//////////////////////////////////////////////////////////////////////
// Author:     Bob Hillhouse
// Date:       September 10, 1998
// Copyright:  American Biometric Company 1998.  All rights reserved.
// Purpose:    This module provides a smart card reader independant 
//             ISO level interface.
// ID:         $Id: dbiscapi.h,v 1.4 1999/01/28 13:56:32 bob Exp $
//////////////////////////////////////////////////////////////////////

#ifndef __DBISCAPI_H__
#define __DBISCAPI_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "unixdefs.h"

// Basic parameters for interfacing to commands 
#define ISO_T0                  (0)
#define ISO_T1                  (1)

#define ISO_CLA                 (0)
#define ISO_INS                 (1)
#define ISO_P1                  (2)
#define ISO_P2                  (3)
#define ISO_P3                  (4)

#define ISO_SW1                 (0)
#define ISO_SW2                 (1)

#define ISO_CASE0               (0)
#define ISO_CASE1               (1)
#define ISO_CASE2S              (2)
#define ISO_CASE3S              (3)
#define ISO_CASE4S              (4)
#define ISO_CASE_T1             (5) // Special T=1 response handling 
                                    // (let card determine response size )

// Maximum length of ATR 
#define ISO_MAX_ATR             64  // should be 32, but why not make it a bit bigger 

// Smartcard speeds 
#define DEFAULT_BAUD             0  // 13.4 KBaud 
#define BAUD_13440               0  // 13.4 KBaud 
#define BAUD_26880               1  // 26.9 KBaud 
#define BAUD_53760               2  // 53.7 KBaud 
#define BAUD_78125               3  // 78.1 KBaud
#define BAUD_107520              4  // 107.5 KBaud - T=0 not supported 
#define ISO_MAX_SPEEDS           5

#define ISO_CHKSUM_LRC           0  // Xor checksum (default) 
#define ISO_CHKSUM_CRC           1  // CRC option (2 bytes) 

// IsoGetInfo Operations 
#define ISO_GET_RESET            1  // Get reset status from driver 
#define ISO_GET_SW               2  // report previous command's status word 
#define ISO_GET_ACTUAL_BAUD      3  // report actual baud rate of available speeds 

// SetParameter types 
#define ISO_SET_SPEED            1  // Manually set library baud rate 
#define ISO_SET_EXTRA_PROC       2  // Set the extra procedure byte mode 
#define ISO_SET_PROTOCOL         3  // Manually change the protocol 
#define ISO_SET_WORKING          4  // Set working waiting time 
#define ISO_SET_SEQUENCE         5  // Set sequence counter for T=1 comms 
#define ISO_SET_CHKSUM           6  // Set the checksum to CRC or LRC 


#define ISO_WWT_DEFAULT          127 //18ms * 127 = 2.29 seconds
#define ISO_WWT_MAXIMUM          255 //18ms * 255 * 16 = seconds

//////////////////////////////////////////////////
// Description
//   - This function intializes the smart card reader.
//     It must be called before any further smart card
//     commands can be called. 
// Parameters
//   - szSCRPath is a zero terminated string which
//     is the full path to the smart card reader DLL
//     (or shared object).  If szSCRPath is NULL then
//     the full path is taken from:
//        Win32 ->  HKEY_LOCAL_MACHINE\SOFTWARE\American Biometric
//                     BioMouse Driver\SCRLibrary
//        Unix  ->  /etc/abcscr.conf
// Postcondition
//   -  On success it returns DBI_FAST_OK and on error 
//      it returns an error as described in dbierrno.h
//   -  If this function is called multiple times, subsequent
//      calls after a successful call increment a counter.
//      CloseLibrary will decrement the counter and when the
//      counter is 0, the library will be closed.
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_InitLibrary( char *szSCRPath );

//////////////////////////////////////////////////
// Description
//   - This function closes the smart card reader.
//     It should be called before the application
//     exits to allow the library to clean itself
//     up and free any allocated memory.
//   - If multiple successful InitLibrary calls were made, 
//     the same number of CloseLibrary calls 
//     must be made in order for clean up to occur.
// Postcondition
//   - On success it returns DBI_FAST_OK and on
//     failure it returns an error code as 
//     described in dbierrno.h
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_CloseLibrary();

//////////////////////////////////////////////////
// Function:  DBI_SC_Reset() 
// Parameters:
// - bAtr is an array of bytes which will be filled with the 
//   smart card's answer to reset on succes and will be cleared
//   on failure.  
// - pwLength is a pointer to a WORD that on exit holds the 
//   length of the answer to reset string that was placed in bAtr. 
// Preconditions:
// - bAtr must pobAtr must have at least DBI_SC_MAX_ATR bytes
//   allocated or memory will be overwritten.  Only the first 
//   DBI_SC_MAX_ATR are modified.
// - pwLength must point to a WORD
// Description:
// - This function performs an ISO reset of a smart card.
//   It must be called before any smart card transactions can 
//   take place.  On success it returns DBI_FAST_OK and on error 
//   it will return an error as described in dbierrno.h
// - On success bAtr is set to the smart card's answer to 
//   reset and pwLength is set to the length of the answer to reset
//   (in bytes), and the function returns DBI_FAST_OK
// - On failure an error code is returned as described in dbierrno.h
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Reset(BYTE *pbAtr, DWORD *pdwLength);

//////////////////////////////////////////////////
// Function:  DBI_SC_Off()
// Parameters:
// Preconditions:
// Description:
//   - This function turns off the smart card.  It should be called
//     before the user removes the smart card to minimize the possibility
//     of damaging the card.  While it is unlikely that the user will
//     damage their smart card by removing it while it is powered, it
//     is significantly safer to power off the card first.
//   - On success this function returns DBI_FAST_OK and on error it 
//     returns an error code as described in dbierrno.h
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Off(void);

//////////////////////////////////////////////////
// Function:  DBI_IsoPts()           - performs ISO7816 PTS
// Parameters:  
// - Speed is an index into a speed table bTimeData?
//   It sets up speed parameters for the baud rate between
//   the biomouse and the smart card
// - Protocol sets the ISO 7816 protocol used between this 
//   library and the smart card.  Presently only T=0 and T=1 
//   protocols are supported; however, the ISO spec supports T=0 to
//   T=15.
// Postconditions:
// - Both the library and smartcard speeds and protocols are modified by
//   this routine on success.
// - On failure the card may be in an undetermined state and an error
//   code as described in dbierrno.h is returned.
// Description:
// - This function can be used to change the baud rate and protocol
//   on any smart card that supports ISO 7816 PTS (protocol type selection)
// - On success both the library and the smart card switch to the new
//   speed and protocol and the function returns DBI_FAST_OK.
// - On failure, an error code is returned as described in dbierrno.h
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Pts(DWORD dwSpeed, DWORD dwProtocol);

//////////////////////////////////////////////////
// Function:  DBI_IsoRaw()           - raw comms with smartcard 
// Parameters:
// - TxLength is the length of the transmit buffer that will be
//   sent to the smart card.
// - TxBuffer is the transmit buffer that will be sent.  It is a
//   pointer to an array of bytes.
// - RxLength is both the length of the receive buffer and the 
//   number of bytes expected back from the smart card.
// - RxBuffer is the receive buffer which is filled with the smart
//   card's response on success.
// Postcondition:  
// - The library speed and protocol settings are not modified
//   by this command even though the smart card's speed and protocol can 
//   be modified with this command.
// Description:  
// - This function performs an uncontrolled IO transaction
//   with the smart card.  There are no protocol rules imposed on the
//   transaction.  This function should be used only if an IsoCase1-4 
//   cannot be used instead.
// - On success the Transmit buffer has been sent to the smart card,
//   the expected number of bytes have been received from the smart card,
//   and the function returns DBI_FAST_OK.
// - On failure an error code is returned as described in dbierrno.h.  
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Raw(DWORD dwTxLength, BYTE *pbTxBuffer,
                          DWORD dwRxLength, BYTE *pbRxBuffer);

//////////////////////////////////////////////////
//Function:  DBI_SC_Case1()         - Performs case 1 transaction
// Parameters:
// - The five bytes of the header consist of CLA, INS, P1, P2, P3
//   In an IsoCase1 P3 is always 0 according to the ISO 7816 spec;
//   however, there are no such restrictions with this library. 
// Description:
// - This function performs an ISO Case1 transaction.  A Case1 
//   transaction consists of transmitting a 5 byte command header to the 
//   card and receiving a 2 byte status word.
// - The two bytes of the status are SW1 and SW2 and these need to be 
//   retrieved with an DBI_IsoGet command.
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Case1(BYTE * pbHeader, BYTE pbStatusWord[2]);

//////////////////////////////////////////////////
// Function:  DBI_SC_Case2()         - Performs case 2 transaction
// Parameters:
// - The five bytes of the header consist of CLA, INS, P1, P2, P3
//   In an IsoCase2 P3 dictates the length of the data response, with the 
//   exception that a value of 0 for P3 indicates 256 bytes of 
//   response data are expected.
// - The receive buffer must be large enough to hold the amount of bytes 
//   as described by P3.
//    
// Description: 
// - This function performs an ISO 7816 spec Case2 transaction.
//   On success it returns DBI_FAST_OK and on failure it returns
//   an error code as described in dbierrno.h.
// - The two status bytes for the status words SW1 and SW2
//   which describe the smart card's response to the command need to 
//   retrieve with a DBI_IsoGet command.
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Case2(BYTE *pbHeader, BYTE *pbRxBuffer, BYTE pbStatusWord[2]);

//////////////////////////////////////////////////
//Function:  DBI_SC_Case3()         - Performs case 3 transaction
// Parameters
// - The five bytes of the header consist of CLA, INS, P1, P2, P3
//   In an IsoCase3 P3 dictates the length of the data to transmit, with the 
//   exception that P3 cannot be 0.
// - The transmit buffer must be large enough to hold the amount of bytes 
//   as described by P3.  The transmit buffer will be sent to the smart 
//   smart card in accordance with the ISO 7816 spec.
//
// Description: 
// - This function performs an ISO 7816 spec Case3 transaction.
//   On success it returns DBI_FAST_OK and on failure it returns
//   an error code as described in dbierrno.h.
// - The two status bytes for the status words SW1 and SW2
//   which describe the smart card's response to the command need to 
//   retrieved with a DBI_IsoGet command.
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Case3(BYTE *pbHeader, BYTE *pbTxBuffer, BYTE pbStatusWord[2]);

//////////////////////////////////////////////////
// Function:  DBI_SC_Case4()         - Performs case 4 transaction
// Parameters:
// - The five bytes of the header consist of CLA, INS, P1, P2, P3
//   In an IsoCase4 P3 dictates the length of the data to transmit, with the 
//   exception that P3 cannot be 0.  RxLength dictates the length of the 
//   expected response.  The response is stored in RxBuffer and RxBuffer
//   must point to a buffer large enough to hold the expected number of 
//   bytes RxLength
// - The transmit buffer must be large enough to hold the amount of bytes 
//   as described by P3.  The transmit buffer will be sent to the smart 
//   smart card in accordance with the ISO 7816 spec.
//
// Description: 
// - This function performs an ISO 7816 spec Case4 transaction.
//   On success it returns DBI_FAST_OK and on failure it returns
//   an error code as described in dbierrno.h.
// - The two status bytes for the status words SW1 and SW2
//   which describe the smart card's response to the command need to 
//   retrieved with a DBI_IsoGet command.
// - Presently the IsoCase4 is only support under the T=1 protocol.
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Case4(BYTE *pbHeader, BYTE *pbTxBuffer, 
                            DWORD dwRxLength, BYTE *pbRxBuffer,
                            BYTE bGetClass, BYTE pbStatusWord[2]);

//////////////////////////////////////////////////
// Function:  DBI_SC_Set()           - Sets driver operating parameters
// Parameters:
// - Parameter indicates which library parameter the caller wants 
//   to set.  Valid parameters are comprised of:
//    ISO_SET_SPEED: 
//      - Allows the caller to change the library's speed setting that
//        would normally be changed with DBI_IsoPts, it does not affect
//        the smart card.
//        Valid values:  0 to ISO_MAX_SPEEDS
//   ISO_SET_PROTOCOL:
//      - Allows the caller to change the library's protocol setting that
//        would normally be changed with DBI_IsoPts, it does not affect
//        the smart card.
//        Valid values:  0 or 1.
//   ISO_SET_SEQUENCE:
//      - Allows the caller to set the library's T=1 sequence number.
//        This can be used manually resynchronize the T=1 communication
//        with the smart card.
//        Valid values:  0 or 1.
//   ISO_SET_CHKSUM:
//        Allows the caller to set the library's checksum mode to either
//        ISO_CHKSUM_LRC or ISO_CHKSUM_CRC.  ISO_CHKSUM_LRC performs an
//        xor checksum check and ISO_CHKSUM_CRC performs a 16 bit CRC check.
//   ISO_SET_EXTRA_PROC:
//        Some ISO smart cards return procedure bytes during a
//        Case3 (receive data) transaction while others do not.  
//        Setting EXTRAPROC to TRUE commands the library to expect 
//        a procedure byte and setting it to FALSE commands the library 
//        not to expect a procedure byte.
// Description:
//
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Set(DWORD dwParameter, DWORD dwValue);

//////////////////////////////////////////////////
//Function:  DBI_SC_Get()           - Gets information from driver
// Parameters:
// Parameter is comprised of:
//   SCR_GET_RESET:
//     which returns the reset status of the smart card.
//   ISO_GET_SW: 
//     which returns fills VALUE with the two status words
//     from the smart card
//   ISO_GET_ACTUAL_BAUD: 
//     which fills Value with the actual baud rate
//
// Description:
// - Returns information about the library state.
//   
//////////////////////////////////////////////////
DWORD ABC_DECL DBI_SC_Get(DWORD dwParameter, DWORD *pdwValue);

#ifdef __cplusplus
}
#endif

#endif
