package Sys::Ibam;

use strict;
use warnings;

use AutoLoader;

our $VERSION = '0.05';

require XSLoader;
XSLoader::load('Sys::Ibam', $VERSION);

1;
__END__

=head1 NAME

Sys::Ibam - Perl extension for the intellegent battery monitor

=head1 SYNOPSIS

  use Sys::Ibam;

  my $ibam = Sys::Ibam->new();

  $ibam->update();
  if ($ibam->valid()) {
    $ibam->update_statistics();
  } else {
    $ibam->ignore_statistics();
  }
  $ibam->save();

=head1 DESCRIPTION

Sys::Ibam is a Perl extension for the ingellegent battery monitor by Sebastian
Ritterbusch <Sebastian@Ritterbusch.de>. It uses statistical and adaptive linear
methods to provide accurate estimations of minutes of battery left or of the
time needed until full recharge. It requires /proc/apm or /proc/pmu/.

=head1 METHODS

=head2 new

  my $ibam = Sys::Ibam->new();
  my $ibam = Sys::Ibam->new($do_second_correction);

Returns a new ibam instance. If $do_second_correction is true then the return
value of seconds_{battery,charge}_correction() will be automatically added to
when calling seconds_left_{battery,charge}{,_adaptive}().

=head2 import_old_data

  $ibam->import_old_data();

Imports version 0.1 data from the current directory.

=head2 update_statistics

  $ibam->update_statistics();

Updates statistics of the ibam instance. You should call save() after calling
update_statistics() to write the new statistics to disk.

=head2 ignore_statistics

  $ibam->ignore_statistics();

Ignores the new data read by update. You should call ignore_statistics() if the
ibam data is invalid or if you don't want the data generated with the last
update to be part of the statistics, which will be written to disk.

=head2 save

  $ibam->save();

Saves the current statistics to disk.

=head2 profile_filename

  my $filename = $ibam->profile_filename(4, 'battery');

Takes a profile number (between 0 and 500) and a string. The string tells the
type of the profile. 'battery', 'charge', 'full' and '' (no battery) are valid.
It returns the correspondent filename of the profile or undef if the type is
invalid.

=head2 current_profile_number

  my $num = $ibam->current_profile_number();

Returns the current profile number.

=head2 current_profile_type

  my $type = $ibam->current_profile_type();

Returns the current profile type in the same format that can be passed to
profile_filename().

=head2 set_profile_logging

  $ibam->set_profile_logging(1);

Sets the profile_logging attribute. It'll be enabled if the argument is true or
disabled if it's false.

=head2 profile_logging_setting

  my $logging = $ibam->profile_logging_setting();

Returns the current value of the profile_logging attibute.

=head2 seconds_left_battery_bios

  my $bat = $ibam->seconds_left_battery_bios();

Returns the time left for the battery reported by the bios in seconds.

=head2 seconds_left_battery

=head2 seconds_left_battery_adaptive

=head2 percent_battery_bios

  my $percent = $ibam->percent_battery_bios();

Returns the charge state of the battery reported by the bios.

=head2 percent_battery

=head2 seconds_left_charge

=head2 seconds_left_charge_adaptive

=head2 percent_charge

=head2 seconds_battery_total

=head2 seconds_battery_total_adaptive

=head2 seconds_battery_correction

=head2 seconds_charge_correction

=head2 on_battery

  my $bat_status = $ibam->on_battery();

Returns a true value if we're on battery. If not, it returns false.

=head2 charging

  my $charge_status = $ibam->charging();

Returns a true value if the battery is charged right now.

=head2 valid

  my $valid = $ibam->valid();

Checks if the data generated by the last update() call is valid. If it is, you
should call update_statistics() or ignore_statistics() if it's not. Then you
should save the data to disk calling the save() method.

=head2 update

  $ibam->update();

Updates ibams data. You should check if it's valid and then update or ignore the
statistics.

=head1 SEE ALSO

ibam(1)

=head1 AUTHOR

Florian Ragwitz E<lt>flora@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2004 by Florian Ragwitz

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.4 or,
at your option, any later version of Perl 5 you may have available.


=cut
