package Docker::Names::Random;

## no critic (Documentation::RequirePodAtEnd)
## no critic (Documentation::RequirePodSections)

use strict;
use warnings;
use 5.010;
use feature qw( state );
use utf8;    # The __DATA__ is UTF-8
use English qw(-no_match_vars);

# ABSTRACT: Create random strings like Docker does for container names.

our $VERSION = '0.0.2'; # VERSION: generated by DZP::OurPkgVersion

use Exporter 'import';
our @EXPORT_OK   = qw( docker_name );
our %EXPORT_TAGS = ( 'all' => [qw( docker_name )] );

use YAML::PP;
use Hash::Util qw( lock_keys );


# Internal functions

sub _get_data {
    state $ypp = YAML::PP->new;
    local $INPUT_RECORD_SEPARATOR = undef;
    state $data = $ypp->load_string(<DATA>);
    return $data;
}

sub new {
    my $class = shift;

    my $data = _get_data();
    my $self = { data => $data };
    bless $self, $class;
    lock_keys( %{$self} );
    return $self;
}


sub docker_name {
    my $self = shift;
    my $data;
    if ( defined $self ) {
        $data = $self->{'data'};
    }
    else {

        # We are not running as an object method
        $data = _get_data();
    }

    my ( $adj, $surname );

  CREATE_NAME:
    my $r_for_adj = int( rand( scalar @{ $data->{'adjectives'} } ) );
    $adj = $data->{'adjectives'}->[$r_for_adj]->{'word'};
    my $r_for_name = int( rand( scalar @{ $data->{'names'} } ) );
    $surname = $data->{'names'}->[$r_for_name]->{'surname'};

    if ( $adj eq 'boring' && $surname eq 'wozniak' ) {

        # Steve Wozniak is not boring
        goto 'CREATE_NAME';
    }

    return $adj . q{_} . $surname;
}


1;

=pod

=encoding UTF-8

=head1 NAME

Docker::Names::Random - Create random strings like Docker does for container names.

=head1 VERSION

version 0.0.2

=head1 SYNOPSIS

    # As an object (if you create many, this is more efficient).
    require Docker::Names::Random;

    my $dnr = Docker::Names::Random->new();
    my $random_name1 = $dnr->docker_name();

    # As an imported function.
    use Docker::Names::Random qw( docker_name );
    # OR
    use Docker::Names::Random qw( :all );

    my $random_name2 = docker_name();

=head1 DESCRIPTION

If you are using Docker, you may have noticed that it
creates random names for containers when you haven't provided any
specific name. These names are a combination of an adjective
and a proper name of an individual. The individuals are
famous men and women picked from the history of scientific exploration
and engineering.

This package allows you to use the same system in your own programs.
You would get combinations like
I<interesting_mendeleev>,
I<epic_engelbart>,
I<lucid_dhawan>,
I<recursing_cori>,
I<ecstatic_liskov> and
I<busy_ardinghelli>.

The combination I<boring_wozniak> is not allowed because
L<Steve Wozniak|https://en.wikipedia.org/wiki/Steve_Wozniak> is not boring.
This same limitation exists in the
L<original code|https://github.com/moby/moby/blob/master/pkg/namesgenerator/names-generator.go>.

=for stopwords overthruster Docker::Names::Random Docker Names Random

=head1 STATUS

This module is currently being developed so changes in the API are possible,
though unlikely.

=head1 METHODS

=head2 new

Create object.

    require Docker::Names::Random;
    my $dnr = Docker::Names::Random->new();

=head2 docker_name

Return a random string.

    my $random_name = $dnr->docker_name();

=head1 FUNCTIONS

=head2 docker_name

Return a random string.

    use Docker::Names::Random qw( docker_name );
    # OR
    use Docker::Names::Random qw( :all );

    my $random_name = docker_name();

=head1 AUTHOR

Mikko Johannes Koivunalho <mikko.koivunalho@iki.fi>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by Mikko Johannes Koivunalho.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

__DATA__
---
adjectives:
- word: admiring
- word: adoring
- word: affectionate
- word: agitated
- word: amazing
- word: angry
- word: awesome
- word: beautiful
- word: blissful
- word: bold
- word: boring
- word: brave
- word: busy
- word: charming
- word: clever
- word: cool
- word: compassionate
- word: competent
- word: condescending
- word: confident
- word: cranky
- word: crazy
- word: dazzling
- word: determined
- word: distracted
- word: dreamy
- word: eager
- word: ecstatic
- word: elastic
- word: elated
- word: elegant
- word: eloquent
- word: epic
- word: exciting
- word: fervent
- word: festive
- word: flamboyant
- word: focused
- word: friendly
- word: frosty
- word: funny
- word: gallant
- word: gifted
- word: goofy
- word: gracious
- word: great
- word: happy
- word: hardcore
- word: heuristic
- word: hopeful
- word: hungry
- word: infallible
- word: inspiring
- word: interesting
- word: intelligent
- word: jolly
- word: jovial
- word: keen
- word: kind
- word: laughing
- word: loving
- word: lucid
- word: magical
- word: mystifying
- word: modest
- word: musing
- word: naughty
- word: nervous
- word: nice
- word: nifty
- word: nostalgic
- word: objective
- word: optimistic
- word: peaceful
- word: pedantic
- word: pensive
- word: practical
- word: priceless
- word: quirky
- word: quizzical
- word: recursing
- word: relaxed
- word: reverent
- word: romantic
- word: sad
- word: serene
- word: sharp
- word: silly
- word: sleepy
- word: stoic
- word: strange
- word: stupefied
- word: suspicious
- word: sweet
- word: tender
- word: thirsty
- word: trusting
- word: unruffled
- word: upbeat
- word: vibrant
- word: vigilant
- word: vigorous
- word: wizardly
- word: wonderful
- word: xenodochial
- word: youthful
- word: zealous
- word: zen
names:
- explanation: Muhammad ibn Jābir al-Ḥarrānī al-Battānī was a founding father of astronomy.
  link: https://en.wikipedia.org/wiki/Mu%E1%B8%A5ammad_ibn_J%C4%81bir_al-%E1%B8%A4arr%C4%81n%C4%AB_al-Batt%C4%81n%C4%AB
  surname: albattani
- explanation: Frances E. Allen, became the first female IBM Fellow in 1989. In 2006, she became the first female recipient of the ACM's Turing Award.
  link: https://en.wikipedia.org/wiki/Frances_E._Allen
  surname: allen
- explanation: June Almeida - Scottish virologist who took the first pictures of the rubella virus -
  link: https://en.wikipedia.org/wiki/June_Almeida
  surname: almeida
- explanation: Kathleen Antonelli, American computer programmer and one of the six original programmers of the ENIAC -
  link: https://en.wikipedia.org/wiki/Kathleen_Antonelli
  surname: antonelli
- explanation: Maria Gaetana Agnesi - Italian mathematician, philosopher, theologian and humanitarian. She was the first woman to write a mathematics handbook and the first woman appointed as a Mathematics Professor at a University.
  link: https://en.wikipedia.org/wiki/Maria_Gaetana_Agnesi
  surname: agnesi
- explanation: Archimedes was a physicist, engineer and mathematician who invented too many things to list them here.
  link: https://en.wikipedia.org/wiki/Archimedes
  surname: archimedes
- explanation: Maria Ardinghelli - Italian translator, mathematician and physicist -
  link: https://en.wikipedia.org/wiki/Maria_Ardinghelli
  surname: ardinghelli
- explanation: Aryabhata - Ancient Indian mathematician-astronomer during 476-550 CE
  link: https://en.wikipedia.org/wiki/Aryabhata
  surname: aryabhata
- explanation: Wanda Austin - Wanda Austin is the President and CEO of The Aerospace Corporation, a leading architect for the US security space programs.
  link: https://en.wikipedia.org/wiki/Wanda_Austin
  surname: austin
- explanation: Charles Babbage invented the concept of a programmable computer.
  link: https://en.wikipedia.org/wiki/Charles_Babbage.
  surname: babbage
- explanation: Stefan Banach - Polish mathematician, was one of the founders of modern functional analysis.
  link: https://en.wikipedia.org/wiki/Stefan_Banach
  surname: banach
- explanation: Buckaroo Banzai and his mentor Dr. Hikita perfected the "oscillation overthruster", a device that allows one to pass through solid matter. -
  link: https://en.wikipedia.org/wiki/The_Adventures_of_Buckaroo_Banzai_Across_the_8th_Dimension
  surname: banzai
- explanation: John Bardeen co-invented the transistor -
  link: https://en.wikipedia.org/wiki/John_Bardeen
  surname: bardeen
- explanation: Jean Bartik, born Betty Jean Jennings, was one of the original programmers for the ENIAC computer.
  link: https://en.wikipedia.org/wiki/Jean_Bartik
  surname: bartik
- explanation: Laura Bassi, the world's first female professor
  link: https://en.wikipedia.org/wiki/Laura_Bassi
  surname: bassi
- explanation: Hugh Beaver, British engineer, founder of the Guinness Book of World Records
  link: https://en.wikipedia.org/wiki/Hugh_Beaver
  surname: beaver
- explanation: Alexander Graham Bell - an eminent Scottish-born scientist, inventor, engineer and innovator who is credited with inventing the first practical telephone -
  link: https://en.wikipedia.org/wiki/Alexander_Graham_Bell
  surname: bell
- explanation: Karl Friedrich Benz - a German automobile engineer. Inventor of the first practical motorcar.
  link: https://en.wikipedia.org/wiki/Karl_Benz
  surname: benz
- explanation: Homi J Bhabha - was an Indian nuclear physicist, founding director, and professor of physics at the Tata Institute of Fundamental Research. Colloquially known as "father of Indian nuclear programme"-
  link: https://en.wikipedia.org/wiki/Homi_J._Bhabha
  surname: bhabha
- explanation: Bhaskara II - Ancient Indian mathematician-astronomer whose work on calculus predates Newton and Leibniz by over half a millennium -
  link: https://en.wikipedia.org/wiki/Bh%C4%81skara_II#Calculus
  surname: bhaskara
- explanation: Sue Black - British computer scientist and campaigner. She has been instrumental in saving Bletchley Park, the site of World War II codebreaking -
  link: https://en.wikipedia.org/wiki/Sue_Black_(computer_scientist)
  surname: black
- explanation: Elizabeth Helen Blackburn - Australian-American Nobel laureate; best known for co-discovering telomerase.
  link: https://en.wikipedia.org/wiki/Elizabeth_Blackburn
  surname: blackburn
- explanation: Elizabeth Blackwell - American doctor and first American woman to receive a medical degree -
  link: https://en.wikipedia.org/wiki/Elizabeth_Blackwell
  surname: blackwell
- explanation: Niels Bohr is the father of quantum theory.
  link: https://en.wikipedia.org/wiki/Niels_Bohr.
  surname: bohr
- explanation: Kathleen Booth, she's credited with writing the first assembly language.
  link: https://en.wikipedia.org/wiki/Kathleen_Booth
  surname: booth
- explanation: Anita Borg - Anita Borg was the founding director of the Institute for Women and Technology (IWT).
  link: https://en.wikipedia.org/wiki/Anita_Borg
  surname: borg
- explanation: Satyendra Nath Bose - He provided the foundation for Bose–Einstein statistics and the theory of the Bose–Einstein condensate. -
  link: https://en.wikipedia.org/wiki/Satyendra_Nath_Bose
  surname: bose
- explanation: Katherine Louise Bouman is an imaging scientist and Assistant Professor of Computer Science at the California Institute of Technology. She researches computational methods for imaging, and developed an algorithm that made possible the picture first visualization of a black hole using the Event Horizon Telescope. -
  link: https://en.wikipedia.org/wiki/Katie_Bouman
  surname: bouman
- explanation: Evelyn Boyd Granville - She was one of the first African-American woman to receive a Ph.D. in mathematics; she earned it in 1949 from Yale University.
  link: https://en.wikipedia.org/wiki/Evelyn_Boyd_Granville
  surname: boyd
- explanation: Brahmagupta - Ancient Indian mathematician during 598-670 CE who gave rules to compute with zero -
  link: https://en.wikipedia.org/wiki/Brahmagupta#Zero
  surname: brahmagupta
- explanation: Walter Houser Brattain co-invented the transistor -
  link: https://en.wikipedia.org/wiki/Walter_Houser_Brattain
  surname: brattain
- explanation: Emmett Brown invented time travel.
  link: https://en.wikipedia.org/wiki/Emmett_Brown (thanks Brian Goff)
  surname: brown
- explanation: Linda Brown Buck - American biologist and Nobel laureate best known for her genetic and molecular analyses of the mechanisms of smell.
  link: https://en.wikipedia.org/wiki/Linda_B._Buck
  surname: buck
- explanation: Dame Susan Jocelyn Bell Burnell - Northern Irish astrophysicist who discovered radio pulsars and was the first to analyse them.
  link: https://en.wikipedia.org/wiki/Jocelyn_Bell_Burnell
  surname: burnell
- explanation: Annie Jump Cannon - pioneering female astronomer who classified hundreds of thousands of stars and created the system we use to understand stars today.
  link: https://en.wikipedia.org/wiki/Annie_Jump_Cannon
  surname: cannon
- explanation: Rachel Carson - American marine biologist and conservationist, her book Silent Spring and other writings are credited with advancing the global environmental movement.
  link: https://en.wikipedia.org/wiki/Rachel_Carson
  surname: carson
- explanation: Dame Mary Lucy Cartwright - British mathematician who was one of the first to study what is now known as chaos theory. Also known for Cartwright's theorem which finds applications in signal processing.
  link: https://en.wikipedia.org/wiki/Mary_Cartwright
  surname: cartwright
- explanation: George Washington Carver - American agricultural scientist and inventor. He was the most prominent black scientist of the early 20th century.
  link: https://en.wikipedia.org/wiki/George_Washington_Carver
  surname: carver
- explanation: Vinton Gray Cerf - American Internet pioneer, recognised as one of "the fathers of the Internet". With Robert Elliot Kahn, he designed TCP and IP, the primary data communication protocols of the Internet and other computer networks.
  link: https://en.wikipedia.org/wiki/Vint_Cerf
  surname: cerf
- explanation: Subrahmanyan Chandrasekhar - Astrophysicist known for his mathematical theory on different stages and evolution in structures of the stars. He has won nobel prize for physics -
  link: https://en.wikipedia.org/wiki/Subrahmanyan_Chandrasekhar
  surname: chandrasekhar
- explanation: 'Sergey Alexeyevich Chaplygin (Russian: Серге́й Алексе́евич Чаплы́гин; April 5, 1869 – October 8, 1942) was a Russian and Soviet physicist, mathematician, and mechanical engineer. He is known for mathematical formulas such as Chaplygin''s equation and for a hypothetical substance in cosmology called Chaplygin gas, named after him.'
  link: https://en.wikipedia.org/wiki/Sergey_Chaplygin
  surname: chaplygin
- explanation: Émilie du Châtelet - French natural philosopher, mathematician, physicist, and author during the early 1730s, known for her translation of and commentary on Isaac Newton's book Principia containing basic laws of physics.
  link: https://en.wikipedia.org/wiki/%C3%89milie_du_Ch%C3%A2telet
  surname: chatelet
- explanation: Asima Chatterjee was an Indian organic chemist noted for her research on vinca alkaloids, development of drugs for treatment of epilepsy and malaria -
  link: https://en.wikipedia.org/wiki/Asima_Chatterjee
  surname: chatterjee
- explanation: Pafnuty Chebyshev - Russian mathematician. He is known fo his works on probability, statistics, mechanics, analytical geometry and number theory
  link: https://en.wikipedia.org/wiki/Pafnuty_Chebyshev
  surname: chebyshev
- explanation: Bram Cohen - American computer programmer and author of the BitTorrent peer-to-peer protocol.
  link: https://en.wikipedia.org/wiki/Bram_Cohen
  surname: cohen
- explanation: David Lee Chaum - American computer scientist and cryptographer. Known for his seminal contributions in the field of anonymous communication.
  link: https://en.wikipedia.org/wiki/David_Chaum
  surname: chaum
- explanation: Joan Clarke - Bletchley Park code breaker during the Second World War who pioneered techniques that remained top secret for decades. Also an accomplished numismatist
  link: https://en.wikipedia.org/wiki/Joan_Clarke
  surname: clarke
- explanation: Jane Colden - American botanist widely considered the first female American botanist -
  link: https://en.wikipedia.org/wiki/Jane_Colden
  surname: colden
- explanation: Gerty Theresa Cori - American biochemist who became the third woman—and first American woman—to win a Nobel Prize in science, and the first woman to be awarded the Nobel Prize in Physiology or Medicine. Cori was born in Prague.
  link: https://en.wikipedia.org/wiki/Gerty_Cori
  surname: cori
- explanation: Seymour Roger Cray was an American electrical engineer and supercomputer architect who designed a series of computers that were the fastest in the world for decades.
  link: https://en.wikipedia.org/wiki/Seymour_Cray
  surname: cray
- explanation: 'This entry reflects a husband and wife team who worked together:'
  link: https://en.wikipedia.org/wiki/Samuel_Curran
  surname: curran
- explanation: Marie Curie discovered radioactivity.
  link: https://en.wikipedia.org/wiki/Marie_Curie.
  surname: curie
- explanation: Charles Darwin established the principles of natural evolution.
  link: https://en.wikipedia.org/wiki/Charles_Darwin.
  surname: darwin
- explanation: Leonardo Da Vinci invented too many things to list here.
  link: https://en.wikipedia.org/wiki/Leonardo_da_Vinci.
  surname: davinci
- explanation: A. K. (Alexander Keewatin) Dewdney, Canadian mathematician, computer scientist, author and filmmaker. Contributor to Scientific American's "Computer Recreations" from 1984 to 1991. Author of Core War (program), The Planiverse, The Armchair Universe, The Magic Machine, The New Turing Omnibus, and more.
  link: https://en.wikipedia.org/wiki/Alexander_Dewdney
  surname: dewdney
- explanation: Satish Dhawan - Indian mathematician and aerospace engineer, known for leading the successful and indigenous development of the Indian space programme.
  link: https://en.wikipedia.org/wiki/Satish_Dhawan
  surname: dhawan
- explanation: Bailey Whitfield Diffie - American cryptographer and one of the pioneers of public-key cryptography.
  link: https://en.wikipedia.org/wiki/Whitfield_Diffie
  surname: diffie
- explanation: Edsger Wybe Dijkstra was a Dutch computer scientist and mathematical scientist.
  link: https://en.wikipedia.org/wiki/Edsger_W._Dijkstra.
  surname: dijkstra
- explanation: Paul Adrien Maurice Dirac - English theoretical physicist who made fundamental contributions to the early development of both quantum mechanics and quantum electrodynamics.
  link: https://en.wikipedia.org/wiki/Paul_Dirac
  surname: dirac
- explanation: Agnes Meyer Driscoll - American cryptanalyst during World Wars I and II who successfully cryptanalysed a number of Japanese ciphers. She was also the co-developer of one of the cipher machines of the US Navy, the CM.
  link: https://en.wikipedia.org/wiki/Agnes_Meyer_Driscoll
  surname: driscoll
- explanation: Donna Dubinsky - played an integral role in the development of personal digital assistants (PDAs) serving as CEO of Palm, Inc. and co-founding Handspring.
  link: https://en.wikipedia.org/wiki/Donna_Dubinsky
  surname: dubinsky
- explanation: Annie Easley - She was a leading member of the team which developed software for the Centaur rocket stage and one of the first African-Americans in her field.
  link: https://en.wikipedia.org/wiki/Annie_Easley
  surname: easley
- explanation: Thomas Alva Edison, prolific inventor
  link: https://en.wikipedia.org/wiki/Thomas_Edison
  surname: edison
- explanation: Albert Einstein invented the general theory of relativity.
  link: https://en.wikipedia.org/wiki/Albert_Einstein
  surname: einstein
- explanation: 'Alexandra Asanovna Elbakyan (Russian: Алекса́ндра Аса́новна Элбакя́н) is a Kazakhstani graduate student, computer programmer, internet pirate in hiding, and the creator of the site Sci-Hub. Nature has listed her in 2016 in the top ten people that mattered in science, and Ars Technica has compared her to Aaron Swartz. -'
  link: https://en.wikipedia.org/wiki/Alexandra_Elbakyan
  surname: elbakyan
- explanation: Taher A. ElGamal - Egyptian cryptographer best known for the ElGamal discrete log cryptosystem and the ElGamal digital signature scheme.
  link: https://en.wikipedia.org/wiki/Taher_Elgamal
  surname: elgamal
- explanation: Gertrude Elion - American biochemist, pharmacologist and the 1988 recipient of the Nobel Prize in Medicine -
  link: https://en.wikipedia.org/wiki/Gertrude_Elion
  surname: elion
- explanation: James Henry Ellis - British engineer and cryptographer employed by the GCHQ. Best known for conceiving for the first time, the idea of public-key cryptography.
  link: https://en.wikipedia.org/wiki/James_H._Ellis
  surname: ellis
- explanation: 'Douglas Engelbart gave the mother of all demos:'
  link: https://en.wikipedia.org/wiki/Douglas_Engelbart
  surname: engelbart
- explanation: Euclid invented geometry.
  link: https://en.wikipedia.org/wiki/Euclid
  surname: euclid
- explanation: Leonhard Euler invented large parts of modern mathematics.
  link: https://de.wikipedia.org/wiki/Leonhard_Euler
  surname: euler
- explanation: Michael Faraday - British scientist who contributed to the study of electromagnetism and electrochemistry.
  link: https://en.wikipedia.org/wiki/Michael_Faraday
  surname: faraday
- explanation: Horst Feistel - German-born American cryptographer who was one of the earliest non-government researchers to study the design and theory of block ciphers. Co-developer of DES and Lucifer. Feistel networks, a symmetric structure used in the construction of block ciphers are named after him.
  link: https://en.wikipedia.org/wiki/Horst_Feistel
  surname: feistel
- explanation: Pierre de Fermat pioneered several aspects of modern mathematics.
  link: https://en.wikipedia.org/wiki/Pierre_de_Fermat
  surname: fermat
- explanation: Enrico Fermi invented the first nuclear reactor.
  link: https://en.wikipedia.org/wiki/Enrico_Fermi.
  surname: fermi
- explanation: Richard Feynman was a key contributor to quantum mechanics and particle physics.
  link: https://en.wikipedia.org/wiki/Richard_Feynman
  surname: feynman
- explanation: Benjamin Franklin is famous for his experiments in electricity and the invention of the lightning rod.
  link: ''
  surname: franklin
- explanation: Yuri Alekseyevich Gagarin - Soviet pilot and cosmonaut, best known as the first human to journey into outer space.
  link: https://en.wikipedia.org/wiki/Yuri_Gagarin
  surname: gagarin
- explanation: Galileo was a founding father of modern astronomy, and faced politics and obscurantism to establish scientific truth.
  link: https://en.wikipedia.org/wiki/Galileo_Galilei
  surname: galileo
- explanation: Évariste Galois - French mathematician whose work laid the foundations of Galois theory and group theory, two major branches of abstract algebra, and the subfield of Galois connections, all while still in his late teens.
  link: https://en.wikipedia.org/wiki/%C3%89variste_Galois
  surname: galois
- explanation: Kadambini Ganguly - Indian physician, known for being the first South Asian female physician, trained in western medicine, to graduate in South Asia.
  link: https://en.wikipedia.org/wiki/Kadambini_Ganguly
  surname: ganguly
- explanation: William Henry "Bill" Gates III is an American business magnate, philanthropist, investor, computer programmer, and inventor.
  link: https://en.wikipedia.org/wiki/Bill_Gates
  surname: gates
- explanation: Johann Carl Friedrich Gauss - German mathematician who made significant contributions to many fields, including number theory, algebra, statistics, analysis, differential geometry, geodesy, geophysics, mechanics, electrostatics, magnetic fields, astronomy, matrix theory, and optics.
  link: https://en.wikipedia.org/wiki/Carl_Friedrich_Gauss
  surname: gauss
- explanation: Marie-Sophie Germain - French mathematician, physicist and philosopher. Known for her work on elasticity theory, number theory and philosophy.
  link: https://en.wikipedia.org/wiki/Sophie_Germain
  surname: germain
- explanation: Adele Goldberg, was one of the designers and developers of the Smalltalk language.
  link: https://en.wikipedia.org/wiki/Adele_Goldberg_(computer_scientist)
  surname: goldberg
- explanation: Adele Goldstine, born Adele Katz, wrote the complete technical description for the first electronic digital computer, ENIAC.
  link: https://en.wikipedia.org/wiki/Adele_Goldstine
  surname: goldstine
- explanation: Shafi Goldwasser is a computer scientist known for creating theoretical foundations of modern cryptography. Winner of 2012 ACM Turing Award.
  link: https://en.wikipedia.org/wiki/Shafi_Goldwasser
  surname: goldwasser
- explanation: James Golick, all around gangster.
  link: ''
  surname: golick
- explanation: Jane Goodall - British primatologist, ethologist, and anthropologist who is considered to be the world's foremost expert on chimpanzees -
  link: https://en.wikipedia.org/wiki/Jane_Goodall
  surname: goodall
- explanation: Stephen Jay Gould was was an American paleontologist, evolutionary biologist, and historian of science. He is most famous for the theory of punctuated equilibrium -
  link: https://en.wikipedia.org/wiki/Stephen_Jay_Gould
  surname: gould
- explanation: Carolyn Widney Greider - American molecular biologist and joint winner of the 2009 Nobel Prize for Physiology or Medicine for the discovery of telomerase.
  link: https://en.wikipedia.org/wiki/Carol_W._Greider
  surname: greider
- explanation: Alexander Grothendieck - German-born French mathematician who became a leading figure in the creation of modern algebraic geometry.
  link: https://en.wikipedia.org/wiki/Alexander_Grothendieck
  surname: grothendieck
- explanation: Lois Haibt - American computer scientist, part of the team at IBM that developed FORTRAN -
  link: https://en.wikipedia.org/wiki/Lois_Haibt
  surname: haibt
- explanation: Margaret Hamilton - Director of the Software Engineering Division of the MIT Instrumentation Laboratory, which developed on-board flight software for the Apollo space program.
  link: https://en.wikipedia.org/wiki/Margaret_Hamilton_(scientist)
  surname: hamilton
- explanation: Caroline Harriet Haslett - English electrical engineer, electricity industry administrator and champion of women's rights. Co-author of British Standard 1363 that specifies AC power plugs and sockets used across the United Kingdom (which is widely considered as one of the safest designs).
  link: https://en.wikipedia.org/wiki/Caroline_Haslett
  surname: haslett
- explanation: Stephen Hawking pioneered the field of cosmology by combining general relativity and quantum mechanics.
  link: https://en.wikipedia.org/wiki/Stephen_Hawking
  surname: hawking
- explanation: Martin Edward Hellman - American cryptologist, best known for his invention of public-key cryptography in co-operation with Whitfield Diffie and Ralph Merkle.
  link: https://en.wikipedia.org/wiki/Martin_Hellman
  surname: hellman
- explanation: Werner Heisenberg was a founding father of quantum mechanics.
  link: https://en.wikipedia.org/wiki/Werner_Heisenberg
  surname: heisenberg
- explanation: Grete Hermann was a German philosopher noted for her philosophical work on the foundations of quantum mechanics.
  link: https://en.wikipedia.org/wiki/Grete_Hermann
  surname: hermann
- explanation: Caroline Lucretia Herschel - German astronomer and discoverer of several comets.
  link: https://en.wikipedia.org/wiki/Caroline_Herschel
  surname: herschel
- explanation: Heinrich Rudolf Hertz - German physicist who first conclusively proved the existence of the electromagnetic waves.
  link: https://en.wikipedia.org/wiki/Heinrich_Hertz
  surname: hertz
- explanation: Jaroslav Heyrovský was the inventor of the polarographic method, father of the electroanalytical method, and recipient of the Nobel Prize in 1959. His main field of work was polarography.
  link: https://en.wikipedia.org/wiki/Jaroslav_Heyrovsk%C3%BD
  surname: heyrovsky
- explanation: Dorothy Hodgkin was a British biochemist, credited with the development of protein crystallography. She was awarded the Nobel Prize in Chemistry in 1964.
  link: https://en.wikipedia.org/wiki/Dorothy_Hodgkin
  surname: hodgkin
- explanation: 'Douglas R. Hofstadter is an American professor of cognitive science and author of the Pulitzer Prize and American Book Award-winning work Goedel, Escher, Bach: An Eternal Golden Braid in 1979. A mind-bending work which coined Hofstadter''s Law: "It always takes longer than you expect, even when you take into account Hofstadter''s Law."'
  link: https://en.wikipedia.org/wiki/Douglas_Hofstadter
  surname: hofstadter
- explanation: Erna Schneider Hoover revolutionized modern communication by inventing a computerized telephone switching method.
  link: https://en.wikipedia.org/wiki/Erna_Schneider_Hoover
  surname: hoover
- explanation: Grace Hopper developed the first compiler for a computer programming language and  is credited with popularizing the term "debugging" for fixing computer glitches.
  link: https://en.wikipedia.org/wiki/Grace_Hopper
  surname: hopper
- explanation: Frances Hugle, she was an American scientist, engineer, and inventor who contributed to the understanding of semiconductors, integrated circuitry, and the unique electrical principles of microscopic materials.
  link: https://en.wikipedia.org/wiki/Frances_Hugle
  surname: hugle
- explanation: Hypatia - Greek Alexandrine Neoplatonist philosopher in Egypt who was one of the earliest mothers of mathematics -
  link: https://en.wikipedia.org/wiki/Hypatia
  surname: hypatia
- explanation: Teruko Ishizaka - Japanese scientist and immunologist who co-discovered the antibody class Immunoglobulin E.
  link: https://en.wikipedia.org/wiki/Teruko_Ishizaka
  surname: ishizaka
- explanation: Mary Jackson, American mathematician and aerospace engineer who earned the highest title within NASA's engineering department -
  link: https://en.wikipedia.org/wiki/Mary_Jackson_(engineer)
  surname: jackson
- explanation: Yeong-Sil Jang was a Korean scientist and astronomer during the Joseon Dynasty; he invented the first metal printing press and water gauge.
  link: https://en.wikipedia.org/wiki/Jang_Yeong-sil
  surname: jang
- explanation: Mae Carol Jemison -  is an American engineer, physician, and former NASA astronaut. She became the first black woman to travel in space when she served as a mission specialist aboard the Space Shuttle Endeavour -
  link: https://en.wikipedia.org/wiki/Mae_Jemison
  surname: jemison
- explanation: Betty Jennings - one of the original programmers of the ENIAC. https://en.wikipedia.org/wiki/ENIAC -
  link: https://en.wikipedia.org/wiki/Jean_Bartik
  surname: jennings
- explanation: Mary Lou Jepsen, was the founder and chief technology officer of One Laptop Per Child (OLPC), and the founder of Pixel Qi.
  link: https://en.wikipedia.org/wiki/Mary_Lou_Jepsen
  surname: jepsen
- explanation: Katherine Coleman Goble Johnson - American physicist and mathematician contributed to the NASA.
  link: https://en.wikipedia.org/wiki/Katherine_Johnson
  surname: johnson
- explanation: Irène Joliot-Curie - French scientist who was awarded the Nobel Prize for Chemistry in 1935. Daughter of Marie and Pierre Curie.
  link: https://en.wikipedia.org/wiki/Ir%C3%A8ne_Joliot-Curie
  surname: joliot
- explanation: Karen Spärck Jones came up with the concept of inverse document frequency, which is used in most search engines today.
  link: https://en.wikipedia.org/wiki/Karen_Sp%C3%A4rck_Jones
  surname: jones
- explanation: A. P. J. Abdul Kalam - is an Indian scientist aka Missile Man of India for his work on the development of ballistic missile and launch vehicle technology -
  link: https://en.wikipedia.org/wiki/A._P._J._Abdul_Kalam
  surname: kalam
- explanation: 'Sergey Petrovich Kapitsa (Russian: Серге́й Петро́вич Капи́ца; 14 February 1928 – 14 August 2012) was a Russian physicist and demographer. He was best known as host of the popular and long-running Russian scientific TV show, Evident, but Incredible. His father was the Nobel laureate Soviet-era physicist Pyotr Kapitsa, and his brother was the geographer and Antarctic explorer Andrey Kapitsa. -'
  link: https://en.wikipedia.org/wiki/Sergey_Kapitsa
  surname: kapitsa
- explanation: Susan Kare, created the icons and many of the interface elements for the original Apple Macintosh in the 1980s, and was an original employee of NeXT, working as the Creative Director.
  link: https://en.wikipedia.org/wiki/Susan_Kare
  surname: kare
- explanation: Mstislav Keldysh - a Soviet scientist in the field of mathematics and mechanics, academician of the USSR Academy of Sciences (1946), President of the USSR Academy of Sciences (1961–1975), three times Hero of Socialist Labor (1956, 1961, 1971), fellow of the Royal Society of Edinburgh (1968).
  link: https://en.wikipedia.org/wiki/Mstislav_Keldysh
  surname: keldysh
- explanation: Mary Kenneth Keller, Sister Mary Kenneth Keller became the first American woman to earn a PhD in Computer Science in 1965.
  link: https://en.wikipedia.org/wiki/Mary_Kenneth_Keller
  surname: keller
- explanation: Johannes Kepler, German astronomer known for his three laws of planetary motion -
  link: https://en.wikipedia.org/wiki/Johannes_Kepler
  surname: kepler
- explanation: Omar Khayyam - Persian mathematician, astronomer and poet. Known for his work on the classification and solution of cubic equations, for his contribution to the understanding of Euclid's fifth postulate and for computing the length of a year very accurately.
  link: https://en.wikipedia.org/wiki/Omar_Khayyam
  surname: khayyam
- explanation: Har Gobind Khorana - Indian-American biochemist who shared the 1968 Nobel Prize for Physiology -
  link: https://en.wikipedia.org/wiki/Har_Gobind_Khorana
  surname: khorana
- explanation: Jack Kilby invented silicone integrated circuits and gave Silicon Valley its name. -
  link: https://en.wikipedia.org/wiki/Jack_Kilby
  surname: kilby
- explanation: Maria Kirch - German astronomer and first woman to discover a comet -
  link: https://en.wikipedia.org/wiki/Maria_Margarethe_Kirch
  surname: kirch
- explanation: Donald Knuth - American computer scientist, author of "The Art of Computer Programming" and creator of the TeX typesetting system.
  link: https://en.wikipedia.org/wiki/Donald_Knuth
  surname: knuth
- explanation: Sophie Kowalevski - Russian mathematician responsible for important original contributions to analysis, differential equations and mechanics -
  link: https://en.wikipedia.org/wiki/Sofia_Kovalevskaya
  surname: kowalevski
- explanation: Marie-Jeanne de Lalande - French astronomer, mathematician and cataloguer of stars -
  link: https://en.wikipedia.org/wiki/Marie-Jeanne_de_Lalande
  surname: lalande
- explanation: Hedy Lamarr - Actress and inventor. The principles of her work are now incorporated into modern Wi-Fi, CDMA and Bluetooth technology.
  link: https://en.wikipedia.org/wiki/Hedy_Lamarr
  surname: lamarr
- explanation: Leslie B. Lamport - American computer scientist. Lamport is best known for his seminal work in distributed systems and was the winner of the 2013 Turing Award.
  link: https://en.wikipedia.org/wiki/Leslie_Lamport
  surname: lamport
- explanation: Mary Leakey - British paleoanthropologist who discovered the first fossilized Proconsul skull -
  link: https://en.wikipedia.org/wiki/Mary_Leakey
  surname: leakey
- explanation: Henrietta Swan Leavitt - she was an American astronomer who discovered the relation between the luminosity and the period of Cepheid variable stars.
  link: https://en.wikipedia.org/wiki/Henrietta_Swan_Leavitt
  surname: leavitt
- explanation: Esther Miriam Zimmer Lederberg - American microbiologist and a pioneer of bacterial genetics.
  link: https://en.wikipedia.org/wiki/Esther_Lederberg
  surname: lederberg
- explanation: Inge Lehmann - Danish seismologist and geophysicist. Known for discovering in 1936 that the Earth has a solid inner core inside a molten outer core.
  link: https://en.wikipedia.org/wiki/Inge_Lehmann
  surname: lehmann
- explanation: Daniel Lewin - Mathematician, Akamai co-founder, soldier, 9/11 victim-- Developed optimization techniques for routing traffic on the internet. Died attempting to stop the 9-11 hijackers.
  link: https://en.wikipedia.org/wiki/Daniel_Lewin
  surname: lewin
- explanation: Ruth Lichterman - one of the original programmers of the ENIAC. https://en.wikipedia.org/wiki/ENIAC -
  link: https://en.wikipedia.org/wiki/Ruth_Teitelbaum
  surname: lichterman
- explanation: Barbara Liskov - co-developed the Liskov substitution principle. Liskov was also the winner of the Turing Prize in 2008. -
  link: https://en.wikipedia.org/wiki/Barbara_Liskov
  surname: liskov
- explanation: Ada Lovelace invented the first algorithm.
  link: https://en.wikipedia.org/wiki/Ada_Lovelace (thanks James Turnbull)
  surname: lovelace
- explanation: Auguste and Louis Lumière - the first filmmakers in history -
  link: https://en.wikipedia.org/wiki/Auguste_and_Louis_Lumi%C3%A8re
  surname: lumiere
- explanation: Mahavira - Ancient Indian mathematician during 9th century AD who discovered basic algebraic identities -
  link: https://en.wikipedia.org/wiki/Mah%C4%81v%C4%ABra_(mathematician)
  surname: mahavira
- explanation: Lynn Margulis (b. Lynn Petra Alexander) - an American evolutionary theorist and biologist, science author, educator, and popularizer, and was the primary modern proponent for the significance of symbiosis in evolution. -
  link: https://en.wikipedia.org/wiki/Lynn_Margulis
  surname: margulis
- explanation: Yukihiro Matsumoto - Japanese computer scientist and software programmer best known as the chief designer of the Ruby programming language.
  link: https://en.wikipedia.org/wiki/Yukihiro_Matsumoto
  surname: matsumoto
- explanation: James Clerk Maxwell - Scottish physicist, best known for his formulation of electromagnetic theory.
  link: https://en.wikipedia.org/wiki/James_Clerk_Maxwell
  surname: maxwell
- explanation: Maria Mayer - American theoretical physicist and Nobel laureate in Physics for proposing the nuclear shell model of the atomic nucleus -
  link: https://en.wikipedia.org/wiki/Maria_Mayer
  surname: mayer
- explanation: 'John McCarthy invented LISP:'
  link: https://en.wikipedia.org/wiki/John_McCarthy_(computer_scientist)
  surname: mccarthy
- explanation: Barbara McClintock - a distinguished American cytogeneticist, 1983 Nobel Laureate in Physiology or Medicine for discovering transposons.
  link: https://en.wikipedia.org/wiki/Barbara_McClintock
  surname: mcclintock
- explanation: Anne Laura Dorinthea McLaren - British developmental biologist whose work helped lead to human in-vitro fertilisation.
  link: https://en.wikipedia.org/wiki/Anne_McLaren
  surname: mclaren
- explanation: 'Malcolm McLean invented the modern shipping container:'
  link: https://en.wikipedia.org/wiki/Malcom_McLean
  surname: mclean
- explanation: Kay McNulty - one of the original programmers of the ENIAC. https://en.wikipedia.org/wiki/ENIAC -
  link: https://en.wikipedia.org/wiki/Kathleen_Antonelli
  surname: mcnulty
- explanation: Gregor Johann Mendel - Czech scientist and founder of genetics.
  link: https://en.wikipedia.org/wiki/Gregor_Mendel
  surname: mendel
- explanation: Dmitri Mendeleev - a chemist and inventor. He formulated the Periodic Law, created a farsighted version of the periodic table of elements, and used it to correct the properties of some already discovered elements and also to predict the properties of eight elements yet to be discovered.
  link: https://en.wikipedia.org/wiki/Dmitri_Mendeleev
  surname: mendeleev
- explanation: Lise Meitner - Austrian/Swedish physicist who was involved in the discovery of nuclear fission. The element meitnerium is named after her -
  link: https://en.wikipedia.org/wiki/Lise_Meitner
  surname: meitner
- explanation: Carla Meninsky, was the game designer and programmer for Atari 2600 games Dodge 'Em and Warlords.
  link: https://en.wikipedia.org/wiki/Carla_Meninsky
  surname: meninsky
- explanation: Ralph C. Merkle - American computer scientist, known for devising Merkle's puzzles - one of the very first schemes for public-key cryptography. Also, inventor of Merkle trees and co-inventor of the Merkle-Damgård construction for building collision-resistant cryptographic hash functions and the Merkle-Hellman knapsack cryptosystem.
  link: https://en.wikipedia.org/wiki/Ralph_Merkle
  surname: merkle
- explanation: Johanna Mestorf - German prehistoric archaeologist and first female museum director in Germany -
  link: https://en.wikipedia.org/wiki/Johanna_Mestorf
  surname: mestorf
- explanation: Maryam Mirzakhani - an Iranian mathematician and the first woman to win the Fields Medal.
  link: https://en.wikipedia.org/wiki/Maryam_Mirzakhani
  surname: mirzakhani
- explanation: Gordon Earle Moore - American engineer, Silicon Valley founding father, author of Moore's law.
  link: https://en.wikipedia.org/wiki/Gordon_Moore
  surname: moore
- explanation: Samuel Morse - contributed to the invention of a single-wire telegraph system based on European telegraphs and was a co-developer of the Morse code -
  link: https://en.wikipedia.org/wiki/Samuel_Morse
  surname: morse
- explanation: Ian Murdock - founder of the Debian project -
  link: https://en.wikipedia.org/wiki/Ian_Murdock
  surname: murdock
- explanation: May-Britt Moser - Nobel prize winner neuroscientist who contributed to the discovery of grid cells in the brain.
  link: https://en.wikipedia.org/wiki/May-Britt_Moser
  surname: moser
- explanation: John Napier of Merchiston - Scottish landowner known as an astronomer, mathematician and physicist. Best known for his discovery of logarithms.
  link: https://en.wikipedia.org/wiki/John_Napier
  surname: napier
- explanation: John Forbes Nash, Jr. - American mathematician who made fundamental contributions to game theory, differential geometry, and the study of partial differential equations.
  link: https://en.wikipedia.org/wiki/John_Forbes_Nash_Jr.
  surname: nash
- explanation: John von Neumann - todays computer architectures are based on the von Neumann architecture.
  link: https://en.wikipedia.org/wiki/Von_Neumann_architecture
  surname: neumann
- explanation: Isaac Newton invented classic mechanics and modern optics.
  link: https://en.wikipedia.org/wiki/Isaac_Newton
  surname: newton
- explanation: Florence Nightingale, more prominently known as a nurse, was also the first female member of the Royal Statistical Society and a pioneer in statistical graphics
  link: https://en.wikipedia.org/wiki/Florence_Nightingale#Statistics_and_sanitary_reform
  surname: nightingale
- explanation: Alfred Nobel - a Swedish chemist, engineer, innovator, and armaments manufacturer (inventor of dynamite) -
  link: https://en.wikipedia.org/wiki/Alfred_Nobel
  surname: nobel
- explanation: Emmy Noether, German mathematician. Noether's Theorem is named after her.
  link: https://en.wikipedia.org/wiki/Emmy_Noether
  surname: noether
- explanation: Poppy Northcutt. Poppy Northcutt was the first woman to work as part of NASA’s Mission Control.
  link: http://www.businessinsider.com/poppy-northcutt-helped-apollo-astronauts-2014-12?op=1
  surname: northcutt
- explanation: Robert Noyce invented silicone integrated circuits and gave Silicon Valley its name. -
  link: https://en.wikipedia.org/wiki/Robert_Noyce
  surname: noyce
- explanation: Panini - Ancient Indian linguist and grammarian from 4th century CE who worked on the world's first formal system -
  link: https://en.wikipedia.org/wiki/P%C4%81%E1%B9%87ini#Comparison_with_modern_formal_systems
  surname: panini
- explanation: Ambroise Pare invented modern surgery.
  link: https://en.wikipedia.org/wiki/Ambroise_Par%C3%A9
  surname: pare
- explanation: Blaise Pascal, French mathematician, physicist, and inventor -
  link: https://en.wikipedia.org/wiki/Blaise_Pascal
  surname: pascal
- explanation: Louis Pasteur discovered vaccination, fermentation and pasteurization.
  link: https://en.wikipedia.org/wiki/Louis_Pasteur.
  surname: pasteur
- explanation: Cecilia Payne-Gaposchkin was an astronomer and astrophysicist who, in 1925, proposed in her Ph.D. thesis an explanation for the composition of stars in terms of the relative abundances of hydrogen and helium.
  link: https://en.wikipedia.org/wiki/Cecilia_Payne-Gaposchkin
  surname: payne
- explanation: Radia Perlman is a software designer and network engineer and most famous for her invention of the spanning-tree protocol (STP).
  link: https://en.wikipedia.org/wiki/Radia_Perlman
  surname: perlman
- explanation: Rob Pike was a key contributor to Unix, Plan 9, the X graphic system, utf-8, and the Go programming language.
  link: https://en.wikipedia.org/wiki/Rob_Pike
  surname: pike
- explanation: Henri Poincaré made fundamental contributions in several fields of mathematics.
  link: https://en.wikipedia.org/wiki/Henri_Poincar%C3%A9
  surname: poincare
- explanation: Laura Poitras is a director and producer whose work, made possible by open source crypto tools, advances the causes of truth and freedom of information by reporting disclosures by whistleblowers such as Edward Snowden.
  link: https://en.wikipedia.org/wiki/Laura_Poitras
  surname: poitras
- explanation: 'Tat’yana Avenirovna Proskuriakova (Russian: Татья́на Авени́ровна Проскуряко́ва) (January 23 [O.S. January 10] 1909 – August 30, 1985) was a Russian-American Mayanist scholar and archaeologist who contributed significantly to the deciphering of Maya hieroglyphs, the writing system of the pre-Columbian Maya civilization of Mesoamerica.'
  link: https://en.wikipedia.org/wiki/Tatiana_Proskouriakoff
  surname: proskuriakova
- explanation: Claudius Ptolemy - a Greco-Egyptian writer of Alexandria, known as a mathematician, astronomer, geographer, astrologer, and poet of a single epigram in the Greek Anthology -
  link: https://en.wikipedia.org/wiki/Ptolemy
  surname: ptolemy
- explanation: C. V. Raman - Indian physicist who won the Nobel Prize in 1930 for proposing the Raman effect. -
  link: https://en.wikipedia.org/wiki/C._V._Raman
  surname: raman
- explanation: Srinivasa Ramanujan - Indian mathematician and autodidact who made extraordinary contributions to mathematical analysis, number theory, infinite series, and continued fractions. -
  link: https://en.wikipedia.org/wiki/Srinivasa_Ramanujan
  surname: ramanujan
- explanation: Sally Kristen Ride was an American physicist and astronaut. She was the first American woman in space, and the youngest American astronaut.
  link: https://en.wikipedia.org/wiki/Sally_Ride
  surname: ride
- explanation: Rita Levi-Montalcini - Won Nobel Prize in Physiology or Medicine jointly with colleague Stanley Cohen for the discovery of nerve growth factor (
  link: https://en.wikipedia.org/wiki/Rita_Levi-Montalcini)
  surname: montalcini
- explanation: Dennis Ritchie - co-creator of UNIX and the C programming language. -
  link: https://en.wikipedia.org/wiki/Dennis_Ritchie
  surname: ritchie
- explanation: Ida Rhodes - American pioneer in computer programming, designed the first computer used for Social Security.
  link: https://en.wikipedia.org/wiki/Ida_Rhodes
  surname: rhodes
- explanation: Julia Hall Bowman Robinson - American mathematician renowned for her contributions to the fields of computability theory and computational complexity theory.
  link: https://en.wikipedia.org/wiki/Julia_Robinson
  surname: robinson
- explanation: Wilhelm Conrad Röntgen - German physicist who was awarded the first Nobel Prize in Physics in 1901 for the discovery of X-rays (Röntgen rays).
  link: https://en.wikipedia.org/wiki/Wilhelm_R%C3%B6ntgen
  surname: roentgen
- explanation: Rosalind Franklin - British biophysicist and X-ray crystallographer whose research was critical to the understanding of DNA -
  link: https://en.wikipedia.org/wiki/Rosalind_Franklin
  surname: rosalind
- explanation: Vera Rubin - American astronomer who pioneered work on galaxy rotation rates.
  link: https://en.wikipedia.org/wiki/Vera_Rubin
  surname: rubin
- explanation: Meghnad Saha - Indian astrophysicist best known for his development of the Saha equation, used to describe chemical and physical conditions in stars -
  link: https://en.wikipedia.org/wiki/Meghnad_Saha
  surname: saha
- explanation: Jean E. Sammet developed FORMAC, the first widely used computer language for symbolic manipulation of mathematical formulas.
  link: https://en.wikipedia.org/wiki/Jean_E._Sammet
  surname: sammet
- explanation: Mildred Sanderson - American mathematician best known for Sanderson's theorem concerning modular invariants.
  link: https://en.wikipedia.org/wiki/Mildred_Sanderson
  surname: sanderson
- explanation: Satoshi Nakamoto is the name used by the unknown person or group of people who developed bitcoin, authored the bitcoin white paper, and created and deployed bitcoin's original reference implementation.
  link: https://en.wikipedia.org/wiki/Satoshi_Nakamoto
  surname: satoshi
- explanation: Adi Shamir - Israeli cryptographer whose numerous inventions and contributions to cryptography include the Ferge Fiat Shamir identification scheme, the Rivest Shamir Adleman (RSA) public-key cryptosystem, the Shamir's secret sharing scheme, the breaking of the Merkle-Hellman cryptosystem, the TWINKLE and TWIRL factoring devices and the discovery of differential cryptanalysis (with Eli Biham).
  link: https://en.wikipedia.org/wiki/Adi_Shamir
  surname: shamir
- explanation: Claude Shannon - The father of information theory and founder of digital circuit design theory. (
  link: https://en.wikipedia.org/wiki/Claude_Shannon)
  surname: shannon
- explanation: Carol Shaw - Originally an Atari employee, Carol Shaw is said to be the first female video game designer.
  link: https://en.wikipedia.org/wiki/Carol_Shaw_(video_game_designer)
  surname: shaw
- explanation: Dame Stephanie "Steve" Shirley - Founded a software company in 1962 employing women working from home.
  link: https://en.wikipedia.org/wiki/Steve_Shirley
  surname: shirley
- explanation: William Shockley co-invented the transistor -
  link: https://en.wikipedia.org/wiki/William_Shockley
  surname: shockley
- explanation: 'Lina Solomonovna Stern (or Shtern; Russian: Лина Соломоновна Штерн; 26 August 1878 – 7 March 1968) was a Soviet biochemist, physiologist and humanist whose medical discoveries saved thousands of lives at the fronts of World War II. She is best known for her pioneering work on blood–brain barrier, which she described as hemato-encephalic barrier in 1921.'
  link: https://en.wikipedia.org/wiki/Lina_Stern
  surname: shtern
- explanation: Françoise Barré-Sinoussi - French virologist and Nobel Prize Laureate in Physiology or Medicine; her work was fundamental in identifying HIV as the cause of AIDS.
  link: https://en.wikipedia.org/wiki/Fran%C3%A7oise_Barr%C3%A9-Sinoussi
  surname: sinoussi
- explanation: Betty Snyder - one of the original programmers of the ENIAC. https://en.wikipedia.org/wiki/ENIAC -
  link: https://en.wikipedia.org/wiki/Betty_Holberton
  surname: snyder
- explanation: Cynthia Solomon - Pioneer in the fields of artificial intelligence, computer science and educational computing. Known for creation of Logo, an educational programming language.
  link: https://en.wikipedia.org/wiki/Cynthia_Solomon
  surname: solomon
- explanation: Frances Spence - one of the original programmers of the ENIAC. https://en.wikipedia.org/wiki/ENIAC -
  link: https://en.wikipedia.org/wiki/Frances_Spence
  surname: spence
- explanation: Michael Stonebraker is a database research pioneer and architect of Ingres, Postgres, VoltDB and SciDB. Winner of 2014 ACM Turing Award.
  link: https://en.wikipedia.org/wiki/Michael_Stonebraker
  surname: stonebraker
- explanation: Ivan Edward Sutherland - American computer scientist and Internet pioneer, widely regarded as the father of computer graphics.
  link: https://en.wikipedia.org/wiki/Ivan_Sutherland
  surname: sutherland
- explanation: Janese Swanson (with others) developed the first of the Carmen Sandiego games. She went on to found Girl Tech.
  link: https://en.wikipedia.org/wiki/Janese_Swanson
  surname: swanson
- explanation: Aaron Swartz was influential in creating RSS, Markdown, Creative Commons, Reddit, and much of the internet as we know it today. He was devoted to freedom of information on the web.
  link: https://en.wikiquote.org/wiki/Aaron_Swartz
  surname: swartz
- explanation: Bertha Swirles was a theoretical physicist who made a number of contributions to early quantum theory.
  link: https://en.wikipedia.org/wiki/Bertha_Swirles
  surname: swirles
- explanation: Helen Brooke Taussig - American cardiologist and founder of the field of paediatric cardiology.
  link: https://en.wikipedia.org/wiki/Helen_B._Taussig
  surname: taussig
- explanation: Valentina Tereshkova is a Russian engineer, cosmonaut and politician. She was the first woman to fly to space in 1963. In 2013, at the age of 76, she offered to go on a one-way mission to Mars.
  link: https://en.wikipedia.org/wiki/Valentina_Tereshkova
  surname: tereshkova
- explanation: Nikola Tesla invented the AC electric system and every gadget ever used by a James Bond villain.
  link: https://en.wikipedia.org/wiki/Nikola_Tesla
  surname: tesla
- explanation: Marie Tharp - American geologist and oceanic cartographer who co-created the first scientific map of the Atlantic Ocean floor. Her work led to the acceptance of the theories of plate tectonics and continental drift.
  link: https://en.wikipedia.org/wiki/Marie_Tharp
  surname: tharp
- explanation: Ken Thompson - co-creator of UNIX and the C programming language -
  link: https://en.wikipedia.org/wiki/Ken_Thompson
  surname: thompson
- explanation: Linus Torvalds invented Linux and Git.
  link: https://en.wikipedia.org/wiki/Linus_Torvalds
  surname: torvalds
- explanation: Youyou Tu - Chinese pharmaceutical chemist and educator known for discovering artemisinin and dihydroartemisinin, used to treat malaria, which has saved millions of lives. Joint winner of the 2015 Nobel Prize in Physiology or Medicine.
  link: https://en.wikipedia.org/wiki/Tu_Youyou
  surname: tu
- explanation: Alan Turing was a founding father of computer science.
  link: https://en.wikipedia.org/wiki/Alan_Turing.
  surname: turing
- explanation: Varahamihira - Ancient Indian mathematician who discovered trigonometric formulae during 505-587 CE -
  link: https://en.wikipedia.org/wiki/Var%C4%81hamihira#Contributions
  surname: varahamihira
- explanation: Dorothy Vaughan was a NASA mathematician and computer programmer on the SCOUT launch vehicle program that put America's first satellites into space -
  link: https://en.wikipedia.org/wiki/Dorothy_Vaughan
  surname: vaughan
- explanation: Sir Mokshagundam Visvesvaraya - is a notable Indian engineer.  He is a recipient of the Indian Republic's highest honour, the Bharat Ratna, in 1955. On his birthday, 15 September is celebrated as Engineer's Day in India in his memory -
  link: https://en.wikipedia.org/wiki/Visvesvaraya
  surname: visvesvaraya
- explanation: Christiane Nüsslein-Volhard - German biologist, won Nobel Prize in Physiology or Medicine in 1995 for research on the genetic control of embryonic development.
  link: https://en.wikipedia.org/wiki/Christiane_N%C3%BCsslein-Volhard
  surname: volhard
- explanation: Cédric Villani - French mathematician, won Fields Medal, Fermat Prize and Poincaré Price for his work in differential geometry and statistical mechanics.
  link: https://en.wikipedia.org/wiki/C%C3%A9dric_Villani
  surname: villani
- explanation: Marlyn Wescoff - one of the original programmers of the ENIAC. https://en.wikipedia.org/wiki/ENIAC -
  link: https://en.wikipedia.org/wiki/Marlyn_Meltzer
  surname: wescoff
- explanation: Sylvia B. Wilbur - British computer scientist who helped develop the ARPANET, was one of the first to exchange email in the UK and a leading researcher in computer-supported collaborative work.
  link: https://en.wikipedia.org/wiki/Sylvia_Wilbur
  surname: wilbur
- explanation: Andrew Wiles - Notable British mathematician who proved the enigmatic Fermat's Last Theorem -
  link: https://en.wikipedia.org/wiki/Andrew_Wiles
  surname: wiles
- explanation: Roberta Williams, did pioneering work in graphical adventure games for personal computers, particularly the King's Quest series.
  link: https://en.wikipedia.org/wiki/Roberta_Williams
  surname: williams
- explanation: Malcolm John Williamson - British mathematician and cryptographer employed by the GCHQ. Developed in 1974 what is now known as Diffie-Hellman key exchange (Diffie and Hellman first published the scheme in 1976).
  link: https://en.wikipedia.org/wiki/Malcolm_J._Williamson
  surname: williamson
- explanation: Sophie Wilson designed the first Acorn Micro-Computer and the instruction set for ARM processors.
  link: https://en.wikipedia.org/wiki/Sophie_Wilson
  surname: wilson
- explanation: Jeannette Wing - co-developed the Liskov substitution principle. -
  link: https://en.wikipedia.org/wiki/Jeannette_Wing
  surname: wing
- explanation: Steve Wozniak invented the Apple I and Apple II.
  link: https://en.wikipedia.org/wiki/Steve_Wozniak
  surname: wozniak
- explanation: The Wright brothers, Orville and Wilbur - credited with inventing and building the world's first successful airplane and making the first controlled, powered and sustained heavier-than-air human flight -
  link: https://en.wikipedia.org/wiki/Wright_brothers
  surname: wright
- explanation: Chien-Shiung Wu - Chinese-American experimental physicist who made significant contributions to nuclear physics.
  link: https://en.wikipedia.org/wiki/Chien-Shiung_Wu
  surname: wu
- explanation: Rosalyn Sussman Yalow - Rosalyn Sussman Yalow was an American medical physicist, and a co-winner of the 1977 Nobel Prize in Physiology or Medicine for development of the radioimmunoassay technique.
  link: https://en.wikipedia.org/wiki/Rosalyn_Sussman_Yalow
  surname: yalow
- explanation: Ada Yonath - an Israeli crystallographer, the first woman from the Middle East to win a Nobel prize in the sciences.
  link: https://en.wikipedia.org/wiki/Ada_Yonath
  surname: yonath
- explanation: 'Nikolay Yegorovich Zhukovsky (Russian: Никола́й Его́рович Жуко́вский, January 17 1847 – March 17, 1921) was a Russian scientist, mathematician and engineer, and a founding father of modern aero- and hydrodynamics. Whereas contemporary scientists scoffed at the idea of human flight, Zhukovsky was the first to undertake the study of airflow. He is often called the Father of Russian Aviation.'
  link: https://en.wikipedia.org/wiki/Nikolay_Yegorovich_Zhukovsky
  surname: zhukovsky
