require "date"
require "dpklib/collection"
require "dpklib/dpkstd"
require "dpklib/timeutils"
require "parsedate"


module Dpklib
  HourParseError = Dpklib.new_error_class(TimeParseError) { |str|
    "Cannot parse into hour: #{str}"
  }
  
  class Hour
    include Comparable

    attr_accessor :sec_count
    def initialize(*args)
      @sec_count = 0
      self.set_hms *args
    end
    
    def hour
      @sec_count / 3600
    end

    def min
      (@sec_count % 3600) / 60
    end

    def sec
      (@sec_count % 60)
    end
    
    def hour=(h)
      @sec_count = (h * 3600) + (@sec_count % 3600)
    end

    def min=(m)
      @sec_count = (self.hour * 3600) + (m * 60) + (@sec_count % 60)
    end

    def sec=(s)
      @sec_count = (self.hour * 3600) + (self.min * 60) + s
    end

    def set_hms(h = 0,m = 0,s = 0)
      @sec_count = h * 3600 + m * 60 + s
    end

    def set_time(time)
      self.set_hms time.hour, time.min, time.sec
    end

    def ==(other)
      case other
      when Hour
	@sec_count == other.sec_count
      else
	false
      end
    end
    alias === ==

    def <=>(other)
      @sec_count <=> other.sec_count
    end

    def +(other)
      ret = self.clone
      ret.sec_count += other.sec_count
      ret
    end

    def hash
      @sec_count.hash
    end

    def to_s
      sprintf("%02d:%02d:%02d", hour, min, sec)
    end

    def to_time(date = Date.new)
      Dpklib.date_to_time(date) + sec_count
    end

    def inspect
      "#{super}[#{to_s}]"
    end
  end #/Hour

  class HourEnumerator
    include Enumerable
    
    attr_reader :range, :step
    def initialize
      @range = @step = nil
    end

    def range=(range)
      if range.first > range.last then
	raise ArgumentError, "sorry, decremental step is not supported now."
      end
      @range = range
    end

    def step=(step)
      if step.sec_count == 0 then
	raise ArgumentError, "not incremental step."
      end
      @step = step
    end

    def each(&block)
      range = self.range
      step_sec = self.step.sec_count
      
      hour = range.first.clone
      while (range === hour) do
	yield hour.clone
	hour.sec_count += step_sec
      end
    end

  end #/HourEnumerator
  
  class HourParser
    attr_accessor :text
    def perform
      hour = Hour.new
      @text.empty? && raise_error
      tokens = @text.split(%r"[:.\-]", 3)
      
      case tokens.size
      when 1
        str = tokens[0]
        hour.hour = parse_token(str[0...2])
        hour.min = parse_token(str[2...4])
        hour.sec = parse_token(str[4...6])
        hour
      else
        hour.hour = parse_token(tokens[0])
        hour.min = parse_token(tokens[1])
        hour.sec = parse_token(tokens[2])
        hour
      end
    end

    protected
    def parse_token(token)
      if token && !(token.empty?) then
        Dpklib.decimal_integer(token) rescue raise_error
      else
        0
      end
    end

    def raise_error
      raise HourParseError, @text
    end
  end #/HourParser

  class << Hour
    alias [] new
    def from_time(time)
      hour = Hour.new
      hour.set_time time
      hour
    end

    def step(range, step_hour, step_min = 0, step_sec = 0, &block)
      enum = HourEnumerator.new
      enum.range = range
      enum.step = Hour.new(step_hour, step_min, step_sec)

      if block_given? then
	enum.each &block
      else
	enum
      end
    end

    def parse(string)
      parser = HourParser.new
      parser.text = string
      parser.perform
    end
  end #/<< Hour

  # parse_hour is different from Hour.parse
  # at a point of using parsedate.
  def self.parse_hour(string)
    hourargs = ParseDate.parsedate(string)[3..5]
    hourargs.include?(nil) && Dpklib::HourParseError.raise(string)
    Hour[*hourargs]
  end
end #/Dpklib

