
module Dpklib
  class ResourceNotFoundError < StandardError
    def initialize(tried, loader)
      msg = "'#{tried}' is not found in resource path: #{loader.load_path.inspect}."
      super(msg)
    end
  end #/ResourceNotFoundError

  module ResourceFindable
    def load_path
      raise(ScriptError, "ResourceFindable must be with load_path defined.")
    end

    def find(path)
      found_path = nil
      load_path.each { |loaddir|
        fullpath = File.expand_path(path, loaddir)
        if File.exist?(fullpath) then
          found_path = fullpath
          break
        end
      }
      found_path
    end

    def find_surely(path)
      find(path) || raise(ResourceNotFoundError.new(path, self))
    end
    alias [] find_surely

    def load(path)
      resource_load do
	Kernel.load(path)
      end
    end

    def require(path)
      resource_load do
	Kernel.require(path)
      end
    end

    def resource_load(&block)
      thread = Thread.start {
	load_path.reverse_each do |path|
	  $LOAD_PATH.unshift path
	end
	yield
      }
      thread.join
    end

    def open(path, *openargs, &block)
      path = find(path)
      File.open(path, *openargs, &block)
    end

    def read(path)
      string = nil
      open(path) { |input|
        string = input.read
      }
      string
    end
  end #/ResourceFindable

  class ResourceLoader
    include ResourceFindable
    attr_reader :load_path

    def initialize(load_path = nil)
      @load_path = load_path || []
    end

    def <<(path_or_pathes)
      case path_or_pathes
      when String; @load_path.unshift(path_or_pathes)
      when Array; @load_path.replace(path_or_pathes + @load_path)
      else; fail_unacceptable
      end
      self
    end

    def >>(path_or_pathes)
      case path_or_pathes
      when String; @load_path << path_or_pathes
      when Array; @load_path.concat(path_or_pathes)
      else; fail_unacceptable
      end
      self
    end

    private
    def fail_unacceptable(val)
      raise(TypeError, "not acceptable value: #{val}")
    end
  end #/ResourceLoader

  class << self
    def parse_env_to_pathes(envname)
      loadpath = ENV[envname]
      loadpath ? loadpath.split(":") : []
    end
  end #/<< self

  DEFAULT_RESOURCE_PATH_ENVNAME = "RESOURCE_PATH"

  RESOURCE = begin
    pathes = Dpklib.parse_env_to_pathes(DEFAULT_RESOURCE_PATH_ENVNAME)
    ResourceLoader.new(pathes)
  end
end #/Dpklib
