require 'digest/md5'

module Dpklib
  class ShadowPassword
    DEFAULT_DIGEST_CLASS = Digest::MD5
  end #/ShadowPassword
  
  class ShadowPasswordEntry < Struct.new(:username, :shadow)
    def initialize(digest_class = nil)
      @digest_class = digest_class || ShadowPassword::DEFAULT_DIGEST_CLASS
      super()
    end
    
    def verify(username, password)
      validated = true
      validated &&= (username == self.username)
      validated &&= (@digest_class.new(password) == self.shadow)
      validated
    end
    
    def password=(password)
      self.shadow = @digest_class.hexdigest(password)
    end
    
    def to_s
      "#{username}:#{shadow}"
    end
    
    ParseError = Class.new(StandardError)
    def self.parse(string, digest_class = nil)
      entry = new(digest_class)
      entry.username, entry.shadow = string.split(":", 2)
      entry.username || raise(ParseError)
      entry
    end
    
    def self.create(username, password, digest_class = nil)
      entry = new(digest_class)
      entry.username = username
      entry.password = password
      entry
    end
  end #/ShadowPasswordEntry

  class ShadowPasswordManager
    attr_accessor :digest_class
    def initialize
      @digest_class = nil
      @entries_hash = {}
    end

    def verify(username, password)
      entry = @entries_hash[username]
      entry ? entry.verify(username, password) : false
    end
    
    def clear
      @entries_hash.clear
    end
    
    def add(entry)
      @entries_hash[entry.username] = entry
    end
    alias << add
    
    def add_line(shadow_line)
      entry = ShadowPasswordEntry.parse(shadow_line, @digest_class)
      add(entry)
    end
    
    def add_text(shadow_text)
      shadow_text.each_line do |shadow_line|
        shadow_line = shadow_line.strip
        shadow_line.empty? || add_line(shadow_line)
      end
    end

    def delete(username)
      @entries_hash.delete(username)
    end
    
    def add_from_file(shadow_file)
      File.open(shadow_file) do |input|
        add_text(input.read)
      end
    end

    def to_text
      entries = []
      @entries_hash.each { |username, entry|
        entries << entry.to_s
      }
      entries.join("\n")
    end
  end #/ShadowPasswordManager
end #/Dpklib
