/* cache.c -- pam_smb cache management code 
   
   Copyright (c) Dave Airlie 2000
   airlied@samba.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   */

#include "config.h"

#include <stdlib.h>
#include <fcntl.h>
#include <syslog.h>
#include <errno.h>
#include <stdio.h>
#include "constants.h"
#include "tdb.h"
#include "cache.h"
#include "pamsmbd.h"

/*
 * Initalise the cache database
 */
int serv_initcachedb()
{
  int type, flags, mode;

  cache_db = tdb_open(cache_db_file, 0, TDB_CLEAR_IF_FIRST, O_RDWR | O_CREAT | O_TRUNC, 0600);

  if (cache_db==NULL)
    {
      perror("tdb_open failed");
      exit(-1);
    }
}

/*
 * Add an entry to the cache
 */
int add_entry_to_cache(usercache_data *entry)
{
  int res;
  TDB_DATA key;
  TDB_DATA value;
  time_t curtime;

  key.dptr = entry->ux_user;
  key.dsize = strlen(entry->ux_user)+1;

  curtime=time(NULL);
  entry->usertimeout=curtime;

  value.dptr = (char *)entry;
  value.dsize = sizeof(usercache_data);
  
  tdb_writelock(cache_db);
  res=tdb_store(cache_db, key, value, TDB_REPLACE);
  tdb_writeunlock(cache_db);
  if (res==0)
    {
      cur_cache_size++;
      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_add: inserted entry\n");
      return 0;
    }

  syslog(LOG_AUTHPRIV | LOG_ERR, "cache_add: error on db insert\n");
  perror("Error on tdb_store\n");
  sigterm_handler(0);
}

    
/*
 * Check if an entry is in the cache
 */
int check_entry_in_cache(usercache_data *entry)
{
  int res;
  TDB_DATA key;
  TDB_DATA value;
  usercache_data *check;

  key.dptr = entry->ux_user;
  key.dsize = strlen(entry->ux_user)+1;

  value=tdb_fetch(cache_db, key);
  if (value.dptr!=NULL)
    {
      /* Check if password is correct */
      check=(usercache_data *)value.dptr;
      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: found entry checking passwords");
   
      if (!(strcmp(entry->password, check->password)))
	{
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: found entry\n");
	  free(value.dptr);
	  return CACHE_OK;
	}
      else
	{
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: password wrong\n");
	  free(value.dptr);
	  return CACHE_ENOENT;
	}
    }
  else
    {
      if (debug)   syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_check: no entry\n");
      return CACHE_ENOENT;
    }
  syslog(LOG_AUTHPRIV | LOG_ERR, "cache_check: error on db fetch\n");

  return CACHE_ENOENT;  
}


/*
 * Handler for SIGALRM signal, does cache cleanout
 */
void sigalrm_handler(int signum)
{

  /* do cache management */
  int res, finished=0;
  TDB_DATA key, nextkey;
  TDB_DATA value;
  usercache_data *cacheent;
  time_t curtime;

  syslog(LOG_AUTHPRIV | LOG_NOTICE, "cache_clean: initialised\n");

  curtime=time(NULL);
  
  key=tdb_firstkey(cache_db);
  if (!key.dptr) return;
  while (!finished)
    {
      value=tdb_fetch(cache_db,key);
      if (value.dptr) 
	{
	  cacheent=(usercache_data *)value.dptr;

	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_clean: checking entry %s tleft: %d\n", cacheent->ux_user, curtime-cacheent->usertimeout);

	  if (curtime-cacheent->usertimeout > CACHE_TIMEOUT)
	    {
	      res=tdb_delete(cache_db, key);
	      if (res==0)
		{
		  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache_clean: purged %s\n", key.dptr);
		  cur_cache_size--;
		}
	      else
		if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache: error purging entry");
	    }
	 
	}
      nextkey=tdb_nextkey(cache_db,key);
      free(key.dptr);
      free(value.dptr);
      key=nextkey;
      if (key.dptr==NULL)
	finished=1;
    }
  
  return;
}
