/* map_user.c -- User Mapping code
  
   Copyright (c) Dave Airlie 2000
   airlied@samba.org

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include "config.h"
#include "constants.h"

#include <syslog.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>

#include "cache.h"
#include "tdb.h"
#include "pamsmbd.h"

/* unix2nt is keyed on unix username and returns NT domain/username pair 
   nt2unix is keyed on DOMAIN\username and returns unix username */
TDB_CONTEXT *unix2nt_map_db;
TDB_CONTEXT *nt2unix_map_db;

/*
 * Initalise the cache database
 */
int map_user_initdb()
{
  int type, flags, mode;

  unix2nt_map_db = tdb_open(unix2nt_db_file, 0, TDB_CLEAR_IF_FIRST, O_RDWR | O_CREAT | O_TRUNC, 0600);
  if (unix2nt_map_db==NULL)
    {
      perror("tdb_open failed");
      exit(-1);
    }
  
  nt2unix_map_db = tdb_open(nt2unix_db_file, 0, TDB_CLEAR_IF_FIRST, O_RDWR | O_CREAT | O_TRUNC, 0600);
  if (nt2unix_map_db==NULL)
    {
      perror("tdb_open failed");
      exit(-1);
    }
}

int map_user_read_map_file(char *map_file)
{
  FILE *fl;
  char buf[255], *p1, *p2, *p3;
  char nt_user[MAX_NT_LEN], nt_domain[MAX_NTDOM_LEN], ux_user[MAX_UU_LEN];
  
  fl=fopen(map_file, "r");
  if (fl==NULL)
    return -1;
  
  while (!feof(fl) && fgets(buf,255, fl))
    {
      p1=strchr(buf, '=');
      if (p1!=NULL) 
	{
	  memcpy(ux_user, buf, p1-buf);
	  ux_user[p1-buf]='\0';
	  
	  p2=strchr(buf, '\\');
	  
	  if (p2==NULL)
	    {
	      strcpy(nt_user, p1+1);
	      if (nt_user[strlen(nt_user)-1]==10)
		nt_user[strlen(nt_user)-1]='\0';
	      strcpy(nt_domain, defaultdomain);
	    }
	  else
	    {
	      memcpy(nt_domain, p1+1, p2-p1-1);
	      nt_domain[p2-p1-1]='\0';
	      strcpy(nt_user, p2+1);
	      if (nt_user[strlen(nt_user)-1]==10)
		nt_user[strlen(nt_user)-1]='\0';
	    }
	  map_user_add_entry(ux_user, nt_user, nt_domain);
	  
	}
    } 
  fclose(fl);
}

int map_user_add_entry(char *unix_user, char *nt_user, char *nt_domain)
{
  TDB_DATA key;
  TDB_DATA value;
  int res;

  char nt_full[MAX_LOGIN_LEN];
  char ux_user[MAX_UU_LEN];

  strcpy(ux_user, unix_user);

  strcpy(nt_full, nt_domain);
  strcat(nt_full, "\\");
  strcat(nt_full, nt_user);

  key.dptr = ux_user;
  key.dsize = strlen(ux_user)+1;

  value.dptr = nt_full;
  value.dsize = strlen(nt_full)+1;
  
  tdb_writelock(unix2nt_map_db);
  res=tdb_store(unix2nt_map_db, key, value, TDB_REPLACE);
  tdb_writeunlock(unix2nt_map_db);

  if (res!=0)
    {
      fprintf(stderr, "map_add: error in unix2nt\n");
      exit(-1);
    }

  key.dptr = nt_full;
  key.dsize = strlen(nt_full)+1;

  value.dptr = ux_user;
  value.dsize = strlen(ux_user)+1;

  tdb_writelock(nt2unix_map_db);
  res=tdb_store(nt2unix_map_db, key, value, TDB_REPLACE);
  tdb_writeunlock(nt2unix_map_db);

  if (res!=0)
    {
      fprintf(stderr, "map_add: error in nt2unix\n");
      exit(-1);
    }
}

int map_user(usercache_data *user)
{
  int res;
  char nt_user[MAX_NT_LEN];
  char ux_user[MAX_UU_LEN];
  char nt_domain[MAX_NTDOM_LEN];
  char nt_full[MAX_LOGIN_LEN];
  char buf[255], *bufptr;
  int domlen, namelen, mapped=0;
  char *lc, *slash;
  TDB_DATA key;
  TDB_DATA value;

  /* Initialise the variables for db search */
  strcpy(nt_user, user->nt_user);
  /* Convert to lowercase */
  lc=nt_user;
  while (*lc != '\0') {
    *lc=tolower(*lc);
    lc++;
  }
  lc++; *lc=0;

  strcpy(ux_user, nt_user);

  strcpy(nt_domain, user->nt_domain);
  
  strcpy(nt_full, nt_domain);
  strcat(nt_full, "\\");
  strcat(nt_full, nt_user);

  /* Search NT->Unix database first for long->short conversion */

  key.dptr=nt_full;
  key.dsize=strlen(nt_full)+1;
  
  value=tdb_fetch(nt2unix_map_db, key);
  if (value.dptr!=NULL)
    {
      /* If found replace name for pam with unix short name */
      strcpy(buf, value.dptr);
      free(value.dptr);
      strcpy(user->ux_user, buf);
      mapped=1;
    }
  else
    {
      /* Search Unix->NT database for short->long conversion */
      key.dptr=ux_user;
      key.dsize=strlen(ux_user)+1;
      value=tdb_fetch(unix2nt_map_db, key);
      
      if (value.dptr!=NULL)
	{
	  slash=strchr(value.dptr, '\\');
	  domlen=slash-value.dptr;
	  strncpy(user->nt_domain, value.dptr, domlen);
	  user->nt_domain[domlen]='\0';
	  
	  namelen=strlen(value.dptr)-domlen-1;
	  strncpy(user->nt_user, slash+1, namelen);
	  user->nt_user[namelen]='\0';
	  
	  free(value.dptr);
	  mapped=1;
	}
    }
  if (mapped)
    if (debug)
      syslog(LOG_AUTHPRIV | LOG_DEBUG, "Mapped unix: %s, nt %s\\%s\n", user->ux_user, user->nt_domain, user->nt_user);
  return 0;
}

int Valid_MapUser(usercache_data *user)
{
  return(Valid_User(user->nt_user, user->password, user->pdc_nbname, user->bdc_nbname, user->nt_domain));
}
