/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* main.c -- main source file */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>
#include <assert.h>
#ifdef DEBUG
# include <signal.h>
#endif

#include "square.h"
#include "parse.h"
#include "write.h"

#ifdef DEBUG
# include "debug.h"
# include "sig2str.h"
#endif

#define usage() printf(usage_message, argv[0])
#define fusage(file) fprintf(file, usage_message, argv[0])

enum format {unknown, text, binary};

int square_main(int, char **);
int squarec_main(int, char **);

/* Static helper functions */
static enum format parse_format(const char *);
static enum format get_format(FILE *, const char *);

const char version_message[] =
  PACKAGE_STRING " "
  "with"
#ifndef HAVE_PTHREAD
  "out"
#endif
  " multi-threading\n"
  "Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>\n"
  "\n"
  PACKAGE_NAME " comes with ABSOLUTELY NO WARRANTY.\n"
  "You may redistribute copies of " PACKAGE_NAME "\n"
  "under the terms of the GNU General Public License\n"
  "version 3, or (at your option) any later version.\n"
  "For more information about these matters, see the file named COPYING.";

/* Call the correct main() function based on the name we were run as */
int main(int argc, char **argv) {
  char *progname, *bname;
  enum {SQUARE, SQUAREC} mode;

#ifdef DEBUG
  struct sigaction action;

  action.sa_handler = debug_abort;
  sigemptyset(&(action.sa_mask));
  action.sa_flags = SA_NODEFER; /* Since we will block all debug
				   signals (except SIGABRT) anyway and
				   we DON'T want to block SIGABRT (so
				   that we have a way to dump
				   core). */

  /* Add all the debug signals (except SIGABRT) to the signal mask */
  for (size_t i = 0; i < debug_signals_entries; i++) {
    if (debug_signals[i] != SIGABRT &&
	sigaddset(&(action.sa_mask), debug_signals[i])) {
      int errno_save = errno;
      char *signame = sig2str(debug_signals[i]);

      if (signame) {
	fprintf(stderr, "%s: cannot add SIG%s (%d) to the signal mask: %s\n",
		argv[0], signame, debug_signals[i], strerror(errno_save));
      }
      else {
	fprintf(stderr,
		"%s: cannot add unknown signal (%d) to the signal mask: %s\n",
		argv[0], debug_signals[i], strerror(errno_save));
      }

      return -1;
    }
  }

  /* Trap signals which indicate bugs */
  for (size_t i = 0; i < debug_signals_entries; i++) {
    if (sigaction(debug_signals[i], &action, NULL)) {
      int errno_save = errno;
      char *signame = sig2str(debug_signals[i]);

      if (signame) {
	fprintf(stderr, "%s: cannot trap SIG%s (%d): %s\n",
		argv[0], signame, debug_signals[i], strerror(errno_save));
      }
      else {
	fprintf(stderr, "%s: cannot trap unknown signal (%d): %s\n",
		argv[0], debug_signals[i], strerror(errno_save));
      }

      return -1;
    }
  }
#endif /* !DEBUG */

  /* Get the mode */
  progname = strdup(argv[0]); /* Make a copy since basename(3) can
				 modify its argument */
  bname = basename(progname);
  mode = strcasecmp(bname, "squarec") ? SQUARE : SQUAREC;

  free(progname);

  return (mode == SQUARE) ?
    square_main(argc, argv) :
    (mode == SQUAREC) ?
    squarec_main(argc, argv) :
    -1;
}

int square_main(int argc, char **argv) {
  square_t square;
  FILE *file;
  char *filename;
  char keep_going = 0, skip_trivial = 1;
  enum format format = unknown;
  int ret;
#ifdef HAVE_PTHREAD
  unsigned long max_threads;
  char max_threads_set = 0;
#endif
  const char usage_message[] =
    "Usage: %s [OPTIONS] [FILE]\n"
    "Solve FILE as a magic square.\n"
    "\n"
    "With no FILE, or when FILE is -, read standard input.\n"
    "Default format (see -f) is \"text\" if extension is not `.bin' or FILE (stdin if\n"
    "not given) is a tty, \"binary\" otherwise.\n"
    "\n"
    "  -k, --keep-going      keep going after the first solution\n"
    "  -a, --all             print trivial differences (rotations/reflections);\n"
    "                          implies --keep-going\n"
    "  -f, --format=FORMAT   parse FILE as FORMAT; FORMAT can be \"text\" or \"binary\"\n"
#ifdef HAVE_PTHREAD
    "  -j, --jobs=THREADS    use at most THREADS threads when solving (0 is the same\n"
    "                          as 1); default is optimum number for your machine\n"
    "  --threads=THREADS     synonym for --jobs\n"
#endif
    "  -h, --help            display this help and exit\n"
    "  -v, --version         print version information and exit\n";

  /* Parse the options */
  while (1) {
    int c;
    static struct option long_options[] = {
      {"keep-going", no_argument,       NULL, 'k'},
      {"all",        no_argument,       NULL, 'a'},
      {"format",     required_argument, NULL, 'f'},
      {"jobs",       required_argument, NULL, 'j'},
      {"threads",    required_argument, NULL, 'j'},
      {"help",       no_argument,       NULL, 'h'},
      {"version",    no_argument,       NULL, 'v'},
      {NULL,         0,                 NULL,  0 }
    };

    if ((c = getopt_long(argc, argv, "kaf:j:hv", long_options, NULL)) == -1)
      break;

    switch (c) {
#ifdef HAVE_PTHREAD
      char *endptr;
#endif

    case 'k':
      keep_going = 1;
      break;
    case 'a':
      skip_trivial = 0;

      /* `--all' implies `--keep-going' since it would have no effect
           otherwise. */
      keep_going = 1;

      break;
    case 'f':
      if ((format = parse_format(optarg)) == unknown) {
	fprintf(stderr, "%s: invalid format \"%s\"\n", argv[0], optarg);
	return 1;
      }

      break;
    case 'j':
#ifdef HAVE_PTHREAD
      errno = 0;
      max_threads = strtoul(optarg, &endptr, 0);

      if (errno) {
	fprintf(stderr, "%s: too many threads (%s)\n", argv[0], optarg);
	return 1;
      }

      if (*endptr) {
	fprintf(stderr, "%s: %s: not an integer\n", argv[0], optarg);
	return 1;
      }

      /* Don't forget that we are a thread too! */
      if (max_threads)
	max_threads--;

      max_threads_set = 1;
#else /* !HAVE_PTHREAD */
      fprintf(stderr, "%s: no thread support\n", argv[0]);
      return 3;
#endif

      break;
    case 'h':
      usage();
      return 0;
    case 'v':
      puts(version_message);
      return 0;
    case '?':
      fusage(stderr);
      return 1;
    default:
      fprintf(stderr,
	      "%1$s: error: getopt_long(3) returned `%2$c' (ASCII %2$d)\n",
	      argv[0], c);

      return -1;
    }
  }

  /* Shift the args to get rid of the options. */
  argv[optind - 1] = argv[0];
  argv += optind - 1;
  argc -= optind - 1;

  if (argc > 2) {
    fusage(stderr);
    return 1;
  }

#ifdef HAVE_PTHREAD
  if (!max_threads_set) {
    /* Determine optimum number of threads for this machine. */
    long opt_threads = sysconf(_SC_NPROCESSORS_ONLN);

    if (opt_threads < 0) {
      fprintf(stderr,
	      "%s: unable to determine optimum number of threads: %m "
	      "(hint: try -j THREADS)\n", argv[0]);

      return -1;
    }

    if (!opt_threads) {
      fprintf(stderr, "%s: you appear to have no available CPU(s)!\n",
	      argv[0]);

      return -1;
    }

    /* Remember, this is the number of *new* threads to start. We are
       a thread. */
    max_threads = opt_threads - 1;
  }
#endif

  if (argc > 1 && strcmp(argv[1], "-")) {
    filename = argv[1];

    if (!(file = fopen(filename, (format == text) ? "r" : "rb"))) {
      fprintf(stderr, "%s: cannot open %s: %m\n", argv[0], filename);
      return 2;
    }
  }
  else {
    file = stdin;
    filename = "<stdin>";
  }

  if (format == unknown)
    format = get_format(file, filename);

  /* `format' should not be `unknown'. */
  assert(format != unknown);

  if ((format == text) ?
      parse_human(&square, file) :
      parse_machine(&square, file)) {
    fprintf(stderr, "%s: could not parse %s: %m\n", argv[0], filename);
    return 1;
  }

  if (fclose(file)) {
    fprintf(stderr, "%s: cannot close %s: %m\n", argv[0], filename);
    return 2;
  }

  /* Solve the square */
  ret = square_solve(&square, NULL, NULL, keep_going, skip_trivial, stdout
#ifdef HAVE_PTHREAD
		     , max_threads
#endif
		     );

  /* We're done with this */
  square_destroy(&square);

  if (ret) {
    fprintf(stderr, "%s: unable to solve square: %m\n", argv[0]);
    return 1;
  }

  return 0;
}

int squarec_main(int argc, char **argv) {
  square_t square;
  FILE *infile = NULL, *outfile = NULL;
  char *infilename, *outfilename;
  enum format informat = unknown, outformat = unknown;
  const char usage_message[] =
    "Usage: %s [-o OUTFILE [-f FORMAT]] [INFILE] [-f FORMAT]\n"
    "Read INFILE and output to OUTFILE, possibly converting between formats.\n"
    "\n"
    "With no INFILE, or when INFILE is -, read standard input.\n"
    "Default format (see -f) is \"text\" if extension is not `.bin' or INFILE/OUTFILE\n"
    "(stdin/stdout if not given) is a tty, \"binary\" otherwise.\n"
    "\n"
    "  -f, --format=FORMAT    parse FILE as FORMAT; FORMAT can be \"text\" or \"binary\";\n"
    "                           applies to INFILE if given after INFILE, or to\n"
    "                           OUTFILE if given after OUTFILE (see -o)\n"
    "  -o, --output=OUTFILE   write output to OUTFILE\n"
    "  -h, --help             display this help and exit\n"
    "  -v, --version          print version information and exit\n";

  /* Parse the options (twice, once for the options before INFILE,
     once after). */
  for (char i = 0; i < 1; i++) {
    /* The last file which was specified on the command line */
    static enum {NONE, INFILE, OUTFILE} last_file = NONE;

    while (1) {
      int c;
      static struct option long_options[] = {
	{"format",  required_argument, NULL, 'f'},
	{"output",  required_argument, NULL, 'o'},
	{"help",    no_argument,       NULL, 'h'},
	{"version", no_argument,       NULL, 'v'},
	{NULL,     0,                 NULL,  0 }
      };

      if ((c = getopt_long(argc, argv, "+f:o:hv", long_options, NULL)) == -1)
	break;

      switch (c) {
	enum format format;

      case 'f':
	if ((format = parse_format(optarg)) == unknown) {
	  fprintf(stderr, "%s: invalid format \"%s\"\n", argv[0], optarg);
	  return 1;
	}

	if (last_file == OUTFILE) {
	  outformat = format;
	}
	else
	  /* Assume INFILE is stdin for now if `last_file' is NONE */
	  informat = format;

	break;
      case 'o':
	outfilename = optarg;

	if (!strcmp(outfilename, "-")) {
	  outfile = stdout;
	  outfilename = "<stdout>";
	}
	else {
	  /* Open the output file */
	  if (!(outfile = fopen(outfilename, "w"))) {
	    fprintf(stderr, "%s: cannot open %s: %m\n", argv[0], outfilename);
	    return 2;
	  }
	}

	last_file = OUTFILE;

	break;
      case 'h':
	usage();
	return 0;
      case 'v':
	puts(version_message);
	return 0;
      case '?':
	fusage(stderr);
	return 1;
      default:
	fprintf(stderr,
		"%1$s: error: getopt_long(3) returned `%2$c' (%2$d)\n",
		argv[0], c);

	return -1;
      }
    }

    /* Shift the args to get rid of the options. */
    argv[optind - 1] = argv[0];
    argv += optind - 1;
    argc -= optind - 1;

    if (!i && argc > 1) {
      /* We have at least one argument, the first of which is
	 INFILE. */
      if (argc < 2 || !strcmp(argv[1], "-")) {
	infile = stdin;
	infilename = "<stdin>";
      }
      else {
	/* Make sure -f wasn't given before INFILE and OUTFILE */
	if (informat != unknown) {
	  fprintf(stderr, "%s: -f must be given after INFILE\n", argv[0]);
	  fusage(stderr);

	  return 1;
	}

	infilename = argv[1];

	/* Open the input file (we can't know what mode to use ("r" or
	   "rb") until we get the input format, and it wouldn't really be
	   that hard to just wait until we know the format to open the
	   file, but it's good if it's buggy on Windoze). */
	if (!(infile = fopen(infilename, "r"))) {
	  fprintf(stderr, "%s: cannot open %s: %m\n", argv[0], infilename);
	  return 2;
	}
      }

      last_file = INFILE;

      /* Shift the argument */
      argv[1] = argv[0];
      argv++;
      argc--;
    }
    else {
      /* INFILE was never given */
      infile = stdin;
      infilename = "<stdin>";

      break;
    }
  }

  /* Make sure there are no more arguments */
  if (argc > 1) {
    fusage(stderr);
    return 1;
  }

  if (!outfile) {
    /* Default to stdout */
    outfile = stdout;
    outfilename = "<stdout>";
  }

  /* Get the formats if they weren't specified explicitly */
  if (informat == unknown)
    informat = get_format(infile, infilename);

  if (outformat == unknown)
    outformat = get_format(outfile, outfilename);

  /* Now parse the square */
  if (((informat == text) ?
       parse_human :
       parse_machine)(&square, infile)) {
    fprintf(stderr, "%s: could not parse %s: %m\n", argv[0], infilename);
    return 3;
  }

  if (fclose(infile)) {
    fprintf(stderr, "%s: unable to close %s: %m\n", argv[0], infilename);
    return 2;
  }

  /* And write the output file */
  if (!((outformat == text) ?
	write_human :
	write_machine)(&square, outfile)) {
    fprintf(stderr, "%s: could not write %s: %m\n", argv[0], outfilename);
    return 3;
  }

  if (fclose(outfile)) {
    fprintf(stderr, "%s: could not write %s: %m\n", argv[0], outfilename);
    return 3;
  }

  square_destroy(&square);
  return 0;
}

/***************************\
|* Static helper functions *|
\***************************/

/* Parse the format from a string such as "binary" or "text",
   returning the format or `unknown' if the string is not a valid
   format. */
static enum format parse_format(const char *str) {
  size_t str_size = strlen(str);

  if (!strncasecmp(str, "text", str_size))
    return text;

  if (!strncasecmp(str, "binary", str_size))
    return binary;

  return unknown;
}

/* Get the format based on the extension of the file and whether it is
   a tty. On error, exit()'s rather than returns. If `file == stdin'
   (or stdout if it's the output file). */
static enum format get_format(FILE *file, const char *name) {
  int fd = fileno(file);
  int flags = fcntl(fd, F_GETFL);
  char is_stdio =
    (flags & O_WRONLY) ? (fd == STDOUT_FILENO) :
    (flags & O_RDWR) ? (fd == STDIN_FILENO || fd == STDOUT_FILENO) :
    /* O_RDONLY */ (fd == STDIN_FILENO);
  char *extension = (is_stdio) ? NULL : strrchr(name, '.');

  if (fd == -1) {
    /* This shouldn't happen because `file' should be valid. */
    fprintf(stderr, "%s: cannot get file descriptor: %m\n",
	    program_invocation_name);
    exit(-1);
  }

  /* Get the format based on whether the file is a tty. */
  if (isatty(fd)) {
    /* Yes, I know. I should re open the file as mode "r" instead of
       "rb". But screw those M$ Windoze lusers! (You're not one, are
       you?) */

    return text;
  }

  if (errno != ENOTTY) {
    fprintf(stderr, "%s: unable to determine if %s is a tty: %m\n",
	    program_invocation_name, name);

    exit(2);
  }

  /* It's not a tty, so get the format based on its extension if it's
     not stdin or stdout (depending on the flags). */
  if (is_stdio || (extension && !strcmp(extension, ".bin")))
    return binary;

  /* If the extension is not `.bin' or there is no extension and the
     file is not stdin (in which case it should be binary if it's not
     a tty), default to text. */
    return text;
}
