/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
/*
 * Created by IntelliJ IDEA.
 * User: rpandey
 * Date: 7 nov. 2002
 * Time: 12:03:13
 * To change template for new class use
 * Code Style | Class Templates options (Tools | IDE Options).
 */
package fr.gouv.culture.sdx.repository;

import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.configuration.ConfigurationUtils;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.save.SaveParameters;
import fr.gouv.culture.sdx.utils.save.Saveable;
import oracle.sql.BLOB;
import org.apache.avalon.excalibur.io.IOUtil;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

public class ORACLERepository extends JDBCRepository {

    public void configure(Configuration configuration) throws ConfigurationException {
        // We will need the datasource _identifier here, so calling super
        super.configure(configuration);
        //well we need a good dsi value to work with so we verify it
        ConfigurationUtils.checkConfAttributeValue(ATTRIBUTE_DSI, this.dsi, configuration.getLocation());
    }

    /** Returns an SQL query that could create the table.
     * <p>
     * This query should not have any parameter.
     * @return The query.
     */

    protected String getTableCreationQuery() {
        // TODO: at some time, implement the proper handling of the original document (column data_t)
        //changed dataType of FIELD_DATA_TRANSFORMED from "LONGBLOB NOT NULL" TO  only "LONGBLOB"
        return "CREATE TABLE " + getTableName() + " ( " + FIELD_ID + " VARCHAR(255) NOT NULL, " + FIELD_DATA + " BLOB, PRIMARY KEY (" + FIELD_ID + "))";
    }

    /** Returns an SQL query that could add a document.
     * <p>
     * This query should have three parameters, the first for the id, the second
     * for the doctype and the third for the data.
     * @return The query.
     */
    protected String getDocumentAddQuery() {
        //TODOImplement: at some time, implement the proper handling of the original document (column data_t).
        return "INSERT INTO " + getTableName() + " (" + FIELD_ID + "," + FIELD_DATA + ") VALUES (?, EMPTY_BLOB())";
    }
    
    /** 
	 * @see fr.gouv.culture.sdx.utils.database.AbstractJDBCDatabase#getAllEntitiesWithLimitQuery(long, long)
	 * ORACLE implementation
	 * SELECT * FROM &gt;table_name&lt; LIMIT &gt;offset&lt;,&gt;number&lt; ;
	 */
	protected String getEntriesWithLimitQuery(long offset, long number) {
		String query = "SELECT * FROM " + getTableName();
		query += " LIMIT " + String.valueOf(offset) + "," + String.valueOf(number) + ";";
		return query;
	}

    /** Adds a document to the repository.
     *
     * @param doc   A document.
     * @param c     A connection to the repository.
     * @throws fr.gouv.culture.sdx.exception.SDXException
     */
    public synchronized void add(Document doc, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        Utilities.checkDocument(super.getLog(), doc);
        this.checkConnection(c);

        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = this.getDocumentAddQuery();
        final InputStream is = doc.openStream();

        try {
            Template template = new Template(conn, queryString);
            final String docId = doc.getId();
            QueryExecutor qe = new QueryExecutor() {
                public void prepare(PreparedStatement ps) throws SQLException {
                    ps.setString(PARAM_INDEX_FIELD_ID, docId);
                }
            };
            template.execute(qe, Template.MODE_EXECUTE_UPDATE);
            conn.commit();

            Template template2 = new Template(conn, "SELECT " + FIELD_DATA + " FROM " + getTableName() + " WHERE " + FIELD_ID + " = ? FOR UPDATE");
            QueryExecutor qe2 = new QueryExecutor() {
                public void prepare(PreparedStatement ps) throws SQLException {
                    ps.clearParameters();
                    ps.setString(PARAM_INDEX_FIELD_ID, docId);
                }

                public void collect(ResultSet rs) throws SQLException, IOException {
                    if (rs.next()) {
                        oracle.sql.BLOB oraBlob = ((oracle.jdbc.OracleResultSet) rs).getBLOB(FIELD_DATA);
                        OutputStream os = oraBlob.getBinaryOutputStream();
                        int chunk = oraBlob.getChunkSize();
                        byte[] buffer = new byte[chunk];
                        int length = -1;
                        while ((length = is.read(buffer)) != -1)
                            os.write(buffer, 0, length);
                        os.flush();
                        os.close();
                    }
                }
            };
            conn.setAutoCommit(false);
            template2.execute(qe2, Template.MODE_EXECUTE_QUERY);
            //this is will give us the original file
            conn.commit();
        } catch (SQLException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_ADD_DOC, args, e);
        } finally {


            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_CLOSE_STREAM, null, e);
                }
            }
        }
    }

    /** Writes the content of a document to an output stream.
     *
     * @param doc A document.
     * @param os An output stream.
     * @param c A connection to the repository.
     * @throws SDXException
     */
    public void get(Document doc, final OutputStream os, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        Utilities.checkDocument(super.getLog(), doc);
        Utilities.checkOutputStream(super.getLog(), os);
        this.checkConnection(c);

        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentGetQuery();
        try {
            Template template = new Template(conn, queryString);
            final String docId = doc.getId();
            QueryExecutor qe = new QueryExecutor() {
                public void prepare(PreparedStatement ps) throws SQLException {
                    ps.setString(PARAM_INDEX_FIELD_ID, docId);
                }

                public void collect(ResultSet rs) throws SQLException, IOException {
                    if (rs.next()) {
                        oracle.sql.BLOB oraBlob = ((oracle.jdbc.OracleResultSet) rs).getBLOB(FIELD_DATA);
                        IOUtil.copy(oraBlob.getBinaryStream(), os);
                    }
                }
            };
            template.execute(qe, Template.MODE_EXECUTE_QUERY);

        } catch (SDXException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_GET_DOC, args, e);
        }

    }

    /** Opens an input stream to read the content of a document.
     *
     * @return          The stream.
     * @param encoding  An encoding (may be null).
     * <p>If <code> null</code> or invalid we use a default.
     * <p>TODOImplement use of encoding currently not implemented , will do soon.
     * @param doc       A document to read.
     * @param c         A connection to the repository.
     * @throws SDXException
     */
    public InputStream openStream(Document doc, String encoding, RepositoryConnection c) throws SDXException {
        //ensuring we have valid objects
        Utilities.checkDocument(super.getLog(), doc);
        this.checkConnection(c);
        //verifying the encoding, if not valid we use a default, see checkEncoding() for details
        encoding = checkEncoding(encoding);
        Connection conn = ((JDBCRepositoryConnection) c).getConnection();
        String queryString = getDocumentGetQuery();
        try {
            Template template = new Template(conn, queryString);
            final String docId = doc.getId();
            QueryExecutor qe = new QueryExecutor() {
                InputStream l_is = null;

                public void prepare(PreparedStatement ps) throws SQLException {
                    ps.setString(PARAM_INDEX_FIELD_ID, docId);
                }

                public void collect(ResultSet rs) throws SQLException {
                    if (rs.next()) {
                        oracle.sql.BLOB oraBlob = ((oracle.jdbc.OracleResultSet) rs).getBLOB(FIELD_DATA);
                        l_is = oraBlob.getBinaryStream();
                    }
                }

                public InputStream getInputStream() {
                    return l_is;
                }
            };
            template.execute(qe, Template.MODE_EXECUTE_QUERY);
            InputStream is = qe.getInputStream();
            if (is == null) {
                String[] args = new String[2];
                args[0] = doc.getId();
                args[1] = this.getId();
                throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_GET_DOC, args, null);
            }
            return is;
        } catch (SDXException e) {
            String[] args = new String[3];
            args[0] = doc.getId();
            args[1] = this.getId();
            args[2] = e.getMessage();
            throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_GET_DOC, args, e);
        }
    }

    /**
     * Creates the table.
     */
    protected void createTable(Connection conn) throws SDXException {
        PreparedStatement ps = null;
        try {
            ps = conn.prepareStatement(getTableCreationQuery());
            ps.executeUpdate();
        } catch (SQLException e) {
            String[] args = new String[2];
            args[0] = this.getTableName();
            args[1] = e.getMessage();
            /*we don't throw a specific oracle exception, as an exception is thrown when a table exists, we must
            attempt to create this table each init() call, as oracle doesnt give us useful metadata about a database*/
            if (e.getErrorCode() != 955)
                throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_CREATE_TABLE, args, e);
        } finally {
            if (ps != null) {
                try {
                    ps.close();
                } catch (SQLException e1) {
                    String[] args1 = new String[2];
                    args1[0] = this.getId();
                    args1[1] = e1.getMessage();
                    throw new SDXException(super.getLog(), SDXExceptionCode.ERROR_CLOSE_SQL_PREPARED_STATEMENT, args1, e1);
                }
            }
        }
    }
    
    /** Save the repository
	 * @see fr.gouv.culture.sdx.utils.save.Saveable#backup(fr.gouv.culture.sdx.utils.save.SaveParameters)
	 */
	public void backup(SaveParameters save_config) throws SDXException {
		super.backup(save_config);
		if(save_config != null)
			if(save_config.getAttributeAsBoolean(Saveable.ALL_SAVE_ATTRIB,false))
			{
				save_config.setAttribute(Node.Name.TYPE,"ORACLE");	
			}
	}
	
	/** Restore the repository
	 * @see fr.gouv.culture.sdx.utils.save.Saveable#restore(fr.gouv.culture.sdx.utils.save.SaveParameters)
	 */
	public void restore(SaveParameters save_config) throws SDXException {
		super.restore(save_config);
	}

}
