/*
Copyright (C) 2000-2010  Ministere de la culture et de la communication (France), AJLSM
See LICENCE file
*/
package fr.gouv.culture.util.apache.cocoon.xml;

import fr.gouv.culture.util.apache.avalon.excalibur.concurrent.Mutex;
import org.apache.cocoon.xml.XMLConsumer;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

public abstract class AbstractSynchronizedXMLConsumer extends Mutex implements SynchronizedXMLConsumer {

    protected XMLConsumer xmlConsumer = null;

    /**
     * Receive an object for locating the origin of SAX document events.
     *
     * @param locator An object that can return the location of any SAX
     *                document event.
     */
    public void setDocumentLocator(Locator locator) {
        try {
            if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.setDocumentLocator(locator);
        } catch (SynchronizationException e) {
            e.printStackTrace();  //TODO ?: what else can be done here?
        }
    }

    /**
     * Receive notification of the beginning of a document.
     */
    public void startDocument()
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.startDocument();

    }

    /**
     * Receive notification of the end of a document.
     */
    public void endDocument()
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.endDocument();

    }

    /**
     * Begin the scope of a prefix-URI Namespace mapping.
     *
     * @param prefix The Namespace prefix being declared.
     * @param uri The Namespace URI the prefix is mapped to.
     */
    public void startPrefixMapping(String prefix, String uri)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.startPrefixMapping(prefix, uri);

    }


    /**
     * End the scope of a prefix-URI mapping.
     *
     * @param prefix The prefix that was being mapping.
     */
    public void endPrefixMapping(String prefix)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.endPrefixMapping(prefix);

    }

    /**
     * Receive notification of the beginning of an element.
     *
     * @param uri The Namespace URI, or the empty string if the element has no
     *            Namespace URI or if Namespace
     *            processing is not being performed.
     * @param loc The local name (without prefix), or the empty string if
     *            Namespace processing is not being performed.
     * @param raw The raw XML 1.0 name (with prefix), or the empty string if
     *            raw names are not available.
     * @param a The attributes attached to the element. If there are no
     *          attributes, it shall be an empty Attributes object.
     */
    public void startElement(String uri, String loc, String raw, Attributes a)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.startElement(uri, loc, raw, a);

    }


    /**
     * Receive notification of the end of an element.
     *
     * @param uri The Namespace URI, or the empty string if the element has no
     *            Namespace URI or if Namespace
     *            processing is not being performed.
     * @param loc The local name (without prefix), or the empty string if
     *            Namespace processing is not being performed.
     * @param raw The raw XML 1.0 name (with prefix), or the empty string if
     *            raw names are not available.
     */
    public void endElement(String uri, String loc, String raw)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.endElement(uri, loc, raw);

    }

    /**
     * Receive notification of character data.
     *
     * @param ch The characters from the XML document.
     * @param start The start position in the array.
     * @param len The number of characters to read from the array.
     */
    public void characters(char ch[], int start, int len)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.characters(ch, start, len);

    }

    /**
     * Receive notification of ignorable whitespace in element content.
     *
     * @param ch The characters from the XML document.
     * @param start The start position in the array.
     * @param len The number of characters to read from the array.
     */
    public void ignorableWhitespace(char ch[], int start, int len)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.ignorableWhitespace(ch, start, len);

    }

    /**
     * Receive notification of a processing instruction.
     *
     * @param target The processing instruction target.
     * @param data The processing instruction data, or null if none was
     *             supplied.
     */
    public void processingInstruction(String target, String data)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.processingInstruction(target, data);

    }

    /**
     * Receive notification of a skipped entity.
     *
     * @param name The name of the skipped entity.  If it is a  parameter
     *             entity, the name will begin with '%'.
     */
    public void skippedEntity(String name)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.skippedEntity(name);

    }

    /**
     * Report the start of DTD declarations, if any.
     *
     * @param name The document type name.
     * @param publicId The declared public _identifier for the external DTD
     *                 subset, or null if none was declared.
     * @param systemId The declared system _identifier for the external DTD
     *                 subset, or null if none was declared.
     */
    public void startDTD(String name, String publicId, String systemId)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.startDTD(name, publicId, systemId);

    }

    /**
     * Report the end of DTD declarations.
     */
    public void endDTD()
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.endDTD();

    }

    /**
     * Report the beginning of an entity.
     *
     * @param name The name of the entity. If it is a parameter entity, the
     *             name will begin with '%'.
     */
    public void startEntity(String name)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.startEntity(name);

    }

    /**
     * Report the end of an entity.
     *
     * @param name The name of the entity that is ending.
     */
    public void endEntity(String name)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.endEntity(name);

    }

    /**
     * Report the start of a CDATA section.
     */
    public void startCDATA()
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.startCDATA();

    }

    /**
     * Report the end of a CDATA section.
     */
    public void endCDATA()
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.endCDATA();

    }


    /**
     * Report an XML comment anywhere in the document.
     *
     * @param ch An array holding the characters in the comment.
     * @param start The starting position in the array.
     * @param len The number of characters to use from the array.
     */
    public void comment(char ch[], int start, int len)
            throws SAXException {
        if (isAcquired() && this.xmlConsumer != null) this.xmlConsumer.comment(ch, start, len);

    }
}
