/**
 * University of Illinois/NCSA Open Source License
 *
 * Copyright (c) 2001,2002 The Board of Trustees of the University of Illinois
 * All rights reserved.
 *
 * Developed by:  Open Archives Initiative Metadata Harvesting Project
 *                University of Illinois at Urbana-Champaign
 *                http://oai.grainger.uiuc.edu/
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal with the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 *  . Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimers.
 *  . Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimers in the
 *    documentation and/or other materials provided with the distribution.
 *  . Neither the names of Open Archives Initiative Metadata Harvesting
 *    Project, University of Illinois at Urbana-Champaign, nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this Software without specific prior written permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE CONTRIBUTORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS WITH THE SOFTWARE.
 *
 * DC.Title:          OAIError
 *
 * DC.Description:    Class representing an OAI Set
 *
 * DC.Contributor:    Yuping Tseng, ytseng1@uiuc.edu, University og Illinois at Urbana-Champaign
 * DC.Contributor:    Tom Habing, thabing@uiuc.edu, University of Illinois at Urbana-Champaign
 *
 * DC.Date.Modified:  Modtime: 5/13/02 11:58a
 *
 * DC:Rights:         University of Illinois/NCSA Open Source License
 *
 * DC.Relation:       For more information see http://oai.grainger.uiuc.edu/
 * DC.Relation:       See http://www.openarchives.org/OAI/openarchivesprotocol.html
 *
 */
package fr.gouv.culture.oai;

import fr.gouv.culture.oai.util.OAIUtilities;
import org.apache.excalibur.xml.sax.XMLizable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This class represents an OAI Error see section 3.6 of OAI spec for more info.
 *
 *
 */
public class OAIError extends OAIObjectImpl implements XMLizable {

/*
3.6 Error and Exception Conditions
In event of an error or exception condition, repositories must indicate OAI-PMH errors, distinguished from HTTP Status-Codes, by including one or more error elements in the response. While one error element is sufficient to indicate the presence of the error or exception condition, repositories should report all errors or exceptions that arise from processing the request. Each error element must have a code attribute that must be from the following table; each error element may also have a free text string value to provide information about the error that is useful to a human reader. These strings are not defined by the OAI-PMH.

Error Codes
    Description. Applicable Verbs

badArgument
    The request includes illegal arguments, is missing required arguments, includes a repeated argument, or values for arguments have an illegal syntax. all verbs
badResumptionToken
    The value of the resumptionToken argument is invalid or expired. ListIdentifiers, ListRecords, ListSets
badVerb
    Value of the verb argument is not a legal OAI-PMH verb, the verb argument is missing, or the verb argument is repeated.  N/A
cannotDisseminateFormat
    The metadata format identified by the value given for the metadataPrefix argument is not supported by the item or by the repository. GetRecord, ListIdentifiers, ListRecords
idDoesNotExist
    The value of the identifier argument is unknown or illegal in this repository. GetRecord, ListMetadataFormats
noRecordsMatch
    The combination of the values of the from, until, set and metadataPrefix arguments results in an empty list. ListIdentifiers, ListRecords
noMetadataFormats
    There are no metadata formats available for the specified item. ListMetadataFormats
noSetHierarchy
     The repository does not support sets. ListSets, ListIdentifiers, ListRecords
*/

    //Constants for error conditions
    public static final int ERROR_UNKNOWN = -1;
    public static final int ERROR_BAD_ARGUMENT = 0;
    public static final int ERROR_BAD_RESUMPTION_TOKEN = 1;
    public static final int ERROR_BAD_VERB = 2;
    public static final int ERROR_CANNOT_DISSEMINATE_FORMAT = 3;
    public static final int ERROR_ID_DOES_NOT_EXIST = 4;
    public static final int ERROR_NO_RECORDS_MATCH = 5;
    public static final int ERROR_NO_METADATA_FORMATS = 6;
    public static final int ERROR_NO_SET_HIERARCHY = 7;
	public static final int ERROR_BAD_DATE = 8;

    //Constants (Strings) for error conditions
    public static final String ERROR_STRING_UNKNOWN = "unknown";
    public static final String ERROR_STRING_BAD_ARGUMENT = "badArgument";
    public static final String ERROR_STRING_BAD_RESUMPTION_TOKEN = "badResumptionToken";
    public static final String ERROR_STRING_BAD_VERB = "badVerb";
    public static final String ERROR_STRING_CANNOT_DISSEMINATE_FORMAT = "cannotDisseminateFormat";
    public static final String ERROR_STRING_ID_DOES_NOT_EXIST = "idDoesNotExist";
    public static final String ERROR_STRING_NO_RECORDS_MATCH = "noRecordsMatch";
    public static final String ERROR_STRING_NO_METADATA_FORMATS = "noMetadataFormats";
    public static final String ERROR_STRING_NO_SET_HIERARCHY = "noSetHierarchy";
	public static final String ERROR_STRING_BAD_DATE = "badDate";


    /**The error state*/
    int intErrorCode = ERROR_UNKNOWN;
    /**The error state string name*/
    String stringErrorCode = ERROR_STRING_UNKNOWN;
    /**The explanation*/
    String reason = "";

    /**
     * Construct a empty object.
     */
    public OAIError(int errorCode) {
        setUp(errorCode);
    }

    /**Simple constructor
     *
     * @param errorCode
     * @param reason
     */
    public OAIError(int errorCode, String reason) {
        this(errorCode);
        this.reason = reason;
    }


    /**Establishes the error state
     * and its String name
     *
     * @param errorCode
     */
    protected void setUp(int errorCode) {
        this.intErrorCode = errorCode;
        switch (this.intErrorCode) {
            case ERROR_BAD_ARGUMENT:
                this.stringErrorCode = ERROR_STRING_BAD_ARGUMENT;
                break;
            case ERROR_BAD_RESUMPTION_TOKEN:
                this.stringErrorCode = ERROR_STRING_BAD_RESUMPTION_TOKEN;
                break;
            case ERROR_BAD_VERB:
                this.stringErrorCode = ERROR_STRING_BAD_VERB;
                break;
            case ERROR_CANNOT_DISSEMINATE_FORMAT:
                this.stringErrorCode = ERROR_STRING_CANNOT_DISSEMINATE_FORMAT;
                break;
            case ERROR_ID_DOES_NOT_EXIST:
                this.stringErrorCode = ERROR_STRING_ID_DOES_NOT_EXIST;
                break;
            case ERROR_NO_RECORDS_MATCH:
                this.stringErrorCode = ERROR_STRING_NO_RECORDS_MATCH;
                break;
            case ERROR_NO_METADATA_FORMATS:
                this.stringErrorCode = ERROR_STRING_NO_METADATA_FORMATS;
                break;
            case ERROR_NO_SET_HIERARCHY:
                this.stringErrorCode = ERROR_STRING_NO_SET_HIERARCHY;
                break;
            case ERROR_BAD_DATE:
                this.stringErrorCode = ERROR_STRING_BAD_ARGUMENT;
                break;
            default:
                this.stringErrorCode = ERROR_STRING_UNKNOWN;
                break;


        }
    }


    /**Send the error events to the provided content handler while respecting
     * the oai pmh format for error evetns
     * @param contentHandler    The handler to feed with events
     * @throws SAXException
     */
    public void toSAX(ContentHandler contentHandler) throws SAXException {
        /*
        <error code="noSetHierarchy">This repository does not
           support sets</error>
        */
        ContentHandler currentHandler = super.contentHandler;

        super.setContentHandler(contentHandler);
        AttributesImpl atts = new AttributesImpl();

        if (OAIUtilities.checkString(this.stringErrorCode))
            atts.addAttribute("", OAIObject.Node.Name.CODE, OAIObject.Node.Name.CODE, OAIObject.Node.Type.CDATA, this.stringErrorCode);

        super.sendElement(OAIObject.Node.Xmlns.OAI_2_0, OAIObject.Node.Name.ERROR, OAIObject.Node.Name.ERROR, atts, this.reason);

        super.setContentHandler(currentHandler);
    }


}



