/*
Copyright (C) 2000-2010  Ministere de la culture et de la communication (France), AJLSM
See LICENCE file
 */
package fr.gouv.culture.sdx.utils;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**	Statics methods utilities for SAX
 *
 */
public class SAXUtils {

	public static String prefixNodeName(String prefix, String elemName) {
		if (!Utilities.checkString(elemName)) return null;
		String qualifiedElementName = "";
		if (Utilities.checkString(prefix))
			qualifiedElementName += prefix + ":";
		qualifiedElementName += elemName;
		return qualifiedElementName;
	}

	/**Sends a element with the simple character content
	 * provided by the <code>elemContent</code> param
	 *
	 * @param uri   The namespace uri
	 * @param localName The local name
	 * @param qName The qualified name
	 * @param atts The attributes
	 * @param elemContent The character content
	 * @throws SAXException
	 */
	public static void sendElement(ContentHandler handler, String uri, String localName, String qName, Attributes atts, String elemContent) 
		throws SAXException {
		if (atts == null) atts = new AttributesImpl();
		handler.startElement(uri, localName, qName, atts);
		SAXUtils.sendElementContent(handler, elemContent);
		handler.endElement(uri, localName, qName);
	}

	/**Sends character content from the provided string
	 *
	 * @param content The string of characters to send
	 */
	public static void sendElementContent(ContentHandler handler, String content) throws SAXException {
		sendElementContent(handler, content, false, null);
	}
	/**	Sends character content from the provided string and filters non XML characters
	 * @param handler	The ContentHandler
	 * @param content	The String to sends
	 * @param escape	Boolean to activate the filter on non XML characters
	 * @throws SAXException
	 */
	public static void sendElementContent(ContentHandler handler, String content, boolean escape) 
		throws SAXException {
		sendElementContent(handler, content, escape, null);
	}
	
	
	/**	Sends character content from the provided string and replaces non XML characters
	 * 	by another character, space by default.
	 * 	See {@linkplain http://www.w3.org/TR/2004/REC-xml-20040204/#charsets}
	 * @param handler	The contentHandler
	 * @param content	The String to sends
	 * @param escape	Boolean to activate the filter on non XML characters
	 * @param replace	The String used to replace non XML character. Default is space " "
	 * @throws SAXException
	 */
	public static void sendElementContent(ContentHandler handler, String content, boolean escape, Character replace)
		throws SAXException {
		if (Utilities.checkString(content)) {
			char[] cs = content.toCharArray();
			if ( escape )
			{
				/*
				 * On doit supprimer les caracteres non XML. On les transforme en espace.
				 * La regle est ici:
				 *
				 * Char ::= #x9 | #xA | #xD | [#x20-#xD7FF] | [#xE000-#xFFFD] | [#x10000-#x10FFFF]
				 * (any Unicode character, excluding the surrogate blocks, FFFE, and FFFF.)
				 * [http://www.w3.org/TR/2004/REC-xml-20040204/#charsets]
				 *
				 * On ne va pas considérer [#x10000-#x10FFFF] car c'est hors de la limite Java.
				 */
				// On fait attention aux caractères interdits en XML
				char r = '\u0020';
				if ( replace!=null )
					r = replace.charValue();
				for (int i=0; i<cs.length; i++) {
					char cc = cs[i];
					if ( !( cc == 0x9 || cc == 0xA || cc == 0xD 
							|| (cc >= 0x20 && cc <= 0xD7FF) 
							|| (cc >= 0xE000 && cc <= 0xFFFD) ) )
					{
						cs[i] = r;
					}
				}
			}
			handler.characters(cs, 0, cs.length);
		}
	}

}
