// Copyright (C) 1995 The Santa Fe Institute.
// No warranty implied, see LICENSE for terms.


#import <collections.h>
#import <simtools.h>

#import "BinChrom.h"
#import "PopSwarm.h"

#include <stdlib.h>		// for atof

#define STRSIZE 20

@implementation PopulationSwarm

void printGenealogy( char* _first, char* _second, float _parenthood ) {
// Prints a message to standard output with the required format
  printf( "(%s %s %.2f)\n", _first, _second, _parenthood );
}

// Comparison function
int compFunc( id a, id b ) {

    float tmp1, tmp2;
    
    sscanf( (char*) a, "%f", &tmp1 );
    sscanf( (char*) b, "%f", &tmp2 );
    if ( tmp1 > tmp2 ) 
	return -1;
    else
	return 1;
}

// ------------------------------ Swarm functions --------------

+createBegin: (id) aZone {

  PopulationSwarm *obj;
  ProbeMap * probeMap;

  obj = [super createBegin:aZone];

  // init defaults
  obj->initPopSize = 100;
  obj->popSize=obj->initPopSize; // Without probe
  obj->chromLen = 0;
  obj->mating = 0.1;		// Good for the demo problem
  obj->mut = 0.05;
  obj->dup = 0;
  obj->kill = 0;
  obj->swap = 0;
  obj->remrei = 0;		// Probabilities of application of genOps
  obj->generations = 100;	// Generations to run
  obj->evalFunc = 0;		// Evaluation function
  obj->verbose= 0;		// Print genealogy or not

  // Create probes
  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // MODIFY: add your own probes for your parameters.
  [probeMap addProbe: [probeLibrary getProbeForVariable: "initPopSize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "generations"
				     inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "mating"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "mut"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "dup"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "kill"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "remrei"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "verbose"
				    inClass: [self class]]];

  // Message probes
  [probeMap addProbe: [probeLibrary getProbeForMessage: "addPopulation:"
				    inClass: [self class]]];
  
  [probeLibrary setProbeMap: probeMap For: [self class]];
  
  return obj;
}

-setChromLength: (unsigned) _len {
  chromLen = _len;
  return self;
}

-setEvaluationFunc: (float(*)( id )) _evalFunc {
  evalFunc = _evalFunc;
  return self;
}

// __________________________
// Build simulation objects
-buildObjects {

  [super buildObjects];
  newPopList = [List create: [self getZone]];

  // Create Uname generator
  Generator = [UName create:[self getZone] setBaseName: "gen0X"];

  // Create initial random population
  [self addPopulation: initPopSize];
   
  // Create intial map, to keep evaluated genes
  popList = [Map createBegin: [self getZone]];
  [popList setCompareFunction: compFunc];
  popList = [popList createEnd];

  return self;
}

-buildActions {
  [super buildActions];

  // MODIFY: schedule your own agents as appropriate.
  modelActions = [ActionGroup create: [self getZone]];
  [modelActions createActionTo: self message: M(evaluate)];
  [modelActions createActionTo: self message: M(newGeneration) ];

  modelSchedule = [Schedule createBegin: [self getZone]];
  [modelSchedule setRepeatInterval: 1];
  modelSchedule = [modelSchedule createEnd];
  [modelSchedule at: 0 createAction: modelActions];
  
  return self;
}
//_______________________________________________________________
-activateIn: (id) swarmContext  {
//_______________________________________________________________
  [super activateIn: swarmContext];

  [modelSchedule activateIn: self];

  return [self getSwarmActivity];
}

//_______________________________________________________________
-createEnd {
//_______________________________________________________________
  
  return [super createEnd];
}

// From protocol
//_______________________________________________________________________
-(void) addPopulation: (unsigned) _num {
				// Adds new chromosomes to the population
				// Set popSize variable
//_______________________________________________________________________

  unsigned i;
  for ( i = 0; i < _num; i++ ) {
    BinChromosome* gen;
    gen = [BinChromosome createBegin: [self getZone]];
    [[[gen setLength: chromLen] setRandomInit] setID: [Generator getNewName]];
    gen = [gen createEnd];
    
    [newPopList addFirst: gen];
  }
  popSize+= _num;
    
}

//_______________________________________________________________
-evaluate {			// Evaluates all member of the population
//_______________________________________________________________

    // Takes them from the new list and places then in the Population map,
    // keyed with fitness

    id index, member;
    index = [newPopList begin: [self getZone]];

    while ( ( member = [index next]) ) {
      char* saux = calloc( STRSIZE, sizeof( char ) );
				// Must be allocated to be used as key
      float result =  ((*evalFunc)( member )) ;
      [member setFitness: result];
      sprintf( saux, "%f", result );
      if (! [popList at: (id) saux insert: member ] ) {
	[SourceMessage raiseEvent : "Something is wrong\n" ];
      }
    }
    [index drop];

    // now remove all members of the list
    index = [newPopList begin: [self getZone]];
    while ( ( member = [index next]) ) {
	[index remove];
    }
    [index drop];

    return self;
}

-(unsigned) getGenerationCount {
  return generationCount;
}

-(unsigned) getGenerations {
  return generations;
}

-newGeneration {

    id index, stiff;
    unsigned killQuant = popSize*( mating + mut + dup + kill +swap + remrei), 
	i;
    char genID[10];

    // Increase generation Count
    generationCount++;

    if ( generationCount == generations ) {
      if ( verbose ) {		// print population
	[self printPopID];
      }
      return 0;
    }
    // Kill the worst mutProb, and substitute them by the siblings of the
    // others
    i = 0; 
    index = [ popList begin: [self getZone]];
    [index setLoc: End];

    while( ( i < killQuant) && (stiff = [index prev] ) ){
      if ( verbose ) {
	printf("[%s %f]\n", [stiff getID], [stiff getFitness] );
      }
      [stiff drop];		// Eliminate gen
      [index remove];
      i++;
    }
    [index drop];

    // Create new basename for this generation
    sprintf( genID, "gen%dX", generationCount );
    [Generator setBaseName: genID ];
    // Now make the new ones as siblings from the first kill percent
    // and another randomly chosen
    for ( i = 0; i < popSize*mating; i ++ ) {
	BinChromosome *gen, *parentA, *parentB;
	parentA = [popList atOffset: 
			     [uniformUnsRand getUnsignedWithMin: 0L 
					     withMax:[popList count] - 1] ];
	parentB = [popList atOffset: 
			     [uniformUnsRand getUnsignedWithMin: 0L 
					     withMax:[popList count] - 1] ];
	
	gen = [BinChromosome createBegin: [self getZone]];
	[gen setID: [Generator getNewName]];
	
	[gen Mate: parentA  And: parentB withMutation: mut];
	
	if ( ! (gen = [gen createEnd] ) ) {
	  [SourceMessage raiseEvent: "Error creating chromosome\n"];
	  return nil;
	}
	if ( verbose ) {	// print generations
	  printGenealogy( [parentA getID], [gen getID], 0.5 );
	  printGenealogy( [parentB getID], [gen getID], 0.5 );
	}
	[newPopList addFirst: gen];
    }
    for ( i = 0; i < popSize*mut; i ++ ) {
      BinChromosome *gen, *parent;
      parent = [popList atOffset:
			  [uniformUnsRand getUnsignedWithMin: 0L 
					  withMax:[popList count] - 1] ];
      gen = [BinChromosome createBegin: [self getZone]];
      [gen setID: [Generator getNewName]];
      [gen Clone: parent withMutation: mut];

      if ( ! (gen = [gen createEnd] ) ) {
	[InvalidCombination raiseEvent: "Error creating chromosome"];
      }
      if ( verbose ) {	// print generations
	printGenealogy( [parent getID], [gen getID], mut );
      }
      [newPopList addFirst: gen];
    }

    for ( i = 0; i < popSize*dup; i ++ ) {
	BinChromosome *gen, *parent;
	unsigned geneIndex;
	parent = 
	    [popList atOffset:
		       [uniformUnsRand getUnsignedWithMin: 0L 
				       withMax:[popList count] - 1]];
	
	geneIndex = [uniformUnsRand getUnsignedWithMin: 0L 
				    withMax:[parent getLength] - 1];
	gen = [BinChromosome createBegin: [self getZone]];
	[gen setID: [Generator getNewName]];
	[gen Duplicate: parent From: geneIndex To: geneIndex 
	     withMutation: mut];

	if ( ! (gen = [gen createEnd] ) ) {
	  [InvalidCombination raiseEvent:"Error creating chromosome\n"];
	}
	if ( verbose ) {	// print generations
	  printGenealogy( [parent getID], [gen getID], 0.1 );
	}
	[newPopList addFirst: gen];
     } 

     for ( i = 0; i < popSize*swap; i ++ ) {
	BinChromosome *gen, *parent;
	unsigned geneIndex1, geneIndex2;
	parent = 
	    [popList atOffset: [uniformUnsRand getUnsignedWithMin: 0L 
				       withMax:[popList count] - 1]];
	geneIndex1 = [uniformUnsRand getUnsignedWithMin: 0L 
				withMax:[parent getLength] - 1];
	geneIndex2 = [uniformUnsRand getUnsignedWithMin: geneIndex1 
				withMax:[parent getLength] - 1];
	gen = [BinChromosome createBegin: [self getZone]];
	[gen setID: [Generator getNewName]];
	[gen Swap: parent : geneIndex1 with: geneIndex2 size: 1];

	if ( ! (gen = [gen createEnd] ) ) {
	     [InvalidCombination raiseEvent: "Error creating chromosome\n"];
	}
	if ( verbose ) {	// print generations
	  printGenealogy( [parent getID], [gen getID], 0.1 );
	}
	[newPopList addFirst: gen];
     } 

     for ( i = 0; i < popSize*remrei; i ++ ) {
	BinChromosome *gen, *parent;
	unsigned geneIndex1, geneIndex2;
	parent = 
	    [popList atOffset: [uniformUnsRand getUnsignedWithMin: 0L 
				       withMax:[popList count] - 1]];
	geneIndex1 = [uniformUnsRand getUnsignedWithMin: 0L 
				withMax:[parent getLength] - 1];
	geneIndex2 = [uniformUnsRand getUnsignedWithMin: geneIndex1 
				withMax:[parent getLength] - 1];
	gen = [BinChromosome createBegin: [self getZone]];
	[gen setID: [Generator getNewName]];
	[gen RemoveReinsert: parent : geneIndex1 To: geneIndex2 size: 1];
	if ( ! (gen = [gen createEnd] ) ) {
	  [InvalidCombination raiseEvent: "Error creating chromosome\n"];
	}
	if ( verbose ) {	// print generations
	  printGenealogy(  [parent getID], [gen getID], 0.1 );
	}
	[newPopList addFirst: gen];
    } 
    return self;
}

-getBest{			// Returns the best genome
//    printf( "Total %d\n", [population count]);
//    id index = [population begin: [self getZone]], member = [index next];// First member
//    printf( "Miembro %s con clave %s\n", [member getGenotype], (char* )[index getKey] );
//    [index dropFrom: [self getZone]];
    return [popList first];
}


-(double) getBestFitness{		// Fitness of the best genome
    float tmp;
    id index = [popList begin: [self getZone]];// First member
    char key[STRSIZE];
    [index next];
    strcpy( key, (char * ) [index getKey] );
    [index drop];
    sscanf( key, "%f", &tmp );
    return tmp;
}

-(id) getList {
  return popList;
}

-printPopID {			// prints population
  unsigned i;
  printf( "Last Generation -- \n" );
  for ( i = 0; i < [popList count]; i ++ ) {
    printf( "%s\n", [ [popList atOffset:i] getID ] );
  }
  return self;
}  
@end

