/* 
  WorldManager.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  17-OCT-1996

*/

#import <simtools.h>
#import <simtoolsgui.h>

#import "SCLGlobals.h"
#import "SCLDisplay.h"

#import "Version.h"

#import "Hole.h"
#import "Substrate.h"
#import "Link.h"
#import "Catalyst.h"

#import "CellEditor.h"
#import "UserCellEditor.h"
#import "AgentManager.h"
#import "ParticleManager.h"
#import "PRNG.h"

#import "WorldManager.h"

@implementation WorldManager


+createBegin: (id) aZone {
  WorldManager * finalSelf;
  id <MessageProbe> messageProbe;
  id <ProbeMap> probeMap;

  finalSelf = [super createBegin: aZone];

  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [finalSelf class]];
  probeMap = [probeMap createEnd];

  messageProbe = 
    [probeLibrary getProbeForMessage: "saveToFileNamed:"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  messageProbe = 
    [probeLibrary getProbeForMessage: "loadFromFileNamed:"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  [probeMap addProbe: 
    [probeLibrary getProbeForVariable: "worldTime"
       inClass: [finalSelf class]]];
  [probeMap addProbe: 
    [probeLibrary getProbeForVariable: "prngState"
       inClass: [finalSelf class]]];
  [probeMap addProbe: 
    [probeLibrary getProbeForVariable: "prngCount"
       inClass: [finalSelf class]]];

  messageProbe = 
    [probeLibrary getProbeForMessage: "createEmptyWorldWithX:Y:"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  messageProbe = 
    [probeLibrary getProbeForMessage: "createDefaultWorld"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  messageProbe = 
    [probeLibrary getProbeForMessage: "fillWithHoles"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  messageProbe = 
    [probeLibrary getProbeForMessage: "fillWithSubstrate"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  [probeMap addProbe: 
    [probeLibrary getProbeForMessage: "adjustSubstrateNumberBy:"
       inClass: [finalSelf class]]];
  [probeMap addProbe: 
    [probeLibrary getProbeForMessage: "adjustLinkNumberBy:"
       inClass: [finalSelf class]]];
  [probeMap addProbe: 
    [probeLibrary getProbeForMessage: "adjustCatalystNumberBy:"
       inClass: [finalSelf class]]];

  messageProbe = 
    [probeLibrary getProbeForMessage: "setWorldTime:"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  messageProbe = 
    [probeLibrary getProbeForMessage: "setPRNGstate:"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];
  messageProbe = 
    [probeLibrary getProbeForMessage: "setPRNGcount:"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  //messageProbe = 
  //  [probeLibrary getProbeForMessage: "refreshDisplay"
  //    inClass: [self class]];
  //[messageProbe setHideResult: 1];
  //[probeMap addProbe: messageProbe];

  messageProbe = 
    [probeLibrary getProbeForMessage: "update"
      inClass: [self class]];
  [messageProbe setHideResult: 1];
  [probeMap addProbe: messageProbe];

  [probeLibrary setProbeMap: probeMap For: [finalSelf class]];
  
  return finalSelf;
}


-createEnd {
  WorldManager *finalSelf;

  finalSelf = [super createEnd];
  [probeDisplayManager createProbeDisplayFor: finalSelf];

  finalSelf->worldTime = 0;
  [Link setupProbeMap: [finalSelf getZone]];  // Has to be done *somewhere*...

  return (finalSelf);
}

-(void) drop {
  [self dropWorld];
  [super drop];
}

-(unsigned) setWorldTime: (unsigned) time {
  worldTime = time;

  return worldTime;
}

-(unsigned) getWorldTime {
  return worldTime;
}

-setBackgroundColor: (Color) color {
  backgroundColor = color;

  return self;
}

-(Color) getBackgroundColor {
  return backgroundColor;
}

-refreshDisplay{
  if (world != nil) { // Defensive...
    //    [worldRaster erase];
    [worldRaster fillRectangleX0: 0 Y0: 0
      X1: ([world getSizeX] * 8) Y1: ([world getSizeY] * 8) 
      Color: backgroundColor];
    [[bondMgr getAgentList] forEach: M(drawSelfOn:) : worldRaster];
    [worldDisplay display];
    [worldRaster drawSelf];
    [probeDisplayManager update];
  }

  return self;
}

-setPRNGstate: (unsigned) newState {
  prngState = newState;
  [prng setPRNGstate: prngState];

  return self;
}

-(unsigned) getPRNGstate {
  prngState = [prng getPRNGstate]; // Defensive...

  return prngState;
}

-setPRNGcount: (unsigned) newCount {
  prngCount = newCount;
  [prng setPRNGcount: prngCount];

  return self;
}

-(unsigned) getPRNGcount {
  prngCount = [prng getPRNGcount]; // Defensive...

  return prngCount;
}

-update {
  if (world != nil) { // Defensive...
    [holeMgr processQueues];
    [substrateMgr processQueues];
    [linkMgr processQueues];
    [catalystMgr processQueues];

    [bondMgr processQueues];

    [userCellEditor update];

    [self getPRNGstate];
    [self getPRNGcount];
    [self refreshDisplay];
  }

  return self;
}

-step {
  if (world != nil) { // Defensive...
    worldTime++;

    [[catalystMgr getAgentList] forEach: M(initialiseStep)];
    [[linkMgr getAgentList] forEach: M(initialiseStep)];
    [[substrateMgr getAgentList] forEach: M(initialiseStep)];
    [[holeMgr getAgentList] forEach: M(initialiseStep)];

    [[catalystMgr getAgentList] forEach: M(step)];
    [holeMgr processQueues];
    [substrateMgr processQueues];
    [linkMgr processQueues];

    [[linkMgr getAgentList] forEach: M(step)];
    [holeMgr processQueues];
    [substrateMgr processQueues];
    [linkMgr processQueues];
    [bondMgr processQueues];

    [[substrateMgr getAgentList] forEach: M(step)];
    [[holeMgr getAgentList] forEach: M(step)];
    [[bondMgr getAgentList] forEach: M(step)];

    [bondMgr processQueues];

    [userCellEditor update];

    [self getPRNGstate];
    [self getPRNGcount];
    [self refreshDisplay];
  }

  return self;
}


-dropWorld {
  [modelSwarm setStateStopped]; // Defensive...
  if (world != nil) {
    /* We assume that if world != nil then *all* of the
       associated objects must be dropped... conversely, if
       world == nil already, we assume *none* of the associated
       objects need be dropped. */

    [userCellEditor drop];
    userCellEditor = nil;

    [bondMgr drop];
    bondMgr = nil;

    [catalystMgr drop];
    catalystMgr = nil;
    [linkMgr drop];
    linkMgr = nil;
    [substrateMgr drop];
    substrateMgr = nil;
    [holeMgr drop];
    holeMgr = nil;

    [worldDisplay drop];
    worldDisplay = nil;

    [worldRaster drop];
    worldRaster = nil;

    [world drop];
    world = nil;
  }

  return self;
}

-createWorldWithX: (int) xSize Y: (int) ySize {
  [self dropWorld]; // Defensive...

  world = [DiscreteToroid createBegin: [self getZone]];
  [world setSizeX: xSize Y: ySize];
  world = [world createEnd];

  catalystMgr = [ParticleManager createBegin: [self getZone]];
  [catalystMgr setAgentClass: [Catalyst class]];
  [catalystMgr setWorld: world];
  catalystMgr = [catalystMgr createEnd];

  linkMgr = [ParticleManager createBegin: [self getZone]];
  [linkMgr setAgentClass: [Link class]];
  [linkMgr setWorld: world];
  linkMgr = [linkMgr createEnd];

  substrateMgr = [ParticleManager createBegin: [self getZone]];
  [substrateMgr setAgentClass: [Substrate class]];
  [substrateMgr setWorld: world];
  substrateMgr = [substrateMgr createEnd];

  holeMgr = [ParticleManager createBegin: [self getZone]];
  [holeMgr setAgentClass: [Hole class]];
  [holeMgr setWorld: world];
  holeMgr = [holeMgr createEnd];

  bondMgr = [AgentManager createBegin: [self getZone]];
  [bondMgr setAgentClass: [Bond class]];
  bondMgr = [bondMgr createEnd];

  worldRaster = [ZoomRaster create: [self getZone]];
  [worldRaster setColormap: colormap];
  [worldRaster setZoomFactor: 3];
  [worldRaster setWidth: ([world getSizeX] * 8) 
                  Height: ([world getSizeY] * 8) ];
  [worldRaster setWindowTitle: "SCL World"];
  [worldRaster pack];

  worldDisplay = [SCLDisplay createBegin: [self getZone]];
  [worldDisplay setDisplayWidget: worldRaster];
  [worldDisplay setDiscrete2dToDisplay: world];
  [worldDisplay setObjectCollection: nil];
  [worldDisplay setDisplayMessage: M(drawSelfOn:)];
  worldDisplay = [worldDisplay createEnd];

  userCellEditor = [UserCellEditor create: [self getZone]];

  // Also, tell the world raster to send mouse clicks to the worldDisplay
  // this allows the user to right-click on the display to probe the bugs.
  [worldRaster setButton: ButtonRight Client: worldDisplay 
    Message: M(makeProbeAtX:Y:)];
  [worldRaster setButton: ButtonLeft Client: userCellEditor 
    Message: M(setRasterX:Y:)];

  return self;
}

-createEmptyWorldWithX: (int) xSize Y: (int) ySize {
  int x, y;
  Hole *hole;

  [self createWorldWithX: xSize Y: ySize];
  for (x = 0; x < xSize; x++)
    for (y = 0; y < ySize; y++) {
      hole = [holeMgr createAgent];
      [hole unwarpToX: x Y: y];
    }

  [self setWorldTime: 0];
  [self update];

  return self;
}

-dropAgent: (id) agent {
  Class agentClass;

  agentClass = [agent class];
  if (agentClass != nil) { // Defensive
    if (agentClass == [Catalyst class]) 
      [catalystMgr dropAgent: agent];
    else if (agentClass == [Link class]) 
      [linkMgr dropAgent: agent];
    else if (agentClass == [Substrate class])
      [substrateMgr dropAgent: agent];
    else if (agentClass == [Hole class])
      [holeMgr dropAgent: agent];
    else
      [InternalError raiseEvent:
        "Unrecognised agentClass %d encountered.",
        agentClass];
  }

  return self;
}
    

-fillWithSubstrate {
  int x, y;
  int xSize, ySize;
  id agent;
  Substrate *substrate;

  xSize = [world getSizeX];
  ySize = [world getSizeY];

  for (x = 0; x < xSize ; x++)
    for (y = 0; y < ySize; y++) {
      agent = [world getObjectAtX: x Y: y];
      if ([agent class] != [Substrate class]) {
        [self dropAgent: agent];
	substrate = [substrateMgr createAgent];
	[substrate unwarpToX: x Y: y];
      }
    }

  [self update];

  return self;
}

-(int) addRandomSubstrate: (int) numNewSubstrates {
  int x, y;
  int numHoles;
  int newSubstrateCount;
  Hole *hole;
  Substrate *substrate;

  [self update]; // Defensive...
  newSubstrateCount = 0;
  numHoles = [holeMgr getCount];
  while ((newSubstrateCount < numNewSubstrates) && (numHoles > 0)) {
    hole = [holeMgr randomAgent];
    x = [hole getX];
    y = [hole getY];
    [holeMgr dropAgent: hole];
    numHoles--;
    substrate = [substrateMgr createAgent];
    [substrate unwarpToX: x Y: y];
    newSubstrateCount++;
    [holeMgr processQueues];
    [substrateMgr processQueues];
  }

  [substrateMgr processQueues];

  [self update];

  return newSubstrateCount;
}

-(int) removeRandomSubstrate: (int) numRemovedSubstrates {
  int x, y;
  int numSubstrates;
  int removedSubstrateCount;
  Hole *hole;
  Substrate *substrate;

  [self update]; // Defensive...
  removedSubstrateCount = 0;
  numSubstrates = [substrateMgr getCount];
  while ((removedSubstrateCount < numRemovedSubstrates) 
      && (numSubstrates > 0)) {
    substrate = [substrateMgr randomAgent];
    x = [substrate getX];
    y = [substrate getY];
    [substrateMgr dropAgent: substrate];
    hole = [holeMgr createAgent];
    [hole unwarpToX: x Y: y];
    numSubstrates--;
    removedSubstrateCount++;
    [substrateMgr processQueues];
    [holeMgr processQueues];
  }

  [substrateMgr processQueues];

  [self update];

  return removedSubstrateCount;
}

-(int) adjustSubstrateNumberBy: (int) adjustment {
  if (adjustment > 0)
    adjustment = [self addRandomSubstrate: adjustment];
  else
    if (adjustment < 0)
      adjustment = -([self removeRandomSubstrate: (-adjustment)]);

  return adjustment;
}


-(int) addRandomLink: (int) numNewLinks {
  int x, y;
  int numHoles;
  int newLinkCount;
  Hole *hole;
  Link *link;

  [self update]; // Defensive...
  newLinkCount = 0;
  numHoles = [holeMgr getCount];
  while ((newLinkCount < numNewLinks) && (numHoles > 0)) {
    hole = [holeMgr randomAgent];
    x = [hole getX];
    y = [hole getY];
    [holeMgr dropAgent: hole];
    numHoles--;
    link = [linkMgr createAgent];
    [link unwarpToX: x Y: y];
    newLinkCount++;
    [holeMgr processQueues];
    [linkMgr processQueues];
  }

  [linkMgr processQueues];

  [self update];

  return newLinkCount;
}

-(int) removeRandomLink: (int) numRemovedLinks {
  int x, y;
  int numLinks;
  int removedLinkCount;
  Hole *hole;
  Link *link;

  [self update]; // Defensive...
  removedLinkCount = 0;
  numLinks = [linkMgr getCount];
  while ((removedLinkCount < numRemovedLinks) 
      && (numLinks > 0)) {
    link = [linkMgr randomAgent];
    x = [link getX];
    y = [link getY];
    [linkMgr dropAgent: link];
    hole = [holeMgr createAgent];
    [hole unwarpToX: x Y: y];
    numLinks--;
    removedLinkCount++;
    [linkMgr processQueues];
    [holeMgr processQueues];
  }

  [linkMgr processQueues];

  [self update];

  return removedLinkCount;
}

-(int) adjustLinkNumberBy: (int) adjustment {
  if (adjustment > 0)
    adjustment = [self addRandomLink: adjustment];
  else
    if (adjustment < 0)
      adjustment = -([self removeRandomLink: (-adjustment)]);

  return adjustment;
}

-fillWithHoles {
  int x, y;
  int xSize, ySize;
  id agent;
  Hole *hole;

  xSize = [world getSizeX];
  ySize = [world getSizeY];

  for (x = 0; x < xSize ; x++)
    for (y = 0; y < ySize; y++) {
      agent = [world getObjectAtX: x Y: y];
      if ([agent class] != [Hole class]) {
        [self dropAgent: agent];
	hole = [holeMgr createAgent];
	[hole unwarpToX: x Y: y];
      }
    }

  [self update];

  return self;
}


-(int) addRandomCatalyst: (int) numNewCatalysts {
  int x, y;
  int numHoles;
  int newCatalystCount;
  Hole *hole;
  Catalyst *catalyst;

  [self update]; // Defensive...
  newCatalystCount = 0;
  numHoles = [holeMgr getCount];
  while ((newCatalystCount < numNewCatalysts) && (numHoles > 0)) {
    hole = [holeMgr randomAgent];
    x = [hole getX];
    y = [hole getY];
    [holeMgr dropAgent: hole];
    numHoles--;
    catalyst = [catalystMgr createAgent];
    [catalyst unwarpToX: x Y: y];
    newCatalystCount++;
    [holeMgr processQueues];
    [catalystMgr processQueues];
  }

  [catalystMgr processQueues];

  [self update];

  return newCatalystCount;
}

-(int) removeRandomCatalyst: (int) numRemovedCatalysts {
  int x, y;
  int numCatalysts;
  int removedCatalystCount;
  Hole *hole;
  Catalyst *catalyst;

  [self update]; // Defensive...
  removedCatalystCount = 0;
  numCatalysts = [catalystMgr getCount];
  while ((removedCatalystCount < numRemovedCatalysts) 
      && (numCatalysts > 0)) {
    catalyst = [catalystMgr randomAgent];
    x = [catalyst getX];
    y = [catalyst getY];
    [catalystMgr dropAgent: catalyst];
    hole = [holeMgr createAgent];
    [hole unwarpToX: x Y: y];
    numCatalysts--;
    removedCatalystCount++;
    [catalystMgr processQueues];
    [holeMgr processQueues];
  }

  [catalystMgr processQueues];

  [self update];

  return removedCatalystCount;
}

-(int) adjustCatalystNumberBy: (int) adjustment {
  if (adjustment > 0)
    adjustment = [self addRandomCatalyst: adjustment];
  else
    if (adjustment < 0)
      adjustment = -([self removeRandomCatalyst: (-adjustment)]);

  return adjustment;
}


-createDefaultWorld {
  int i;
  int x, y;
  Particle *particle;
  Substrate *substrate;
  Catalyst *catalyst;
  BOOL holeFound;

  int initialNumSubstrates = 200;
  int initialNumCatalysts = 5;
  int xSize = 20;
  int ySize = 20;

  [self createEmptyWorldWithX: xSize Y: ySize];

  // Add the substrates...
  i = 0;
  while (i < initialNumSubstrates) {
    // Find a hole...
    holeFound = NO;
    while(!holeFound) {
      // This is going to get very slooooow if the 
      // material particle density in the world gets anywhere
      // near 1.0!!!
      x = [prng getUniformWithRange: xSize];
      y = [prng getUniformWithRange: ySize];
      particle = [world getObjectAtX: x Y: y];
      if ([particle isMemberOf: [Hole class]])
        holeFound = YES;
    }

    [holeMgr dropAgent: particle];
    substrate = [substrateMgr createAgent];
    [substrate unwarpToX: x Y: y];

    i++;
  }
  [holeMgr processQueues];
  [substrateMgr processQueues];

  // Add the catalysts...
  i = 0;
  while (i < initialNumCatalysts) {
    // Find a hole...
    holeFound = NO;
    while(!holeFound) {
      // This is going to get very slooooow if the 
      // material particle density in the world gets anywhere
      // near 1.0!!!
      x = [prng getUniformWithRange: xSize];
      y = [prng getUniformWithRange: ySize];
      particle = [world getObjectAtX: x Y: y];
      if ([particle isMemberOf: [Hole class]])
        holeFound = YES;
    }

    [holeMgr dropAgent: particle];
    catalyst = [catalystMgr createAgent];
    [catalyst unwarpToX: x Y: y];

    i++;
  }
  [holeMgr processQueues];
  [catalystMgr processQueues];

  [userCellEditor update];
  [self refreshDisplay];

  return self;
}

-saveTo: (id <OutFile>) outFile {
  [outFile putString: "# World state...\n"];

  [prng saveTo: outFile];

  [outFile putString: "# World Time...\n"];
  [outFile putInt: [worldManager getWorldTime]];  // Really need putUnsigned!
  [outFile putNewLine];

  [outFile putString: "# World Size...\n"];
  [outFile putInt: [world getSizeX]];
  [outFile putString: " "];
  [outFile putInt: [world getSizeY]];
  [outFile putNewLine];

  [holeMgr saveTo: outFile];
  [substrateMgr saveTo: outFile];
  [linkMgr saveTo: outFile];
  [catalystMgr saveTo: outFile];

  [bondMgr saveTo: outFile];

  return self;
}


-loadFrom: (id <InFile>) inFile {
  int intParm;
  int xSize, ySize;
  // char c;

  [self dropWorld];

  [inFile skipLine]; // "# World state...\n" 

  //   [inFile getChar: &c];
  // while (c != '\n') {
  //   putchar(c);
  //   [inFile getChar: &c];
  //  }
  // fflush(stdout);

  [prng loadFrom: inFile];

  [inFile skipLine]; // "# World Time...\n"

  [inFile getInt: &intParm]; // Really need getUnsigned!
  [worldManager setWorldTime: intParm];
  [inFile skipLine];

  [inFile skipLine]; // "# World Size...\n"
  [inFile getInt: &xSize];
  [inFile getInt: &ySize];
  [inFile skipLine];
  
  // OK, now we have the wherewithall to create the "empty"
  // configuration...
  [self createWorldWithX: xSize Y: ySize];

  [holeMgr loadFrom: inFile];
  [substrateMgr loadFrom: inFile];
  [linkMgr loadFrom: inFile];
  [catalystMgr loadFrom: inFile];

  [bondMgr loadFrom: inFile];

  [self update];
 
  return self;
}

-saveToFileNamed: (char *) filename {
  id <OutFile> outFile;

  [modelSwarm setStateStopped]; // Defensive...

  outFile = [OutFile create: [self getZone] withName: filename];

  [outFile putString: "# "];
  [outFile putString: [Version getVersionString]];
  [outFile putNewLine];

  [self saveTo: outFile];
 
  [outFile drop];

  return self;
}

-loadFromFileNamed: (char *) filename {
  id <InFile> inFile;

  [modelSwarm setStateStopped]; // Defensive...

  inFile = [InFile create: [self getZone] withName: filename];

  [inFile skipLine]; // Version info...
  // Possibly should check for compatibility (;-)

  [self loadFrom: inFile];

  [inFile drop];

  return self;
}

@end
