/* 
  Particle.m

  Barry McMullin <mcmullin@eeng.dcu.ie>
  27-SEP-1996

*/

#import <math.h>
#import <simtools.h>

#import "PRNG.h"
#import "SCLGlobals.h"
#import "Particle.h"

@implementation Particle


static double defaultMobilityFactor = 1.0;
static Color defaultColor = 1;
/*
  These are sortof, kindof, class variables (as opposed to
  instance variables). They should only be accessed
  through the relevant +set and +get *class* methods. If 
  subclasses want have something
  other than the default values they will have to
  provide their own class variables *and* override the
  relevant +set and +get methods...
*/

+(void) setClassMobilityFactor: (double) inMobilityFactor {
  defaultMobilityFactor = inMobilityFactor;
}

+(double) getClassMobilityFactor {
  return(defaultMobilityFactor);
}

+(void) setClassColor: (Color) color {
  defaultColor = color;
}

+(Color) getClassColor {
  return(defaultColor);
}

-createEnd {
  Particle *finalSelf;

  finalSelf = [super createEnd];
  [finalSelf setParticleColor: [[finalSelf class] getClassColor]];

  return finalSelf;
}

-setParticleColor: (Color) c {
  particleColor = c;
  return self;
}

-(BOOL) canMove {
  return (!hasMoved);
}

-setHasMoved {
  hasMoved = YES;

  return self;
}

-initialiseStep {
  hasMoved = NO;

  return self;
}

-doMotion {

  /* NB: It's important to use [isa getClassMobilityFactor] here
     (rather than referring directly to defaultMobilityFactor)
     because we may be executing -doMotion in a subclass with
     a different mobilityFactor (i.e. where +getMobilityFactor
     has been overridden). */

  neighbor_t neighbor;
  Particle *neighborParticle;
  double motionProbability;

  if ([self canMove]) {
    neighbor = [vnNeighbor getRandom];
    /* *Always* use von Neumann neighborhood for motion -
       using the Moore neighborhood would make it much harder
       to implement the impermeability of links... */

    neighborParticle = [self getAgentAtNeighbor: neighbor];

    if (neighborParticle == nil) 
      [InternalError raiseEvent:
	 "nil encountered at neighbor %d of particle %d.\n",
	 neighbor, self];
    else {
      if ([neighborParticle canMove]) {
	motionProbability = 
	  sqrt([isa getClassMobilityFactor] * 
	  [[neighborParticle class] getClassMobilityFactor]);
	if ([prng getTossWithProb: motionProbability]) {
	  [self swapWithAgent: neighborParticle];
          hasMoved = YES;
          [neighborParticle setHasMoved];
	}
      }
    }
  }

  return self;
}

-step {
  [self doMotion]; // Only default behaviour is doMotion.

  return self;
}


-drawSelfOn: (id <Raster>) r {
  /* 
    Default particle is a 3x3 square.
    Subclasses may override to get different shapes...
    */
  int rasterX, rasterY;
  int i,j;

  rasterX = 8 * x;
  rasterY = 8 * y;

  for (i = 0; i < 5; i++) {
    for (j = 0; j < 5; j++) {
      [r drawPointX: rasterX Y: rasterY Color: particleColor];
      rasterY++;
    }
  rasterY -= 5;
  rasterX++;
  }

  return self;
}


@end
