// file: AgentObserverSwarm.m
// swarm room 1996 

#import "AgentObserverSwarm.h"


@implementation AgentObserverSwarm

// createBegin: here we set up the default observation parameters.
+createBegin: (id) aZone {
  AgentObserverSwarm * obj;
  
  // Superclass createBegin to allocate ourselves.
  obj = [super createBegin: aZone];

  // Fill in the relevant parameters (only one, in this case).
  obj->displayFrequency = 1;
  obj->zoomFactor = 3;

  [self setAgentObserverSwarmProbeMap: aZone];
  [self setAgentModelSwarmProbeMap: aZone];
  [self setCellProbeMap: aZone];
  [self setAgentProbeMap: aZone];
  
  return obj;
}

// make some probe maps

+setAgentObserverSwarmProbeMap: (id) aZone {
  ProbeMap * probeMap;

  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [self class]];
  probeMap = [probeMap createEnd];

  // Add in a bunch of variables, one per simulation parameters
  [probeMap addProbe: [probeLibrary getProbeForVariable: "displayFrequency"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "zoomFactor"
				    inClass: [self class]]];  
  [probeMap addProbe: [probeLibrary getProbeForMessage: "displayMaize"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForMessage: "displayElevation"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForMessage: "displaySoil"
				    inClass: [self class]]];  
  [probeMap addProbe: [probeLibrary getProbeForMessage: "displayWater"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForMessage: "displayElevWater"
				    inClass: [self class]]];
  [probeMap addProbe: [probeLibrary getProbeForMessage: "displaySoilWater"
				    inClass: [self class]]];
  
  // Now install our custom probeMap into the probeLibrary.
  [probeLibrary setProbeMap: probeMap For: [self class]];
  return self;
}

+setAgentModelSwarmProbeMap: (id) aZone {
  ProbeMap * probeMap;

  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [AgentModelSwarm class]];
  probeMap = [probeMap createEnd];

  // Add in a bunch of variables, one per simulation parameter
  [probeMap addProbe: [probeLibrary getProbeForVariable: "allAgents"
				    inClass: [AgentModelSwarm class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldXSize"
				    inClass: [AgentModelSwarm class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "worldYSize"
				    inClass: [AgentModelSwarm class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "which_yield"
				    inClass: [AgentModelSwarm class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "random_yield"
				    inClass: [AgentModelSwarm class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "time"
				    inClass: [AgentModelSwarm class]]];

  // Now install our custom probeMap into the probeLibrary.
  [probeLibrary setProbeMap: probeMap For: [AgentModelSwarm class]];
  return self;
}

+setCellProbeMap: (id) aZone {
  ProbeMap * probeMap;
  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [Cell class]];
  probeMap = [probeMap createEnd];
  
  // Add in a bunch of variables, one per simulation parameters
  [probeMap addProbe: [probeLibrary getProbeForVariable: "x"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "y"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "farming_plots"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "maize_pot"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "water"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "elevation"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "soil"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForMessage: "getNumHouses"
				    inClass: [Cell class]]];
  [probeMap addProbe: [probeLibrary getProbeForMessage: "probeSettlers"
				    inClass: [Cell class]]];  
  
  // Now install our custom probeMap into the probeLibrary.
  [probeLibrary setProbeMap: probeMap For: [Cell class]];
  return self;
}

+setAgentProbeMap: (id) aZone {
  ProbeMap * probeMap;
  probeMap = [EmptyProbeMap createBegin: aZone];
  [probeMap setProbedClass: [Agent class]];
  probeMap = [probeMap createEnd];
  
  // Add in a bunch of variables, one per simulation parameters
  [probeMap addProbe: [probeLibrary getProbeForVariable: "tag"
				    inClass: [Agent class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "x"
				    inClass: [Agent class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "y"
				    inClass: [Agent class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "num_family"
				    inClass: [Agent class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "maize_storage"
				    inClass: [Agent class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "tot_plots"
				    inClass: [Agent class]]];
  [probeMap addProbe: [probeLibrary getProbeForVariable: "plot_need"
				    inClass: [Agent class]]];
  
  // Now install our custom probeMap into the probeLibrary.
  [probeLibrary setProbeMap: probeMap For: [Agent class]];
  return self;
}


// createEnd: create objects we know we'll need. In this case, none,
// but you might want to override this.
-createEnd {
  return [super createEnd];
}


// Create the objects used in the display of the model. This code is
// fairly complicated because we build a fair number of widgets. It's
// also a good example of how to use the display code.
-buildObjects {
  id modelZone;					  // zone for model.
  int i;
  Agent * agentToProbe;

  [super buildObjects];
  
  // First, we create the model that we're actually observing. The
  // model is a subswarm of the observer. We also create the model in
  // its own zone, so storage is segregated.
  modelZone = [Zone create: [self getZone]];
  agentModelSwarm = [AgentModelSwarm create: modelZone];
  
  // Now create probe objects on the model and ourselves. This gives a
  // simple user interface to let the user change parameters.
  [probeDisplayManager createProbeDisplayFor: agentModelSwarm];
  [probeDisplayManager createProbeDisplayFor: self];

  // Instruct the control panel to wait for a button event: we halt here
  // until someone hits a control panel button so the user can get a
  // chance to fill in parameters before the simulation runs
  [controlPanel waitForControlEvent];
  // Check now if the user hit the quit button: if so, abort.
  if ([controlPanel getState] == ControlStateQuit)
    return self;


  // OK - the user has specified all the parameters for the simulation.
  // Now we're ready to start.

  // First, let the model swarm build its objects.
  [agentModelSwarm buildObjects];

  // Now get down to building our own display objects.

  // First, create a colormap: this is a global resource, the information
  // here is used by lots of different objects.
  colormap = [XColormap create: [self getZone]];

  // set up 64 colors for maize fertility: 0-63
  for (i = 0; i < 64; i++)
    [colormap setColor: i ToRed: 0 Green: (double) (i+10.0)/74.0 Blue: 0.2];

  // set up 64 colors for soil types: 64-127
  for (i=0;i<64;i++)
    [colormap setColor: 64+i ToRed: (double) (i+10.0)/74.0
	      Green: (double) (i+10.0)/74.0 Blue: 0.1];    

  // set up 64 colors for elevation: 128-191
  for (i=0;i<64;i++)
    [colormap setColor: 128+i ToRed: 0.4 Green: (double) (i+10.0)/74.0 Blue: 0.6];

  // set up 8 colors for water availability: 192-199
  for (i=0;i<8;i++)
    [colormap setColor: 192+i ToRed: 0.0 Green: 0.1 Blue: (double) (i+2.0)/10.0];
 
  // colors for the agents
  [colormap setColor: 200 ToName: "red"];
  [colormap setColor: 201 ToName: "firebrick"];
  [colormap setColor: 202 ToName: "white"];
  [colormap setColor: 203 ToName: "yellow"];

  // now set the agents color to red for easy viewing
  [[agentModelSwarm getAgentList] forEach: M(setColor:) : (void *) 200];
  
  // Next, create a 2d window for display, set its size, zoom factor, title.
  worldRaster = [ZoomRaster create: [self getZone]];
  [worldRaster setColormap: colormap];
  [worldRaster setZoomFactor: zoomFactor];
  [worldRaster setWidth: [[agentModelSwarm getWorld] getSizeX]
	       Height: [[agentModelSwarm getWorld] getSizeY]];
  [worldRaster setWindowTitle: "Anasazi World"];
  [worldRaster pack];				  // draw the window

  // And also create an Object2dDisplay: this object draws agents on
  // display the cells
  cellDisplay = [Object2dDisplay createBegin: [self getZone]];
  [cellDisplay setDisplayWidget: worldRaster];       // which widget
  [cellDisplay setDiscrete2dToDisplay: [agentModelSwarm getWorld]];
  [cellDisplay setObjectCollection: [agentModelSwarm getCellList]];
  [cellDisplay setDisplayMessage: M(drawSelfOn:)];   // object's draw method
  cellDisplay = [cellDisplay createEnd];

  // displays the agents
  agentDisplay = [Object2dDisplay createBegin: [self getZone]];
  [agentDisplay setDisplayWidget: worldRaster];       // which widget
  [agentDisplay setDiscrete2dToDisplay: [agentModelSwarm getWorld]];    // what world
  [agentDisplay setObjectCollection: [agentModelSwarm getAgentList]];
  [agentDisplay setDisplayMessage: M(drawSelfOn:)];   // object's draw method
  agentDisplay = [agentDisplay createEnd];

  // Also, tell the world raster to send mouse clicks to the cellDisplay
  // this allows the user to right-click on the display to probe the bugs.
  [worldRaster setButton: ButtonRight Client: cellDisplay
	       Message: M(makeProbeAtX:Y:)];

  // have globally defined GRAPH if want to have display of these graphs
  if (GRAPH) {
    // Create the graph widget to display num plots
    avgGraph = [EZGraph createBegin: [self getZone]];
    [avgGraph setTitle: "Average size of plots and family vs. time"];
    [avgGraph setAxisLabelsX: "time" Y: "average size"];
    avgGraph = [avgGraph createEnd];
  
    // graph avg number of plots
    [avgGraph createAverageSequence: "plots"
	      withFeedFrom: [agentModelSwarm getAgentList]
	      andSelector: M(getNumPlots)];

    // graph avg mumber in household
    [avgGraph createAverageSequence: "family"
	      withFeedFrom: [agentModelSwarm getAgentList]
	      andSelector: M(getFamilySize)];
    
    // Create the graph widget to display yield
    yieldGraph = [EZGraph createBegin: [self getZone]];
    [yieldGraph setTitle: "Yields Actual/Average vs. time"];
    [yieldGraph setAxisLabelsX: "time" Y: "yield"];
    yieldGraph = [yieldGraph createEnd];

    // graph average yield realized by agents
    [yieldGraph createAverageSequence: "average"
		withFeedFrom: [agentModelSwarm getAgentList]
		andSelector: M(getYield)];

    // graph avg yield over all cells
    [yieldGraph createAverageSequence: "actual"
		withFeedFrom: [agentModelSwarm getCellList]
		andSelector: M(getMaizePotential)];

    // Create the graph widget to display num plots
    popGraph = [EZGraph createBegin: [self getZone]];
    [popGraph setTitle: "Number Households Simulated/Estimated vs. time"];
    [popGraph setAxisLabelsX: "time" Y: "households"];
    popGraph = [popGraph createEnd];
  
    // graph num of agents simulated
    [popGraph createSequence: "simulated"
	      withFeedFrom: agentModelSwarm
	      andSelector: M(getNumAgents)];

    // graph num of agents estimated
    [popGraph createSequence: "estimated"
	      withFeedFrom: agentModelSwarm
	      andSelector: M(getEstAgents)];
  }

  // Finally, we create a Probe display to probe a particular heatbug.
  // Probes can also be created on the fly, we just do this here for demo.
  agentToProbe = [[[agentModelSwarm getAgentList] begin: [self getZone]] next];
  [agentToProbe setColor: 202];
  [probeDisplayManager createProbeDisplayFor: agentToProbe];
  
  // All done - we're ready to build a schedule and go.
  return self;
}  

// Create the actions necessary for the simulation. This is where
// the schedule is built (but not run!)
// Here we create a display schedule - this is used to display the
// state of the world and check for user input. This schedule should
// be thought of as independent from the model - in particular, you
// will also want to run the model without any display.
-buildActions {
  [super buildActions];
  
  // First, let our model swarm build its own schedule.
  [agentModelSwarm buildActions];
  
  // Create an ActionGroup for display: a bunch of things that occur in
  // a specific order, but at one step of simulation time. Some of these
  // actions could be executed in parallel, but we don't explicitly
  // notate that here.
  displayActions = [ActionGroup create: [self getZone]];
  // Schedule up the methods to draw the display of the world
  [displayActions createActionTo: cellDisplay message: M(display)];
  [displayActions createActionTo: agentDisplay message: M(display)];
  [displayActions createActionTo: worldRaster message: M(drawSelf)];

  // Now schedule the update of graphs
  if (GRAPH) {
    [displayActions createActionTo: avgGraph  message: M(step)];
    [displayActions createActionTo: yieldGraph  message: M(step)];
    [displayActions createActionTo: popGraph  message: M(step)];
  }
  
  // Schedule the update of the probe displays
  [displayActions createActionTo: probeDisplayManager message: M(update)];
  // Finally, schedule an update for the whole user interface code.
  // This is crucial: without this, no graphics update and the control
  // panel will be dead. It's best to put it at the end of the display schedule
  [displayActions createActionTo: controlPanel        message: M(doTkEvents)];

  // And the display schedule. Note the repeat interval is set from our
  // own Swarm data structure. Display is frequently the slowest part of a
  // simulation, so redrawing less frequently can be a help.
  displaySchedule = [Schedule createBegin: [self getZone]];
  [displaySchedule setRepeatInterval: displayFrequency]; // note frequency!
  displaySchedule = [displaySchedule createEnd];
  [displaySchedule at: 0 createAction: displayActions];
  
  return self;
}  

// activateIn: - activate the schedules so they're ready to run.
// The swarmContext argument has to do with what we were activated *in*.
// Typically the ObserverSwarm is the top-level Swarm, so it's activated
// in "nil". But other Swarms and Schedules and such will be activated
// inside of us.
-activateIn: (id) swarmContext {
  // First, activate ourselves (just pass along the context).
  [super activateIn: swarmContext];

  // Activate the model swarm in ourselves. The model swarm is a
  // subswarm of the observer swarm.
  [agentModelSwarm activateIn: self];

  // Now activate our schedule in ourselves. This arranges for the
  // execution of the schedule we built.
  [displaySchedule activateIn: self];
  
  // Activate returns the swarm activity - the thing that's ready to run.
  return [self getActivity];
}

// You could override the "go" method here if you want something special
// to happen when the model and observer actually start running. But
// the default GUISwarm go is probably good enough.

// more display stuff
-(int) displayMaize {
  [[agentModelSwarm getCellList] forEach: M(setDisplay:) : (void *) 0];
  return 0;
}

-(int) displaySoil {
  [[agentModelSwarm getCellList] forEach: M(setDisplay:) : (void *) 1];
  return 1;
}

-(int) displayElevation {
  [[agentModelSwarm getCellList] forEach: M(setDisplay:) : (void *) 2];
  return 2;
}

-(int) displayWater {
  [[agentModelSwarm getCellList] forEach: M(setDisplay:) : (void *) 3];
  return 3;
}

-(int) displayElevWater {
  [[agentModelSwarm getCellList] forEach: M(setDisplay:) : (void *) 4];
  return 4;
}

-(int) displaySoilWater {
  [[agentModelSwarm getCellList] forEach: M(setDisplay:) : (void *) 5];
  return 5;
}

@end
