// file: Cell.m
// defines characteristics of each cell

#import "Cell.h"
#import <simtoolsgui.h>

@implementation Cell

// creation stuff
// initialization stuff
- setWorld: (id <Grid2d>)w
{
  world = w;
  return self;
}

- setX: (int)inx Y: (int)iny
{
  x = inx;
  y = iny;
  [world putObject: self atX: x Y: y];	  // add cell to the world
  return self;
}

- setSettlerSet: a
{
  settlerSet = a;
  return self;
}

- createEnd
{
  // init cell state, no usage yet
  farming_plots = 0;
  if (settlerSet == nil)
    {
      fprintf(stderr,"ERROR: (%d,%d) no settler cell exists\n",x,y);
    exit(-1);
    }
  return self;
}


// agent manipulation
- addSettler: agent
{
  [settlerSet add: agent];
  return self;
}

- removeSettler: agent
{
  return ([settlerSet remove: agent]);
}

- probeSettlers
{
  if ([self getNumHouses] > 0)
    {
      id index;
      Agent *agentToProbe;

      index = [settlerSet begin: [self getZone]];
      while ((agentToProbe = [index next]) != nil)
        [probeDisplayManager createProbeDisplayFor: agentToProbe];
    }
  return self;
}


// set cell state

- setMaizePotential: (int)veg
{
  // units read are SPUB (lbs beans per acre), we want SPUM
  // where SPUM = (SPUB*10+4)*2.368 (gives yield in kg maize 
  // for 1 ha [VW 1994:123])
  if (veg > -1)
    maize_pot = (int)((veg*10)+4) * 2.36775;
  else
    {
      fprintf(stderr,"ERROR: (%d,%d) maize potential cannot be negative %d\n",x,y,veg);
      exit(-1);
    }
  return self;
}

- setWaterType: (int)w
{
  //  fprintf(stderr,"TEST: (%d,%d) set water type %d\n",x,y,w);
  water = w;
  return self;
}

- setElevation: (int)height
{
  //  fprintf(stderr,"TEST: (%d,%d) set elevation %d\n",x,y,height);
  elevation = height;
  return self;
}

- setSoilType: (int)s
{
  //  fprintf(stderr,"TEST: (%d,%d) set soil type %d\n",x,y,s);
  soil = s;
  return self;
}

// farming info

- setFarmPl: (int)num
{
  farming_plots =  num;
  return self;
}

- changeFarmPl: (int)num
{
  farming_plots +=  num;
  if (farming_plots < 0)
    {
      fprintf(stderr,"ERROR: (%d,%d) farm ha less than than 0 (have %d, want %d)\n",
              x,y,farming_plots,num);
      exit(-1);
    }
  if (farming_plots > 10)
    {
      fprintf(stderr,"ERROR: (%d,%d) farm ha greater than 4 (have %d, want %d)\n",
              x,y,farming_plots,num);
      exit(-1);
    }
  return self;
}

// get cell state

- (id)getSettlerSet
{
  return (settlerSet);
}

- (int)getMaizePotential
{
  return (maize_pot);
}

- (int)getWaterType
{
  return (water);
}

- (int)getElevation
{
  return (elevation);
}

- (int)getSoilType
{
  return (soil);
}

- (int)getFarmPl
{
  return (farming_plots);
}

- (int)getNumHouses
{
  return ([settlerSet count]);
}


// display
- setDisplay: (int)d
{
  // 0 displays maize potential
  // 1 displays water
  // 2 displays elevation
  // 3 displays soil
  // 4 displays elevation and water
  // 5 displays soil and water
  display = d;
  return self;
}

- drawSelfOn: (id <Raster>)r
{
  int t;

  switch (display)
    {
    case 0:
      t = (maize_pot-50) / 20;
      if (t<0) t = 0;
      if (t>63) t = 63;
      [r drawPointX: x Y: y Color: t];
      break;
    case 1:
      t = soil / 2;
      if (t>63) t = 63;
      [r drawPointX: x Y: y Color: 64+t];
      break;
    case 2:
      t = elevation / 2;
      if (t>63) t = 63;
      [r drawPointX: x Y: y Color: 128+t];
      break;
    case 3: 
      t = (water<<1)|1;
      [r drawPointX: x Y: y Color: 192+t];
      break;
    case 4:
      t = elevation / 2;
      if (t>63) t = 63;
      [r drawPointX: x Y: y Color: 128+t];
      if (water >0){
	t = (water<<1)|1;
	[r drawPointX: x Y: y Color: 192+t];
      }
      break;
    case 5:
      t = soil / 2;
      if (t>63) t = 63;
      [r drawPointX: x Y: y Color: 64+t];
      if (water >0){
	t = (water<<1)|1;
	[r drawPointX: x Y: y Color: 192+t];
      }
      break;
    default:
      fprintf(stderr,"ERROR: (%d,%d) display for cell not set correctly\n",x,y);
      exit(-1);
    }
  return self;
}

// debuggging stuff
- debugX: (int)inx Y: (int)iny
{
  if ((x == inx && y == iny) || inx == -1 )
    {
      printf("x: %d y: %d settlers: %d planted: %d\n",
             x,y,[settlerSet count],farming_plots);
    }
  return self;
}

@end
