/*
 * Copyright (C) 2008-2014 Free Software Foundation, Inc.
 * Copyright (C) 2017 Red Hat, Inc.
 *
 * Authors: Simon Josefsson, Nikos Mavrogiannopoulos, Martin Ukrop
 *
 * This file is part of GnuTLS.
 *
 * GnuTLS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuTLS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>
 */

#ifndef GNUTLS_TESTS_TEST_CHAINS_H
# define GNUTLS_TESTS_TEST_CHAINS_H

/* *INDENT-OFF* */

#define MAX_CHAIN 10

static const char *chain_with_no_subject_id_in_ca_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIFVDCCBDygAwIBAgIQR+AAAAAAXtwVoBdbGUjUQDANBgkqhkiG9w0BAQsFADBv\n"
	"MQswCQYDVQQGEwJUVzESMBAGA1UEChMJVEFJV0FOLUNBMRowGAYDVQQLExFTZWN1\n"
	"cmUgU1NMIFN1Yi1DQTEwMC4GA1UEAxMnVFdDQSBTZWN1cmUgU1NMIENlcnRpZmlj\n"
	"YXRpb24gQXV0aG9yaXR5MB4XDTE2MDcyNjAyMTY1NloXDTE5MDcyNjE1NTk1OVow\n"
	"gaAxCzAJBgNVBAYTAlRXMQ8wDQYDVQQIEwZUYWl3YW4xDzANBgNVBAcTBlRhaXBl\n"
	"aTEjMCEGA1UEChMaTmF0aW9uYWwgVGFpd2FuIFVuaXZlcnNpdHkxMzAxBgNVBAsT\n"
	"KkNvbXB1dGVyIGFuZCBJbmZvcm1hdGlvbiBOZXR3b3JraW5nIENlbnRlcjEVMBMG\n"
	"A1UEAwwMKi5udHUuZWR1LnR3MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\n"
	"AQEAxs5vKRdaicZTmxC+vY6ZZ/wCJNxFJGsRTlyRmDHr/fXD6mTcZSjsRY+LpAbF\n"
	"RqPKj5zuQtXdk9j4UGieajWSqmDNZy4+5gEqmjgU+vrYCK+uWHwAzdDCSWgGv29/\n"
	"2/QOyl22OkpdiFv0wf00Rz86l9Oua3Zml8LW/LB8JO5w15yhlo+VgMy7mFFaBItG\n"
	"sV5IEq8CFfaz0+T7/Bf17u2Ckl1jLKIJHp5Qm0FrPA4a0KEVg9RNYeo5evieI9et\n"
	"UpAXO2EEHsakg+8yBrm4llRt7onb32hiZvAstak2FaHPRHdZtDQY1QrmTBYgPdzx\n"
	"sOV5bE+NTeE04eEA22g7HbQsRwIDAQABo4IBuDCCAbQwHwYDVR0jBBgwFoAU+AfC\n"
	"aCT/hZXL2x7jM5wqT5cgVnswKQYDVR0OBCIEIGjmBQmIw+x+peJLZ3wJMHsAsbyN\n"
	"kz5qkj/ZA5UH0pu2MFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9zc2xzZXJ2ZXIu\n"
	"dHdjYS5jb20udHcvc3Nsc2VydmVyL1NlY3VyZXNzbF9yZXZva2Vfc2hhMl8yMDE0\n"
	"LmNybDAXBgNVHREEEDAOggwqLm50dS5lZHUudHcwgYEGCCsGAQUFBwEBBHUwczBE\n"
	"BggrBgEFBQcwAoY4aHR0cDovL3NzbHNlcnZlci50d2NhLmNvbS50dy9jYWNlcnQv\n"
	"c2VjdXJlX3NoYTJfMjAxNC5jcnQwKwYIKwYBBQUHMAGGH2h0dHA6Ly90d2Nhc3Ns\n"
	"b2NzcC50d2NhLmNvbS50dy8wNwYDVR0gBDAwLjAsBgsrBgEEAYK/JQEBGTAdMBsG\n"
	"CCsGAQUFBwIBFg93d3cudHdjYS5jb20udHcwCQYDVR0TBAIwADAOBgNVHQ8BAf8E\n"
	"BAMCBaAwHQYDVR0lBBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMA0GCSqGSIb3DQEB\n"
	"CwUAA4IBAQC7AMzY2nvycER/Kh5gZltVcWhOYotMJoLcsdw5Vl1drauE4eK6hc/E\n"
	"vB6YeSRHVr5+XTksSHRj4HXgaUBYb/llOstOwF5sygwOneMj8TF7KP8Lz1OMbUao\n"
	"iiitO5MOgfx5bQeWrUAwQerbZDp0ApDQcJfHG8HfUhJhYiXLWoXlb41SFwSSUtk5\n"
	"VRqAIdjlg9wxhxP+0qz6llhL2ycJfrgX+eZyceoeoeaLTEpiev2jWfcTwmuz7CMn\n"
	"ggIDPuvJ5cZ5Sh9hvzfNP7Vp2s+dV4idydwUTzZv0oC60/UQs7HUpjThno0S9h2u\n"
	"s42h0Usx0Dxa7Mr8ddLLort1k2hlHHko\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIFsTCCA5mgAwIBAgIQQAEzU+QAAAAAAAAMw26IjTANBgkqhkiG9w0BAQsFADBR\n"
	"MQswCQYDVQQGEwJUVzESMBAGA1UEChMJVEFJV0FOLUNBMRAwDgYDVQQLEwdSb290\n"
	"IENBMRwwGgYDVQQDExNUV0NBIEdsb2JhbCBSb290IENBMB4XDTE0MTAyODA3Mjc1\n"
	"NloXDTI0MTAyODE1NTk1OVowbzELMAkGA1UEBhMCVFcxEjAQBgNVBAoTCVRBSVdB\n"
	"Ti1DQTEaMBgGA1UECxMRU2VjdXJlIFNTTCBTdWItQ0ExMDAuBgNVBAMTJ1RXQ0Eg\n"
	"U2VjdXJlIFNTTCBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCASIwDQYJKoZIhvcN\n"
	"AQEBBQADggEPADCCAQoCggEBANvg/lFTogGmz2qOXalRCWoWfkoUErMRSiYoGy8j\n"
	"UovBDasYVeOmXPLtGvVjLvqxaiKGYWbO0hzXj4RFaLwTM7gOJw5GaleMNj+PL4To\n"
	"1N2VKQxVAJ29LvpSJGvip4Vp0aRTMb3/xliIssln4fpVaA59AlwuAyN5Rrii6HZ4\n"
	"5kFT2defRirNpt2cBFDUkGkjfgP7htybTtwBFr8UfUQLsxtgd7QmC0ufAZVqkIaD\n"
	"NS6bgtLWxAUxUJooaBhNMryTC/YJgT16aA+n0/F0+NEeLjZk4R4v05XOm8lGeLKH\n"
	"aVZKrbCrC/bRD5Kr6QbhVQ4OHrmxk1muJ/1v5VGn57wFJEUCAwEAAaOCAWUwggFh\n"
	"MB8GA1UdIwQYMBaAFEjbzd6O6UlyWojosdg9B7O5a2ZQMB0GA1UdDgQWBBT4B8Jo\n"
	"JP+FlcvbHuMznCpPlyBWezAOBgNVHQ8BAf8EBAMCAQYwOAYDVR0gBDEwLzAtBgRV\n"
	"HSAAMCUwIwYIKwYBBQUHAgEWF2h0dHA6Ly93d3cudHdjYS5jb20udHcvMEkGA1Ud\n"
	"HwRCMEAwPqA8oDqGOGh0dHA6Ly9Sb290Q0EudHdjYS5jb20udHcvVFdDQVJDQS9n\n"
	"bG9iYWxfcmV2b2tlXzQwOTYuY3JsMBIGA1UdEwEB/wQIMAYBAf8CAQAwdgYIKwYB\n"
	"BQUHAQEEajBoMDwGCCsGAQUFBzAChjBodHRwOi8vc3Nsc2VydmVyLnR3Y2EuY29t\n"
	"LnR3L2NhY2VydC9yb290NDA5Ni5jcnQwKAYIKwYBBQUHMAGGHGh0dHA6Ly9yb290\n"
	"b2NzcC50d2NhLmNvbS50dy8wDQYJKoZIhvcNAQELBQADggIBAJ6AKBrUBHfH5JfL\n"
	"O38QwmyVYu3L36QsEdpkGhQaeHbpK/xLfr+amO4ZoxNolgwhm/vVWHFh2kjK+TXN\n"
	"/81YQljx8hjOx6Uy2JGEkVEKisW/QSl0bcRh4L4D62sZShnd8BplfMhTRwEJXZW3\n"
	"bR5yvy/DLrNuHRGPfrPb0v37L7nwdu+WMOUvv15KwV+2VlI+cnrREYWWAYixDm2N\n"
	"jK2yMCoUjEvwCcylnnwsg4CR8Il41eA7G1HZgC3qY4/Udgzu1igZcxRH46Hg4Hrq\n"
	"c8NRPuCsYRMoo2jh6sGz93BFaKY+DQbEvl/cH5qyKtzZvz7tLEsFkKRNkmcOfCQP\n"
	"77fMm8/aTpzzyjOvxNJ00XMDoxY7I+tSd/6Xxy7/vJ3VZQX4mFSg8NTlIp2ytoTv\n"
	"V1fus99V2NpLt93AX8ywDcOCma/rLqlT8Qu7nb5a+5r6sKMd81wfVWJYwncl5Ge1\n"
	"3L2goay+21EaFTgd0xp6B4BMRZ3TtAH/ZU133+78Pmt9ImasaJlPAh8UTBFwktbC\n"
	"CmKmiGvrZkLcvHXL0R18quyHY+6oZbKa1eZZsUcjDoErj8stId+W8QK7xpCbMYni\n"
	"w76NH/gy+5aJmxcQxMcyE3BkgMnMa4+TsIr2LSknvDmObW/mhMMAKHBfAQvMttLB\n"
	"iuqgTqRkGf4Y7kSH4hF/BbsO+9xn\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIFWTCCBEGgAwIBAgIQQAEzU+QAAAAAAAAMyl0baTANBgkqhkiG9w0BAQsFADBf\n"
	"MQswCQYDVQQGEwJUVzESMBAGA1UECgwJVEFJV0FOLUNBMRAwDgYDVQQLDAdSb290\n"
	"IENBMSowKAYDVQQDDCFUV0NBIFJvb3QgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
	"HhcNMTQxMDI4MDczODMxWhcNMzAxMDI4MTU1OTU5WjBRMQswCQYDVQQGEwJUVzES\n"
	"MBAGA1UEChMJVEFJV0FOLUNBMRAwDgYDVQQLEwdSb290IENBMRwwGgYDVQQDExNU\n"
	"V0NBIEdsb2JhbCBSb290IENBMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKC\n"
	"AgEAsAXbyOuMxG6KIe+OTZxxCh9ScO1tgpyXxddMTkVJy0BCtRI0bBnCdKQxX4UC\n"
	"l+xDMwpT0pyMjre4edsr1WryjmbE7isBB5LUs9AC31D2Va9mDsvgR2AvKzI5NVI6\n"
	"KIP4exbGGLhi1kclkc7wGRJNrWP10z91XynwoTAcKqCYphW97v0ZNvDikUOP+srW\n"
	"ECdJTO/dwfGFcJvK6qhaQ/xthm9z6TdFqfA2x8yIdR67bAb/m2s+F+xhqnF8xh2i\n"
	"90npFbU81qFh9RH3BW8d/RG+0DAHwimwCU4m3OOiqJFqH8KRRYhc5Zi4caUVGcl8\n"
	"dRHMcHRPLZsdkUT9Viig/ruGasj6XAtY3MZLdsirItlzD6X0WgKJP0+eIoLuonRT\n"
	"Kj1TJ2kdbI4yLGQAJmNhNk6jRrc/fbMtrG2QopWizs/agucHNBmW6bghqil+pji+\n"
	"jilKIWZ5H7PDtQln3tbUB0bzKtrmIjdgy4G2D6AP6ciVf79VkQV6zz0VwG/eCZQB\n"
	"g9c0G8xApfC4m2fVmJE7p4R4lSakWgj4K3S0AAQ837gUjujfqY1sZ5IzHcC30uyS\n"
	"yL4JvywpBW8Ca57vvL8qvFvAUI9BcHGHsk23BKmEozKvru5rF4uysf5s4ZCMiKiX\n"
	"SM7ITcvzBs9fagpCsR4edy+OoOaSDgb8BSLSJuExUX0y3A8CAwEAAaOCAR0wggEZ\n"
	"MB8GA1UdIwQYMBaAFGo4WyaN3ota8k96VIMZGOMINaa6MB0GA1UdDgQWBBRI283e\n"
	"julJclqI6LHYPQezuWtmUDAOBgNVHQ8BAf8EBAMCAQYwOAYDVR0gBDEwLzAtBgRV\n"
	"HSAAMCUwIwYIKwYBBQUHAgEWF2h0dHA6Ly93d3cudHdjYS5jb20udHcvMEIGA1Ud\n"
	"HwQ7MDkwN6A1oDOGMWh0dHA6Ly9Sb290Q0EudHdjYS5jb20udHcvVFdDQVJDQS9y\n"
	"ZXZva2VfMjA0OC5jcmwwDwYDVR0TAQH/BAUwAwEB/zA4BggrBgEFBQcBAQQsMCow\n"
	"KAYIKwYBBQUHMAGGHGh0dHA6Ly9yb290b2NzcC50d2NhLmNvbS50dy8wDQYJKoZI\n"
	"hvcNAQELBQADggEBACkLbsSU3GJZk3paTF3cmT6OqPv5oI8b2SdwbfguXkhpYRdA\n"
	"EImqA7gcy9+8bChUH8Uh9UuWv/xMR8oLd8PMZntvuTYIafnBke2P1p6iIuiCt4nI\n"
	"qtAg50qsIysvT/ZPFGvxekWhh8hx56ekuYBryc7NiiWczdMJpTL6JNFRfzwxmUfq\n"
	"H6dvboTNrtiuNWrWy+C+E8GiMcvuH+km37cGxg1FCuerRVcetwGbMfXyBUBFhtgC\n"
	"G9BLIdsgguMi6U/jXSvEORIXKMa5Z7F94bJ8duU2hNp0Xqw4tmv47qEDxLDEXBJM\n"
	"bwbaOkRltjaXDBh50EaXMx+rUqjO3le0KBM4t6w=\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIFQTCCAymgAwIBAgICDL4wDQYJKoZIhvcNAQELBQAwUTELMAkGA1UEBhMCVFcx\n"
	"EjAQBgNVBAoTCVRBSVdBTi1DQTEQMA4GA1UECxMHUm9vdCBDQTEcMBoGA1UEAxMT\n"
	"VFdDQSBHbG9iYWwgUm9vdCBDQTAeFw0xMjA2MjcwNjI4MzNaFw0zMDEyMzExNTU5\n"
	"NTlaMFExCzAJBgNVBAYTAlRXMRIwEAYDVQQKEwlUQUlXQU4tQ0ExEDAOBgNVBAsT\n"
	"B1Jvb3QgQ0ExHDAaBgNVBAMTE1RXQ0EgR2xvYmFsIFJvb3QgQ0EwggIiMA0GCSqG\n"
	"SIb3DQEBAQUAA4ICDwAwggIKAoICAQCwBdvI64zEbooh745NnHEKH1Jw7W2CnJfF\n"
	"10xORUnLQEK1EjRsGcJ0pDFfhQKX7EMzClPSnIyOt7h52yvVavKOZsTuKwEHktSz\n"
	"0ALfUPZVr2YOy+BHYC8rMjk1Ujoog/h7FsYYuGLWRyWRzvAZEk2tY/XTP3VfKfCh\n"
	"MBwqoJimFb3u/Rk28OKRQ4/6ytYQJ0lM793B8YVwm8rqqFpD/G2Gb3PpN0Wp8DbH\n"
	"zIh1HrtsBv+baz4X7GGqcXzGHaL3SekVtTzWoWH1EfcFbx39Eb7QMAfCKbAJTibc\n"
	"46KokWofwpFFiFzlmLhxpRUZyXx1EcxwdE8tmx2RRP1WKKD+u4ZqyPpcC1jcxkt2\n"
	"yKsi2XMPpfRaAok/T54igu6idFMqPVMnaR1sjjIsZAAmY2E2TqNGtz99sy2sbZCi\n"
	"laLOz9qC5wc0GZbpuCGqKX6mOL6OKUohZnkfs8O1CWfe1tQHRvMq2uYiN2DLgbYP\n"
	"oA/pyJV/v1WRBXrPPRXAb94JlAGD1zQbzECl8LibZ9WYkTunhHiVJqRaCPgrdLQA\n"
	"BDzfuBSO6N+pjWxnkjMdwLfS7JLIvgm/LCkFbwJrnu+8vyq8W8BQj0FwcYeyTbcE\n"
	"qYSjMq+u7msXi7Kx/mzhkIyIqJdIzshNy/MGz19qCkKxHh53L46g5pIOBvwFItIm\n"
	"4TFRfTLcDwIDAQABoyMwITAOBgNVHQ8BAf8EBAMCAQYwDwYDVR0TAQH/BAUwAwEB\n"
	"/zANBgkqhkiG9w0BAQsFAAOCAgEAXzSBdu+WHdXltdkCY4QWwa6gcFGn90xHNcgL\n"
	"1yg9iXHZqjNB6hQbbCEAwGxCGX6faVsgQt+i0trEfJdLjbDorMjupWkEmQqSpqsn\n"
	"LhpNgb+E1HAerUf+/UqdM+DyucRFCCEK2mlpc3INvjT+lIutwx4116KD7+U4x6WF\n"
	"H6vPNOw/KP4M8VeGTslV9xzU2KV9Bnpv1d8Q34FOIWWxtuEXeZVFBs5fzNxGiWNo\n"
	"RI2T9GRwoD2dKAXDOXC4Ynsg/eTb6QihuJ49CcdP+yz4k3ZB3lLg4VfSnQO8d57+\n"
	"nile98FRYB/e2guyLXW3Q0iT5/Z5xoRdgFlglPx4mI88k1HtQJAH32RjJMtOcQWh\n"
	"15QaiDLxInQirqWm2BJpTGCjAu4r7NRjkgtevi92a6O2JryPA9gK8kxkRr05YuWW\n"
	"6zRjESjMlfGt7+/cgFhI6Uu46mWs6fyAtbXIRfmswZ/ZuepiiI7E8UuDEq3mi4TW\n"
	"nsLrgxifarsbJGAzcMzs9zLzXNl5fe+epP7JI8Mk7hWSsT2RTyaGvWZzJBPqpK5j\n"
	"wa19hAM8EHiGG3njxPPyBJUgriOCxLM6AGK/5jYk4Ve6xx6QddVfP5VhK8E7zeWz\n"
	"aGHQRiapIVJpLesux+t3zqY6tQMzT3bR51xUAV3LePTJDL/PEo4XLSNolOer/qmy\n"
	"KwbQBM0=\n"
	"-----END CERTIFICATE-----\n"
};

/* This chain was generated by a modified gnutls lib. The script tests/suite/certs/create-chain.sh
 * was used after modifying it to generate RSA-PSS certificates and set 64 byte salt in intermediate
 * CA, and 48-byte otherwise. Then _gnutls_x509_write_sign_params() was modified to set a 32-byte salt
 * when it would have set a 64-byte one. That way signatures from the intermediate certificate restricted
 * to 64-byte salts will be incorrectly set to 32-bytes. */
static const char *rsa_pss_chain_smaller_salt_in_sig_fail[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDiTCCAkGgAwIBAgIUMquMu6/Azo9N40rNZ1z7hkotqC0wPQYJKoZIhvcNAQEK\n"
	"MDCgDTALBglghkgBZQMEAgKhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAICogMC\n"
	"ASAwDzENMAsGA1UEAxMEQ0EtMTAgFw0xOTAyMTIyMDU0NTlaGA85OTk5MTIzMTIz\n"
	"NTk1OVowEzERMA8GA1UEAxMIc2VydmVyLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IB\n"
	"DwAwggEKAoIBAQDrEJ5ONj7OYNjDZ3johFKItvX6BFJ7ejLfNELvT7I9hsiGJBr5\n"
	"Q/NgeQolSXLKHYG0L5Lxu1fbHINzC43NEivY3KMKKl0+MdXWwAr0yW/cTeuDc/+e\n"
	"YqGT3TpCcxa/0dJ+Y3zAS1DqsHjNOxyYBvyKATyvFKo+oAwOqtR/OLflUvoXvYZV\n"
	"YByseOLhE70Vfuk8yppRcKwokwk/3S6dZjoxK1K3PBQGARJNaUChtx5iM1qMrluK\n"
	"uDj7yV9DYhtyhSmYvcZ1gb3t0aAxGoGbfdOHa7XMovzfRDUPbwvkKUJqcNfGkeGn\n"
	"pZRzbA8D/YrjFtm7QVgf6yD20DbZChzoxRWzAgMBAAGjdzB1MAwGA1UdEwEB/wQC\n"
	"MAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0O\n"
	"BBYEFM/CHpfVzdNRBMYfqBXUieW9m9oFMB8GA1UdIwQYMBaAFDBBFsyy+oqRFlRx\n"
	"MH5qlHt7guXUMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAICoRowGAYJKoZI\n"
	"hvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEgA4IBAQADuShUlCXrs5K6Yu7mKvoyZztJ\n"
	"dQFuxv4WDvbhoZ19GEEg6icRUoaA3tWKf7tNRnqQklMLhWIZParXtt+xz7q5K6ic\n"
	"kX5oGzzUNryAx5DJkZCCffdA1FaQjCEI6Cy5cEnGifXyacwA7BViUwMnWvJRSKYi\n"
	"gvBVKc1TBwA+vPIzlSb3COo1zhshxM+C7mhzspDFkceXV7qapFDMj7M/GbgqH7h0\n"
	"yuJv2bymytjXadR43LuG6yqqsFvIPHYBcyPq3Uzu+57UJbHhAlkTXaAXfZkc1Ut7\n"
	"Xz8pOEzcxZHl4SEgsO6KeT2uQUE1Zx5AgwaNfuMmg0aFJep8vKcQ1jvdzxS2\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDojCCAlqgAwIBAgIUYIZPL5Kf86B0XYSKAdI8dv4HJY8wPQYJKoZIhvcNAQEK\n"
	"MDCgDTALBglghkgBZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAIBogMC\n"
	"ATAwDzENMAsGA1UEAxMEQ0EtMDAgFw0xOTAyMTIyMDU0NTlaGA85OTk5MTIzMTIz\n"
	"NTk1OVowDzENMAsGA1UEAxMEQ0EtMTCCAVIwPQYJKoZIhvcNAQEKMDCgDTALBglg\n"
	"hkgBZQMEAgKhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAICogMCAUADggEPADCC\n"
	"AQoCggEBANCQ6fUJYYI3OTDYIcyshBdnVBQq0uGjHg/04niCpoAZi/nlfP3tCRZS\n"
	"k44kMt6hla9cEkdj5mzeGFlG5AYG9C5MimyYwTJ5Sho6t8ct4wPESeypuDbcvMRX\n"
	"MTLM/9+ZECkDgKA238z4sNX0T0ppsCXy8IK0Jmn7bky6lqNmaMTjYWy7Tu4kQOMX\n"
	"7RE4tv/WlaH95d7zHYuaAf5dNY5GJ/cGrkYLrL1KpN/UU/4KKxvWs3EbsnDvrTcs\n"
	"mzLrTOIaedrrNXY6FsGE3+XKDCo+Z80LsrySpCozAECrEFCENMfS3ptOwI+Vblb1\n"
	"Kar8+4+7uMxbGY/RJ/gGIKGYibkpzicCAwEAAaNkMGIwDwYDVR0TAQH/BAUwAwEB\n"
	"/zAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBQwQRbMsvqKkRZUcTB+apR7e4Ll\n"
	"1DAfBgNVHSMEGDAWgBR1lWzS3rLSrmdPPgma8JL4j1PJgzA9BgkqhkiG9w0BAQow\n"
	"MKANMAsGCWCGSAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIB\n"
	"MAOCAQEAnYZf5bo7ZtysyLO/3QjAM+o1IWXinH97XANEbs5oZOK/rQNLBIpOLaYp\n"
	"YcnziJTEIqvy+7/KNwdjLcKZ4f5PBlDHBsr70XeJmMc+9/ZadY14BHZUEWNfBPx5\n"
	"dZR55/g62CdermdCJEoY6XdIMqdTHrdwmBIS/7g/dciQt0+lrjanX14VLAVRUAIu\n"
	"HMn5C4ZGeBDd8av3P+VIqdkFfpAYlZ5BsYqshel4pnAyhpUO5wTmY7cm78fqctyX\n"
	"qmQ0PRLQXmlqrL2oJtlGcSWlT0u1bS0gJPpvszataCZhnX/O9x6yzzgeUpP4I/AR\n"
	"KS4ZXRehFmQH4xS1eq5fmWiTzbvWHA==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDgTCCAjmgAwIBAgIUUVxp7I/ecuDCjWdn2Rng+TBNidUwPQYJKoZIhvcNAQEK\n"
	"MDCgDTALBglghkgBZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAIBogMC\n"
	"ATAwDzENMAsGA1UEAxMEQ0EtMDAgFw0xOTAyMTIyMDU0NTlaGA85OTk5MTIzMTIz\n"
	"NTk1OVowDzENMAsGA1UEAxMEQ0EtMDCCAVIwPQYJKoZIhvcNAQEKMDCgDTALBglg\n"
	"hkgBZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAIBogMCATADggEPADCC\n"
	"AQoCggEBAMcPAwX89KK6Nz39xdQRbSy9Ax7XzKAqtmmIczRVTKqsdQh4bm/gDuD6\n"
	"Edxjl02cISBLczWV13brINSBI+QX/eLPyBmGGzI4ryyJuP+1qc0NMjDAlfYw+kXF\n"
	"NZz02W6svxvrrt26mKJ1F+K/bZE+s9XHN0DW+hifQBBr8HX3BWJ9g6yj6YPd55pm\n"
	"kQQcVgRG3BG1EMkJGK4LNesGdJGTHy+uqgtcykrMjh25uhr0oTOG6UjVYjXalZ5o\n"
	"rOqo6CV+uGPmJYW2pBOlAOmblMMXSHXhIAhRBY8+h01BCsCU5wlEfPIsvclP2gSG\n"
	"RVbM/9XgS/+4yN0fD+oXgi5Jh6TCYz8CAwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB\n"
	"/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1UdDgQWBBR1lWzS3rLSrmdPPgma8JL4j1PJ\n"
	"gzA9BgkqhkiG9w0BAQowMKANMAsGCWCGSAFlAwQCAaEaMBgGCSqGSIb3DQEBCDAL\n"
	"BglghkgBZQMEAgGiAwIBMAOCAQEAqudvb92hfo7iAS63u902onL2XwhfS9IZtu3D\n"
	"Lum78Q8nzhWf+YSls4/o8ln/Erv8LfrrhxoPEVpxQTPCbj/mmHez3hh+xrb0ZUVQ\n"
	"pi5gE6kkkzzvL1VEMce85RLbm4AyVDl4onU2gaFXTxpMpKwBTZoKRbLcG2TsQgyW\n"
	"Kgq+XnyT/1AC2vp4Ou8G1MIh5bkfetTeo2KJ3lmEVGoUh0k0diayDwaBgBDeX7hl\n"
	"XvKrG/hhhWPVWNDXdQsiYYKVty76yM3vJiK9No1+jPZzNTv+pZaRqJiQ/ZaCICvC\n"
	"uK/63Yrle+W/W1Jdj23/kSSL94ugw7PFwbqo2gPkECbG2Mk8pw==\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_with_sha1_fail[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDbzCCAi+gAwIBAgIMWXnYiAFPuZgAfKycMDUGCSqGSIb3DQEBCjAooAkwBwYF\n"
	"Kw4DAhqhFjAUBgkqhkiG9w0BAQgwBwYFKw4DAhqiAwIBIDAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMCAXDTE3MDcyNzEyMTE1MloYDzk5OTkxMjMxMjM1OTU5WjATMREwDwYDVQQD\n"
	"EwhzZXJ2ZXItMjCCASAwCwYJKoZIhvcNAQEKA4IBDwAwggEKAoIBAQDF+5tQbOKq\n"
	"cH51hLYVtBFObTC6FdIdS5bS/7TEZnVR8y19RPJ5QFCVIQ4zs477x6MyDd6WeyRz\n"
	"3jjbkEBXZ0DxCIDCXN1SFxPC+x8REJF2BR0Q72ayap9SC1DvLvlKiSoWXgUi7nKV\n"
	"uYIMcC7bJyvKFk7b7j0YHccHb0Xe8I4unKMnvB+hYSVqqLGRENoIFYDDW90r4Evd\n"
	"pZv35/+Q3K053FMuRIndn0kZ3qH3eQUtIssbQS9n3HucTpiR7K+0HcxiEDX3N+DA\n"
	"91slhmk7jV4gt5ae7e5K162N7W8MQBmZhY8urxySQHFeHLW/JFQ9OJ9+Z106ISX1\n"
	"+i5n8XizBBdVAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9j\n"
	"YWxob3N0MA8GA1UdDwEB/wQFAwMHgAAwHQYDVR0OBBYEFJ2ub+hyk2QafGrbuKlN\n"
	"wYh91vRIMB8GA1UdIwQYMBaAFKOCzV4ao1MVHrBy+M1ib/5nd2p3MDUGCSqGSIb3\n"
	"DQEBCjAooAkwBwYFKw4DAhqhFjAUBgkqhkiG9w0BAQgwBwYFKw4DAhqiAwIBIAOC\n"
	"AQEAd6oXlHPD739T+attYU+gqERx0KHGZfYx09GcEFCTsECAlMY3hpG0uCD8HVsN\n"
	"bKwYb1R/F/nwMBcJD0xteL8FO4qB9FNpnuFzmTeRcVdf+vpLgzHEZFi+twzEJip9\n"
	"gfQg4xXY/F3bh1ARG3l5TDmvLQSv/cidse5HQ7mFSsjAgCJpKnCaqt+6pNzWvofq\n"
	"yHI+COJ8rMUTrgp2fhpxBz1pGMmhX7HwJciI5jLPy5Q1WEYpgfrJcMb6q6pAKvlQ\n"
	"Jevy9IdKhauADb3S/vPDy0xY0SJW14mBVtI6+EizUcx6XuirU5pbYyyeOxb9Y6Wv\n"
	"7CfpUMRomkMpPhkfYWChnjm1BA==\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDkjCCAkqgAwIBAgIMWXnYhzaP/YD+JqXpMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTIxMTUxWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggFKMDUGCSqGSIb3DQEBCjAooAkwBwYFKw4DAhqhFjAU\n"
	"BgkqhkiG9w0BAQgwBwYFKw4DAhqiAwIBIAOCAQ8AMIIBCgKCAQEAp3zhvSFFwRFU\n"
	"HhrpxBSkoKnMz+D/RoA12Onl4q3dv6FAnZd33lCXAaEgdOqGzK/gb1QUnw1f/fgG\n"
	"7GBXCW2EvfasMVV3lQQAJjlCXNjgUavZ8uDUihsJ49ORHZ7s6eMNfAeI8m02sj2a\n"
	"lV2TQAYa1PETbVr4K58QHvLQaaBXN6oA7G1hZobt78W40wBKkMzTkwvYh1vRXSGb\n"
	"+/Qszs07fhim32nBE/42OJ8TwxPGLfNkx027u8eD4Ebl8zCubOGHPko56s/ip/3A\n"
	"UiTMY2jR9dcHisfW8CKOjvKlmTf8CXSw7uxTHqhx7PFTAMBsStMnUpx9JVR9wyKg\n"
	"Y7QYlBpfoQIDAQABo2QwYjAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMH\n"
	"BAAwHQYDVR0OBBYEFKOCzV4ao1MVHrBy+M1ib/5nd2p3MB8GA1UdIwQYMBaAFNUc\n"
	"loHmlMVV0q6firCdA/RFcfsxMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAA4IBAQAM/BlBEyKGyydb\n"
	"nF24GI0mKJnWYMWgOIk6Bc1jdh9khPe0MfDKZLFHZqNBWLEHByTDBL9Z8LGACXYd\n"
	"L1JAlA5BNgp1p489qz/SCOdriqLV7tAAND9uBzEALYLckHhra+4dlLurFhioEIqP\n"
	"dcZC3ocEqBd9L1SMPOCUniGtALXeEwJkv5TnXMfP0vaIzGVt4x7+KeCF4x3HhJN7\n"
	"3SwCpN/mhpATMQYY3Fen0UBeRWTUbRjsjRwZzO+J7BtEUNl7eMyum5pOFhhltYXZ\n"
	"De7kx7bEX+qbVg1F26PsNsBlSJgbCglr8fMlEuHBpG7viWkt1nqJW9rhcYb0bhre\n"
	"NjSVlh5j\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDeTCCAjGgAwIBAgIMWXnYhy/+QNZimT/9MD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTIxMTUxWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggFSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAA4IBDwAwggEKAoIBAQDJ\n"
	"vn4RQAG2vcJiv0ZstD5d27Fp5Lhw/BjpArUsYeVTyqaVy+y+juSc0DZJU6RBSdKi\n"
	"g38iAtdpVVKyV1R3HE+jgLpPULtFsp9Fziz2bDUXFvJLaktbqPtwKBEUFGcHTeiD\n"
	"RgyKBvlwpP94A4FDYBDQQgHAbZKUIfkQrPiObsQU72+P2TtuTUTl9PIwFS0faNKX\n"
	"r8dRN14Tvt/JhpOF0qhrOLBoxVVATBArJflAT3ZfiWW9QQj5DoMogS9PCWBQKmhD\n"
	"K0xWp4RCeiKKZHQH9jCsJSJ3vBcEiql/TrW+h9kb7ZkHJsrpKYzcDGCJFyjWmxjD\n"
	"3QV7bRdjoVh4Hr/tW783AgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0P\n"
	"AQH/BAUDAwcGADAdBgNVHQ4EFgQU1RyWgeaUxVXSrp+KsJ0D9EVx+zEwPQYJKoZI\n"
	"hvcNAQEKMDCgDTALBglghkgBZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUD\n"
	"BAIBogMCAUADggEBALFp6rRbdfbJgxZ82VtIFDipowjEiuIgCaZAym1JNyBuyy9a\n"
	"OHLz+dMpLvmZm8U1EGlRafCYyY2HL6NTQvn95iaFLpsccMV0jo/jZWWd5GH8lU0+\n"
	"LAkatUfK28Ue7F38i84zWJpXKobEEZKT2CxBJ30jxaUezA4AGLifF4Wsj1UyHhuR\n"
	"q+U51DjJ+MvZEEcGEQ+kbUwbBKbSFJrZ4BRCqJvCU5HjtAuzJu1QYQ0qjTmd+n63\n"
	"cyr37omknmF0LMPH4+q7G7scPS42dNIvPmJ8v3QuGm95WnUI1cPaci7BEozkV7ik\n"
	"8Cqz1BnOnAx84GjQ16B2QeRFicuqKHGPGPB/qEg=\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_with_diff_mgf_oid_fail[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDgTCCAjmgAwIBAgIMWXndVCXFljE2JCDNMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAICoRowGAYJKoZIhvcNAQEIMAsGCSqGSIb3DQEBCKIDAgEwMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI3MTIzMjIwWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\n"
	"AQEAsuqXehkktDwKeniqCvZJGSO65fOgB0xAnOCuvl7Ijtjgv4NwjfQP5nYGyb+Y\n"
	"oNX0BgXTo9pIcFWIDptOlHs5bKKo8XgzffgILMk307B9DGBx7xpVINVj+WoI3DFT\n"
	"fJSaO8jcwxRAXr3hhVY/hQ60/cKD0Cecejlo7LJbZ4wAnjsYhF5G49o7IVuIqZzJ\n"
	"26lzCPQvrNAY+Pwc7zxN2RitvrC2UUueT8S56NxnRLAgghWDTXTi+02pZakuojli\n"
	"YikV8xs5YZIUdk/y69i4U7hwfGFR5sv6M+o0KK9kZb7KlELXpO/BOs80itIeEi8c\n"
	"60xCL7w/jM6Ecvp8f2QBDPRWFQIDAQABo3cwdTAMBgNVHRMBAf8EAjAAMBQGA1Ud\n"
	"EQQNMAuCCWxvY2FsaG9zdDAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTDlByu\n"
	"iQUCjywT/rAp9FyNmYEFTzAfBgNVHSMEGDAWgBSvAud7TcW4ObAj/q6BHxW1ij9E\n"
	"5TA9BgkqhkiG9w0BAQowMKANMAsGCWCGSAFlAwQCAqEaMBgGCSqGSIb3DQEBCDAL\n"
	"BgkqhkiG9w0BAQiiAwIBMAOCAQEAPBxwYHtQVqHew1LAxEH7knPJNataUvpdxroI\n"
	"E1kR7PUgj6ODazEEq3Cv+5evhF5fhkmAA0+k5hEJzfnbRnSsVvBzIkkF+G1hfDwm\n"
	"WWEM0EQYFiUNO1aJuAMxZi6M6dZbEr8ZfJUd3ySrc9iFN8RGgh7mhtQKPkhw2ZYv\n"
	"WyaYkLOBwyeGm6Kybozi+6l8IveZmzF8YXph0XAaGCdjVjMYarEyI6u0BltR1Wne\n"
	"rvt7CioAMWUbYauEc7/2ffCYleLn83f4puVQyx0/f6uuJ3LD1tUh0aXYhGl7fJRc\n"
	"G6I5gSmWui8cRxPzX9/d8a4Jw8bKfTYJnLM0p0O8NSLzP47nYw==\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDmjCCAlKgAwIBAgIMWXndVB4vBSZAonB1MD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTIzMjIwWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggFSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIC\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwA4IBDwAwggEKAoIBAQDf\n"
	"ha9h+Z0v5sLF+9yJPasc9dpi1cT2+BsDqYucfKEa9Pwgrs3OKCk7jFrIqiG5h0C4\n"
	"womDzUHxhuyJ6XOMjN1r1VCaZKNN0CBihWzEDHt1sxmY/xPkVYPtJOvstRmHiUSu\n"
	"w/iuV8UdvRrcNvcmfgwhxCvTee0UG4e/K15z7G5N5x1t5PwK3upM3rCIGVVeLfXw\n"
	"CACj+JXS4l2sPNW3sZ9aayEe0BwGuym33SyGYl8R8aTFk9ss2jfB6se4NFH0B2QY\n"
	"pNZHWwYJWVHRH0c5251kw412tYR2tILnrkUs43cWc5dY/mhcHXaIec2gPURes2bB\n"
	"CIJRAkXxfvuaPZpMfvmDAgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0P\n"
	"AQH/BAUDAwcEADAdBgNVHQ4EFgQUrwLne03FuDmwI/6ugR8VtYo/ROUwHwYDVR0j\n"
	"BBgwFoAUGQ4GFn446XTVG8jGCz3ok6tf3SswPQYJKoZIhvcNAQEKMDCgDTALBglg\n"
	"hkgBZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAIBogMCAUADggEBADNS\n"
	"zFchDH+tULFXxrJPAnDV1KMjYskMYzwMdUFHwjQ3A/4uLodhvqH2b62HWQIPG37a\n"
	"3FOmFSQTxx5qd5fVN9LgcPc0gMowjlq+M7lJjlKfzDPceN2XyvxNCf97NqYJTHya\n"
	"RFNFm5Of9dN9YDQ7kdu2FIcFEczIbkIQDFcizTrdPgZ3QkVEnrDSldtUEdgUxIf5\n"
	"aSmGtPhhG8+x/uqcFKF3Ese8pNkCvL8pPkLSgtTEGmV/eFtKNb0VArdQfQ4g3BKO\n"
	"6DyWE/hasa0r/b8WK5LwIMQnItkQvlwIBEPU/XS8STL9GySCBag+Pt6DtIzaJVcs\n"
	"oHpzmj1Pz6hk2R21u5Y=\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDeTCCAjGgAwIBAgIMWXndVBE6CEbh8MIQMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTIzMjIwWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggFSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAA4IBDwAwggEKAoIBAQDq\n"
	"Fk/MQ+4AVJPMYobr2MJSKJQ5DDLdBmmhv6pjvCO/v21oakGlz73XhtlA7lm9uLRP\n"
	"9gCXyIZCBHk9uofl3f7g9sBxdZrAvcDlonqekW0yUtzFcnm0gcqUmJZe8uXkpIDP\n"
	"GBcSegQXhsE6pFrQXVRuR2RVtz7U9a4ItYZwvVSqTRQtPOlsdzgNGPiFzY/3Qawp\n"
	"4ssrQyDl03Up0pTlws68kUnrONkGGcxXIEiqSZJhrCpRdN2BvDSSY7mzxTvG5MR3\n"
	"RhTJz/PJn7C/QvqcrlJJKvJ1GZzdiRMczlKAOASu7bJppjLhHmuNMDOxPt5dgLqI\n"
	"tH9Rfl+s7yMDa6var+NTAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0P\n"
	"AQH/BAUDAwcGADAdBgNVHQ4EFgQUGQ4GFn446XTVG8jGCz3ok6tf3SswPQYJKoZI\n"
	"hvcNAQEKMDCgDTALBglghkgBZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUD\n"
	"BAIBogMCAUADggEBAIMiwgKxkpfnDsQvsdVvxKQgoQR7QXnGZTRNx66YGjHqFT3T\n"
	"VkOJWMa7IvPvfQegi9C2xEwOGPOR13fJlroxya+xKmzgebWhD4tKxMjPbMWYFrCh\n"
	"qTJWJz9+MyST9/3YXu7Ao0+kIDJfQVipCvRWah4WKqi2HUhNnxLz2yrAc0f4zsbN\n"
	"FHfsFK5bKnahSlXlGV+ZrALqieQGjKh66IqB8P2vfmIjpbTh+srGuyG/QxxlUk+/\n"
	"XDk8S4T+fnV+41K3INmG7wkjP54dIwrv6YjhJKo4nCaO+GB0oLA8SCI1doClSCyK\n"
	"zWdV+iY4TwA8+UEEeQCu7mGo71XOTg9/YaiOc6U=\n"
	"-----END CERTIFICATE-----\n"
};


static const char *rsa_pss_chain_increasing_salt_size_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEsTCCAumgAwIBAgIMWXnOxy72g1dtZFCEMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI3MTEzMDE1WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIB0jA9BgkqhkiG9w0BAQowMKANMAsGCWCGSAFl\n"
	"AwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBYAOCAY8AMIIBigKC\n"
	"AYEAuXu6rL2zC75jxQIc8DVdVITJQGXztZ4rTJgYBvncFm0rSKDAa7vKyIaro8Mz\n"
	"8F/vhTYb108sjv4TnY4/Nc7toV91plmLdhXhMj/ZqPckAVvPUayHSe9AF2I8HJTH\n"
	"s3IFZIkEi0Q+2uAvkfpZCTBq4Frsl3ZXPYtRr5RkWNqzN35DpUnHm83q+TPuA+K9\n"
	"TvWZjeP+xWIkk+41hRInB74qLiz1AMzrZX/DnFkYHPCi+uoeV8AVBbUZ12uO4KG4\n"
	"IWDhs9irWtxiMNZiDIvRnKiEgUGgYOIto44KuyesxMPOdz7A9ATcQAIdUq1EqAHq\n"
	"b+Xm9n4nmekwYxCmBH4IH6eSbQAfgewuFWfqjjBWjF13N0xj0iBwkdDzx/ThN4Yf\n"
	"ejAJRw+L8SURYY3FbQYhg6B+emws7RE0bSFRwu6Au8O/XiA+qLTwivr0Joqry6d5\n"
	"GV0wdBK4h+ulRScR6YBJdnYJBGPF3A7QnyPfnBWUfkNk+zpqOIXuy35lO5qGOZLv\n"
	"yfGHAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0\n"
	"MA8GA1UdDwEB/wQFAwMHgAAwHQYDVR0OBBYEFARQW59hEnCVOewEitaEuYxbZb40\n"
	"MB8GA1UdIwQYMBaAFB9SkyW6Or/XyMyRdrHiC5w5MbNXMD0GCSqGSIb3DQEBCjAw\n"
	"oA0wCwYJYIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFA\n"
	"A4IBgQALdAXW0tx6B+7nuGA7/F3ed/pXj2oG+Gu5otbQHda6gnvGp/JAGr2p45Mt\n"
	"55F30e6YolhCVWPaSaPSxAjQSg1vy80/QTey5aPZ0jZEKklzyid7+q1s3mOeIDA2\n"
	"ajaN9Tq7uW+YxuMD5V9CSpTu28Fdqy3g8K8579hr14ch30IjDWhxngJLxH5vRnAT\n"
	"1WAd6xLU+tBehMozcenW5lkRzmnD8SeXIZa+TOcQa1IzzU/VwrGKaLs9dNFOoi/b\n"
	"9wZkVBA0vp7jRA0gnJFQkldLetlB0h5JB9Kpcw/lsqKF5Oe+DBXPlaNcpx7PVZhL\n"
	"zFmH590jgEogaVvUJ1TxeUsNwUltsaXNvlkFcxmMUdziEXLwKkdfxog94owojuu4\n"
	"2xn+wsntn+02G0r9w6FVW54cXpxwvC3BbuIMAI9Xn57GKgr+SzazubimtHLw1o7x\n"
	"Xzs3EIn08vZDvjHXcRKmKAkhtDP6m5ni10tajfAIKBzjwv+YPJOBN2Qk8mnE6gBY\n"
	"dwiMNUs=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEmjCCAtKgAwIBAgIMWXnOxx7hUkOmAFYAMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTEzMDE1WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgFAA4IBjwAwggGKAoIBgQCt\n"
	"QbjYb0pxJjIRpfGTD0oo2JrUDHOaOt/ON5BGdH6YnN0O4SBuYMzYI0WwX4CWUoeH\n"
	"qYCnQqOhoZv3FoIU1dTIv7YUHW47drn9DywSIfYWoLI0snDu/QhB23SN11Xgcmux\n"
	"pWiyp2PXE/WEFgvzN9/Y9Dx+xapyckl3M3ydF0fNgXVtnvgEzPgtsR/H/6Trzad4\n"
	"0ev6Yvj+ZmoOwgYr3UQgQew7uCfbw1EavG6135mwW2QsDxBDpWiG2hKXeb5zDftg\n"
	"+uZbb5q+TyRdLwDtwmtcjX6nlNIxssRJaNMej8MfG6D/U68MMclvwVHe+QhC59bE\n"
	"1rft3TVajNKTfWNgWgIrtt1XBXzOAH7a4ZSsSLScfrASEbJPIsWTgefObTw3xZjw\n"
	"cGmCTZu6sgpXYIvk6oUwWhI52SaulhApbUkB1yUUnYUJ24YmI62YA/LNmA3w3avw\n"
	"YvVIBhNV49sCUCDi78TUCc7QLn1jJ8Y+LuEZ7MIRW7yyj4zZ/Q/ket9mS97JMjsC\n"
	"AwEAAaNkMGIwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQAMB0GA1Ud\n"
	"DgQWBBQfUpMlujq/18jMkXax4gucOTGzVzAfBgNVHSMEGDAWgBRCDm/4nfcRVSZx\n"
	"8GuAUop0AzlBwDA9BgkqhkiG9w0BAQowMKANMAsGCWCGSAFlAwQCAaEaMBgGCSqG\n"
	"SIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOCAYEACNo6j6f2JLpwbbQ/7vE6gRqC\n"
	"4sx0dxAiAb+W44dIETB4XR6rhRTr/rFApMAnf7zZ6YlzKIT7DqyeSUmal1iz93rD\n"
	"pNo1AW0Th7P548jm+wtjEH2/+BTJeNzxh+ruqCpocMMfMqmiPxUDwpdKtZaGTnV/\n"
	"rcxbfwFS3F27U3WlJo+FW/fq2/+m1kh/9wAyiogd1HC/FO+TSoprw9kvr1dn+OzU\n"
	"FCVrq+BVN8QrFj6SVSiaxn3/eWYnwlj5mj7WLw3XXNcImXT6uIkHWLyI4uEO0qnD\n"
	"sV9/lf4ApqTvaQR8qWPpN1pWK+fWOH+LbWHbrcuLUhGfZReIrky5onhIZJiIbOGm\n"
	"sLJfiYxtdfo7r4RvtrnPwcKSEJvNdVTeU0psmEbGbatnrVpGezNu5zGp+PvgddVu\n"
	"KQ7UREY31lu7x/PFT7dDDkWWv0cI5ny9bpci/Ua0FF5RF3ovgOyo0uEWrfUPNyvG\n"
	"0BAkOtUmY+HilBtj0qAadFcbV2hHfdk1nHG6YlBh\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEeTCCArGgAwIBAgIMWXnOxxfLoQ2OARFgMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTEzMDE1WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgA4IBjwAwggGKAoIBgQDL\n"
	"qyZ+7NdkwHddKKozoo740ywb0L/ntRkqvtcevRgTJQiwKwEIac74m7x7o3d0LXj7\n"
	"TyKqcR96RtlLhgppsCiQ8ByWNTh77zG0+phZ7oUuYJRY5HhHKo+mjEJ2DBEhsKmg\n"
	"aoyliZ9GYaPni6aJFyb2xasEyDugllcFpJlVteebcg8MwGZ0sQOR8GBjldtDIFrt\n"
	"snPiwHScvIbDMy87xOTsvUpr/I1VHX8JEiPa6baWTiDuR/uX3izcXniBUZV7anik\n"
	"UjOjpUcw8DohnYeqIXO+MiC+2gLsyqsQD5cWhRkurLFDS3/5RSvYL5C8EPvYdJxz\n"
	"xd9ayWwSGLj4cAS94CWJaifmMnmkq9cYPZkmILe2sikvq7LpkUENxDxklSPeQXRc\n"
	"Cb+JC8btqlJWNjuZOvv9nU4Cp7K++27Rzkdw0fp0fiiRxSqPAQqpba2LyRvIgsd5\n"
	"raqOK8gC6Dibpe6hoJzEctBtuiENXyiAmxS+A0TE2LrEBbwc0OZQ3rymY/7OlHMC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBRCDm/4nfcRVSZx8GuAUop0AzlBwDA9BgkqhkiG9w0BAQowMKANMAsGCWCG\n"
	"SAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOCAYEAcFw0\n"
	"2OUUobSl5zfhxl/e3DVLICiq5CPdbv6ZArAKpCGWNoUaHIqeecJIfJAUsw6USrRJ\n"
	"+TxL/VD6FvvtBm7ZPjjbuVvXfu+3KWTbD9aEWmblbSH6C2V8t1XIROTy+kIjIWv9\n"
	"Ihb+iFEuffqeVnoLW04BERQ22bXfubvzGsf/cidtDhRv6/wQVHWzZxImKQcGCxSX\n"
	"XjehwnOcYLhakd5O9rcn1YwzljETLCpGbd3jhrDFO+yBlq+IQ9UnNLqT44hY1wAK\n"
	"QoXLuiEmkuTaYXRs3t6J1u9HSUIfu072fAoI9GxvLSQ9YErzc06nmJxCAcra0lH+\n"
	"5AiEd4WjtHDeRqLoOKOGVc9FO3BueBrrVJCU/CxIEMFPeCXu6WfaR1KxqQFCWi49\n"
	"bCwAGwgYpSNMrOB/v2ybiJHCW+xcZ6/+DSrKKjWIRRb+GwpOYN92WcFG8Cq2PwA1\n"
	"ku2e9OkNgsFKxhj8uWFv+3Pr1nrBFzCQ1Q4BjbopB8zR7vivYm70XqsGZxsa\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_pkcs11_pss_pkcs1_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEgTCCArmgAwIBAgIMWXnPNhh4NAScQCZHMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI3MTEzMjA2WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAryX+DLdg6z21lpupqpL2oracNWNtHFtRvHo3P+RLXYVl2twWAcbNB9Zs4jpC\n"
	"uzhOMqrmcWlfstGtBoDA4YjWqk4T0nvlRIGLVP74AayGiggr46FwVon5XRTyCIkV\n"
	"4GL550e7wkpNfafb/wYSZ1YxyqcMSzWw9ynWK1O737zIdMoVafht9CpJnPq6NI3Y\n"
	"P14sJtQqWY9gV5lEpsNeXulOcOFdB2s6D9Qke8IqijSpTj3bpMBvJ8xkUPHxBoyN\n"
	"hYvPBvpRSIB3hBr85PCZaGHOJOUljm2iV6Xdj1sVe+bo4MG1CnVhCFt9L99oOwg3\n"
	"j3jTrvTfs496wBkz8EzdpeWRX97rVLHWhIjsLX73daJHfpI6ZnLTv2CIAuK0w48r\n"
	"AT9XVW74z7DEPniUdB6Nku9R7yNfiULbX/4LMEA4TpTJVPRE1YcmRcTVuw4Cauxu\n"
	"hkqw584YCWwPc4r7vhEKPZPMsrR/YOZPXTDSHAhVVCJCOruvYWn88uWIj+GnEzLG\n"
	"MCcpAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0\n"
	"MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFM39DeRiwBay6YrB4tohGli+Ucy6\n"
	"MB8GA1UdIwQYMBaAFEXZSb6vYdOsgU2XjleiZ5ifrnwbMD0GCSqGSIb3DQEBCjAw\n"
	"oA0wCwYJYIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEg\n"
	"A4IBgQA1fBiRmGUDDjdr6M/5kwECqebfefncXH6lvBT0aCfdPcMMS2O0b5aY9eFO\n"
	"B74cNbou0AEIZMWb7tUPfG+Q0LgO5+A15n1bZZ2RAuwQp4uX7zwkfpE4fA1NR3wZ\n"
	"PuMkKlLEaCXYdRFlCjJktOVRu0UGTSSuW7EZYlx8xJm7SeqfYg5lTs6SndMxSXAX\n"
	"35iJ3Di9OOtEr+Qd3M9qjYIov9GYvf/QRfKfqWdPYhxW8uGkj88kcJc3F1TnV6vA\n"
	"jqeCvGRbL+cFCIwsKmF6s1wMatuNTrKbqeskZhoNg8geRidY9IBZb+GxkwLTokiJ\n"
	"JFLkiF5PGVaRbS8rwQ4nfy4ZPJzXAapXSpdhu5sdQINQvdYfJTkuM6SvkbfoTl0n\n"
	"v0NnHRYZfIkAftdr8C01RaM09Gc4f92yQsQvwroBQlmbJ+vNJZYtarU0dBu1CHdV\n"
	"ndQflknAQjt+kIRbrYM+I/Kh3IbZ10vKV5nr/p3RBj7Nu8NiyfkiiwJaibpxZz2U\n"
	"t56r8oM=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEOjCCAqKgAwIBAgIMWXnPNg/J0oxWLPnLMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTEzMjA2WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgA4IBjwAwggGKAoIBgQC1\n"
	"v36Xtc881eB+V3rg7XkhFy87Qc9g7OnCDczZ7cde3zSa337PZ6lUqpj4PfLwEBWr\n"
	"3rQWR7axb3LuNj40IPcBpZsmhk+twc8LnidqqIt17hza7pHfovSrBBupanaqVcpr\n"
	"PrbIoaiONdqYa54pVbW/FXTY7fiWHeAxxLAS/QvqLwCdhFfMSawNblU6FDAPyNbE\n"
	"aAGGY9VgN63fHdxW085J+vdd6lvDk7BG/pWUd0VpB6ltJwsX/NtgVPWcyiHm+uAF\n"
	"diN171SucT++e9H1OvkgVFnDG1Mccy6fJJ1dEkMparu7bn5Uji25cquVz1aYplty\n"
	"LUxgMdpMrg3qvaz4B0C7hdHFGLYmlKCzaljJ1tA3P83+Fkwa5WlA5r4rS4fDsLJJ\n"
	"tQJDLawTvtgatplXJHepEV1YAqiOAi+2iA86hI6jTIHpgqoerlEvRvzT1krX3yaB\n"
	"eIwFjYgztBUBdIZ7UXIIryWztb9LKtpFIPVDGAes8oBPJH4/8ti+AIwDjlWOevsC\n"
	"AwEAAaNkMGIwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQAMB0GA1Ud\n"
	"DgQWBBRF2Um+r2HTrIFNl45XomeYn658GzAfBgNVHSMEGDAWgBQM0WnAYYMNuJPW\n"
	"D0SVZOQJJanMrTANBgkqhkiG9w0BAQsFAAOCAYEAtjXmbN8Nj0F42Gz6XFEsdOtB\n"
	"vPyj1k7zRFIp1roH8IQqqOjtEkAj6XuP0INvIUpy5/J1PIJafU0O27pjg37IdmVt\n"
	"H8fhZUmPbrSDNE8+VHwcWsQtUFK/rw40lnNL7ymaRXQNM5HNKfYdt7T3WAGQw1MU\n"
	"kbW3efAAUa5iVZKp9W9RX128bB4vjW9q/9xUbn+tGujZXb6lZ+uK3iPthtteAy0p\n"
	"XNQZw5BMiMSOV1apuU+uIcXOCgbgiU0ix3CwHB5C3yd+hkK7dd8GwD0dz2WreL81\n"
	"0m8Ym2rhrVhmnya262QVE3sufNt0yi7VjLCNrvcdeDLypypYlz2l0xCcl0bAHqL5\n"
	"KhcVsqXNQ0Pp8V2oaPVQRIGuS+yo+eJSe24U27KShBO/0AGY9AGcNALrl6P6n6ZS\n"
	"HMecIKkS5EQdlTxAuohO9cVd4167sVE/+I3iY8GmwQ8Lq422hEY3zQdZFJBtJ7zs\n"
	"ti9MSSipLUVJgD7t9r9nPHklupVm4atvYjxlxMC+\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID6TCCAlGgAwIBAgIMWXnPNSlMcWT4IBgMMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTEzMjA1WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC4\n"
	"VdQEElPvqHTn7BS7asLCvk56B0pkd34wDWELzifq0mVVNb1KlKzllm320EpQ3t1C\n"
	"ak9vYm4NogSwujcXaR1ZnMeOyEoXay2qCWD75kjUTo3yTesb7YIU2cHF7dftH69J\n"
	"oCeQC1+oiv+JrVMluVMpGuNB6TQiKvOFlKtmYXuIhXS9L3x6JQbvPdBWzr43iJv3\n"
	"vq83z7B+yU9EtURD/ub9vzHsZrQD3gDf9tJqg/laBvsRDqDP7U2vzgh9RV9UBUOD\n"
	"8yBN4n1TJyL3b0SqxLEXDdf843VXUSm1a8saO/bvmH6IRZyzy7BZ3Vdz8e2GWMBJ\n"
	"uHr8jiwD+Fohhygc0l3eS25h/cdVhs36b+QM4BCVJE/BpzznfBx8Orn4qWCQenw2\n"
	"d3GA19Kya4ilDKjkLEkuevDM3FM2G4bu4SHkEMcMgbMjuR8+QjgTU0oJyTSxFikg\n"
	"0aTr3NNfXeu3RQ3E324Xa4W/z87hQ08rO1lQYdzQ5qHHsF1iTv0N3mfTKtVhXdsC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBQM0WnAYYMNuJPWD0SVZOQJJanMrTANBgkqhkiG9w0BAQsFAAOCAYEAFxHR\n"
	"t6sFU7LLkGzu8LQxOijtghiF81do7LuydlBGI0fr/Y7KIkknpYWMxv6VZh3s1WV6\n"
	"/ll/0f9pK4+G0i7r1H5+i4HvfV5zu3ApyK8GBgxZvrnbCIkGUpocXOsdnAtxJSC1\n"
	"WvQrqCAZKixQDWwyt9YmWm51KEm1JJtWviE10ShaqVj35Z489tKUd2fGfepodP0d\n"
	"Vr5RLlV+e8r/ib8ZP8ORs+2wqLEADXFk1Ve6qkvoExt+zhcgcgzAQH6gzi33BKof\n"
	"uU7REvZ9A9sL9uj62Pk/Pj6LE6I2xA6nvVrSkT1waZ3c1iufoR7vUrxKz7pI9tGX\n"
	"h2sNiR1M3sKEeppgXG2Y/RWn4rK7OVnIwr4DKU8AtPi7KG8ERLriaHlU5nN2XAsB\n"
	"tas625DwjY7W6H3FAeD+9nTx0lNUffh3udRrzgCLobu2My5srY51orxS4GQJZldd\n"
	"PoIw3F5XljrbtW7UtE1Lh9uuQhUXw/QXTBGYa4WXKQSJy4W3KpDRbHWsISGL\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_sha512_sha384_sha256_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEsTCCAumgAwIBAgIMWXnNyRuyw21uHTizMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAICoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI3MTEyNjAxWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIB0jA9BgkqhkiG9w0BAQowMKANMAsGCWCGSAFl\n"
	"AwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOCAY8AMIIBigKC\n"
	"AYEAycDE2FpGA7tkLcxcnUqA6BVw2xsyl4IB4K+fRHkAF/4lAozRAmIFMlcuPM+b\n"
	"YZ4qRhrsvaMKFXxgCIuSz965Vvw6twm+E2XM3oBnkgohUSI++j7E+/5cyrp40jGA\n"
	"QUhzg83VoeJN1FIVJiWiZt5zTalQ1khRm2asVJiA22dqQdQZiGBy55jUQ7aGEv0Q\n"
	"MkQwTbfbhCGbpGyYz4Qx21IWp+AhXXePQ1j3NH9LCLJK90EIdq68XIVUmNtYi8t9\n"
	"devj5+nEAs68uSidEWp9bIB7y9ndL1TSfQ6svLmwLB1Rl5D3rM9ZmmB9/92rECKp\n"
	"PdoyxkBf7BU3zTPP5Q9qlMqfL1fnVAcXGsVhhiUcWNPPTMarLcwtKgN2pvNQT7nG\n"
	"QhnmIl1kNSYSZTPamahF9GOcTL6rVjadu2UBd5+3XRS/MXYlPYWy9YQeau3cDrn7\n"
	"03pXT8qFbc25IWsex//hV/7rk79HbDdwnoUg6rPaHBMRl51QFaVVAi8m2CjHa/f3\n"
	"5FjjAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0\n"
	"MA8GA1UdDwEB/wQFAwMHgAAwHQYDVR0OBBYEFDG/Tlzzm+E/1Z1rDlDQfq+KoAZm\n"
	"MB8GA1UdIwQYMBaAFJKBxhYjx+AnDvsqgUqc1TK82POAMD0GCSqGSIb3DQEBCjAw\n"
	"oA0wCwYJYIZIAWUDBAICoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEw\n"
	"A4IBgQBYrsdt/jkyeoH3JeXWqIBLyK6uZFC5Lh7FuK82Mv7iinwtPVL5VQtGAJml\n"
	"jquuQ51S1UmZ6NUDW+wcA5hro1KAAb1wDOd5BmvUGLSUIPImT8YX/npLGtKINBBw\n"
	"N56xsHgSKuptU3/Nbe8Yxgxk4xwzeJa+O40zWxuYryZHs8jBO3I1QLVV/iQqrpjG\n"
	"dbxOEdOGyyU21CbpiCOuiHRraVq2IOgWohS5Y1+9AkHm0ir5QWFfxQrFnNA5WODx\n"
	"XIBr5gG1mRj+bwo2kNxSFW20DTgJRL3KpINNa5JJxemzXcv37rFeYyKrCQnGe3bw\n"
	"DQZwA1gDCA7zAo5QGm4I72cwRN7CoLxChdJvCGcDeYmkEBnKr3KJ4qYOLbqcGWQf\n"
	"ErC7NEDprI0OytJkyMXX3CN9yLkfLqF9uiKhrO8oRrIfeN6Re195ORJDZxyj6Txq\n"
	"eGrvjT5cmOs5c2Lx8MIsa1ItX0iVdZQlTkgu3/Pe4MT/dUfTpPaExSMBGUVh8YRy\n"
	"bN522fM=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEmjCCAtKgAwIBAgIMWXnNyQ1BARJPavyUMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIDoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCA6IDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTEyNjAxWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIC\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwA4IBjwAwggGKAoIBgQDL\n"
	"TmQzgdiY75JQHpi5ZDkblnTj5k4utI5jV6HNWomMcGyj5Ofy/F7xRe1r/bQB/G70\n"
	"fnuy5v4iVooEgDOs7GSs/7yqagqkvGUeIDskmjmNznlveBAEqfuDkiFWgfe/2SZ6\n"
	"iami7UzWDC5VObQY2isPPwfWllGTvq1eacfPdcZO8D5McaMVAlTmnqwZ1plRy93t\n"
	"n0FoikDr7nX2OXmeSpUVPYXhK/SMTYCXlNKY3sAkZGsg+YLslyR6xkLMPK7ZYZQY\n"
	"UNynwSeC2KJD5d+8f4g/eBlYq6F93f0/oN0t2C3D1NGWEapCUwt2uk2U6H954quQ\n"
	"QWBz6RzTPP+DhNlqNy2F1OqFmfyz2CqcthnjSvX+wYmDGQ1i3dOJv2RuZggKoqdD\n"
	"T++OIO18ui5pxJJY4rXbXW6Q20Q1r7L9ekth9xPQMfzQrvmNiSYGIin1Dg/sIx/S\n"
	"jcWpVUfoKhWodVKFqqCTh2BQIrhjMp58HsDnweqZplmol+TIYmisDOPKkJeUfTMC\n"
	"AwEAAaNkMGIwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQAMB0GA1Ud\n"
	"DgQWBBSSgcYWI8fgJw77KoFKnNUyvNjzgDAfBgNVHSMEGDAWgBQHMqYj24DquhwW\n"
	"ZWWW+RYVmFdhODA9BgkqhkiG9w0BAQowMKANMAsGCWCGSAFlAwQCA6EaMBgGCSqG\n"
	"SIb3DQEBCDALBglghkgBZQMEAgOiAwIBQAOCAYEAq2O55VSNICePERXCHlsi25cl\n"
	"u4PPwy+U/MO2A6RPbJV034O28FWGe4riQ90QFHteHS1Sv+taItyob8xs+jk2rFY0\n"
	"TlwoH6Q2zMWzpW0Z7ccnpSE98YzSNwf0Qx3xSGmCvJB1NDFdSbYrpU856nGanwyQ\n"
	"PbSpJHcnIBgXK04FwgBGhgJ0emm1jPQ5NkXhz/LG9I8tyeC+9AauLGfYd/mt/Iwi\n"
	"HNzZy3lo46SKFGY7rC8yk0O9zXo5+SDEPbgv4MiDBsQLJfjxED8J2eWW8W6N22K0\n"
	"nUyQzQyfN26D/Vrpy4Zb5RvxwVCTeUyy5w4FFUfvSxWLDkTHaBMdNTyADmTNEDt5\n"
	"zzVec9wSWk57TXExyG8k/ViA0UI242KzcG536N6Q4axrrRN87vEHQPmK40j6eC+e\n"
	"NtLeOw/tINMxSj3xFWA2LaMpJmMUQSK39oPP6ct6rIe1gG8J2dmdYzTSGrmtBtJQ\n"
	"9qmDO0u0Fv5MbEWYMkULtTQg4/jyJlQD4S+/C2Nk\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEeTCCArGgAwIBAgIMWXnNyDCiIyBT0gz/MD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIDoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCA6IDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MTEyNjAwWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAID\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCA6IDAgFAA4IBjwAwggGKAoIBgQDE\n"
	"9H7sCj+cbftdBLcDr+u/YR4RKZkjBRk7BSV+uy8ROrk0HRzPDdxK2z4apoIFwqhP\n"
	"kAWgaAVItniryu8TqDpYUMWjqQd2wRPtNBGkwQPBCHNQwehDd+tf0ru/4hRL5pcx\n"
	"A14594DS+oJBQHVfbZPCweITYIrpcx6tKzvMnIlc37pr4KsTftRrCOTl9CZiTenR\n"
	"vFYkAXucR6l9zHoiFABK13IpeLB3+1wlSe+hiClaEoG31XCbXq7hLK1Uf2i1y2il\n"
	"6/fty3BfGqoBQ2BZ9AjmVMCD8dkN6v/+0mXED3nJY/3QReDcxIxiWpxA/8k5Gzox\n"
	"S9mgCCewfWCKKgJt37eZvm4JwZ/yP6ZALrM9uy3ZhCYd4snImy6E0qg1VC3OJFZp\n"
	"4wJgpDLbAKXliMQN0ppswDRiqwlpZW1gUCpybf5vlVN0Eyw6jQ2/6CeTmNb95a2b\n"
	"5H30q5ZPt4NiyepfgA+IRqHexwgAr00/MMbxdtuW/2Y8WQizv7LdE/h6AmXkC2UC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBQHMqYj24DquhwWZWWW+RYVmFdhODA9BgkqhkiG9w0BAQowMKANMAsGCWCG\n"
	"SAFlAwQCA6EaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgOiAwIBQAOCAYEAdCu7\n"
	"QA/Bx9HFjc9794p+Kw6UBrtMdtp46XIyUfYasp6j7Ta6BV4y20OW5DCdldtuqjEy\n"
	"S+nFfUdKI6a/YRLibl8N7VFK8Fqhrh5ltJN7J+Y5eSiku9SHG94iTY6tFCNa5n20\n"
	"DUZ9QxIBi1fMVwCf1yyt9VGx4U7CG0XS9QGkLg/GtFEgyr/4IALw4W2BeluKAQvN\n"
	"Pulyogq4u/Jwh1uEJEMufLXFIDftGxWixmD5aSu1ReDl18svG05ycLyn5KlYuqJp\n"
	"Hve/VX2yegD4clMK+yUtZ+x9ODzVfJ/j1HeBpnLrVzxBNL0HyPuhfb9CNf6kHwdF\n"
	"1lqXopRcidYrqyJp7NHoEhWSszg4RJNE5WOjymbu+swRCJV1ISi9tTxXrIQnbdqD\n"
	"+2FewNFyLsnxfMSnNlF9sIY4o2tnu/8ktWC/sx5lAxLfsp+ZMr2jfT1IalnlK8LZ\n"
	"8YLslBiZkbG03zyadTegADLN4c1u+FHqKtkPnuLm+Eez2V76kYkLfGGbof5X\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_sha384_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEfzCCAregAwIBAgIMWXi5ih/ITJ/QWOoqMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI2MTU0NzIyWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBoDALBgkqhkiG9w0BAQoDggGPADCCAYoCggGB\n"
	"AMWYGBwas+HiF+8VbtcNMJeJydmHi4GyXFlqmrIkKheVdcag590oNPi/XlLL8a/F\n"
	"ZY6rZkofMJmghpUy8UNfboh2J/orwtvcGHWz63FIdWU/XyOK7+kTeYzrZTZWpFSb\n"
	"6m/SCl2r62IDZoCGBHVY4+c3lWj3VNO5ZoLqB4YihKsrGEnQwuu3hxVjl4dnuktr\n"
	"PJHukt8i9JKtEL2snh87QaiN1jfhSAPverDGNKIJvuAvd5RilbGDSPDT0Unp6Ckq\n"
	"dgXgX6UOZ12Kk1DYWinnvAgIjMQme45nvLrwHVGpt1GfvzLteXNeg0+K+QBldviI\n"
	"vppxvtahwDrPmsjwWP977zPNntOhU3apS01P9HvPe4xNgFlYYaTMPRTVXkOZSYEu\n"
	"RyzQIM03YpB4sOJFMyFFH91VghKOiN4zIHPtz2z0k9gUJwUKN+DZMhWbbtDxwpW5\n"
	"FYi2QEVQohrZSehh6jrh81dVVBgvMCt9SCWXkt82eA79PLJxYTLlG/tn7oALxLdm\n"
	"NwIDAQABo3cwdTAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAP\n"
	"BgNVHQ8BAf8EBQMDB4AAMB0GA1UdDgQWBBROq/95r17hi5grwFWV+sYKZ75+iTAf\n"
	"BgNVHSMEGDAWgBTt5tc//kUHWmueMbpe11tTHkNo4jA9BgkqhkiG9w0BAQowMKAN\n"
	"MAsGCWCGSAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOC\n"
	"AYEAtZTjZJBvVRW2DCY93yLVwBCJJDmeRf1qzR9+js9Ws+pu6ZL5qIdidYpNpB/5\n"
	"CGnjEdohq3Nd8vgItjLmP6p/JCSVEV8nc9W4IcnUCiZYzggq8CKDK4VSwZPKrdyX\n"
	"dW62d9GWnQ4VqvLF/h8hedrxujKVEZhVJDoHV4BFWacQra81Rp0PzT2d4RzoA/qc\n"
	"H0wOZQxcyazDPXc5JI7rN3P7uFaOwdnYsB5cDC4/XOygv/ULEd5J4OD7PM3T3+nw\n"
	"VnwyaqMPtMJNsnO6j2qPhFj2q8DxOpxsyV+o3DxG8s7SfdcRdMKOPH2T7x1lcXm8\n"
	"l5QWd99kHEFUqwB+4R+2TKH3C8AsDnJq1x/OmfpyC0l3ZSSS64MdzyrSQKn56NWr\n"
	"kQjVOpMMvGrdFkofZhNe4Dw+XApzFtsKaNbwZqGwlzvoPASlJxEmLHbwGQQQtMaY\n"
	"JfNRuUtZSf2dQdFJXKpt6vClMbFhp9t00G5+XgU2jNi8KyaNQ49VVM5SDUmMxOo3\n"
	"zPQF\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEaDCCAqCgAwIBAgIMWXi5igaqxw34FqT6MD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAICoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTU0NzIyWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggGgMAsGCSqGSIb3DQEBCgOCAY8AMIIBigKCAYEAzlBF\n"
	"LH8Y+QjFNrlqALm29hRiIcjqecggFN/4RzhJ3HmfZ0pAx+WGbv/55hW1e2Chh+E2\n"
	"TCfaX6pOy4Brtu95+mcySteQPG7h64HYr2c0Ts1c8HhSJy8g/5Y8PhTiQjp2ai84\n"
	"pDJ9lgnzbjX/YtES0A0m+sbin86lScO2m567/0HG8DexwwOJMOM+DguTViK6C2KJ\n"
	"OAqLuavjDZnNr146rLU7xbRqYDe0mKiINJdw2f4qif2QP0gmKmAoprogKaCm/94q\n"
	"XFzwUDkgsYKQf8swiDIvgVT10UEvn492MtE3uo1s6VaCW3gDKRB7yaMmrcPu2ZkY\n"
	"Emt3LN8mMt0w19e/ce8BoP8qokNRCv+bbPzFeP467IwdC4oX80C6zmEG0cDP4BeR\n"
	"Mt6fAWPg14p2yE1DP3wtp++iGUrdNj+N6bkZbRc5Z4KjZIKcQC/BFWXqvY8wQkOi\n"
	"R6dirCm+wycvILjectNNwA1c4BQ3ZYbqQu1g0M1ytcthtwfmk+TJ2qQibUfpAgMB\n"
	"AAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4E\n"
	"FgQU7ebXP/5FB1prnjG6XtdbUx5DaOIwHwYDVR0jBBgwFoAU0YmZVorhECgBkUd2\n"
	"fU8+DltCWHswPQYJKoZIhvcNAQEKMDCgDTALBglghkgBZQMEAgKhGjAYBgkqhkiG\n"
	"9w0BAQgwCwYJYIZIAWUDBAICogMCATADggGBADh6vayMUYWLPnIRezxYzOIEST/H\n"
	"IQiCcYobFpaecyMnt47wndTmRw5bWJljwcH+lY5VLAGK3CegSQnANSngbnrraR04\n"
	"D7031eQgZtYuzQPsqiGp4k+Z8cXz9b/3xgUThBmET9Iu5k8IZvhmIkCIbARMPA05\n"
	"kP0ouTNevmkpCZWCggbXPe5H9O9gZ7MC4z1WVwklte1p6+BUuiCLnp+hpLsMAl6f\n"
	"DI22SgdRiOfNtretrW+U5sNvsbvbHb18+iBfB/6ESQ4zJprkWUSLSK9nfB++j5LC\n"
	"GLoMsDaWFAtvq5hBVOLfcBJgVpZPmqynTeHysjeF/H1j4mws/7//85MAGRmnHdQe\n"
	"nFJ8xtj4k9RaKWtlUnsF4LSo/i+34cU8+ETxjNZU+UrupXz4KmHIho+DhYpJtW4A\n"
	"yYasAvX9fTuStHJdLfpMGVN0lkOd2kuDhsLpsmdsMOZrQw9PPxlfw9NCdWxYFmp6\n"
	"8g7df51DKxx/CDrLXQLzqp1gc8kgM2eIZ18z0g==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEeTCCArGgAwIBAgIMWXi5iTjjcE3NZtmRMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAICoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTU0NzIxWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIC\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwA4IBjwAwggGKAoIBgQDD\n"
	"7BbETCcU6Y6htR4lbDPrdS7XzDZ3m7gjjsgmZxZqPll9STNrqeWdOe27bdn5t2yQ\n"
	"vOxv7JN08hV8JJ8BzlCOVY04RX5whVypdtjgolvbpNmNdG4lpoBN6ZfV9AojrQx5\n"
	"JS2gGZcA8tPpPk29anX6N7QxNcIfbaHUK4IiVBMR/fpAq56DzBVP/gTPC7FiQ1kr\n"
	"iiLVQZriqnoBDf96igaEtDnz1cbIluaEH/lKtTgjlA7Dx8AOfS2VfkKHEJceerAO\n"
	"97hz/i02Jwtmn7JHHCujHKYjuwy8Ieqk5gGk/Lny+sdtNWLkNhWQIzyJXSQWUeiB\n"
	"mh6MpCYUr5x+3RVl9xDaR2ltV9DF5hL2yE3ZYzbhpU6LLLn82DK1PARc8CinIZyW\n"
	"coo46BFc7NfXUnOnMZsCBZnBiR9ZIpnZLBHeAxMIIsl4PiyJrw0JC0NS8Bd2Bgbo\n"
	"JdpGV+nx4hDVsrkINpBaYJJUXbxgD3noEKZhamtiYO4UTRk1w74ceHwJKAH0hWsC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBTRiZlWiuEQKAGRR3Z9Tz4OW0JYezA9BgkqhkiG9w0BAQowMKANMAsGCWCG\n"
	"SAFlAwQCAqEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgKiAwIBMAOCAYEAvuww\n"
	"C9ndJly8/Cp6WHu8Wt45a+o6Zup2mMiLX5Rg9Y0i2ZgrbXvY4tbQIsexQWNXuKBk\n"
	"U/p2ReBB6gletZgtyNKeCHTqpGyHemGinP+Yqw88j2Z3S8xOHN/zR+uK+iUZEKZk\n"
	"/atCDW4BdocaYOGJjTwXI72qsBeyJSDF3qCrAgYcynzd3wmNdl97OdL+/COgIy46\n"
	"pYf8skbqAd6hOymTqGD4wI+TNPfjXmstjC6leeUvmAX2HuCOJWkwVp0r0JGs1qK2\n"
	"8meesblAVeW2XqL0ScN511ME42rxt6wwSelTepaKlMAsysN58L/JRgC+VqD2S4yS\n"
	"2fe6OfuWItbOWDfvCRgufUzHzgGnslf5U0Ejze8xLhryBXz17Mnx8PGXss3xRqva\n"
	"Urgy49T8eth03GV8mTnEbW4ixH/5LRq6Te5ttSeuHVtdpnDAbB8MKZQtiNk5mDNd\n"
	"PXNZOty0RPQ5q2L1Q1ANd/zsY3bCoCOu7ErJBXYMyXCBxFtTUWOHvxnDt3cZ\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_sha512_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEfzCCAregAwIBAgIMWXi5rCrgP3sDVAJlMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI2MTU0NzU2WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBoDALBgkqhkiG9w0BAQoDggGPADCCAYoCggGB\n"
	"AJxfUNbtAMdj9L8j8H5MVFC4WUATVth6BleRmS8B5L6SlTaMofYBXUAPsWuLm1UX\n"
	"NnrPzkLmpt1QaHl8lwgdISiP+Ka/81H9fs6gdEZigadMO+y0/YT8/E1CVIhmmuGQ\n"
	"2Luo6cdYZ15i7OirtPqm7TPzHeE2xhL8JX/4iuTUphM6f1yLhC9pnv+4FCtODkcT\n"
	"bl2IpKuTImfmxkymJeUUkwpR0CvUi0PFuafwLVvGMo1qyx8wkvz2bqq68l2VzqPP\n"
	"fvbGWjOX+dpHnPH0wUtKI6pvO2v9SwwuKBywRU11ahgovBGG/6n+HubwVGp6S+V/\n"
	"UShmILq0LSOyVdbYqF9nlKp9UYTDf266gUoDoSZjFw1Mfx3ntGh7uS6vLNvGdSEl\n"
	"1JWe3VDEanyYzgKBrFPOKOL5leKRFQtNFjPTIzYxakhtu4wFGw8yqrDACsNYloZo\n"
	"VPys7B69OTvvkrkNRHLZYHFgveUFAq1JDHMPtFHO680najV1R0kyaYurVXmm6MS5\n"
	"qwIDAQABo3cwdTAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAP\n"
	"BgNVHQ8BAf8EBQMDB4AAMB0GA1UdDgQWBBTkNsRnUzHbTJkw8PC5cGXFKwsIyzAf\n"
	"BgNVHSMEGDAWgBQ1g4DM6qmT8uJskGxeUUUSIPe3iTA9BgkqhkiG9w0BAQowMKAN\n"
	"MAsGCWCGSAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOC\n"
	"AYEAIPSvyEoh5C7wP7M5hqJnfzlV0rs1jeLNNgfc8GCbxKYepCZysD6JufFDBnYu\n"
	"qE4/RrZs0kDJ1RB2sMqTcwUEuZBIBLZtaRj6Y9dsBrPkC3+1DuMVT4+AoeQZMKTh\n"
	"lLsMWba7Cm/lV1SG+cxKv8Kj7mrsrFNrv77td12rofIpYt1gxhgz0k5GrsNtRpyX\n"
	"jpTk8PgiJQyGDeEG6HrG0F4VvFzvy6XeIOjwuiXTbpzC4PwpUt8WHkp/Xm8Loiv1\n"
	"5P++mByZtXSCd87BIADD0YapQaXDV0a8NWM2QUTcZJw/p2LrY5QOkO0VepWkifbN\n"
	"1jIbc8kilMNH5yQh2JWrDRz81ntpcIlFLwYQVzJguZ5Tr6lHnv6d2mnBUzbbv/bs\n"
	"scMlxv+55LPviiHIkidIxfBENMbU2dB1u3hu+aEOQJ5W1w7RSuRJHi281HathkIo\n"
	"jbo0ZCxUzXIB41YIoESYDOg6FGkoT2I2vCHyW7lLzPnPa7a3PolGeLn64dmmn3mr\n"
	"Nr7V\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEaDCCAqCgAwIBAgIMWXi5rBKSNwkPo4olMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIDoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCA6IDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTU0NzU2WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggGgMAsGCSqGSIb3DQEBCgOCAY8AMIIBigKCAYEA8Afg\n"
	"aY9tKN/1UwFdqmDTbxcxiGDQFTDKDFt4zLEy8HoqsiTLEycydVJeAEuw1WNrph1x\n"
	"nphDETOsiG429CEkIj4rpNaPSevQmfkUP+NFqKgf3egUInmXzSMnKuc3eiDXzSC9\n"
	"mcYzcs3O6kDruoTBcmujSQxdcPYdj08BkM2uD1PlHVeE1h66axt82I74q8ntT1Zx\n"
	"IM4TaLSao/Xdn1i5AYHwJj3DzjKlYDuLqkAiyQDI/NrRS007MYRLN4Ebu6bvkuzN\n"
	"6m7eXYPugV+lSkGSLTi0cbG0wkUqcR1X5JzBqHyXU0epoz3/PpVBwMUNHMun3s7z\n"
	"TQt5OJY97BeY6l/Wj259iBYj41UvEghT67smaM8zvwFb51+fCPLKPUXG4A2Ksx0k\n"
	"H+HIP2TIIQbuM4KAS3VmyFNoxzOXs89BdxJCQ+D83RZHSYn4t+76fiSzV+I4baGi\n"
	"DbPVU7cM5CrOcfTohP83jpOgM/LbPyptGu6S6GKMx93HVLP6LtnZE736dO5XAgMB\n"
	"AAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4E\n"
	"FgQUNYOAzOqpk/LibJBsXlFFEiD3t4kwHwYDVR0jBBgwFoAULmo+wdwsHxfVzvUw\n"
	"NyVK9++NokUwPQYJKoZIhvcNAQEKMDCgDTALBglghkgBZQMEAgOhGjAYBgkqhkiG\n"
	"9w0BAQgwCwYJYIZIAWUDBAIDogMCAUADggGBAIXyJ4S/dWmkPV3kBUENfIXaLV57\n"
	"mGJjnR/EnUX4gVVxDfKDTNGq2Y1ksCeY1JmvjSHZVkX/D4p3BCHF8bHpLvS7Edts\n"
	"4NpoL3A4MBdupwDFtF/0Fo4VdZM4ztLL4gBCq2pnukCkbyELCPpe3d/yVujsJNrQ\n"
	"4faiJMwCjep+3q0ZiytlsN8M3bdGy8ocbzPAi2rMTvQ8I+2e5kLTJmatJ4Qbut25\n"
	"d1rfJ4ruMt2QOrSlYSENKkA3zjRAg4a2xvVPyOVZBEj48366b1uuji/sOQRckZ/w\n"
	"3eoeffRfWQXO2y0/K9TUqZM+6n10N32ZkR45I+XSQ13qS73l4QS4djay9z/bAMeb\n"
	"/zgaf6J790LULzDBEvhPZLNn4bBu/t7WVj2NI+frQvAHyQ9ZhBYkow84qF+//zK9\n"
	"d/VzQbBQOJFX9TWdWgUxklrWnXE0gmxzGBdq+cMQyHulVVbgShftCRJ8jn8e0Cl1\n"
	"dl+Cpj08yyLpT9/ZmL8ytgD3Iobw0wPHppb/jQ==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEeTCCArGgAwIBAgIMWXi5rAVAVYZtSJyOMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIDoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCA6IDAgFAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTU0NzU2WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAID\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCA6IDAgFAA4IBjwAwggGKAoIBgQDA\n"
	"DS6aZqSaUEDHL+1DTQtaYK8IO4GgdfI33+C2VdIe3LGjUJMsOjpKjodrSe46gTyW\n"
	"fBAZSPHntcbckOJovv6fXKDPWgvh0s3AWZjVGtGVidXbEFzpps+YKOLnrVMbvsJn\n"
	"yKIZed1RvJhpGs3pH8hvI0dV5udpypGmtiGUQe/6qC0kUfVix1luN+8A/SyI9tIS\n"
	"vzgDORlHI6/2w/HlVX7lMIjpf880rx2BX5XKDpYmpMqj5yOOK15gKyZLUGjC7OWs\n"
	"cuMGZz697fkmdRjFDLALQYkIvhFKiPPJJttV6ZB+wQ9daCgKihFl1cVD9w3AznY8\n"
	"lcspJvGcFp3wtIvAvQ84P4F6g8nhs79Rdng5F2Qkr8NRoeGKfGLJT4HFOTwIG8Xa\n"
	"Ily+i38KTpLIJklb+qzd5hURONxiaFTfG+FDcrySSaUC7+OD4ZitDLkfCidcfMmW\n"
	"pSvld03gmw6C58zpaNZdztQy5rs5Cw3nI7RFNU9ADEySg7i6aUe52tvmgJkaIMkC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBQuaj7B3CwfF9XO9TA3JUr3742iRTA9BgkqhkiG9w0BAQowMKANMAsGCWCG\n"
	"SAFlAwQCA6EaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgOiAwIBQAOCAYEAS2F3\n"
	"Tg2gTIXkyg8vV9OCfH2YgYQpTFLXyfLOLiuQrSXgHUOCO1RV1Jkjb8+Q5hMrPOMZ\n"
	"Dygb3LYlF6fEWW76Z3QUqlshIfCbWE7+MAQSbjHN7Awbq9c44edvm0JuAavVsk7M\n"
	"Nxx8ISNqlO1BRlaX0PscNyiRBhrPrUzAfPaiiguKXMMhFnp0V/aRor47ZePhRrRT\n"
	"NKX5aSBrVl02gQldK5uKbywa2k4SDG9OuxtHfBwjshHoWbvxg3S9lvhgzR4yaXWh\n"
	"944QjceIEelUz7czweTUdFc5SSheUg78SGnXWOsR0CljXl+r+Laq/Fg69VZffjJJ\n"
	"8A4J6eUf/2Y3h+sJ88/Le80bG7NUREfhjYfC7VAqPiHcGTYQUjw+Ti1yWmWt1rS1\n"
	"khg529kYVRone3UN1/Fs3hquwjZfL5ZqUR67pyMI+QTVvVeSBCthSafrhCCr8xpO\n"
	"OpQuLLx9v/QXqtg8IhdVoDxsQmdLU7aeM/KfuTujcqFeacs8ng2g2hEGGFa0\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_chain_sha256_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEfzCCAregAwIBAgIMWXi5YhOIRWm7jtzAMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI2MTU0NjQyWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBoDALBgkqhkiG9w0BAQoDggGPADCCAYoCggGB\n"
	"AKrMNcBiv53Gc/nZlcHxl5DVW7kqKzpfcy4ghZXEM57/7FztPaYU5waGbAcci/yh\n"
	"m06lFD0kAyN6FzMZ87+o4PK6Ov7EwAg6WtQ5eMvvK7NXMnX1tSGlhQYY+a+30n5g\n"
	"gO4c6TXB4mwAWeWCk8RH28apwPnQVUDhxpmtzP/XAKkrdp1oTVUKQaSrkRrzBYY7\n"
	"66x0ea1Rqx25piSkN41FxGVYLgKY4UbAmVJTGWTM9MPLwCCjq4DZO9QaHT/dGpG1\n"
	"FPEtmfw4ttSqJfGPT5ikEfVRw/h7WaPfrf+oHVktd4/TMHXGIbVvEqueTuVXm7CK\n"
	"k8bKvMOpS2FFFtPJbbatUJMfenN+i9epgiquuMdkp23ZCutDmdNMHmy0M6RSF3vd\n"
	"6bg55Ghd71OJNyFZ3o1QoeokrL8qmFER23L6q87iCed0Vtw7Sqd+84HETZ71RI7r\n"
	"Qg0kKd+sQBA7vKy+XpmHOiBsmLEffGHTL/pP6Hl7EtP72NdY3Oayh3QqAuPIoxYP\n"
	"NwIDAQABo3cwdTAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAP\n"
	"BgNVHQ8BAf8EBQMDB4AAMB0GA1UdDgQWBBQuRFAWSS9Dn8SzG6lK1E82fyDDVjAf\n"
	"BgNVHSMEGDAWgBR0nX+9KcaAS/Dvj0qxpYFUZWXFRDA9BgkqhkiG9w0BAQowMKAN\n"
	"MAsGCWCGSAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOC\n"
	"AYEApluSi9xK1Tzgvrw8bob4yhNthQeD5E4d9kDCqiI1EPooyuj4M8FlzWUyYgk5\n"
	"IQ6wZcylosGzJSVJp1LcFKpVzRAxqJk49M7fr6XwqKjn9bezjbqt+VjHEmG9imeo\n"
	"sG6tKQ3fnOlP1+RgmWuq8ZDM1gkv6h3WPQHuqYrcyGxvgh2JYp6TDzGriTLEj/MX\n"
	"PX7zJr8qSM4YJzt47LQ2k4zDay9G9SLem+tcmjc70Gd1eULCl0YAJ+e35QkhKmHO\n"
	"aEs4nXXR6S7kVTIojF64e45JhqySOZailZIoCTxYGV8fqIJc9iBDmKeruGfgUzM+\n"
	"0+FAprmhcqFOv+GN5b9oPbcWuQPwzCqMNWd0nBI+w5M0oktfdVvtsUw8Dak1xJIb\n"
	"mCuTlvj35a2i5fA3j7BYT//Mj/hTNJXTvIPUVhQ72xIZ9MjkiAQM49wz1xZtQQII\n"
	"ugHEN5LS4QUdjM0/2ZhAFUUpbgPg5Gg3dqMw4tuYYDoLe0+SO3ZH5jIN+5HrbQAJ\n"
	"pKuA\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEaDCCAqCgAwIBAgIMWXi5YTrysOexHvTRMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTU0NjQxWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggGgMAsGCSqGSIb3DQEBCgOCAY8AMIIBigKCAYEAvhHk\n"
	"wO9njQ/tVzWKDXgTAcP2cxSAbYNmjgflCkhdrF+Bn2BKR4e0q9wbY2sgK52rCTqg\n"
	"gVB+1Aem/C7U8b81X1JlsKuh9jarmJwPVK/oRFW+L8qbmHRKktgNYIbS4qgyQUqR\n"
	"vh2hF48Rrr5HudtvvL5oqSrqNvpZkUzPzIkV61C14+ujeZiMuycYG8qT6UL2fRTC\n"
	"/qGwQWUvaJWSCEQhibMTeE8G6zO+O6hSxNZJVYdM6HT/AutEE+hvkfY3sy2PAAmr\n"
	"2ea96Tt3KJiifDvdwH3iGhfHShBY66pvKF7BFpyOhrxCZaDxb5XqfiQWLZAM688G\n"
	"sYqOLLaMgsL7KMUh2c9OA3bHiPed5nLap4cFBYEDT3j010waJIYLykRcRYV8ijc1\n"
	"jLmy0JDW+Gvy7JvuVMLisvqmafQIapAyFjvxOnw/H/TD7HRaXmkDAK/0XwPDObXl\n"
	"VsTrHH4MxRpmyuHTMrns7usuQiZABwmqk++Hx60YiepwDepAyyhBBXqyecD9AgMB\n"
	"AAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4E\n"
	"FgQUdJ1/vSnGgEvw749KsaWBVGVlxUQwHwYDVR0jBBgwFoAU+BBJLVPFaMA3pFQ8\n"
	"ePgsaRye/NEwPQYJKoZIhvcNAQEKMDCgDTALBglghkgBZQMEAgGhGjAYBgkqhkiG\n"
	"9w0BAQgwCwYJYIZIAWUDBAIBogMCASADggGBAJCZIQzChHgacm23b6shZucxQOUz\n"
	"k5AQRSQYTc9RIKvp70fupjFY13wpX7EIyJut+x4UAbXWKm558siMWz1Bgz0zF831\n"
	"2F2aQMLoZzQ+85IjXbgSFLmjOZggBhQbb16LfXX+YNs2zeGkmpGyLDNbID3FlteY\n"
	"Qig4ELp3XARl4gzbc6R+t78AidkyGxjFVfYzI75hA7Y4GC4kWqoa+Wn1MLlrnZRB\n"
	"r/J3ECyB+Kg+izdrDlPtvfcoiAFXcQjoiB3+BG6WrT5fEBzNJXr8rdo5rSxpxKjz\n"
	"ilEnHQrcQqWIH/BeeLTmjMYEhYVamCqxc/bxxHhCMly6dOQmtcJdc8EufssqUAno\n"
	"qPU18vBYTAoKSpS2euAlOY4SYGaLUtAKqdHV7sx+hglRW3jZfmKN/RJD4kOM36vB\n"
	"fbQ6KjVn4GbFgJF12xB/fW3C2eabTIcG0tTl9+JiOW7YQrQqIkmFNA8PSA/a9Avm\n"
	"/258r4j2gke3u2P2wWbZBaF+DoukvkJsCHBH5w==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEeTCCArGgAwIBAgIMWXi5YS3G4a8ITSrbMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTU0NjQxWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIB\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgA4IBjwAwggGKAoIBgQCd\n"
	"NTGy7akZ+d66CnQ/g3usM8mr+qonNxTmQXvjPslvXYz62OFpa0lMDfO4Cj2ylt/U\n"
	"XMjIhXzfJ9/btovlXSQMi4jAByXesl+Pn3uklcy4W9gaQYqAyxZtFriy7VTpolkb\n"
	"0lr+gVBc6Qr9Rko/HpmlMJcGJTSbm/oVjvAnLpC6MEuBtR0uWoPb4JNOS7Ycwq4g\n"
	"KpoGOeZchqnFLx4LUl7m83UQsAoeMeilHV7xhbgDNI3x2vmVTYs2MNU7H1VS2VRU\n"
	"f26IUtp+m8k4KAS8uZBXOeycYkXll6vONhcFlr5m5ZHH++/nxu3HjjisdW7rrSst\n"
	"NGYHR7E6/2jSG7PnKfxb59/PTfV8NETIG+ZpFeWeWmJn6DnOIsQ3MQyaLGqmRCae\n"
	"OIRcljx65gN8nRxHN+H35d1+GIfAUQ5v4UZgRUIpmMPpwUi3+zX1nweaS1Orjqdl\n"
	"pEHIxCWB0CGc+/TmTL7qiJdujspeos54iZLJOchvxmI8/HR7kzH5xgwzqCDUAEkC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBT4EEktU8VowDekVDx4+CxpHJ780TA9BgkqhkiG9w0BAQowMKANMAsGCWCG\n"
	"SAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOCAYEAH1xu\n"
	"N7vSt3LLfjZ/0Cby3xtZNi1+m1FCh6Vn3NLofFYRVmHIItK9PszRaUN21UKc+PnG\n"
	"XkvVU6GePfMHMdLOY8AR2We9OgIAxE2aYva5TyG1EjrfmCth/Qj9VMYLuIu02QZT\n"
	"WTNBKCAv69EokxeGgk2zHrhwMvdHIcngTnLT2P41Pi8IIaYy9O8cIpmOAmbXh183\n"
	"Vk30ik/6V4jc/Vc+QWRGj2DBTSGuXvg9UpKX5TZKp7li3w0oczL4BPLEyyFbZwGV\n"
	"9ZIi9G0Viqch/7sTT82extpXgtWAbDf25QCBPKuOS2t7uRjacEpFDQVa+97haTK6\n"
	"JpxZ1Wi99SXrasKkqS3AQTzUWAuzfQ8BU/s1XC1mMaZo107XXIc3tWyv4QhQV9Wk\n"
	"1R8QhPE7bozAkyU8USWnaT433stfmkceu1ZwjX1kts6oAxk7xQUFZ+MVNYlyehyM\n"
	"n42VNJ1m0aKvz9LPUWbgcd7dFBTBuHM0YnVaGhY3h/dvrmubxylxbG9xCp4M\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_invalid_self_sig[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIID5zCCAk+gAwIBAgIMWXl72CvN5943shfAMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MDUzNjI0WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggGgMAsGCSqGSIb3DQEBCgOCAY8AMIIBigKCAYEA73m5\n"
	"jEDnwLR5SoJJQ5yjJO09+ngRY4WwxN2RMX3g75JtBSyTjIC6OKvXwgJDGVuUnl/q\n"
	"yU9bNvM0PKoRLeylqTAESfv5tjNS8ievydgaO8tJBMhSGyzHZvHQOc6VkOJAIIcm\n"
	"WyduEyAm7oqNSDD5cDfXtvTCIW9FsjBgL0sG9h8j1Dyl3ZPM0UkFkmaXJSkIkVPa\n"
	"2Cl619MnzVTngsUpJmIpKzl/uomM+Aj3QnzT7V6XtImaC2T5/rH896Jr5hVTImiL\n"
	"5V8YjTVb7XWt93GLmhFBykCJFXCNAbWyax8TBy9JRUAsgouSA9q/dJ6GbRLOcWXb\n"
	"Rr+MvuZYuODY8p076GWItrd0LVabJbr3CKwE/RBqxY5rvBb/wSYNYcG0yt3Oo5hQ\n"
	"t01WqgmOdlUXIEBQ2Xx/kwPvSatWd9f0WyhlXv6eCugEJzYpK+75xpAysAIbLHZv\n"
	"33Tr+eArgNVoaJGYNpZlwW6ZmkRjuTKYe6iHYs72uh5H7lNaALyfXw/gaBwRAgMB\n"
	"AAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4E\n"
	"FgQUhi5JW/4DoCk5RGaJdF7DUsbXKUwwDQYJKoZIhvcNAQELBQADggGBALdQ3Xeu\n"
	"4b1lZq+w2G6e6MTpAljGtyCN2hdc2xaKYOSH7AIuW8Ahiuo1rhUskYzBGqQ/rmuu\n"
	"p/uo744ve6wbrIt+DyUZJGxGKOxZab+Hg/9rW/fx119kk3tCQMxSE6yI0YeuXT6B\n"
	"hPL4LtoeZvncQcuueEIUmrMfPRYBuWfQsmhAXE0R3o4lx2DfnPtjSwUo9A+4AVtO\n"
	"4sp29n2qfGsOIKrfMPrXZGQ5XWhAw0Ra0CIlFHhRxoov28rssiUIqLKS0ZrmOMvV\n"
	"R5LuJWaLEb5W5E8LwU2Yxu3+0BFtRSy57HedebmOeDkmVuG4C8Km50ymdonafFQV\n"
	"rfW9Ni1eqUMBD4BXQD0HDIEmQiBEBE7/K3rS9ezEIxpQWgK6dlYoaxGkybuHEos5\n"
	"r0rLvi9CmmJRdV6+4jeuY3sUBRva1v8FYXHNZ/USgxRdUfpomElwu2C/af7X/ov4\n"
	"NUWJzE9Efdc4rtkD8BJ0iwq4tfRx9Smp2hqsChVuRu8mkQooNdB4wS7MpA==\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *rsa_pss_invalid_chain_with_pkcs1_sig[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIC1jCCAT6gAwIBAgIMWXl85RG4QFslQjj5MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MDU0MDUzWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0xMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEyi7p\n"
	"7NiEAL4EjBrqbN0jULt4cHYYezi+iaJngGgJMeltsiNuPCjFRyBvvfdvQu6Va09j\n"
	"CBYE8F4dyUUDOdB7TqN3MHUwDAYDVR0TAQH/BAIwADAUBgNVHREEDTALgglsb2Nh\n"
	"bGhvc3QwDwYDVR0PAQH/BAUDAweAADAdBgNVHQ4EFgQUicpVAAomaffbIpb7diFW\n"
	"RyPMN6EwHwYDVR0jBBgwFoAU877t3BygfVZ2TIFitsZClUs6rX0wDQYJKoZIhvcN\n"
	"AQELBQADggGBAGW5Mu/NO0CFExmnw0f0G9jlOl/NDiVhLXDdZjlSLqi3T2Q+/an2\n"
	"DCCeEVnsDdAFC/+Xnmd2e5hm1aj4PdA7Ua935MCDEzd4fqkdGTgGBqk8eEjWaJT4\n"
	"cJidvPE+lVQEfAoVYA7MRzgtoa69MCJY82mdvK3ojJdezb3Is7eqg/FCItbhNQUc\n"
	"NSCHCe0kuZt2w35KMxtEbAPsdaquB9IpIyyIYDOUhOI2n8NBGdYTWk4ESQWYcy6S\n"
	"5aaKy5uut4PWvAOUjPzJBiiAaDZj4Ajn7rZiODxkMZE+1TIk+S4qM5H0Dt6JyksY\n"
	"GN+0SnyXB0/6y3qM6ZDXRMjkUdPADnz0mwotfigZ5TMOdaxo9zgr6hFrIolGB3ot\n"
	"KCL0ISS2l9eB64uebYbGyMOpSMOoGfypR7bwr8XFBxl0TbBosmZTMJkbM4CjsJLA\n"
	"WPLHhT4lmVinzcbdOdycTFOKwECNHo4mTBjiUrrhndSjlshUZJe99XJuT6FXnyUE\n"
	"XbkbJWt5LqAlxQ==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIERzCCAn+gAwIBAgIMWXl85Q6kSh5LQoCEMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI3MDU0MDUzWhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggGgMAsGCSqGSIb3DQEBCgOCAY8AMIIBigKCAYEAyQQD\n"
	"sSFXjTnXQaa57GLvSU6C5p4U6xQL7eE+Eia7A4lNMyRnkhYtGADCD5052gijhdXe\n"
	"L8R5D59sQeCti4JGkeYF4uwKb7uyGgQGyaDD9sdwJBkZoZ5mcyPJq8uG/93CFf7o\n"
	"ygo0rnFktHE+Hs4EUbJAdf/kxFij8PxJJFSeyz9GrmZgkUMFavJZcDsgclfp0xkE\n"
	"fA9NkoERDalxutViqjS9OtcqdEfl2Wj1/kVXYb9uYHMM4lCeGppz8s5c4CzXbz0F\n"
	"E1OXGGPpvw9IPPesMrBByD8Oijx/5aadBgTfhQ6Qd07bmJahn0S+jhZXoUru5tT2\n"
	"TvrAhJ/6lDJfFZ3rygT5P8vpC+J5O82zVY2sAMq5cfUb33FteXLDy2/RqSktRhHm\n"
	"/8bnQqPtw/2QeL6oUYPN21uWrLmvIc5pJVdz6yW2vhWkhxEVCIrh0zPGvuUpYQEN\n"
	"MN5ePj7rfzEsvRsKLZo+DbmKMa0AFLNBzwmz+0Ei7H8NqQjw4wlrbcjmi5GzAgMB\n"
	"AAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4E\n"
	"FgQU877t3BygfVZ2TIFitsZClUs6rX0wPQYJKoZIhvcNAQEKMDCgDTALBglghkgB\n"
	"ZQMEAgGhGjAYBgkqhkiG9w0BAQgwCwYJYIZIAWUDBAIBogMCASADggGBAJSw9jj3\n"
	"jbO31cViLd2k82vxE5o/SBgxru+HZZDvsBmSxNetSFGSymn+um7g8T/t8fEjxDvV\n"
	"jXDKVnHoj+xuYR3tS+qV58MZ5F9oFrcKiw5QWLicT/DTG/mKwEsKysdibbbtXcQQ\n"
	"zIbVHJI5IH/6e9qw+U4f5J7MzFYWjzo5TavvOo2ZL+kzFuxC61X81a+0XQjP4bgQ\n"
	"r5CFauTvay6zOevvafOBlMQ3V77UouFshbIdTXJHRAcdKUGrTwONq4tDoOVSa5E1\n"
	"AjGZop71IXNncNAcyx9fivxrYfFAozKum4GH1S0lywD+wDCSbC6nkfaB/C12F/gR\n"
	"tEb7+DwZbiYroERhE/zOMUKcGFw0+3Jo4jyAOicmSetknhvmhIoNycaR9KfA5xB+\n"
	"Avw4VnNI5M8TVmjgyu0rq52gH21d1mc3alhOh5syNtLdBq62ookxkSwhaebXgpsJ\n"
	"/0oI+7Bmh6f8ySWcjUFLW+3bPKhuRh6WmkDHQZbECOkp9N3BPBQsdbfqTA==\n"
	"-----END CERTIFICATE-----\n"
};

static const char *rsa_pss_invalid_chain_with_wrong_hash[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEfzCCAregAwIBAgIMWXi1MgbpHZw/A0zTMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEgMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwNzI2MTUyODUwWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBoDALBgkqhkiG9w0BAQoDggGPADCCAYoCggGB\n"
	"AM0Hb+o5YO1we4mN3Z9X11+6i4EwmSYoXkaq4/UAJweoyi6xyDw1wA23jJUHTxpF\n"
	"9dLv/30Qo3bo7fdakJt4fV8tGBlNrm2nPkE9LmQl+dIjtBdmNO/X2qHHVt0WyxqH\n"
	"XEejb6oDNPtF30/dfe1qeuIGAB5cSdmm3Zrkk5/JmfNBuALqHuKol8hympvxXWMs\n"
	"WjCnCd3QGLGIx4nxx1tYKoDVM5xgVq2QLa5h0o5eMb4BYI8KY4dXGzYr6quJtF3g\n"
	"Uw0fX9fo+OclSqx+2TaCnJfWhKFHbS++s/RvOBdDaGJZJcMIs9Kmb1HI4Dmb9ZDL\n"
	"g7B9kraw/hPOgKzGpbCgwJYhxC93L7fNrO7X99b2wIyM5+/Q9at6qa3hm0DEcYTC\n"
	"MZfyLNBfJSQOQBV6NZGwRv7KGqaZRvsiWU5za+dL5wKIwDIxaNX4HIyFoatzLf9R\n"
	"skjyogRE7bXBVk+/UGS3aof2RWl39sOCVbqfauRtFXVKoAmht2kMne+nSrkYi5vR\n"
	"jwIDAQABo3cwdTAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAP\n"
	"BgNVHQ8BAf8EBQMDB4AAMB0GA1UdDgQWBBRJUF1FDf/yxGw2btyh9Ps4cNgZWjAf\n"
	"BgNVHSMEGDAWgBRuLwzcgNymMpZ87aXHIYLe25ggYTA9BgkqhkiG9w0BAQowMKAN\n"
	"MAsGCWCGSAFlAwQCAaEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgGiAwIBIAOC\n"
	"AYEAAowi/Rj1T32pah3Zwo7FVeADn6lKf2SZwAdfMLN2xOYlSgNz+F9Z9Mz4x123\n"
	"GRIoYRmTun2BnwtBVPxtwqaeD88NaftXM2fPyA+q/gYDwmiMzLf3cDC98ngj38xa\n"
	"9YS9k3Mc5CsV5lJ1yiUmpdVwP/dzk4mcvL5+3rb5E0lyv8eEx0r3+rPXnoICkSAO\n"
	"uD0ZYg8AXdsRqToQwutwP02azKEaecjklEkSG81kxlEw6xALkm2qgn7ZogfRYjQK\n"
	"TS5UZldErnWOyw85cTCUYpNs+uy1Lpug91CEI1uXK6pBpb+5HsenKlg6UwBgfG12\n"
	"EQnqbFzOZ1fQ5OgFofLkh/vvNByMmanltqKCV/pz824zi7Va6JoAz9EF35U674gX\n"
	"NB8wf7zMF6mEPAsUR8RIgvkjFTwOxrIfggx6LVs+b1eYTQFzJJF4/zQg17NIlYu6\n"
	"4iOev2OAU65F5P0FJY3IWPWhIl/1ki3drgo/CjZtjYhcZf6EZzS5lY5VxlE0KlwU\n"
	"y8mX\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEaDCCAqCgAwIBAgIMWXi1MTL9A06S6xisMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAIBoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAaIDAgEwMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTUyODQ5WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTEwggGgMAsGCSqGSIb3DQEBCgOCAY8AMIIBigKCAYEApvFV\n"
	"pkvVLfaMySlvBx/erBif/1eW6E2K1i5mBE6CUf22rswc9obFxzJvEE2+HSgtvXzY\n"
	"2q7e+OV9HAi3isQFs3aU1mJGhdweSeill+QiX2KH/MlfxVlZiGAztoswKwNn4ei0\n"
	"Q+B4ixBqRRptpGhvdIMuo4+MxOGqMB4ylhCmW9vp4bjDfbvfatknVLlRxFiNY3vn\n"
	"egD2WbJem4xZfhxWLvJXyvxw1APpDjz9ba9QKNBsmRCWwmXfONJu1M/IMlekqCS5\n"
	"qT+KT9/5E4sMAsBpPAC7St2BBre+OpjpKbhMS6+QJ0td78Rw+ao1yDJKoErXIF5a\n"
	"UuK1uf1nX2CJpBXAdePBP3GWY8yicVYDJojRk5y52GSYslGGKQFw+Sa3M3M038dB\n"
	"0ItPUVPc6KqgMH/WoYNuAosxObSPMXWKTMCLLG4+Ay8kf2LQeNqExFBof5VMOvjx\n"
	"pyf0JsRWxuibFyIsKBPJOQfqavj+yDYUpHjH3KgQWxwFS2zHvh9hSFdVmudPAgMB\n"
	"AAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4E\n"
	"FgQUbi8M3IDcpjKWfO2lxyGC3tuYIGEwHwYDVR0jBBgwFoAUC5eGQfaYKQYzk7b9\n"
	"ZjLg2BcNVT0wPQYJKoZIhvcNAQEKMDCgDTALBglghkgBZQMEAgGhGjAYBgkqhkiG\n"
	"9w0BAQgwCwYJYIZIAWUDBAIBogMCATADggGBACCc/elZRc5w7KFiDtrfiOAQ3m/M\n"
	"uvBabVUyWSdHVq/48n84APdKqUZdi/9qc9vXoV6hu6tE3HEv5oppzHoRUNtuO0WG\n"
	"uPiuA6diyvwO5JZrnAzCevKB+5j5bdfdXh62ScgTGsXyIOBTbW2R4gQkMe+yo3Xb\n"
	"U03hf0PDzpmJL9DcekjNOFphvYyh25jW2sWdlym8pj/Q58Zhx8uDfsMV2yAFzNlX\n"
	"a4K6rXeS1V7zw6qHon0/sZgSc8nSHUNYzXk9BoteZUT3n9T9mDB3tR0fHJx4VlOq\n"
	"EywY0pPylGHfDcKEAJoWR2YuGsLrHS+ZEPwiqkbjEn4O8CijT0dVYC5M8poUwYmC\n"
	"kZDIKeiHMPoUjhRNGQ+W2sXvbsseFDm68R+fqbxP+GHbqThdCaxtqzCK/BNUYF0d\n"
	"e2dvHPdydbqyL9EhfBwvR7blIfvkn6UF0OK7rs+ch+OZUW71+SI70mETVp8N3JXr\n"
	"wCwxtVkyM+5nB+iTDFs7uYw05+QLg7cSnZL5zw==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEeTCCArGgAwIBAgIMWXi1MSBUNuk9dyXbMD0GCSqGSIb3DQEBCjAwoA0wCwYJ\n"
	"YIZIAWUDBAICoRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwMA8xDTAL\n"
	"BgNVBAMTBENBLTAwIBcNMTcwNzI2MTUyODQ5WhgPOTk5OTEyMzEyMzU5NTlaMA8x\n"
	"DTALBgNVBAMTBENBLTAwggHSMD0GCSqGSIb3DQEBCjAwoA0wCwYJYIZIAWUDBAIC\n"
	"oRowGAYJKoZIhvcNAQEIMAsGCWCGSAFlAwQCAqIDAgEwA4IBjwAwggGKAoIBgQCg\n"
	"V/fvWuWb1hyqlrWXKgz8L/xyi1j6/2Mc0EOxB1tSDilrYB7EjnR+WIsXmLc5qi4u\n"
	"4/30ddw1JJLPSE/IuzbxRBKMOdLAodycx/XzYCfZbCEbeBJ3F0Avoc2rPdwd4Jxh\n"
	"aCI3aw3KSopPgTpihJXqfmeo9bjwjuppaewdIoM007XoV6TGRfSMJL4KZK5gvW5G\n"
	"5hrHF/nPCI3nNFH+ocFn0yHRRmbq4IxIfkHku5AxX4PW18c7bJ7AA37o61YzDXgf\n"
	"jY22rLVSEVeAfQL8HmPY/yLTY5r6stJKWQDRrXGPxtLWwtbkGx6AGAry+o/0eGo9\n"
	"o3X+h3YYL2lX8jhCG75p0ZKIRoRAVRD67WM5HmGcv9m5YbfHwLZ+SSqAfclS2UL9\n"
	"Czrss5lAqEcatoxSC+L3iee6kVDk7OPtv6KLYA1OoAJIgWapWcJpBNgo4PU8Ps+e\n"
	"yN3BPjBZu1O1j8RnIeprLSlP9X4I4i2WPWAWHOi2S20pHdvC6/+qZqSqVJn65QUC\n"
	"AwEAAaNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYAMB0GA1Ud\n"
	"DgQWBBQLl4ZB9pgpBjOTtv1mMuDYFw1VPTA9BgkqhkiG9w0BAQowMKANMAsGCWCG\n"
	"SAFlAwQCAqEaMBgGCSqGSIb3DQEBCDALBglghkgBZQMEAgKiAwIBMAOCAYEAD9JR\n"
	"Wjsn3Vb/eiLqMmy1YZyCBBUePifHFOjK9edxzcef2+eds5fbfZKE9HwebFydbaCl\n"
	"2vfylBRhHiPx35pk2m7dwt07WAjVyLyoF/XWAbJPhzj9nEUxP15uZz0L2YR4PFK7\n"
	"w6PFSgAbnqcNW+Eva95t0sSaClMfOQUZizwJUXJHaF1+NML//oG/zar07QSbIC4V\n"
	"WU00anUDHxikhj7FGndhquV3bAICW+LlFiVASK73wvhcWNjBQ/fIOHH61kl9XOGR\n"
	"4t3/YgC4DnaRX2eFknQqTVKt5GTMRRLPVrSzcPQVMkHkA9q3qMRZ1OcJ4jeIF0Ur\n"
	"PYhowVrqif6tb8jWe635nFBnaXiwtOAGBQfGTbbbT6z0Rm6t+QunKxDfb8CTwZvq\n"
	"6VQGMBrYbu1xHowWVtP9c7WAVtIIbXn4pUWND3tq4kCxyr8EKOO+AxSxPqRKRHUa\n"
	"OInT1o1n73r11Arm1dmlup5cJjJ4SrJoqxGemdNVrLYR6x8//2ZjiS/N93gC\n"
	"-----END CERTIFICATE-----\n"
};

static const char *unknown_critical_extension_on_endcert[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIENDCCApygAwIBAgIMWLRKXgVfpidUGiL3MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwMjI3MTU0ODQ2WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEArLFdJ0t8aIAEwXk0+BaXSCWtCrHyotXZxxOImQk48khIB4zQZjdbsbIrE7BK\n"
	"RW9h8WpUNUTCMRLrIiAL072YbnCyf/aHoaumOGYpHdqo+nD05RH81x67L82v8o+q\n"
	"vw5VeWOfEU+HnAJxazXe3WqvjDvxDLxlvRG7WBECfdmdzdexot9VgLNv1hD3Cam5\n"
	"/FBOGwKLkU+cK57BAUcgRqFLFcXZdq/6Joy99O/WVMkYXfDUExRog6EJuzLsQqLZ\n"
	"symssmps56OdGNeSwACcRUXYQRI2Fp6kWpQ1kynroSDms4q6hz/oIre47UtmYNsu\n"
	"LnUYLNRAbpRuCLs2uKHjXJoQHV5HCGAq9Whtk7vuNDoD28VO9T0CBn1GbXATDmro\n"
	"UldPZSFvsW8MUFQcuRbScxrLekbR94GwD8sPdw1siDmI4M1SshS5IyXctK5rfxF3\n"
	"9qJ8DxwWI3ZSp/N/2HbAa7KA9RsGe548cjfxMbHBv15vZbW490Jrypbl5zTzHwWB\n"
	"W/nRAgMBAAGjgYkwgYYwDwYDzgcIAQH/BAUaqwGs/jAMBgNVHRMBAf8EAjAAMBQG\n"
	"A1UdEQQNMAuCCWxvY2FsaG9zdDAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTc\n"
	"dq+IHuXY6ZiVq2QRqYMjCWC/wzAfBgNVHSMEGDAWgBT705Z3PM2jlhgsbpzfeFfd\n"
	"uTqdMDANBgkqhkiG9w0BAQsFAAOCAYEAiPZuXN7FdUo22v1IKyNmGZ0GOzv9N43X\n"
	"OnJkfDAbMjtO6KxCtqACSFh2cJ4NqUV+GqSo0fNgZIBvXVaL5GOGEvStTs8xw8oH\n"
	"GIJXQcY3o+6S5/u/6OU8CYnsIqQGGjIhuUzIsrdcFrRrX4nY9v8zr1o8B7MbN0pG\n"
	"seh8bLU5ih41OH7wjqx3/nuQLs7aui5sRiS+Ug7lF8N9VeOlDCdCeiWo7jNSbTkV\n"
	"Gt5lcAI6W1paOrNaqwDQ0WeNSM+QLyOXWXR9b4Ck7T4Kcx+5ZSYCKul3msyyA4rV\n"
	"cAqvaKs/M6IqFHaQGSwgrZFIBcBrViuAWZbdlpaCUNV2bBsxf/ro4Fe5Z3hJz4/r\n"
	"4RhJffrzMN1aaINi5rFYgMSSHGSy3O5L1yYMdOa6FiJSzYMmtJDFejNLU+e5vM8V\n"
	"wEBzUaG8UCrOaOumcQcdb2J7sjAoQ+Ghn24/4jmW+A2mFCmAvKfBHRH89BUzEKt5\n"
	"5IHTMGbPWEe483Nyh3Sx1pXcoQDVgD3l\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIID/zCCAmegAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE3MDIyNzE1NDg0NVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEArw8KL4omF9ALedLd\n"
	"qAgnh+mTAQr1XlAy/1DiwA8yCFIPMucSHyMCJlvmRs/gtgemuU4Bn8NsCu9CcXUI\n"
	"s9mMhrmE098HLn/2McNW5wGNVfK3/8+8N0cok01MuWBjCcZ70k0aYEkFVivVdo4z\n"
	"CScmZrQYqqtmiTpsJOft+EU479mDaHK0F2caEZqWdyAmpgQ81eMtI/1kFzRbjYCE\n"
	"Pgati1YrcbS8QLd6NRA6Lk3WxSZpIFQ5gyhvdR3z3onWZM6Mcqj47ZsVK/nvXUSM\n"
	"2I6JACx2AfDWnbO1c2agZlxRvjlcPkrOiIg9KQs13IAAQ/VeB9KI9PrTgLwNhLtw\n"
	"gsc8fgn/y4GL2VWolPuNVG0cIP4b4EnPH3ppRfvGe3lkm1zRUkTDht1QRgYMMel0\n"
	"HSosbt6zXxMKRF+5tsI2ntnXz1xMVMsE8MEgK58BBOTuJevPLLTnkh9WOih6IfDN\n"
	"t99yFt86x7KbRyddivRP60OftilhUaBcKz/vpKpE0xyB/QPDAgMBAAGjZDBiMA8G\n"
	"A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU+9OWdzzN\n"
	"o5YYLG6c33hX3bk6nTAwHwYDVR0jBBgwFoAUJolqIJH3QVRxmq4o/abMSF3P4REw\n"
	"DQYJKoZIhvcNAQELBQADggGBADpJZ6ydM6+kUcoDGgZnycH2+73A7OuIM8nhVnMg\n"
	"9+SsiLrjD9rL3gMhlxgvsCGG3So0AeZTJwAQWW/B0vTIoScX1LQ9YuB1vv2l8kkF\n"
	"Sr/m9TZcP2KiLCcyzkUQ+NvssL5oVXP4/yZDfPasBR5tTLgKaHEhENKLx+1OW9/j\n"
	"SAGtyP1vxnAd/4kPeuS+2L9ehKTQLUhztE9ezguMOEnBmOcn4QujHvOAAJVh2HgU\n"
	"MWcIo4MFhpEYo5CobIDJ2+amRW3ywnIGOQfFQOWUfT4XlgoKvirEqWIOKiqmzEZu\n"
	"sYKbFPhTrFwZmpEwLEw+8B3iz/hW4c7Dp1/oOdrPSr28iQYfJWMyJTm/F67/wzGZ\n"
	"eWYs6Kxc644gtBckJLGjVPHcu4dwpWpKnzK1cwzN8GY16/m/7eA1XsFWThQOR+KH\n"
	"tF34Cs3xbu2DBtbMecpeWAOYR1lQg4IRHrCVrB2hxve+/bXrocs7EDlw4rTfqkz6\n"
	"WqtmyLapzQDNj/NWymF1+X4c7g==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID3jCCAkagAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE3MDIyNzE1NDg0NVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA2Y2wvebbAVPqOsDr\n"
	"1D5yGc7B/TlTvuzwAX6RrR6zwuwr3NkeP172A/S1JMqZgl/MnV+PLaCGQxKgMg3V\n"
	"7pIe9qenytKs6QqGTWt3z3C1HerGT9LQ052JclCStmC1jPMwOUXFiUNunvpMPMMB\n"
	"97KegAnoYhViQ/IS0iDmHbdEWaFRFcvqVXyaxZP9eW3bkEZ+0pF2yaYZE2NqwEgi\n"
	"aSWpG6VStRC86GJCTswnpIEaS3gTGZzqmAw7eHEqz0lhZB6o//Pyl2TvUexwu52X\n"
	"McxWflgCTdbZmjnuyA5LZp8GnsQ3TbOzkP6J+BeZDPjSKvNtgZEz7Bt+NXQGxm4k\n"
	"9/5813WkA29QEyVHZOQUvxTCyr3B3lgxxvZzXQYcmqpWdDZyZr2jxJaizp5BVKRc\n"
	"OxhExiEhOUnwwwwNMj0Wyxz894QFrRqy947+lMWrH/snQ/owZX2IZ918Bm23IpJl\n"
	"bUb5Dnyfr3PIa3XfAkJWRCM5xqKb+xOZJaR+1lEbYYqOuQDDAgMBAAGjQzBBMA8G\n"
	"A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4EFgQUJolqIJH3\n"
	"QVRxmq4o/abMSF3P4REwDQYJKoZIhvcNAQELBQADggGBAKnD4dpow4u9YpnVZ4tL\n"
	"cOrkB/TuSIGbuTG1K3CWvjDNoPwe0AXvhexorVuYMWVtTGz1EY4ugopP7qvMf4Yp\n"
	"lcZNamtTDi4Y6zTrcKHe6ckedETi5TD6d7pNIe5wgn2p8hptfcyPwlpPjx01XqFm\n"
	"Zg7OcoBwhe6UYsM6C0UpUQTqv80aOhAzB5zHHUAoIVSGQBGgTip5QxTq6cE4OX1I\n"
	"eobqAjL/J+1Wnc5tN6ctDaO0HbXxTnPt3FkJlkFmnx46o5n6R3KmyfWs7QSks0PN\n"
	"/V3hQ4uV0PagQ+z76As4AsNHNGIKCiZevqOUWoU46PlFOR3W65i6dw2iRLW/qmCQ\n"
	"Mn8OIzqvcPOwYS4a9vDAX5TR23gPkUixlPnxo5sTPgpTT3PRhJekIoCgL3ncBs8I\n"
	"p4/TbPht4m0L2eCodGkCm1QCbvbhE1JA8ZpSjxjICODHEeD5w0RXT0FWprCMvrO3\n"
	"uExu6AxfNhtLsV8E/HXAGTguetUysJgqqXdCUDXTQbrIHw==\n"
	"-----END CERTIFICATE-----"
};

static const char *unknown_critical_extension_on_intermediate[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEITCCAomgAwIBAgIMWLRKghUUPjZQy+hWMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwMjI3MTU0OTIyWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEA2WRm/jJ2Ul4TynGmCamCsUv2qLvQmFknfg5HEjPGpx/sKnzkkavke3IdltLS\n"
	"/UZVw1ZwoMKBh/5C4tG1Y7mKHdmHmuISyfzB/vHBNx212UZJpOiwuHHhV94Nmdbd\n"
	"X5sD6aFluJOWRAL8PB2vPJF8q+X1PNDxilDEevar/ULNR4+rHMlqRUtbnpqHXVmd\n"
	"eYgUAYRmJatzyqhlGJ3mFZPiOwCxvR4b1RKzFx5zgz1vy+B+0a+h+wRgDZ3iCwMO\n"
	"mtAgYQ7z/jBNzdbkVZVMhCQm8UF4Pt2ITtLSfuZ0lDzntfC504TQABaFooAorBMF\n"
	"y4+OpIyrAXGAVct90Rtkq2F0sVfGtz81xMUw8iVteURalUM/XH4Owz71BJcOKIDO\n"
	"PbWyGTMXzWcdCgEhL3P4QG0HMjyhjyv+BAND7Q0e2KGjfnGmO2tRuR6h6nBUkjiZ\n"
	"UicMEAyCxJAuMMBItJTIoXeNFsttlYIuAMHFV6A8ekjWhhZsg7LhlLH+k5/uz3rB\n"
	"UPGXAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0\n"
	"MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFMBahZT/Z4RDB4CkB5lAxAGSCUcd\n"
	"MB8GA1UdIwQYMBaAFAMeecLjN1JGgju+TsSezQDYARdtMA0GCSqGSIb3DQEBCwUA\n"
	"A4IBgQBV35WByM7cfucUIIaYgAE7RWI1mVz9QGacEzMX0jtJB3pI78KQ+4i4XRHM\n"
	"DqES71RbHUTsrIUUEMvzVcpGSak0cvx+9QC6wkNn8w/SDqkJnF4/72MRGXigHGcI\n"
	"L9/dSj0yKUHMzjo621cVlG8AgmCG8MYInTXY5pvqQ59mvD1A0HLUfJmwoCwcLpx0\n"
	"JIM7/f9CfK2kHVBIvdhV7cHr+pDPHZ5wjVIqF2WE9TnC05caxm/yR5aauGptM8YN\n"
	"nfoOnsqFGK5nmD94tAPzCbPjQQqyRQQALm9/oJPMG0blJPkch/hsJixujM1hcENX\n"
	"ylV5Odw2LYGy2qFi1Ku85F/cJFvsF6b215ziQKlXBdwEa9Nno0jjxa+g8zFktUa1\n"
	"6Y/W06RqQKKVdn6bfhYmOi5PL+faiA9yrGnBy3i/32Glxs19sSUODa9babAFKTlM\n"
	"099jOSh6YDQWOa/z/eQ8ueaL7nfwe9cWBOsjwUUO0BJwx3uNQ70ijf3FBbLe1PKz\n"
	"yQNcZEI=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEEDCCAnigAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE3MDIyNzE1NDkyMloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA4osbQlC5tHJGl+kD\n"
	"Of7n3E3DfyF1g1XK7XyYpbjmguAVuxz9K2qVMVYc+GUWUN7CQeCONFBV7+49pDFm\n"
	"47jw26fQ/x6tj9Omq0pHqTSEmNpsZTh622PqHulThK1R7R1p2T9vnfXUd/AnqWYa\n"
	"ECBp6BSW6WKScL36rqVcKc8I62s4z330rrIEOksWFVSOo52FwSaM2lEH2nj9MGPv\n"
	"P387c+J5fG22g6Ho6k/A9YJZgs6bjq+XXgqBAVxYjY++YgjTwmcFq9tMQqmxEOCL\n"
	"jF0yGCUnV+oz1J5+WTfhkm+yyq9hDOaf6GoCYVMiIpcpnTtZegMTL06Bd4nawODT\n"
	"QCuzg2FOmHzSP6XehGPtHzRTsKDJNjl6Q9xTnoUPOcoECYI1xMHy4lbKH236BY4G\n"
	"wjyFpfiplr8p4ZOdlgat/+d/eR0gxJ+A6ZbdUsrjNrMo3Jec9y8ZHKR5OLsQxe4+\n"
	"UIYlA8iGce0BVGjLlXyE9jE1i8ku0s8QQo523GB8Va8ljSQJAgMBAAGjdTBzMA8G\n"
	"A84HCAEB/wQFGqsBrP4wDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQA\n"
	"MB0GA1UdDgQWBBQDHnnC4zdSRoI7vk7Ens0A2AEXbTAfBgNVHSMEGDAWgBQghKJA\n"
	"xiymxB9WDU44awpl6BIZ8TANBgkqhkiG9w0BAQsFAAOCAYEABOkLX1UW8oDD7TnZ\n"
	"9jtpkkW6CrSBI5aIVcKLNVMrXUGiZ9s0rONFXHLEugIgkjP++zUBnflc4KpwlNnK\n"
	"NXp0KtCDhQkIXCWn7RYZ633v0XOrop5VuYV+qtpd2oNDsbCIrkph3llnPjRbhIC9\n"
	"FdkPW5ZvfnJuW5WOwVBZKe192XJXBHQNPLuA6GbE1ouyaHKrs71W0UAi3ij//tAk\n"
	"okwYcK6Y7fIuVZqsSCK9U1/p40zdLwdoeJz1/3U2lTuZwDlh3f12DTFGBaDYamS8\n"
	"4utiRfZPQ0U9t8J8HDj9o+ShX0VMXw116zwiJqw+B86hslT0jAocrJtfGyANsP7h\n"
	"p1OtDVfgisJVK4JaPUoVsIkl8+TBV3sy3jFdc1jpqRg7127Z3Q8nHSHX1mHdI/Ra\n"
	"M3DUhUuTJpk7/vQgfQ+NsfwdvNHWyb4+zhqJyUuEQS0yQ7SfGznzHQhPYks1RmP3\n"
	"afa4/D152QSDszhHvTrXyPiFVRuAGk9lh0h8nw1mwygoL1OI\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID3jCCAkagAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE3MDIyNzE1NDkyMloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA8QUnCRcC8HlaTt+z\n"
	"+ui60g8eRO/NqaJFZ4Q/GL65v8dZKRqH10aWZavLXhryCGr2tH2GX0by7pkZqpoD\n"
	"FEuARVoiNuXOVntn+HPlsiRyPCpgisN8tS3cCugvzNrVOlPlfvihX9M3oAYFImc7\n"
	"Lpk2K+FW4s8ThJr7YOSg1BmuQiY9vyZ3fgaOBMmMqOke8MEkXc0xoZ4M14ZgbcXA\n"
	"FpnPHLQK8HZHq4K1NVzF23gKhtB8lrQBf5O8tpqzIFjyiCYiGLhZi9v3Rv4bMBUS\n"
	"1bje2PiZ7QIuA5KzTEG45NDiCv6gl1u+uIuWc2VJpRDyMECgKOwhmQs+lnqt3wdq\n"
	"wgtVABAakxPjBRoXv27wNo5nhuQruuESiWQHbly33mMnT9MFkEoeRxJWJpCrgzOY\n"
	"wtT2eXJPzd61HQmr99JygQR/roB4B0AAHOUxeSgsjrH8ptzQSeE7Uk2HSyz8YG3a\n"
	"NfBvYMCCXZLW422ZG4G1wEjan9to+pDj/a4O51ClCT5B7of7AgMBAAGjQzBBMA8G\n"
	"A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4EFgQUIISiQMYs\n"
	"psQfVg1OOGsKZegSGfEwDQYJKoZIhvcNAQELBQADggGBAEO9TkE/bgMkFm37usHr\n"
	"d03a67vZahubHweidEDsZwWf9h8PiaAIUth3r0TLR5HFGj5zh73hOWCggMmxJv+V\n"
	"8UL1oq2jYAXeDkA7MbjADIP3Yl9C1fS+Bq2Dgqht3pGWq5Uej0wClu75bHEKLZWl\n"
	"miKvuFscponO9L68ARPBsZ2woQ9F0+2buV7PizylIZv8BnYAxoHAEHcq2RIqj06L\n"
	"si/plPx6UzbfWrfW6H9OqjB5TDJ8hSwLEzUg+RFYH34RZ1D2Thl8GKB62lIpmP9K\n"
	"7f2wSE9X6gWivyObhmQyAHiL9rQPQsXrpp7paqeknxEtPITxugOunYLPfuqddkso\n"
	"33BGnJ6qVjJ6sTBziQmPcwNInm2/91Iw0BSZP0/mCBot+9rSHD38i/sdHbJWh9WT\n"
	"mOVu3yT5v+39mC8Yd0ykgpO1/47jdPr5FyVz7yAfwb3/MHgjx20UQV0fn2R37LAk\n"
	"AsKMxlHYGz6LYjxK06ZOT3GbBny23aGoooAPqBZenFWXgA==\n"
	"-----END CERTIFICATE-----\n"
};

static const char *unknown_critical_extension_on_root[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEITCCAomgAwIBAgIMWLRKjwNqHC0gaxzuMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTcwMjI3MTU0OTM1WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAqaopWnZqPqFMWM4p62PIxlBCidmiUgefxCphPqq3HX0C5IaHB+xC1a/p51h2\n"
	"C6hTzChKpVWCJ2obxt4/EqgJRVA5ujYA5F7Cw65oy3GecHgMzprFiTksRIA/n5SM\n"
	"whYQBFy939T2UDCyK5bfQGzDTDXoy4ZKoano5cmD8uuVODLlyyj2cv/XYsDGwAZb\n"
	"h4+UGJzld1YYzrKaKb/9kCczsDLR51F40ZbLtWtQ9dvKXcIfhKEaFuWI8MjmJYzJ\n"
	"WyKhMVEA3Eat+GnrKTWka5nFPqFha3XbVGb1ASJjdjvUFl6v8RlJ+wqsIni1WaIp\n"
	"TZY6BCQmEc83PtvahjImoquFt4cHofrbQPK4A1Kaiw5SJMWJvmaZpiAlDnb8Wl88\n"
	"TZBtkg1s/XVdD5GbY7r5FZRh52/9e4L43ZThZAGW91EYoNrVr9713OBfwiDpyGRw\n"
	"bba5jmHtpzKU4xYANizc1A+ioaCnMbJM8iq68EUkcP0F/enH+TMRxQRMUQEKd4gx\n"
	"mYUHAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0\n"
	"MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFAxBxXFjDnXrtfECorTP1cf/us6s\n"
	"MB8GA1UdIwQYMBaAFGGzwr5dJM/noimXvK/tlXJzNCQZMA0GCSqGSIb3DQEBCwUA\n"
	"A4IBgQCIij2aGijFZxVV2kdzSbEWWVV9hAFHF0azEHTQwmSQ0nO/zVWJV7pTjMMi\n"
	"qC2zAyfR/zc8BtxGBF1bh4U3Ke5Jy9cr8CYuS2/mvMERopwiQkUtVTABpvKebGqE\n"
	"fkbbDs2Qu87u0RxPVUZqyiPNDtVq6+MM70/QP4FRM5uwOp3nSMDiaajlX8jfnZe4\n"
	"A1Wa8yduEygiEAbiu0xn2rwjbfW8430Ep4e0lEIBZfWarL61vQaBBv4UmOcIWfaF\n"
	"CVOhJGwuf7DYThUrkcs2mLO84ip7O5HLBpvrqDaymHEZAnidrfoayyRjJaQLvg7y\n"
	"4s65ivmTyRpQnXLw/tAbWE6lwN1xKp3Z5ahcCN9wwMmv0kcSDi33rwAJl4l6h/Le\n"
	"K5c/qyrGxnQ4AL2s98WUL7NaZyv9kuFsQ90ECjVfwhfUEmFqnWPPusKo+LvK4Un5\n"
	"6+uniip3wX/8GCDV+cLQywuDIdo755ZKMaf3UjSfsknWJmpJYl5u0K18uLAzIod0\n"
	"PVow3ek=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIID/zCCAmegAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE3MDIyNzE1NDkzNFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAxIF0pvly30S//JQK\n"
	"HGVka0CNqz2KC9oDoy90YBpIxlwdO2qiQyBlEytGXU4omVFGZZdpAwQ0XaMaEVDK\n"
	"pmdhYPJAcdCPUVkae/qPNS3FWEdSnFlu2qWhiFLgGOUNYn05zA68otm5ahFeD8oB\n"
	"lVv8I/NEU7rQmEHZy4Bg2xTSl/hVVgqbzPtLdAiOzAutWWiQwq4zoVYJVCu6Ut8P\n"
	"hzpSYKAO8W81/tAiyfcZeoARA/QF8NGcSuWBto1hCb43GkZGFp5cKxSJklX06D8g\n"
	"HkUFWkIavOaetywFv1dnUnVKJ2QEyh+40iwnnw2+HpBO3eVo/qArayBWNb/baNm/\n"
	"kYZyIqzFZs2Mf38Uu4MGIJMB9RkcMUuw3o3u1HNMeFIqPeTpvQ8r888uxQTgZcfw\n"
	"l1tGijdnNBNzdvIMTTAOZyLpdDxx/WoGA8lz1Ks3auZ+d0dzO/2q2b0fiYBAY58h\n"
	"Ou1uRjmFX4YpVGcC9+2zmJGGMlkEu/iiJT8dmZc5ydPujY9BAgMBAAGjZDBiMA8G\n"
	"A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUYbPCvl0k\n"
	"z+eiKZe8r+2VcnM0JBkwHwYDVR0jBBgwFoAUjdMy0N5jaoCQ6Ucqd269BlwiH/Mw\n"
	"DQYJKoZIhvcNAQELBQADggGBABUXMEKzD93AwIJZ0R2n2gJzlc7lnKF8BP70ZP5h\n"
	"JTYz+tHymc+Nw6eW7yQ5o9tnJue3Z+ueOzN/+UP7dWH7NRnaJ/BOJuKXcwGrH6tD\n"
	"prP2+HR2tPIuEGmOoe9bnFAiX7evTC3wt+AeIGgYSZgPwR7gF69Pudk8ISrMg0oh\n"
	"Q4pM9YUj5cIWiBBD20vQj6wJCvzi5tyJjEhjT6fJVlcKFZBveOG5RaPB91GiuJ6s\n"
	"JW6z1Q2cCMtvsRZnj7jFDH/m4ZlmK/h9oD1HxCP1rcEmHykOh4RnyyrlT+Vuzw+u\n"
	"jtNaz0Fr9y20a39ENrmMI719/vuZxx2eExko/ta0EbirJw2u+Ygjdn6c6Bc+BR/C\n"
	"IisDCHVO1U+nk8zLo0Rd1JInJH+3jYzn9OAClqBoKxp4bSIo5z419t2jZTAWUNcQ\n"
	"Rj+7s8XN/z7pfNfecZJ/n9KTfqbowqfZaMR/UdmVQZTmN3Xk9CnQsGpotbNDqQfH\n"
	"jz5Y36//zejV2aJeOMWZeAw/QA==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID7zCCAlegAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE3MDIyNzE1NDkzNFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEArr1s4R3Fq1St6vdT\n"
	"ZLTWWZ40bG72Ue/Z94c+m0dujKk+SnXEOe8sEzZLzizaoBiA8IDc1sFpLHDruWsD\n"
	"jLACBOPDahfIGIfeJ1R8L23HEgbg7RXdJ8Bx1YsfaASr8L7AihDBM6BTPfGipf6u\n"
	"Ul2oI0vyo8WxW+0BdJ/qTMJwT/vFqn1XBisuvwa23SVAe0UU66kYXmSxUDZv3YWv\n"
	"P3vqVBGdbussChIYn5xyfCaaLgC3VQNkhmswSfyFQBJ7VO/nHI1a35E705WSKGys\n"
	"IZdg3mA08jLOReCsgAO4vnhubA3jk9/LFFIbJulNQ0j3jcqOyfCtsXEmvgbq2b6D\n"
	"ksBg/x1riV/FAW/nRwW/vqaKDtDTx6XcjFnMjfvBzkON8F+M76qTl4lVbv9fUgEE\n"
	"hBppfHlvu3sQ7ysVYgTDdCM0neTZ0Gm7dOd/9qnrA7ZzQBLRBQD7+en23jyuX/PT\n"
	"TEL2pO3kGo9H9zgTC8TcGyNxhJ9APIu531xkHJyse5IkORYJAgMBAAGjVDBSMA8G\n"
	"A84HCAEB/wQFGqsBrP4wDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwYA\n"
	"MB0GA1UdDgQWBBSN0zLQ3mNqgJDpRyp3br0GXCIf8zANBgkqhkiG9w0BAQsFAAOC\n"
	"AYEAjPW9Y5HZpXnSpjKPryxWNtQSZVRh1b/dGgKOOqjvlXajU/FbMjXTnpm3wdSO\n"
	"PCXGyUkBonuoWlbxcRLsShH8O8LUH50drYnTYiNYqYq3e23u2LrQHbB1YlQnRvdR\n"
	"JP2AOl/d8mP9c3GBhu0dxzTiH5zyiCMxOWChdPbpKCTNGD2aZIUoXsUwLv2ICsue\n"
	"qQIr4f+pBztpNAJdwfMFtqdB1Aeg0UW3+aCjD4XEjISYd9VfXnPg0OaBqddy/MA3\n"
	"Itef4+O7XNQi7w6FWmuZWdEJC2juygybA0jv6fvFKV5HUxxsdMaUwy/Ibt/MOhUd\n"
	"e4857GRICFKq8Q8FUbDiuw868aTkKSVqXoZFoTLQsXn53GbSGvNqwpqQXK9JMPCn\n"
	"c6XIbeF9zgGWbswgdNhs3u8J9tkIxd03cIqaCjzmsl9TByjg5hGEw4+hXbt9PDFa\n"
	"M5avPfGI7g0CdoSKSzAishaYtBn+HCRHyZTpxATrv7/fo0fTSs32EIHmro0Xk5b9\n"
	"qFGX\n"
	"-----END CERTIFICATE-----\n"
};

/* the key purpose in the intermediate certificate is not the expected one */
static const char *kp_fail1[] = {
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDOjCCAiKgAwIBAgIMVB/VrzLxJphTIbssMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
    "BgNVBAMTBENBLTEwIhgPMjAxNDA5MjIwNzU0MjNaGA85OTk5MTIzMTIzNTk1OVow\n"
    "EzERMA8GA1UEAxMIc2VydmVyLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
    "AoIBAQCjtW076msqq69wke2Nc8NMD6UpxVQ1oteCh91lhIgGS+KBunt+C1Hfnipr\n"
    "iTEC0A7/DWGynWZBcK8LHbVyG32fP45S7BcR1SimbE0HD1aWFSboQegghrF+NISG\n"
    "HJgUTvTvUKn8pEUxowHfU3eGM3er4QZ7hyerijOb8/W2PFqkDjEZse6uPzKOoawL\n"
    "Trm88cCuzSwKuE3Fftvc6tfzorXVKiFIGlbNBw+bpCVUMpjx4w5Ug4BbJpD/e7Hf\n"
    "U+BCkjfKzWMDJ54rUQLbYUTh4QknHUoZ0W+RMKpeEM4esHt7HdJtZYKh/Lzi5GjB\n"
    "VRAfAA+5khCRwC2uJv1cpzqVjDOHAgMBAAGjgY0wgYowDAYDVR0TAQH/BAIwADAU\n"
    "BgNVHREEDTALgglsb2NhbGhvc3QwEwYDVR0lBAwwCgYIKwYBBQUHAwEwDwYDVR0P\n"
    "AQH/BAUDAwegADAdBgNVHQ4EFgQU7Vhq5oDQ/UgzdS+yRZGbBuD5GxcwHwYDVR0j\n"
    "BBgwFoAUWXJAO7Nmy8yMdi66LZQGgaO+plkwDQYJKoZIhvcNAQELBQADggEBABSD\n"
    "xzc5ZeUpD+6dvIwptMYLLL5qMI5/GmhSHRuI7pPkf7JMOrfOfPaDRKJY2HIoHRB+\n"
    "68iUeE+C3I17lpuXcTGU4OvLrqBQ19orSfWvaQMl8yOR0Uzjn98jc4wuA2nMOnBu\n"
    "nV8Yx/rJvBK/uJYBjDaMdKtddBvdUoaOco8tflOLDBz7aBKTO31qynKGWgtPDasG\n"
    "DsshOcDgstMwhcBXbIrliDvDhBeWZnjG6E+9yf1ppUoVMp8UU6H12lA3en1GzI3v\n"
    "E9NHhkJkFk6uUIp09sWLfw/MGiU/rIb9Kj7qjOoE0RAirOJdqSnb+XRwtD5U8u2Y\n"
    "JcxB7MEBdJsNPbxRVtc=\n"
    "-----END CERTIFICATE-----\n",
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDFjCCAf6gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTQwOTIyMDc1NDIzWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCyxSDH68ZW2CHV\n"
    "6aozE8LMXnOZXDOuli49867VsS7SLr49jxL6jzLx/oJN3JfITcn+ohYNifnA7TBa\n"
    "VeDSXMTG9tJrU8FlOTCk2Vsc939YiJ9tKNX4rPD+OeSbWgxOnVlkI5zZYnq/+i1E\n"
    "UvahCHyP93GXl0zR9hsSptJPD27mX653clPqPoTNBA+qSDTb/GK0Yvgfioaqk0PO\n"
    "q8pcKjZ2N+qg7st+y5Rj/92g9E6vpdHt9DBfL1THkeaJ0VKfvsea8fj1y1AwwWvT\n"
    "9TWIYCLCiYYtMv2Oqsf3lMMU+Zs5DO1FqHglXrnwjGMN6tf57OYol72Pih3enejM\n"
    "Syb2U2z3AgMBAAGjeTB3MA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYB\n"
    "BQUHAwkwDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUWXJAO7Nmy8yMdi66LZQG\n"
    "gaO+plkwHwYDVR0jBBgwFoAU8DKVyOiO2uYcwEgw+8kRGgRXCPYwDQYJKoZIhvcN\n"
    "AQELBQADggEBAHudcSXdtTdRoMkxRZUxPu3RaJFaFJr+y0SU+3gBBjbMuYIUD5b5\n"
    "neTl5NDwtgPQS41ldf7oMVWaFGdV6OZzkfPqWgNCsDPy7xSGHXLQ11SSNsw+J9mW\n"
    "zVdiM+BQOWvd4WJ0FKJ+pO1WVyTiIaqnSmETzgviX93YueV53h7Z02sGifp4X2Xh\n"
    "aauF8xrG62ELNBC5kYFkAWrnNtSMuykbpfZT+l4nFI9ytxHLRFMaH2jpbfdL0pmQ\n"
    "oktFJ1fln6N0S71doSUTvfy2iPcvVqjIKW6YD/Pyrr6ThEbVSGzY4FogzUMQf3fN\n"
    "J2Q6Eb+S3ZB28Sm377QU5XKQ1ANpOa4ozHk=\n"
    "-----END CERTIFICATE-----\n",
    NULL,
    "-----BEGIN CERTIFICATE-----\n"
    "MIIC9TCCAd2gAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTQwOTIyMDc1NDIzWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCYp4Ek815xc7wy\n"
    "5AEJ2AzJSyTnVWR4HTsaWBOVGCpE7Jrz3hHPUzE/9gRM1Oq5ROWJ7DUVFoD9pP8A\n"
    "if2hRYwSqCNaMswMj3ReDLm2iKROrFhYR9Rr6kcucmRDcrN1SqmmIvZT80uxPXhT\n"
    "TRVWYHc9kjGvgC3U16O+265d44mgxGw2UMwMirh5X5u6JFrfTShBXj08UJoNsj+1\n"
    "6Tp/x6+vO8iaGWcRPetAWzwMGNSH4CanWuHZBdL7jLV/OamfvKhBHhEGF/JN1KDn\n"
    "MSLrQJib8T2WVfVdmLWM6FbDiw2i1KeleCij354MNuI+Azr8aTJma5cI1BWU5Ndq\n"
    "tbHShdCxAgMBAAGjWDBWMA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYB\n"
    "BQUHAwEwDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4EFgQU8DKVyOiO2uYcwEgw+8kR\n"
    "GgRXCPYwDQYJKoZIhvcNAQELBQADggEBAD4RUNg3EQ5yJ1LfXWKFTXIhSg6B92Y8\n"
    "QTSq+1RWEDP51J7i60A0yTDZi0XZUrfNIv+0gzw8F9l/QEWM+IXXnHUhhluZW/xf\n"
    "PiQPtz3Tv5uMIxs24vMIZYeMzXJ+N1rY2JbYsEWhkfSsJA8LD08gw0azlPKx/wXy\n"
    "aBlknDfeCcrpXhnJdpMxj/N7nLPTQuPe7/VmZjYc8VkopjKMrZS+3KcEoXbr5Zjr\n"
    "zm2mY/IHu6AZgtsWMhakONbH6I9rsZt5VlTPO72VmkjYtHhFfvjs2fPH1Gdu52y7\n"
    "P4gcTXWeMficO8uzeHv17J0+qBBYxRe5Fkri1i1JRjJcBqVaK3JPUzQ=\n"
    "-----END CERTIFICATE-----\n"};

/* the key purpose in the anchor certificate is not the expected one */
static const char *kp_fail2[] = {
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDIzCCAgugAwIBAgIMVB/V7wN6fXUuqVU3MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
    "BgNVBAMTBENBLTEwIhgPMjAxNDA5MjIwNzU1MjdaGA85OTk5MTIzMTIzNTk1OVow\n"
    "EzERMA8GA1UEAxMIc2VydmVyLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
    "AoIBAQDlUn8aFZ8w0fXI0ewO8+UQ0KKw1hUbH9A7DLJKj/bEyFepKh7JKnBTugs8\n"
    "LEE0BKxMjC2smx5sbIKPGRljva6qST6j52HjWQfVMjL4u3M8eH9y+t/ltatMUZ6a\n"
    "GFpchgNNp66/PW5F2aar73H+KpIBT+Lz0mfpjY48LS+c/ZaLfufbbJ49eNPIZPgk\n"
    "nsrX+41YCC3axlUCXeBxdAkCaE3ff5G/pWBPKg+Mx7iS4bjvOAPgjX8cM/ZqLfP5\n"
    "5o+AQqocXw1/uouvpO7rsww+0PgVQnZxoVX2QK3i8l2kIoIBLNaSLTx617tlKZgL\n"
    "6KW+6B4BJePzNN5UsOgWMo4ZIq9pAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYD\n"
    "VR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFJBE\n"
    "Ci49lpyQRB7rCapUONuivibPMB8GA1UdIwQYMBaAFOjA+VXcj2Ujv5sKm7aczyNr\n"
    "bTvOMA0GCSqGSIb3DQEBCwUAA4IBAQAUnlwhZzSUkD4IZ3g4HgfGD0Tlf7ZMZbUw\n"
    "wtdM9sGzYS9n4r2Wn1pVHEzlzqyI1UCnZq2cqYcDfLUncIBYTOveGyrkzq791Mnx\n"
    "1HbmcsjQbEWr/ywCEX55ZWGiDrLkK85TRS+BYNVA4sZnAzvzKwwKOMOGfiKcj7rq\n"
    "XBWpUSbRua29uQOk+P1US10bxiD777OZtp+woJlAUPaB3U4XZsbCCbl/ln2BgpoP\n"
    "61Qn3BdnfSrl+BsjlpDBUItnyVICCRRH109XpHmvXC0SnceGnl0S0rZhFXMZ5uhK\n"
    "icfV5FLumEOPw+pmFIY9z1B/e/EQfbWuhycQT2J5R1RfrwMbRI8D\n"
    "-----END CERTIFICATE-----\n",
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDFjCCAf6gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTQwOTIyMDc1NTI2WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvp14uRd38IoFu\n"
    "CW+O/T5k4eLDbll8hZkoJkV6fm7oGQ0LcZRvhvQaHRxO50QzzNYoLMpDVw9cMMJr\n"
    "cyy0C7jehOOLX4JVTWPyoynOycHTmqs9BZmjx5H/cFZRxYg9MQFmIDVj7jjaLf6u\n"
    "vqDR02ab+tHesT9Zc/r/0THAqPoFio3rVXNyQpMQbsywxFRqhG0X0JkQ63xBKJQ/\n"
    "ZvlfNyLnM6D1cG2Kjq1hbjyTNdofGUHY+CGuuoEAqIRnKR80rpUw5cdxy2bAjBgN\n"
    "ubCD7QQlxMUmARjwAgM6lhWB+EXhbcMqZ+dmL0elmST8E8Y5sY/UkEorzigIHjr2\n"
    "Tyl0/LSzAgMBAAGjeTB3MA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYB\n"
    "BQUHAwkwDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU6MD5VdyPZSO/mwqbtpzP\n"
    "I2ttO84wHwYDVR0jBBgwFoAUEjFgu7hCgPDa1wiPd0d1kW33LNIwDQYJKoZIhvcN\n"
    "AQELBQADggEBALB1rjTwAzC0ZFSTyGg6wYo0S/zo+RKFbezHQf+TVQ6mUjbAs6Zs\n"
    "KdTm5rvWXd669RqN9pVWZhlP4gBHAe/0koIjZQ92LOkqosEGsfWbzKfy0Ey3+MWs\n"
    "2d5Qwunzm7D/PnsrYCtZoNIIr0KiAMjyfW3cBB/n1vcaZDAnHnkoUWLUfsgoBFxq\n"
    "yGHJU966soKKl6yMAz8+pn8SDkXFjGXNCJSEcLFq2mwSVTJO2bLDKsnknAdkJr2A\n"
    "1n3vudMERHj4IjXo798Qj/qGozYECDgKLBzBN6+8HUA2r+qelNl04EI+neFaQ5bb\n"
    "nU6OfwULZEc+I9iucS+6I1hXmqqI/jcBIRM=\n"
    "-----END CERTIFICATE-----\n",
    NULL,
    "-----BEGIN CERTIFICATE-----\n"
    "MIIC9TCCAd2gAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTQwOTIyMDc1NTI2WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDW9zXIvd+4Qf7Q\n"
    "dzGbDAR+tc8rOac7FE+YJkWTx+fcIjtaiXaPRA5r7xCUawBAPaH4DvRqmqxwG+Js\n"
    "CD3YMERMh184XHkPWX/s8P14lwmzm/k65GF7PLTcGEwQN67PgRkvV9l9iXq2Krgk\n"
    "syfAd+7mTqkx63PPJgVqIa8Sk9Ljdp/GBlvAaFRpm8fvsKmRwDO+AjLTLw7Ou4kN\n"
    "XQwPejnoreUb19dK5naA5ODrliv7mPc082g9o8NGULMcidGndBry5D/hYSY2zvUJ\n"
    "lxy8M0LHQxdx1d3UIAY5i+sT5OUQBRvoXETZup7Ve6aoKfBc63PWiNAe7wuyGpWR\n"
    "4dLskuZjAgMBAAGjWDBWMA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYB\n"
    "BQUHAwEwDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4EFgQUEjFgu7hCgPDa1wiPd0d1\n"
    "kW33LNIwDQYJKoZIhvcNAQELBQADggEBAFePqEhyTZeZhm+1l10cre94O/0awOTk\n"
    "mDLhJJY2AI9zqAgnaWXg7YAf7nz3Z1Jez02fu7F8c9r7E010YE1W+2kECuM4uN6r\n"
    "9X7VnUW9VRXgiLFIi7mLFtqh+VnwR3xvnKYRFM12pXDFoWhyYfFWVUeNnXdKSN6E\n"
    "4sMPBoyuSB47yit4BUkEanbnGbZG74G7ldRg9HXJqmkOJl+1HALpAstBE/MCM7TF\n"
    "lXgkj8eCaQwOKcK6bl+BM7dExwmYbOY/ILsDHx3/AB/eT7K8kKwE+pAzmHPobX4A\n"
    "np3FeZ2muHFtNx32NuatbIk1VI2pbskc0iQLwBl06SLDR/5Hq848MDM=\n"
    "-----END CERTIFICATE-----\n"};

/* the key purposes in all certificates match */
static const char *kp_ok[] = {
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDIzCCAgugAwIBAgIMVB/WCA0b8cgGIwgHMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
    "BgNVBAMTBENBLTEwIhgPMjAxNDA5MjIwNzU1NTJaGA85OTk5MTIzMTIzNTk1OVow\n"
    "EzERMA8GA1UEAxMIc2VydmVyLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
    "AoIBAQDACAXnEalw36OPk35Kv8Goj7xRaeaAz4W4w1vUZ6IJCiNIzIknvlnU5vi8\n"
    "MigzloHYXmhHRyfHGggoPZyGtDNiKIeZLRE81CVS9UzM4Uu5naKmU4zBMmmqEvvh\n"
    "/IHHDXd+Ky89WXcI4rTFjzGzvViQinN+1E7BViFplOBvGHB0qa/v2saqyoD9HRJL\n"
    "RKYbSGG85T1u8B7M7mdnBuQyk12r6lsxsNfScnkNH04jUtgL1i+Susd7zo1waqi+\n"
    "lXvvW5P8gB9cfa32cXsBNGReOZoVk7G60JjLDmB+fWetVn7o42wG0gZ6TsSTsrqn\n"
    "WLNMOxw3W8WLapR4vwU2BzjTZfRjAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYD\n"
    "VR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFDL9\n"
    "2DAwMqU6DnAE8/MOefs0AFO8MB8GA1UdIwQYMBaAFO47x5DoYJnwbtfPj/PIxKmH\n"
    "JZQqMA0GCSqGSIb3DQEBCwUAA4IBAQBv3kR0R0VIu4mJ7oERxCPosJafnJPOGRMz\n"
    "7F4zk3j7kwaP6OGHRJwcvxNLRAr+YIXheFDEJTwkEtA5pbir9wIG0fb+FT/o4ggs\n"
    "4r2DqEZo9rZNatPMTKswbA3kmeKxRUe/AVsqeSz5Na5HNrHAGFZplUpyGMHT09f1\n"
    "rHfOTlsq8dorGtE14UyJc9CY1GhHZVNFSPgyaKxSTVBr4qsD1WVKPcARWjL9Qp7E\n"
    "0Gnh/O+eLdC4V7izmAfaS7kwyYiyJoWSvce4hMweEfWytOaBJMDg4pFDLabhfJpo\n"
    "IZXCSc3/qHmf03lU+ntppPOXdl5niZ0YnwRmf8uYJb13EAepaiyQ\n"
    "-----END CERTIFICATE-----\n",
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDFjCCAf6gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTQwOTIyMDc1NTUyWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCp39LwuWEO/TO7\n"
    "0X9G8P3+EoWL8TDyGmAGST3+qM9pSunLkIeYnY9RtiT8/W5Tt1G7GD7RaU1J+IoL\n"
    "UsEOpT/Apx2Pl3AcXLQlhrtSSZj+yBBWSvY2C+Affhr34sIEYXlE85DseSvotJb/\n"
    "ebHoOFPziji95gR+l9L7aQe7RKaebTfNGbSuTzn1e1YYmnJqtNmfLcrgALblCYBm\n"
    "XMlJRy5NCiWk1D/BNIGsp7Qqtx5yE/h+92bY7js7s/Vzmbhs3LmFYoKMvBOWqbrc\n"
    "OQCFZxLX4jaNg64OEVOX+OqbAy+bssoMWiXAZfBz3018Xnf88lrsYVL1ZH6QDQL7\n"
    "sEIZ/IBPAgMBAAGjeTB3MA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYB\n"
    "BQUHAwkwDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU7jvHkOhgmfBu18+P88jE\n"
    "qYcllCowHwYDVR0jBBgwFoAU1fwQDT/4ciYUpHQqev0LK+2shjswDQYJKoZIhvcN\n"
    "AQELBQADggEBABjGLC1H8oL/TjwBBbrFSwRtmbY1ElLO6lZvniWnfsOk7R/DdoBb\n"
    "rzNcdUOcXoH6RbuZ4XA2ROqLb9RAor/V+A4CWHLLfToHKTYFE4vH7iN99gk+OEy6\n"
    "G7CR2jYNHVikrX3eEUqdMby9+mY1K9GAz8+MojBUTPllc7Gzp0TsLzWXxvhSR84Z\n"
    "zVJAS7bE605pKeABTD3b4aSskn0yt4UYEgVfw2hOnXDaZMfQLgp46z2PBuKXo+Xq\n"
    "JpqbTJMuWA96J4A51RloNWBESzYhBCppeGIlGOhryqMDreVJ+MxZiuqgcbvm/7qc\n"
    "QTLpHAc7dU1/X8/QgqKZIVnpWbmGRNsnTuU=\n"
    "-----END CERTIFICATE-----\n",
    NULL,
    "-----BEGIN CERTIFICATE-----\n"
    "MIIC9TCCAd2gAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTQwOTIyMDc1NTUyWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCoyMH9n/NaDPOx\n"
    "+LFRDvrIPyCpc1cCfq/CLjdcUYll6IYCDn/p2XTSFFCzeLY37yyAX7VuHcbfAQWD\n"
    "0Ax8IL5EUBdQ0BVEkw5ck5pOsqbnLU991zAvUpN1C+u7ogB92atTaeDR9TUE4bMX\n"
    "EW+k6us9WQQe9A/w5rnOr9baR+lvndQW6Mun+7bhhX0KdezUosTd6xfW9tOXSOso\n"
    "jkk9wW+PKdRCmfmqENNLMAIkQ6RES5LTO9KFGlbaCfJjxPVmT7V53nRsY4j+v1rT\n"
    "nNMK6JshtbjQCaVM3nvXXQCJ0nRtUGUcS2JeRpc1C6h+TsHPDo7kNPKGYLDi5Zps\n"
    "9WOArkLVAgMBAAGjWDBWMA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYB\n"
    "BQUHAwkwDwYDVR0PAQH/BAUDAwcGADAdBgNVHQ4EFgQU1fwQDT/4ciYUpHQqev0L\n"
    "K+2shjswDQYJKoZIhvcNAQELBQADggEBABoD6Zl+A3DU4KTa4n+002J4ddl85O4+\n"
    "Qj1NOFfGZ8qP8S5SuzlUIUzRMvIacCSXOwcBFer4UtKe5x/O5i9F1G4eKt09vGGF\n"
    "+XZNjhBOOqqEVUEcwCMecdU5aDaFWx3g7ixrzOlA17Ida/j/QtJZVyhJJBm9wxfW\n"
    "peFcl/aR3/PPn3eULbTTMK+mUe96PwW2FrEA7ecNBxhCkcIvt42IWqkqTD/1Mg6B\n"
    "BukSgD3VAQumnglSuu+G+F+KJ0zFPdmu6IaudpQ92hM6NeK1vJiiP1Mv0ALsk04C\n"
    "Byazcl/VWffXsBIE8OI3k25rFXGn5IAVxzLNGpRFhWfKXbREXICC868=\n"
    "-----END CERTIFICATE-----\n"};

/* This is the same chain as modified1 but with no modification */
static const char *modified2[] = {
    "-----BEGIN CERTIFICATE-----\n"
    "MIIFXjCCBEagAwIBAgIQHYWDpKNVUzEFx4Pq8yjxbTANBgkqhkiG9w0BAQUFADCB\n"
    "tTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
    "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
    "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMm\n"
    "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwHhcNMTQwMjI3\n"
    "MDAwMDAwWhcNMTUwMjI4MjM1OTU5WjBnMQswCQYDVQQGEwJVUzETMBEGA1UECBMK\n"
    "V2FzaGluZ3RvbjEQMA4GA1UEBxQHU2VhdHRsZTEYMBYGA1UEChQPQW1hem9uLmNv\n"
    "bSBJbmMuMRcwFQYDVQQDFA53d3cuYW1hem9uLmNvbTCCASIwDQYJKoZIhvcNAQEB\n"
    "BQADggEPADCCAQoCggEBAJdfieOPrf4Arf1Iled/ii97407ZnjpaB5xxm49Q4Pz3\n"
    "+5xmD0LYre7Cjn1A7W3ZlHki5zFVZpW9Jb/3PfSEDY5slyjkLD2jdl2gVefSthSZ\n"
    "tYxb5eYv79tIEN0U9AZ8/VaGwUokl8n1MitcECxNLMe4LqoVmS29nXITTTzX5t3I\n"
    "4dUeMBDNI+xgVpJSpxwzA+/L+wxoj5Sb4YJ/Y+iUknCkjX6PpaZMRWBEE0dqvG02\n"
    "qlxXesAV0nmKYvjbtqAyoW6vgjP85h6gJEESIqTTZy1HOgFpO8XT05CpGDcjhP1s\n"
    "TvXF7Vx1aj+xDidGLLW188G35oFKIhEyHQV2V7vzRAUCAwEAAaOCAbUwggGxMFAG\n"
    "A1UdEQRJMEeCEXVlZGF0YS5hbWF6b24uY29tggphbWF6b24uY29tgghhbXpuLmNv\n"
    "bYIMd3d3LmFtem4uY29tgg53d3cuYW1hem9uLmNvbTAJBgNVHRMEAjAAMA4GA1Ud\n"
    "DwEB/wQEAwIFoDAdBgNVHSUEFjAUBggrBgEFBQcDAQYIKwYBBQUHAwIwQwYDVR0g\n"
    "BDwwOjA4BgpghkgBhvhFAQc2MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZl\n"
    "cmlzaWduLmNvbS9jcHMwHwYDVR0jBBgwFoAUDURcFlNEwYJ+HSCrJfQBY9i+eaUw\n"
    "RQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL1NWUlNlY3VyZS1HMy1jcmwudmVyaXNp\n"
    "Z24uY29tL1NWUlNlY3VyZUczLmNybDB2BggrBgEFBQcBAQRqMGgwJAYIKwYBBQUH\n"
    "MAGGGGh0dHA6Ly9vY3NwLnZlcmlzaWduLmNvbTBABggrBgEFBQcwAoY0aHR0cDov\n"
    "L1NWUlNlY3VyZS1HMy1haWEudmVyaXNpZ24uY29tL1NWUlNlY3VyZUczLmNlcjAN\n"
    "BgkqhkiG9w0BAQUFAAOCAQEAOeZfjkI0yR/nutCMHp5/uB/evkB8qIYxh1KKbhPB\n"
    "TmpykmJxiLKrBBcBBu9kW5lMbNPSNclE4sCyN0dxCJHwPm7ubNUxsmDSTPTJOx0M\n"
    "zl0WZVaZ7eX3nw1kj8jSoK0f5n87RzKK85MwBFsEn73Z2pDvxTcd72BE0T1UJLcU\n"
    "2A5uHAJyvm2QpOWBIRKlJHIHCcu5xjj5yLnGO9id0cjOjEgj9f1Mo4hzawL5vQfk\n"
    "o/xFxAsA70bk2trv54kgLvhmAW+B6OpN3Z/xB4mWNOw3G8bg/u+pCLvd8KRO5V8K\n"
    "TlgO1NTaOgYC6OAF3USNMhuNZh6h5tWA3mA8rFr8ZsayhA==\n"
    "-----END CERTIFICATE-----\n",
    "-----BEGIN CERTIFICATE-----\n"
    "MIIF7DCCBNSgAwIBAgIQbsx6pacDIAm4zrz06VLUkTANBgkqhkiG9w0BAQUFADCB\n"
    "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
    "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
    "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
    "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
    "aG9yaXR5IC0gRzUwHhcNMTAwMjA4MDAwMDAwWhcNMjAwMjA3MjM1OTU5WjCBtTEL\n"
    "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
    "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
    "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMmVmVy\n"
    "aVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwggEiMA0GCSqGSIb3\n"
    "DQEBAQUAA4IBDwAwggEKAoIBAQCxh4QfwgxF9byrJZenraI+nLr2wTm4i8rCrFbG\n"
    "5btljkRPTc5v7QlK1K9OEJxoiy6Ve4mbE8riNDTB81vzSXtig0iBdNGIeGwCU/m8\n"
    "f0MmV1gzgzszChew0E6RJK2GfWQS3HRKNKEdCuqWHQsV/KNLO85jiND4LQyUhhDK\n"
    "tpo9yus3nABINYYpUHjoRWPNGUFP9ZXse5jUxHGzUL4os4+guVOc9cosI6n9FAbo\n"
    "GLSa6Dxugf3kzTU2s1HTaewSulZub5tXxYsU5w7HnO1KVGrJTcW/EbGuHGeBy0RV\n"
    "M5l/JJs/U0V/hhrzPPptf4H1uErT9YU3HLWm0AnkGHs4TvoPAgMBAAGjggHfMIIB\n"
    "2zA0BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZlcmlz\n"
    "aWduLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEAMHAGA1UdIARpMGcwZQYLYIZIAYb4\n"
    "RQEHFwMwVjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2Nw\n"
    "czAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQG\n"
    "A1UdHwQtMCswKaAnoCWGI2h0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMtZzUu\n"
    "Y3JsMA4GA1UdDwEB/wQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglp\n"
    "bWFnZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNo\n"
    "dHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjAoBgNVHREEITAfpB0w\n"
    "GzEZMBcGA1UEAxMQVmVyaVNpZ25NUEtJLTItNjAdBgNVHQ4EFgQUDURcFlNEwYJ+\n"
    "HSCrJfQBY9i+eaUwHwYDVR0jBBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJ\n"
    "KoZIhvcNAQEFBQADggEBAAyDJO/dwwzZWJz+NrbrioBL0aP3nfPMU++CnqOh5pfB\n"
    "WJ11bOAdG0z60cEtBcDqbrIicFXZIDNAMwfCZYP6j0M3m+oOmmxw7vacgDvZN/R6\n"
    "bezQGH1JSsqZxxkoor7YdyT3hSaGbYcFQEFn0Sc67dxIHSLNCwuLvPSxe/20majp\n"
    "dirhGi2HbnTTiN0eIsbfFrYrghQKlFzyUOyvzv9iNw2tZdMGQVPtAhTItVgooazg\n"
    "W+yzf5VK+wPIrSbb5mZ4EkrZn0L74ZjmQoObj49nJOhhGbXdzbULJgWOw27EyHW4\n"
    "Rs/iGAZeqa6ogZpHFt4MKGwlJ7net4RYxh84HqTEy2Y=\n"
    "-----END CERTIFICATE-----\n",
    NULL,
    "-----BEGIN CERTIFICATE-----\n"
    "MIIExjCCBC+gAwIBAgIQNZcxh/OHOgcyfs5YDJt+2jANBgkqhkiG9w0BAQUFADBf\n"
    "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
    "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
    "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
    "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
    "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
    "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
    "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
    "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
    "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
    "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
    "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
    "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
    "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
    "AAGjggGRMIIBjTAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
    "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjA9\n"
    "BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVy\n"
    "aXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwNAYD\n"
    "VR0lBC0wKwYJYIZIAYb4QgQBBgpghkgBhvhFAQgBBggrBgEFBQcDAQYIKwYBBQUH\n"
    "AwIwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAHBgUr\n"
    "DgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVyaXNp\n"
    "Z24uY29tL3ZzbG9nby5naWYwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhho\n"
    "dHRwOi8vb2NzcC52ZXJpc2lnbi5jb20wDQYJKoZIhvcNAQEFBQADgYEADyWuSO0b\n"
    "M4VMDLXC1/5N1oMoTEFlYAALd0hxgv5/21oOIMzS6ke8ZEJhRDR0MIGBJopK90Rd\n"
    "fjSAqLiD4gnXbSPdie0oCL1jWhFXCMSe2uJoKK/dUDzsgiHYAMJVRFBwQa2DF3m6\n"
    "CPMr3u00HUSe0gST9MsFFy0JLS1j7/YmC3s=\n"
    "-----END CERTIFICATE-----\n"
};

/* Empty intersection of 2 permitted DNS names,
 * non-intuitive constraints order (more specific higher) */
static const char *nc_bad0[] = {
  /* Alternative DNSname: two.example.org */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEJzCCAo+gAwIBAgIMV4T0BxqceieCt/KBMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
  "BgNVBAMTBENBLTIwIBcNMTYwNzEyMTM0MzM1WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
  "ETAPBgNVBAMTCHNlcnZlci0zMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
  "AYEAm1IOBuzu9Pya9O3FKhQKus22xPlm+fVex8EV+p3IymnZGZUlDeTX5OcxCOm4\n"
  "G87KNl/UQjgCB6n2FPiIYFbH9skxyvW8ZlG+M4so5yg7mwRjB8QPe0yEOLyxaLaa\n"
  "uNp9icjtPJgPpIrEgppevfiP4iXrRGakzpjayazVCDTp9+XAhdWEi43mN6fgpM8V\n"
  "Yc5sstkEueCjIfhApBzReMTvEUs3jCtmpqIvm07zVLpCh3sWh5MPSZtcw6UiKZdb\n"
  "rRoaypznSkQDGQXCTZ92gSnkg0m86OIOHNQcxLXqfbrNJ7QZBf1wpi04s4DHNHSC\n"
  "k9TpKe/dbDO4vgMgBNrcZ/9B7y95Pe+XJawG3klGhz2zGG7DmvWNygtUcM9nqk/P\n"
  "f7TQhwsU4McmyxvVb09OVwk/2zEaPswv6MFvoxOskcQ5aYhJZs6wLDG3hh8yE4fr\n"
  "BBvJb53flMnuSIWLfzeGUg4eeS8xP7ORApwLM0K0VGLaT4V9lpmWFLot0hv7XAcH\n"
  "jeTVAgMBAAGjfTB7MAwGA1UdEwEB/wQCMAAwGgYDVR0RBBMwEYIPdHdvLmV4YW1w\n"
  "bGUub3JnMA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFERr13TeLMJ3q5QS2W4O\n"
  "HiqwpM0RMB8GA1UdIwQYMBaAFDTfJRBdiC6+QinO/HA/E7TWxeHrMA0GCSqGSIb3\n"
  "DQEBCwUAA4IBgQAiOgI7RgzjDBHgliXb2Q9iuCq/o/08Fz2he8AzTJ0fw+Xd+g40\n"
  "HWnhZZxlnSq/XFircrHwLuMyG2B6HJ9gXWg7SI/5PG9fVz0USC0tcxKzA87iB2sx\n"
  "KWzdfmzBM32ioTFEisH9YQqCVXc3Umol15r3dAZsKGRKQzYjVG8APJS4LYZTX918\n"
  "Yg06jCmp+ZhyRHVhQ1NbrX9geOK8tuZoTQ/10iI1+eIF50a43qA0H8YDuyQbrZA3\n"
  "ECdVIQVCUQTVlTx+JMl7DoZnm+m+BrisAAuq/4TeJwm2Es3IF4SPB/pwaZyx8YnK\n"
  "xqne/auI6Rq7nfsi3owxBjjX1YamlmM6UWdvIsejsy92im2G0+J5s55yw+fCGXE5\n"
  "5mItHVWOiviaPa95NU3NeD8RkUUFI568GM8GnIcSfJi1yxed8UApbCiZMbIIN8fl\n"
  "5mMgyZv2QJXbJxhIiCQixn8nYsj2iaJu9Ns6zd5cFaQSmQxIEUfCiNZ9kO0xwpor\n"
  "tHWgZdawxv2CfGg=\n"
  "-----END CERTIFICATE-----\n",
  /* Name Constraints (critical):
    Permitted: DNSname: example.org */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEIDCCAoigAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
  "MCAXDTE2MDcxMjEzNDMzNVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAtpx8p5POIgdnDbBr\n"
  "fH1kByvm2PP+iN4UJhJYY1V7EMiucC/CU5HoYhi/KpBwoY+28oaISEMr0KEf/3rv\n"
  "CTZRBxuqxCboK6+u/dDOlyYeM0dU57jpKmgCrETLMq92QaIEhNzv88cTaWP0OGzv\n"
  "2klLqim5AJC2J/XWqHGprfdhf9GCWurMT+km7LPIClDHfwnmrPHuNhelfPCVzKpO\n"
  "9S9+Lq5KpaV45DRQtMve5NjUju1q9LotEeEdlu5bnomIK3SyfS+n5AZnLNVAqmMg\n"
  "kSB1ymtWqn4wiw3hCBz8biSlkeowdh37cm3j0za27R3IjFnIQLD44Ena3pTU8v+P\n"
  "4/k1OML8UWXpigP5QuTSASx0fXiShHf3baY1HnEqULfYvi+IUb6wMs/3f13NVVBE\n"
  "z+LsjiWlwqB0fK5lefO32cEDvtSMlIxgt3FUDCo3/rLAh4ZorURONh4MUWiODTSl\n"
  "417JOLB/miH37jodViv6zfbtTvw/+GbZM9TnvHlzqvZj5nLFAgMBAAGjgYQwgYEw\n"
  "DwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzARoA8wDYILZXhhbXBsZS5vcmcw\n"
  "DwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUNN8lEF2ILr5CKc78cD8TtNbF4esw\n"
  "HwYDVR0jBBgwFoAU4SfGxDtCWqGQsk7xBIooEZNCoMYwDQYJKoZIhvcNAQELBQAD\n"
  "ggGBABJZw4MHkE+8Fg+r/ET/kJ0n0NtsB57O3ogPpe/0/EWpsEJsjnRzimfu5NjS\n"
  "PIcEKk/l2Ij8vbmDxb1uNsZmeYphdjb+w/D44OnxahxeLELwZPHWpJLvuf5S8bsz\n"
  "Z0bZFNkDUXYbKDX8kWr1gNCKURBS344fRfe8HzZsG68stouvCuOh5pvre7mGGMJI\n"
  "5/OMISmQiKIGLpUi1YOSRM25VMZ6GnzgYiN/bcZU1ph+R0lQv7/RRZ7oiaYmFBTi\n"
  "FfWIE2hsJla3mbhCnUUp18MpRu4+gPirCVhNQ+ii9FPklcIhXxOrq6cqfX/YAcWO\n"
  "uF70tZK/+Z7UXqGYJeQ8pdmlzjNGSH7Q6D+QKNAjZ+Ovb7zEh3NmyTT2XEykMR6+\n"
  "bQYaGGcRu8Uvz4wHDaqeUuF/vgTiFaJ8kwNGX8Xb1x+ok5QrJAKZzvy59kojz8L0\n"
  "ukQ6SqsvZ6SkJRbHHEh39YPNdC66O58KTiayjKgxQmVHsMOhraI1+YmPntCNBqNN\n"
  "AvhLDg==\n"
  "-----END CERTIFICATE-----\n",
  /* Name Constraints (critical):
    Permitted: DNSname: one.example.com */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEJDCCAoygAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDcxMjEzNDMzNFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAt7EPrrm4e4JEPXVI\n"
  "3s6eNQCyQv24LU9HD+7hDMEOFf//DoHeb8QqSDJdiCk+Hax5ydKZR2h4HZRmj5HO\n"
  "s6nxh0AWL645fKcvfk9Oj3r1roLWxH9Kk/UR246s7FcujhDzEz3LEOZUedeMY2CS\n"
  "tyjPLsKoP0jiDslRk0Yt4m7OfayB71B26qq92SzRr2YlMvf6AWHEiZhCRqVNidDV\n"
  "LxdMwqIkO8s93DN8Kw74X8U5o5vTjmmDiW1HVrqsxOuImnjQ4qTUiDv0JbzTQbTp\n"
  "uPOlJ5u/qMTK1jsGDcgfnojHLrsyuuTPR4v6Rmebpi0HHrT2PkxLeGtQEUxM7TeS\n"
  "Ccq+eva9zm4UngonS2/nkfYawLDkP3XQ7cJQueNKLC5etDr9NqhFaD624InblWGy\n"
  "V7jtEJRwRPH9FeMG7HyWb4BHYz36dCsMLbsCrCLIH8H7r/1nswVxlL5SRwiL06fK\n"
  "11pwae1uyNgQuvjno4zHKM5V+mJe1Tz//2X3bfb7crFPQgsxAgMBAAGjgYgwgYUw\n"
  "DwYDVR0TAQH/BAUwAwEB/zAhBgNVHR4BAf8EFzAVoBMwEYIPb25lLmV4YW1wbGUu\n"
  "Y29tMA8GA1UdDwEB/wQFAwMHBAAwHQYDVR0OBBYEFOEnxsQ7QlqhkLJO8QSKKBGT\n"
  "QqDGMB8GA1UdIwQYMBaAFJm3gYrByx1mGmb4CnWXtNzxwGapMA0GCSqGSIb3DQEB\n"
  "CwUAA4IBgQAU53SjH5nO+ah/pAQaIDuxaJ6yaFWt1ZuW8riu/dTqn9vI0R4K6WCh\n"
  "EZ/rf4Z4YWMLm0+wI/+1CbFHtuZ9savA4qx7rtXQw5mF1JTEBsBM/chiXZ50euKW\n"
  "DRE2e8egOESxDQWk5cnaAxtbiRYXu/KYGqFcGeRvSoy85gIwfjBtweYn+rOwM9Yi\n"
  "9JsrKwsdFlzvzB6+ozDMCHncqtkU3DqI9QD80oP033z45EJxWxOhd6YhnrZN9SKp\n"
  "E/lnc/XuY3NflVE5PGT5efrfGkAfbp2fWPfvc2PP0Lh172zoPy3mBwcXpWdij+H2\n"
  "JCzwEqzxQzLpACtFy0kwq9HhzfgcdbbFmUbNweIf30eVG0XQ35myZy9Q1LQINhaj\n"
  "UN0Ao7qtLUtC8z5DlUFMuEHQBLhFkmuRHJHCkFRqLO0nHFYmKxtQ2nNmbHt1909s\n"
  "I20OEegNTFV8luCbFahoILckFlsbep9P4d0wOMjZuJkLyModK7Yx+CdOpq6/Cegg\n"
  "gt+aIvJzHEY=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIID3jCCAkagAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDcxMjEzNDMzM1oYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAmmrn3nN2bIFYipZN\n"
  "ED4nbWSc0ZYDbo8VzqjCKNVhMbYJlu07tV0qDK/1IDHf9awo5AladB4NEH3oJi9c\n"
  "sCtiBtJ9UnqD+gxsJFNtpOfNRfelOE8R7suXAxDxJto7YLtXnLCcMx+UMkhlDfVi\n"
  "Yy5Hqua//+EFyffokOPJ0/JCxFvTd9ldbNnxgLL27yDJBL1e4SMSw03/wKoLS0nW\n"
  "Sjzcu1+Y2sdj6CNVDtZjGmDLMNtyykX4BDz71PrlR1euktkuHS1HMthQdj3rSWjU\n"
  "Rehe7LxjYG548SpnIVA93EOfDyqLhjpKUL8+rA0cKBIsaJK+TyUNQ8XYa98djBAj\n"
  "gjRYRsPkZt/FH2BTg+4XSHWMrmfEbxyxqAf6euUkY4Z+Y2xkUHQl5GdYk44Rb/+4\n"
  "NxSBBKSj+6SqK2f0o3WTHXwJTeX+B0rV2x507hFqf6lRGzwzffrXKqH3yxfqbycl\n"
  "XlahOiBJ1xKNrR0XGeq9yPcrWv/RYvYt4JJp9OV1U2Mz3DRRAgMBAAGjQzBBMA8G\n"
  "A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUmbeBisHL\n"
  "HWYaZvgKdZe03PHAZqkwDQYJKoZIhvcNAQELBQADggGBABaf05+i07lJI74gv87t\n"
  "87BuaYEvySlQuUqycCvEs31RXFxJQhpHS7RvqPw6vqDv418SZwd/hNaC7a1JU0gL\n"
  "Zuha61y9u6/HbmeCBSgXYcd+4M/2oPz6WcJ9uoOZk8D2NxafubVtyXH26O6tMEnK\n"
  "0JJuV6q7fsqvIHf+tvRs/fTD7gKtyAsj1OoO3EjkRRQPnHOR4anXr1jxDFvldHEs\n"
  "qhlibWotfyvS4BvSk8nEo+/hrXs86cQDqCg1bbbz04sTQVHW1/kCKYl7c/HQGnTT\n"
  "I3Yc7pFq7n5sNP31XN1a8VaGiKseNXmxjhS9XlIvQ1qB5ObE+Dm0tWQbrDo73udb\n"
  "dW+I2/Pcij0tGBi8Cxe/PZKv5wio4NpWGTNiF6PMSaUp+lqX2iLYfjjl7osr3Hph\n"
  "gnwxlST3q0Av0+91jCfj6IZ9YRHLakceaRxcj8zLoVGpQqTdJjuH4Sy7nKoL58G1\n"
  "96Asqk2NsUztvRfw5pYFoe7ZUgsa4M+0/nZxOPd2UeodMA==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Name constraints: Empty excluded DNSname, empty Common name */
static const char *nc_bad1[] = {
/* DNSname: localhost
   DNSname: www.example.com
   Common name: (empty) */
"-----BEGIN CERTIFICATE-----\n"
"MIIDSzCCAjOgAwIBAgIMU/xqxDpxZ3J5cUcrMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTEwIhgPMjAxNDA4MjYxMTA4NTJaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEChMIc2VydmVyLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDP3GV/JSLCCmx09hJINJZC8fSUBE1IVbZsY/q00rZOw5KwPioLiMOIup7W\n"
"na8YJ2ama0GJjU86PlJDhBH6soaY24ZCW5kKhvfnSw2TkpW6umL7psvuBslRoMxA\n"
"t12MpapZDFZixZjV44Bstuyt9sI1ze3au+5C7E4+z40o/3uvbIiN2iz4bPgwPIMu\n"
"5V/bVTei6uAcu4fNHh/AGnAUJa201QsUhM4+VRFc0XmanjSulySD3obwERDneqab\n"
"77gnIP6zwuFXxHgucbmzU8DIgVhes2k4v6AB1nPxlpUL5+E+W4XDg4ckGGkfxgcn\n"
"dGYvuv3pwIyHvb7Z0A8D6bE435cnAgMBAAGjgZ4wgZswDAYDVR0TAQH/BAIwADAl\n"
"BgNVHREEHjAcgglsb2NhbGhvc3SCD3d3dy5leGFtcGxlLmNvbTATBgNVHSUEDDAK\n"
"BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBS5dbTqiPbhe7Fg\n"
"e/Bs92qsNgWSzjAfBgNVHSMEGDAWgBSgAJcc9Q5KDpAhkrMORPJSboq3vzANBgkq\n"
"hkiG9w0BAQsFAAOCAQEAdkVCQQ5mCuV5WTqCeH7AtXxYM7IV7q6O7uqirKwDUzGD\n"
"wB5shurAPMn9IG46O68P+BPLMfwszFDgszNrjuMsKb7hLT5+pVsd6XRQeimPJ5rN\n"
"/szMfALLAcw8Yxt+1YbxlgUgybFsiK2zdBpAshU+FzMRvFfq9rnqW/VXM0J6ghz3\n"
"VqLbviOY5KpCLzfG0yM+CTcKXVFau9QZK962AfXzUwaCymw1cRHzQlpdMQtTtcIp\n"
"nci6MKXViEdeHbPLcZe9+vzSSpFh5u/l47w+2B1oz7mndFFpxkw37zDaVH5yAFxK\n"
"+5VijiKxH6nmniLUX8Zsv82YBaO0liNb2fOZopxQGQ==\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
 * Excluded DNSname: (empty) */
"-----BEGIN CERTIFICATE-----\n"
"MIIDFTCCAf2gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEwODUyWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC+iPUnEs+qmj2U\n"
"Rz8plNAE/CpeUxUfNNVonluu4DzulsxAJMN78g+Oqx+ggdkECZxHLISkzErMgiuv\n"
"bG+nr9yxjyHH2YoOAgzgknar5JkOBkKp1bIvyA950ZSygMFEHX1qoaM+F/1/DKjG\n"
"NmMCNUpR0c4m+K22s72LnrpMLMmCZU0fnqngb1+F+iZE6emhcX5Z5D0QTJTAeiYK\n"
"ArnO0rpVEvU0o3nwe3dDrT0YyoCYrzCsCOKUa2wFtkOzLZKJbMBRMflL+fBmtj/Q\n"
"7xUe7ox62ZEqSD7W+Po48/mIuSOhx7u+yToBZ60wKGz9OkQ/JwykkK5ZgI+nPWGT\n"
"1au1K4V7AgMBAAGjeDB2MA8GA1UdEwEB/wQFMAMBAf8wEgYDVR0eAQH/BAgwBqEE\n"
"MAKCADAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBSgAJcc9Q5KDpAhkrMORPJS\n"
"boq3vzAfBgNVHSMEGDAWgBQ/lKQpHoyEFz7J+Wn6eT5qxgYQpjANBgkqhkiG9w0B\n"
"AQsFAAOCAQEAoMeZ0cnHes8bWRHLvrGc6wpwVnxYx2CBF9Xd3k4YMNunwBF9oM+T\n"
"ZYSMo4k7C1XZ154avBIyiCne3eU7/oHG1nkqY9ndN5LMyL8KFOniETBY3BdKtlGA\n"
"N+pDiQsrWG6mtqQ+kHFJICnGEDDByGB2eH+oAS+8gNtSfamLuTWYMI6ANjA9OWan\n"
"rkIA7ta97UiH2flvKRctqvZ0n6Vp3n3aUc53FkAbTnxOCBNCBx/veCgD/r74WbcY\n"
"jiwh2RE//3D3Oo7zhUlwQEWQSa/7poG5e6bl7oj4JYjpwSmESCYokT83Iqeb9lwO\n"
"D+dr9zs1tCudW9xz3sUg6IBXhZ4UvegTNg==\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEwODUyWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC2cd2vlg/9nuzi\n"
"6S6/qyJRnaUoFutajTwGqfQKZpbqXI7TcFZwKidzAZlZiU+sAvkY8d/9cadlbrde\n"
"S9HGv31QmexWjgWAMGNpeyiPlXIN8xGzIbZRM3FCih0bnIyibdwgAuU14dUrChGD\n"
"sQ4SAmRUpThkB8anvC10PIsxfnifBwJI6dGQZb1KOxVOIWg7Gb5tNFkZILBGv8wk\n"
"cbycIBYC3lRX8svUj9mMiro53f+4ZGbi4DcSLIdw4ebAczfBd+uHM2jkHFZUNuAY\n"
"7rGZAAuqEh5IE0QHS9CV6mg6Pf9+sLGMBZUbix2sxRntAEyz8+kO7W2zgmKPla4+\n"
"y54cIUmBAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUP5SkKR6MhBc+yflp+nk+asYGEKYwDQYJKoZIhvcNAQELBQAD\n"
"ggEBAC4N4dUoGDTGf93DtjEcZzqGoAd6TsCYy6k5zeYMZFwogfArg0IMurcFDLeN\n"
"PXe3xl9RiUjbiZMaHWju81kIO34z0NLd705XR9QFEc+xiuZOMmm4SxciAF5xo+Hh\n"
"Fhc9cVa8Icm2ju86Q4yhJziYrElH8VwHTBE0k+RE1cK65F5PQFGGBlpGm9EMcYTv\n"
"EQQATPLuWwKRAFNJBx2t3DAeMseo/Iq6Snd/UfdqgLkV61YtbzqL8bu+a8rgMAYz\n"
"ovgORsI48TlbU4H7YI+vzPO33tRV2m4dOxppMHzv8Ie2LIIfqYn0HRd87c06djEA\n"
"EpXfXGqxjX5vAtNPO5fGGzghol4=\n"
"-----END CERTIFICATE-----",
NULL
};

/* Name constraints: Multiple-level constraints, intersection empty */
static const char *nc_bad2[] = {
/* DNSname: www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDQDCCAiigAwIBAgIMU/x5DBI1pGSO2eYZMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTMwIhgPMjAxNDA4MjYxMjA5NDhaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEChMIc2VydmVyLTQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDL/hbj+RVDQ5sA4VR2FZ+P+/jju7jkUFUovwhCrWpGXFs0WDokcOkcXc3f\n"
"0yturFWazVEKaaZECiDpGf6iXmNoJA6fPT+G0gPlIL6wh4wKQG+vwVYsX5ZkXOMR\n"
"sl7BqCpeCChkth54mClEwOYW3WohdXqiJfxoFVdgnisbqkxYtz9aXzYE71cZIFAx\n"
"nL7V/gY+G/m0iZCdfh7YEDlT+qtLkGyHsyyTxwUH4yyqcsFl4WWG6wAdKF5U69yw\n"
"uo61J5wpE+yDyS0u4Cjw67d29OIHsT7GAq+fP69vMoEHPvPUM/aA68AycybV2OYt\n"
"8OJAyZqf/6zvnlrbLuk08kWf1TD3AgMBAAGjgZMwgZAwDAYDVR0TAQH/BAIwADAa\n"
"BgNVHREEEzARgg93d3cuZXhhbXBsZS5jb20wEwYDVR0lBAwwCgYIKwYBBQUHAwEw\n"
"DwYDVR0PAQH/BAUDAwegADAdBgNVHQ4EFgQUz+XUeM8bwvK3BD7NVRRYSeZSlA0w\n"
"HwYDVR0jBBgwFoAU/n2NeCCnwHQwFpKwi42A3H+w9W8wDQYJKoZIhvcNAQELBQAD\n"
"ggEBAPJZO19PhmxQ1ZRDLQWZWxuQgD2Wwy8sS+wnlUo/TZM7+pT5ICjLdEKgETPd\n"
"HOqgAMQFaUpp5vx1jUBmmKdPOmwEnV/2zbw3GrYwAQjxunXD66iHjYbodl9zBumM\n"
"NXDGsHnKYNu9sPdQSMLC7OEOrKvEhH2afOvYDORQbSGXh7+3js7Mzggy0NoYtxnK\n"
"4wqt6g73SFkV82mTQpUBK218ROjuWVBUmWxq2JU+qvsAKbhz+Tjr9+kmFcNBRgmA\n"
"Zga26uoQhd6YP9DKbCvf3sK4bi6A5NROeLf9BzJHWkGani4F9wOjRmLVnLlB3BPi\n"
"tHZaLDU9fUnf6I6p3nu6LPTH3JU=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Permitted: DNSname: example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCIYDzIwMTQwODI2MTIwOTQ4WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQD6kCE6KmDCkagX\n"
"fvq0rx73h7zn23phJYBmugfp5fPQYIJ1463dGmlFK4Lfkz2V4StgM2mbFFEkcyVC\n"
"pz+PyeSDgyWJJ/RwG690lLfu5JfNLvwxj/rFNK6rS7EpADew6RgURCpEMt6z1uEk\n"
"+IQsxKoXQmAcdtc/ubPFWInotg7Avoid0sG69s/+hq/nlGE9A8JMFnsLh/n01d+F\n"
"9dWsjrNiZ+mfTE8w0MVTq4+8mvmPmnjKsiu0rgqaVTmYpZW5chz9gGrZCr/Wr5CL\n"
"zPsAYaWie+wo8cR5qMEoX+JPHqM8eP9K1v+uYc03aD3u1/QYdxY73OLn31+jYAqq\n"
"tRJjgSERAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oA8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU/n2N\n"
"eCCnwHQwFpKwi42A3H+w9W8wHwYDVR0jBBgwFoAUJ2ngSf1CqeYVG3eUNcH7Lt6/\n"
"ECUwDQYJKoZIhvcNAQELBQADggEBAIQruSBUyQ268Js61XT3nlr1Y9HxmKA2DIuM\n"
"WcAicE1XIpuxDpZ/VmKH0/o2JVR1A7uwSMEnHdShHixMbpYrHRDnZITxs2lsJijE\n"
"r7YdqadH7EbjDIXv1DJcPnNaeqFPbyXEWqLYoQf6UPBLVRWeKISPN0hMaIZv4Y/X\n"
"OcBceajAr0XGxASRFDky26M01AVPZoYjgT7vLp835yk9BY5+q0GxlSJl6HbQ5ESA\n"
"IoC3Limt72niobmvEryQDq7qUUoR7hB1SMKfyX/qktxT3UCBLKXHsp80ECJ2A7Sd\n"
"YrHjFE6LnWHwGJFYZ1eYKiOjglVRGv3+bNX07bQBWKzRbLWYM+0=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Excluded: DNSname: example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCIYDzIwMTQwODI2MTIwOTQ4WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDUbGpRRMdnTzbG\n"
"R7J9qYJoHOhgRp3lEa9sBP7/pNNIsEhXmvzBu5J0buJpAfRPmWcoZauKsVrKnE69\n"
"CFlTDVCIahQ5gtJkGdjrDrQDFFCMnKC04Lhq+EmbASBTn5GRQwJqEUi9xRpj0yOL\n"
"0XGlMp4JS44eAL4giywzPtOAZaJlr4kdOnSPK2SHFVwQGfQiNmzD5ajmsjM3k4o3\n"
"R2gXAsudyasQzRAjFyeo2ry7klPUPS5RHJ6B2n87e9kLGrYb8+O9I9FNc/w4J49W\n"
"AovVr5vcs9Km25jLUn43KDprDhpXddEraz6WyZJRMTZVRRUizET3gmojZFFD4zOQ\n"
"mneVYerpAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oQ8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUJ2ng\n"
"Sf1CqeYVG3eUNcH7Lt6/ECUwHwYDVR0jBBgwFoAUyFGHFFLCWYOQTLmh8jJpGyxS\n"
"bSUwDQYJKoZIhvcNAQELBQADggEBALGFCZXC1KPBBPMtLJNfhNBtBBC4i5q+1Qeo\n"
"aJL7dKVuBn79WAuND8rvJvrPKpGTmyxkcnqRXSBVH3c+Xi+v5ykLrtHJ2x4TOrmg\n"
"RBAaBqDuecQ9Ec0dCc5ODKwjdI/wEOGAS4sfrMXzQCv+UJqi2lE0fo/xDmS/azCc\n"
"WUjFSQOuWnCJIIAIyWlF2bPtdtiaydHKkTcG7c/zwrxRaWE2Q2G+dm+itpJ7sCtx\n"
"ZFfGMLUl7mDadhiYrxq1SnwrObMwbngPNZyUBi2G7jnXlyFc9X/w6fVIULLxN+bn\n"
"IzHWcRrBZ/ShdvCStmgbTlKNtvg0LWAk7QWzy2ibaXS5jp2r+Fc=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Excluded: DNSname: example.net
   Excluded: DNSname: example.org */
"-----BEGIN CERTIFICATE-----\n"
"MIIDMTCCAhmgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTIwOTQ3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDsnSNH0g4bvAXx\n"
"zJAfs+XFWpNIpJtD/1H3Ei0ezfH5Ug7GNpHKlYKLCIShqCFj9WSwzSyKNGCHafdt\n"
"PPhKo9uAo2bSaBZjmzxfqSOGDEXZ+4LlRgSPX2Arc0i97ZsPf0nkWLfrxlTOAQIm\n"
"DxDNrWlGrCl1dfPiB+EyMzo+++MCdTGEsdEdRFm85QNjOOTiaTeUpUh5G27+hbuk\n"
"PaRlZ1GHJYlrLHK/2qw9/Mw+gNnfn/Efw+lNeYuQ3tco8IAMN0jB8x1hDfOxTx93\n"
"mrFzAdGTfsYZc31YapATk2re8IJGeKSCY4XP2HvYZEE1fYdw8ZcqZ/Gv1RdXyxvc\n"
"6oT5r/PNAgMBAAGjgZMwgZAwDwYDVR0TAQH/BAUwAwEB/zAsBgNVHR4BAf8EIjAg\n"
"oR4wDYILZXhhbXBsZS5uZXQwDYILZXhhbXBsZS5vcmcwDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUyFGHFFLCWYOQTLmh8jJpGyxSbSUwHwYDVR0jBBgwFoAUTBVq\n"
"WCSLkLZte75Q9bgKeM851qowDQYJKoZIhvcNAQELBQADggEBAGKVEQfAzrWj8wmQ\n"
"l4sm+i/pgK0I07jNMvgUDsvAmjlkndWxoX1ROe0Nd2I3d5te0+G9MR6CTOByr8VE\n"
"NLyXEsrk++BsfLk/0UNFHwq97QLTzzyEXQYQnDza8R1jdlr7XpGZOoWczi08yMAk\n"
"UiJyq2xaqerTlIYp01T9a3Nb5tWFyUVekJeyJQakj2VLaKkl4hCfK3h/HFBNJ3yf\n"
"AvBu77wQeh6n8osNDCpW9e1KRAGisDCFrTMUlyxQIK/OXhjLzu7qDKShdNnfNRmc\n"
"H4W9ODLIm8AX1S0udg9OebPhNWfM2bDFzI/dIX+yHp6q0oepbT11rKG8G+5M25uU\n"
"AUhTFC4=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTIwOTQ3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCZSS6SaBALkN4W\n"
"Tao6pqaDmi1edtBskNHdO/lYomElQtIkw6wQHlJjIdjF3vnhEWU9+HOIZrWKth4X\n"
"u+naDZquOb4GWPq/X6/KBoQ4hq/XZJaFEDPeciNcVeylWVlHi1OeGm8uHZxAK/6d\n"
"wpGoe/0K+QaLFdbm/srw1LGvCwbLwNDKePX9TgOfVKdZtGZUdDDo6TXUmhNG+QeP\n"
"7Fv1n2PjQFkXiRwVLgJj06DvR+ft81x2gjEVS+vxWg0+cbJvBI2ItpNGnIWvbwl7\n"
"BTyNRjvsi7ljFn+SfaRBLXE4aygQFQ9UCHNNYtkBO73BXv/SgcFXzSDDN5ZMfpg9\n"
"SSWkEApFAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUTBVqWCSLkLZte75Q9bgKeM851qowDQYJKoZIhvcNAQELBQAD\n"
"ggEBAH3bezfaVpnyqZRJsZ8sHAIpJWa03mHl/mqRpT0qg45Agzwo7yb5dubiHUBJ\n"
"BK/kAMhICjHAH+6E5XubGVSFvbzBX1FeKQQgzghN4niosOCLZPUtl8gJGZlsOoCy\n"
"6HldkxXa26GBZR0NvJb/p83VA7w5Zlp5j7Rp2VkWwRniaPex39dogDX3IwnoZKzL\n"
"ogyeNQPG2qLDBdZRAVng0eJK1Ml5PHxoEkcFwFsxd4B1cJV0VCMk7X7oEc9qBtUB\n"
"Ye/bst72puWDK1lBhT6EFhDDbY9xKm7pvUkGx80gWm9JZ0xGCaoM4tyEAaCd9tYZ\n"
"JFvnIEGJGeGjlRLJZGS4mZ/Q5mI=\n"
"-----END CERTIFICATE-----\n",
NULL
};

/* Name constraints: DNSname in excluded range */
static const char *nc_bad3[] = {
/* CN=www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDPDCCAiSgAwIBAgIMU/xvqR+qZTQTaWIIMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTEwIhgPMjAxNDA4MjYxMTI5NDVaGA85OTk5MTIzMTIzNTk1OVow\n"
"LTEYMBYGA1UEAxMPd3d3LmV4YW1wbGUuY29tMREwDwYDVQQKEwhzZXJ2ZXItMjCC\n"
"ASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJ46qo4PFrBfYlQXSgtbk/rO\n"
"0aO8/Gh/bIh4c/JX0RfqC55bnfbkO9SwwR+eU76INyVb1exmd7qsl4R2EgS8V3Gf\n"
"3K5k4tNecMfxT98MWiuSSp8Q8+affUF5t9TSLujL1dckMlPfH9hdxCYhJGH51mkf\n"
"wr3oEmwNXsA9OQ8oxq2i8WxQTJGUXkwx/k2L2NRF3L8vjRnXRfKSISkkDXeKYMvo\n"
"V5ElQwlKo0sonttUIGOVav8Cf4GnFQzSJW+RfANTniGIq16jE+flKz1kQYRLLoeA\n"
"fgH/1vI1v5xqMURNW/BQlawAE0HGj4MAyfebhsWmhqmcNqGBf1OfHMNdB1vamGkC\n"
"AwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDATAPBgNV\n"
"HQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBRiCra2BJERmr2/+Klot32criLTbTAfBgNV\n"
"HSMEGDAWgBToCEW507CZ42L/fn7H+DLcx+zW/DANBgkqhkiG9w0BAQsFAAOCAQEA\n"
"fsG45/VuJzw5DRbrE6o67T3EgFfPCzr+xc0JmTJSCHvWIx+2O1VspmJiArNTwQ5O\n"
"l8Hq2Sag9Wi0cyRC8lVKPbC7Im2fZ4m4endOhiEmaOHBCru5bIFRwDvtG3u+yEYI\n"
"rzRU+6PdwmLYwc+ks8qEqACw772nElJxOWXmYEMtFpYh8eujfzjmUGIJyTotrm72\n"
"WX8phKA/xogZaSLD21t8u77PE/JEcJ2LXAa9dq6pGYru1vyuRqq8ZeWiVAAqD6hZ\n"
"cglKk8dLi6esywQMGEGqhRx9y1A0mPZO+M599GOgWTbShUB3pUyaLLLLnD9Dciwq\n"
"4E4iP9rdfgStOfz12BsKOw==\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Excluded: DNSname: example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEyOTQ1WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDW2Z64Ax+MNj1j\n"
"2QM9hjP4ybs+mz51vfDAlDVVdh2bzJOHyjTIlzI71QloH9aPuv5d92tTxe7/7afY\n"
"fWC61AF0WKj7x8h570IW0Zye4ITEnDLlqk5Bn46IP9fWDq5xYVEGMaYT4l409Fyw\n"
"JUZjfXqMefAXhj55wZoz+WMM0AB5LABlojLkV+iPMVJgfYWhcVijd92Yebp8R2/+\n"
"z1nF0vQtV01tatWTEiJajPRHZCwVe71rXEf02nYiqCw5RwLZrsug5LZ+K8LoBbeE\n"
"ezcJT5y8uf4mpTmTj2Po7Kby22yl1wkVV925a2Of7ufDL3d56SIM1foNXAAmlFar\n"
"M5Y9hIZLAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oQ8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU6AhF\n"
"udOwmeNi/35+x/gy3Mfs1vwwHwYDVR0jBBgwFoAUWU1t/YrnYJGhEfuoDj42A4ui\n"
"j0gwDQYJKoZIhvcNAQELBQADggEBAMj2jjHmYLWxGMkLOIQ/MGtvchZ+v6vmEl7m\n"
"GaoHkz0sxFeJqs7mwcybvwG4tlHB/PhaLTH4HfN6PLNbRA4oamr3gFsEtd/JRihw\n"
"X/5CvdJdu/d7uN36yrD5ZTJmt5v1sAXqzkVYXHUSQLOLTIVfwQfUv8IrxTWgbhNI\n"
"mIi55bjCyOWYzZsZ5kTDNFcBkoYiMks2fVuUdP8xrxoweedVswUdkwg1TyWLikG3\n"
"47VuQP3eA7+zEkFUeywG89DTOpDURAlvBzaVTjKn++3RgH/A4Wa+MX6HTHXjxBIU\n"
"1uGcMjhPjc99F81RaYdIlFsQiQ74b5RwdSvGo0e67ssgar0XKgw=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEyOTQ1WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDYTAYDM0lZ+RMg\n"
"M1M3q4n6H/VebQ1CG1ztkinGzj7eH+fNyi9Wq5EqB/0S7jVPEuD0o5jBrwI6XFoS\n"
"MQiWtqWHGh47qijX8y/oc75Sn/2b1gGF3zDWM9LygZZW2+QOIrvK5TcU+rAmXKsA\n"
"765z0nTIbL3vNr9n0yEM3E13tk3Qjqx+OLhJ/ZyLKW+w+BuhLp79LcVtjNnlVfvC\n"
"nVgLvo69YGdJxhPUjjVqKwTlvptyzELQSSQMenPmvhz2kRXjQ/6jog4tb1qkzfpP\n"
"eYB0MVgSLeWBgNF3VLTSH06RHvXEQcdP2e3AR67sJxd6UJ4vOo1widQs0yWTZpCB\n"
"ZJawOPqDAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUWU1t/YrnYJGhEfuoDj42A4uij0gwDQYJKoZIhvcNAQELBQAD\n"
"ggEBACUbydVQKZi5ulzn/lQK5x/DZySJflrevZ1plV7BVBhZhlvBO0VARGNn+NW5\n"
"G9RqY+itmyBbW/Fl43gWiHQynYneK9tYBub7WeJqr9iTX4zvI7V8fk/vbyfVRODX\n"
"cJ8JzeLYqi6Hm1PK7Q9dz0rgyulXeuCyDeQ4jzoGIm2l7atUoGZB0f9YCJyeV2ew\n"
"t8jMZr2sSVMgvT87S/EHMe5q5YAJQzDBAadH64icaxW3e03UeH6JYblohsZVQTIE\n"
"wl60jozIStml73oyocfytsErDdKArrSSHxHaygAqoVu+9O5U90vwK6VDuGF0YzZj\n"
"ZKOAu2HuFHpCMbYzUYi3FMOUU5k=\n"
"-----END CERTIFICATE-----\n",
NULL
};

/* Name constraints: Multiple-level constraints, different subdomains */
static const char *nc_bad4[] = {
/* DNSname: sub2.example.org */
"-----BEGIN CERTIFICATE-----\n"
"MIIDMjCCAhqgAwIBAgIBBzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCAXDTA0MDIyOTA3MjE0MloYDzk5OTkxMjMxMjM1OTU5WjARMQ8wDQYDVQQDEwZz\n"
"ZXJ2ZXIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCtq4QWPb53ClQL\n"
"hpCyCf2oX/WnrX2FaPv8m4J29fpkSTn7uuasEjLzwKvltXPJp5fE9jJnX2JyrS5z\n"
"tBGcH7/OnxEOGtZN19gLQLRPqCjzGkez7moQEbpnO/M8xeUuil4CbuhcnTA93vjf\n"
"i2mxQMgjS/Ffblbbv5QR6R97Eu01w2gbar7S0nj3ctl2lYiBWFIcBVbunVgtlC8L\n"
"JgW6tv6jay/GZSUBTw8ijh0o0S1ZEVqoOBszDdLHFwO6m3XpMNocYKh2Eva9LcA+\n"
"NmVWywaAk84RiMttMjlc+Y3Q3UhEemgh0RTE5oEIFjV9Am4uBM8LcwvmIat2oYtM\n"
"rbhczTG9AgMBAAGjgZQwgZEwDAYDVR0TAQH/BAIwADAbBgNVHREEFDASghBzdWIy\n"
"LmV4YW1wbGUub3JnMBMGA1UdJQQMMAoGCCsGAQUFBwMBMA8GA1UdDwEB/wQFAwMH\n"
"oAAwHQYDVR0OBBYEFM2roPUnRWzJSXG+5jGoBJDX5ZshMB8GA1UdIwQYMBaAFM2r\n"
"oPUnRWzJSXG+5jGoBJDX5ZshMA0GCSqGSIb3DQEBCwUAA4IBAQARXEGjiewd62mb\n"
"kiJ5ezzxgsnfLBlhBdrt6bTEWS1Ehnnx1HMfnerFuSmjk8SGsVXgHsqmzY5cg3Sy\n"
"449+VqkMESOiQRmj4DmNNLu25ShLWlcfDPXBFZavPtNKUrBEWfthI7ZvHBeNe18O\n"
"kkgYJGKLmMm5+Bz2uDOTBTPUmZkX98sUZgdcFyN8OI2q22MLWed0a+ZNZFRUN/Iw\n"
"9gg6fzSmMBShAIxei04CZE2Wx32a9tr+mNqHa+Puch2s2EqlYJBlDGa2QgMWA8IP\n"
"09CHQRxeICqV+cX6zJP4znQFqySX5rWWeD4FKkdQr8T1IX1bCcGKCkj1ulEntOjK\n"
"8ZOkdWuv\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Permitted: DNSname: sub1.example.org */
"-----BEGIN CERTIFICATE-----\n"
"MIIDQjCCAiqgAwIBAgIBBzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCAXDTA0MDIyOTA3MjE0MloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
"QS0yMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArauEFj2+dwpUC4aQ\n"
"sgn9qF/1p619hWj7/JuCdvX6ZEk5+7rmrBIy88Cr5bVzyaeXxPYyZ19icq0uc7QR\n"
"nB+/zp8RDhrWTdfYC0C0T6go8xpHs+5qEBG6ZzvzPMXlLopeAm7oXJ0wPd7434tp\n"
"sUDII0vxX25W27+UEekfexLtNcNoG2q+0tJ493LZdpWIgVhSHAVW7p1YLZQvCyYF\n"
"urb+o2svxmUlAU8PIo4dKNEtWRFaqDgbMw3SxxcDupt16TDaHGCodhL2vS3APjZl\n"
"VssGgJPOEYjLbTI5XPmN0N1IRHpoIdEUxOaBCBY1fQJuLgTPC3ML5iGrdqGLTK24\n"
"XM0xvQIDAQABo4GmMIGjMA8GA1UdEwEB/wQFMAMBAf8wGwYDVR0RBBQwEoIQc3Vi\n"
"MS5leGFtcGxlLm9yZzAiBgNVHR4BAf8EGDAWoBQwEoIQc3ViMS5leGFtcGxlLm9y\n"
"ZzAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBTNq6D1J0VsyUlxvuYxqASQ1+Wb\n"
"ITAfBgNVHSMEGDAWgBTNq6D1J0VsyUlxvuYxqASQ1+WbITANBgkqhkiG9w0BAQsF\n"
"AAOCAQEAR8wKnufviUhLtx0og+P/F52BcdQGgVoBdIpix57DeBQTn9PXuF70hPjg\n"
"a/ZeSR2b65JnyO49i9bBX8ctg/FR+LBHdxMRcIN+VxyKFBtyFc2cxW5a4BWaD9SG\n"
"hW6gzWviV3XwTfTKuY8h710cEP73Yyu1FOuejI34Jiz0sNXqhPc+08k9maFicjfk\n"
"Ftpft0y0YM1DJmUMDMQkpShb4kojLwsSYTxU8DTRHXvBHrAdH4Np50tm6FYIXUNS\n"
"iXlrg/c4VPwEsf5/sR+Ga60LPejZsrvhtvimUsGdNNVVWRtsgcT2jHXxyR1N1vJx\n"
"qPlLDNLLWp9nfCk/55QqSf34dcKomg==\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Permitted: DNSname: example.org */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIDCCAgigAwIBAgIBBzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCAXDTA0MDIyOTA3MjE0MloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
"QS0xMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArauEFj2+dwpUC4aQ\n"
"sgn9qF/1p619hWj7/JuCdvX6ZEk5+7rmrBIy88Cr5bVzyaeXxPYyZ19icq0uc7QR\n"
"nB+/zp8RDhrWTdfYC0C0T6go8xpHs+5qEBG6ZzvzPMXlLopeAm7oXJ0wPd7434tp\n"
"sUDII0vxX25W27+UEekfexLtNcNoG2q+0tJ493LZdpWIgVhSHAVW7p1YLZQvCyYF\n"
"urb+o2svxmUlAU8PIo4dKNEtWRFaqDgbMw3SxxcDupt16TDaHGCodhL2vS3APjZl\n"
"VssGgJPOEYjLbTI5XPmN0N1IRHpoIdEUxOaBCBY1fQJuLgTPC3ML5iGrdqGLTK24\n"
"XM0xvQIDAQABo4GEMIGBMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0eAQH/BBMwEaAP\n"
"MA2CC2V4YW1wbGUub3JnMA8GA1UdDwEB/wQFAwMHBAAwHQYDVR0OBBYEFM2roPUn\n"
"RWzJSXG+5jGoBJDX5ZshMB8GA1UdIwQYMBaAFM2roPUnRWzJSXG+5jGoBJDX5Zsh\n"
"MA0GCSqGSIb3DQEBCwUAA4IBAQBr5K+BzFT2skeVkZ0fR8g6KcevlvAP7sOm8Ktm\n"
"nINBFi9ercrD4XAqkRTCYxDZ/6D9s10lf+CW4QLw2GOQer1D+znBzZZCqFfZ2+uL\n"
"ibQ+pth5IgsSxkxctowv7x3Y4C95TNAcWs9bWIHKgLqXhotk6mccEvxkyaPIL5yX\n"
"U2bs+tNpaqiwdCNjTJCo1+xdQwWZfk/oOMBn6Fihuc6eGP6L6Zj2J7TqO2hafrs4\n"
"lg1rYDU+cTtDy/eVt2UoYXLDnIm2EPcnTAaR6K3cLlh1HbeaUTDU2nE04riE8ntA\n"
"Mpu+t5TUUq+Zcx17zn54/W5oLC0wCGzzyLhh3/mLh6RLDYub\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC3jCCAcagAwIBAgIBBzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCAXDTA0MDIyOTA3MjE0MloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
"QS0wMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEArauEFj2+dwpUC4aQ\n"
"sgn9qF/1p619hWj7/JuCdvX6ZEk5+7rmrBIy88Cr5bVzyaeXxPYyZ19icq0uc7QR\n"
"nB+/zp8RDhrWTdfYC0C0T6go8xpHs+5qEBG6ZzvzPMXlLopeAm7oXJ0wPd7434tp\n"
"sUDII0vxX25W27+UEekfexLtNcNoG2q+0tJ493LZdpWIgVhSHAVW7p1YLZQvCyYF\n"
"urb+o2svxmUlAU8PIo4dKNEtWRFaqDgbMw3SxxcDupt16TDaHGCodhL2vS3APjZl\n"
"VssGgJPOEYjLbTI5XPmN0N1IRHpoIdEUxOaBCBY1fQJuLgTPC3ML5iGrdqGLTK24\n"
"XM0xvQIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAw\n"
"HQYDVR0OBBYEFM2roPUnRWzJSXG+5jGoBJDX5ZshMA0GCSqGSIb3DQEBCwUAA4IB\n"
"AQBtm548YFAtyKUyUyfz/iDPwgFq3PtxN0gfA2ZgrHG0vXDq49eDv7sQueTjY21T\n"
"GGkgyEEOlcbPzDC7hMrfaYSEGwrwbTkrWnrqk0rorFjjkjlKkx/9EuDQ/gjyQGrn\n"
"EvnFWwmxq+vambeRGxPWFRR9r+ugbL/xciZfT6OyoU8SZY4EpFu3e4ydmskdkt7R\n"
"qRPOpaHBuKygw4Uq5mhNaZDMy3FjtP+wt84UOOZbjqFQ8K8AZdAaDp3TDBtzoHED\n"
"lgv+PQMzYNnSsPVdvBC2fkgIWzZlyjeGMMtkUp3E7FeyFU1302g3xINRBmUcoO0c\n"
"jo6K7k6FZy3n6DsJ0q/os2wA\n"
"-----END CERTIFICATE-----\n",
NULL
};

/* Name constraints: IPAddress_v4 in excluded range */
static const char *nc_bad5[] = {
  /* IPAddress: 203.0.113.10 */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEHDCCAoSgAwIBAgIMV3OYdiegYYrQTBXhMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
  "BgNVBAMTBENBLTEwIBcNMTYwNjI5MDk0NDIyWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
  "ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
  "AYEA3Gp+KHwqvpNlEVkKyW1etHhOCXJPde/OJ7INv5RAd2xH5pQJI/mY7ET8RAz3\n"
  "oh4t9Ev65yljuhE1Q5qDGkjZrAhSpY8OwIQ7ZKtxH6zdKTsegAr60cfShgk2bDef\n"
  "wxbz92UBccyqzTQqWdi0Xvt9nVOkNL6EJWJ9eOW7s5fQgeEBlHYIzZyZs8ndiYqw\n"
  "Hpy6gg4XG777A0pgq96o8ybNBIHALV9aq1UGLWXGvEbi+yc6pQMYCg7OMcZq9aVO\n"
  "KBlR6HIxEohhFIWctmQDwd6ey73YV5XrDTRB7Qpe66/pqygQZNqMSaa4KL4z9KcL\n"
  "YNUtIFC39ivNBGT1gM5V3c69mkR38kYwRy0UU7C84/8t45+e6ZmSqe63RKKioaJU\n"
  "LlTCgHXwI1oeYlt7Pnm0kaWgArDJxBk5n1YxEoqkVXJ+lFy/qIF67Dosv3bNh2PU\n"
  "iPpLmbuFmdqq+E9DENeqji21CcPkvY1ekdib4JLolNJCVaRLnXPc5o0UEZrhCqfq\n"
  "UYfBAgMBAAGjcjBwMAwGA1UdEwEB/wQCMAAwDwYDVR0RBAgwBocEywBxCjAPBgNV\n"
  "HQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBRcj6YutTxQa7DfoqMSjPPQgv+p0TAfBgNV\n"
  "HSMEGDAWgBSSabOPqlecJ2jDyxpE3oaYyIBWBDANBgkqhkiG9w0BAQsFAAOCAYEA\n"
  "OyCT0ywwikYVSKEK2npyzAx7mkbkfrAHmkcCnVEgRonXKIWOL8lbRQp0m85ynVea\n"
  "aTfAVc0SNc7DMPR4tw+1gCpgsqhGEuip2Ltol9bhUovzfIpveUdJgeCI/NHBPAHH\n"
  "kwYgua8dcOwjb2SDCieXvWlE5DZUHWHNYxgTKQVB/WfAjmQbuF5olusFc2UtSk3z\n"
  "cnpBDb0+jUHfqCo8On+Wk5cc76mCvrcl9DLRFvPWN6T5IhisGVA93M1lIMQ9J9jN\n"
  "Q8SxZ7OeNaOXhBX2j27KzV7IXy0vqajdQDui1XV3Cos+EUFAt4Es8gUg5I8gNETl\n"
  "DqGglGUz2fVnLAQLClnVkZ0ExF3kDao7JwU2nl2XbzPcwwGGqLf06s5sEWinAZ2x\n"
  "k4L6YBjhauxndIkk0567/+GVeUaJxtruYA9rn/vJO6kEZXUy9eY0BF3GXQ4CKYG/\n"
  "FmFqfCwQ71LshpMsKncFp66QESAwmq9qaUqXoWSYtRABJKb9gYH42Fx0EY7QFjhq\n"
  "-----END CERTIFICATE-----\n",
  /* Name Constraints (critical):
    Permitted: IPAddress: 203.0.113.0/24
    Excluded:  IPAddress: 203.0.113.0/26 */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEKzCCApOgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDYyOTA5NDQyMVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA33/Xbt7c8Y+UxoU3\n"
  "4O0f+OTJF59lqe4UJK9owqXpiyt4d4hWwZV9EkPT9ZaysPaL05+AXdfXsfUz3ClE\n"
  "RPvK1zjR7WBqhu7dQ8RGNomaP0gY7eVqa0/OcDe1Z4ogmf27WbAUVR4xDZP18Qqy\n"
  "HB9BDStOJ24wbVfYgbJq3djb886yHDch5/mtdI4/C6aZqayaLjmoNs7UovuRlQV/\n"
  "Tdid3rleUeoH9xFD3CeI+PZMIbhyQww66qMiCkrTtUN3EOtrsvWwuxUTH1dedEI3\n"
  "tboI8UO+e2VJsDjbdzGIV0+HnvwoPQHYRlrog4RW9pNSTst+DeA4XY8+/FVnfhdT\n"
  "vDdbcmit4BxdtZf9npP5iaIQTOHNntdAAPOFCoALiCMF0parEKNwNFHa98hXr+Nj\n"
  "u4/oQM57fMcOC1z4y1pt6HK0zzFNhwHUrZUlAgcnXJ9bW510m6vu2LEFgJb7mM3L\n"
  "ku+ddujNOoL0BxsaaUJCh2IP2xC0rXGi6tNwR1Bv5RBnenDxAgMBAAGjgY8wgYww\n"
  "DwYDVR0TAQH/BAUwAwEB/zAoBgNVHR4BAf8EHjAcoAwwCocIywBxAP///wChDDAK\n"
  "hwjLAHEA////wDAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBSSabOPqlecJ2jD\n"
  "yxpE3oaYyIBWBDAfBgNVHSMEGDAWgBTAJHQ/p0WPW1fW+sKB5zbefu6HDTANBgkq\n"
  "hkiG9w0BAQsFAAOCAYEAbPahfACWMab9ccrS+pBdz+4hiy+LmvOP4FRFDz/zOCsG\n"
  "iZc61cYPWmRYWGsyWz2CksTg+Ktx/85PPvKTmEctK9NSehUgKESur3CPTp32k8YZ\n"
  "4NMXYXtwNYLPe5F6uehdPst5h7lg3Rz2pPqKopQ8lfv2qYTjtNCJFZ9wCtijqWwJ\n"
  "gZA9kFfPlHeY7kdnTY/Uhw1UVC0ty6daWXkKq4fhiiCXqcIO81GQHwOYySQSJef+\n"
  "KQ3m7QHascLKSzAISbK9Ncy/C/fIcM7pQmODKPwY/K3V1CQky6jmoZQ5LUbpXEW3\n"
  "qOjCElV2WMnkj5WuxDUeoVEKtnFGXI0UY5EY1z89GP7BatbNKLyodfp9qgjuXgr9\n"
  "DacHAUWOH6sH/wg4d5pgySVJeos5l6DzwgkyECaAGFM2t08waO2Xk9SgpmfMCxCM\n"
  "Gop4d5bIEMC8qScQmA4S1wfuoQQRRZiy2A4QckXoJ1OxLj1XVgossO21h8f9uNFJ\n"
  "nfnYYSRbIp+X0mo7JjHO\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIID3jCCAkagAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDYyOTA5NDQyMVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAs4uT4SEkVU0hdD6p\n"
  "aAkXBPeVl2y37jNB88rwam6kbHKfTQ/JpTpiXULo/dCmw0fr8np/FfoMZTUk8DMm\n"
  "zIsPjgx0yDx/PeUeA1Jw+KA60rApvxhtHhw2XLkF9hkU3YVwdcKfe1Cy3ZUT+3Xr\n"
  "rfoLLyRTR8qhj2wS/ARTx989cGbOpMGgFQxOrFZhm8N2/ORYDYJQYNTeqjEmHsuu\n"
  "wR0lvKj81hVe3mu6OsK4THbVpfeNW52zWW1FXwE3xbEXuckhzs2ZAc7t7SWzqD0u\n"
  "8lVWb6vXI7LGv217EpIRwtqa9zUHXLt4UqV+kqiUxLUOjxfitJI8CPYGPhc7aocC\n"
  "popaFqZJcLYvSJJY6Ya2ihZ4VyXQoXD+N8nQsFIm6dernNYQH2rB8plknAWbQqUG\n"
  "bhjTKX/oF2LCk8ow0KumqQre1UgcPLNiLQRAhyVdXgCMQ1bd0mFrwXUT2m2X+0sn\n"
  "DmW1by2bH2cevKtVaVMHq1Y6K7mcMb7pyQak7gFYAdu3D1u5AgMBAAGjQzBBMA8G\n"
  "A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUwCR0P6dF\n"
  "j1tX1vrCgec23n7uhw0wDQYJKoZIhvcNAQELBQADggGBABfMsNqRJrgIsQdneT1B\n"
  "s6fXdAlPkvbRvnScn+i/OOQxolgDQwge9A+0JxqTMtvxs+X5ahi79pPzqqzH1iYV\n"
  "2eBY1JpmYbdOYBVoLUQG7GG6ZRJWcDtAjGhnxgoc+lxVePpSsiZWpXFzAYbV7jQa\n"
  "2BXvzAk+uuBCUqKhrkmGZ+ZdQ8SlDjrQyblhKLy3vPi2LhfeNwTWAv2Qx8V2XRyB\n"
  "BQSchuMBq19mSppzjN+e5ixiRoNx02fH8dfZ0bu4ONUZcoECwuCmKW6UchQTL8Yf\n"
  "GJH9beqEAHevvE8Qn5aZcpX/BPBpx5jzOFXAD4Oqg+tY9T/vfgdElD2fQOI0gJQV\n"
  "bzC83bbAkqrjeZXyn0srzegPRyVczuv26SYGjhBsLVk5pAt6Msfnu5g3MC3BjKgc\n"
  "/gYV2bFtj3NMu8bAPYkKE9G/+0PT/a3/B5k3iw9FMxuZD/vjYIdSo/oloM+Ht9WG\n"
  "EYaTPDOTm4sXdqMVlPIyhwxADNrPnlAke68rKN7H5ChozQ==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};


/* Different name constraint types (testing symmetry of merging) */
static const char *nc_bad6[] = {
  /* Subject alternative name (not critical):
    DNSname: ddd.com */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEHzCCAoegAwIBAgIMV5diNwZSNyHAsK8oMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
  "BgNVBAMTBENBLTIwIBcNMTYwNzI2MTMxNDMxWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
  "ETAPBgNVBAMTCHNlcnZlci0zMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
  "AYEAv27tPNVuhJu/74JaGtQH4s4ZWnIsvEiqVCUveI3YCZYNTeE0FC5fzyqcsx3Z\n"
  "K9R8WVlF7SCadNY8E7aTjFiO7x3vcHvwW6eWY7kb0ZwUlzrvaqXp8BlpwpWUP0wq\n"
  "da8TZ5S2qrYC/itHHr8Xwo3Ec3rwj8ZO/96JjDKUgbkQVGoXR85+oWdABTSZcpbL\n"
  "T4UwCkBeYbkOD2jejMlaOHy+128mYeqW8Nh+bhF9EFVWFWr9XBSpfpCguD4JvsAf\n"
  "1zFWmgZOIosLWoLg5O4Po2i4rZCQLGxSKanvHR7npfpqbUgGKHHIGVQNHeLWz4f6\n"
  "D/VwRyxiZ1+VhR9kRvcQ3SD86QAInoEPQf1F1QnFqq76mfSobjKhQC5utg6DdMvH\n"
  "GU+smT24nRtx0Q7nILWzDPr/DdvpVPpFOP58thPLB3PJnc/dobwadsi6IslvdZ+/\n"
  "K/e2Ec4G9sOTMN++0+WM60wPula7t5GTr6SAmxZbrYXZlQH0jy75DBxThO1fMDn5\n"
  "pMa3AgMBAAGjdTBzMAwGA1UdEwEB/wQCMAAwEgYDVR0RBAswCYIHZGRkLmNvbTAP\n"
  "BgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBQyjEcacSHAUrf91hhII8DfODB1ozAf\n"
  "BgNVHSMEGDAWgBR0larTRnVL1RZtrqpleTpk+hEkqzANBgkqhkiG9w0BAQsFAAOC\n"
  "AYEADwaR444xUTrdl/WYBrPF69SXw7Plw7d4fh/65417wiEr4mjX3pfH6owVKGgq\n"
  "dzBjtkq8+MKV5W11L/S4tpVtUJt3AVrOCQKUiQiHuB4LSKX99YBBlZ9GYSeRgp2n\n"
  "iPqaAlREizXBJyPAH6tpuEyUp0rwP5qFjDj+Ks32jJUmoiJ/rnBMFBIdlWHpan0a\n"
  "7lIQi1uKIzQBELHmb13Yk/K99ilYaT5V/hPFRI02Q6/nLCTyuL7jLBijB8o8nwzp\n"
  "crXGwROBXKdn8NE+SrLPALM1/Hms/KANzLI6CjjZrO88YyNvWjCs8K8/sags8XiJ\n"
  "bxEIM/sEMV0I28E3T7kDywdf8OUvADYQIJidsrwv+n+UcI3MTZjax/295afjgBYF\n"
  "Zn1wEB/N2t22XlXvkbefYRaYvv1rCb7WaSvvzUaNsq2CL0uG353roiuti++MM1Cf\n"
  "wraL3SaauiI5gIjF03wElMyuAnKctk5cig7YVrhWAUx6fP5IOQdczRhP5SeUQBNX\n"
  "sEnr\n"
  "-----END CERTIFICATE-----\n",
  /* Name Constraints (critical):
    Permitted: DNSname: ccc.com */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEGjCCAoKgAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
  "MCAXDTE2MDcyNjEzMTQzMFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAo48gh92ygu5fGfzP\n"
  "nVwd8wLN8+1+k2WKOWupimwHdNG34KGKABlnjEjTZWuzZD05xf4V879YO8fiEWCT\n"
  "Nty70MmFyGXp9iCQpjIyTHhPVIIXfHKH1FbMvTiCicTtgFlcxFS5XnlhfN9H3Js1\n"
  "o0IA7Zyoq28s/Jkv6+b2YEr9OyNc4tMu6vLmEEnaDYwa0uLr7Tut6bTt+46bP1//\n"
  "d75xRsidnAM2tbYrhJ7MgTVTPYt9ewr8JcL0LDb4wH26fitddGAKQUE3VCQ/anyi\n"
  "PeOjoKryRaAsMtzDTfBRAx9eKIkKSYdA9c/dQmPO33Gxp2XctllVwPav5PDRSCdO\n"
  "F0dUJ0fqSNm95rpwCHw1B/MfSug1ye2AG3F8L+NX8SAqOgFgeZmukscrty3mwqOH\n"
  "sQRnSuu1+x6jUypwGoNifWuAZjdUiNflC6JfXwuZ8lS8xdvEkFSyK52xkcrWdvFu\n"
  "FNh8WsnpWkbBwaY1Y0nD1RY1thgMcbcJWtOiUTDOJKtNbQP5AgMBAAGjfzB9MA8G\n"
  "A1UdEwEB/wQFMAMBAf8wGQYDVR0eAQH/BA8wDaALMAmCB2NjYy5jb20wDwYDVR0P\n"
  "AQH/BAUDAwcEADAdBgNVHQ4EFgQUdJWq00Z1S9UWba6qZXk6ZPoRJKswHwYDVR0j\n"
  "BBgwFoAUjuF/dTI8YtA8wD+BlmP2MTUZNTwwDQYJKoZIhvcNAQELBQADggGBAIe7\n"
  "7o7DdYEj1mj5HU38fJ+8Dbugya98LODWSXXDpyxmFdn/A/poARj4DovFwXJ/0K+I\n"
  "BQ3jDvoJu6sV3/V2WKL613euX7vIK1sahbzlVHtBWxUAvCyg05ZOFp59ZxahHLO7\n"
  "6BF1QQK/PKtABTijCtcsoEGU7Smntn1mu5pMXoUHb8Sm+kkR7Wwm/z93lnHASG5B\n"
  "uGPQaXnZ4cMFa7+4tBe8Dz4huSQObye+iHVnkizQmEighHVK+4qKExMOTwJB5ba0\n"
  "6K8mKY7YAGDGUgjJzXVmGJPp7mb74AEJiyGobYB+Om9kPkWyAZW09EYl9GnCDl7u\n"
  "kiI8IqfoA6yvxHerQTsjS55TXTPmOl9dvYpnF07QxK8UwtZvf55zniKHjsP7TBoe\n"
  "9IJt7xxMeYbv6eDbxIpUp8HiljTTxKPKFZaFEOCFBjk4G/yUQNNiNUYdvsRdhvq2\n"
  "v7o3zwatl2i87lXGD7C78o2jqZDHoZI49XKN6ZW6dw3tpnjrEJeom8MwJfjHww==\n"
  "-----END CERTIFICATE-----\n",
  /* Name Constraints (critical):
    Permitted: RFC822Name: eee.com */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEGjCCAoKgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDcyNjEzMTQyOVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAwoXjtDX6uN+SF7qF\n"
  "dEWvxYb9Q3gu0PT0N7a17FaystBIGmw5lT9PnMUJKHPxNwrn+QdD1ReEmf6K4D8q\n"
  "vQzIzbCSBIV8bBZLA1/iH/hRaYvCvQLIaSnDEiDkkswkOyu/k7RDdB5H8q8Ey5x+\n"
  "udlqQGDsjHVQCKeWfAHXl+NRQ7ONj69cVbI1qz4r1KhnVt9BsxSmXw5n2BBul3Ns\n"
  "BY6yD8WD7js2sGxSMuv8hjFozc1t2u6APmkEJwiGx0fe+0Wu2HHNOUu+oyeiuPLq\n"
  "6G/Nz1B15E15iY+Fi6g7Qa9k97UuLPiIzjqD4JtnSerBxhkossWa7KrOw7no5Ws+\n"
  "d1lpZ92zGt8kTpi7KxQIV/zxvaXVBKv3br/FmKYxswFrbkXoYaFXrdsPs49B9OyX\n"
  "WY1zMWhbU4uddPG964yxDz2oTehRjvvxF+1oSAaoLn4chlPpx8p3unj5Y/h8xr/+\n"
  "Ciwblu3LE/1boa1zS/wM89UwysakVsaAxO3XjZyJPAJsTndTAgMBAAGjfzB9MA8G\n"
  "A1UdEwEB/wQFMAMBAf8wGQYDVR0eAQH/BA8wDaALMAmBB2VlZS5jb20wDwYDVR0P\n"
  "AQH/BAUDAwcEADAdBgNVHQ4EFgQUjuF/dTI8YtA8wD+BlmP2MTUZNTwwHwYDVR0j\n"
  "BBgwFoAU12oooOGTdVttQK9BdMpkS9Ch5Y0wDQYJKoZIhvcNAQELBQADggGBAEN8\n"
  "7Uou6LgI5RKyRL8UB4pLs53+mkUPoXCkCbv9mdFAqoi9OP89ALrF11tlME9PQZFj\n"
  "90sxnK4S7ZTDBeS12AUlEEQLCTIrqTMVTyiiklS4bSck4TOGFCaQhrsh7ZliEqkm\n"
  "oVsxYs0DYW66lVam/+K+5i6mO7Z4HGo5GDbU8nhMNHlighURfmzJ10xh6Z5s6uSA\n"
  "K16OzIOiSqUXp+nT7/NmWSzVgegCqScXLHZ6VDHDzfjURHrqnNTGg4ByMJo1R9DF\n"
  "Ne6khju2qP5j71Fwxz3RyZwwdhrPN3QSmMGyuFK3GS5WOYTeg35/Fa54UGVCKSM9\n"
  "Dsdg2lHS4+Bxkcj6fSj++5RE2CILOP2vV08VdV5IVFNWSNXhNM4JPOaSjKzIXNuq\n"
  "VdXTXxYBn90WA4/WNvUiUaVSioLvf/8Fv57/5JVwTsqM/I5TKZm/sbVePdX/3G/O\n"
  "7AmDbp8t5zInW1EUev7ThIRWlr2Jr4NivzeHcjRwdQW0HT0kwJfJSTzH/4RzAg==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIID3jCCAkagAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDcyNjEzMTQyOVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAzipouUl1Uqxd0TpN\n"
  "JO8syBj5zG5Rasydf5ucdnu5P1gjRvoHulrlYy5/NUA+7N3QLN6fo4cjRZ890p18\n"
  "7xqp/EdPUaFhv8bgyiaB8pXuQWTldqcuPSwUzXBHAmqhQ5lE7oomyGA4nqEJreas\n"
  "ishVn8v1M3FS6woAiJAxP5vbUOMkoxmoGZy6KkHNGZI+GUSDkoZBWTod8fFSMDYg\n"
  "kQk8QAGctCAG6Ms9NWKQpNHbTmz9478AC4phZR5QzvstzD6WpEvnXTI7K3ByoSN2\n"
  "0OeQzl5u3HbNJYDydOZbYuSnbADGL/QpBqirmcMUi28lXuSJ6c6jdvnOODqbahvr\n"
  "zvJDw6FQtn3OMdZnnSvWhkg6SxvImi99ABpS7SLNHDxm6CWrvbuZVFR21Tl/uxZl\n"
  "4je2QW6WaFQRO2J7iVWY2YJtDHLAGA7esfofnvRNSHEh1e5n8G67X5xGGFIIvBu9\n"
  "YSFmwTYZcu+DOa5DOU4tAQtftuWcK3XdcRJBn0X0NbBG13YxAgMBAAGjQzBBMA8G\n"
  "A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU12oooOGT\n"
  "dVttQK9BdMpkS9Ch5Y0wDQYJKoZIhvcNAQELBQADggGBABVza/dPn0VSs8frs2nw\n"
  "Pd3QoclMKNJ+myDtn5P4q4X0MNKVUQqz7HZ2RPa4UH5PDRp1PbYP45z56wIoYfJb\n"
  "i/sdDax0Nd0w6rmpl+koP0m24p8asxt0KenGQiBYfeEy2pxFn5aEKoDnyNAdhZRQ\n"
  "2Ug+pp9OMJoTqAPcDGbooHs5j5wHcIfO0bRhl7/QnDc5HK6ll4SmOrfJ8LpkO0BY\n"
  "vQK9zWC5Y3zck5uz8tNSCiGgt6y8MZQisC3JYWkzXE9xOjNfUBN+KhdF8ETgaBMZ\n"
  "QxguU5VbKtzqG8cpl+6tgW/Otrarirov4EVGhn2B9gwy8cIBe5Q6hXhnzYCpJjVq\n"
  "NWGSJ8T0gD+KZt/zxIAWWaYdMUXR7nQ5S+gY3Kn/OoTBZMrKJ3LSEQW42USrQwAl\n"
  "p6YA+vX9EI4B1r5uGDoxklkxAAOrO4sddw7MuncjeQzahqbI4IQmxj+GfjTXZ5Df\n"
  "gmQvlqt9tOTE0uIbWQT+iaXo28yZ+D0ymU8QazjG7UX9dg==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* This chain has name constraints on the CA but the end certificate
 * has no name on CN or subjectAltname. */
static const char *nc_good0[] = {
    /* empty CN, empty SAN */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIDIjCCAgqgAwIBAgIMVRJquRMxIN9nRLG4MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
    "BgNVBAMTBENBLTAwIhgPMjAxNTAzMjUwNzU4NDlaGA85OTk5MTIzMTIzNTk1OVow\n"
    "EzERMA8GA1UEChMIc2VydmVyLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
    "AoIBAQDEiLbqiSgPhAlwQXdn6Z7EoYMRiNHX+6cnlI195lv5LPdupiZ7EPF6y8rs\n"
    "H/PyRwLhOuhfyAbBibBTWKt1n100UqAcr0Z/l2+zBorc/APhKrysDlWyYUjs/Ly1\n"
    "pNQ4V9POpuCqEcPsGWcSr3ULqdRH2PoguWoF0el59fOyioUV+PbKAHPxs342fqu0\n"
    "Plk3Bru4kf4R0U3L1r0DogFUYmjhGyhxKAezEikQGgrwlo30LD/31uuzJWs5x1+l\n"
    "sGOKHIfxiWlBS8781QKuIWHT8Z+qsnPXobQ4ss2jF4qHjwIesJr8vq0OP2mQ/Ilh\n"
    "WwcNJJtWwHE6O6Vj1kWUQ9kDuWQ3AgMBAAGjdjB0MAwGA1UdEwEB/wQCMAAwEwYD\n"
    "VR0lBAwwCgYIKwYBBQUHAwEwDwYDVR0PAQH/BAUDAwegADAdBgNVHQ4EFgQU8U1K\n"
    "y6Tq/tczkOCK7CZhUAiuUcwwHwYDVR0jBBgwFoAUvSbAZPs2w5eQBjoLYPh+uKyj\n"
    "ZzswDQYJKoZIhvcNAQELBQADggEBAG2H5HeItTKZVgNd2hFPfi0QUvheX8mjMniV\n"
    "uaS/1zDSvAJOJue2GEpiYeTLvjd4WA5J09ZRSAuVWR9hg44m17TF2pod3YFBfGwx\n"
    "8RkEp5W8LBdwPzcgbLVxkWmwZTo1v4Xv679uyVTeB306vfkTrsa0C82S5zJd1Wyt\n"
    "/bFaNnxxb6KzVhFEctaVPVZdwrj7Q6XASH1oCfe/l50UcUkK853cXhV3CNJ8OUOY\n"
    "h8O7FN/s5oZ23R9eX2D9mHJ/ccucv46ofAmQ9TjLZIACp89IfoMf61MOUG51BIlt\n"
    "t494m704KMI2Y4hci5fHZ4UbcykjNpwkAMnCuk0K8K4gAFT6SZ8=\n"
    "-----END CERTIFICATE-----\n",
    NULL,
    /* Name constraints (critical):
	Permitted: DNSname: example.com */
    "-----BEGIN CERTIFICATE-----\n"
    "MIIC/zCCAeegAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
    "MCIYDzIwMTUwMzI1MDc1ODQ5WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
    "BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDUEBLGhjEj8pLO\n"
    "cU3UVmC+FcN3OKXH5lqm4pdxP6rbk0C9WKgavGV9MfEali767BaLHaEzxdCpcbxr\n"
    "YdBKFcij7ucF9YqpoDD5HnMLhOEHKnQD0nk5wWYw7Q9ULy0wqEy15nfDdunDbYK5\n"
    "TG7K2nsKcyPkEs637bJPBSOVSpn7mT49OnpSNpZcD361SqRh+OY8Iorr1m9DsrfW\n"
    "8J9JCf4VlaL821PzoA/EEReabPI9TM10QYpN4J0JQsQnmuU/0WdaEJtq4pFZIZfJ\n"
    "WtGGS6GX5faOkyMj5SuunZIjLal/+GRSVk1m3vfmDUp4MVKAqfqQ068Ix/aLEipX\n"
    "8mNnyp95AgMBAAGjYjBgMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0eAQH/BBMwEaAP\n"
    "MA2CC2V4YW1wbGUuY29tMA8GA1UdDwEB/wQFAwMHBAAwHQYDVR0OBBYEFL0mwGT7\n"
    "NsOXkAY6C2D4friso2c7MA0GCSqGSIb3DQEBCwUAA4IBAQBSfcTha0878cAy3Peg\n"
    "V9z+5rNr7/3Awf5HRTRr2VAATloXJ7iyvuvWmPlIKek40W+Xh4aHjZLjdfuPWyeV\n"
    "EXAcEEwhxY4t8NRATzgBy8WyP64LRQnFsmo4p5VbT0ddcqUqwDDYgbNPdLmfLUwV\n"
    "JS5DNFSsDco5Ng1DoQCLoIkuLMwD7g7YfMyUq8HupEI9TuhcXC5FUZbt0KjkQk9c\n"
    "fbBiEvZcxlmLQRiC0sWFmuBteMyrnw3Y68jpl12ORyB/oVpCvXlYm4ViCCh5uyx3\n"
    "Ml+FbR8ws+dEvGKmer50Lfw6/WSyEb/zWlLUUqClbJChLVnGMjgvwUqrLSKUcUw6\n"
    "DsYI\n"
    "-----END CERTIFICATE-----\n",
    NULL
};

/* Name constraints: DNSname in permitted range, intermediate certs without constraints */
static const char *nc_good1[] = {
/* DNSname: www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDQDCCAiigAwIBAgIMU/xyoxPcYVSaqH7/MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTMwIhgPMjAxNDA4MjYxMTQyMjdaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEChMIc2VydmVyLTQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDkemVOFdbhBX1qwjxQHr3LmPktNEVBmXjrIvyp++dN7gCYzubnpiLcBE+B\n"
"S2b+ppxBYm9ynKijhGrO+lZPCQRXWmqUg4YDfvnEqM4n04dCE98jN4IhwvWZyP3p\n"
"+U8Ra9mVIBAY2MReo1dcJQHNmo560xzxioHsGNQHAfYgVRHiE5hIXchYbWCkBrKt\n"
"XOoSSTmfgCF3L22p6S1q143VoKUr/C9zqinZo6feGAiTprj6YH0tHswjGBbxTFLb\n"
"q3ThbGDR5FNYL5q0FvQRNbjoF4oFitZ3P1Qkrzq7VIJd9k8J1C3g/16U2dDTKqRX\n"
"ejX7maFZ6oRZJASsRSowEs4wTfRpAgMBAAGjgZMwgZAwDAYDVR0TAQH/BAIwADAa\n"
"BgNVHREEEzARgg93d3cuZXhhbXBsZS5jb20wEwYDVR0lBAwwCgYIKwYBBQUHAwEw\n"
"DwYDVR0PAQH/BAUDAwegADAdBgNVHQ4EFgQUAEYPmcA7S/KChiet+Z6+RRmogiww\n"
"HwYDVR0jBBgwFoAUjxZogHO3y4VdOLuibQHsQYdsGgwwDQYJKoZIhvcNAQELBQAD\n"
"ggEBABlA3npOWwl3eBycaLVOsmdPS+fUwhLnF8hxoyKpHe/33k1nIxd7iiqNZ3iw\n"
"6pAjnuRUCjajU+mlx6ekrmga8mpmeD6JH0I3lq+mrPeCeFXm8gc1yJpcFJ/C2l4o\n"
"+3HNY7RJKcfoQxIbiKOtZ6x9E0aYuk3s1Um3Pf8GLwENoou7Stg5qHsLbkN/GBuP\n"
"n3p/4iqik2k7VblldDe3oCob5vMp0qrAEhlNl2Fn65rcB4+bp1EiC1Z+y6X8DpRb\n"
"NomKUsOiGcbFjQ4ptT6fePmPHX1mgDCx+5/22cyBUYElefYP7Xzr+C8tqqO3JFKe\n"
"hqEmQRsll9bkqpu2dh83c3i9u4g=\n"
"-----END CERTIFICATE-----\n",
/* - */
"-----BEGIN CERTIFICATE-----\n"
"MIIDATCCAemgAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCIYDzIwMTQwODI2MTE0MjI3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC/4ofaL+ilmmM+\n"
"bGaFRy5GYQXtkD8sA3+/GWsunR928fQS68Zh6iWU+gPm52i7Gfbh7piKWA5Tb63w\n"
"unbS6dPsfPSvgRMZGKJpzxqVcBQAnTS4MuDPlXNg3K3HMyVtbxekII8jFeGEJuCL\n"
"mBMT4dI48IZRzj+2mir38w2cQPfomaKtjg2jMokG8Z9/4+SU9VJCcY1/yZk8fCbS\n"
"dBbwhnDq10yvhPCHgX6KMYmoJr28CYgH29Q9sDP1XN3VvAx5X+PtW/6pyF0U5E2e\n"
"gRzVv7Hr3FJKvytbNxRMCoy2YOyvsTP0fIhiXdtkulTKXyiq4cxA+aYByOu1FjU4\n"
"NicWbiZ/AgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUjxZogHO3y4VdOLuibQHsQYdsGgwwHwYDVR0jBBgwFoAUwAx0\n"
"aL2SrsoSZcZUuFlq0O17BSgwDQYJKoZIhvcNAQELBQADggEBAGQvj8SquT31w8JK\n"
"tHDL4hWOU0EwVwWl4aYsvP17WspiFIIHKApPFfQOD0/Wg9zB48ble5ZSwKA3Vc3B\n"
"DJgd77HgVAd/Nu1TS5TFDKhpuvFPJVpJ3cqt3pTsVGMzf6GRz5kG3Ly/pBgkqiMG\n"
"gv6vTlEvzNe4FcnhNBEaRKpK5Hc5+GnxtfVoki3tjG5u+oa9/OwzAT+7IOyiIKHw\n"
"7F4Cm56QAWMJgVNm329AjZrJLeNuKoQWGueNew4dOe/zlYEaVMG4So74twXQwIAB\n"
"Zko7+wk6eI4CkI4Zair36s1jLkCF8xnL8FExTT3sg6B6KBHaNUuwc67WPILVuFuc\n"
"VfVBOd8=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Permitted: DNSname: example.com
   Excluded:  DNSname: example.org
 */
"-----BEGIN CERTIFICATE-----\n"
"MIIDMzCCAhugAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCIYDzIwMTQwODI2MTE0MjI3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDIf3as4EONSgWu\n"
"Mbm9w3DbKd/su1UWlrYrcpVqmU3MKD5jXBxyoThSBWxmq1+wcNDmE1on6pHY1aad\n"
"k3188JKMC83wEcyQXaiH3DlTYFXXkkI+JJNUGlfAMSoXG248SpoCIOhCETUG03iP\n"
"Z3AZludaHYsv4akAh1Kl6qn66+bKM53l/YhoQDxhoGaYvO8ZSwKnx5DEiq447jpW\n"
"M+sUFe38RPaMjHpyc1GRctvQDzJGm+8ZRujYDH+fGNzVDDlRyRnsVanFGNdyfhmy\n"
"BN2D2+2VEvzAWlaGg2wQN8gF3+luavIVEgETXODZPa5FF7ulmQmhqGrZcw6WtDmY\n"
"hUbNmbL7AgMBAAGjgZUwgZIwDwYDVR0TAQH/BAUwAwEB/zAuBgNVHR4BAf8EJDAi\n"
"oA8wDYILZXhhbXBsZS5jb22hDzANggtleGFtcGxlLm9yZzAPBgNVHQ8BAf8EBQMD\n"
"BwQAMB0GA1UdDgQWBBTADHRovZKuyhJlxlS4WWrQ7XsFKDAfBgNVHSMEGDAWgBTg\n"
"+khaP8UOjcwSKVxgT+zhh0aWPDANBgkqhkiG9w0BAQsFAAOCAQEASq5yBiib8FPk\n"
"oRONZ4COgGqjXvigeOBRgbHf9AfagpoYDbOKDQS8Iwt9VHZfJxdcJ1OuM1aQqXlN\n"
"dUyf+JdR/24Nv1yrhL+dEfRGka6Db96YuPsbetVhNIiMm2teXDIPgGzAKuTm4xPA\n"
"6zyNVy5AwfDQ5hIZ+EUsfOoerIElNyAbh66law4MWuiv4oyX4u49m5lxLuL6mFpR\n"
"CIZYWjZMa0MJvWMKGm/AhpfEOkbT58Fg5YmxhnKMk6ps1eR6mh3NgH1IbUqvEYNC\n"
"eS42X3kAMxEDseBOMths0gxeLL+IHdQpXnAjZppW8zEIcN3yfknul35r8R6Qt9aK\n"
"q5+/m1ADBw==\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDATCCAemgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTE0MjI2WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDIe0eOnLaV750K\n"
"4+mVaAftRrJp8t68KJivcRFpkl0ucQs6gwNf9EsVkHineOR3RXypjJ7Hsv+4PIKp\n"
"BhEOTprYUKcBaxHK/NIezV6NrO1AwuD6MtJDQF9jGpSy0F3eRUoBCjVYhTl+JxcZ\n"
"hGHPJd8WMeanQWY4xG4gTwtpjF3tPU5+JGQwLk5SbcLicM2QMG3CapZinOGK3/XC\n"
"Fjsvf5ZhxnixayhfiX/n9BmeP1yxz7YORNYPlL8z1CcLZqJsyjZnNkVwNvl4ft9I\n"
"FOKBLoOTSGocHFIFXh5b50GG6QHgvN+TiAwdpfRTUskWVg8VVIh7ymgDoI2jQhk4\n"
"EeMaZHd/AgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQU4PpIWj/FDo3MEilcYE/s4YdGljwwHwYDVR0jBBgwFoAU6XJK\n"
"EOUYTuioWHG+1YBuz0yPFmowDQYJKoZIhvcNAQELBQADggEBAJOCrGvbeRqPj+uL\n"
"2FIfbkYZAx2nGl3RVv5ZK2YeDpU1udxLihc6Sr67OZbiA4QMKxwgI7pupuwXmyql\n"
"vs9dWnNpjzgfc0OqqzVdOFlfw8ew2DQb2sUXCcIkwqXb/pBQ9BvcgdDASu+rm74j\n"
"JWDZlhcqeVhZROKfpsjsl+lHgZ7kANwHtUJg/WvK8J971hgElqeBO1O97cGkw/in\n"
"e8ooK9Lxk3Td+WdI8C7juCYiwsGqFEKuj7b6937uzvpFmm1fYDdOHhTMcHTHIVTr\n"
"uxSSurQ4XSDF6Iuel3+IdpLL79UYJ7Cf4IhBWj0EloF6xWTA6nUYl3gzKpx1Tg1U\n"
"x2+26YY=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTE0MjI2WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCqLuVrTyiqz+Zs\n"
"9Qw5V2Z1y1YSWU6aRDMs+34rP2gwT41C69HBh2LLRS04iJUVQydwnEJukwKlTNRn\n"
"1lEpvWgtYmySWA2SyI4xkVzCXgwv0k7WyLwa39hfNY1rXAqhDTL8VO0nXxi8hCMW\n"
"ohaXcvsieglhN5uwu6voEdY3Gwtx4V8ysDJ2P9EBo49ZHdpBOv+3YLDxbWZuL/tI\n"
"nYkBUHHfWGhUHsRsu0EGob3SFnfiooCbE/vtmn9rUuBEQDqOjOg3el/aTPJzcMi/\n"
"RTz+8ho17ZrQRKHZGKWq9Skank+2X9FZoYKFCUlBm6RVud1R54QYZEIj7W9ujQLN\n"
"LJrcIwBDAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQU6XJKEOUYTuioWHG+1YBuz0yPFmowDQYJKoZIhvcNAQELBQAD\n"
"ggEBAEeXYGhZ8fWDpCGfSGEDX8FTqLwfDXxw18ZJjQJwus7bsJ9K/hAXnasXrn0f\n"
"TJ+uJi8muqzP1V376mSUzlwXIzLZCtbwRdDhJJYRrLvf5zfHxHeDgvDALn+1AduF\n"
"G/GzCVIFsYNSMdKGwNRp6Ucgl43BPZs6Swn2DXrxxW7Gng+8dvUS2XGLLdH6q1O3\n"
"U1EgJilng+VXx9Rg3yCs5xDiehASySsM6MN/+v+Ouf9lkoQCEgrtlW5Lb/neOBlA\n"
"aS8PPQuKkIEggNd8hW88YWQOJXMiCAgFppVp5B1Vbghn9IDJQISx/AXAoDXQvQfE\n"
"bdOzcKFyDuklHl2IQPnYTFxm/G8=\n"
"-----END CERTIFICATE-----\n",
NULL
};

/* Name constraints: IPAddress_v6 in permitted range, intermediate certs without constraints */
static const char *nc_good2[] = {
  /* IPAddress: 2001:db8:4000:: */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEKDCCApCgAwIBAgIMV3OWtCJqV9nu6MtYMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
  "BgNVBAMTBENBLTMwIBcNMTYwNjI5MDkzNjUyWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
  "ETAPBgNVBAMTCHNlcnZlci00MIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
  "AYEA34kB6fm9NLwz3CGzya+pVo6qWXVxkaXiKnKYk7nzcT5nozZYERR8gd3FWmAX\n"
  "T1lzNC92Yxbz7zvonD8cxmRqMNGXqNrHrGmO9u0pr8eFnIRhSEscSWv3gwoMKH+C\n"
  "+FajvWM9XK7+ndiXLL4ReFtrPcryrN3B0duMX+l2vgIBlBsIoj6m/n6ExQ8uyE3a\n"
  "VyzM2gZgvjX3hw3ieLcOURRVRSsINlNcDZeUB/oGNkWKECUAEbmarutqeoYDZOyI\n"
  "LBxcFhZ3/l2khMHuU5G/uQlBVuR45LqEk2LkRtG3MiidmFOnvHbFwFucT1JexwAs\n"
  "5YAeBVfIkO/ZaHTnfL4d/z2GKniNyCmbwQ6kBqG8kK/EGWkpPwIHu8KRap8LmrrS\n"
  "YY2pRT7L5UCmsFsWyTm3N4n6QYImCnn8h9IY4zKtQfzfbh10wWgd4tqtJZELQjgL\n"
  "DvxNsv443bqJ1vWvwmV9X8O0G4nSjcMsgQQCPYWTfnNpcVVOa80n2p23xyG58hdl\n"
  "hQ0HAgMBAAGjfjB8MAwGA1UdEwEB/wQCMAAwGwYDVR0RBBQwEocQIAENuEAAAAAA\n"
  "AAAAAAAAADAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBSRsZYeRAUgOYrmLOR7\n"
  "W7JTQVTJ5TAfBgNVHSMEGDAWgBQVna/6ANq7at0JSC476WWFVlOauzANBgkqhkiG\n"
  "9w0BAQsFAAOCAYEAZ/NaFq9GGqsXyIB8Lgmf8GqmHCq5EC3qPijOf34McVtAfZU8\n"
  "0q2ZNkBlV+P14n4DUdNKoLwjZ9jL33IiyRWPNdz2SgqOgu0rdN2xXc/Hq5Wu/bBw\n"
  "ZtASxQBV1crYvtoRCTBmei8j+WJ1Qfh+1QNar1mgofCR99Sqx+x0HWC9nAF6aAMd\n"
  "6t4GS+E3Cmpu/MzgCHHr7yT0XkltJ7m/oLjSwR4RFepNAc5zMOD2ujxwB+5t+mnE\n"
  "fO5i3SF8OZEnq0c8LtiZvn0bbVtRMcRFpi5kLzV+5b4J4y2BocR8cSvbp/GOvyYi\n"
  "AYKfwboXK9nJmaxsYdT9zKSp5sPETMMXD7ZDtaM0jKunb9sUF98FsK1j1I416cQI\n"
  "ChBUCeANDhHTAmaxPKpyjnHsEtbA27z5l5bQmUXSTc5vVYEj4HhNAE89T/4AbMbH\n"
  "6hvlsD0t9cq9nWly0CC7UIoI1Llv7TVgIKVieLQ5DIZHGL/VuPFHNqfFZDzPByjf\n"
  "kU9hILqMbOM6P182\n"
  "-----END CERTIFICATE-----\n",
  /* (no name constraints set) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIID/zCCAmegAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
  "MCAXDTE2MDYyOTA5MzY1MloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0zMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAxCKO/30Vk2JmTvrV\n"
  "j/yi2Bi2rx+SQbIwJRmpGXedQfO90q9EXXZ7INQzo+xhk3dsuzkgTxeVnf7ONJwh\n"
  "r6KoEgoUXSH+R2MaX1+LjrD2MU3zDO11DE+nAv8hgOZJqYawmgve9RMsVREUcnCu\n"
  "85pT7bIsNqtuaXG7/mkMib1YfHwiMCl3u0jwhTWf0FuSRqe9ozqHo8DR89vHyrjl\n"
  "t7FsrUnECJLZtDlf0VUG0lekZIY+WL7w54j4C9z+e8ZnKPpGIJyTgS4W/yXslyNR\n"
  "VECiQHKW/gwSJBDMLODesJsSpgwh2NZ7VmreWTGSoFgS84Hteogj8jY0C7Ky6DHo\n"
  "kYLpHcWo3FBIZ6oSiWR49zcJ1r26+JnHQEE1Kxt9Rpn6m6E2k4yW6hBFIZcWnFSQ\n"
  "LKPtHMQS+soqC/qj4fFNqm7/OJof33LAG6T3cC8wtgtom7n9Jd1RZvkGtlAbNzzm\n"
  "Wwh4SUyMyq9/dQ3WkJ1RVzVLhqYi9+QiHuVrqVSFO8dmL1/zAgMBAAGjZDBiMA8G\n"
  "A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUFZ2v+gDa\n"
  "u2rdCUguO+llhVZTmrswHwYDVR0jBBgwFoAU2X5m+TBOqGUVhmhVmzQagc4mEqEw\n"
  "DQYJKoZIhvcNAQELBQADggGBAHbh1c2UYg23aHZN/cVkAD6fqOykWqF9ZeqI9r0c\n"
  "BGpvnhcVMOdGitpIYk3cW5a2UYmo6SkfOHos6yB3fZKHz525YhS7Tg96t+c4es8D\n"
  "GlbJdr/O2vwPVsBseyvk+AHPBrcJDooiqD1rXeJWUiIREVBC1hICYaEDTSdPKxRX\n"
  "93sRnfEFpZMcWd2dsqOUwwwN6VIeZAxoeysM2O6qkqDIjIVOK613oMYUEKBSyZ+w\n"
  "d9Ds5sStkaLXxNJ06q3Mst5rE+IgpznpYvJBtpp6HQQeqiTXI9lIoU2oZda8UChA\n"
  "Tc6iNFl+oQVvaMMUo2YlbEKbQ4UPbxT6wx5LfF3imbReMtRQCbs7uvCkTNTkY2mf\n"
  "LFTMPMBjbLaY5ogx3vRZQd3833vC9iUcgBewyJc81BcEzI6F9rcg9quzkAnXdUsV\n"
  "zcMfadJlDrnPm/n3mNiHZs70MQ/dXQtbaD5H6T9BME5sRwAmW7VJ/ySeytkoUw8z\n"
  "leNeFV8T+J9lz0g5hWY78QJaTQ==\n"
  "-----END CERTIFICATE-----\n",
  /* Name Constraints (critical):
    Permitted: IPAddress: 2001:db8::/32
    Excluded:  IPAddress: 2001:db8::/34 */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEWzCCAsOgAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
  "MCAXDTE2MDYyOTA5MzY1MVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAq03TF1WvguiqwoZG\n"
  "XbM/lSligqO1tCd8dhAsa1lZmuHqcZRsQLs4Dq4Ffh00AVWDnn734hGzyvBA2LNj\n"
  "tSH+slECZ55lVWKvJ0D8ip481YFP4CTWcJbM4cvkZdM82+ygYERy/WfR9cJqyrLX\n"
  "tYvvs1b15fb7s0alE0gJK4j0RbzDjp1DsHFKzG2bNJxJ39xnfa6h4zwDJ+NgJAQN\n"
  "z84OEw5ECZlKp2HbZZdTx0rXFYiyucao/Ugs1rZ3SPzcPg+EJJMSTdxypD8qWaGg\n"
  "xP0UrmYxl0D9+m+pV8YftD3h9yFDB0DC0eaXyST224mJDUYR4E6tBSFetWgLkc6l\n"
  "+1os8Ys6SvrlKvtxr0xQFxV1LhMX/gZgntyflljj/DWYpo+uaA74bkhOzIxEpa1Z\n"
  "BXMLauKJ7dzm2aHYIgFlYxu1TIjib0D/UaEp3wmoZ6pDUpxjoiqjfDc6WxV5b2Gj\n"
  "TIZ6qwLcADF90estBeLEtkcf8xk71JzFe0FGL4bDkPPKax0jAgMBAAGjgb8wgbww\n"
  "DwYDVR0TAQH/BAUwAwEB/zBYBgNVHR4BAf8ETjBMoCQwIocgIAENuAAAAAAAAAAA\n"
  "AAAAAP////8AAAAAAAAAAAAAAAChJDAihyAgAQ24AAAAAAAAAAAAAAAA/////8AA\n"
  "AAAAAAAAAAAAADAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBTZfmb5ME6oZRWG\n"
  "aFWbNBqBziYSoTAfBgNVHSMEGDAWgBQZXWHEGPF89Ep5BX76GGJZxvxVQDANBgkq\n"
  "hkiG9w0BAQsFAAOCAYEAl5x0spmwJ0kKiVOLu8WRRtHb6DK6pSu+jGxGh4GNwCFw\n"
  "bvX8u6QBlCu9xW4afd6/a1PduPtoRQltWeZaB2SDWnnjclKpaG8A9736YV9XKHdL\n"
  "QX6GZcKSa2r81aAaHSZqxo60HfMPbCLWiWwWDX6O284kLumq7m2Z+pTKmb/Fmdqc\n"
  "i292pyamXuj8eMsYNGvxzknwe3jr4HZhNfdjRvsLTI6ovEGsa6tdIgszSOrLT/kh\n"
  "yu8zt9gljas6aBJ2rzT1OECaHQ74IkVZkhA6C4tSf2grH4yDZ4oZrcgJFHF/saC6\n"
  "5uj5niAxmJrlaBeb+dwl+c5aNFo51zZRYktoQuvCGykWwqc8XwZBKu8MGNsEjEo7\n"
  "wtfgu/bzXrLUJluXnzVBNCcXwDzsgIxlpJcFZ5aqaVhfYrl8cd5Wa3FrkHvkGyBP\n"
  "aXS1nd/tvl96i6p60w/VkX6FlSknXh8IdkjcChckJv3AukUrV0U1ViTaVkWpVhQL\n"
  "JKM8n9POeVPK4pUlaXAd\n"
  "-----END CERTIFICATE-----\n",
  /* (no name constraints set) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIID/zCCAmegAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDYyOTA5MzY1MVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAnf+axUDkYDVQNCvR\n"
  "hvLcW2nWKZo3g2JciS/h+V2gKWjDDNWxg6luyxlCFsDCaNx+t3j2YJECRZvNMLDP\n"
  "QiE7U8+GKtJgR3FhHnA5xu/IbBWbNYUO08Q1s+w82dfy6QxRRDOXXh0fjAbd153J\n"
  "K/9FDpyCuJ/RC9RKphwOz5YsSlMbUpYCCm0pten44HzyT/F2hwVLEEnLn1uuM8x2\n"
  "HggBYH6WHFiGJLZH6h9ajt04JjyvcTDIaGw4ttET1doBnnRU+6CDiFYeYFqzUDkG\n"
  "4lKKPxyXsqC0B20vV/N03c5DnODkMnTGPQ696HuhNtf4+i6PRhTieX2iu8uDOPcF\n"
  "aOfokwfuUx3Ws4dShfvSMN/jFpdpOVn41dceY1Dbqy3tMF0YVFY6SSvRtvUQAvmz\n"
  "KYmaYwKKpe/yMDqICAdE3fjkHHkQeQri6FP7RrLfbsprLtQlS/1ZclT9CnIz3uES\n"
  "7C4a9OVvxIz+RZoIps/q94PB5fqvzXsmDIukV9VFKpAjrXWVAgMBAAGjZDBiMA8G\n"
  "A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUGV1hxBjx\n"
  "fPRKeQV++hhiWcb8VUAwHwYDVR0jBBgwFoAU5xR9/WMP7pTmHW6KSB7HSeK6exkw\n"
  "DQYJKoZIhvcNAQELBQADggGBABBXBE8psujTJ83QlsnyQYUk2MPfvm5CuJGuC4rr\n"
  "uc3FFNi1A9h1XR44J5iDf84SZcaNZgYag79SHaQ0OdHHYPdJ6Yc+59IIf8PncoGd\n"
  "wgeU61u8uIZL53yfDSE7o0t0UrJpBNb7oGF2adRJs7ZqyUZyN6A9+74jcrxllpjI\n"
  "oha9WgugzP9CUqWYgv/MDNuwAtv+1znNFgv8C8mkhbr8wmf737XsQzfrYFK1ibhw\n"
  "8rBGJLnLT+Xh+CgaDYeZrnS/oSMXCJETTbnPWfJGacX6FtB829fYhO6VMDqTy74p\n"
  "k04UcXRxH7ZnRXbCImqnGTfNZtXYLu1oCDC/Ubi08ev2r8Lxrg8B/F6ME0hihiHh\n"
  "8X1ggr2fDll2mUwBnalRgRYfYc3MRfQv1oy/lQDp0A2rpbKQiX8ji8r1y9xDcu/B\n"
  "ERwaNiUxyK+Tx3BiMIN4EP7+yGh/f71PDvlZYBkXNYrQBaCpW8fj2hr6N/B4kCPr\n"
  "/AJhLcEQybcD4IKiqywx8kx3Ig==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIID3jCCAkagAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MCAXDTE2MDYyOTA5MzY1MFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
  "QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEApAgApH8PdlQKWr+K\n"
  "ja6KYXYYNxdhs4JoiOBxr3OQuS/ocqp1HU5vVRXKiu0XpqkB631wmC1tTk7N4LYZ\n"
  "+4ffM82JhkK0y1FwV4soqeuwYraF//8ekgmCxi7tqie0WDmXc+aO2O8pruFHTjCV\n"
  "TCMAer+wspFEQgHVsbFRAr+zyZpz8bn3Ywy801aM+807sEyNeeIMR0UnM5uELKvF\n"
  "iHXJwdPdXM48sGe6DHJtSw4OLx3+xqRMlhVBAb0/yYLv1HDFwul0IRBfjj96rXgc\n"
  "bWLiKIjZCl64+Y+UbbHp71pT09T/tzu15tvjHoIGLudWIaZHCnU10fQS1ySL/Xjm\n"
  "n0xxze2AVSzoYoiw3ldkTvik9gqESC+uu+QXhzhxKe+GKEd6oGE+8KOxAkTJT6BO\n"
  "vXKa6R7XDgI4AXeNlTgOZXDAqmhPFjARaLS8jdUIKJLHSaqlzX0+XLilgTGMNaXA\n"
  "4Sm5pKnJcoCpL4OisgiJnuIRshO10IKgM5YqeytbkjeJzqX9AgMBAAGjQzBBMA8G\n"
  "A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU5xR9/WMP\n"
  "7pTmHW6KSB7HSeK6exkwDQYJKoZIhvcNAQELBQADggGBAARJRZIqmdoMRq3iYxh9\n"
  "vApIvuQ8cImjusmIaYzZpAvjcOiX5RGuN/69e7Os6QxnN+H6TAj3IX+a0Msu+P/c\n"
  "NrrQWttd+uR2xZic+dNXzhsEI5+o4G7W9srnDLU01FmlytvH1CSgLYm8uv3Q8G3/\n"
  "RVEYmOphvHUUDYJeFIUDyaC88k52tyZ1SeSkveLRy+vf7GkHicVVMAOuyiQV3aQD\n"
  "M8o3QFyrncw5i608d8JArJZ7LXhx+S+37rMsBGHnXAyKjv8zNt/YW3IjAA4ifr+m\n"
  "rbTurPyCNxKFdhTBQaF1ofQaKVNEIdSjNCB+5RXUXmoAELsiRQS4LGN6NhMfBlbu\n"
  "YAMUmDjwu/LkDGLbZHX0cGiDuLc/qefVc1QBAQZ2zoCAnYZU6itnzqlayOijNj8n\n"
  "0aHMa8P8rb9gzOKcNOz147lLK5oHjYgeYOy3hpUDT/k7wyELWb20GKaaWLgGqWS0\n"
  "W5U6UgHQoBLyOvHIOkbCRVyIPLh9ijufA0LpUdk2Lf1Sww==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *v1_intermed_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC3jCCAcagAwIBAgIBATANBgkqhkiG9w0BAQsFADAOMQwwCgYDVQQDEwNDQTIw\n"
  "IhgPMjAxNDAyMTIxNDU0NTJaGA85OTk5MTIzMTIzNTk1OVowDjEMMAoGA1UEAxMD\n"
  "Q0EyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAwZPztwmz136OBSTT\n"
  "t4pXys9dTfaOBajrO2s4JcPa0C+7D2wfWD941q1q01TV5+QgLDTF7OO5VSXt2W3p\n"
  "cKlXS/Ll2N8sxTaULcVCodFrHOdfHV1V65VlWhJOnPdpboBtM2V8Iory+d2mNXZT\n"
  "wkcNJ/Z8YBUZCTeR1zaLjq0GAITyJoMiI4+x9Djc+iBDGJarRW7A/JyDN4EFjDzw\n"
  "svdWpHg710I+qtKnlMO/whEmw9r3L486JTSlrrrruUSVGY9UWJpv62az1jbu63d8\n"
  "6/PBp0xbBpiv1xA0qSSquN/THurTZ0Y0MS0vbpnAYkws8YxnFAV1TU4B7AZ0IQId\n"
  "Zjo6HQIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAw\n"
  "HQYDVR0OBBYEFNVrkTf8WoQKImozc6dBqx8J8tvhMA0GCSqGSIb3DQEBCwUAA4IB\n"
  "AQBpX+j6Fd5aLnMs55qqbFBTWU9bH4/fGh6OVcJztZyvRTOQMLoha9Lsa2C1c1u0\n"
  "kjj3coRWIq8YH8FbOhu0x3pij5dcnn1FQCKcwEmjdDf6ltxplkZXpR86yW2ZyR2W\n"
  "WmIPUrMPJNFkBbgVKFyYoj+9QUyoWHAWNSLJhqBI5v5CRNYIIat1Nt5SuTDm3ggw\n"
  "GUfMH/snytxVq23tj+02pBCdahTqN1w83W1yFX39URChPpl9RZ6HcIg3DFrXhXte\n"
  "lA+/t8l+o7w7POJ4xMyRtbTuGpGHQac+VJBWKFkduY3sbXN2GdQPL6/VvKH115Tr\n"
  "Bos85afmGYPR/gUP0hVSlFzj\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *v1_root_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDAjCCAeqgAwIBAgIMVDP8wwGyCHAlXREsMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
  "BgNVBAMTBENBLTAwIhgPMjAxNDEwMDcxNDQ2MjdaGA85OTk5MTIzMTIzNTk1OVow\n"
  "EzERMA8GA1UEAxMIc2VydmVyLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
  "AoIBAQCp7I46Ga7QxXCq6SegwcKCbnMeIG2P5IaJXknjBy4rq7P3dqEdEmm/UdwS\n"
  "Wp2jH+k+OFvzFe2FR1fY7UBSFdpLTaMz8YIQ1ESPu2afAxWCE1drJnphVCZyMskp\n"
  "d9P9p+TXE4Y7ppxPRTvp2D/chfIcByIKPKwsjU37hrgFehb7Jolq3Er0pOPitSPj\n"
  "KFVKNAktu8Z411S1hQdO7+jjr8pbFoROm2VDbYRpowCHw9ZhlC51SFKeqPTslUdv\n"
  "53pmq1p02d3WMvKWuFRAIMs1UPba5prN9UI7jZztR0o7xnHm3KtWk+o4+YSDCM5R\n"
  "EoiGvyhKgDoM3B/KQG+1rin2/uMdAgMBAAGjVjBUMAwGA1UdEwEB/wQCMAAwFAYD\n"
  "VR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFDoL\n"
  "zL5Pd48Psh5oe9VGrkmXclumMA0GCSqGSIb3DQEBCwUAA4IBAQA21lKw/Vr7P4hl\n"
  "VBY27GOGIh7Tw0dFN8HiYX0jFRlyn7zgoRYfJMuLxtC+jJl02s+iljl9gdTlcAgt\n"
  "dWs74QdKUIMHfoT12WOcIwAIMZLBspbUjn6+eoVPE6zCOfrChRCv4dM4BCz/kg6w\n"
  "MqxM/UE+OS+AgO8hHN1boMbBWMcMR/ylpJE2P3nHGgTg6xsZrEn9aH8y+uqUuScn\n"
  "P13H74zV1f1tTi2QT7Y/lpNKuB8vvqJgcUdxGaVQhz3q+YooSwmkDaWTz9HIAqpr\n"
  "77BBxvr5eok6o528h4qJ5nA3NgLw8nIgyBP2vN/N4CsmRAZ/vHVbGEOeWRkH0rGE\n"
  "SemQfe/B\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICljCCAX4CAQAwDQYJKoZIhvcNAQELBQAwDzENMAsGA1UEAxMEQ0EtMDAiGA8y\n"
  "MDE0MTAwNzE0NDYyNloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAnTMg33+4Q3ULz+//A9Wn\n"
  "m0DCq9X1ne0q8TfCriOo4Zc2VuFsR3Bm90btX9KqVEfByD8Ba4fb/oF0F6+sb4Ej\n"
  "imPI6PSwHXEX0BPSHcmv6lb+iXHRwpqsx+r9GIuPS+0vuTu07sj1yjszlx7aNXOx\n"
  "hsDAFaedzO8/9nCjbrQ79cLcGusPETjGFAD9vDfBTUNtebPiW9CCNsmRUVWOm6/s\n"
  "5kfy1AfcH6FZCoqvmoELz55JWzYHrWSQhgIopJ0DLfYoF8fg3XeIMUNLk+Lrcoe6\n"
  "ZDCxDxGwXReNfIWnf1l1OnZAdMAt/egr9jeQR+l9y4jBhcssW7Wb8M6+wvinsMOG\n"
  "MQIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQBosiz3l31++ZErMs09bnGe+DEQfL+t\n"
  "iWRmhzLC7kF64+O4Eu+IkWvEUv+LEhHm2GLrqaKu7FAlWUMWRX3GHHxAOtNeLR7U\n"
  "qXBZtq56wHA+fMv8+lqLQuT6eKLNbbuxVWZ/E1qaMax7rlQrtEpAC9ruuafcPlC5\n"
  "U4YbJ4VOhfBnJzQ6KlFtbqOkGr7v/l2d9NRxorAWawPVhIteZv1Ahiu++5g+dn/R\n"
  "z8ehN9SEm+c6C5mWrqHiQka3yi060gO8kBcumM/cE6BxffiOUxy2gsPC2ZrI8xkB\n"
  "ghrpQ87AmWXsvVk03U1l0vHpiE3kXb5FIAbWW7In1mfULqLKgeqllMhD\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *pathlen_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC6DCCAdCgAwIBAgIIUvuTdCOiZ3IwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0EyMCIYDzIwMTQwMjEyMTUyOTU3WhgPOTk5OTEyMzEyMzU5NTlaMA4xDDAK\n"
  "BgNVBAMTA0NBMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMGT87cJ\n"
  "s9d+jgUk07eKV8rPXU32jgWo6ztrOCXD2tAvuw9sH1g/eNatatNU1efkICw0xezj\n"
  "uVUl7dlt6XCpV0vy5djfLMU2lC3FQqHRaxznXx1dVeuVZVoSTpz3aW6AbTNlfCKK\n"
  "8vndpjV2U8JHDSf2fGAVGQk3kdc2i46tBgCE8iaDIiOPsfQ43PogQxiWq0VuwPyc\n"
  "gzeBBYw88LL3VqR4O9dCPqrSp5TDv8IRJsPa9y+POiU0pa6667lElRmPVFiab+tm\n"
  "s9Y27ut3fOvzwadMWwaYr9cQNKkkqrjf0x7q02dGNDEtL26ZwGJMLPGMZxQFdU1O\n"
  "AewGdCECHWY6Oh0CAwEAAaNGMEQwEgYDVR0TAQH/BAgwBgEB/wIBADAPBgNVHQ8B\n"
  "Af8EBQMDBwQAMB0GA1UdDgQWBBTVa5E3/FqECiJqM3OnQasfCfLb4TANBgkqhkiG\n"
  "9w0BAQsFAAOCAQEAtQudk32tQ30ldwLy5QyNzwpxTq1izycXGMkh3LvNUQrxmwzl\n"
  "8EPi1d4bxdAi3ghwppImJPZ1aWOrSl9cxl7kH4clq/QdG6bKhhr/40ImZctV35fA\n"
  "Kd1/aDlUUNQIv7cD/T8fb8rMmZ7RPoLsgLcVfodKwafY+X/y4ZacA2uF2L2dX37T\n"
  "etQprA+hjeKu6rej9eb+ERZqYChDvp7FNbJ5fOnIZ9iG1Z714fUeuRDzvosJl6n8\n"
  "aVIRHXdZbhCgKdJTR4bvFPGVFL86xLMkV7jhCImNBN9rmd59wD6g79nTUUoPDM3r\n"
  "rpNkoLGmlBhUorRWbx0YAz9UojNdd4GWMefwZw==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *cve_2014_0092_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDtDCCAmygAwIBAgIETeC0yjANBgkqhkiG9w0BAQsFADAZMRcwFQYDVQQDEw5H\n"
  "bnVUTFMgVGVzdCBDQTAeFw0xMTA1MjgwODM5MzlaFw0zODEwMTIwODM5NDBaMC8x\n"
  "LTArBgNVBAMTJEdudUFBQSBUZXN0IFNlcnZlciAoUlNBIGNlcnRpZmljYXRlKTCC\n"
  "AVIwDQYJKoZIhvcNAQEBBQADggE/ADCCAToCggExALRrJ5glr8H/HsqwfvTYvO1D\n"
  "hmdUXdq0HsKQX4M8AhH8E3KFsoikZUELdl8jvoqf/nlLczsux0s8vxbJl1U1F/Oh\n"
  "ckswwuAnlBLzVgDmzoJLEV2kHpv6+rkbKk0Ytbql5gzHqKihbaqIhNyWDrJsHDWq\n"
  "58eUPfnVx8KiDUuzbnr3CF/FCc0Vkxr3mN8qTGaJJO0f0BZjgWWlWDuhzSVim5mB\n"
  "VAgXGOx8LwiiOyhXMp0XRwqG+2KxQZnm+96o6iB+8xvuuuqaIWQpkvKtc+UZBZ03\n"
  "U+IRnxhfIrriiw0AjJ4vp4c9QL5KoqWSCAwuYcBYfJqZ4dasgzklzz4b7eujbZ3L\n"
  "xTjewcdumzQUvjA+gpAeuUqaduTvMwxGojFy9sNhC/iqZ4n0peV2N6Epn4B5qnUC\n"
  "AwEAAaOBjTCBijAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAT\n"
  "BgNVHSUEDDAKBggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBR2\n"
  "B1hM6rUp9S2ABoyDSoINCeyT3jAfBgNVHSMEGDAWgBRNVrdqAFjxZ5L0pnVVG45T\n"
  "AQPvzzANBgkqhkiG9w0BAQsFAAOCATEBdNWmTsh5uIfngyhOWwm7pK2+vgUMY8nH\n"
  "gMoMFHt0yuxuImcUMXu3LRS1dZSoCJACBpTFGi/Dg2U0qvOHQcEmc3OwNqHB90R3\n"
  "LG5jUSCtq/bYW7h/6Gd9KeWCgZczaHbQ9IPTjLH1dLswVPt+fXKB6Eh0ggSrGATE\n"
  "/wRZT/XgDCW8t4C+2+TmJ8ZEzvU87KAPQ9rUBS1+p3EUAR/FfMApApsEig1IZ+ZD\n"
  "5joaGBW7zh1H0B9mEKidRvD7yuRJyzAcvD25nT15NLW0QR3dEeXosLc720xxJl1h\n"
  "h8NJ7YOvn323mOjR9er4i4D6iJlXmJ8tvN9vakCankWvBzb7plFn2sfMQqICFpRc\n"
  "w075D8hdQxfpGffL2tEeKSgjyNHXS7x3dFhUpN3IQjUi2x4f2e/ZXg==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDUDCCAgigAwIBAgIBADANBgkqhkiG9w0BAQsFADAZMRcwFQYDVQQDEw5HbnVU\n"
  "TFMgVGVzdCBDQTAeFw0xMTA1MjgwODM2MzBaFw0zODEwMTIwODM2MzNaMBkxFzAV\n"
  "BgNVBAMTDkdudVRMUyBUZXN0IENBMIIBUjANBgkqhkiG9w0BAQEFAAOCAT8AMIIB\n"
  "OgKCATEAnORCsX1unl//fy2d1054XduIg/3CqVBaT3Hca65SEoDwh0KiPtQoOgZL\n"
  "dKY2cobGs/ojYtOjcs0KnlPYdmtjEh6WEhuJU95v4TQdC4OLMiE56eIGq252hZAb\n"
  "HoTL84Q14DxQWGuzQK830iml7fbw2WcIcRQ8vFGs8SzfXw63+MI6Fq6iMAQIqP08\n"
  "WzGmRRzL5wvCiPhCVkrPmwbXoABub6AAsYwWPJB91M9/lx5gFH5k9/iPfi3s2Kg3\n"
  "F8MOcppqFYjxDSnsfiz6eMh1+bYVIAo367vGVYHigXMEZC2FezlwIHaZzpEoFlY3\n"
  "a7LFJ00yrjQ910r8UE+CEMTYzE40D0olCMo7FA9RCjeO3bUIoYaIdVTUGWEGHWSe\n"
  "oxGei9Gkm6u+ASj8f+i0jxdD2qXsewIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/\n"
  "MA8GA1UdDwEB/wQFAwMHBgAwHQYDVR0OBBYEFE1Wt2oAWPFnkvSmdVUbjlMBA+/P\n"
  "MA0GCSqGSIb3DQEBCwUAA4IBMQAesOgjGFi1zOYpA/N3gkUVRcBHDxmN7g2yOcqH\n"
  "VfhFc+e4zhOehR11WCt2RgzNlnYVmV5zBmQBdTAt8Po/MVhLCDW1BULHlLvL0DFc\n"
  "4sB1RlcGeQcCKQa4b+Q9VWf4f6TfuEWZQC5j5stiXjVgOqrOMrzKZ2eKWA4JsL9s\n"
  "V+7ANSZE+hOt1X1mA8moyqe95U2Ecih+nFJSWSBd1WFiEzVnXv4FVWPXbH9HERDK\n"
  "VbasjofWWmQO1YlQPishLgm1IbwqOkOk4sDgoLuUZ4GgP0DDeN6EmRDOzByrv+9u\n"
  "f45Bl9IQf4IJNPLU9lEqjyMOydqT6kBi7fjV5ICuQZ4EeVJsOGuX7PqNyoDzJHLv\n"
  "ferRfNLr6eQSHSxBhS0cVyDjb5gCawK6u7xTU+b7xikEie9k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Triggers incorrect verification success on older versions */
static const char *cve_2008_4989_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIB6zCCAVQCCQCgwnB/k0WZrDANBgkqhkiG9w0BAQUFADA9MQswCQYDVQQGEwJE\n"
  "RTEXMBUGA1UEChMOR05VIFRMUyBBdHRhY2sxFTATBgNVBAMTDGludGVybWVkaWF0\n"
  "ZTAeFw0wODExMDMxMjA1MDRaFw0wODEyMDMxMjA1MDRaMDcxCzAJBgNVBAYTAkRF\n"
  "MRcwFQYDVQQKEw5HTlUgVExTIEF0dGFjazEPMA0GA1UEAxMGc2VydmVyMIGfMA0G\n"
  "CSqGSIb3DQEBAQUAA4GNADCBiQKBgQDKdL9g5ErMLOLRCjiomZlNLhy0moWGaKIW\n"
  "aX6vyUIfh8d6FcArHoKoqhmX7ckvod50sOYPojQesDpl7gVaQNA6Ntr1VCcuNPef\n"
  "UKWtEwL0Qu9JbPnUoIYd7mAaqVQgFp6W6yzV/dp63LH4XSdzBMhpZ/EU6vZoE8Sv\n"
  "VLdqj5r6jwIDAQABMA0GCSqGSIb3DQEBBQUAA4GBAH4QRR7sZEbjW00tXYk/3O/Z\n"
  "96AxJNg0F78W5B68gaJrLJ7DTE2RTglscuEq1+2Jyb4AIziwXpYqxgwcP91QpH97\n"
  "XfwdXIcyjYvVLHiKmkQj2zJTY7MeyiEQQ2it8VstZG2fYmi2EiMZIEnyJ2JJ7bA7\n"
  "bF7pG7Cg3oEHUM0H5KUU\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (not signed by next cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICADCCAWmgAwIBAgIJAIZ4nkHQAqTFMA0GCSqGSIb3DQEBBQUAMDUxCzAJBgNV\n"
  "BAYTAkRFMRcwFQYDVQQKEw5HTlUgVExTIEF0dGFjazENMAsGA1UEAxMEcm9vdDAe\n"
  "Fw0wODExMDMxMjA0NDVaFw0wODEyMDMxMjA0NDVaMD0xCzAJBgNVBAYTAkRFMRcw\n"
  "FQYDVQQKEw5HTlUgVExTIEF0dGFjazEVMBMGA1UEAxMMaW50ZXJtZWRpYXRlMIGf\n"
  "MA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDvBpW8sAhIuUmNvcBE6wv/q7MtM1Z9\n"
  "2I1SDL8eJ8I2nPg6BlCX+OIqNruynj8J7uPEQ04ZLwLxNXoyZa8057YFyrKLOvoj\n"
  "5IfBtidsLWYv6PO3qqHJXVvwGdS7PKMuUlsjucCRyXVgQ07ODF7piqoVFi9KD99w\n"
  "AU5+9plGrZNP/wIDAQABoxAwDjAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUA\n"
  "A4GBAGPg+M+8MsB6zHN2o+jAtyqovrTTwmzVWEgfEH/aHC9+imGZRQ5lFNc2vdny\n"
  "AgaJ9/izO5S6Ibb5zUowN2WhoUJOVipuQa2m9AviOgheoU7tmANC9ylm/pRkKy/0\n"
  "n5UVzlKxDhRp/xBb7MWOw3KEQjiAf2Z3wCLcCPUqcJUdJC4v\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (trusted CA cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEIDCCAwigAwIBAgIQNE7VVyDV7exJ9C/ON9srbTANBgkqhkiG9w0BAQUF\n"
  "ADCBqTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYG\n"
  "A1UECxMfQ2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UE\n"
  "CxMvKGMpIDIwMDYgdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNl\n"
  "IG9ubHkxHzAdBgNVBAMTFnRoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EwHhcNMDYx\n"
  "MTE3MDAwMDAwWhcNMzYwNzE2MjM1OTU5WjCBqTELMAkGA1UEBhMCVVMxFTAT\n"
  "BgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYGA1UECxMfQ2VydGlmaWNhdGlvbiBT\n"
  "ZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UECxMvKGMpIDIwMDYgdGhhd3RlLCBJ\n"
  "bmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxHzAdBgNVBAMTFnRoYXd0\n"
  "ZSBQcmltYXJ5IFJvb3QgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
  "AoIBAQCsoPD7gFnUnMekz52hWXMJEEUMDSxuaPFsW0hoSVk3/AszGcJ3f8wQ\n"
  "LZU0HObrTQmnHNK4yZc2AreJ1CRfBsDMRJSUjQJib+ta3RGNKJpchJAQeg29\n"
  "dGYvajig4tVUROsdB58Hum/u6f1OCyn1PoSgAfGcq/gcfomk6KHYcWUNo1F7\n"
  "7rzSImANuVud37r8UVsLr5iy6S7pBOhih94ryNdOwUxkHt3Ph1i6Sk/KaAcd\n"
  "HJ1KxtUvkcx8cXIcxcBn6zL9yZJclNqFwJu/U30rCfSMnZEfl2pSy94JNqR3\n"
  "2HuHUETVPm4pafs5SSYeCaWAe0At6+gnhcn+Yf1+5nyXHdWdAgMBAAGjQjBA\n"
  "MA8GA1UdEwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBR7\n"
  "W0XPr87Lev0xkhpqtvNG61dIUDANBgkqhkiG9w0BAQUFAAOCAQEAeRHAS7OR\n"
  "tvzw6WfUDW5FvlXok9LOAz/t2iWwHVfLHjp2oEzsUHboZHIMpKnxuIvW1oeE\n"
  "uzLlQRHAd9mzYJ3rG9XRbkREqaYB7FViHXe4XI5ISXycO1cRrK1zN44veFyQ\n"
  "aEfZYGDm/Ac9IiAXxPcW6cTYcvnIc3zfFi8VqT79aie2oetaupgf1eNNZAqd\n"
  "E8hhuvU5HIe6uL17In/2/qxAeeWsEG89jxt5dovEN7MhGITlNgDrYyCZuen+\n"
  "MwS7QcjBAvlEYyCegc5C09Y/LHbTY5xZ3Y+m4Q6gLkH3LpVHz7z9M/P2C2F+\n"
  "fpErgUfCJzDupxBdN49cOSvkBPB7jVaMaA==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 3 ends with trusted v1 RSA-MD2 chain */
static const char *verisign_com_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCDCCBPCgAwIBAgIQakrDGzEQ5utI8PxRo5oXHzANBgkqhkiG9w0BAQUFADCB\n"
  "vjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
  "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjE4MDYGA1UEAxMv\n"
  "VmVyaVNpZ24gQ2xhc3MgMyBFeHRlbmRlZCBWYWxpZGF0aW9uIFNTTCBTR0MgQ0Ew\n"
  "HhcNMDcwNTA5MDAwMDAwWhcNMDkwNTA4MjM1OTU5WjCCAUAxEDAOBgNVBAUTBzI0\n"
  "OTc4ODYxEzARBgsrBgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMIRGVs\n"
  "YXdhcmUxCzAJBgNVBAYTAlVTMQ4wDAYDVQQRFAU5NDA0MzETMBEGA1UECBMKQ2Fs\n"
  "aWZvcm5pYTEWMBQGA1UEBxQNTW91bnRhaW4gVmlldzEiMCAGA1UECRQZNDg3IEVh\n"
  "c3QgTWlkZGxlZmllbGQgUm9hZDEXMBUGA1UEChQOVmVyaVNpZ24sIEluYy4xJTAj\n"
  "BgNVBAsUHFByb2R1Y3Rpb24gU2VjdXJpdHkgU2VydmljZXMxMzAxBgNVBAsUKlRl\n"
  "cm1zIG9mIHVzZSBhdCB3d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjEZMBcGA1UE\n"
  "AxQQd3d3LnZlcmlzaWduLmNvbTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEA\n"
  "xxA35ev879drgQCpENGRQ3ARaCPz/WneT9dtMe3qGNvzXQJs6cjm1Bx8XegyW1gB\n"
  "jJX5Zl4WWbr9wpAWZ1YyJ0bEyShIGmkU8fPfbcXYwSyWoWwvE5NRaUB2ztmfAVdv\n"
  "OaGMUKxny2Dnj3tAdaQ+FOeRDJJYg6K1hzczq/otOfsCAwEAAaOCAf8wggH7MAkG\n"
  "A1UdEwQCMAAwHQYDVR0OBBYEFPFaiZNVR0u6UfVO4MsWVfTXzDhnMAsGA1UdDwQE\n"
  "AwIFoDA+BgNVHR8ENzA1MDOgMaAvhi1odHRwOi8vRVZJbnRsLWNybC52ZXJpc2ln\n"
  "bi5jb20vRVZJbnRsMjAwNi5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXBjAq\n"
  "MCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQGA1Ud\n"
  "JQQtMCsGCCsGAQUFBwMBBggrBgEFBQcDAgYJYIZIAYb4QgQBBgorBgEEAYI3CgMD\n"
  "MB8GA1UdIwQYMBaAFE5DyB127zdTek/yWG+U8zji1b3fMHYGCCsGAQUFBwEBBGow\n"
  "aDArBggrBgEFBQcwAYYfaHR0cDovL0VWSW50bC1vY3NwLnZlcmlzaWduLmNvbTA5\n"
  "BggrBgEFBQcwAoYtaHR0cDovL0VWSW50bC1haWEudmVyaXNpZ24uY29tL0VWSW50\n"
  "bDIwMDYuY2VyMG0GCCsGAQUFBwEMBGEwX6FdoFswWTBXMFUWCWltYWdlL2dpZjAh\n"
  "MB8wBwYFKw4DAhoEFI/l0xqGrI2Oa8PPgGrUSBgsexkuMCUWI2h0dHA6Ly9sb2dv\n"
  "LnZlcmlzaWduLmNvbS92c2xvZ28uZ2lmMA0GCSqGSIb3DQEBBQUAA4IBAQBEueAg\n"
  "xZJrjGPKAZk1NT8VtTn0yi87i9XUnSOnkFkAuI3THDd+cWbNSUzc5uFJg42GhMK7\n"
  "S1Rojm8FHxESovLvimH/w111BKF9wNU2XSOb9KohfYq3GRiQG8O7v9JwIjjLepkc\n"
  "iyITx7sYiJ+kwZlrNBwN6TwVHrONg6NzyzSnxCg+XgKRbJu2PqEQb6uQVkYhb+Oq\n"
  "Vi9d4by9YqpnuXImSffQ0OZ/6s3Rl6vY08zIPqa6OVfjGs/H45ETblzezcUKpX0L\n"
  "cqnOwUB9dVuPhtlX3X/hgz/ROxz96NBwwzha58HUgfEfkVtm+piI6TTI7XxS/7Av\n"
  "nKMfhbyFQYPQ6J9g\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCjCCBPKgAwIBAgIQESoAbTflEG/WynzD77rMGDANBgkqhkiG9w0BAQUFADCB\n"
  "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
  "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
  "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
  "aG9yaXR5IC0gRzUwHhcNMDYxMTA4MDAwMDAwWhcNMTYxMTA3MjM1OTU5WjCBvjEL\n"
  "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
  "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjE4MDYGA1UEAxMvVmVy\n"
  "aVNpZ24gQ2xhc3MgMyBFeHRlbmRlZCBWYWxpZGF0aW9uIFNTTCBTR0MgQ0EwggEi\n"
  "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC9Voi6iDRkZM/NyrDu5xlzxXLZ\n"
  "u0W8taj/g74cA9vtibcuEBolvFXKQaGfC88ZXnC5XjlLnjEcX4euKqqoK6IbOxAj\n"
  "XxOx3QiMThTag4HjtYzjaO0kZ85Wtqybc5ZE24qMs9bwcZOO23FUSutzWWqPcFEs\n"
  "A5+X0cwRerxiDZUqyRx1V+n1x+q6hDXLx4VafuRN4RGXfQ4gNEXb8aIJ6+s9nriW\n"
  "Q140SwglHkMaotm3igE0PcP45a9PjP/NZfAjTsWXs1zakByChQ0GDcEitnsopAPD\n"
  "TFPRWLxyvAg5/KB2qKjpS26IPeOzMSWMcylIDjJ5Bu09Q/T25On8fb6OCNUfAgMB\n"
  "AAGjggH0MIIB8DAdBgNVHQ4EFgQUTkPIHXbvN1N6T/JYb5TzOOLVvd8wEgYDVR0T\n"
  "AQH/BAgwBgEB/wIBADA9BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYc\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2NwczA9BgNVHR8ENjA0MDKgMKAuhixo\n"
  "dHRwOi8vRVZTZWN1cmUtY3JsLnZlcmlzaWduLmNvbS9wY2EzLWc1LmNybDAgBgNV\n"
  "HSUEGTAXBglghkgBhvhCBAEGCmCGSAGG+EUBCAEwDgYDVR0PAQH/BAQDAgEGMBEG\n"
  "CWCGSAGG+EIBAQQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFn\n"
  "ZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNodHRw\n"
  "Oi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjApBgNVHREEIjAgpB4wHDEa\n"
  "MBgGA1UEAxMRQ2xhc3MzQ0EyMDQ4LTEtNDgwPQYIKwYBBQUHAQEEMTAvMC0GCCsG\n"
  "AQUFBzABhiFodHRwOi8vRVZTZWN1cmUtb2NzcC52ZXJpc2lnbi5jb20wHwYDVR0j\n"
  "BBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJKoZIhvcNAQEFBQADggEBAFqi\n"
  "sb/rjdQ4qIBywtw4Lqyncfkro7tHu21pbxA2mIzHVi67vKtKm3rW8oKT4BT+is6D\n"
  "t4Pbk4errGV5Sf1XqbHOCR+6EBXECQ5i4/kKJdVkmPDyqA92Mn6R5hjuvOfa0E6N\n"
  "eLvincBZK8DOlQ0kDHLKNF5wIokrSrDxaIfz7kSNKEB3OW5IckUxXWs5DoYC6maZ\n"
  "kzEP32fepp+MnUzOcW86Ifa5ND/5btia9z7a84Ffelxtj3z2mXS3/+QXXe1hXqtI\n"
  "u5aNZkU5tBIK9nDpnHYiS2DpKhs0Sfei1GfAsSatE7rZhAHBq+GObXAWO3eskZq7\n"
  "Gh/aWKfkT8Fhrryi/ks=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE/zCCBGigAwIBAgIQY5Jrio9Agv2swDvTeCmmwDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
  "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
  "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
  "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
  "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
  "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
  "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
  "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
  "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
  "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
  "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
  "AAGjggHKMIIBxjAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
  "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjBt\n"
  "BggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFnZS9naWYwITAfMAcGBSsOAwIa\n"
  "BBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNodHRwOi8vbG9nby52ZXJpc2lnbi5j\n"
  "b20vdnNsb2dvLmdpZjA9BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYc\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7Lvw\n"
  "MAnzQzn6Aq8zMTMwgYAGA1UdIwR5MHehY6RhMF8xCzAJBgNVBAYTAlVTMRcwFQYD\n"
  "VQQKEw5WZXJpU2lnbiwgSW5jLjE3MDUGA1UECxMuQ2xhc3MgMyBQdWJsaWMgUHJp\n"
  "bWFyeSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eYIQcLrkHRDZKTS2OMp7A8y6vzAg\n"
  "BgNVHSUEGTAXBglghkgBhvhCBAEGCmCGSAGG+EUBCAEwDQYJKoZIhvcNAQEFBQAD\n"
  "gYEAUNfnArcMK6xK11/59ADJdeNqKOck4skH3qw6WCAYQxfrcn4eobTInOn5G3Gu\n"
  "39g6DapSHmBex2UtZSxvKnJVlWYQgE4P4wGoXdzV69YdCNssXNVVc59DYhDH05dZ\n"
  "P4sJH99fucYDkJjUgRUYw35ww0OFwKgUp3CxiizbXxCqEQc=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[3] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 2 ends with trusted v1 RSA-MD2 cert */
static const char *citibank_com_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIENDCCA52gAwIBAgIQauOJMlH5Ob2tFZ6rJMBdjjANBgkqhkiG9w0BAQUFADCB\n"
  "ujEfMB0GA1UEChMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVy\n"
  "aVNpZ24sIEluYy4xMzAxBgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2Vy\n"
  "dmVyIENBIC0gQ2xhc3MgMzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMg\n"
  "SW5jb3JwLmJ5IFJlZi4gTElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjAeFw0w\n"
  "ODA4MjkwMDAwMDBaFw0xMDA4MjkyMzU5NTlaMHgxCzAJBgNVBAYTAlVTMRMwEQYD\n"
  "VQQIEwpOZXcgSmVyc2V5MRIwEAYDVQQHFAlXZWVoYXdrZW4xEjAQBgNVBAoUCUNp\n"
  "dGlncm91cDERMA8GA1UECxQId2hnLW9hazYxGTAXBgNVBAMUEHd3dy5jaXRpYmFu\n"
  "ay5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBALQJbSYtbndsIlslFveP\n"
  "IlVNE38HnUD56BHcwfvcb8rQItXeHzYmgOf/RgHPTKG3LEZOxKqM0QpcZtEJ6xwV\n"
  "cTG7Wjw/FrMisN8aO4JWaxe8dFGajstlEMxz43G5zlprb9jzjnbIvvcnz0ILikOQ\n"
  "qmcThopBTs1+d4j7w/yEJo1zAgMBAAGjggF6MIIBdjAJBgNVHRMEAjAAMAsGA1Ud\n"
  "DwQEAwIFoDBGBgNVHR8EPzA9MDugOaA3hjVodHRwOi8vY3JsLnZlcmlzaWduLmNv\n"
  "bS9DbGFzczNJbnRlcm5hdGlvbmFsU2VydmVyLmNybDBEBgNVHSAEPTA7MDkGC2CG\n"
  "SAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlzaWduLmNv\n"
  "bS9ycGEwKAYDVR0lBCEwHwYJYIZIAYb4QgQBBggrBgEFBQcDAQYIKwYBBQUHAwIw\n"
  "NAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC52ZXJpc2ln\n"
  "bi5jb20wbgYIKwYBBQUHAQwEYjBgoV6gXDBaMFgwVhYJaW1hZ2UvZ2lmMCEwHzAH\n"
  "BgUrDgMCGgQUS2u5KJYGDLvQUjibKaxLB4shBRgwJhYkaHR0cDovL2xvZ28udmVy\n"
  "aXNpZ24uY29tL3ZzbG9nbzEuZ2lmMA0GCSqGSIb3DQEBBQUAA4GBAFDXKsxtWkoo\n"
  "HBkNjcCvcnjNAo3Pe+eOtLHb39e5qhkNQLPGA/1/7AofY9KmEtSV2LVGeuuJI4Pi\n"
  "Lg7fPl9Q0OE/oHJpj5JkObBP9Wo1vbrDR2nGWUlCRWm20rH81dTn7OcDxarwGWsR\n"
  "ewTCNmpKYaMx8Q1dyMYunHJApu+fbrHu\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDgzCCAuygAwIBAgIQJUuKhThCzONY+MXdriJupDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNOTcwNDE3MDAwMDAwWhcNMTExMDI0MjM1OTU5WjCBujEfMB0GA1UEChMWVmVy\n"
  "aVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVyaVNpZ24sIEluYy4xMzAx\n"
  "BgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2VydmVyIENBIC0gQ2xhc3Mg\n"
  "MzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMgSW5jb3JwLmJ5IFJlZi4g\n"
  "TElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjCBnzANBgkqhkiG9w0BAQEFAAOB\n"
  "jQAwgYkCgYEA2IKA6NYZAn0fhRg5JaJlK+G/1AXTvOY2O6rwTGxbtueqPHNFVbLx\n"
  "veqXQu2aNAoV1Klc9UAl3dkHwTKydWzEyruj/lYncUOqY/UwPpMo5frxCTvzt01O\n"
  "OfdcSVq4wR3Tsor+cDCVQsv+K1GLWjw6+SJPkLICp1OcTzTnqwSye28CAwEAAaOB\n"
  "4zCB4DAPBgNVHRMECDAGAQH/AgEAMEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHAQEw\n"
  "KjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL0NQUzA0BgNV\n"
  "HSUELTArBggrBgEFBQcDAQYIKwYBBQUHAwIGCWCGSAGG+EIEAQYKYIZIAYb4RQEI\n"
  "ATALBgNVHQ8EBAMCAQYwEQYJYIZIAYb4QgEBBAQDAgEGMDEGA1UdHwQqMCgwJqAk\n"
  "oCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAAgB7ORolANC8XPxI6I63unx2sZUxCM+hurPajozq+qcBBQHNgYL+Yhv\n"
  "1RPuKSvD5HKNRO3RrCAJLeH24RkFOLA9D59/+J4C3IYChmFOJl9en5IeDCSk9dBw\n"
  "E88mw0M9SR2egi5SX7w+xmYpAY5Okiy8RnUDgqxz6dl+C2fvVFIa\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Self-signed certificate */
static const char *pem_self_cert[] = {
  "-----BEGIN CERTIFICATE-----\n"
    "MIIDgjCCAmygAwIBAgIBADALBgkqhkiG9w0BAQUwSzELMAkGA1UEBhMCQlIxFDAS\n"
    "BgNVBAoTC01pbmFzIExpdnJlMSYwJAYDVQQDEx1UaGFkZXUgTGltYSBkZSBTb3V6\n"
    "YSBDYXNjYXJkbzAeFw0wODA1MzAxOTUzNDNaFw0wODExMjYxOTUzNDNaMEsxCzAJ\n"
    "BgNVBAYTAkJSMRQwEgYDVQQKEwtNaW5hcyBMaXZyZTEmMCQGA1UEAxMdVGhhZGV1\n"
    "IExpbWEgZGUgU291emEgQ2FzY2FyZG8wggEfMAsGCSqGSIb3DQEBAQOCAQ4AMIIB\n"
    "CQKCAQC4D934O6wrXJbMyu1w8gu6nN0aNUDGqrX9UgaB/4xVuYhPlhjH0z9Dqic9\n"
    "0pEZmyNCjQmzDSg/hnlY3fBG0i9Iel2oYn1UB4SdcJ2qGkLS87y2ZbMTS1oyMR7/\n"
    "y9l3WGEWqwgjIvOjGstcZo0rCIF8Qr21QGX22KWg2HXlMaZyA9bGtJ+L+x6f2hoo\n"
    "yIPCA30VMvIgHjOSPQJF3iJFE4Uxq1PQ65W91NyI6/bRKFOmFdCUJW8tqqvntYP8\n"
    "hEE08wGlKimFNv7CqZuRI8QuOnhZ7pBXkyvQpW8yHrORlOHxSjkNQKjddt92TCJb\n"
    "1q6eKv2CtCuDLgCuIy0Onr4U9n+hAgMBAAGjeDB2MA8GA1UdEwEB/wQFMAMBAf8w\n"
    "HgYDVR0RBBcwFYITbWFpbC5taW5hc2xpdnJlLm9yZzATBgNVHSUEDDAKBggrBgEF\n"
    "BQcDATAPBgNVHQ8BAf8EBQMDB6QAMB0GA1UdDgQWBBQ/5v42y0jBHUKEfqpPmr5a\n"
    "WsjCGjALBgkqhkiG9w0BAQUDggEBAC/WfO2yK3vM9bG0qFEj8sd0cWiapMhf5PtH\n"
    "jigcPb/OKqSFQVXpAdNiUclPRP79Ih3CuWiXfZ/CW0+k2Z8tyy6AnEQItWvoVh/b\n"
    "8lS7Ph/f9JUYHp2DtgsQWcNQbrUZOPFBu8J4MD6cDWG5Uxwl3YASg30ZdmMDNT8B\n"
    "HshYz0HUOAhYwVSI3J/f7LFhD5OpjSroHgE7wA9UJrerAp9f7e3e9D7kNQ8DlvLP\n"
    "kz6Jh+5M/xD3JO1yl+evaCp3LA+z4M2xiNvtzkAEgj3t6RaJ81Sh5XGiooDYZ14R\n"
    "DgEBYLTUfBYBPzoaahPEdG/f0kUjUBJ34fkBUSjJKURPTHJfDfA=\n"
    "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 2, CA constraint FALSE in v3 CA cert)*/
static const char *thea_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC7DCCAlWgAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJERTEM\n"
  "MAoGA1UECBMDUkxQMSAwHgYDVQQKExdUZWNobmlzY2hlIFVuaXZlcnNpdGFldDEb\n"
  "MBkGA1UECxMSRmFjaGJlcmVpY2ggUGh5c2lrMQswCQYDVQQDEwJDQTAeFw0wODA5\n"
  "MTExMDUyMDdaFw0xODA5MDkxMDUyMDdaMIGTMQswCQYDVQQGEwJERTEMMAoGA1UE\n"
  "CBMDUkxQMRcwFQYDVQQHEw5LYWlzZXJzbGF1dGVybjEgMB4GA1UEChMXVGVjaG5p\n"
  "c2NoZSBVbml2ZXJzaXRhZXQxGzAZBgNVBAsTEkZhY2hiZXJlaWNoIFBoeXNpazEe\n"
  "MBwGA1UEAxMVdGhlYS5waHlzaWsudW5pLWtsLmRlMIGfMA0GCSqGSIb3DQEBAQUA\n"
  "A4GNADCBiQKBgQC/gTUrXSeNvuRH+ibdR7zvlCGs+66C6tDaq14SpEDiY/FEw/S4\n"
  "mkhsHohiQkmqpcPJ0FONok7bvJryKZwwhGFHeESvvWjFVNIdxFgf6Jx2McKsRzBD\n"
  "nbgVNeK6bywh2L5WgOeckRm0vUxCwX+jWtETorNHSYnZI9smmBtJ1FIPkQIDAQAB\n"
  "o3sweTAJBgNVHRMEAjAAMCwGCWCGSAGG+EIBDQQfFh1PcGVuU1NMIEdlbmVyYXRl\n"
  "ZCBDZXJ0aWZpY2F0ZTAdBgNVHQ4EFgQUS0IiRshnnlH2bneYeCn6OkY9nZAwHwYD\n"
  "VR0jBBgwFoAU+rCwSUUzK53X9W5otZG4okyY/rswDQYJKoZIhvcNAQEFBQADgYEA\n"
  "g0f6XFxpUL2hncpQcnKorNYdOkZkZHiKqu2SINtla+IbLZFO4nVVO+LKt+RCo2o7\n"
  "tZIMLEU3aCeH5dgSEKQeyL5MPMg3MbA6ezjOBTkT/YgngzM4CMLOKcvAMLncfH/z\n"
  "GYBW1DXijIy1r/SxO0k9zy8OEtKeOOUO0GqQTWuTOOg=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICvzCCAiigAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJERTEM\n"
  "MAoGA1UECBMDUkxQMSAwHgYDVQQKExdUZWNobmlzY2hlIFVuaXZlcnNpdGFldDEb\n"
  "MBkGA1UECxMSRmFjaGJlcmVpY2ggUGh5c2lrMQswCQYDVQQDEwJDQTAeFw0wODA5\n"
  "MTExMDQ3NDRaFw0xODA5MDkxMDQ3NDRaMGcxCzAJBgNVBAYTAkRFMQwwCgYDVQQI\n"
  "EwNSTFAxIDAeBgNVBAoTF1RlY2huaXNjaGUgVW5pdmVyc2l0YWV0MRswGQYDVQQL\n"
  "ExJGYWNoYmVyZWljaCBQaHlzaWsxCzAJBgNVBAMTAkNBMIGfMA0GCSqGSIb3DQEB\n"
  "AQUAA4GNADCBiQKBgQC76RbqsB5J+VvU1KbBCrkIL3lgY8BxgFvYF3HiHgxtCdqq\n"
  "BmRpAaDBcVAuEb1ihhP68181sYQ1UPMY+zwBwXVNSVvjeBba1JjGmagwPnJXOCay\n"
  "7Cw5orY8KB7U33neEOGrlz1EKQGVaPsr993wGD/7AmntuVuxrRVpzoDP5s0PIwID\n"
  "AQABo3sweTAJBgNVHRMEAjAAMCwGCWCGSAGG+EIBDQQfFh1PcGVuU1NMIEdlbmVy\n"
  "YXRlZCBDZXJ0aWZpY2F0ZTAdBgNVHQ4EFgQU+rCwSUUzK53X9W5otZG4okyY/rsw\n"
  "HwYDVR0jBBgwFoAU+rCwSUUzK53X9W5otZG4okyY/rswDQYJKoZIhvcNAQEFBQAD\n"
  "gYEAUT+LmosiDHGuLAZmY40obam0eexJzn/g++mDy3FMh3WmMBKSsfwFsFsQ4k7N\n"
  "lv1SCfTYeh2hpw/DQzkiYZUkcQI4mBR4hG5Zv56AfYQLGeLtN4VOOCMxguftvzv0\n"
  "kziQa2QW+VzVJqV1gpRCRT30Jaa9s4u6ipO9DT5N03F4CcI=\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 3 ends with trusted v1 RSA-MD2 cert, similar to
   verisign_com_chain above */
static const char *hbci_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEczCCA9ygAwIBAgIQeODCPg2RbK2r7/1KoWjWZzANBgkqhkiG9w0BAQUFADCB\n"
  "ujEfMB0GA1UEChMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVy\n"
  "aVNpZ24sIEluYy4xMzAxBgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2Vy\n"
  "dmVyIENBIC0gQ2xhc3MgMzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMg\n"
  "SW5jb3JwLmJ5IFJlZi4gTElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjAeFw0w\n"
  "ODA2MTAwMDAwMDBaFw0wOTA3MzAyMzU5NTlaMIG2MQswCQYDVQQGEwJERTEPMA0G\n"
  "A1UECBMGSGVzc2VuMRowGAYDVQQHFBFGcmFua2Z1cnQgYW0gTWFpbjEsMCoGA1UE\n"
  "ChQjU3Bhcmthc3NlbiBJbmZvcm1hdGlrIEdtYkggJiBDby4gS0cxKTAnBgNVBAsU\n"
  "IFRlcm1zIG9mIHVzZSBhdCB3d3cudmVyaXNpZ24uY29tMSEwHwYDVQQDFBhoYmNp\n"
  "LXBpbnRhbi1ycC5zLWhiY2kuZGUwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGB\n"
  "AK1CdQ9lqmChZWaRAInimuK7I36VImTuAVU0N6BIS4a2BbblkiekbVf15GVHGb6e\n"
  "QV06ANN6Nd8XIdfoxi3LoAs8sa+Ku7eoEsRFi/XIU96GgtFlxf3EsVA9RbGdtfer\n"
  "9iJGIBae2mJTlk+5LVg2EQr50PJlBuTgiYFc41xs9O2RAgMBAAGjggF6MIIBdjAJ\n"
  "BgNVHRMEAjAAMAsGA1UdDwQEAwIFoDBGBgNVHR8EPzA9MDugOaA3hjVodHRwOi8v\n"
  "Y3JsLnZlcmlzaWduLmNvbS9DbGFzczNJbnRlcm5hdGlvbmFsU2VydmVyLmNybDBE\n"
  "BgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0dHBzOi8v\n"
  "d3d3LnZlcmlzaWduLmNvbS9ycGEwKAYDVR0lBCEwHwYJYIZIAYb4QgQBBggrBgEF\n"
  "BQcDAQYIKwYBBQUHAwIwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRw\n"
  "Oi8vb2NzcC52ZXJpc2lnbi5jb20wbgYIKwYBBQUHAQwEYjBgoV6gXDBaMFgwVhYJ\n"
  "aW1hZ2UvZ2lmMCEwHzAHBgUrDgMCGgQUS2u5KJYGDLvQUjibKaxLB4shBRgwJhYk\n"
  "aHR0cDovL2xvZ28udmVyaXNpZ24uY29tL3ZzbG9nbzEuZ2lmMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAJ03R0YAjYzlWm54gMSn6MqJi0mHdLCO2lk3CARwjbg7TEYAZvDsKqTd\n"
  "cRuhNk079BqrQ3QapffeN55SAVrc3mzHO54Nla4n5y6x3XIQXVvRjbJGwmWXsdvr\n"
  "W899F/pBEN30Tgdbmn7JR/iZlGhIJpY9Us1i7rwQhKYir9ZQBdj3\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDgzCCAuygAwIBAgIQJUuKhThCzONY+MXdriJupDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNOTcwNDE3MDAwMDAwWhcNMTExMDI0MjM1OTU5WjCBujEfMB0GA1UEChMWVmVy\n"
  "aVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVyaVNpZ24sIEluYy4xMzAx\n"
  "BgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2VydmVyIENBIC0gQ2xhc3Mg\n"
  "MzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMgSW5jb3JwLmJ5IFJlZi4g\n"
  "TElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjCBnzANBgkqhkiG9w0BAQEFAAOB\n"
  "jQAwgYkCgYEA2IKA6NYZAn0fhRg5JaJlK+G/1AXTvOY2O6rwTGxbtueqPHNFVbLx\n"
  "veqXQu2aNAoV1Klc9UAl3dkHwTKydWzEyruj/lYncUOqY/UwPpMo5frxCTvzt01O\n"
  "OfdcSVq4wR3Tsor+cDCVQsv+K1GLWjw6+SJPkLICp1OcTzTnqwSye28CAwEAAaOB\n"
  "4zCB4DAPBgNVHRMECDAGAQH/AgEAMEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHAQEw\n"
  "KjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL0NQUzA0BgNV\n"
  "HSUELTArBggrBgEFBQcDAQYIKwYBBQUHAwIGCWCGSAGG+EIEAQYKYIZIAYb4RQEI\n"
  "ATALBgNVHQ8EBAMCAQYwEQYJYIZIAYb4QgEBBAQDAgEGMDEGA1UdHwQqMCgwJqAk\n"
  "oCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAAgB7ORolANC8XPxI6I63unx2sZUxCM+hurPajozq+qcBBQHNgYL+Yhv\n"
  "1RPuKSvD5HKNRO3RrCAJLeH24RkFOLA9D59/+J4C3IYChmFOJl9en5IeDCSk9dBw\n"
  "E88mw0M9SR2egi5SX7w+xmYpAY5Okiy8RnUDgqxz6dl+C2fvVFIa\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* End-entity cert signed using RSA-MD5. */
static const char *mayfirst_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDVTCCAr6gAwIBAgIDCHp1MA0GCSqGSIb3DQEBBAUAMFoxCzAJBgNVBAYTAlVT\n"
  "MRwwGgYDVQQKExNFcXVpZmF4IFNlY3VyZSBJbmMuMS0wKwYDVQQDEyRFcXVpZmF4\n"
  "IFNlY3VyZSBHbG9iYWwgZUJ1c2luZXNzIENBLTEwHhcNMDgwNTE5MDUyOTE5WhcN\n"
  "MDkxMDE5MDUyOTE5WjCBxDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFHN1cHBvcnQu\n"
  "bWF5Zmlyc3Qub3JnMRMwEQYDVQQLEwpHVDY5MDc5ODgwMTEwLwYDVQQLEyhTZWUg\n"
  "d3d3LnJhcGlkc3NsLmNvbS9yZXNvdXJjZXMvY3BzIChjKTA3MS8wLQYDVQQLEyZE\n"
  "b21haW4gQ29udHJvbCBWYWxpZGF0ZWQgLSBSYXBpZFNTTChSKTEdMBsGA1UEAxMU\n"
  "c3VwcG9ydC5tYXlmaXJzdC5vcmcwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGB\n"
  "AN0TWIZwJ/hIfMHc08/bBMlzZ5WucJqEvxU/ZnxPo/H6V/m4v1iLpM2hip2c5cg0\n"
  "BcEMc/TBHQ1UEV8sb0Lh91kWfiMB1Sp+L2Fpz/wnhsivXC5j6jq9IcPqmOZOXBYX\n"
  "k04W1B6FKTvk9KrZJ0at2J44hp4SsAfWQI0eCKuas+R1AgMBAAGjgb0wgbowDgYD\n"
  "VR0PAQH/BAQDAgTwMB0GA1UdDgQWBBS0D4iuCxp35TLADTkINq2AhgTYVTA7BgNV\n"
  "HR8ENDAyMDCgLqAshipodHRwOi8vY3JsLmdlb3RydXN0LmNvbS9jcmxzL2dsb2Jh\n"
  "bGNhMS5jcmwwHwYDVR0jBBgwFoAUvqigdHJQa0S3ySPY+6j/s1draGwwHQYDVR0l\n"
  "BBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcN\n"
  "AQEEBQADgYEAXNWYnrO1mZgBSCwPlWhVa2aOKGCFmehLIcAPEBN+8xhXuOeigYBm\n"
  "ic5ShCO583ttgHNCV3Y5dW9sNhv1US4vSb6soKjgUlG11fJKUqU8mwFKvbs7TUSq\n"
  "j6h+1uvlfFI34WzODjJloY4QSM7FmbnW+HCiFKYyvra3iUqjcl9AeR4=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICkDCCAfmgAwIBAgIBATANBgkqhkiG9w0BAQQFADBaMQswCQYDVQQGEwJV\n"
  "UzEcMBoGA1UEChMTRXF1aWZheCBTZWN1cmUgSW5jLjEtMCsGA1UEAxMkRXF1\n"
  "aWZheCBTZWN1cmUgR2xvYmFsIGVCdXNpbmVzcyBDQS0xMB4XDTk5MDYyMTA0\n"
  "MDAwMFoXDTIwMDYyMTA0MDAwMFowWjELMAkGA1UEBhMCVVMxHDAaBgNVBAoT\n"
  "E0VxdWlmYXggU2VjdXJlIEluYy4xLTArBgNVBAMTJEVxdWlmYXggU2VjdXJl\n"
  "IEdsb2JhbCBlQnVzaW5lc3MgQ0EtMTCBnzANBgkqhkiG9w0BAQEFAAOBjQAw\n"
  "gYkCgYEAuucXkAJlsTRVPEnCUdXfp9E3j9HngXNBUmCbnaEXJnitx7HoJpQy\n"
  "td4zjTov2/KaelpzmKNc6fuKcxtc58O/gGzNqfTWK8D3+ZmqY6KxRwIP1ORR\n"
  "OhI8bIpaVIRw28HFkM9yRcuoWcDNM50/o5brhTMhHD4ePmBudpxnhcXIw2EC\n"
  "AwEAAaNmMGQwEQYJYIZIAYb4QgEBBAQDAgAHMA8GA1UdEwEB/wQFMAMBAf8w\n"
  "HwYDVR0jBBgwFoAUvqigdHJQa0S3ySPY+6j/s1draGwwHQYDVR0OBBYEFL6o\n"
  "oHRyUGtEt8kj2Puo/7NXa2hsMA0GCSqGSIb3DQEBBAUAA4GBADDiAVGqx+pf\n"
  "2rnQZQ8w1j7aDRRJbpGTJxQx78T3LUX47Me/okENI7SS+RkAZ70Br83gcfxa\n"
  "z2TE4JaY0KNA4gGK7ycH8WUBikQtBmV1UsCGECAhX2xrD2yuCRyv8qIYNMR1\n"
  "pHMc8Y3c7635s3a0kr/clRAevsvIO1qEYBlWlKlV\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test V1 CA without basicConstraint. */
static const char *v1ca[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE/zCCA+egAwIBAgIQBSsgZODO6vk6ayagofBQJDANBgkqhkiG9w0BAQUFADCB\n"
  "sDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
  "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNTEqMCgGA1UEAxMh\n"
  "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBMB4XDTA4MDUwNTAwMDAw\n"
  "MFoXDTA5MDUyMjIzNTk1OVowczELMAkGA1UEBhMCVVMxETAPBgNVBAgTCElsbGlu\n"
  "b2lzMRAwDgYDVQQHFAdEdSBQYWdlMSQwIgYDVQQKFBtBcmdvbm5lIE5hdGlvbmFs\n"
  "IExhYm9yYXRvcnkxGTAXBgNVBAMUEGF1dGgyLml0LmFubC5nb3YwgZ8wDQYJKoZI\n"
  "hvcNAQEBBQADgY0AMIGJAoGBAMg6YPOXsPQedkLUug3RoMjv/OB+SfuDgGXxtef5\n"
  "iE0SjCcsKT5v+bfxt2+ccs7IN7kWn1luJ5NTb0ZrdE6LQoYp9oLsaX/ukOnxKUMY\n"
  "YhJJyHgutPtwyPvfZTZPpATWycJnZGIehY1S6thwxeofUyE3ykec2lalULzwXgel\n"
  "iC97AgMBAAGjggHTMIIBzzAJBgNVHRMEAjAAMAsGA1UdDwQEAwIFoDBEBgNVHR8E\n"
  "PTA7MDmgN6A1hjNodHRwOi8vU1ZSU2VjdXJlLWNybC52ZXJpc2lnbi5jb20vU1ZS\n"
  "U2VjdXJlMjAwNS5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXAzAqMCgGCCsG\n"
  "AQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMB0GA1UdJQQWMBQG\n"
  "CCsGAQUFBwMBBggrBgEFBQcDAjAfBgNVHSMEGDAWgBRv7K+g3Yqk7/UqEGctP1WC\n"
  "vNfvJTB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZl\n"
  "cmlzaWduLmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL1NWUlNlY3VyZS1haWEudmVy\n"
  "aXNpZ24uY29tL1NWUlNlY3VyZTIwMDUtYWlhLmNlcjBuBggrBgEFBQcBDARiMGCh\n"
  "XqBcMFowWDBWFglpbWFnZS9naWYwITAfMAcGBSsOAwIaBBRLa7kolgYMu9BSOJsp\n"
  "rEsHiyEFGDAmFiRodHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvMS5naWYw\n"
  "DQYJKoZIhvcNAQEFBQADggEBAEAKzE4gXeyjRDBSgAgWIaCozbWK+b1ct4aZhWZt\n"
  "ihAyASxsNgDjDJzkInINjXoO5FWuJHDuoyyWHBQBb7t96+KgFu/4ye90VUDNTuin\n"
  "mmqdOKeLSHVnlhfvGLCdrhWSWg/jZmAjYrXYRwkvxehl9IcHmOgNrHV3INdrSTdZ\n"
  "ZCVLL74tuMqhMMm/NJ0tdEmWgpJe+/0dky2F2gAB+mFXlyzFvCLoyS2Vl0PW/BxM\n"
  "Ly5t+scmAbgni9gzmFTNhbKHd0s2UE395z4ra6fUdZ0BClFgMDvUnb6kJ/uyKRSa\n"
  "h7uQbWFJbA8aNgGLvfTf6o9n+GwbZkcgtBgIVENt8wzqg2I=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEnDCCBAWgAwIBAgIQdTN9mrDhIzuuLX3kRpFi1DANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNMDUwMTE5MDAwMDAwWhcNMTUwMTE4MjM1OTU5WjCBsDELMAkGA1UEBhMCVVMx\n"
  "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
  "dCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cu\n"
  "dmVyaXNpZ24uY29tL3JwYSAoYykwNTEqMCgGA1UEAxMhVmVyaVNpZ24gQ2xhc3Mg\n"
  "MyBTZWN1cmUgU2VydmVyIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\n"
  "AQEAlcMhEo5AxQ0BX3ZeZpTZcyxYGSK4yfx6OZAqd3J8HT732FXjr0LLhzAC3Fus\n"
  "cOa4RLQrNeuT0hcFfstG1lxToDJRnXRkWPkMmgDqXkRJZHL0zRDihQr5NO6ziGap\n"
  "paRa0A6Yf1gNK1K7hql+LvqySHyN2y1fAXWijQY7i7RhB8m+Ipn4G9G1V2YETTX0\n"
  "kXGWtZkIJZuXyDrzILHdnpgMSmO3ps6wAc74k2rzDG6fsemEe4GYQeaB3D0s57Rr\n"
  "4578CBbXs9W5ZhKZfG1xyE2+xw/j+zet1XWHIWuG0EQUWlR5OZZpVsm5Mc2JYVjh\n"
  "2XYFBa33uQKvp/1HkaIiNFox0QIDAQABo4IBgTCCAX0wEgYDVR0TAQH/BAgwBgEB\n"
  "/wIBADBEBgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0\n"
  "dHBzOi8vd3d3LnZlcmlzaWduLmNvbS9ycGEwMQYDVR0fBCowKDAmoCSgIoYgaHR0\n"
  "cDovL2NybC52ZXJpc2lnbi5jb20vcGNhMy5jcmwwDgYDVR0PAQH/BAQDAgEGMBEG\n"
  "CWCGSAGG+EIBAQQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UEAxMRQ2xhc3Mz\n"
  "Q0EyMDQ4LTEtNDUwHQYDVR0OBBYEFG/sr6DdiqTv9SoQZy0/VYK81+8lMIGABgNV\n"
  "HSMEeTB3oWOkYTBfMQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIElu\n"
  "Yy4xNzA1BgNVBAsTLkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlv\n"
  "biBBdXRob3JpdHmCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQEFBQADgYEA\n"
  "w34IRl2RNs9n3Nenr6+4IsOLBHTTsWC85v63RBKBWzFzFGNWxnIu0RoDQ1w4ClBK\n"
  "Tc3athmo9JkNr+P32PF1KGX2av6b9L1S2T/L2hbLpZ4ujmZSeD0m+v6UNohKlV4q\n"
  "TBnvbvqCPy0D79YoszcYz0KyNCFkR9MgazpM3OYDkAw=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzEL\n"
  "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQL\n"
  "Ey5DbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9y\n"
  "aXR5MB4XDTk2MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UE\n"
  "BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGf\n"
  "MA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69q\n"
  "RUCPhAwL0TPZ2RHP7gJYHyX3KqhEBarsAx94f56TuZoAqiN91qyFomNFx3In\n"
  "zPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/isI19wKTakyYbnsZogy1Olhec9vn2a\n"
  "/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0GCSqGSIb3DQEBAgUAA4GBALtM\n"
  "EivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Dolbwdj2wsqFHMc9ikwFPw\n"
  "TtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNycAA9WjQKZ7aKQRUzk\n"
  "uxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test CACert chain with RSA-MD5 signature. */
static const char *cacertrsamd5[] = {
  /* chain[0] (EE cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE3zCCAsegAwIBAgICbmgwDQYJKoZIhvcNAQEFBQAwVDEUMBIGA1UEChMLQ0Fj\n"
  "ZXJ0IEluYy4xHjAcBgNVBAsTFWh0dHA6Ly93d3cuQ0FjZXJ0Lm9yZzEcMBoGA1UE\n"
  "AxMTQ0FjZXJ0IENsYXNzIDMgUm9vdDAeFw0wOTAxMTYyMjI5NDdaFw0xMTAxMTYy\n"
  "MjI5NDdaMBsxGTAXBgNVBAMTEGZyeS5zZXJ2ZXJhbWEuZGUwgZ8wDQYJKoZIhvcN\n"
  "AQEBBQADgY0AMIGJAoGBAMh3WSR8Dxw/zDAZpHMjA1To0HJIaoyR9TqzQfLgn7Yr\n"
  "sQFrRMefVMWYHiEFAVJTRcO5GuUtkw3IxALLlyNMl7xJbZESzRKw3Tz3NtM3DopB\n"
  "8L7rI8ANy7Hh6P5QRMWJ9OJyiLhSpAi0TuJeGr+kKovHRj64V2/NtoPgDsytHMt9\n"
  "AgMBAAGjggF2MIIBcjAMBgNVHRMBAf8EAjAAMDQGA1UdJQQtMCsGCCsGAQUFBwMC\n"
  "BggrBgEFBQcDAQYJYIZIAYb4QgQBBgorBgEEAYI3CgMDMAsGA1UdDwQEAwIFoDAz\n"
  "BggrBgEFBQcBAQQnMCUwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5v\n"
  "cmcvMIHpBgNVHREEgeEwgd6CEGZyeS5zZXJ2ZXJhbWEuZGWgHgYIKwYBBQUHCAWg\n"
  "EgwQZnJ5LnNlcnZlcmFtYS5kZYIQZnJ5LnNlcnZlcmFtYS5kZaAeBggrBgEFBQcI\n"
  "BaASDBBmcnkuc2VydmVyYW1hLmRlghIqLmZyeS5zZXJ2ZXJhbWEuZGWgIAYIKwYB\n"
  "BQUHCAWgFAwSKi5mcnkuc2VydmVyYW1hLmRlggd6cHViLmRloBUGCCsGAQUFBwgF\n"
  "oAkMB3pwdWIuZGWCCSouenB1Yi5kZaAXBggrBgEFBQcIBaALDAkqLnpwdWIuZGUw\n"
  "DQYJKoZIhvcNAQEFBQADggIBAEWSsOlLbjdRjijMmOnDc2RcLQ5PQC9pjUW+bzGR\n"
  "KTJbf8Hf/wSdmHAam+UsIM6HzdQVi058dGyb8/NJQJD+9Dgv1m57x1prLerkt6xq\n"
  "UQCYmOpMxCJOykLqzEUnou9WtL5FaD+wBlOuqWFy0Cy2O3LHXkSkaMR+gdxC4pkI\n"
  "wSkI2SDdC0juvnoVI7iBaaIhYI/1FwV56hc6lxsAslf0NbtiiwhneVbHm5XRK1d4\n"
  "tabVKwOHnEuDyAnZd1yM1EqXKz+NwBlhoKWhC0fVUByID5A2WGEejBJcW/lVrYft\n"
  "4+sJpnwS+/VDS5yrDXMqMdYGE8TVMy7RsaoUdaeFQYv4Go48BBGDJB5uEkBJiSq8\n"
  "ViZA4iEKujBa5zKJ+CZXy3D/eHLBKUL+ayc9dLeeFTPZU0jYb83kE1wtlnWwF4J1\n"
  "8lUQI10nLFg+ALoZoAmFZej19XgbyG6im+ZRFuwrpV6F3HJRP+AMNInsLoQTuD9I\n"
  "l2gftVaIU1MqUmVMBcUeeNXG1BZ9vRonKzAC4Otfk1B6aW4Lz0E+sZ+HfCMicD3j\n"
  "N01KAeNZ64j8emgnLffurb7qUWbanTpMEzxrelBRufxJkXcn6BcFcxPBVgFnsMgF\n"
  "tP7e7N/cm55pI8Et+Gjp+ORJetSio118yu9bf7etSAJWOS6tQ2Ac7JeKP+a8jsvq\n"
  "Uyx7\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (Class 3 CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCDCCA/CgAwIBAgIBATANBgkqhkiG9w0BAQQFADB5MRAwDgYDVQQKEwdSb290\n"
  "IENBMR4wHAYDVQQLExVodHRwOi8vd3d3LmNhY2VydC5vcmcxIjAgBgNVBAMTGUNB\n"
  "IENlcnQgU2lnbmluZyBBdXRob3JpdHkxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA\n"
  "Y2FjZXJ0Lm9yZzAeFw0wNTEwMTQwNzM2NTVaFw0zMzAzMjgwNzM2NTVaMFQxFDAS\n"
  "BgNVBAoTC0NBY2VydCBJbmMuMR4wHAYDVQQLExVodHRwOi8vd3d3LkNBY2VydC5v\n"
  "cmcxHDAaBgNVBAMTE0NBY2VydCBDbGFzcyAzIFJvb3QwggIiMA0GCSqGSIb3DQEB\n"
  "AQUAA4ICDwAwggIKAoICAQCrSTURSHzSJn5TlM9Dqd0o10Iqi/OHeBlYfA+e2ol9\n"
  "4fvrcpANdKGWZKufoCSZc9riVXbHF3v1BKxGuMO+f2SNEGwk82GcwPKQ+lHm9WkB\n"
  "Y8MPVuJKQs/iRIwlKKjFeQl9RrmK8+nzNCkIReQcn8uUBByBqBSzmGXEQ+xOgo0J\n"
  "0b2qW42S0OzekMV/CsLj6+YxWl50PpczWejDAz1gM7/30W9HxM3uYoNSbi4ImqTZ\n"
  "FRiRpoWSR7CuSOtttyHshRpocjWr//AQXcD0lKdq1TuSfkyQBX6TwSyLpI5idBVx\n"
  "bgtxA+qvFTia1NIFcm+M+SvrWnIl+TlG43IbPgTDZCciECqKT1inA62+tC4T7V2q\n"
  "SNfVfdQqe1z6RgRQ5MwOQluM7dvyz/yWk+DbETZUYjQ4jwxgmzuXVjit89Jbi6Bb\n"
  "6k6WuHzX1aCGcEDTkSm3ojyt9Yy7zxqSiuQ0e8DYbF/pCsLDpyCaWt8sXVJcukfV\n"
  "m+8kKHA4IC/VfynAskEDaJLM4JzMl0tF7zoQCqtwOpiVcK01seqFK6QcgCExqa5g\n"
  "eoAmSAC4AcCTY1UikTxW56/bOiXzjzFU6iaLgVn5odFTEcV7nQP2dBHgbbEsPyyG\n"
  "kZlxmqZ3izRg0RS0LKydr4wQ05/EavhvE/xzWfdmQnQeiuP43NJvmJzLR5iVQAX7\n"
  "6QIDAQABo4G/MIG8MA8GA1UdEwEB/wQFMAMBAf8wXQYIKwYBBQUHAQEEUTBPMCMG\n"
  "CCsGAQUFBzABhhdodHRwOi8vb2NzcC5DQWNlcnQub3JnLzAoBggrBgEFBQcwAoYc\n"
  "aHR0cDovL3d3dy5DQWNlcnQub3JnL2NhLmNydDBKBgNVHSAEQzBBMD8GCCsGAQQB\n"
  "gZBKMDMwMQYIKwYBBQUHAgEWJWh0dHA6Ly93d3cuQ0FjZXJ0Lm9yZy9pbmRleC5w\n"
  "aHA/aWQ9MTAwDQYJKoZIhvcNAQEEBQADggIBAH8IiKHaGlBJ2on7oQhy84r3HsQ6\n"
  "tHlbIDCxRd7CXdNlafHCXVRUPIVfuXtCkcKZ/RtRm6tGpaEQU55tiKxzbiwzpvD0\n"
  "nuB1wT6IRanhZkP+VlrRekF490DaSjrxC1uluxYG5sLnk7mFTZdPsR44Q4Dvmw2M\n"
  "77inYACHV30eRBzLI++bPJmdr7UpHEV5FpZNJ23xHGzDwlVks7wU4vOkHx4y/CcV\n"
  "Bc/dLq4+gmF78CEQGPZE6lM5+dzQmiDgxrvgu1pPxJnIB721vaLbLmINQjRBvP+L\n"
  "ivVRIqqIMADisNS8vmW61QNXeZvo3MhN+FDtkaVSKKKs+zZYPumUK5FQhxvWXtaM\n"
  "zPcPEAxSTtAWYeXlCmy/F8dyRlecmPVsYGN6b165Ti/Iubm7aoW8mA3t+T6XhDSU\n"
  "rgCvoeXnkm5OvfPi2RSLXNLrAWygF6UtEOucekq9ve7O/e0iQKtwOIj1CodqwqsF\n"
  "YMlIBdpTwd5Ed2qz8zw87YC8pjhKKSRf/lk7myV6VmMAZLldpGJ9VzZPrYPvH5JT\n"
  "oI53V93lYRE9IwCQTDz6o2CTBKOvNfYOao9PSmCnhQVsRqGP9Md246FZV/dxssRu\n"
  "FFxtbUFm3xuTsdQAw+7Lzzw9IYCpX2Nl/N3gX6T0K/CFcUHUZyX7GrGXrtaZghNB\n"
  "0m6lG5kngOcLqagA\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (Root CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIHPTCCBSWgAwIBAgIBADANBgkqhkiG9w0BAQQFADB5MRAwDgYDVQQKEwdSb290\n"
  "IENBMR4wHAYDVQQLExVodHRwOi8vd3d3LmNhY2VydC5vcmcxIjAgBgNVBAMTGUNB\n"
  "IENlcnQgU2lnbmluZyBBdXRob3JpdHkxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA\n"
  "Y2FjZXJ0Lm9yZzAeFw0wMzAzMzAxMjI5NDlaFw0zMzAzMjkxMjI5NDlaMHkxEDAO\n"
  "BgNVBAoTB1Jvb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEi\n"
  "MCAGA1UEAxMZQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJ\n"
  "ARYSc3VwcG9ydEBjYWNlcnQub3JnMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC\n"
  "CgKCAgEAziLA4kZ97DYoB1CW8qAzQIxL8TtmPzHlawI229Z89vGIj053NgVBlfkJ\n"
  "8BLPRoZzYLdufujAWGSuzbCtRRcMY/pnCujW0r8+55jE8Ez64AO7NV1sId6eINm6\n"
  "zWYyN3L69wj1x81YyY7nDl7qPv4coRQKFWyGhFtkZip6qUtTefWIonvuLwphK42y\n"
  "fk1WpRPs6tqSnqxEQR5YYGUFZvjARL3LlPdCfgv3ZWiYUQXw8wWRBB0bF4LsyFe7\n"
  "w2t6iPGwcswlWyCR7BYCEo8y6RcYSNDHBS4CMEK4JZwFaz+qOqfrU0j36NK2B5jc\n"
  "G8Y0f3/JHIJ6BVgrCFvzOKKrF11myZjXnhCLotLddJr3cQxyYN/Nb5gznZY0dj4k\n"
  "epKwDpUeb+agRThHqtdB7Uq3EvbXG4OKDy7YCbZZ16oE/9KTfWgu3YtLq1i6L43q\n"
  "laegw1SJpfvbi1EinbLDvhG+LJGGi5Z4rSDTii8aP8bQUWWHIbEZAWV/RRyH9XzQ\n"
  "QUxPKZgh/TMfdQwEUfoZd9vUFBzugcMd9Zi3aQaRIt0AUMyBMawSB3s42mhb5ivU\n"
  "fslfrejrckzzAeVLIL+aplfKkQABi6F1ITe1Yw1nPkZPcCBnzsXWWdsC4PDSy826\n"
  "YreQQejdIOQpvGQpQsgi3Hia/0PsmBsJUUtaWsJx8cTLc6nloQsCAwEAAaOCAc4w\n"
  "ggHKMB0GA1UdDgQWBBQWtTIb1Mfz4OaO873SsDrusjkY0TCBowYDVR0jBIGbMIGY\n"
  "gBQWtTIb1Mfz4OaO873SsDrusjkY0aF9pHsweTEQMA4GA1UEChMHUm9vdCBDQTEe\n"
  "MBwGA1UECxMVaHR0cDovL3d3dy5jYWNlcnQub3JnMSIwIAYDVQQDExlDQSBDZXJ0\n"
  "IFNpZ25pbmcgQXV0aG9yaXR5MSEwHwYJKoZIhvcNAQkBFhJzdXBwb3J0QGNhY2Vy\n"
  "dC5vcmeCAQAwDwYDVR0TAQH/BAUwAwEB/zAyBgNVHR8EKzApMCegJaAjhiFodHRw\n"
  "czovL3d3dy5jYWNlcnQub3JnL3Jldm9rZS5jcmwwMAYJYIZIAYb4QgEEBCMWIWh0\n"
  "dHBzOi8vd3d3LmNhY2VydC5vcmcvcmV2b2tlLmNybDA0BglghkgBhvhCAQgEJxYl\n"
  "aHR0cDovL3d3dy5jYWNlcnQub3JnL2luZGV4LnBocD9pZD0xMDBWBglghkgBhvhC\n"
  "AQ0ESRZHVG8gZ2V0IHlvdXIgb3duIGNlcnRpZmljYXRlIGZvciBGUkVFIGhlYWQg\n"
  "b3ZlciB0byBodHRwOi8vd3d3LmNhY2VydC5vcmcwDQYJKoZIhvcNAQEEBQADggIB\n"
  "ACjH7pyCArpcgBLKNQodgW+JapnM8mgPf6fhjViVPr3yBsOQWqy1YPaZQwGjiHCc\n"
  "nWKdpIevZ1gNMDY75q1I08t0AoZxPuIrA2jxNGJARjtT6ij0rPtmlVOKTV39O9lg\n"
  "18p5aTuxZZKmxoGCXJzN600BiqXfEVWqFcofN8CCmHBh22p8lqOOLlQ+TyGpkO/c\n"
  "gr/c6EWtTZBzCDyUZbAEmXZ/4rzCahWqlwQ3JNgelE5tDlG+1sSPypZt90Pf6DBl\n"
  "Jzt7u0NDY8RD97LsaMzhGY4i+5jhe1o+ATc7iwiwovOVThrLm82asduycPAtStvY\n"
  "sONvRUgzEv/+PDIqVPfE94rwiCPCR/5kenHA0R6mY7AHfqQv0wGP3J8rtsYIqQ+T\n"
  "SCX8Ev2fQtzzxD72V7DX3WnRBnc0CkvSyqD/HMaMyRa+xMwyN2hzXwj7UfdJUzYF\n"
  "CpUCTPJ5GhD22Dp1nPMd8aINcGeGG7MW9S/lpOt5hvk9C8JzC6WZrG/8Z7jlLwum\n"
  "GCSNe9FINSkYQKyTYOGWhlC0elnYjyELn8+CkcY7v2vcB5G5l1YjqrZslMZIBjzk\n"
  "zk6q5PYvCdxTby78dOs6Y5nCpqyJvKeyRKANihDjbPIky/qbn3BHLt4Ui9SyIAmW\n"
  "omTxJBzcoTWcFbLUvFUufQb1nA5V9FrWk9p2rSVzTMVD\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test Certicom cert with ECC-SHA256 signature. */
static const char *ecc_cert[] = {
  /* chain[0] (ECC cert) */
"-----BEGIN CERTIFICATE-----\n"
"MIICbzCCAhSgAwIBAgIIZLkW6EZO5PQwCgYIKoZIzj0EAwIwgZsxFDASBgNVBAsT\n"
"C1NBTVBMRSBPTkxZMRcwFQYDVQQKEw5DZXJ0aWNvbSBDb3JwLjEQMA4GA1UEBxMH\n"
"VG9yb250bzEQMA4GA1UEBBMHT250YXJpbzE5MDcGA1UEAxMwdGxzLnNlY2cub3Jn\n"
"IEVDQyBzZWNwMjU2cjEgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MQswCQYDVQQGEwJD\n"
"QTAeFw0wOTA1MDcwMDAwMDBaFw0xNTA1MDEwMDAwMDBaMIGYMRQwEgYDVQQLEwtT\n"
"QU1QTEUgT05MWTEXMBUGA1UEChMOQ2VydGljb20gQ29ycC4xEDAOBgNVBAcTB1Rv\n"
"cm9udG8xEDAOBgNVBAgTB09udGFyaW8xNjA0BgNVBAMTLXRscy5zZWNnLm9yZyBF\n"
"Q0Mgc2VjcDI1NnIxIFNlcnZlciBDZXJ0aWZpY2F0ZTELMAkGA1UEBhMCQ0EwWTAT\n"
"BgcqhkjOPQIBBggqhkjOPQMBBwNCAATf63kPhr3D6a2scrHWVr0oOXQMnBDT6Jv/\n"
"ifqzt4/xTbXsZNEyD96nyh82sk0tM+FVfBlsIwGc7vqBfyq0mC/Io0MwQTAOBgNV\n"
"HQ8BAf8EBAMCA4gwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwEwFwYDVR0RBBAwDoIM\n"
"dGxzLnNlY2cub3JnMAoGCCqGSM49BAMCA0kAMEYCIQDfacZHsdsj6SXQ2hyJS4Do\n"
"SMclqGLo2Sop7hfAeEJA+wIhAOMo7eLya44SIcuzrLBpg29g5ZzYOeuEzRcg9mch\n"
"AB1w\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIICTjCCAfagAwIBAgIICvq6Bj3Av6EwCQYHKoZIzj0EATCBmzEUMBIGA1UECxML\n"
"U0FNUExFIE9OTFkxFzAVBgNVBAoTDkNlcnRpY29tIENvcnAuMRAwDgYDVQQHEwdU\n"
"b3JvbnRvMRAwDgYDVQQEEwdPbnRhcmlvMTkwNwYDVQQDEzB0bHMuc2VjZy5vcmcg\n"
"RUNDIHNlY3AyNTZyMSBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkxCzAJBgNVBAYTAkNB\n"
"MB4XDTA2MDUwMTAwMDAwMFoXDTE1MDUwMTAwMDAwMFowgZsxFDASBgNVBAsTC1NB\n"
"TVBMRSBPTkxZMRcwFQYDVQQKEw5DZXJ0aWNvbSBDb3JwLjEQMA4GA1UEBxMHVG9y\n"
"b250bzEQMA4GA1UEBBMHT250YXJpbzE5MDcGA1UEAxMwdGxzLnNlY2cub3JnIEVD\n"
"QyBzZWNwMjU2cjEgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MQswCQYDVQQGEwJDQTBZ\n"
"MBMGByqGSM49AgEGCCqGSM49AwEHA0IABB2oofFVa6akTK6hpaJLs+6skdhn0sQp\n"
"uJwVwG99T0VZY8v7q6NMIWrpYQFmOxQyVVNlxWWyr2cLYJTyqx/zuDejIzAhMA4G\n"
"A1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MAkGByqGSM49BAEDRwAwRAIg\n"
"W2KAhfAKWFoh47A7muk8K10cGqOKRtb9lCgdOltj19oCIG+ZJQv5m+RnL4X2Ti0y\n"
"ZJzOOuzBQVGiUFwZdn1dLv4X\n"
"-----END CERTIFICATE-----\n",
  NULL
};

static const char *verisign_com_chain_g5[] = {
  /* chain[0] */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIFXjCCBEagAwIBAgIQHYWDpKNVUzEFx4Pq8yjxbTANBgkqhkiG9w0BAQUFADCB\n"
 "tTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
 "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
 "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMm\n"
 "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwHhcNMTQwMjI3\n"
 "MDAwMDAwWhcNMTUwMjI4MjM1OTU5WjBnMQswCQYDVQQGEwJVUzETMBEGA1UECBMK\n"
 "V2FzaGluZ3RvbjEQMA4GA1UEBxQHU2VhdHRsZTEYMBYGA1UEChQPQW1hem9uLmNv\n"
 "bSBJbmMuMRcwFQYDVQQDFA53d3cuYW1hem9uLmNvbTCCASIwDQYJKoZIhvcNAQEB\n"
 "BQADggEPADCCAQoCggEBAJdfieOPrf4Arf1Iled/ii97407ZnjpaB5xxm49Q4Pz3\n"
 "+5xmD0LYre7Cjn1A7W3ZlHki5zFVZpW9Jb/3PfSEDY5slyjkLD2jdl2gVefSthSZ\n"
 "tYxb5eYv79tIEN0U9AZ8/VaGwUokl8n1MitcECxNLMe4LqoVmS29nXITTTzX5t3I\n"
 "4dUeMBDNI+xgVpJSpxwzA+/L+wxoj5Sb4YJ/Y+iUknCkjX6PpaZMRWBEE0dqvG02\n"
 "qlxXesAV0nmKYvjbtqAyoW6vgjP85h6gJEESIqTTZy1HOgFpO8XT05CpGDcjhP1s\n"
 "TvXF7Vx1aj+xDidGLLW188G35oFKIhEyHQV2V7vzRAUCAwEAAaOCAbUwggGxMFAG\n"
 "A1UdEQRJMEeCEXVlZGF0YS5hbWF6b24uY29tggphbWF6b24uY29tgghhbXpuLmNv\n"
 "bYIMd3d3LmFtem4uY29tgg53d3cuYW1hem9uLmNvbTAJBgNVHRMEAjAAMA4GA1Ud\n"
 "DwEB/wQEAwIFoDAdBgNVHSUEFjAUBggrBgEFBQcDAQYIKwYBBQUHAwIwQwYDVR0g\n"
 "BDwwOjA4BgpghkgBhvhFAQc2MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZl\n"
 "cmlzaWduLmNvbS9jcHMwHwYDVR0jBBgwFoAUDURcFlNEwYJ+HSCrJfQBY9i+eaUw\n"
 "RQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL1NWUlNlY3VyZS1HMy1jcmwudmVyaXNp\n"
 "Z24uY29tL1NWUlNlY3VyZUczLmNybDB2BggrBgEFBQcBAQRqMGgwJAYIKwYBBQUH\n"
 "MAGGGGh0dHA6Ly9vY3NwLnZlcmlzaWduLmNvbTBABggrBgEFBQcwAoY0aHR0cDov\n"
 "L1NWUlNlY3VyZS1HMy1haWEudmVyaXNpZ24uY29tL1NWUlNlY3VyZUczLmNlcjAN\n"
 "BgkqhkiG9w0BAQUFAAOCAQEAOeZfjkI0yR/nutCMHp5/uB/evkB8qIYxh1KKbhPB\n"
 "TmpykmJxiLKrBBcBBu9kW5lMbNPSNclE4sCyN0dxCJHwPm7ubNUxsmDSTPTJOx0M\n"
 "zl0WZVaZ7eX3nw1kj8jSoK0f5n87RzKK85MwBFsEn73Z2pDvxTcd72BE0T1UJLcU\n"
 "2A5uHAJyvm2QpOWBIRKlJHIHCcu5xjj5yLnGO9id0cjOjEgj9f1Mo4hzawL5vQfk\n"
 "o/xFxAsA70bk2trv54kgLvhmAW+B6OpN3Z/xB4mWNOw3G8bg/u+pCLvd8KRO5V8K\n"
 "TlgO1NTaOgYC6OAF3USNMhuNZh6h5tWA3mA8rFr8ZsayhA==\n"
 "-----END CERTIFICATE-----\n",
  /* chain[1] */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIF7DCCBNSgAwIBAgIQbsx6pacDIAm4zrz06VLUkTANBgkqhkiG9w0BAQUFADCB\n"
 "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
 "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
 "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
 "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
 "aG9yaXR5IC0gRzUwHhcNMTAwMjA4MDAwMDAwWhcNMjAwMjA3MjM1OTU5WjCBtTEL\n"
 "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
 "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
 "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMmVmVy\n"
 "aVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwggEiMA0GCSqGSIb3\n"
 "DQEBAQUAA4IBDwAwggEKAoIBAQCxh4QfwgxF9byrJZenraI+nLr2wTm4i8rCrFbG\n"
 "5btljkRPTc5v7QlK1K9OEJxoiy6Ve4mbE8riNDTB81vzSXtig0iBdNGIeGwCU/m8\n"
 "f0MmV1gzgzszChew0E6RJK2GfWQS3HRKNKEdCuqWHQsV/KNLO85jiND4LQyUhhDK\n"
 "tpo9yus3nABINYYpUHjoRWPNGUFP9ZXse5jUxHGzUL4os4+guVOc9cosI6n9FAbo\n"
 "GLSa6Dxugf3kzTU2s1HTaewSulZub5tXxYsU5w7HnO1KVGrJTcW/EbGuHGeBy0RV\n"
 "M5l/JJs/U0V/hhrzPPptf4H1uErT9YU3HLWm0AnkGHs4TvoPAgMBAAGjggHfMIIB\n"
 "2zA0BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZlcmlz\n"
 "aWduLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEAMHAGA1UdIARpMGcwZQYLYIZIAYb4\n"
 "RQEHFwMwVjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2Nw\n"
 "czAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQG\n"
 "A1UdHwQtMCswKaAnoCWGI2h0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMtZzUu\n"
 "Y3JsMA4GA1UdDwEB/wQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglp\n"
 "bWFnZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNo\n"
 "dHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjAoBgNVHREEITAfpB0w\n"
 "GzEZMBcGA1UEAxMQVmVyaVNpZ25NUEtJLTItNjAdBgNVHQ4EFgQUDURcFlNEwYJ+\n"
 "HSCrJfQBY9i+eaUwHwYDVR0jBBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJ\n"
 "KoZIhvcNAQEFBQADggEBAAyDJO/dwwzZWJz+NrbrioBL0aP3nfPMU++CnqOh5pfB\n"
 "WJ11bOAdG0z60cEtBcDqbrIicFXZIDNAMwfCZYP6j0M3m+oOmmxw7vacgDvZN/R6\n"
 "bezQGH1JSsqZxxkoor7YdyT3hSaGbYcFQEFn0Sc67dxIHSLNCwuLvPSxe/20majp\n"
 "dirhGi2HbnTTiN0eIsbfFrYrghQKlFzyUOyvzv9iNw2tZdMGQVPtAhTItVgooazg\n"
 "W+yzf5VK+wPIrSbb5mZ4EkrZn0L74ZjmQoObj49nJOhhGbXdzbULJgWOw27EyHW4\n"
 "Rs/iGAZeqa6ogZpHFt4MKGwlJ7net4RYxh84HqTEy2Y=\n"
 "-----END CERTIFICATE-----\n",
  /* chain[2] */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIExjCCBC+gAwIBAgIQNZcxh/OHOgcyfs5YDJt+2jANBgkqhkiG9w0BAQUFADBf\n"
 "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
 "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
 "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
 "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
 "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
 "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
 "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
 "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
 "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
 "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
 "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
 "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
 "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
 "AAGjggGRMIIBjTAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
 "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjA9\n"
 "BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVy\n"
 "aXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwNAYD\n"
 "VR0lBC0wKwYJYIZIAYb4QgQBBgpghkgBhvhFAQgBBggrBgEFBQcDAQYIKwYBBQUH\n"
 "AwIwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAHBgUr\n"
 "DgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVyaXNp\n"
 "Z24uY29tL3ZzbG9nby5naWYwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhho\n"
 "dHRwOi8vb2NzcC52ZXJpc2lnbi5jb20wDQYJKoZIhvcNAQEFBQADgYEADyWuSO0b\n"
 "M4VMDLXC1/5N1oMoTEFlYAALd0hxgv5/21oOIMzS6ke8ZEJhRDR0MIGBJopK90Rd\n"
 "fjSAqLiD4gnXbSPdie0oCL1jWhFXCMSe2uJoKK/dUDzsgiHYAMJVRFBwQa2DF3m6\n"
 "CPMr3u00HUSe0gST9MsFFy0JLS1j7/YmC3s=\n"
 "-----END CERTIFICATE-----\n",
 NULL,
  /* chain[4] - alt CA */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIE0zCCA7ugAwIBAgIQGNrRniZ96LtKIVjNzGs7SjANBgkqhkiG9w0BAQUFADCB\n"
 "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
 "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
 "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
 "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
 "aG9yaXR5IC0gRzUwHhcNMDYxMTA4MDAwMDAwWhcNMzYwNzE2MjM1OTU5WjCByjEL\n"
 "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
 "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2ln\n"
 "biwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJp\n"
 "U2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9y\n"
 "aXR5IC0gRzUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1\n"
 "nmAMqudLO07cfLw8RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbex\n"
 "t0uz/o9+B1fs70PbZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIz\n"
 "SdhDY2pSS9KP6HBRTdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQG\n"
 "BO+QueQA5N06tRn/Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+\n"
 "rCpSx4/VBEnkjWNHiDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/\n"
 "NIeWiu5T6CUVAgMBAAGjgbIwga8wDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8E\n"
 "BAMCAQYwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAH\n"
 "BgUrDgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVy\n"
 "aXNpZ24uY29tL3ZzbG9nby5naWYwHQYDVR0OBBYEFH/TZafC3ey78DAJ80M5+gKv\n"
 "MzEzMA0GCSqGSIb3DQEBBQUAA4IBAQCTJEowX2LP2BqYLz3q3JktvXf2pXkiOOzE\n"
 "p6B4Eq1iDkVwZMXnl2YtmAl+X6/WzChl8gGqCBpH3vn5fJJaCGkgDdk+bW48DW7Y\n"
 "5gaRQBi5+MHt39tBquCWIMnNZBU4gcmU7qKEKQsTb47bDN0lAtukixlE0kF6BWlK\n"
 "WE9gyn6CagsCqiUXObXbf+eEZSqVir2G3l6BFoMtEMze/aiCKm0oHw0LxOXnGiYZ\n"
 "4fQRbxC1lfznQgUy286dUV4otp6F01vvpX1FQHKOtw5rDgb7MzVIcbidJ4vEZV8N\n"
 "hnacRHr2lVz2XTIIM6RUthg/aFzyQkqFOFSDX9HoLPKsEdao7WNq\n"
 "-----END CERTIFICATE-----\n",
 NULL
};

/* TLS feature extension chains */

/* intermediate has no TLS feature extension */
static const char *tls_feat_inter_no_ext[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEPzCCAqegAwIBAgIMV2qxjR6UmcYfmTkvMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTYwNjIyMTU0MTAxWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAvRR62ekBwabwe49K5Yhj4EGFVapmZvQbZy/lh+OIDjysGRbqDEOk02lh5gvU\n"
	"8LByMvLu2/3adrpnmU0wRlw0wF63efyzFQgEvHew5dTEOXXksxLtcNQUz4CgdUmt\n"
	"ckpp3w1kO7zaK9glGLKWG9QstgJIPv0ggoeFBaPPKPbhTIqdDDdWLEXzFwboOLYL\n"
	"QdaVWtSgGPQnVvDdGaKhpkvZndzF1pwRCu0szt7jktW31kOJq3sCNjt5AwjkGjKV\n"
	"n4rNSz9PfxQ6npzt+fQ4w6VTJyqGZQjopxPkFkXieCL8ls6XqDJeqFXENpdpRgmb\n"
	"hWiCWBhHxSon2yUefjcV7w1b3r/r5B9P111GfpIaaFkzlF2HG1qDadCwD9A5Egjb\n"
	"KWoxyzVsYcnu+ODP/zJ3DHUq/tWby8B5ZQ0HtuvHoMJ+/S6vEA3QKKwU8kr+QX7S\n"
	"tmrAo9mWgHlkinLEqfuYApmSlYoSEEG0l+p/sSnvwq8N7doxRbcvhUng/Uiq1OIp\n"
	"FprBAgMBAAGjgZQwgZEwGgYIKwYBBQUHARgEDjAMAgEFAgEGAgEHAgEIMAwGA1Ud\n"
	"EwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAw\n"
	"HQYDVR0OBBYEFBJiXEt+XVJWrIPFtet3wyFI/OE5MB8GA1UdIwQYMBaAFJRCUeHF\n"
	"92wGbuofUbyXXlm2fqzgMA0GCSqGSIb3DQEBCwUAA4IBgQC1HZ3QO4OCMgVFUVFT\n"
	"+lLArzr1ZNqYxbYLDW8j77YYv4HNe822DtoH118yEpXmOsUvJbAZrEqaLpfRUDI6\n"
	"wTd/755sm7mqP0bpG3J5x+WdE00iTQW9mChQ5NGNqqB1eXwn+dEtEvsSDOtVMuxd\n"
	"0jbiR6YQINFsCCUdLJ3uFfwj23084BBC2RqBb3JEU6VaD5CZfTo6q/kn6ZjE057e\n"
	"kIoVqhLDTZvuHQTj2LpNiEcVd1z9mjy3AWuvN38NLUQZJb75Rnpd8GQG6lFQW/Qq\n"
	"ykbzt3OF1tX5SWCUux1yYXA5ng4X85kWArkOjJL6IOKlVAK96fN4v2uymDI5D+p+\n"
	"ynbyhescPkmbdNG5cJCxu2gArEBcfbxkTqQpOgAhNq5P2Bphd+40HTFa/x3O3uDs\n"
	"PCPlf4tpGBJ2hXWnRheHIqyGX0hanekb1fSN2aMdat5V4w9bAx0oSd3pohmysuf8\n"
	"uboqRVcJ1itfSb/uJyTt6l+6VpDKw4M3sg9GQqx2jiLBU4k=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIID/zCCAmegAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NDEwMVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA2vvT8gYwa/7P0JSC\n"
	"fDk7nliapNHj5yFDuf3zj8Pqm1et+CbFeoF6r46cb5+xWa9IBdiZzQMgf7ehrNyB\n"
	"98s8ynYf0pR3uW8J1hEKc06alJwlZwxYywRU+83ROB2Es6rkGKUhjCVdi7p6ddlu\n"
	"IfjbQe2vZgb8b1Naa9pjs2/jLhbnlkaZSdMKHIcc9+vFS/99pQvCJR0gDiMNaYTm\n"
	"p+tiZ81t5XFhSjD75z0S2NNBLwA4IIG/i17UoecatYzjDOt+B9hH/1oB+O10Orhk\n"
	"B1ktekP89wFq89tc7tGx885lzcv6WlQqt75KxZfgZyxWfhrWB8G/atelzZg+7nQe\n"
	"NYqhgYz+9fiOkoc/93jLoWssQPOTDzCFZ96qHCKrrs/P3WNLirXDR9Q+cAoxhDW0\n"
	"V3p8Ar9WaJimx7+oeG0rxmX4Zp5awa5x6EJEBBpCmje8xPuFRuLZsU+W7M4nNMSu\n"
	"9vS4V2fb0Tx68wz5v3n3ifMHKk3Xtdt60IJT8/blKhghFi4HAgMBAAGjZDBiMA8G\n"
	"A1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUlEJR4cX3\n"
	"bAZu6h9RvJdeWbZ+rOAwHwYDVR0jBBgwFoAUwsbfrUyVADCii3gYFBMkBr4eMo8w\n"
	"DQYJKoZIhvcNAQELBQADggGBAIHldlbqpj1me0d2oHtvmLeqeiWV/RRPGRnNJQc5\n"
	"Iz7R9K+K3P2Mk9pHSNnTx1ewT6dvv5luZK8rSTHW1wf829iXHR1O7ivkGC/wUldc\n"
	"LCFDs8s+n9RW7jrE6Cp20cE7HQruBn4gk4JdRUTwkq/fwmwhdMsZId+79wRr0otj\n"
	"L+n3Vu/0DtGuXHlQvzppihmCkILWS3Uy9iKq205s/2zPO595aA0n49y40pyPdsNr\n"
	"BhDsMf5xmZjVerzurL9S/YHTPmCvNCsX0RwCRQvVFicwyngD4dUW57gGkeUrSmaL\n"
	"zwP+s6YdWYs1itByL7afDULcOwIs84GZ6BBWwN65FBBZTgIjuDwcVFMMt5VGwiOl\n"
	"4G11HAkDOPxlmgzReHw4M4/ynR1/NWEAdMksQqgjj5aUhfZBF9pgO2L9CFzFz/sY\n"
	"uY59mki+wuMYh15j/MQnSnGrd4peA3duikH1mpwuBzehIqh51kfGYsjpajy9qNtB\n"
	"Z4CohSJdLVYUHrq/QIYscunEmg==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID+jCCAmKgAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NDEwMVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAk5rI3effp+0REqgO\n"
	"Vhzb0LoqWBikxW87r3vKHssDA7ZuhuPoVj/tnhYVtTPYN4a6OAsl2UQc82nplKS7\n"
	"m8zsIiAc+SX5leMFJ0WKIZER1MJIr7mK3/iSxFLMoJ57n048beIKJCzEAzMyS/+i\n"
	"RFGq5VsDC2O8hPCrE9Ncen6WzR0mmcmFqhtYd+F2jgvGaNqyv27SrVjI2qIAISSQ\n"
	"8z00k0dwgW7eAs8BYOXzZLKlKMVFRGRWoIlOWBmDpqreZIcid0K4N5Ga4b+owU1t\n"
	"bw9F4q3wPvFqFX7MZK+dpARwNT/jo5ylxYAjz9k1elfCkdf+RqDmFcXEXRDVBIEI\n"
	"IHGJoeiMBMcYY6vpvs1H9Yeo929GFQtPW+yKMM5EvpSoJII/NgoBy4hXxnRvZD6E\n"
	"K1Yo3EfxM2P3CCCUd+Pqq0xmlNODpMcUYmjWRzboK5BmbM9fLoG/POpNm9lUf4Nd\n"
	"C2mCsEvyjmZ9JiETtaiI2FE8SmsGhDyjGOkg5/EnKnOhSis5AgMBAAGjXzBdMBoG\n"
	"CCsGAQUFBwEYBA4wDAIBBQIBBgIBBwIBCDAPBgNVHRMBAf8EBTADAQH/MA8GA1Ud\n"
	"DwEB/wQFAwMHBgAwHQYDVR0OBBYEFMLG361MlQAwoot4GBQTJAa+HjKPMA0GCSqG\n"
	"SIb3DQEBCwUAA4IBgQAJHqCuEWfHglPs6HMaQB/n+SpXGdX6ql3Hw0PjfVwneZaa\n"
	"C/DVVyeCoYn7V1WjGm1PvoDfn5edkMdB8JbfcnSjIG/TCJMnQ3ST9rIT35TEz7yO\n"
	"ptTNRaMkLEJVyScS+ifGvOwIe4T/mERnC6qyskrdOFQVAY2KnYH20cF1sRn7tnnk\n"
	"OJ0uSvzCHghdPIYpQMbGHRKIrFYyLmmbZO02uTnuS5AI0h5R94FBQXtLfVIvBKBW\n"
	"LmLi3A7s0YITOqv3I8nBZNKgQj2aXHGqI7lN8cGDjeHgo5L5Yf0qKTmP+KkauujT\n"
	"4mHc69fmdv4cwKQjoz5rYWW6FwPGGyIE97bKNC52K35G/4M6V40bu3PCLx4kyE5e\n"
	"WtJvoYQox2J/Ht6gwNIfoGHBbHM99rRxYbbxaSkPSdiRUq8TdZ2W4ZJf02Fzbmhe\n"
	"639r3BNCZuYn1Orut5hEe+h5ABbuQYW8ruD55LgNEUMqhvmfxQjabxB4vh1XGyXL\n"
	"zQ778SGDR2hiRF1W4+c=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

/* end certificate has no extension */
static const char *tls_feat_no_ext[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEITCCAomgAwIBAgIMV2qzJyxXCwS84av4MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTYwNjIyMTU0NzUxWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAxAcB54ZehtTEHjz/HaU0lhN/vs5VbUmF//rkkpUx7ApPvJBzHaD4DnALE+8A\n"
	"6G3TDzlwDR7dJw9PPZAbrIs8EsW+W09raL+mp6t2klfCVRJimZCa+4k/vVPqzyZr\n"
	"SOiUYFqhE7q2RfPDzkzJGxj7SEr+7Y7+TaE233lAb0sIZqayq+KUjnYw03nGGHqF\n"
	"qg9fqdrdHild/uW8VhHovpoeYJU381Vv0w71s7Oeq6touq0Qgw7GlX94TctcXUvP\n"
	"wq4VXXToy2s1kr9w9tcvMuXrX360WebhujY22Mkkp98KL8gYrK3N1PJ5adGESYIb\n"
	"XPYROZClmp+nL88rM5xV58CZ2VWkqZkzOSrwIaQ/X5EHq7Cj4oXpM5eq9398JB3H\n"
	"dw7fg0ODyaj0AZXuyg1OK5qbh0KXAUwmpSbSEYqOXJEhO8CSaOC1bMIXLW5s6axv\n"
	"7tkNuH6LHvS7GInZYEKiOlgBRlNH37D4A0JqcSGO3itNVlZb2WqS0zwl+80UwraX\n"
	"9dzhAgMBAAGjdzB1MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0\n"
	"MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFGcXQmzK5g6/NC//Mwc6qN9/jPrD\n"
	"MB8GA1UdIwQYMBaAFDnIakD2t4t0UWvGolQEkhIYF9mdMA0GCSqGSIb3DQEBCwUA\n"
	"A4IBgQB+G/D477nbbY/gxRznLbFi7xMC0arwpuYdWa8BvkPwSACSlJI1heCnDPZk\n"
	"S4jt6SPcSzZzrKtALUxS0MoEAOL9p6pXI9cJRKGmnC5clrtw5rHWqAe2DfvzJCsP\n"
	"yAHcY2g3jnC01knLYy1G0y7phxXAlqiN9MofQuE5j5919wYvPrYfbG+DwKQgbC9l\n"
	"FhTBs7FwSD6hnzd2JK5ldXnAA14Ekh0NLa3fLgd/THiihRIil4VTISYdgwpUfcH1\n"
	"mo+HO6Fc1l7U8HKlzRLoF7BtM/AOLYWLMtf3RiHkoIz1eLK9SPrGxANc/zKDksRy\n"
	"QpZxQj1n3a9m4JRJnbnAxcRGQzB4oe6BroFG3pJH9KhyNaSehDob3ckebt8w/W1Z\n"
	"WqWmPr8lLPO9y2DNYqfuSqMNd9gdUzGc6PB10yyBDnviLjY6JioUxKR5y1FcuvQ3\n"
	"CufS1HyfbF9ITW8peLXryMFen+VwQgtoP5aHISLOMh2+DLEpe6+MHo8HOpELFGsW\n"
	"z/CW3fw=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEHDCCAoSgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NDc1MVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAoJ/AHuj/w4qJQqWh\n"
	"WGIcYCGz/Kshs0/zqWHvOfuJZk9rN9laZ13LKSf8uh79jydrGmEyzPWSVsjXDWBy\n"
	"EdPplNsc9PWx9vOfwsusYOypLWztJF9sMc/fpEMNT7qZTs2XFfAjkMek9U39d5Dc\n"
	"X8isUGmK5QyjstuSQ9lRR0usonv6tF2g356oW6C7D6vFI03/1gj7MljT4pmeiavz\n"
	"BkP5ztNYso1De35Z9c8pQoHkxbPySMBVvDPzEirz28cgF8CJgx3PmXsHMHkfhBwV\n"
	"3HsptSryU3FICTlxmwUBEDm6BecDWtJJrP/GP2xHtbQnfQEVg2uOIKgdIO7Ibr5G\n"
	"zlDJu5dO6NBKSRzjc+w1y20FkVPnocYOvYuV2a1HKOA5d5YnhLvSY7iU3ArNwk4a\n"
	"FHKQNGRmEF4uN6lTYEYwTM46Zib0skVHu/oUlERW/bho43/0b/MzVphiMa0UmMZ3\n"
	"fdOYmvgNrJCWd2Sa3XzSWaWXuRey3L8Pa6QQSv9TQk6tGVjPAgMBAAGjgYAwfjAa\n"
	"BggrBgEFBQcBGAQOMAwCAQUCAQYCAQcCAQgwDwYDVR0TAQH/BAUwAwEB/zAPBgNV\n"
	"HQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBQ5yGpA9reLdFFrxqJUBJISGBfZnTAfBgNV\n"
	"HSMEGDAWgBRbRGd/MX25P8RaPZb9sBLHfi4hQDANBgkqhkiG9w0BAQsFAAOCAYEA\n"
	"srvZMaXdDvUx7jmlVp0VsH1ftHJyvcfU3tv98pPQPRWN/yeBgxKP7aSYmUevKAk+\n"
	"bWig7aXfjt/JEEJkbxoOeNtSpJnsosaQpRnzGqRe0YDfvaiolXf/GjGpbaUX21qW\n"
	"WYW4ND3jE+x38xYYK2LIvmbga0q+D8gIVIXmFJfuPkzeliH72FjgMDXgJ3jHUhCb\n"
	"7ngpToboxWJPNuNBbSCXt2IZgvEimvUCQo5kSGdqVwkUq7KCWuDdE/RtUuQhhruY\n"
	"eNWkIE/H5koJOi+UnxpDI2jrFchAMgH4x88bmfSaUxvEtSKrL/b/aDuuRfrURMP8\n"
	"y8rYCEFigO3e1Y5JF651nw0R5FcdijvD/8nWzP+BBmt+OMY5lQ1Fcmrtiv/YLKlF\n"
	"Ofj2k5rZo+9cjx9xlit2xOOQqPdnNyw4BgkG4rDPzTPiJr5ExBJB7roGKKSmxcJk\n"
	"7goFMr3qx2qiACZCcqIGY2NBCEeGh9y4KySXGI2WO0NLD530+lJb74C9lCukSDrx\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID+jCCAmKgAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NDc1MFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA3OI4hcWn3Sbs4RaA\n"
	"VB6N9KHBNExW1d7c51Plx7pbwUGFfCR2gmRTvIOOVyN4K6l+fx7ZIwSLDQuETR2t\n"
	"ETL2rA4qANIKdoVIDkoQci1tdR9HWYE9o4y1pMhLWO19ZSR525BzAH59ToAC6OKP\n"
	"w3m36hdYd2QaVA6uDknVQBmLwhE8sq+ufTtOjcClnfdfNb2NIGpao77/UQvhKrOF\n"
	"pfvMLaEhlTF365N4aulP5ln7FmSaUHHwQY52TT8hz3x29r4pZKXqV2VnP1rkXUCr\n"
	"ifzd9r57FxkHxpsWq9eXXtFGJmHKmCFP0MHanHNIEpoWbsA1h3C6/ml6FjxfOSDc\n"
	"CagADxW3L+iGUMFXOYAWFTtZZQK/QHw7AcWyLZZQyF6OkttzCLsMBXQ5MOEPsMbY\n"
	"JEONdcr2MjZxeYesaL2fO1SQJrfRJgeOWO4RtBDiVSHKrjBwtKiF7B0fZah2k0jN\n"
	"oxjhxkqyWraoYMHuAXYsi05U2Lz89JhL6YKflM2isFOBirRNAgMBAAGjXzBdMBoG\n"
	"CCsGAQUFBwEYBA4wDAIBBQIBBgIBBwIBCDAPBgNVHRMBAf8EBTADAQH/MA8GA1Ud\n"
	"DwEB/wQFAwMHBgAwHQYDVR0OBBYEFFtEZ38xfbk/xFo9lv2wEsd+LiFAMA0GCSqG\n"
	"SIb3DQEBCwUAA4IBgQBpYKqR4iZnt6Igx/kNDhCE+QDMJjCEe4gp1kZjwatK+9Ke\n"
	"Qpeakq46KkiPrWICsfpqesfLRFMIuF/5M/Rq4wV5ilAX4PQUnUC7di7ehZKZBECW\n"
	"nZ9xzFRY8xZj1K0Em/Re2vAzipuo1UaBhCfAFz82axyLBSr4qACBGjrOXBAeCkzf\n"
	"SfYhi470oKA7AmD6HIVJB1s6kIqpLJGvsFo8mUWhjyYQ++aTRWyL1NvDieGuYIAg\n"
	"1fpUqVRX8YBEkh97QtG7RU8esb0NKhsT0c5NGi8ETAE+gleJJEw92vl1sU7/lYCd\n"
	"sg9F5zeByMphomTENX7AyAm5eSal+BYgiSMuPKRzG9Y1iFzpCP20fcCfQAExPLx4\n"
	"WLILYo6wMuVL4ivz2H/dhwmMKLbccpkI+OI7MTrSHrWRpG2QZvkONCV87ePC5yQ9\n"
	"N6YgtFdHVccxoAN1WQp2kUfduN13p1L/xVubudG9sF33hzF8hkaBCecs6qExfZDY\n"
	"o0JpbiaXRk7Uo3Wm9IU=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};


/* Intermediate has a subset of the CA's list */
static const char *tls_feat_inter_subset_fail[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEPzCCAqegAwIBAgIMV2qzzBH0Lg+EVI9YMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTYwNjIyMTU1MDM2WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAv4MqFiAHJqcSstTIdr/C0bD5EWO8Zu3dcmLtFykb0/AjgBAUj942otkhQv4W\n"
	"Wd5Qq5inBoUE8vgek6zShDws7Q5UiiW853vDwXBVvZwNc/4D1VpvOn7pMX/ILjQ2\n"
	"zcTIntInWmw+yGWDwYwA1C2aMNzFSTxFVhXYqgdzbf4EbpsFNlF0qKL3qeDlKGd1\n"
	"RKBopWg3D/+S8pv0glbLWIkgkNDwBNN0QwEeUsOgP2ice0yNmM+tZI8xNnQJ8Xrk\n"
	"GbLznXmGQW4mYoDYyFUG6yPi2ZAjwLuj6eY83q1Qv3RBatASKrxo10SfZiLDoiul\n"
	"0yhL6b33kFxkVAmefZKTCidKZG0tYronvadwy+wu2jNlZ85npT5qNbH1A4R5ooEK\n"
	"cWPc+AFXezHYp4k3OYboT0FyZOMM5bfo4prRjV6+OwXFzWaXpZR6WhjQgKgN/SeY\n"
	"jop7QJE16Umd/Wd0s7q0rpUFUfPxaAV1/EFIT8BvRxAseLYzHkHiNFODkicVAMxM\n"
	"E26FAgMBAAGjgZQwgZEwGgYIKwYBBQUHARgEDjAMAgEFAgEGAgEHAgEIMAwGA1Ud\n"
	"EwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAw\n"
	"HQYDVR0OBBYEFE37gAQSIYi41ewmh+rDfLM4y4smMB8GA1UdIwQYMBaAFJrg4B8X\n"
	"tXcN8n7yyTZ6kKCWkYK5MA0GCSqGSIb3DQEBCwUAA4IBgQCSE4ClejSjGLyhWwMS\n"
	"SDTihndqC3eNygAbYVFe0Xp1B7wzjDJsAvAMQvdCUheIryIcNIm8IiMOp2tM0M/U\n"
	"Iy9AY9+VWcRXxpqQwURBGXM9/ly1Gr48SlhHUK7fg6/2GLdloM21Cvqmmo+eCnRB\n"
	"WAhdFr33ADKexqop4GJVY+bVNSDSaknwUx3Xoa5NpOHihYMbA524UEyBIG5msuY+\n"
	"KKMZjfm7dNYIjH12D5HzSwdwrQWvlmkbLBAgAtlOrOfOAs9hmxTaQDKkAEBIbdCI\n"
	"OYUT5GAsNXaOlcb8ImvP7OQfRln0MoS+6yXiYkJ5IfcDotmZkq+Jc6VvPVnbh+EN\n"
	"nnlQxaOYVSo4Bl8UjwdRjuCfPEVFcr7pmRlij8duxoXD6PJ3B30ONEPrqTp6NCPi\n"
	"VL7bbkvsWGMLrtBWJHH5cGzBmHdm5TWSi8/5YMOFzwEAouwjQ1ux+2LhUWBpFikT\n"
	"8SSHjIY97PZZn8mwiZECt7TfiTuj0x4R1cOKpyt6ey2KPvs=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEFTCCAn2gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTAzNVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAvIcLRN8LHydTAhOA\n"
	"akrJ3w6CB9wN07XpVHbv4VFgfHuNGBS2ZRkdEg6/LsSZAeZ0doQMJUzSvcNyp/+e\n"
	"CPHfSMWuZX+C9mE9oCzEl+5bjrjphXp1AUutQ9I9zcI8TAQmFKQjC4lm0tj5tQp2\n"
	"ApClJbgLCnP6/5vb2y6GypMZnGeG+/4kC/zOoOef1zCSoeeqoiH3bbdW9xZhO6W2\n"
	"aYGiC8vP9M0e5rnMTiZ8cPVsWWgzumgZRpvfwWb5yjVqZ/wMe8uvwiU2Db/iDmKi\n"
	"HN2VctR+qAI8+Tw6Ax1nA4P9fSFP+wDZ2gwY2cCtM4/o5KyfK5KBfYtYcVa1vLd0\n"
	"wYg9Vf91uMdXoYfdCbQy/auGcTSiV0lf3HtnJvh2DJPkYt+CwKzrpbK3sYs3Z/81\n"
	"8LoDRwkKB1F5XDl0TIxyXxTgvCgx0u2nP1dQCUseNhG5ZUL3699McbgV36+Tgv5o\n"
	"3xenBhq3dgV456HZZo12mx4HBbjiKhq2XDRDcMjT1grEBV4/AgMBAAGjejB4MBQG\n"
	"CCsGAQUFBwEYBAgwBgIBBQIBBjAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQF\n"
	"AwMHBAAwHQYDVR0OBBYEFJrg4B8XtXcN8n7yyTZ6kKCWkYK5MB8GA1UdIwQYMBaA\n"
	"FHv4+czgWyHRnQHCkDoag3Q4SGZ9MA0GCSqGSIb3DQEBCwUAA4IBgQAFwVvLQaQl\n"
	"tshFLPAKTq71sdS/8pANVUqV3D6ypmeOFqr9X4p9joZl4WDDhTgVIBi4AP7PJEYC\n"
	"KO3Hh1h+zsGDj9jZKr7Np/IjIA9/Bqzd1Yw8DHLsHCZ+Jb49kQNnXm3azbwpGp8j\n"
	"jbyE1PNCca3cccXvoPlD/P+PMxxz9OORsln09kpIFTZMrO82KeQF+rUyGDKzVnjL\n"
	"2z8tq6pKLl6fDzDP7x0bwAMyFmWjheb2Q0SvMYrXgspP9oFmpYRdvxUcMmYkyvNA\n"
	"Zcfjh4tkLvmTSnI0mfy7GiekWaeKOLsS+7PTq+A4oirQ0bQo0hS5RHhXYUGoI5ZD\n"
	"b6/SvS5xoiw2L7VRuzd77kxH3YkczPEKv3Hgi14YxLF5UCbCxjSi6Dg6HxphM/KU\n"
	"baKV+jnSNK9PF6dCvv7U+Qm4DHZxUBplUVaVCLZizW8HT52I9eVN5Xjt1cWRIDHl\n"
	"n9F7dMY0np7Q6s9+nemgt5+i1JdDixTpC0cHZraUWHqkY7Bzczl+Kyw=\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID+jCCAmKgAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTAzNVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA5iScbMCdDf26x3JQ\n"
	"uv3U7vkj+nP1YGRQ+RS/Z0C8sdnC00SqK0jRBRk+//12DLWlLIWaFpoLPSd5cFXM\n"
	"AGukWaUiex6hYsIvs+Q2ocdckI2R8LbBOhH5pwFnMQ8lLFNHFXMjAvOaQV1oj/PY\n"
	"4nbwRRs2pY12jPz8HMyYIIS0VM4wA1/jg9WhdNRVBe5+ZvCVPDhzRuOLiikZVrEZ\n"
	"kKdpPugQNBqt2U2LO8sYjr9lmsG2u+mmjJJNrc6p9fOds4B3vPLq8W2dfaYS4GEc\n"
	"qkEuKhdqeP41uEUAbj1jSUZh16tyu75dHIpJcI6ZscNP8zyZ+CP/8jxfRl1G9OVu\n"
	"qUvDrE+3krq+ntZmaINjf7c9sWKD+m2lx3kVMHHUlXNCH0nn1V4P3svQnlg1W3za\n"
	"eEs88+pYq2VIJhNvYnP3rsCF8LgupHTdGALe47uW+7vdjrNBBmms9+hYOmvdFKfC\n"
	"dbeblk08wkVkW9BkMZVJ1iJVUVfZwebwWL96NSOLJnPhA8W1AgMBAAGjXzBdMBoG\n"
	"CCsGAQUFBwEYBA4wDAIBBQIBBgIBBwIBCDAPBgNVHRMBAf8EBTADAQH/MA8GA1Ud\n"
	"DwEB/wQFAwMHBgAwHQYDVR0OBBYEFHv4+czgWyHRnQHCkDoag3Q4SGZ9MA0GCSqG\n"
	"SIb3DQEBCwUAA4IBgQBLeaUX61VidXpsoa4zNZZWeqobd6chwn8mOJduQhFwc9M6\n"
	"UpYTx2w9EazWbNLru+kvuvAI/D77YMF1vzhksR+yC9Rw/Ljfj3abIA71z04vYYnu\n"
	"eLu3uLEXORJtxbDiY5QImp40gT/cAOcyDvGvVkcMJIKsGwoXS+DAlmQcJUWGh9Cc\n"
	"hJLGYJKPoFL/f5rn4pTmupE3sDnAo7wXVLicCOqBd48NbSFF4q2y1uDke9WcpiJT\n"
	"encWcZMdmDLFjVNUzPTehou7PdtGc+2+fC/c29wwTVPwrcV/UvWHbdSWdKnx+d83\n"
	"op4Zk0CZwP2I3XzRP3+oxPwdWr+N4cP4Fc4sQ0DFPH0duIXI9cMBaUQ7y4q4z5AM\n"
	"laK7POGUBEqPxEQAEh+5YRw/7g3/f+UT5vD9x0keFV7/NpuEkPGTZDynEV6uw1bj\n"
	"Gt1QEcLQGKvBq9Fac3kf7CZXXPNws23mXkfulg0StmXGG+jasiWY5gnqAB05JxPe\n"
	"CUouZrHY/Ww2QjP4vug=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};


/* Intermediate has unrelated values to the CA's */
static const char *tls_feat_inter_unrelated_fail[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEPzCCAqegAwIBAgIMV2q0ZAC3KVYGK9LYMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTYwNjIyMTU1MzA4WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAoc2zmiJS0B7jfKnMPqJzK0yp24xIiM0TWFIHsEDtjA9wLmwtvclUovIQIh0H\n"
	"wf7ZZt4jce9ZILc+EL9ELakRi8w8C9jDQiV/sfaGECBYh21hJoncyt1ZJXKm4909\n"
	"b9BmJg5n7d4cqgaJ8ELZ957zMSJWA0oDRGje9aTigQcSvWNfkfvZNyJXFHKhkqVd\n"
	"qz4aEq4FjptD9bjZ47uXYp96gCkjZgizbfhvIbLCHdHwxauCwO24Bl+JsVdMmlLJ\n"
	"ps/EzRH7OU3Jyxpn0bp2ap5FvAE7wFuSQZg/QfHSzlZQuTQnDrMuU7iBGhvCDfUg\n"
	"zOYISvcN9Uxp1M+a4pptJPlDyP2A5FqinIp42bgRSneBCoFKGOyHbn4JE/RK+WCB\n"
	"9JNNmimurprZXVSnzw1DReRCFiwuGx3HR/pVRI5txPQItiMVTyD8cG5j6ON5tX9P\n"
	"noYsZOJrj0iwGhct3E9mfUYJqguxW9eVjxEiZXmkxTjhR2BjIkedPVPwjEPKOFSh\n"
	"vf1nAgMBAAGjgZQwgZEwGgYIKwYBBQUHARgEDjAMAgEFAgEGAgEHAgEIMAwGA1Ud\n"
	"EwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAw\n"
	"HQYDVR0OBBYEFGmo5CTySqacfiOTM1/tqLZ72vqYMB8GA1UdIwQYMBaAFLM/Oolj\n"
	"mHMjh3gOiOJ3M0BRuD0jMA0GCSqGSIb3DQEBCwUAA4IBgQDaSD8K3JvGIvW+8exW\n"
	"XUo4R085hppxH12+nEwqvjtTmx8Re00ihm7taDpFaNHm5/hdPDF7jRqwJlsck22S\n"
	"bCHi9mSv416Ts2/KjaQlyKXQbWiO5yxCFrODcC6JFU8wy3XdtfUoRU6VUiWXuOiz\n"
	"JgxpwxWmPkakSEzUpr7VvZL/UVYkiLIIuvZ2AXsHCwJY6H9Dprct/iCFYIo8w7t/\n"
	"vHSmZLE6DLuES/EmXhwP/vMowummaKFWvj95E57g5nNAnFGysbH5QgXz0XlvDwC9\n"
	"iLK5fSrnDG0qcuVOkEPNRyml3ecs+uYgVAtZf71sr3+20+eYUBcllPy6DrQNUA8Y\n"
	"UiMAWQJxKIxnh2NP7AXbSJnb3/Gbo2SThuqihRs70+1RPOeVJjme8lNshYVjF7OR\n"
	"PY8srNWTOumzpb10HIMyMI5fTGQIkvW05D2qNKGu59ZQJomaE0ZV417ZwI11uVHI\n"
	"BwmpSliw959g176DBjFS6AEf7BFXImzTvuMcBzZFkXLZo54=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEGDCCAoCgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTMwN1oYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA+uCPJluPaCX6+PkI\n"
	"EWCpsgEeMlLXdilST/x2IgWtTb6Hv5jpBTJzVzRfnVodo3d9AC4JFpv8r3fOYsnI\n"
	"so9yVJzID4JAsmrFRLm8wk0KSKA7jGDT1HcNAmTSdb6GzpAJtEFZbOfFbOoOJhJR\n"
	"bBYyDwLkGx1Ghugjco2tGeCz6ztpPx3vQPj0Naqib0U28rwWu6a3N+hFxeRl3pgu\n"
	"NleLnkp3sizF31YCbo5A+Hmq67e3y0OIqzfn2AW7LI9L3eQUFpmGUEQz3zhpcH0S\n"
	"eJ4Ft4qMjblPR+zdMHDBKLIlCHIc36cMqcj7YMKknFO13IZprMA686V7MJ4z5z0z\n"
	"7IiWRCkbZ3FJw2F5lwJXViBJLq5rHH6r43qnv3i6aBYXHgiM9JtELnQnHgbcRBOX\n"
	"ZjCdaVZJiNbXuWT1WbKGtnovPqmOLftVSXvTnJN5UnnHITHCz1OUUWYwxq+cDFZ9\n"
	"nSHI2mJV0v1ZoUhe+qhz/qqQvZLC74YEnLFQGooFWu07y41PAgMBAAGjfTB7MBcG\n"
	"CCsGAQUFBwEYBAswCQIBBAIBAwIBAjAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB\n"
	"/wQFAwMHBAAwHQYDVR0OBBYEFLM/OoljmHMjh3gOiOJ3M0BRuD0jMB8GA1UdIwQY\n"
	"MBaAFLAH7D01maBP1WxFfYBm/vCl1mhBMA0GCSqGSIb3DQEBCwUAA4IBgQB8dBgc\n"
	"zJGEEhNEe2myi2CdeajcfJpZFrJYsBaH3lvgree4qohYDz9ojEFj4SWuBe2VqI2E\n"
	"EshNzJ5bOY7xEc5dZmuV9tAGUNY8dKWUFSBK7g+ReGFgGFGHP/Rvf5d/hnin5yqb\n"
	"q2TdD+XH3PEnsM+FbI5W2TPTiOp1DKIf5vqgXUrNuKQbDxTMzA8rdlB6tyaAEin0\n"
	"bivEFfwSQSpOvxyYvw9SHnAd67hpl9yvVv+polOsLXdA/p7wD0gqJ1GTta4VpPXb\n"
	"cJSf98OSML5f0MJuHcT3VjBEoiqT4KCZo2YBr6oAgpDFWd2SVvzUB5OPiVnryWVH\n"
	"o8rIFXCkRYKN37t9fVgejz07/0IEHPvp4Cdzf4LVAFIdjD7jOClXZzQvtc5tOOx4\n"
	"idO1sU9NXiksUmi8KF4fPrjZb5THX5/J4oFkfKCzrL/GZumSVYzzdrWkbM936wFq\n"
	"QXgvUPN/ZHICeseK/zJnlzNh7lVKU1aANnsaQgXwKp697u/VCEIQ0OBKpgY=\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID+jCCAmKgAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTMwN1oYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAvGvJCizqUP+R/Ytx\n"
	"SxyW/jSWy+i2mU5DgQJbeJ6UTxFcbB7LLOgwdmEpm/g7Wv4evm73G1CIR0ujCuor\n"
	"Rm8HvxnbfoHO8Czm0uOORp0UlqQ/1c03L6wDj00mdmiWGaC6ViLBvgiNhw3NUnwc\n"
	"eDyo61f5ujYpnK1Wj9oFHljqkjoSneaYl4YkJtCdbTb5lgKlg6burYFDNiF5EaKK\n"
	"5mzY8M9GYVwIu0t7GK8WbzrDjgH/Evqg53C/1NBnyh+qmt1FylLELNaYzq5fLiwI\n"
	"bh7UylPCgRzLDlaOMknVfOP70rapIBCYH7CzUUgQhJGlmlxowZfc9PgHnw8VLaOR\n"
	"dJjtSugLSXbMv3YwEiUwGWvyudHIj08DV+ul1p04hvl3pYd25yd1x2xHPg8VAEX3\n"
	"ycN08Uu6D6UfpZdvs+i7VZKKSrl49MR0UIDOj2jK989OrGb4nVFD+zTzcFPJZRew\n"
	"Ebf7nvwmBnu+6gbFt4QXl9ErhQrKEtJdJ+IVJgmj/HwVhseBAgMBAAGjXzBdMBoG\n"
	"CCsGAQUFBwEYBA4wDAIBBQIBBgIBBwIBCDAPBgNVHRMBAf8EBTADAQH/MA8GA1Ud\n"
	"DwEB/wQFAwMHBgAwHQYDVR0OBBYEFLAH7D01maBP1WxFfYBm/vCl1mhBMA0GCSqG\n"
	"SIb3DQEBCwUAA4IBgQAqMU/vm1cK6iCUSJFvi0DpykkVAtC7vo3ZQGeaiEsSaz5U\n"
	"svphDbWlHVN3EdAv7jNmt/R30bEtU5J8xmMU9JtBs2NlYl+65txAAuJUaTmL+ndk\n"
	"Cr/mCKx7gnQi8mYmCmoA3FbtjHik8aufwGKCZ4arP5a3pSUEh4Hi7bydWaDEnmRi\n"
	"p+ngyb9KwYiU0yvz1mF6pmYK62lHX4M5ONM2hw90DvAqMcAWzKI3oBcFgufkWNxE\n"
	"jqQnx31FaQ2uhsPEfMOz1AcYy0axuuK/NxIQyLKLHLYLCF8Ybb9FJNKWYvEAmIAh\n"
	"YwkzvzUfIgez04tNS6OPc2Zn+EZlXIotmK1pBddGUPEnwWCCkzIt331Sip/25Qri\n"
	"2Fij6cxtVmH4EJR4c93NC89P7kcNL/TAKtRRfaPuD/XdVchH3jTmZiR0AlPD+RG9\n"
	"83tWk8f3gfMWOhS3ulWYUvvw5ahhcxlXPyEIGL/LKyjcloSoPre6nOu/hMOhH4QH\n"
	"YFBWeRM4C9FZEqT1a7g=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

	/* All feat extensions ok */
static const char *tls_feat_ok[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEPzCCAqegAwIBAgIMV2q1gxhOXfKDif2GMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTYwNjIyMTU1NzU1WhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEA1ATMiShXx/9z9HzfHeTDWSOWKs1CPosoj/axIdPUk82q0Y052z3a+iDDp//s\n"
	"BquPxSKt4B4fKjIWSD6gfdVhBaHB3MbHMLLZ+6qqIxezFzGclCAyPBBCPfuOPw7y\n"
	"c0FMQAUXVI8r5JBe2Y7AWckNcMRu/rjJ/4UUT+I8NFG4YLZhDlDf2ZzgsuFG9ye+\n"
	"IdgNynZPUeC+0RphRAyg6BnHax3F7Ti9CwIeFl5bKpmiKH3X43pqbFhQUM3gNF66\n"
	"a0rtK/ljkXBvJ+7L9ACyupPxji1y+VmS0lDGZApZB23qVzM9iKxIi7LfmsJ8EwI3\n"
	"hNVvWQkRJVxT6MOrZ1cKXlJ6sz994N0BfZg7oraXn2ojI8I5Dfjly0U7BMJG6kd3\n"
	"RSjy1L5YuAYSbhMWgz6fUT5Jt3hoXMYTiNxJGSXEC0evh6iF7GgqXNHaMd5qh9TT\n"
	"X4eyejCQNDWRQg5y5hNwFT/C18eTtjhD4/QRGDWYVn2tvhsKI/wowmQ6F+2LKhUO\n"
	"u2TxAgMBAAGjgZQwgZEwGgYIKwYBBQUHARgEDjAMAgEFAgEGAgEHAgEIMAwGA1Ud\n"
	"EwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAw\n"
	"HQYDVR0OBBYEFJ1NcokmOpoc1vU98iCH9Qw37tavMB8GA1UdIwQYMBaAFN2L1dSv\n"
	"fFt5IWg1sunSrAMiavK5MA0GCSqGSIb3DQEBCwUAA4IBgQBtzUgsge65IMTf6lB/\n"
	"bJCO67OTXpaIzrFjmnPxc/u4SOg7pfbCH7Usdee4VhGBCRBkqE49Pyvu6Cf+p/MC\n"
	"3jvM+wm1Q3TRYvjeXuJoaoEefwsiMUK0K7eynsuPT76jFdcxMEsWUvexScbJIOX3\n"
	"I7EPF/Cvom/39Ssz46NHSkJKqdlbY1AUfTbT+fH5qP3crfYNiA8eTusWIK8iwWGE\n"
	"PVln8uIiZMHuagy4y2vdEz3Ha5z+Nn9vP0dAfzULSQpVePNwwPVAv4qhd1mtv/Rc\n"
	"uObCM+3o7ejUmvBn4nNCe8k57Hznztb5AGiiynmbGb03lNXHbS70MvCxilAmncqc\n"
	"z/Z8x/N7uLExAEwfed4APEBPWdoQm4wuwLgrLs5mVTPHNfTcyZLOip9zE5pntlGH\n"
	"0i5BWiPb/n7V7wJtG/ZWreGnJk5hh1+u7wEwnPr9MQ0suxGYrWoBb+Rm/QKv4LKE\n"
	"teG7IlYYAvrfsz0vWbxb11G0ROln6UckRrXgDXfYB9PTEoY=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEHDCCAoSgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTc1NVoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEAso72loNaZh/8xNBj\n"
	"xxNXRg8sc4KER9ufI/Fg+nDN4ddnI6ztuGGyUt+NdPHCSydV6mWzcTRrvYIZKwl7\n"
	"H5/lUmgKpNdhiOq9tAlmEYaHeWHl5JuewVgOnpSITwqZv0K9EUoklAtnJ9CQsVqg\n"
	"Om4lmGJDFIC+YnahbI4glt/I6WabsZvEuVtZKbJ6wk0avLqYHdb2w3oZBEs0+VSb\n"
	"/BH8gHYitqK2YMxAJY5rWPIjYEzILOB9k6B6tqG5KLyXXiWLfOYBpxjlFA7cAz5C\n"
	"R0jaQd7b9WU9GPkPExKQ7VQ8IHPDyU9awk32Kmr3pY21eAd/0LPmi8LMug8r+M5B\n"
	"tsqmTwB8wkqj8PdFc1InaKRrcc7BJJj0Befa9yBj4/qsjfNtI893r3h7/WnhqAfw\n"
	"trmzI3DwBHcgFhwTtEeJwXza8wZaGKWgjYd3d+M3hpqgp2DCmWP9IFlrkgITpzEJ\n"
	"FEHcGwVaIM9uyyMbcTExgPeFEm2vtASJ+rOAjg9OAYPKETGbAgMBAAGjgYAwfjAa\n"
	"BggrBgEFBQcBGAQOMAwCAQUCAQYCAQcCAQgwDwYDVR0TAQH/BAUwAwEB/zAPBgNV\n"
	"HQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBTdi9XUr3xbeSFoNbLp0qwDImryuTAfBgNV\n"
	"HSMEGDAWgBQ3S2pCBcQ8ZcCsf3iECpq9I3JC3zANBgkqhkiG9w0BAQsFAAOCAYEA\n"
	"klVFaDyQ9n3P9W2mb/8pWBH+PgQjllTKS3poqkNbPx88DaZMd/Pi63I4gjVoHAVa\n"
	"9uShnrM4E6Z/2Va+XV2vASdV3SaKlw/nnbz7vpJrhjsrdfSgDEV/vh61MOyiQDNw\n"
	"8tqutLy8nnNocj54jXVsta/hWxvvPcbGyhMEFvRLdZjCpQre4ACeZG+p/hDCZlm9\n"
	"MAx3EWVjOAkxOwR6CGcg39Q8HcJZbFd8XjJK9Y1z+ALKTSmeZ5ppPTEeHIjOJOtq\n"
	"w9S0I3+wroV2wVK3+tbzCdJopdMwivYfVH/XCVq9HniJFqtBVwXi1gUqjpbL+AQT\n"
	"weB1mF1bwQNpS9fk5UJOOZ2bNaJ81l+eEgSM9X2g6rEoe6z5BynNGp4cpnHrUP3O\n"
	"HPFd6O6/u/nGeYuHOZxPu0BJz8RnBH/8oyc4+6I5PVx8hfCB57EVtW2yrlDcFWt7\n"
	"QCycdysz5mds93rlsTnCUTpzule4Qiwsq8HIcdN9JIuDa1G9q4Jc5C3/Rz7C2eRd\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID+jCCAmKgAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTc1NFoYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA2I0BfGo6rT8j5EN5\n"
	"EI6lXlmkczZKU7NcoQUbmi1Dss5f54TIzrSGQ4w0Qi2HEo00HPX2NaXz2K3bC+ZC\n"
	"q1MYMnEi6CBNhsD8FS1RocP60kM6dziH6mdHHhmWZvNZocup6SMzNkLPSQSo07H+\n"
	"/24YJTEhN56uWfGAf/7gmFGAhiZJLAzS0uyWimoQk0p4GMroLN9oavgpVOgpTbFO\n"
	"eCA8ruFq8kO/dKlnVgq/I+Y4ObQWsaF6cyoX7OCXSXXF0ebNCNQnHzKIekGKH3zU\n"
	"z9aVFxlPBNlin2xZsM4Bs3xhJay7VimxZfXFgIFr5anGyIZp1hlC6ppJKiRGDErP\n"
	"lCzHATCSy9xrUxZ66b7IEwq3vjaPc86nXBVLNRDeGAkzaORLEu1MenOF9FTFVUcl\n"
	"h8cd1jNqPFLwkM0CZEFbxelSWzILBJjRJykJwydcye9HK866gRCNAraKQ4gC8Umd\n"
	"RvtejTtQ9oyvn6sDe9G3hdYZ/6M7dpxPvDM6+X6CacDtM+8DAgMBAAGjXzBdMBoG\n"
	"CCsGAQUFBwEYBA4wDAIBBQIBBgIBBwIBCDAPBgNVHRMBAf8EBTADAQH/MA8GA1Ud\n"
	"DwEB/wQFAwMHBgAwHQYDVR0OBBYEFDdLakIFxDxlwKx/eIQKmr0jckLfMA0GCSqG\n"
	"SIb3DQEBCwUAA4IBgQANnHNp2P4GcvZz4KKxkKzcvCxUEwT4Wy2DS3GgMy9C3dMR\n"
	"UGy/gpJ2HqepB8xr2NWP6MuB4gGzA37LflvEAz7/MFMinP5mTbpP+G8yuRh2H+vD\n"
	"reRf9LnUvwwnjN1H1+YBoSOdOQNzfrChRo24jjsnQ2PyXz1RYRQ0wVdaMFX32pYq\n"
	"k36zhD+XV8zaj9zXkFkT/WValxzbZkGJVcDrl/4KIcTAi5lq5VdlAp2E3CN2sua4\n"
	"I5HRjZAarefMRxcddX8PPVI1ZNEwrB2jM1uOwrodjhbYbfLPrkOsCQTQh6ETDEYi\n"
	"3Y9nCx+DgIGZ8KyycIlvIjJwyiFOC+YB/BfX0M/dOJ+tl7FU1b8J/qNQTG82rgDH\n"
	"fNRD2DR1+jNb+IQiV/HdJJLyissqYkWVaIeG+6RXXy8d0rfSIWpT8YysgJmWqT4Q\n"
	"QKarLoyzHleMN1fGMXT8P6a6Z4AKl1lr3ev9rHjGr88crXQDGtzQmM6YCi7FKegJ\n"
	"T4pCBf5TMfFHfyuGlF4=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *tls_feat_superset[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIERTCCAq2gAwIBAgIMV2q0/wgRypWYPJf5MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
	"BgNVBAMTBENBLTEwIBcNMTYwNjIyMTU1NTQzWhgPOTk5OTEyMzEyMzU5NTlaMBMx\n"
	"ETAPBgNVBAMTCHNlcnZlci0yMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKC\n"
	"AYEAwBrZ+hFGdXX0cZmlBruD6ciZMUwVxCRuYP9fK3iTAEMGEI60tjsMfuLVOVHi\n"
	"OXME1Y2UO/VpH5jcLZElyqb6MfekxdNwa5XW3KbGSJYczBEicHDaF2sMkpE8TM35\n"
	"bVu4lvsvhp9m4AmeS9zM7jXIfuF6YOtfrwhRCNjI6CJxo+ZzHz7yBEDJ25iWZdzN\n"
	"zRPapY+MwPrgR2GCZV+yN+F4eoizLfX0aQ7MGAtjGfcNFwGeQrRzpBcTXe1dqr2G\n"
	"3jtreKFO+vseS7zZvqJVsnurnzFwN2dCqpeEbiNisKihrbpOPV5C6KBFRyBmJUKr\n"
	"PCAtaJZgsPQQ0oQRCqopeAeThnRNTkHqK/t/r9HketUXc+6dU5D4Okc+NV0ULBMM\n"
	"xR+H6K8AJJoTAxZR/t2Gsaao83fcYvFenu+T6IPsyKftrVAmICHkPlRY/ir0O5OJ\n"
	"7s5NXCygDJKf94Hqc3nU6EU4VBrehvKDf17sOMigmOlFA64mFSn+GyoI5OqBwnr9\n"
	"614VAgMBAAGjgZowgZcwIAYIKwYBBQUHARgEFDASAgEEAgEFAgEGAgEHAgEIAgEJ\n"
	"MAwGA1UdEwEB/wQCMAAwFAYDVR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQF\n"
	"AwMHoAAwHQYDVR0OBBYEFC7as/AaudiD7/rYG+pu00XJhLECMB8GA1UdIwQYMBaA\n"
	"FCJrjNrs/mt/frUxJ+fPkDBA720HMA0GCSqGSIb3DQEBCwUAA4IBgQBXNFbgFC0e\n"
	"YKsmEwOdLLpIedi6k0kdd3Mo42CX3y8lv1JbZokUFUDd7fgJgbtlG2t5Tp+iYRAM\n"
	"6V5NoXOc0eWDYn/K0TQJSWsHnwnAn0SG2ZJt/2A6grqoMa2BJSY6pKfBOOkepJKF\n"
	"Yd8EOFJNnwOMjwzRPoccsx6LBTrOZVAoDgR01RUPDXBoyDOBnq0A2gEpMnJQjjw6\n"
	"w8oBnU/ySKhs6iXNY9xiA+3J/rT8jF5WOCAk78MLewcoEqWYR0QgtkmoHqhY9BzO\n"
	"VyJ5sMzCAnv1Zwi8UerBpg97In44POun9u/qEBSjIZj0Mba6MIWXTN5GzoxJTej+\n"
	"XWJoNxR9J4gTlHnFTxi3s5EcS1lsk6S2rVUCy2mHhSTmrwoXGz8AyDUn3hjfgHx5\n"
	"cmW+nzaDERO3pLZ0MFQEwv9Wdq9pcGnpDtOzimI+AsrGsuYqeY69ZDH1AvKfposU\n"
	"0kblXBxkedBddb1HlvLzU45j9RqV/TfYBJA30FAI451vwoANX6oLl1E=\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEHDCCAoSgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTU0M1oYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0xMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA2z5di/R9W/uEwa1a\n"
	"Oo2+POgZSuzGnikpH78cCsV0u70ieaOolj/X7tJ+9MAfYLfMY1lRl7OboYAshTn6\n"
	"ZfLQa7QkysXsR/EySq5sEVCz/Btw60dZb3UrJEyskR9wZ4L49hUrKiECAIEWykXh\n"
	"F0bi0KAOMziV18bOlgrhmm/AD0mXqLnhIRD/CBZtYiDD1/mClgYtOxJLqyUdj1GH\n"
	"+XKN9BdIlZ7zl4UEuMuGxTkqpmxeYvtQKxoGd2Mf+Y+FK0CiMVTZ0PxHxA/rQ7b4\n"
	"s0evs9a10oXmVnNuYtDtCIMiOY2zJvo38oWAvDJsJH/zmzyPqWoCyDqm4zuxT14I\n"
	"BFlVxN7CAgvrdJgRonmy0yr6vLlXCK/PdzhD0LS8RcfjeDYhTsDY61rfArsB2eWv\n"
	"Y2uyJAOhjYzCDAAPxh87co3WlQyiVS5bFP5SI++y1qgBHPuXmkoPIft8H2HxY6sn\n"
	"XAHdqKgrB78j5Asc6FWfzp3MpatgLuspOmx/fLERxoIWVxo3AgMBAAGjgYAwfjAa\n"
	"BggrBgEFBQcBGAQOMAwCAQUCAQYCAQcCAQgwDwYDVR0TAQH/BAUwAwEB/zAPBgNV\n"
	"HQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBQia4za7P5rf361MSfnz5AwQO9tBzAfBgNV\n"
	"HSMEGDAWgBQHiC/xbZtw756ibmRddGjDpv5VzzANBgkqhkiG9w0BAQsFAAOCAYEA\n"
	"NykGM82KLfZPVpkw/wZSWmBdblYrdSPTyGNm7RAqLodFIW1re5Xtkk6xPJYGWNZO\n"
	"Ac0Oz9IDsGlaEVsXOyyIV6ZmzG4hr9HQh5jvt3nHdXMJMjNz9qJLM4mq/+3yF0Dx\n"
	"B3/HUne8A8k+Rp+F/c/544Rt8zH3pHGKH0ytc1a8Q79q9SUhH3Cc/d0qvmUcuCBG\n"
	"tix6wnfCpRma/6nugFXM2HbR/3C9U9EpNG2nS6mdpuYlm3SKFY6VJd6gpLGc/akr\n"
	"/557JFsC6UrgHgI1c2sSqSWJuhLBRVeFkyvGOH3PSz+F/sUO4UXxf0LcoUsr9Zo0\n"
	"9F6muAj5M7FIVOnBaOIpYVf3pezUUfY0aHaB0GFDJViRNuSxJJBrJEejBKEIXfEp\n"
	"AoBVepgwdvM9V+BTxHYYetpUDZLa9PeQcuehyzN2dNLOwYPeYX1Ax9c/b+WbZRdw\n"
	"JG9uD/Zre0xmdIvlHFIGVh3pZsgdbEWe6jrBRgwcQ7snSMPVvOQxONiLQnTKy8+g\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIID+jCCAmKgAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
	"MCAXDTE2MDYyMjE1NTU0MloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRD\n"
	"QS0wMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA2MAgfmjKkZ3MlXu1\n"
	"aFurNhmGREguCb3fqpHkOXaqCk04rxG/ZSMJkNX3v3IaZXPBM/YinjLrB2hl0Vaj\n"
	"tA+EQgXsPekRZKX9UjvIC1U+8Lutv8rJ15d8ZBgo6fGzj0rkLX5o3gFrweNXuBYh\n"
	"Bz1N0TvD+SWPfauG4UlZcMLnH2Mk0T/oNPsb2xC8F60OH/brgnT3OVGjQmouV8wh\n"
	"eYVutV+GLjeOlcNWLTQNUWPaeq3fiUA+MPXsszW+yEe5KkMpETDtluwGVcyLZMAo\n"
	"k9YXhe/MU47PIXYiE1bH7/vInGf0PZ6ljoRrBMyEEsG1DuysmNBoP5GjpaKS3+4K\n"
	"21R9rr6ylMb7Tx2PlMYzfpWyB/y7naGPBuIF4ExI+h6SzwblPzpKNKqcsAO+T3/G\n"
	"FLkRMzpFfxW8PiyzsUZnBFUDeeFMV1NMMKUhOumQGTJ35vxaPAtqMxqJRHfpsP85\n"
	"GXnJ72ubCekSEgmsIzhcSpevcH4Kreyu03rtkwzFm8gal8EXAgMBAAGjXzBdMBoG\n"
	"CCsGAQUFBwEYBA4wDAIBBQIBBgIBBwIBCDAPBgNVHRMBAf8EBTADAQH/MA8GA1Ud\n"
	"DwEB/wQFAwMHBgAwHQYDVR0OBBYEFAeIL/Ftm3DvnqJuZF10aMOm/lXPMA0GCSqG\n"
	"SIb3DQEBCwUAA4IBgQDFINpl39BzYyziYAh1t8YVvcTgAs/ALikaTPnSPoDEJgJ8\n"
	"QDt1omDPaaOuprcZNIN/23q5I9fNLgcUyDS9dCAqZmtxK8fXorS07U9XuQwN3b8J\n"
	"1P+NOJfzn7oUFVdGZ/0wWkySQVGmIkweTNJ3JCIRBM2FMIeCdZKt8NHcU1F7xPmo\n"
	"IAk4MiM4Mp3m4GZoRxo/VDL5MBKUlw62Er6wpRguGvJvCeoKANwIMgwU3r+kBjR2\n"
	"IvB0ysr/xxVTuOzs41Y9Cnp+5Q+McLKvHXkJKyAsyZU+i0SKf2OvwGnzhkzk21Yt\n"
	"z8jKzog2AmeIqWS3rlu0Hbqo+0QORgIeASU2J86YXo2c0UQfpb3kI7gkQG6jbCLt\n"
	"JgstnWQIZ/NgZ8kI+IsEgCkwlZWcwvvzLN0rU5nuO2psOvHTasPJrY0yTpx6uBVO\n"
	"MhwmNs8vu/29rvWY3pY3X3j+MQcAlb6KpNyJCUaV0Rhi4F385Wy15VJ+FQdykZ28\n"
	"GMGU26jKIvQm0NDWbB4=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

#ifdef ENABLE_GOST
static const char *gost01[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIBfTCCASqgAwIBAgICASIwCgYGKoUDAgIDBQAwNzELMAkGA1UEBhMCUlUxKDAm\n"
	"BgNVBAMMH0NBIGNlcnRpZmljYXRlIChQS0NTMTIgZXhhbXBsZSkwHhcNMTIwNjE0\n"
	"MTkzMDE5WhcNMTcwNjE0MTkzMDE5WjA5MQswCQYDVQQGEwJSVTEqMCgGA1UEAwwh\n"
	"VGVzdCBjZXJ0aWZpY2F0ZSAoUEtDUzEyIGV4YW1wbGUpMGMwHAYGKoUDAgITMBIG\n"
	"ByqFAwICIwEGByqFAwICHgEDQwAEQHlSMCMAJZgoHUCIFmn+eqOgYlQy8h7SfjZ2\n"
	"kMkJ4xTae8jtZYxHLq3P+qJeismsHdmqdqSBbOGGdOaJNNPbAZKjGjAYMAkGA1Ud\n"
	"EwQCMAAwCwYDVR0PBAQDAgWgMAoGBiqFAwICAwUAA0EArslfUeqhW9eFkspn89+C\n"
	"OQEJX6JoghiOjFYlky0XmaaDl3D6EcbID+B6cBEmcXF21xxIEeYJIAqGzOEnMXdT\n"
	"cg==\n"
	"-----END CERTIFICATE-----\n",
	NULL,
	"-----BEGIN CERTIFICATE-----\n"
	"MIIBfTCCASqgAwIBAgIBADAKBgYqhQMCAgMFADA3MQswCQYDVQQGEwJSVTEoMCYG\n"
	"A1UEAwwfQ0EgY2VydGlmaWNhdGUgKFBLQ1MxMiBleGFtcGxlKTAeFw0xMjA2MTQx\n"
	"MzEyMzFaFw0xNzA2MTQxMzEyMzFaMDcxCzAJBgNVBAYTAlJVMSgwJgYDVQQDDB9D\n"
	"QSBjZXJ0aWZpY2F0ZSAoUEtDUzEyIGV4YW1wbGUpMGMwHAYGKoUDAgITMBIGByqF\n"
	"AwICIwEGByqFAwICHgEDQwAEQHxF7QOkGNDlnKWiBwdD80gToowegPcHR1Y1r2ZR\n"
	"RQqB610f3uEWN/EikI7exYVRR0dmCyILLMmgRxX+KU4qmgejHTAbMAwGA1UdEwQF\n"
	"MAMBAf8wCwYDVR0PBAQDAgEGMAoGBiqFAwICAwUAA0EAFnvKPRo2tQkI/iqu/CkP\n"
	"YQJPW43KnRMqkmB/NnGOC5+wdivIA5yJaGbT2sQ1r+n6qyJnG32yV44DrSe7b2DV\n"
	"OA==\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *gost12_256[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIICYjCCAg+gAwIBAgIBATAKBggqhQMHAQEDAjBWMSkwJwYJKoZIhvcNAQkBFhpH\n"
	"b3N0UjM0MTAtMjAxMkBleGFtcGxlLmNvbTEpMCcGA1UEAxMgR29zdFIzNDEwLTIw\n"
	"MTIgKDI1NiBiaXQpIGV4YW1wbGUwHhcNMTMxMTA1MTQwMjM3WhcNMzAxMTAxMTQw\n"
	"MjM3WjBWMSkwJwYJKoZIhvcNAQkBFhpHb3N0UjM0MTAtMjAxMkBleGFtcGxlLmNv\n"
	"bTEpMCcGA1UEAxMgR29zdFIzNDEwLTIwMTIgKDI1NiBiaXQpIGV4YW1wbGUwZjAf\n"
	"BggqhQMHAQEBATATBgcqhQMCAiQABggqhQMHAQECAgNDAARAut/Qw1MUq9KPqkdH\n"
	"C2xAF3K7TugHfo9n525D2s5mFZdD5pwf90/i4vF0mFmr9nfRwMYP4o0Pg1mOn5Rl\n"
	"aXNYraOBwDCBvTAdBgNVHQ4EFgQU1fIeN1HaPbw+XWUzbkJ+kHJUT0AwCwYDVR0P\n"
	"BAQDAgHGMA8GA1UdEwQIMAYBAf8CAQEwfgYDVR0BBHcwdYAU1fIeN1HaPbw+XWUz\n"
	"bkJ+kHJUT0ChWqRYMFYxKTAnBgkqhkiG9w0BCQEWGkdvc3RSMzQxMC0yMDEyQGV4\n"
	"YW1wbGUuY29tMSkwJwYDVQQDEyBHb3N0UjM0MTAtMjAxMiAoMjU2IGJpdCkgZXhh\n"
	"bXBsZYIBATAKBggqhQMHAQEDAgNBAF5bm4BbARR6hJLEoWJkOsYV3Hd7kXQQjz3C\n"
	"dqQfmHrz6TI6Xojdh/t8ckODv/587NS5/6KsM77vc6Wh90NAT2s=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *gost12_512[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIC6DCCAlSgAwIBAgIBATAKBggqhQMHAQEDAzBWMSkwJwYJKoZIhvcNAQkBFhpH\n"
	"b3N0UjM0MTAtMjAxMkBleGFtcGxlLmNvbTEpMCcGA1UEAxMgR29zdFIzNDEwLTIw\n"
	"MTIgKDUxMiBiaXQpIGV4YW1wbGUwHhcNMTMxMDA0MDczNjA0WhcNMzAxMDAxMDcz\n"
	"NjA0WjBWMSkwJwYJKoZIhvcNAQkBFhpHb3N0UjM0MTAtMjAxMkBleGFtcGxlLmNv\n"
	"bTEpMCcGA1UEAxMgR29zdFIzNDEwLTIwMTIgKDUxMiBiaXQpIGV4YW1wbGUwgaow\n"
	"IQYIKoUDBwEBAQIwFQYJKoUDBwECAQICBggqhQMHAQECAwOBhAAEgYATGQ9VCiM5\n"
	"FRGCQ8MEz2F1dANqhaEuywa8CbxOnTvaGJpFQVXQwkwvLFAKh7hk542vOEtxpKtT\n"
	"CXfGf84nRhMH/Q9bZeAc2eO/yhxrsQhTBufa1Fuou2oe/jUOaG6RAtUUvRzhNTpp\n"
	"RGGl1+EIY2vzzUua9j9Ol/gAoy/LNKQIfqOBwDCBvTAdBgNVHQ4EFgQUPcbTRXJZ\n"
	"nHtjj+eBP7b5lcTMekIwCwYDVR0PBAQDAgHGMA8GA1UdEwQIMAYBAf8CAQEwfgYD\n"
	"VR0BBHcwdYAUPcbTRXJZnHtjj+eBP7b5lcTMekKhWqRYMFYxKTAnBgkqhkiG9w0B\n"
	"CQEWGkdvc3RSMzQxMC0yMDEyQGV4YW1wbGUuY29tMSkwJwYDVQQDEyBHb3N0UjM0\n"
	"MTAtMjAxMiAoNTEyIGJpdCkgZXhhbXBsZYIBATAKBggqhQMHAQEDAwOBgQBObS7o\n"
	"ppPTXzHyVR1DtPa8b57nudJzI4czhsfeX5HDntOq45t9B/qSs8dC6eGxbhHZ9zCO\n"
	"SFtxWYdmg0au8XI9Xb8vTC1qdwWID7FFjMWDNQZb6lYh/J+8F2xKylvB5nIlRZqO\n"
	"o3eUNFkNyHJwQCk2WoOlO16zwGk2tdKH4KmD5w==\n"
	"-----END CERTIFICATE-----\n",
	NULL
};
#endif

static const char *rsa_512[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIBTjCB+aADAgECAhQcc65I8jSxWRjcS1czw4MRLIc8qDANBgkqhkiG9w0BAQsF\n"
	"ADAUMRIwEAYDVQQDDAlsb2NhbGhvc3QwHhcNMTkxMjE1MDI1NTU4WhcNMjkxMjEy\n"
	"MDI1NTU4WjAUMRIwEAYDVQQDDAlsb2NhbGhvc3QwXDANBgkqhkiG9w0BAQEFAANL\n"
	"ADBIAkEAwZFO/Vz94lR3/TKz76qRCV2skqthX7PB6YxeLHH3ifWSYR2qCYTBikaA\n"
	"Sm6PGDvAliviIjGjKTkdDdqZX2S94QIDAQABoyMwITAJBgNVHRMEAjAAMBQGA1Ud\n"
	"EQQNMAuCCWxvY2FsaG9zdDANBgkqhkiG9w0BAQsFAANBAHslvfVxod5p+Gt7l4LV\n"
	"M2HBxOt4YM8mRCtyNSmJEGAe+aIzXaiSiRnVkVvjQvdxacu2D4yP52BUo1vzNnCq\n"
	"2UI=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *ed448[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIBhDCCAQSgAwIBAgIUIWKQV5hisum31Z2Fw+PeZ80wqnkwBQYDK2VxMBkxFzAV\n"
	"BgNVBAMTDkdudVRMUyB0ZXN0IENBMCAXDTIwMDMxNjA5MTY1M1oYDzk5OTkxMjMx\n"
	"MjM1OTU5WjAZMRcwFQYDVQQDEw5HbnVUTFMgdGVzdCBDQTBDMAUGAytlcQM6AFsM\n"
	"fQUL5TonNaVrBB7H4UtwnVlolZatMXceHZiWnzMKXOZXlIabi0nTGkvSFu9ed6JJ\n"
	"L7EWarjRAKNDMEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQAMB0G\n"
	"A1UdDgQWBBRMwtFQ9T9Ndw63UP2QGAuIFoYb6TAFBgMrZXEDcwB8hbYLw7KMlb3a\n"
	"Q2YAXiugWt2WcAMtvKgqzjXzUt2jilaDA72d3MCAWQQsMmQfRNSthDIao5CksoDk\n"
	"Xc8qFzckmdBiF7W+UNT3OMisE9yIxF4iA1Sxsji3C0WDUq2jen5Uv9E99H+r47L8\n"
	"U955wKxWJAA=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

/* This contains an expired intermediate CA, which should be superseded. */
static const char *superseding[] = {
	"-----BEGIN CERTIFICATE-----"
	"MIIDrzCCAmegAwIBAgIUcozIBhMJvM/rd1PVI7LOq7Kscs8wDQYJKoZIhvcNAQEL"
	"BQAwJjEkMCIGA1UEAxMbR251VExTIHRlc3QgaW50ZXJtZWRpYXRlIENBMCAXDTIw"
	"MDUzMTEyMTczN1oYDzk5OTkxMjMxMjM1OTU5WjA3MRgwFgYDVQQDEw90ZXN0Lmdu"
	"dXRscy5vcmcxGzAZBgNVBAoTEkdudVRMUyB0ZXN0IHNlcnZlcjCCASAwCwYJKoZI"
	"hvcNAQEKA4IBDwAwggEKAoIBAQCd2PBnWn+b0FsIMbG+f/K+og2iK/BoLCsJD3j9"
	"yRNSHD6wTifYwNTbe1LF/8BzxcwVRCD0zpbpFQawbjxbmBSzrXqQlUFFG11DvNBa"
	"w58rgHGo3TYCrtFIBfLbziyB1w/vWeX0xHvv8MMJ1iRSdY+7Y36a2cV+s85PdO4B"
	"TpZlLfy8LPP6p6+dgVoC+9tTu2H1wARYOVog+jt9A3Hx0L1xxVWTedFoiK2sVouz"
	"fLRjfp5cOwuRHSD2qbpGOAeNVVaOE88Bv3pIGPguMw0qAdEDo20hRYH23LIyvBwB"
	"oCnyFNnAViMtLa2QlXSliV9a9BKOXYjWzAeso2SF4pdHcvd5AgMBAAGjgZMwgZAw"
	"DAYDVR0TAQH/BAIwADAaBgNVHREEEzARgg90ZXN0LmdudXRscy5vcmcwEwYDVR0l"
	"BAwwCgYIKwYBBQUHAwEwDwYDVR0PAQH/BAUDAweAADAdBgNVHQ4EFgQUan6mlccq"
	"Uy1Z64wvRv3xxg4h2ykwHwYDVR0jBBgwFoAUSCM0UwqJMThKWurKttKm3s4dKxgw"
	"DQYJKoZIhvcNAQELBQADggExAKAOMyMLpk0u2UTwwFWtr1hfx7evo2J7dgco410I"
	"DN/QWoe2Xlcxcp1h5R9rX1I3KU2WGFtdXqiMsllCLnrDEKZmlks0uz76bCpKmM99"
	"/1MDlY7mGCr/2PPx53USK5J5JTiqgp6r7qAcDAnpYvrPH45kk7iqwh02DhAxRnGR"
	"CW7KWK8h7uu0Az9iBT2YfV372g4fRDK3fqYzJofQwbhSiUuJ7wyZCRhGOoxMMmDb"
	"KBbc1wAYXW+tlv2cSbfzRvSxMR+CzkyH2tGDxeN//aZUfGmQ8IzWUQ7UtK5z+Q0E"
	"fL6fZtm2SdGabGpV1UYoGpwOtOngK+m0i9SqrMD7g5+SMhc1VuvVuTtxjr5Cha8l"
	"X0HEZtxgFrkdfMD4yLAqiguaCBngtbRmELF5VpebmJbiLVU="
	"-----END CERTIFICATE-----",
	"-----BEGIN CERTIFICATE-----"
	"MIIDkTCCAkmgAwIBAgIUY9cJ4NLNFEaojJHdP1I4Q7OHNJwwDQYJKoZIhvcNAQEL"
	"BQAwGTEXMBUGA1UEAxMOR251VExTIHRlc3QgQ0EwHhcNMTgxMjMxMjMwMDAwWhcN"
	"MjAwNTMwMjIwMDAwWjAmMSQwIgYDVQQDExtHbnVUTFMgdGVzdCBpbnRlcm1lZGlh"
	"dGUgQ0EwggFSMA0GCSqGSIb3DQEBAQUAA4IBPwAwggE6AoIBMQC0ayeYJa/B/x7K"
	"sH702LztQ4ZnVF3atB7CkF+DPAIR/BNyhbKIpGVBC3ZfI76Kn/55S3M7LsdLPL8W"
	"yZdVNRfzoXJLMMLgJ5QS81YA5s6CSxFdpB6b+vq5GypNGLW6peYMx6iooW2qiITc"
	"lg6ybBw1qufHlD351cfCog1Ls2569whfxQnNFZMa95jfKkxmiSTtH9AWY4FlpVg7"
	"oc0lYpuZgVQIFxjsfC8IojsoVzKdF0cKhvtisUGZ5vveqOogfvMb7rrqmiFkKZLy"
	"rXPlGQWdN1PiEZ8YXyK64osNAIyeL6eHPUC+SqKlkggMLmHAWHyameHWrIM5Jc8+"
	"G+3ro22dy8U43sHHbps0FL4wPoKQHrlKmnbk7zMMRqIxcvbDYQv4qmeJ9KXldjeh"
	"KZ+Aeap1AgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE"
	"ADAdBgNVHQ4EFgQUSCM0UwqJMThKWurKttKm3s4dKxgwHwYDVR0jBBgwFoAUHncj"
	"bWcxH5EHm5Yv7PzIRv6M4QMwDQYJKoZIhvcNAQELBQADggExAHP1UAQ/nvuQtRZF"
	"Q4b96yxVwCjMjn7knLyLNtyYGE3466xvE/ofvx5lgaR06ez/G17XP+Ok5SLJNUVc"
	"mplTERCv5CgnX7R5VdGJkkD1repaYxaTtwyJz0AfYEMRUj3jfaeLaiUKJvEW5RRs"
	"I3solY18sy/m/xGrH2X0GTNfKM9BURENABsppt07jxH719nF9m9SynV/Z2hE5hlv"
	"5e5vyPt4wyRPIJLUI3TKAlvb1s40zz3ua7ZTgQL/cOxfY4f9pRKW9CMB3uF69OP9"
	"COAxrmHVZsImmDZ6qO1qQrbY1KN/cX5kG4pKg7Ium723aOlwcWzEDXKumD960fN1"
	"5g+HrjNs6kW+r9Q5QS8qV5s8maZNcxTrMvQ1fF2AKBNI3Z3U7vmtrSeqxIXp3rGH"
	"iJwOKIk="
	"-----END CERTIFICATE-----",
	NULL
};

static const char *superseding_ca[] = {
	"-----BEGIN CERTIFICATE-----"
	"MIIDkzCCAkugAwIBAgIUIs7jB4Q4sFcdCmzWVHbJLESC3T4wDQYJKoZIhvcNAQEL"
	"BQAwGTEXMBUGA1UEAxMOR251VExTIHRlc3QgQ0EwIBcNMjAwNTMxMTIxMzEwWhgP"
	"OTk5OTEyMzEyMzU5NTlaMCYxJDAiBgNVBAMTG0dudVRMUyB0ZXN0IGludGVybWVk"
	"aWF0ZSBDQTCCAVIwDQYJKoZIhvcNAQEBBQADggE/ADCCAToCggExALRrJ5glr8H/"
	"HsqwfvTYvO1DhmdUXdq0HsKQX4M8AhH8E3KFsoikZUELdl8jvoqf/nlLczsux0s8"
	"vxbJl1U1F/OhckswwuAnlBLzVgDmzoJLEV2kHpv6+rkbKk0Ytbql5gzHqKihbaqI"
	"hNyWDrJsHDWq58eUPfnVx8KiDUuzbnr3CF/FCc0Vkxr3mN8qTGaJJO0f0BZjgWWl"
	"WDuhzSVim5mBVAgXGOx8LwiiOyhXMp0XRwqG+2KxQZnm+96o6iB+8xvuuuqaIWQp"
	"kvKtc+UZBZ03U+IRnxhfIrriiw0AjJ4vp4c9QL5KoqWSCAwuYcBYfJqZ4dasgzkl"
	"zz4b7eujbZ3LxTjewcdumzQUvjA+gpAeuUqaduTvMwxGojFy9sNhC/iqZ4n0peV2"
	"N6Epn4B5qnUCAwEAAaNkMGIwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMD"
	"BwQAMB0GA1UdDgQWBBRIIzRTCokxOEpa6sq20qbezh0rGDAfBgNVHSMEGDAWgBQe"
	"dyNtZzEfkQebli/s/MhG/ozhAzANBgkqhkiG9w0BAQsFAAOCATEAcF9R9VGQxTwW"
	"aOjeIeQ9ZJxybaj0BaXC8xR4b9uZloS9d/RBFTjgRbQ82yqaj7f80mgUtabKRfTA"
	"ltV2MgTbJdOjwGzEDtKGhClBbovnEGrYTbPBT9rgfYPt0q7SMBr6AzGAPt+ltwI7"
	"9yntV81qvTxvW5MEEo0j2MuA3NT3oqe+w1rUKNQCWhnN2TUhJGkTlaaMozcgNFaE"
	"Dplop4dtvCGtupxOjC3Nf6FWq1k7iZQxX70AFBYVMpuF7qGh6qDp+T1hmTCSVzxP"
	"SfDQIBjhKgy4clhkuR5SRxhN74RX+/5eiQyVLxzr+eIhqzJhPqUCmVnCLcqYdNRi"
	"hpHic4uJm0wGOKYTI7EG8rb4ZP4Jz6k4iN9CnL/+kiiW5otSl3YyCAuao5VKdDq9"
	"izchzb9eow=="
	"-----END CERTIFICATE-----",
	"-----BEGIN CERTIFICATE-----"
	"MIIDZTCCAh2gAwIBAgIULcrECQOBgPaePBfBHXcyZiU0IiYwDQYJKoZIhvcNAQEL"
	"BQAwGTEXMBUGA1UEAxMOR251VExTIHRlc3QgQ0EwIBcNMjAwNTMxMTIxMTQzWhgP"
	"OTk5OTEyMzEyMzU5NTlaMBkxFzAVBgNVBAMTDkdudVRMUyB0ZXN0IENBMIIBUjAN"
	"BgkqhkiG9w0BAQEFAAOCAT8AMIIBOgKCATEAnORCsX1unl//fy2d1054XduIg/3C"
	"qVBaT3Hca65SEoDwh0KiPtQoOgZLdKY2cobGs/ojYtOjcs0KnlPYdmtjEh6WEhuJ"
	"U95v4TQdC4OLMiE56eIGq252hZAbHoTL84Q14DxQWGuzQK830iml7fbw2WcIcRQ8"
	"vFGs8SzfXw63+MI6Fq6iMAQIqP08WzGmRRzL5wvCiPhCVkrPmwbXoABub6AAsYwW"
	"PJB91M9/lx5gFH5k9/iPfi3s2Kg3F8MOcppqFYjxDSnsfiz6eMh1+bYVIAo367vG"
	"VYHigXMEZC2FezlwIHaZzpEoFlY3a7LFJ00yrjQ910r8UE+CEMTYzE40D0olCMo7"
	"FA9RCjeO3bUIoYaIdVTUGWEGHWSeoxGei9Gkm6u+ASj8f+i0jxdD2qXsewIDAQAB"
	"o0MwQTAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAwHQYDVR0OBBYE"
	"FB53I21nMR+RB5uWL+z8yEb+jOEDMA0GCSqGSIb3DQEBCwUAA4IBMQAeMSzMyuTy"
	"FjXTjxAUv010bsr6e6fI9txq/S1tXmWWJV/8aeARthuOFZO5Jjy3C5aMbac2HDV4"
	"Otu0+JLaoEMSXvorAhValVuq06i5cmaPzvJBcxMWzlEAXfavSwHv5Q+kqNU3z81S"
	"WnjEpMHcl9OyER7o9IhF55Xom2BXY5XL83QOzQ4C3bpKrNevZC7i7zS8NoYRGP+8"
	"w21JseXkWQW4o2hkFqbCcRE1dlMW02iJE28RZ5aBFDIm2Y6zuLaXZIkaO7E41CAw"
	"IUyhowm/S1HcmQnhruAGKJvQtB6jvnhZb7pgnuSkhIvAQgw93CLE985KEua1ifY2"
	"p1d/6ho2TWotHHqDnDkB8pC0Wzai8R+63z18Kt0gROX2QItCyFksjNJqYPbgwZgt"
	"eh1COrLsOJo+"
	"-----END CERTIFICATE-----",
	NULL
};

static const char *rsa_sha1_in_trusted[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIID0jCCAoqgAwIBAgIUezaBB7f4TW75oc3UV57oJvXmbBYwDQYJKoZIhvcNAQEL\n"
	"BQAwGTEXMBUGA1UEAxMOR251VExTIHRlc3QgQ0EwHhcNMjEwNTAzMTQyNzIxWhcN\n"
	"MjIwNTAzMTQyNzIxWjA3MRgwFgYDVQQDEw90ZXN0LmdudXRscy5vcmcxGzAZBgNV\n"
	"BAoTEkdudVRMUyB0ZXN0IHNlcnZlcjCCAVIwDQYJKoZIhvcNAQEBBQADggE/ADCC\n"
	"AToCggExALRrJ5glr8H/HsqwfvTYvO1DhmdUXdq0HsKQX4M8AhH8E3KFsoikZUEL\n"
	"dl8jvoqf/nlLczsux0s8vxbJl1U1F/OhckswwuAnlBLzVgDmzoJLEV2kHpv6+rkb\n"
	"Kk0Ytbql5gzHqKihbaqIhNyWDrJsHDWq58eUPfnVx8KiDUuzbnr3CF/FCc0Vkxr3\n"
	"mN8qTGaJJO0f0BZjgWWlWDuhzSVim5mBVAgXGOx8LwiiOyhXMp0XRwqG+2KxQZnm\n"
	"+96o6iB+8xvuuuqaIWQpkvKtc+UZBZ03U+IRnxhfIrriiw0AjJ4vp4c9QL5KoqWS\n"
	"CAwuYcBYfJqZ4dasgzklzz4b7eujbZ3LxTjewcdumzQUvjA+gpAeuUqaduTvMwxG\n"
	"ojFy9sNhC/iqZ4n0peV2N6Epn4B5qnUCAwEAAaOBkzCBkDAMBgNVHRMBAf8EAjAA\n"
	"MBoGA1UdEQQTMBGCD3Rlc3QuZ251dGxzLm9yZzATBgNVHSUEDDAKBggrBgEFBQcD\n"
	"ATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBRIIzRTCokxOEpa6sq20qbezh0r\n"
	"GDAfBgNVHSMEGDAWgBQedyNtZzEfkQebli/s/MhG/ozhAzANBgkqhkiG9w0BAQsF\n"
	"AAOCATEAXs8lOV231HQerhSGEjZJz0vBuA3biKYlu3cwCTKvF6EOyYMSWOnfqqD0\n"
	"eDhpo1pzGtUa2zYLHagb+sU2NSTe0sqP+PK1giUg8X8/tRtWKk1p/m76yK/3iaty\n"
	"flgz+eMai4xQu2FvAJzIASFjM9R+Pgpcf/zdvkiUPv8Rdm9FieyAZnJSo9hJHLxN\n"
	"x60tfC5yyswdbGGW0GbJ2kr+xMfVZvxgO/x6AXlOaUGQ+jZAu9eJwFQMDW5h5/S1\n"
	"PJkIt7f7jkU33cG+BawcjhT0GzxuvDnnCG0L7/z7bR+Sw2kNKqHbHorzv91R20Oh\n"
	"CIISJPkiiP+mYcglTp1d9gw09GwSkGbldb9ibfc0hKyxiImFfIiTqDbXJcpKH98o\n"
	"W8hWkb20QURlY+QM5MD49znfhPKMTQ==\n"
	"-----END CERTIFICATE-----\n",
	"-----BEGIN CERTIFICATE-----\n"
	"MIID2TCCAkGgAwIBAgIUWsb4DATcefXbo0WrBfgqVMvPGawwDQYJKoZIhvcNAQEL\n"
	"BQAwHjEcMBoGA1UEAxMTR251VExTIHRlc3Qgcm9vdCBDQTAeFw0yMTA1MDMxNDI2\n"
	"MzVaFw0yMjA1MDMxNDI2MzVaMBkxFzAVBgNVBAMTDkdudVRMUyB0ZXN0IENBMIIB\n"
	"UjANBgkqhkiG9w0BAQEFAAOCAT8AMIIBOgKCATEAnORCsX1unl//fy2d1054XduI\n"
	"g/3CqVBaT3Hca65SEoDwh0KiPtQoOgZLdKY2cobGs/ojYtOjcs0KnlPYdmtjEh6W\n"
	"EhuJU95v4TQdC4OLMiE56eIGq252hZAbHoTL84Q14DxQWGuzQK830iml7fbw2WcI\n"
	"cRQ8vFGs8SzfXw63+MI6Fq6iMAQIqP08WzGmRRzL5wvCiPhCVkrPmwbXoABub6AA\n"
	"sYwWPJB91M9/lx5gFH5k9/iPfi3s2Kg3F8MOcppqFYjxDSnsfiz6eMh1+bYVIAo3\n"
	"67vGVYHigXMEZC2FezlwIHaZzpEoFlY3a7LFJ00yrjQ910r8UE+CEMTYzE40D0ol\n"
	"CMo7FA9RCjeO3bUIoYaIdVTUGWEGHWSeoxGei9Gkm6u+ASj8f+i0jxdD2qXsewID\n"
	"AQABo2QwYjAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAwHQYDVR0O\n"
	"BBYEFB53I21nMR+RB5uWL+z8yEb+jOEDMB8GA1UdIwQYMBaAFCApU0Q1pxZL+AW3\n"
	"GctysPWxl+SfMA0GCSqGSIb3DQEBCwUAA4IBgQBbboeDr/rLT1tZWrdHq8FvflGm\n"
	"EpxZIRU4DdDD/SUCWSPQvjBq0MvuKxs5FfJCKrDf2kS2qlZ1rO0AuWwREoDeTOEc\n"
	"arjFoCry+JQ+USqS5F4gsp4XlYvli27iMp3dlnhFXEQQy7/y+gM5c9wnMi8v/LUz\n"
	"AV6QHX0fkb4XeazeJ+Nq0EkjqiYxylN6mP+5LAEMBG/wGviAoviQ5tN9zdoQs/nT\n"
	"3jTw3cOauuPjdcOTfo71+/MtBzhPchgNIyQo4aB40XVWsLAoruL/3CFFlTniihtd\n"
	"zA2zA7JvbuuKx6BOv2IbWOUweb732ZpYbDgEcXp/6Cj/SIUGxidpEgdCJGqyqdC7\n"
	"b58ujxclC6QTcicw+SX5LBox8WGLfj+x+V3uVBz9+EK608xphTj4kLh9peII9v3n\n"
	"vBUoZRTiUTCvH4AJJgAfa3mYrSxzueuqBOwXcvZ+8OJ0J1CP21pmK5nxR7f1nm9Q\n"
	"sYA1VHfC2dtyAYlByeF5iHl5hFR6vy1jJyzxg2M=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *rsa_sha1_in_trusted_ca[] = {
	/* This CA is generated with the same key as rsa_sha1_in_trusted[1], but
	 * self-signed using SHA-1.
	 */
	"-----BEGIN CERTIFICATE-----\n"
	"MIIDYzCCAhugAwIBAgIUahO8CvYPHTAltKCC2rAIcXUiLlAwDQYJKoZIhvcNAQEF\n"
	"BQAwGTEXMBUGA1UEAxMOR251VExTIHRlc3QgQ0EwHhcNMjEwNTAzMTQyMDM1WhcN\n"
	"MjIwNTAzMTQyMDM1WjAZMRcwFQYDVQQDEw5HbnVUTFMgdGVzdCBDQTCCAVIwDQYJ\n"
	"KoZIhvcNAQEBBQADggE/ADCCAToCggExAJzkQrF9bp5f/38tnddOeF3biIP9wqlQ\n"
	"Wk9x3GuuUhKA8IdCoj7UKDoGS3SmNnKGxrP6I2LTo3LNCp5T2HZrYxIelhIbiVPe\n"
	"b+E0HQuDizIhOeniBqtudoWQGx6Ey/OENeA8UFhrs0CvN9Ippe328NlnCHEUPLxR\n"
	"rPEs318Ot/jCOhauojAECKj9PFsxpkUcy+cLwoj4QlZKz5sG16AAbm+gALGMFjyQ\n"
	"fdTPf5ceYBR+ZPf4j34t7NioNxfDDnKaahWI8Q0p7H4s+njIdfm2FSAKN+u7xlWB\n"
	"4oFzBGQthXs5cCB2mc6RKBZWN2uyxSdNMq40PddK/FBPghDE2MxONA9KJQjKOxQP\n"
	"UQo3jt21CKGGiHVU1BlhBh1knqMRnovRpJurvgEo/H/otI8XQ9ql7HsCAwEAAaND\n"
	"MEEwDwYDVR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBQe\n"
	"dyNtZzEfkQebli/s/MhG/ozhAzANBgkqhkiG9w0BAQUFAAOCATEAYLm/4DfUp+mA\n"
	"S/23a2bwybJoPCMzKZpi+veXkqoq/a/BCUkFpqnjpVjz0ujVKK121oeOPBAa/mG1\n"
	"Y3fJYP+b3PloL/6xj/8680TveGirCr0Rp/8XWa8lt+Ge8DM3mfTGWFTWHa0lD9VK\n"
	"gjV1oNZNLe5SKA6dJLAp/NjCxc/vuOkThQPeaoO5Iy/Z6m7CpTLO7T4syJFtDmSn\n"
	"Pa/yFUDTgJYFlGVM+KC1r8bhZ6Ao1CAXTcT5Lcbe/aCcyk6B3J2AnYsqPMVNEVhb\n"
	"9eMGO/WG24hMLy6eb1r/yL8uQ/uGi2rRlNJN8GTg09YR7l5fHrHxuHc/sme0jsnJ\n"
	"wtqGLCJsrh7Ae1fKVUueO00Yx9BGuzLswMvnT5f0oYs0jrXgMrTbIWS/DjOcYIHb\n"
	"w3SV1ZRcNg==\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *rsa_sha1_not_in_trusted[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIID0jCCAoqgAwIBAgIUNCvPV9OvyuVMtnkC3ZAvh959h4MwDQYJKoZIhvcNAQEL\n"
	"BQAwGTEXMBUGA1UEAxMOR251VExTIHRlc3QgQ0EwHhcNMjEwNTA0MDg0NzAzWhcN\n"
	"MjIwNTA0MDg0NzAzWjA3MRgwFgYDVQQDEw90ZXN0LmdudXRscy5vcmcxGzAZBgNV\n"
	"BAoTEkdudVRMUyB0ZXN0IHNlcnZlcjCCAVIwDQYJKoZIhvcNAQEBBQADggE/ADCC\n"
	"AToCggExALRrJ5glr8H/HsqwfvTYvO1DhmdUXdq0HsKQX4M8AhH8E3KFsoikZUEL\n"
	"dl8jvoqf/nlLczsux0s8vxbJl1U1F/OhckswwuAnlBLzVgDmzoJLEV2kHpv6+rkb\n"
	"Kk0Ytbql5gzHqKihbaqIhNyWDrJsHDWq58eUPfnVx8KiDUuzbnr3CF/FCc0Vkxr3\n"
	"mN8qTGaJJO0f0BZjgWWlWDuhzSVim5mBVAgXGOx8LwiiOyhXMp0XRwqG+2KxQZnm\n"
	"+96o6iB+8xvuuuqaIWQpkvKtc+UZBZ03U+IRnxhfIrriiw0AjJ4vp4c9QL5KoqWS\n"
	"CAwuYcBYfJqZ4dasgzklzz4b7eujbZ3LxTjewcdumzQUvjA+gpAeuUqaduTvMwxG\n"
	"ojFy9sNhC/iqZ4n0peV2N6Epn4B5qnUCAwEAAaOBkzCBkDAMBgNVHRMBAf8EAjAA\n"
	"MBoGA1UdEQQTMBGCD3Rlc3QuZ251dGxzLm9yZzATBgNVHSUEDDAKBggrBgEFBQcD\n"
	"ATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBRIIzRTCokxOEpa6sq20qbezh0r\n"
	"GDAfBgNVHSMEGDAWgBQedyNtZzEfkQebli/s/MhG/ozhAzANBgkqhkiG9w0BAQsF\n"
	"AAOCATEAWs/Qa1Ebydwo4Ke2KEdy5cUTSZjnoz93XpbrP9W60MJ4d2DIQPcYUcLF\n"
	"+glez+mRtVXDRtH5V/4yZX1EdgrPVQGeVlO5HbNiYyYw/Yj3H6kzWtUbBxdOAOE/\n"
	"/ul8RCKKMfvYBHCBgjBMW0aFm31Q1Z8m8nanBusyJ0DG1scBHu4/3vTCZthZAxc5\n"
	"3l3t/jjsNRS+k5t6Ay8nEY1tAZSGVqN8qufzO2NBO06sQagp09FTfDh581OBcVtF\n"
	"X7O0cffAWHk3JoywzEWFEAhVPqFlk07wG2O+k+fYZfavsJko5q+yWkxu8RDh4wAx\n"
	"7UzKudGOQ+NhfYJ7N7V1/RFg1z75gE3GTUX7qmGZEVDOsMyiuUeYg8znyYpBV55Q\n"
	"4BNr0ukwmwOdvUf+ksCu6PdOGaqThA==\n"
	"-----END CERTIFICATE-----\n",
	/* ICA with SHA1 signature */
	"-----BEGIN CERTIFICATE-----\n"
	"MIID2TCCAkGgAwIBAgIUYaKJkQft87M1TF+Jd30py3yIq4swDQYJKoZIhvcNAQEF\n"
	"BQAwHjEcMBoGA1UEAxMTR251VExTIHRlc3Qgcm9vdCBDQTAeFw0yMTA1MDQwODQ1\n"
	"NDdaFw0yMjA1MDQwODQ1NDdaMBkxFzAVBgNVBAMTDkdudVRMUyB0ZXN0IENBMIIB\n"
	"UjANBgkqhkiG9w0BAQEFAAOCAT8AMIIBOgKCATEAnORCsX1unl//fy2d1054XduI\n"
	"g/3CqVBaT3Hca65SEoDwh0KiPtQoOgZLdKY2cobGs/ojYtOjcs0KnlPYdmtjEh6W\n"
	"EhuJU95v4TQdC4OLMiE56eIGq252hZAbHoTL84Q14DxQWGuzQK830iml7fbw2WcI\n"
	"cRQ8vFGs8SzfXw63+MI6Fq6iMAQIqP08WzGmRRzL5wvCiPhCVkrPmwbXoABub6AA\n"
	"sYwWPJB91M9/lx5gFH5k9/iPfi3s2Kg3F8MOcppqFYjxDSnsfiz6eMh1+bYVIAo3\n"
	"67vGVYHigXMEZC2FezlwIHaZzpEoFlY3a7LFJ00yrjQ910r8UE+CEMTYzE40D0ol\n"
	"CMo7FA9RCjeO3bUIoYaIdVTUGWEGHWSeoxGei9Gkm6u+ASj8f+i0jxdD2qXsewID\n"
	"AQABo2QwYjAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAwHQYDVR0O\n"
	"BBYEFB53I21nMR+RB5uWL+z8yEb+jOEDMB8GA1UdIwQYMBaAFCApU0Q1pxZL+AW3\n"
	"GctysPWxl+SfMA0GCSqGSIb3DQEBBQUAA4IBgQAewBcAGUGX28I5PDtuJkxoHonD\n"
	"muHdXpYnrz1YXN4b7odNXockz++Xovgj126fo+PeWgmaaCic98ZcGnyVTi9+3oqN\n"
	"2Bf4NNfyzSccgZZTphzbwjMcnc983HLQgsLSAOVivPHj5GEN58EWWamc9yA0VjGn\n"
	"cuYmFN2dlFA8/ClEbVGu3UXBe6OljR5zUr+6oiSp2J+Rl7SerVSHlst07iU2tkeB\n"
	"dlfOD5CquUGSka3SKvEfvu5SwYrCQVfYB6eMLInm7A0/ca0Jn3Oh4fMf2rIg/E3K\n"
	"qsopxsu8BXrLoGK4MxbxPA65JpczhZgilQQi3e3RIvxrvyD2qamjaNbyG5cr8mW4\n"
	"VOLf3vUORbkTi5sE7uRMu2B3z3N7ajsuQM8RHB17hOCB2FO/8rermq/oeJNtx57L\n"
	"5s5NxCHYTksQ4gkpR4gfTIO/zwXJSwGa/Zi2y2wIi/1qr7lppBsKV2rDWX7QiIeA\n"
	"PxOxyJA2eSeqCorz9vk3aHXleSpxsWGgKiJVmV0=\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

static const char *rsa_sha1_not_in_trusted_ca[] = {
	"-----BEGIN CERTIFICATE-----\n"
	"MIIEDTCCAnWgAwIBAgIUd5X8NZput+aNPEd9h92r4KAu16MwDQYJKoZIhvcNAQEL\n"
	"BQAwHjEcMBoGA1UEAxMTR251VExTIHRlc3Qgcm9vdCBDQTAeFw0yMTA1MDMxNDI1\n"
	"MDNaFw0yMjA1MDMxNDI1MDNaMB4xHDAaBgNVBAMTE0dudVRMUyB0ZXN0IHJvb3Qg\n"
	"Q0EwggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQCsFAaMb/iRN+OFqQNh\n"
	"OkkXGZlb+eLerLuB9ELnYwyLIh4MTXh0RjFZdCQLsQHfY/YFv0C50rmoXTA/d3Ef\n"
	"K/P243KjX0XBWjO9TBuN0zth50eq94zf69yxA/a+kmT+O5YLfhi2ELM5F3IjOUoZ\n"
	"lL0IGlFJwauAkaNylp/Evd5nW7g5DUJvMm4A3RXNfZt9gAD4lPRwryQq9jxT48Xu\n"
	"fB0kAPEG/l/Izbz2rYin5+nySL+a0CSNuEbITxidtMhveB747oR0QS2sMQKji1ur\n"
	"pRJ945SHiYJIgVuFAJc9StikSyIrxZgK45kAzcQAyRWWKiMNH5PprGFYJp+ypwhm\n"
	"1t8Bphj2RFJAG3XRRZF/9uJIYc5mEHCsZFZ/IFRaKqyN30kAUijgNt+lW5mZXVFU\n"
	"aqzV2zHjSG8jsGdia3cfBP46Z1q2eAh5jOCucTq1F7qZdVhOFmP9jFE6Uy5Kbwgc\n"
	"kNAnsEllQeJQL2odVa7woKkZZ4M/c72X5tpBU38Rs3krn3sCAwEAAaNDMEEwDwYD\n"
	"VR0TAQH/BAUwAwEB/zAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBQgKVNENacW\n"
	"S/gFtxnLcrD1sZfknzANBgkqhkiG9w0BAQsFAAOCAYEAaZMV71mZ9FYoVdpho61h\n"
	"WWPs5GppQLJ1w70DNtGZ+lFrk/KopeDvOu1i61QLWRzcZCZMl+npiX1KH5kjVo3v\n"
	"C9G8kdMW6EVRk5p6qCJMPFN2U+grMMp50aY5kmw+/v+Lhk5T/VG93l63P91FkUre\n"
	"o8qhOudJExoUnR1uB9M6HMAxVn8Lm/N1LGPiP6A6Pboo716H7mg/A7pv9zoZ6jUp\n"
	"7x693mA/b3I/QpDx/nJcmcdqxgEuW+aRlFXgnYZRFAawxi+5M9EwCWbkSTO4OMHP\n"
	"Qlvak3tJO+wb92b0cICOOtzIPgQ+caiLg9d0FvesALmQzDmNmtqynoO85+Ia2Ywh\n"
	"nxKPlpeImhLN9nGl9sOeW2m4mnA5r0h1vgML4v/MWL4TQhXallc31uFNj5HyFaTh\n"
	"6Mr0g3GeQgN0jpT+aIOiKuW9fLts54+Ntj1NN40slqi3Y+/Yd6xhj+NgmbRvybZu\n"
	"tnYFXKC0Q+QUf38horqG2Mc3/uh8MOm0eYUXwGJOdXYD\n"
	"-----END CERTIFICATE-----\n",
	NULL
};

#if defined __clang__ || __GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 5)
#  pragma GCC diagnostic push
#  pragma GCC diagnostic ignored "-Wunused-variable"
#endif

static struct
{
  const char *name;
  const char **chain;
  const char **ca;
  unsigned int verify_flags;
  unsigned int expected_verify_result;
  const char *purpose;
  time_t expected_time;
  unsigned notfips;
} chains[] =
{
  { "CVE-2014-0092", cve_2014_0092_check, &cve_2014_0092_check[1],
    0,
    GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL, 1412850586},
  { "CVE-2008-4989", cve_2008_4989_chain, &cve_2008_4989_chain[2],
    GNUTLS_VERIFY_ALLOW_BROKEN,
    GNUTLS_CERT_SIGNER_NOT_FOUND | GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID, NULL},
  { "amazon.com ok", verisign_com_chain_g5, &verisign_com_chain_g5[4],
    GNUTLS_VERIFY_ALLOW_BROKEN | GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LOW),
    0, NULL},
  { "verisign.com v1 fail", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_ALLOW_BROKEN,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID, NULL},
  { "verisign.com v1 ok", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_ALLOW_BROKEN | GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LOW),
    0, NULL},
  { "verisign.com v1 not ok due to profile", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LEGACY),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "verisign.com v1 not ok due to profile", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_HIGH),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "citibank.com v1 fail", citibank_com_chain, &citibank_com_chain[2],
    GNUTLS_VERIFY_ALLOW_BROKEN | GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},
  { "expired self signed", pem_self_cert, &pem_self_cert[0],
    0, GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID, NULL},
  { "self signed", pem_self_cert, &pem_self_cert[0],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0, NULL},
  { "ca=false", thea_chain, &thea_chain[1],
    GNUTLS_VERIFY_ALLOW_BROKEN,
    GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},
  { "ca=false2", thea_chain, &thea_chain[1],
    GNUTLS_VERIFY_ALLOW_BROKEN, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},
  { "hbci v1 fail", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_ALLOW_BROKEN | GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},
  { "hbci v1 ok expired", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_ALLOW_BROKEN,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID, NULL},
  { "hbci v1 ok", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_ALLOW_BROKEN|GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0, NULL},
  { "rsa-md5 fail", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "rsa-md5 not ok", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD2,
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "rsa-md5 not ok2", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID, NULL},
  { "rsa-md5 ok", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5, 0, NULL},
  { "rsa-md5 ok - allow broken", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_BROKEN, 0, NULL},
  { "v1ca fail", v1ca, &v1ca[2],
    GNUTLS_VERIFY_ALLOW_BROKEN|GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},

  { "pathlen fail", pathlen_check, &pathlen_check[2],
    GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT | GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL},

  /* Test whether a V1 root certificate is correctly accepted */
  { "v1root fail", v1_root_check, &v1_root_check[1],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT | GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},
  { "v1root ok", v1_root_check, &v1_root_check[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0, NULL},

  /* test whether a v1 intermediate certificate is rejected */
  { "v1invalid fail", v1_intermed_check, &v1_intermed_check[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, NULL},
  { "v1 leaf ok", &v1_intermed_check[1], &v1_intermed_check[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0, NULL},

  { "v1ca expired", v1ca, &v1ca[2],
    GNUTLS_VERIFY_ALLOW_BROKEN,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID , NULL},
  { "v1ca (sha1) ok - allow broken", v1ca, &v1ca[2], /* check GNUTLS_VERIFY_ALLOW_BROKEN */
    GNUTLS_VERIFY_ALLOW_BROKEN|GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0, NULL},
  { "v1ca (sha1) ok - allow sha1", v1ca, &v1ca[2], /* check GNUTLS_VERIFY_ALLOW_SIGN_WITH_SHA1 */
    GNUTLS_VERIFY_ALLOW_SIGN_WITH_SHA1|GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0, NULL},
  { "v1ca2 expired", v1ca, &v1ca[2],
    GNUTLS_VERIFY_ALLOW_BROKEN|GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID, NULL},
  { "v1ca2 ok", v1ca, &v1ca[2],
    GNUTLS_VERIFY_ALLOW_BROKEN|GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT,
    0, NULL},
  { "cacertrsamd5 fail", cacertrsamd5, &cacertrsamd5[2],
    0, GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "cacertrsamd5 ok", cacertrsamd5, &cacertrsamd5[2],
    GNUTLS_VERIFY_ALLOW_BROKEN, 0, NULL},
  { "cacertrsamd5 short-cut not ok", cacertrsamd5, &cacertrsamd5[0],
    GNUTLS_VERIFY_DO_NOT_ALLOW_SAME,
    GNUTLS_CERT_SIGNER_NOT_FOUND | GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "cacertrsamd5 short-cut ok", cacertrsamd5, &cacertrsamd5[1],
    GNUTLS_VERIFY_ALLOW_BROKEN, 0, NULL},
  { "ecc cert ok", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_HIGH), 0, NULL},
  { "ecc cert ok", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_SUITEB128), 0, NULL},
  { "ecc cert not ok (due to profile)", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_ULTRA), 
	GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "ecc cert not ok (due to profile)", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_SUITEB192), 
	GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL},
  { "name constraints: empty CN, empty SAN, permitted dns range", nc_good0, &nc_good0[2], 0, 0, 0, 1427270515},
  { "name constraints: dns in permitted range", nc_good1, &nc_good1[4], 0, 0, NULL, 1412850586},
  { "name constraints: ipv6 in permitted range", nc_good2, &nc_good2[4], 0, 0, NULL, 1467193927},
  { "name constraints: 2 constraints (dns, dns), non-intuitive order", nc_bad0, &nc_bad0[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1468920734},
  { "name constraints: empty CN, empty excluded dns", nc_bad1, &nc_bad1[2], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1412850586},
  { "name constraints: multiple dns constraints, empty intersection", nc_bad2, &nc_bad2[4], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1412850586},
  { "name constraints: dns excluded range", nc_bad3, &nc_bad3[2], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1412850586},
  { "name constraints: basic dns intersection", nc_bad4, &nc_bad4[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1412850586},
  { "name constraints: IP in excluded range", nc_bad5, &nc_bad5[2], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1469540953},
  { "name constraints: 2 constraints (dns, email), non-intuitive order", nc_bad6, &nc_bad6[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, NULL, 1469540953},
  { "not-modified", modified2, &modified2[3], GNUTLS_VERIFY_ALLOW_BROKEN, 0, NULL, 1412850586},
  { "kp-interm", kp_fail1, &kp_fail1[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_PURPOSE_MISMATCH, GNUTLS_KP_TLS_WWW_SERVER, 1412850586},
  { "kp-fin", kp_fail2, &kp_fail2[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_PURPOSE_MISMATCH, GNUTLS_KP_TLS_WWW_SERVER, 1412850586},
  { "kp-ok", kp_ok, &kp_ok[3], 0, 0, GNUTLS_KP_OCSP_SIGNING, 1412850586},
  { "tls features - intermediate no ext", tls_feat_inter_no_ext, &tls_feat_inter_no_ext[3], 0, GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE|GNUTLS_CERT_INVALID, 0, 1466612070},
  { "tls features - end no ext", tls_feat_no_ext, &tls_feat_no_ext[3], 0, GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE|GNUTLS_CERT_INVALID, 0, 1466612070},
  { "tls features - intermediate is subset", tls_feat_inter_subset_fail, &tls_feat_inter_subset_fail[3], 0, GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE|GNUTLS_CERT_INVALID, 0, 1466612070},
  { "tls features - intermediate has unrelated vals", tls_feat_inter_unrelated_fail, &tls_feat_inter_unrelated_fail[3], 0, GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE|GNUTLS_CERT_INVALID, 0, 1466612070},
  { "tls features - end is superset", tls_feat_superset, &tls_feat_superset[3], 0, 0, 0, 1466612070},
  { "tls features - ok", tls_feat_ok, &tls_feat_ok[3], 0, 0, 0, 1466612070},
  { "unknown crit extension on root - fail", unknown_critical_extension_on_root, &unknown_critical_extension_on_root[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_UNKNOWN_CRIT_EXTENSIONS, GNUTLS_KP_TLS_WWW_SERVER, 1488365541},
  { "unknown crit extension on root - success", unknown_critical_extension_on_root, &unknown_critical_extension_on_root[3], GNUTLS_VERIFY_IGNORE_UNKNOWN_CRIT_EXTENSIONS, 0, GNUTLS_KP_TLS_WWW_SERVER, 1488365541},
  { "unknown crit extension on intermediate - fail", unknown_critical_extension_on_intermediate, &unknown_critical_extension_on_intermediate[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_UNKNOWN_CRIT_EXTENSIONS, GNUTLS_KP_TLS_WWW_SERVER, 1488365541},
  { "unknown crit extension on intermediate - success", unknown_critical_extension_on_intermediate, &unknown_critical_extension_on_intermediate[3], GNUTLS_VERIFY_IGNORE_UNKNOWN_CRIT_EXTENSIONS, 0, GNUTLS_KP_TLS_WWW_SERVER, 1488365541},
  { "unknown crit extension on endcert - fail", unknown_critical_extension_on_endcert, &unknown_critical_extension_on_endcert[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_UNKNOWN_CRIT_EXTENSIONS, GNUTLS_KP_TLS_WWW_SERVER, 1488365541},
  { "unknown crit extension on endcert - success", unknown_critical_extension_on_endcert, &unknown_critical_extension_on_endcert[3], GNUTLS_VERIFY_IGNORE_UNKNOWN_CRIT_EXTENSIONS, 0, GNUTLS_KP_TLS_WWW_SERVER, 1488365541},
  { "rsa pss: invalid self sig - fail", rsa_pss_invalid_self_sig, &rsa_pss_invalid_self_sig[0], GNUTLS_VERIFY_DO_NOT_ALLOW_SAME, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 0, 1501138253},
  { "rsa pss: invalid chain with pkcs#1 1.5 sig - fail", rsa_pss_invalid_chain_with_pkcs1_sig, &rsa_pss_invalid_chain_with_pkcs1_sig[2], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 0, 1501138253},
  { "rsa pss: invalid chain with wrong hash (sha384-sha256) - fail", rsa_pss_invalid_chain_with_wrong_hash, &rsa_pss_invalid_chain_with_wrong_hash[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 0, 1501138253},
  { "rsa pss: smaller salt in sig than spki - fail", rsa_pss_chain_smaller_salt_in_sig_fail, &rsa_pss_chain_smaller_salt_in_sig_fail[3], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 0, 1550005473},
  { "rsa pss: chain with sha1 hash - fail", rsa_pss_chain_with_sha1_fail, &rsa_pss_chain_with_sha1_fail[3], 0, GNUTLS_CERT_INVALID, 0, 1501159136},
  { "rsa pss: chain with different mgf hash - fail", rsa_pss_chain_with_diff_mgf_oid_fail, &rsa_pss_chain_with_diff_mgf_oid_fail[3], 0, GNUTLS_CERT_INVALID, 0, 1501159136},
  { "rsa pss: chain with sha256 - ok", rsa_pss_chain_sha256_ok, &rsa_pss_chain_sha256_ok[3], 0, 0, 0, 1501138253},
  { "rsa pss: chain with sha384 - ok", rsa_pss_chain_sha384_ok, &rsa_pss_chain_sha384_ok[3], 0, 0, 0, 1501138253},
  { "rsa pss: chain with sha512 - ok", rsa_pss_chain_sha512_ok, &rsa_pss_chain_sha512_ok[3], 0, 0, 0, 1501138253},
  { "rsa pss: chain with increasing salt size - ok", rsa_pss_chain_increasing_salt_size_ok, &rsa_pss_chain_increasing_salt_size_ok[3], 0, 0, 0, 1501159136},
  { "rsa pss: chain with alternating signatures - ok", rsa_pss_chain_pkcs11_pss_pkcs1_ok, &rsa_pss_chain_pkcs11_pss_pkcs1_ok[3], 0, 0, 0, 1501159136},
  { "rsa pss: chain with changing hashes - ok", rsa_pss_chain_sha512_sha384_sha256_ok, &rsa_pss_chain_sha512_sha384_sha256_ok[3], 0, 0, 0, 1501159136},
  { "no subject id: chain with missing subject id, but valid auth id - ok", chain_with_no_subject_id_in_ca_ok, &chain_with_no_subject_id_in_ca_ok[4], 0, 0, 0, 1537518468},
#ifdef ENABLE_GOST
  { "gost 34.10-01 - ok", gost01, &gost01[2], GNUTLS_VERIFY_ALLOW_BROKEN, 0, 0, 1466612070, 1},
  { "gost 34.10-01 - not ok (due to gostr94)", gost01, &gost01[2], 0,
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL, 1466612070, 1},
  { "gost 34.10-01 - not ok (due to profile)", gost01, &gost01[2], GNUTLS_VERIFY_ALLOW_BROKEN|GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_ULTRA),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL, 1466612070, 1},
  { "gost 34.10-12-256 - ok", gost12_256, &gost12_256[0], 0, 0, 0, 1466612070, 1},
  { "gost 34.10-12-512 - ok", gost12_512, &gost12_512[0], 0, 0, 0, 1466612070, 1},
#endif
  { "rsa-512 - not ok (due to profile)", rsa_512, &rsa_512[0], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_MEDIUM),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL, 1576759855, 1},
  { "ed448 - ok", ed448, &ed448[0], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_ULTRA),
    0, NULL, 1584352960, 1},
  { "superseding - ok", superseding, superseding_ca, 0, 0, 0, 1590928011 },
  { "rsa-sha1 in trusted - ok",
    rsa_sha1_in_trusted, rsa_sha1_in_trusted_ca,
    GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_MEDIUM),
    0, NULL, 1620052390, 1},
  { "rsa-sha1 not in trusted - not ok",
    rsa_sha1_not_in_trusted, rsa_sha1_not_in_trusted_ca,
    GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_MEDIUM),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID, NULL, 1620118136, 1},
  { NULL, NULL, NULL, 0, 0}
};

#if defined __clang__ || __GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 5)
#  pragma GCC diagnostic pop
#endif

/* *INDENT-ON* */

#endif				/* GNUTLS_TESTS_TEST_CHAINS_H */
