/* errors.c -  Error numbers 
 *	Copyright (C) 2001, 2002, 2003 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include "gpgme.h"

static char *(*gpgme_gettext)( const char * msgid ) = NULL;
#define _(a) _gpgme_gettext( a )


void
gpgme_set_gettext_handler( char *(*gettext)( const char *id ) )
{
    gpgme_gettext = gettext;
}


const char*
_gpgme_gettext( const char * a )
{
    return gpgme_gettext? gpgme_gettext( a ) : a;
}


/**
 * gpgme_strerror:
 * @err:  Error code 
 * 
 * This function returns a textual representaion of the given
 * errocode. If this is an unknown value, a string with the value
 * is returned (which is hold in a static buffer).
 * 
 * Return value: String with the error description.
 **/
const char *
gpgme_strerror (gpgme_error_t err)
{
    const char *s;
    static char buf[128];
    
    switch (err) {
    case GPGME_EOF: s=_("End of file"); break;
    case GPGME_No_Error: s=_("No Error"); break;
    case GPGME_General_Error: s=_("General Error"); break;
    case GPGME_Out_Of_Core: s=_("Out Of Core"); break;
    case GPGME_Invalid_Value: s=_("Invalid Value"); break;
    case GPGME_Busy: s=_("Busy"); break;
    case GPGME_No_Request: s=_("No Request"); break;
    case GPGME_Exec_Error: s=_("Exec Error"); break;
    case GPGME_Too_Many_Procs: s=_("Too Many Procs"); break;
    case GPGME_Pipe_Error: s=_("Pipe Error"); break;
    case GPGME_No_Recipients: s=_("No Recipients"); break;
    case GPGME_Inv_Recipients: s=_("Invalid Recipients"); break;
    case GPGME_No_Data: s=_("No Data"); break;
    case GPGME_Conflict: s=_("Conflict"); break;
    case GPGME_Not_Implemented: s=_("Not Implemented"); break;
    case GPGME_Read_Error: s=_("Read Error"); break;
    case GPGME_Write_Error: s=_("Write Error"); break;
    case GPGME_Invalid_Type: s=_("Invalid Type"); break;
    case GPGME_Invalid_Mode: s=_("Invalid Mode"); break;
    case GPGME_File_Error: s=_("File Error"); break;
    case GPGME_Decryption_Failed: s=_("Decryption Failed"); break;
    case GPGME_Signing_Failed: s=_("Signing Failed"); break;
    case GPGME_Encryption_Failed: s=_("Encryption Failed"); break;
    case GPGME_No_Passphrase: s=_("No Passphrase"); break;  
    case GPGME_No_Seckey: s=_("No secret key available"); break;
    case GPGME_No_Pubkey: s=_("No public key available"); break;
    case GPGME_Canceled: s=_("Cancelled"); break;
    case GPGME_Invalid_Key: s=_("Invalid Key"); break;
    case GPGME_Invalid_Engine: s=_("Invalid Engine"); break;
    case GPGME_Bad_Signature: s=_("Bad Signature"); break;
    case GPGME_Bad_Passphrase: s=_("Bad Passphrase"); break;
    case GPGME_Clip_Open: s=_("Cannot open Clipboard"); break;
    case GPGME_Clip_Empty: s=_("Clipboard is empty"); break;
    case GPGME_Clip_Get: s=_("Cannot get Clipboard data"); break;
    case GPGME_Internal_GPG_Problem: s=_("Internal GPG Problem, no more information available.\n"
					"Please run GPG on the console to check this."); break;
    case GPGME_Cipher_IDEA: s=_("Protection algorithm 1 (IDEA) is not supported."); break;
    case GPGME_Conf_InvOption: s=_("Config file (gpg.conf) contains invalid option."); break;
    case GPGME_SC_Failure: s=_("Smartcard operation failure"); break;
    case GPGME_RecpError_NoReason: s = _("No specific reason given"); break;
    case GPGME_RecpError_NotFound: s = _("Not Found" ); break;
    case GPGME_RecpError_Ambigious:s = _("Ambigious specification"); break;
    case GPGME_RecpError_WrongKeyUsage:  s = _("Wrong key usage"); break;
    case GPGME_RecpError_Revoked:  s = _("Key revoked"); break;
    case GPGME_RecpError_Expired:  s = _("Key expired"); break;
    case GPGME_RecpError_NotSecret:s = _("Not a secret key"); break;
    case GPGME_RecpError_Untrusted:s = _("Key not trusted"); break;
    default:  sprintf( buf, "Unknown error ec=%d", err ); s=buf; break;
    }
    
    return s;
} /* gpgme_strerror */

