/* pipefile.c - High level interface for file encryption via pipes
 *	Copyright (C) 2001 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <string.h>

#include "util.h"
#include "context.h"
#include "ops.h"
#include "key.h"

#ifdef USE_GPGME_PIPEFILES
/*
 * File operations based on pipe streams.
 * Warning, this interface should be only used for small files. For real file
 * encryption it's a good idea to use the gpgme_file_xxx functions.
 */

gpgme_error_t
gpgme_op_encrypt_file( gpgme_recipients_t rset, const char *fname, 
					   const char *output )
{	
    gpgme_error_t err = 0;
    gpgme_ctx_t c;
    gpgme_data_t plain, ciph;
    
    err = gpgme_new( &c );
    if ( err )
        return err;
    gpgme_control( c, GPGME_CTRL_ARMOR, 1 );
    
    err = gpgme_data_new_from_file( &plain, fname, 1 );
    if ( err )
        goto leave;
    
    err = gpgme_data_new( &ciph );
    if ( err )
        goto leave;
    
    err = gpgme_op_encrypt( c, rset, plain, ciph );
    if ( err )
        goto leave;
    
    gpgme_data_release_and_set_file( ciph, output );
    
leave:
    if ( plain )
        gpgme_data_release( plain );
    gpgme_release( c );
    return err;
    
} /* gpgme_op_encrypt_file */

gpgme_error_t
gpgme_op_decrypt_file( gpgme_ctx_t c, const char *fname, 
					   const char *output )
{
    gpgme_error_t err = 0;	
    gpgme_data_t ciph, plain;
    
    err = gpgme_data_new_from_file( &ciph, fname, 1 );
    if (err)
        goto leave;
    
    err = gpgme_data_new( &plain );
    if (err) 
        goto leave;
    
    err = gpgme_op_decrypt( c, ciph, plain );
    if (err)
        goto leave;
    
    gpgme_data_release_and_set_file( plain, output );
    
leave:
    if ( ciph )
        gpgme_data_release( ciph );
    gpgme_release( c );
    return err;
} /* gpgme_op_decrypt_file */

gpgme_error_t
gpgme_op_verify_file( const char *fname, const char *detach_sig,
		     gpgme_sigstat_t *r_stat )
{
    gpgme_error_t err = 0;
    gpgme_ctx_t c;
    gpgme_data_t sig;
    gpgme_sigstat_t stat;
    
    err = gpgme_new( &c );
    if ( err )
        return err;
    
    err = gpgme_data_new_from_file( &sig, fname, 1 );
    if ( err )
        goto leave;
    
    err = gpgme_op_verify( c, sig, NULL );
    if ( err ) {
        *r_stat = 0;
        goto leave;
    }
    
    *r_stat = stat;
    
leave:
    if ( sig )
        gpgme_data_release( sig );
    gpgme_release( c );
    return err;
} /* gpgme_op_verify_file */

gpgme_error_t
gpgme_op_sign_file( gpgme_ctx_t c,gpgme_sigmode_t mode, const char *keyid, 
				   const char *fname, const char *output )
{
    gpgme_error_t err = 0;	
    gpgme_key_t key;
    gpgme_data_t plain, ciph;
    
    gpgme_control( c, GPGME_CTRL_ARMOR, 1 );
    
    err = gpgme_data_new_from_file( &plain, fname, 1 );
    if (err)
        goto leave;
    
    err = gpgme_data_new( &ciph );
    if (err)
        goto leave;
    
    err = _gpgme_key_new( &key );
    if (err) 
        goto leave;
    strcpy( key->keys.keyid, keyid );
    gpgme_signers_add( c, key );
    
    err = gpgme_op_sign( c, plain, ciph, mode );
    if (err)
        goto leave;
    
    gpgme_key_release( key );
    gpgme_data_release_and_set_file( ciph, output );
    
leave:
    if ( plain )
        gpgme_data_release( plain );
    gpgme_release( c );
    return err;
} /* gpgme_op_sign_file */

gpgme_error_t
gpgme_op_sign_encrypt_file( gpgme_ctx_t c, gpgme_recipients_t rset,
			   const char *keyid, const char *fname,
			   const char *output )
{
    gpgme_error_t err = 0;
    gpgme_data_t plain, ciph;
    
    gpgme_control( c, GPGME_CTRL_ARMOR, 1 );
    
    err = gpgme_data_new_from_file( &plain, fname, 1 );
    if (err)
        goto leave;
    
    err = gpgme_data_new( &ciph );
    if (err)
        goto leave;
    
    err = gpgme_op_sign_encrypt( c, rset, plain, ciph );
    if (err)
        goto leave;
    
    gpgme_data_release_and_set_file( ciph, output );
    
    
leave:
    if ( ciph )
        gpgme_data_release( plain );
    gpgme_release( c );
    return err;
} /* gpgme_op_sign_encrypt_file */

gpgme_error_t
gpgme_op_import_file(const char *fname)
{
    gpgme_error_t err = 0;
    gpgme_ctx_t c;
    gpgme_data_t keydata;
    
    err = gpgme_new( &c );
    if ( err )
        return err;
    
    err = gpgme_data_new_from_file( &keydata, fname, 1 );
    if ( err )
        goto leave;
    
    err = gpgme_op_import( c, keydata );	
    
leave:
    if ( keydata )
        gpgme_data_release( keydata );
    gpgme_release( c );
    return err;
} /* gpgme_op_import_file */

gpgme_error_t
gpgme_op_export_file( gpgme_recipients_t rset, const char *output )
{
    gpgme_error_t err = 0;
    gpgme_ctx_t c;
    gpgme_data_t keydata;
    
    err = gpgme_new( &c );
    if ( err )
        return err;
    gpgme_control( c, GPGME_CTRL_ARMOR, 1 );
    
    err = gpgme_data_new( &keydata );
    if ( err )
        goto leave;
    
    err = gpgme_op_export( c, rset, keydata );
    if ( err )
        goto leave;	
    
    gpgme_data_release_and_set_file( keydata, output );
    
leave:
    gpgme_release( c );
    return err;
} /* gpgme_op_export_file */
#endif
