/* wptCardPCSC.cpp - PC/SC card API interface
 *	Copyright (C) 2003 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <windows.h>
#include <winscard.h>

#include "gpgme.h"
#include "wptTypes.h"
#include "wptCard.h"


#define MAX_READERS 8

struct pcsc_reader_s {
    SCARD_READERSTATE_A reader_states[MAX_READERS];
    int nreaders;
    char * readers[MAX_READERS];
};
typedef struct pcsc_reader_s *pcsc_reader_t;


static LONG (WINAPI *pcsc_establish_context) (unsigned long scope,
                                const void *reserved1,
                                const void *reserved2,
                                unsigned long *r_context);
static LONG (WINAPI *pcsc_release_context) (unsigned long context);
static LONG (WINAPI *pcsc_list_readers) (unsigned long context, 
					 const char *groups,
					 char *readers, 
					 unsigned long *readerslen);
static LONG (WINAPI *pcsc_connect) (unsigned long context,
                      const char *reader,
                      unsigned long share_mode,
                      unsigned long preferred_protocols,
                      unsigned long *r_card,
                      unsigned long *r_active_protocol);
static LONG (WINAPI *pcsc_disconnect) (unsigned long card, 
				       unsigned long disposition);
static LONG (WINAPI *pcsc_status) (unsigned long card,
                     char *reader, unsigned long *readerlen,
                     unsigned long *r_state, unsigned long *r_protocol,
                     unsigned char *atr, unsigned long *atrlen);
static LONG (WINAPI *pcsc_get_status_change)(unsigned long ctx, 
					     unsigned long timeout,
					     SCARD_READERSTATE * readerstate,
					     unsigned long creaders);

static int pcsc_init = 0;


#define load_one_fnc(ptr, hlib, name) do {			    \
    a = ptr = (void *)GetProcAddress (hlib, name);		    \
    if (!a) {							    \
	char msg[256];						    \
	_snprintf (msg, sizeof msg-1, "Could not load '%s'", name); \
	MessageBox (NULL, msg, "PC/SC Driver", MB_ICONERROR|MB_OK); \
	goto fail;						    \
    } \
} while (0)


int
pcsc_loadlib (int scard_support)
{
    HMODULE hlib;
    FARPROC a;    

    if (!scard_support || pcsc_init)
	return 0;
    hlib = LoadLibrary ("WINSCARD");
    if (!hlib)
	goto fail;
    load_one_fnc (pcsc_establish_context, hlib, "SCardEstablishContext");
    load_one_fnc (pcsc_release_context, hlib, "SCardReleaseContext");
    load_one_fnc (pcsc_list_readers, hlib, "SCardListReadersA");
    load_one_fnc (pcsc_connect, hlib, "SCardConnectA");
    load_one_fnc (pcsc_disconnect, hlib, "SCardDisconnect");
    load_one_fnc (pcsc_status, hlib, "SCardStatusA");
    load_one_fnc (pcsc_get_status_change, hlib, "SCardGetStatusChangeA");
    goto success;

fail:
    FreeLibrary (hlib);
    return -1;
success:
    pcsc_init = 1;
    FreeLibrary (hlib);
    return 0;
}


void
pcsc_free_readers (pcsc_reader_t rd)
{
    int i;

    if (!rd)
	return;

    for (i=0; i < rd->nreaders; i++)
	safe_free (rd->readers[i]);
    safe_free (rd);
}


const char *
pcsc_get_reader (pcsc_reader_t rd, int idx, int * ret_nrd)
{
    if (!rd)
	return NULL;
    if (idx == -1 && ret_nrd) {
	*ret_nrd = rd->nreaders;
	return NULL;
    }
    if (idx >= rd->nreaders)
	idx=0;
    return rd->readers[idx];
}


int
pcsc_scan_readers (pcsc_reader_t * ret_rd)
{
    pcsc_reader_t rd;
    SCARDCONTEXT hctx;
    SCARD_READERSTATE_A rdstat;
    DWORD readers;
    LPSTR rdrstr;
    char * p;
    int i;
    int rc, curr_rd=0;

    *ret_rd = NULL;
    if (!pcsc_init)
	return -1;

    rc = pcsc_establish_context (SCARD_SCOPE_SYSTEM, NULL, NULL, &hctx);
    if (rc != SCARD_S_SUCCESS)
	return -1;

    rc = pcsc_list_readers (hctx, NULL, NULL, &readers);
    if (rc != SCARD_S_SUCCESS)
	return -1;

    rdrstr = malloc (sizeof(char)*readers);
    if (!rdrstr)
	BUG (0);

    rc = pcsc_list_readers (hctx, NULL, rdrstr, &readers);
    if (rc != SCARD_S_SUCCESS) {
	safe_free (rdrstr);
	return -1;
    }

    rd = calloc (1, sizeof *rd);
    if (!rd)
	BUG (0);
	
    p = rdrstr;
    while ((*p != '\0') && (rd->nreaders < MAX_READERS)) {	
	rd->readers[rd->nreaders] = strdup (p);
	p += strlen (p)+1;
	rd->nreaders++;	
    }

    if (!rd->nreaders) {
	safe_free (rdrstr);
	safe_free (rd);
	return 0;
    }

    /* set the initial states */
    for (i=0; i<rd->nreaders; i++) {
	rd->reader_states[i].szReader = rd->readers[i];
	rd->reader_states[i].dwCurrentState = SCARD_STATE_EMPTY;
    }

    while (1) {
	rdstat = rd->reader_states[curr_rd];		
	rc = pcsc_get_status_change (hctx, 0, &rdstat, 1);
	if (rc == SCARD_E_TIMEOUT)
	    continue;
	if (rc != SCARD_S_SUCCESS)
	    ;

	/* next reader */
	curr_rd++;
	if (curr_rd >= rd->nreaders)
	    curr_rd = 0;
    }
    
    safe_free (rdrstr);
    *ret_rd = rd;
    rc = pcsc_release_context (hctx);
    if (rc != SCARD_S_SUCCESS)
	return -1;
    return 0;
}


int
pcsc_get_card_status (void)
{
    pcsc_reader_t rd;
    SCARD_READERSTATE_A rdstat;
    int rc, stat=0;

    rc = pcsc_scan_readers (&rd);
    if (rc)
	return -1;
    rdstat = rd->reader_states[0];
    if (rdstat.dwCurrentState & SCARD_STATE_UNAVAILABLE)
	stat |= CARD_STATE_UNAVAIL;
    if (rdstat.dwCurrentState & SCARD_STATE_EMPTY)
	stat |= CARD_STATE_EMPTY;
    if (rdstat.dwCurrentState & SCARD_STATE_INUSE)
	stat |= CARD_STATE_INUSE;
    if (rdstat.dwCurrentState & SCARD_STATE_EXCLUSIVE)
	stat |= CARD_STATE_EXCLUSI;
    return stat;
}
