/* wptClipEncryptDlg.cpp - Clipboard encrypt dialog
 *	Copyright (C) 2000-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptErrors.h"
#include "wptCommonCtl.h"
#include "wptGPG.h"
#include "wptKeylist.h"
#include "wptNLS.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptRegistry.h"
#include "wptDlgs.h"


BOOL CALLBACK 
clip_encrypt_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static listview_ctrl_t lv = NULL;
    gpgme_keycache_t kc;
    gpgme_recipients_t rset;
    gpgme_ctx_t ctx;
    gpgme_error_t err;
    refresh_cache_s rcs = {0};
    int force_trust = 0, opt = 0;
    int kmode = reg_prefs.keylist_mode? KEYLIST_ENCRYPT_MIN : KEYLIST_ENCRYPT;
    
    switch( msg ) {
    case WM_INITDIALOG:
        SetWindowText( dlg, _("Encryption") );
        	
	if( keycache_get_reload() ) {
	    rcs.kr_reload = rcs.kr_update = 1;
	    rcs.tr_update = 0;        
	    DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_KEYCACHE, GetDesktopWindow(),
			    keycache_dlg_proc, (LPARAM)&rcs );
	}
        kc = keycache_get_ctx( KEYCACHE_PUB );
        if( !kc )
            BUG( NULL );
        lv = keylist_load( GetDlgItem( dlg, IDC_ENCRYPT_KEYLIST ), kc, NULL,
			    kmode, GPGME_ATTR_USERID );
	center_window( dlg );
        SetForegroundWindow( dlg );
        set_active_window( dlg );
        return TRUE;
        
    case WM_DESTROY:
        reset_active_window( );
        if( lv ) {
            keylist_delete( lv );
            lv = NULL;
        }
        return FALSE;
        
    case WM_NOTIFY:
        NMHDR *notify;
        
        notify = (NMHDR *)lparam;
        if( notify && notify->code == NM_DBLCLK && notify->idFrom == IDC_ENCRYPT_KEYLIST )
            PostMessage( dlg, WM_COMMAND, MAKEWPARAM( IDOK, 0 ), NULL );
	if( notify && notify->code == LVN_COLUMNCLICK && notify->idFrom == IDC_ENCRYPT_KEYLIST ) {
	    NMLISTVIEW *p = (LPNMLISTVIEW) lparam;
	    int sortby = 0;

	    switch( p->iSubItem ) {
	    case  0: sortby = GPGME_ATTR_USERID; break;
	    case  1: sortby = GPGME_ATTR_KEYID; break;
	    case  2: sortby = GPGME_ATTR_LEN; break;
	    case  4: sortby = GPGME_ATTR_VALIDITY; break;
	    default: sortby = GPGME_ATTR_USERID; break;
	    }
	    keylist_sort( lv, sortby );
	}
        return TRUE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDOK:
            rset = keylist_get_recipients( lv, &force_trust, NULL );
            if( !gpgme_recipients_count( rset ) ) {
                msg_box( dlg, _("You must select at least one key."), _("Encryption"), MB_ERR );
                gpgme_recipients_release( rset );
                return FALSE;
            }
            if( force_trust )
                opt |= GPGME_CTRL_FORCETRUST;
            if( reg_prefs.use_tmpfiles )
                opt |= GPGME_CTRL_TMPFILES;
            err = gpgme_op_clip_encrypt( rset, opt, &ctx );
            if( err && err != GPGME_Inv_Recipients ) {
		if( err == GPGME_Internal_GPG_Problem )
		    gnupg_display_error ();
		else
		    msg_box( dlg, gpgme_strerror( err ), _("Encryption"), MB_ERR );
		gpgme_release( ctx );
                gpgme_recipients_release( rset );
                return FALSE;
            }
	    else if( err == GPGME_Inv_Recipients ) {
		int ncount = gpgme_recperr_count_items( ctx );
		gpgme_error_t code = GPGME_No_Error;

		while( ncount-- ) {
		    const char *s = gpgme_recperr_get( ctx, ncount, &code );
		    if( s )
			log_box( _("Encryption"), MB_ERR, 
				 _("Unusable recipient \"%s\": %s"), s,
				 gpgme_strerror( code ) );
		}
	    }
            else
		show_msg( dlg, 1500, _("GnuPG Status: Finished") );
            gpgme_recipients_release( rset );
	    gpgme_release( ctx );
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDCANCEL:
            EndDialog( dlg, FALSE );
            return FALSE;

	case IDC_ENCRYPT_FNDCMD:
	{
	    char tmpbuf[64];
	    int n;

	    n = GetDlgItemText (dlg, IDC_ENCRYPT_FIND, tmpbuf, DIM (tmpbuf)-1);
	    if (!n)
		break;
	    n = listview_find (lv, tmpbuf);
	    if (n != -1)
		listview_select_one (lv, n);
	    else
		log_box (_("Encryption"), MB_ERR, _("No recipient found with '%s'"), tmpbuf);
	    break;
	}
        }
        break;
    }
    
    return FALSE;
} /* clip_encrypt_dlg_proc */
