/* wptGPGME.cpp - WinPT GPGME interface
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#include <sys/types.h>
#include <windows.h>

#include "../resource.h"
#include "wptNLS.h"
#include "wptGPG.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptRegistry.h"
#include "wptDlgs.h"

#include "openpgp.h"

BOOL CALLBACK  keycache_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam);
void progress_cleanup (progress_filter_s * pfx);

static gpgme_keycache_t pub = NULL;
static gpgme_keycache_t sec = NULL;
static unsigned int reload = 0;


void
keycache_reload (HWND dlg) 
{	
    refresh_cache_s rcs = {0};

    rcs.kr_reload = rcs.kr_update = 1;
    rcs.tr_update = 1;
    DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_KEYCACHE, dlg,					
		    keycache_dlg_proc, (LPARAM)&rcs );
} /* keycache_reload */

void
keycache_release (void)
{
    int n = gpgme_keycache_count (pub);
    char tmpbuf[64];

    sprintf (tmpbuf, "%d", n);
    set_reg_key (HKEY_CURRENT_USER, "Software\\WinPT", "nKeys", tmpbuf);

    if (pub) {
        gpgme_keycache_release( pub ); 
        pub = NULL;
    }
    if (sec) {
        gpgme_keycache_release( sec );
        sec = NULL;
    }
} /* keycache_release */


gpgme_error_t
keycache_init( const char *pubring, const char * secring )
{
    struct progress_filter_s pfx;
    gpgme_error_t err;
    int val = 0;
    char * p;

    if (reload) {
        keycache_release ();
        reload = 0;
    }
    p = get_reg_entry (HKEY_CURRENT_USER, "Software\\WinPT", "nKeys");
    if (p && *p != ' ')
    {
	val = atoi (p);
	free_if_alloc (p);
	memset (&pfx, 0, sizeof pfx);
    }

    err = gpgme_keycache_new (&pub);
    if (err)
	return err;
    if (val != 0)
	gpgme_keycache_set_cb (pub, progress_callback, &pfx, val);
    err = gpgme_keycache_new (&sec);
    if (!err)
	err = gpgme_keycache_init (pub, NULL, 0);
    if (!err)
	err = gpgme_keycache_init( sec, NULL, 1 );
    if( !err && pubring && *pubring )
	err = gpgme_keycache_prepare( pub, pubring, NULL );
    if( !err && secring && * secring )
	err = gpgme_keycache_prepare( sec, NULL, secring );
    if (!err)
	gpgme_keycache_sync (pub, sec);
    if (val != 0)
	progress_cleanup (&pfx);
    return err;
} /* keycache_init */


void
keycache_set_reload( int yes )
{
    reload = yes;
} /* keycache_set_reload */


int
keycache_get_reload( void )
{
    return reload;
} /* keycache_get_reload */


gpgme_keycache_t
keycache_get_ctx( int is_pub )
{
    return is_pub? pub : sec;
} /* keycache_get_ctx */


static int
get_key( const char *keyid, gpgme_key_t *r_key, int secret )
{
    gpgme_keycache_t cache;
    gpgme_error_t err;
    int mode = secret? KEYCACHE_PRV : KEYCACHE_PUB;

    if( !keyid )
	return WPTERR_GENERAL;
    if( r_key )
	*r_key = NULL;
    cache = keycache_get_ctx( mode );
    if( !cache )
	BUG( NULL );
    err = gpgme_keycache_find_key( cache, keyid, 0, r_key );
    if( err )
	return WPTERR_GENERAL;
    return 0;
} /* get_key */


static int
get_key2( const char * keyid, gpgme_key_t * r_key, int secret )
{
    gpg_iobuf_t inp;
    char * p;
    int rc;

    p = get_gnupg_keyring (1, !NO_STRICT);
    if( !p )
	BUG( NULL );

    inp = gpg_iobuf_open( p );
    if( !inp ) {
	const char *s = winpt_strerror( WPTERR_FILE_OPEN );
	log_box( _("WinPT Error"), 0, "%s: %s", p, s );
	free_if_alloc( p );
	return NULL;
    } 
    gpg_iobuf_ioctl( inp, 3, 1, NULL ); /* disable cache */
    rc = gpgme_getkey_bykeyid( inp, keyid, r_key );

    gpg_iobuf_close( inp );
    free_if_alloc( p );
    return rc;
} /* get_key2 */


int
get_pubkey( const char *keyid, gpgme_key_t *ret_key )
{
    int rc = 0;

    if( pub && sec )
	rc = get_key( keyid, ret_key, 0 );
    else
	rc = get_key2( keyid, ret_key, 0 );
    return rc;
} /* get_pubkey */


int
get_seckey( const char *keyid, gpgme_key_t *ret_skey )
{
    int rc = 0;

    if( pub && sec )
	rc = get_key( keyid, ret_skey, 1 );
    else
	rc = get_key2( keyid, ret_skey, 1 );
    return rc;
} /* get_seckey */


int
count_insecure_elgkeys (void)
{
    gpgme_keycache_t pc;
    gpgme_key_t key;
    int n=0;

    pc = keycache_get_ctx (1);
    if (!pc)
	BUG (0);
    while (!gpgme_keycache_next_key (pc, 0, &key)) {
	if (gpgme_key_get_ulong_attr (key, GPGME_ATTR_ALGO, NULL, 0)
	    == GPGME_PK_ELG_ES)
	    n++;
    }
    gpgme_keycache_rewind (pc);
    return n;
}
