/* wptKeyEditDlgs.cpp - GPG key edit dialogs
 *	Copyright (C) 2002-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>
#include "../resource.h"

#include "wptTypes.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptDlgs.h"
#include "wptNLS.h"
#include "wptUTF8.h"
#include "wptErrors.h"
#include "wptKeylist.h"
#include "wptKeyManager.h"
#include "wptRegistry.h"


enum keyedit_commands {    
    CMD_ADDKEY = 0,
    CMD_ADDUID,
    CMD_ADDPHOTO,
    CMD_ADDREVOKER,
    /*CMD_FPR,*/
    CMD_DELUID,
    CMD_DELKEY,
    CMD_DELPHOTO,
    /*CMD_DELSIG,*/
    CMD_EXPIRE,
    /*CMD_PREF,
    CMD_SHOWPREF,
    CMD_SETPREF,*/
    CMD_UPDPREF,
    CMD_PASSWD,
    CMD_PRIMARY,
    CMD_TRUST,
    /*CMD_REVSIG,*/
    CMD_REVUID,
    CMD_REVKEY,
    CMD_DISABLE,
    CMD_ENABLE,
    /*CMD_SHOWPHOTO,*/
};


struct keyedit_callback_s {
    gpgme_editkey_t ek;
    const char * pass;
    listview_ctrl_t lv;
    void * opaque;
};
typedef struct keyedit_callback_s KEYEDIT_CB;

struct keygen_callback_s {
    int bits;
    int algo;
    u32 expire;
    char * fpr;
};
typedef struct keygen_callback_s KEYGEN_CB;


static subclass_s keyedit_subkey_proc;
static subclass_s keyedit_uid_proc;

int keygen_check_date( SYSTEMTIME * st );
void get_userid_preflist (char ** r_prefs, int * r_flags);

static void
do_init_keylist (HWND dlg, winpt_key_t k)
{
    gpgme_keycache_t pub;
    gpgme_key_t key;
    const char * s, * kid;
    char * u;
    int i, n;

    pub = keycache_get_ctx (1);
    if (!pub)
	BUG (0);

    while( !gpgme_keycache_next_key( pub, 0, &key ) ) {
	s = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
	kid = gpgme_key_get_string_attr (key, GPGME_ATTR_KEYID, NULL, 0);
	if (!s || !strcmp (kid+8, k->keyid+2))
	    continue;
	u = utf8_to_wincp (s, strlen (s));
	SendDlgItemMessage (dlg, IDC_ADDREV_KEYLIST, CB_ADDSTRING, 
			    0, (WPARAM)(char *)u);
	free( u );
    }
    gpgme_keycache_rewind( pub );
    n = SendDlgItemMessage( dlg, IDC_ADDREV_KEYLIST, CB_GETCOUNT, 0, 0 );
    for( i = 0; i < n; i++ ) {
	gpgme_keycache_next_key( pub, 0, &key );
	SendDlgItemMessage( dlg, IDC_ADDREV_KEYLIST, CB_SETITEMDATA, 
			    (WPARAM)(int)i, (LPARAM)key );
    }
    SendDlgItemMessage( dlg, IDC_ADDREV_KEYLIST, CB_SETCURSEL, 0, 0 );
} /* do_init_keylist */


static void
do_add_new_userid( listview_ctrl_t lv, const char * name, const char *email,
		   const char * comment )
{
    char * p;
    size_t n;
    
    n = strlen( name ) + strlen( email ) + 16;
    if( comment )
	n += strlen( comment );
    p = new char[n+1];
    if( !p )
	BUG( NULL );
    if( comment )
	sprintf( p, "%s (%s) %s", name, comment, email );
    else
	sprintf( p, "%s %s", name, email );
    listview_add_item( lv, "" );
    listview_add_sub_item( lv, 0, 0, _("Ultimate" ) );
    listview_add_sub_item( lv, 0, 1, p );
    listview_add_sub_item( lv, 0, 2, _("OK") );
    free_if_alloc( p );
} /* do_add_new_userid */


static void
do_add_new_subkey (listview_ctrl_t lv, KEYGEN_CB * keygen, unsigned int flags)
{
    char info[128], keyid[32];
    const char * expdate, * s;
    
    expdate = keygen->expire? get_key_expire_date (keygen->expire) : _("Never");
    _snprintf( info, sizeof info-1, "%d-bit %s",
	keygen->bits, gpgme_key_expand_attr( GPGME_ATTR_ALGO, keygen->algo ) );
    _snprintf( keyid, sizeof keyid-1, "0x%s", keygen->fpr+32 );
    listview_add_item( lv, "" );
    listview_add_sub_item( lv, 0, 0, info );
    listview_add_sub_item( lv, 0, 1, keyid );
    listview_add_sub_item( lv, 0, 2, get_key_created( time(NULL) ) );
    listview_add_sub_item( lv, 0, 3, expdate );
    if( flags & KM_FLAG_REVOKED ) s = _("Revoked");	
    else if( flags & KM_FLAG_EXPIRED ) s = _("Expired");
    else s = _("OK");
    listview_add_sub_item( lv, 0, 4, s );
} /* do_add_new_subkey */


static int
do_find_userid (const char * keyid, const char * name)
{
    gpgme_uidinfo_t inf;
    gpgme_ctx_t ctx;
    gpgme_error_t err;
    int nitems = 0, pos = -1;
    const char * s;
    char * utf8_name=NULL;

    err = gpgme_new (&ctx);
    if (err)
	BUG (0);
    err = gpgme_op_editkey_get_info (ctx, keyid, &inf);
    if (err) 
    {
	log_box (_("user ID"), MB_ERR, _("Could not get key information for: \"%s\""), name);
	gpgme_release (ctx);
	return -1;
    }
    gpgme_release (ctx);
    nitems = gpgme_editkey_count_items (inf);
    while (nitems--) 
    {
	s = gpgme_editkey_get_string_attr (inf, GPGME_ATTR_NAME, nitems);
	if (!s)
	    continue;
	utf8_name = utf8_to_wincp (s, strlen (s));
	if (!strcmp (utf8_name, name))
	{
	    pos = gpgme_editkey_get_ulong_attr (inf, GPGME_ATTR_LEVEL, nitems);
	    free (utf8_name);
	    break;
	}
	free (utf8_name); utf8_name=NULL;
    }

    gpgme_uid_info_release (inf);
    return pos;
} /* do_find_userid */


BOOL CALLBACK
keyedit_addphoto_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static winpt_key_t k;    
    gpgme_editkey_t ek;
    gpgme_error_t ec;
    gpgme_ctx_t ctx;
    const char * s;    
    char pwd[128], file[128];
    int id;

    switch( msg ) {
    case WM_INITDIALOG:
	k = (winpt_key_t)lparam;
	if( !k )
	    BUG( NULL );
	SetDlgItemText (dlg, IDC_ADDPHOTO_INF, _("Remember that the image is stored within your public key.  If you use a very large picture, your key will become very large as well! Keeping the image close to 240x288 is a good size to use."));
	SetDlgItemText (dlg, IDC_ADDPHOTO_FILEINF, _("Pick an image to use for your photo ID.\nThe image must be a JPEG file."));
	SetDlgItemText (dlg, IDC_ADDPHOTO_PWDINF, _("Passphrase"));
	SetForegroundWindow( dlg );
	break;

    case WM_DESTROY:
	break;

    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE )
            EndDialog( dlg, TRUE );
	break;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {

	case IDC_ADDPHOTO_SELFILE:
	    s = get_filename_dlg( dlg, FILE_OPEN, _("Select Image File"), _("JPEG Files (*.jpg, *.jpeg)\0*.jpg;*.jpeg\0\0"), NULL );
	    if( s && *s )
		SetDlgItemText( dlg, IDC_ADDPHOTO_FILE, s );
	    break;

	case IDOK:
	    if( !GetDlgItemText( dlg, IDC_ADDPHOTO_FILE, file, sizeof file-1 ) ){
		msg_box( dlg, _("Please enter a file name."), _("Add Photo"), MB_ERR );
		return FALSE;
	    }
	    if( get_file_size( file ) == 0 || get_file_size( file ) > 6144 ) {
		id = msg_box( dlg, _("The JPEG is really large.\n"
				     "Are you sure you want to use it?"), 
				     _("Add Photo"), MB_YESNO|MB_INFO );
		if( id == IDNO )
		    return TRUE;
	    }
	    if( k->is_protected ) {
		if( !GetDlgItemText( dlg, IDC_ADDPHOTO_PASS, pwd, sizeof pwd-1 ) ) {
		    msg_box( dlg, _("Please enter a passphrase."), _("Add Photo"), MB_ERR );
		    return FALSE;
		}
	    }
	    ec = gpgme_editkey_new( &ek );
	    if( !ec )
		ec = gpgme_new( &ctx );
	    if( ec )
		BUG( NULL );
	    gpgme_enable_logging( ctx );
	    gpgme_editkey_addphoto_set( ek, file, k->is_protected? pwd : NULL );
	    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_ADDPHOTO );
	    ec = gpgme_op_editkey( ctx, k->keyid );
	    gpgme_editkey_release( ek );
	    if( ec ) {
		gpgme_show_error( dlg, ec, ctx, _("Add Photo"), MB_ERR );
		gpgme_release( ctx );
		return FALSE;
	    }
	    else {
		keycache_set_reload( 1 );
		msg_box( dlg, _("Photo successfully added."), _("GnuPG Status"), MB_OK );
	    }
	    gpgme_release( ctx );
	    EndDialog( dlg, TRUE );
	    break;

	case IDCANCEL:
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }
    return FALSE;
} /* keyedit_addphoto_dlg_proc */


BOOL CALLBACK
keyedit_addrevoker_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static winpt_key_t k;
    static gpgme_key_t seckey;    
    gpgme_editkey_t ek;
    gpgme_ctx_t ctx;
    gpgme_error_t ec;    
    char uid[128], pwd[128];
    

    switch( msg ) {
    case WM_INITDIALOG:
	k = (winpt_key_t)lparam;
	if( !k )
	    BUG( NULL );
	if( get_seckey( k->keyid, &seckey ) )	
	    BUG( NULL );
	if( !k->is_protected )
	    EnableWindow( GetDlgItem( dlg, IDC_ADDREV_PASS ), FALSE );
	do_init_keylist( dlg, k );
	SetDlgItemText (dlg, IDC_ADDREV_INF, _("Appointing a key as designated revoker cannot be undone."));
	SetDlgItemText (dlg, IDC_ADDREV_KEYINF, _("Public key"));
	SetDlgItemText (dlg, IDC_ADDREV_PWDINF, _("Passphrase"));
	SetForegroundWindow( dlg );
	break;

    case WM_DESTROY:	
	break;

    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE )
            EndDialog( dlg, TRUE );
	break;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDOK:	    
	    if( !GetDlgItemText( dlg, IDC_ADDREV_KEYLIST, uid, sizeof uid-1 ) ) {
		msg_box( dlg, _("Please select a user ID."), _("Add Revoker"), MB_ERR );
		return FALSE;
	    }
	
	    if( k->is_protected ) {
		if( !GetDlgItemText( dlg, IDC_ADDREV_PASS, pwd, sizeof pwd-1 ) ) {
		    msg_box( dlg, _("Please enter the passphrase."), _("Add Revoker"), MB_ERR );
		    return FALSE;
		}
	    }
	    ec = gpgme_editkey_new( &ek );
	    if( !ec )
		ec = gpgme_new( &ctx );
	    if( ec )
		BUG( NULL );
	    gpgme_enable_logging( ctx );
	    gpgme_editkey_addrev_set( ek, uid, k->is_protected? pwd : NULL );
	    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_ADDREV );
	    ec = gpgme_op_editkey( ctx, k->keyid );
	    gpgme_editkey_release( ek );
	    memset( pwd, 0, sizeof pwd );
	    if( ec ) {
		gpgme_show_error( dlg, ec, ctx, _("Add Revoker"), MB_ERR );
		gpgme_release( ctx );
		return FALSE;
	    }
	    else {
		msg_box( dlg, _("Revoker successfully addded."), _("GnuPG Status"), MB_OK );
		keycache_set_reload( 1 );
	    }
	    gpgme_release( ctx );
	    EndDialog( dlg, TRUE );
	    break;

	case IDCANCEL:
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }
    return FALSE;
} /* keyedit_addrevoker_dlg_proc */


BOOL CALLBACK
keyedit_adduid_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static KEYEDIT_CB *ctx;
    char *utf8_name = NULL;
    char name[128], email[128], comment[128];
    int rc;
    
    switch ( msg ) {
    case WM_INITDIALOG:
        ctx = (KEYEDIT_CB *)lparam;
        if( !ctx )
            dlg_fatal_error(dlg, "Could not get dialog param!");
#ifndef LANG_DE
        SetWindowText( dlg, _("Add new User ID") );
        SetDlgItemText( dlg, IDC_ADDUID_INFNAME, _("&Name") );
        SetDlgItemText( dlg, IDC_ADDUID_INFEMAIL, _("&Email") );
        SetDlgItemText( dlg, IDC_ADDUID_INFCOMMENT, _("&Comment") );
#endif
        SetForegroundWindow( dlg );
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE ) {
            gpgme_editkey_make_invalid( ctx->ek );
            EndDialog(dlg, TRUE);
        }
        return FALSE;
        
    case WM_COMMAND:
        switch ( LOWORD( wparam ) )  {
        case IDOK:
            rc = GetDlgItemText( dlg, IDC_ADDUID_NAME, name, sizeof name-1 );
            if( !rc || rc < 5 ) {
                msg_box( dlg, _("Please enter a name (min. 5 chars.)"), _("UserID"), MB_ERR );
                return FALSE;
            }
	    if( strchr( name, '@' ) ) {
		msg_box( dlg, _("Please enter the email address in the email field and not in the name field"), _("UserID"), MB_INFO );
		return FALSE;
	    }
	                           
            if( !GetDlgItemText( dlg, IDC_ADDUID_EMAIL, email, sizeof email -1 ) ) {
                msg_box( dlg, _("Please enter an email address."), _("UserID"), MB_ERR );
                return FALSE;
            }
	    if( !strchr( email, '@' ) ) {
		msg_box( dlg, _("Invalid email address."), _("UserID"), MB_ERR );
		return FALSE;
	    }
            
            rc = GetDlgItemText( dlg, IDC_ADDUID_COMMENT, comment, sizeof comment -1 );

	    /* xxx: something is wrong with the encoding :-( */
	    utf8_name = wincp_to_utf8 (name, strlen (name));

	    gpgme_editkey_adduid_set( ctx->ek, utf8_name? utf8_name : name, email, 
				      rc? comment: NULL, ctx->pass );
	    free( utf8_name );
	    if( ctx->lv )
		do_add_new_userid( ctx->lv, name, email, rc?comment : NULL );
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDCANCEL:
            gpgme_editkey_make_invalid( ctx->ek );
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* keyedit_adduid_dlg_proc */


BOOL CALLBACK
keyedit_addsubkey_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static KEYEDIT_CB * ctx;
    static KEYGEN_CB * keygen;
    gpgme_error_t rc;
    int index, size, valid;
    HWND lb;
    
    switch ( msg ) {
    case WM_INITDIALOG:
        ctx = (KEYEDIT_CB *)lparam;
        if( !ctx )
            dlg_fatal_error( dlg, "Could not get dialog param!" );
	keygen = (KEYGEN_CB *)ctx->opaque;
#ifndef LANG_DE
        SetWindowText( dlg, _("Add new Subkey") );
        SetDlgItemText( dlg, IDC_ADDSUBKEY_INFALGO, _("Key type") );
        SetDlgItemText( dlg, IDC_ADDSUBKEY_INFSIZE, _("Size") );
        SetDlgItemText( dlg, IDC_ADDSUBKEY_INFVALID,
                        _("Valid for 'n' days. 0 means forever") );
#endif
        lb = GetDlgItem( dlg, IDC_ADDSUBKEY_ALGO );
        listbox_add_string( lb, "DSA (sign only)");
        listbox_add_string( lb, "ElGamal (encrypt only)" );
        listbox_add_string( lb, "RSA (sign only)");
        listbox_add_string( lb, "RSA (encrypt only)" );
	SetDlgItemInt( dlg, IDC_ADDSUBKEY_VALID, 0, FALSE );
	SetDlgItemInt( dlg, IDC_ADDSUBKEY_SIZE, 2048, FALSE );
        SetForegroundWindow( dlg );
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE ) {
            gpgme_editkey_make_invalid( ctx->ek );
            EndDialog( dlg, TRUE );
        }
        return FALSE;
        
    case WM_COMMAND:
        switch ( LOWORD(wparam) ) {        
        case IDOK:
            lb = GetDlgItem( dlg, IDC_ADDSUBKEY_ALGO );
	    switch (listbox_get_cursel (lb)) {
	    case 0: index = 2; break;
	    case 1: index = 4; break;
	    case 2: index = 5; break;
	    case 3: index = 6; break;
	    default: 
                msg_box( dlg, _("Please select one entry."), _("Add Subkey"), MB_ERR );
                return FALSE;
            }
	    if (gpgver[0] == 1 && gpgver[1] == 2) { /* GPG 1.2.x kludge */
		if (listbox_get_cursel (lb) > 1)
		    index++;
	    }
            size = GetDlgItemInt( dlg, IDC_ADDSUBKEY_SIZE, NULL, TRUE );
            if( !size ) {
                msg_box( dlg, _("Please enter the keysize."), _("Add Subkey"), MB_ERR );
                return FALSE;
            }
            else if( index == 2 && size != 1024 ) {
                msg_box( dlg,_("DSS uses a fixed keysize of 1024. Size changed."), _("Add Subkey"), MB_INFO );
                size = 1024;
            }
            else if( size > 4096 ) {
                msg_box( dlg, _("Chosen size should be between 1024 and 4096. Size changed."), _("Add Subkey"), MB_ERR );
                size = 4096;
            }
            else if( size < 1024 ) {
                msg_box( dlg, _("Keys with a size of less then 1024 are considered insecure.\n"
                                "Size changed to 1024!"), _("Add Subkey"), MB_INFO );
                size = 1024;
            }
            valid = GetDlgItemInt( dlg, IDC_ADDSUBKEY_VALID, NULL, TRUE );
            if( valid < 0 ) {
                msg_box( dlg, _("Please enter the days the key is valid."), _("Add Subkey"), MB_ERR );
                return FALSE;
            }
            rc = gpgme_editkey_addkey_set (ctx->ek, ctx->pass, index, size, valid);
	    if (rc) {
		msg_box (dlg, gpgme_strerror (rc), _("Add Subkey"), MB_ERR);
		return FALSE;
	    }
	    keygen->bits = size;
	    switch (index) {
	    case 2: keygen->algo = GPGME_PK_DSA; break;
	    case 4: keygen->algo = GPGME_PK_ELG_E; break;
	    case 5: keygen->algo = GPGME_PK_RSA_S; break;
	    case 6: keygen->algo = GPGME_PK_RSA_E; break;
	    }
	    if (valid)
		keygen->expire = time (NULL) + valid*24*60*60;
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDCANCEL:
            gpgme_editkey_make_invalid( ctx ->ek );
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* keyedit_addsubkey_dlg_proc */


BOOL
keyedit_add_userid (winpt_key_t k, HWND dlg, listview_ctrl_t lv)
{
    gpgme_error_t ec;
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    KEYEDIT_CB cb;
    char * pass = NULL;
    int cancel = 0;

    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Add user ID"), MB_ERR );
        return FALSE;
    }

    if (k->is_protected) {
	pass = request_passphrase( _("Key Edit"), 1, &cancel );
	if( cancel )	
	    return FALSE;
    }

    ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( dlg );
	   
    memset( &cb, 0, sizeof cb );
    cb.ek = ek;
    cb.pass = k->is_protected? pass : NULL;
    cb.lv = lv;
    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYEDIT_ADDUID,
		      dlg, keyedit_adduid_dlg_proc,         
		     (LPARAM)&cb, _("Add user ID"),
		    IDS_WINPT_KEYEDIT_ADDUID );
    if( !gpgme_editkey_is_valid( ek ) ) {
	free_if_alloc( pass );
	return FALSE;
    }
            
    ec = gpgme_new( &ctx );
    if( ec )
	BUG( dlg );
    gpgme_enable_logging( ctx );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_ADDUID );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Add user ID"), MB_ERR );
    else {
	msg_box(dlg, _("User ID successfully added"), _("GnuPG Status"), MB_OK );
	keycache_set_reload( 1 );
    }
    gpgme_editkey_release( ek );
    gpgme_release( ctx );
    free_if_alloc( pass );
    return TRUE;
} /* keyedit_add_userid */


BOOL
keyedit_add_subkey (winpt_key_t k, HWND dlg, listview_ctrl_t lv)
{
    gpgme_error_t ec;
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    KEYEDIT_CB cb;
    KEYGEN_CB keygen;
    char * pass = NULL;
    int cancel = 0;

    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Add Subkey"), MB_ERR );
        return FALSE;
    }
    if( k->is_protected ) {
	pass = request_passphrase (_("Key Edit"), 1, &cancel);
	if( cancel )	
	    return FALSE;
    }
    ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( dlg );
    
    memset( &keygen, 0, sizeof keygen );
    memset( &cb, 0, sizeof cb );
    cb.ek = ek;
    cb.pass = k->is_protected? pass : NULL;    
    cb.opaque = &keygen;
    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_KEYEDIT_ADDSUBKEY,    
		      dlg, keyedit_addsubkey_dlg_proc,
		      (LPARAM)&cb, _("Add new Subkey" ),        
		      IDS_WINPT_KEYEDIT_ADDSUBKEY );        
    if( !gpgme_editkey_is_valid( ek ) ) {
	free_if_alloc( pass );
	return FALSE;
    }

    ec = gpgme_new( &ctx );
    if( ec )
	BUG( dlg );
    gpgme_enable_logging( ctx );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_ADDKEY );
    gpgme_set_progress_cb( ctx, keygen_cb, NULL );
    keygen_cb_dlg_create ();
    
    ec = gpgme_op_editkey( ctx, k->keyid );
    keygen.fpr = (char *)gpgme_control (ctx, GPGME_CTRL_FPR, -1);
    keygen_cb_dlg_destroy ();
    keygen_cb (NULL, NULL, 0, 0, 0); /* flush */
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Add Subkey"), MB_ERR );
    else {	
	msg_box( dlg, _("Subkey successfully added."), _("GnuPG Status"), MB_OK );
	if( lv )
	    do_add_new_subkey( lv, &keygen, k->flags );
	keycache_set_reload( 1 );
    }
    free_if_alloc( pass );
    gpgme_editkey_release( ek );
    gpgme_release( ctx );
    
    return ec? FALSE : TRUE;
} /* keyedit_add_subkey */


BOOL
keyedit_add_photo( winpt_key_t k, HWND dlg  )
{
    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Add Photo"), MB_ERR );
        return FALSE;
    }
    DialogBoxParam( glob_hinst, (LPCTSTR)IDD_WINPT_KEYEDIT_ADDPHOTO, dlg,
		    keyedit_addphoto_dlg_proc, (LPARAM)k );
    return TRUE;
} /* keyedit_add_photo */


BOOL
keyedit_add_revoker (winpt_key_t k, HWND dlg)
{
    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Add Revoker"), MB_ERR );
        return FALSE;
    }
    DialogBoxParam( glob_hinst, (LPCTSTR)IDD_WINPT_KEYEDIT_ADDREV, dlg,
		    keyedit_addrevoker_dlg_proc, (LPARAM)k );
    return TRUE;
} /* keyedit_add_revoker */


static int
is_idea_protect_algo( const char * keyid )
{
    gpgme_key_t key;
    const char * sym_prefs;
    size_t n;

    if( get_pubkey( keyid, &key ) )
	BUG( NULL );
    sym_prefs = gpgme_key_get_string_attr( key, GPGME_ATTR_KEY_SYMPREFS, NULL, 0 );    
    if( !sym_prefs )
	return 1; /* assume that only v3 keys have no symmetric cipher preferences
		     and thus IDEA is explicit. */
    for( n = 0; sym_prefs[n]; n++ )
	;
    if( (n == 0 || n == 1) && *sym_prefs == 0x01 )
	return 1;
    return 0;
} /* is_idea_protect_algo */


BOOL
keyedit_change_passwd( winpt_key_t k, HWND dlg )
{
    gpgme_error_t ec;
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    char * old_pass = NULL, * new_pass = NULL;
    int cancel = 0;

    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Key Edit"), MB_ERR );
        return FALSE;
    }

    if( !idea_available && is_idea_protect_algo( k->keyid ) ) {
	msg_box( dlg, _("Cannot change passphrase because the key\n"
	                "is protected with the IDEA encryption algorithm."), 
			_("Key Edit"), MB_ERR );
	return FALSE;
    }

    if( k->is_protected ) {
	old_pass = request_passphrase( _("Current (old) Passphrase"), 1, &cancel );
	if( cancel )
	    return FALSE;
    }
    new_pass = request_passphrase( _("New Passphrase" ), 1, &cancel );
    if( cancel ) {
	free_if_alloc( old_pass );
        return FALSE;
    }

    if( is_8bit_string( new_pass ) ) {
	msg_box( dlg, _("The passphrase contains 8-bit characters.\n"
		         "It is not suggested to use charset specific characters."),
			 _("Key Edit"), MB_ERR );
	free_if_alloc( old_pass );
	free_if_alloc( new_pass );
	return FALSE;
    }

    ec = gpgme_new( &ctx );
    if( !ec )	
	ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( NULL );
    gpgme_enable_logging( ctx );
    gpgme_editkey_passwd_set( ek, k->is_protected? old_pass : NULL, new_pass, 0 );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_PASSWD );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Change Passwd"), MB_ERR );
    else
	msg_box( dlg, _("Passphrase successfully changed."),  _("GnuPG status"), MB_OK );
    free_if_alloc( old_pass );
    free_if_alloc( new_pass );
    gpgme_editkey_release( ek );
    gpgme_release( ctx );
    return TRUE;
} /* keyedit_change_passwd */


listview_ctrl_t
subkey_list_init( HWND dlg, winpt_key_t k )
{
    LV_ITEM lvi;
    gpgme_key_t key;
    struct listview_column_s cols[] = {
	{0, 80, (char *)_("Description")},
	{1, 78, (char *)_("Key ID")},
        {2, 66, (char *)_("Creation")},
        {3, 66, (char *)_("Expires")},
        {4, 64, (char *)_("Status")},
	{5, 16,  "C"/*ertify*/},
	{6, 16,  "S"/*ign*/},
	{7, 16,  "E"/*ncrypt*/},
	{8, 16,  "A"/*uth*/},
        {0, 0, 0}
    };
    listview_ctrl_t lv;
    char buf[256], tmp[128];
    const char *t;
    int nkeys = 0, rc = 0, i, bits, j;

    if( get_pubkey( k->keyid, &key ) ) {
	msg_box( dlg, _("Could not find key."), _("Key Edit"), MB_ERR );
	return NULL;
    }
        
    nkeys = gpgme_key_count_items( key, GPGME_ATTR_KEYID );
    if( !nkeys ) {
        msg_box( dlg, _("No subkey(s) found."), _("Key Edit"), MB_ERR );
	return NULL;
    }
        
    rc  = listview_new( &lv );
    if( rc )
	BUG( dlg );
        
    lv->ctrl = GetDlgItem( dlg, IDC_KEYEDIT_KEYLIST );
    for( i = 0; cols[i].fieldname != NULL; i++ )     
	listview_add_column( lv, &cols[i] );
        
    for( i = 0; i < nkeys; i++ ) {
        listview_add_item( lv, "" );
	listview_add_sub_item( lv, 0, 1, "" );
	memset( &lvi, 0, sizeof lvi );
	lvi.mask = LVIF_PARAM;	
	lvi.lParam = (LPARAM )key;
	if( ListView_SetItem( lv->ctrl, &lvi ) == FALSE )
	    return NULL;
    }
        
    listview_set_ext_style( lv );
    for( i = 0; i < nkeys; i++ ) {
	memset( buf, 0, sizeof buf );

	bits = gpgme_key_get_ulong_attr( key, GPGME_ATTR_LEN, NULL, i );
	_snprintf( tmp, sizeof tmp-1, "%d-bit ", bits );
	strcat( buf, tmp );

	j = gpgme_key_get_ulong_attr( key, GPGME_ATTR_ALGO, NULL, i );
	t = gpgme_key_expand_attr( GPGME_ATTR_ALGO, j );
	_snprintf( tmp, sizeof tmp-1, "%s", t );
	strcat( buf, tmp );
	
	listview_add_sub_item( lv, i, 0, buf );
	t = gpgme_key_get_string_attr( key, GPGME_ATTR_KEYID, NULL, i );
	if( !t )
	    t = "DEADBEEFDEADBEEF";
	_snprintf( tmp, sizeof tmp-1, "0x%s", t+8 );
	listview_add_sub_item( lv, i, 1, tmp );

	j = gpgme_key_get_ulong_attr( key, GPGME_ATTR_CREATED, NULL, i );
	t = gpgme_key_expand_attr( GPGME_ATTR_CREATED, j );
	if( !t )
	    t = "????-??-??";
	listview_add_sub_item( lv, i, 2, t );

	j = gpgme_key_get_ulong_attr( key, GPGME_ATTR_EXPIRES, NULL, i );
	if( j ) {
	    t = gpgme_key_expand_attr( GPGME_ATTR_CREATED, j );
	    listview_add_sub_item( lv, i, 3, t );
	}
	else
	    listview_add_sub_item( lv, i, 3, _("Never") );

	if( gpgme_key_get_ulong_attr(key, GPGME_ATTR_KEY_EXPIRED, NULL, i ) )
	    t = _("Expired");
	else if( gpgme_key_get_ulong_attr( key, GPGME_ATTR_KEY_REVOKED, NULL, i ) )
	    t = _("Revoked");
	else
	    t = _("OK");
	listview_add_sub_item( lv, i, 4, t );

	gpgme_key_get_cability( key, GPGME_ATTR_CAN_CERTIFY, i )?
	    t = "*" : t = "";
	listview_add_sub_item( lv, i, 5, t );
	gpgme_key_get_cability( key, GPGME_ATTR_CAN_SIGN, i )?
	    t = "*" : t = "";
	listview_add_sub_item( lv, i, 6, t );
	gpgme_key_get_cability( key, GPGME_ATTR_CAN_ENCRYPT, i )?
	    t = "*" : t = "";
	listview_add_sub_item( lv, i, 7, t );

	gpgme_key_get_cability (key, GPGME_ATTR_CAN_AUTH, i)?
	    t = "*" : t = "";
	listview_add_sub_item (lv, i, 8, t);
    }
    return lv;
} /* subkey_list_init */


static listview_ctrl_t
userid_list_init (HWND dlg, winpt_key_t k)
{
    listview_ctrl_t lv = NULL;
    gpgme_key_t key;
    int nuids = 0, rc, j, u_attr;
    struct listview_column_s cols[] = {
        {0,  72, (char *)_("Validity")},
        {1, 250, (char *)_("Name")},
	{2,  76, (char *)_("Creation")},
        {0, 0, 0}
    };    
    const char *attr;

    if( get_pubkey( k->keyid, &key ) ) {
	msg_box( dlg, _("Could not find key."), _("Key Edit"), MB_ERR );
	return NULL;
    }
        
    nuids = gpgme_key_count_items( key, GPGME_ATTR_USERID );
    if( !nuids ) {
        msg_box( dlg, _("No user ID(s) found."), _("Key Edit"), MB_ERR );
	return NULL;
    }
        
    rc = listview_new( &lv );
    if( rc )
	BUG( dlg );        
    lv->ctrl = GetDlgItem( dlg, IDC_KEYEDIT_UIDLIST );
    for( j = 0; cols[j].fieldname != NULL; j++ )
        listview_add_column( lv, &cols[j] );
        
    for( j = 0; j < nuids; j++ ) {           
	listview_add_item( lv, " " );
	listview_add_sub_item( lv, 0, 1, " " );        
    }

    listview_set_ext_style (lv);
    for (j = 0; j < nuids; j++) {
	if (gpgme_key_get_ulong_attr (key, GPGME_ATTR_UID_REVOKED, NULL, j))
	    attr = _("Revoked");
	else {
	    u_attr = gpgme_key_get_ulong_attr (key, GPGME_ATTR_VALIDITY, NULL, j);
	    attr = gpgme_key_expand_attr (GPGME_ATTR_VALIDITY, u_attr);
	}
	listview_add_sub_item( lv, j, 0, (char *)attr );	

	attr = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, j );
	if( attr ) {
	    char * uid = utf8_to_wincp (attr, strlen (attr));
	    if (uid) {
		listview_add_sub_item( lv, j, 1, uid );
		free( uid );
	    }
	}
	else
	    listview_add_sub_item( lv, j, 1, _("Invalid user ID") );
	u_attr = gpgme_key_get_ulong_attr (key, GPGME_ATTR_UID_CREATED, NULL, j);
	if (u_attr)
	    listview_add_sub_item (lv, j, 2, get_key_created (u_attr));
    }
    if( !k->key_pair ) {
	CheckDlgButton( dlg, IDC_KEYUID_ADD, BST_INDETERMINATE );
	CheckDlgButton( dlg, IDC_KEYUID_REVOKE, BST_INDETERMINATE );	
    }
    return lv;
} /* userid_list_init */


static void
do_init_cmdlist( HWND dlg )
{
    const char *cmdlist[] = {
	"ADDKEY",
	"ADDUID",
	"ADDPHOTO",
	"ADDREVOKER",
	/*"FPR",*/
	"DELUID",
	"DELKEY",
	"DELPHOTO",
	/*"DELSIG",*/
	"EXPIRE",
	/*"PREF",
	"SHOWPREF",
	"SETPREF",*/
	"UPDPREF",
	"PASSWD",
	"PRIMARY",
	"TRUST",
	/*"REVSIG",*/
	"REVUID",
	"REVKEY",
	"DISABLE",
	"ENABLE",
	"SHOWPHOTO",
	NULL
    };
    const char * s;
    int i = 0;

    for( i = 0; (s=cmdlist[i]); i++ ) {
	SendDlgItemMessage( dlg, IDC_KEYEDIT_CMD, CB_ADDSTRING, 0,
			    (LPARAM)(char *)s );
    }
    SendDlgItemMessage( dlg, IDC_KEYEDIT_CMD, CB_SETCURSEL, 0, 0 );
} /* do_init_cmdlist */


static int
is_cmd_openpgp( int cmdid )
{
    switch( cmdid ) {
    case CMD_ADDKEY:
    case CMD_ADDPHOTO:
    case CMD_ADDREVOKER:
    case CMD_DELPHOTO:
    /*case CMD_SHOWPHOTO:*/
    case CMD_UPDPREF:
	return 1;
    }
    return 0;
} /* is_cmd_openpgp */


static void
do_show_help( HWND dlg )
{
    char helptext[2048];

    _snprintf( helptext, sizeof helptext-1,
	_(/*"FPR	    \t\tshow fingerprint\r\n"*/
	 "ADDUID   \t\tadd a user ID\r\n"
	 "ADDPHOTO  \t\tadd a photo ID\r\n"
	 "DELUID    \t\tdelete a user ID\r\n"
	 "ADDKEY    \t\tadd a secondard key\r\n"
	 "DELKEY    \t\tdelete a secondary key\r\n"
	 "ADDREVOKER\t\tadd a revocation key\r\n"
	 /*"DELSIG    \t\tdelete signatures\r\n"*/
	 "EXPIRE    \t\tchange the expire date\r\n"
	 /*"PREF	    \t\tlist preferences (expert)\r\n"
	 "SHOWPREF  \t\tlist preferences (verbose)\r\n"
	 "SETPREF   \t\tset preference list\r\n"*/
	 "UPDPREF   \t\tupdated preferences\r\n"
	 "PASSWD    \t\tchange the passphrase\r\n"
	 "PRIMARY   \t\tflag user ID as primary\r\n"
	 "TRUST	    \t\tchange the ownertrust\r\n"
	 /*"REVSIG    \t\trevoke signatures\r\n"*/
	 "REVUID    \t\trevoke a user ID\r\n"
	 "REVKEY    \t\trevoke a secondary key\r\n"
	 "DISABLE   \t\tdisable a key\r\n"
	 "ENABLE    \t\tenable a key\r\n"
	 /*"SHOWPHOTO \t\tshow photo ID\r\n"*/) );
    msg_box( dlg, helptext, _("Key Edit Help"), MB_OK );
} /* do_show_help */


static int
do_editkey_delkey( winpt_key_t k, HWND dlg, listview_ctrl_t lv  )
{
    int j, id;
    char tmp[64];
    gpgme_error_t ec;
    gpgme_editkey_t ek;
    gpgme_ctx_t ctx;

    if( listview_count_items( lv, 0 ) == 1 ) {
	msg_box( dlg, _("Primary key can not be deleted!"), _("Key Edit"), MB_ERR);
        return FALSE;
    }
    if( (j = listview_get_curr_pos( lv )) == -1 ) {
	msg_box( dlg, _("Please select a key."), _("Key Edit"), MB_ERR );
        return FALSE;
    }
    if( j == 0 ) {
	msg_box( dlg, _("Primary subkey can not be deleted!"), _("Key Edit"), MB_ERR );
        return FALSE;
    }
     
    listview_get_item_text( lv, j, 0, tmp, sizeof tmp -1 );
    id = log_box( _("Key Edit"), MB_YESNO|MB_ICONWARNING,
		    _("\"Subkey %s.\"\n\n"
		      "Anything encrypted to the selected subkey will no longer\n"
		      "be able to be decrypted.\n\n"
		      "Do you really want to delete this subkey?"), tmp );
    if( id == IDNO )
	return FALSE;

    ec = gpgme_new( &ctx );
    if( !ec )
	ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( dlg );
    gpgme_enable_logging( ctx );
    gpgme_editkey_delkey_set_id( ek, j );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_DELKEY );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Delete Subkey"), MB_ERR );
    else {
	listview_del_item( lv, j );
	keycache_set_reload( 1 );    
	status_box( dlg, _("Subkey successfully deleted."), _("GnuPG status") );
    }
    gpgme_editkey_release( ek );
    gpgme_release( ctx );    
    return ec? FALSE : TRUE;
} /* do_editkey_delkey */


static int
do_editkey_expire( winpt_key_t k, HWND dlg, listview_ctrl_t lv )
{
    gpgme_error_t ec;
    gpgme_editkey_t ek;
    gpgme_ctx_t ctx;
    date_s udd = {0};
    char buf[256], * pass = NULL;
    int j, cancel = 0;

    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Key Edit"), MB_ERR );
        return FALSE;
    }
    if ( (j = listview_get_curr_pos( lv )) == -1 ) {
	msg_box( dlg, _("Please select a key."), _("Key Edit"), MB_ERR );
        return FALSE;
    }
            
    listview_get_item_text( lv, j, 3, buf, sizeof buf -1 );    
    if( !strcmp( buf, _("Expired") ) ) {    
	msg_box( dlg, _("Key already expired!"), _("Key Edit"), MB_ERR );
        return FALSE;
    }
    memset( &udd, 0, sizeof udd );
    udd.text = _("Key Expiration Date");
    dialog_box_param( glob_hinst, (LPCSTR)IDD_WINPT_DATE, dlg,    
			date_dlg_proc, (LPARAM)&udd,
		    _("Key Expiration Date"), IDS_WINPT_DATE );
    if( udd.cancel == 1 )
	return FALSE;
    if( !keygen_check_date( &udd.st ) ) {
	msg_box( dlg, _("The date you have chosen lies in the past."), _("Key Edit"), MB_ERR );
        return FALSE;
    }
    if( k->is_protected ) {
	pass = request_passphrase (_("Key Edit"), 1, &cancel );
	if( cancel )
	    return FALSE;
    }
    _snprintf( buf, sizeof buf - 1, "%04d-%02d-%02d",     
		udd.st.wYear, udd.st.wMonth, udd.st.wDay );
    ec = gpgme_editkey_new( &ek );
    if( !ec )
	ec = gpgme_new( &ctx );
    if( ec )
	BUG( dlg );
    gpgme_editkey_expire_set( ek, j, 0, buf, k->is_protected? pass : NULL );
    gpgme_enable_logging( ctx );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_EXPIRE );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Expire Subkey"), MB_ERR );
    else {
	listview_add_sub_item( lv, j, 3, buf );
	keycache_set_reload( 1 );
	msg_box( dlg, _("Subkey expire date successfully set."), _("GnuPG status"), MB_OK );
    }
    free_if_alloc( pass );
    gpgme_release( ctx );
    gpgme_editkey_release( ek ); 
    return TRUE;
} /* do_editkey_expire */


static int
do_editkey_revoke( winpt_key_t k, HWND dlg, listview_ctrl_t lv )
{
    gpgme_ctx_t ctx;
    gpgme_error_t ec;
    gpgme_editkey_t ek;
    char buf[256], * pass = NULL;
    int j, cancel = 0;

    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Key Edit"), MB_ERR );
        return FALSE;
        
    }			
        
    if( (j = listview_get_curr_pos( lv )) == -1 ) {
	msg_box( dlg, _("Please select a key."), _("Key Edit"), MB_ERR );
	return FALSE;
    }
    else if( listview_count_items( lv, 0 ) == 1 ) {
	msg_box( dlg, _("No subkeys were found, if you want to revoke the\n"
	                "whole key, please use the Key Manager command directly.\n\n"
			"This command is only available to revoke single subkeys"),
		 _("Key Edit"), MB_INFO );
	return FALSE;
    }
	    
    listview_get_item_text( lv, j, 3, buf, sizeof buf-1 );
    if( !strcmp( buf, _("Revoked") ) ) {
	msg_box( dlg, _("Key already revoked."), _("Key Edit"), MB_ERR );
	return FALSE;
    }
    
    if( k->is_protected ) {
	pass = request_passphrase (_("Key Edit"), 1, &cancel);
	if( cancel )
	    return FALSE;	    
    }
	    
    ec = gpgme_editkey_new( &ek );
    if( !ec )
	ec = gpgme_new( &ctx );
    if( ec )
	BUG( NULL );
    gpgme_enable_logging( ctx );
    gpgme_editkey_revkey_set( ek, j, 0, k->is_protected? pass : NULL );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_REVKEY );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Revoke Subkey"), MB_ERR );
    else {
	listview_add_sub_item( lv, j, 5, _("Revoked") );
	keycache_set_reload( 1 );
	msg_box( dlg, _("Subkey successfully revoked."), _("GnuPG Status"), MB_OK );
    }
    free_if_alloc( pass );
    gpgme_release( ctx );
    gpgme_editkey_release( ek );
    return TRUE;
} /* do_editkey_revoke */


int
do_editkey_revuid( winpt_key_t k, HWND dlg, listview_ctrl_t lv )
{
    gpgme_ctx_t ctx;
    gpgme_error_t ec;
    gpgme_editkey_t ek;
    char buf[256], t[512], * pass;
    int cancel = 0, id = 0, j;

    if( !k->key_pair ) {
	msg_box( dlg, _("There is no secret key available!"), _("Revoke user ID"), MB_ERR );
	return FALSE;
    }

    if( listview_count_items( lv, 0 ) == 1 ) {
	msg_box( dlg, _("Key has only one user ID."), _("Key Edit"), MB_ERR );	
	return FALSE;
    }

    if( (j = listview_get_curr_pos( lv )) == -1 ) {
	msg_box( dlg, _("Please select a user ID."), _("Key Edit"), MB_ERR );	
	return FALSE;	
    }
	    
    listview_get_item_text( lv, j, 0, buf, sizeof buf - 1 );
    if( strstr( buf, _("Revoked") ) ) {
	msg_box( dlg, _("This user ID has been already revoked."), _("Key Edit"), MB_INFO );
	return FALSE;
    }
	    
    listview_get_item_text( lv, j, 1, buf, sizeof buf -1 );
    _snprintf( t, sizeof t -1, _("user ID \"%s\".\n\n"
		"Do you really want to revoke this user ID?"), buf );
    if( msg_box( dlg, t, _("Key Edit"), MB_YESNO|MB_ICONWARNING ) == IDNO )
	return FALSE;
    if( k->is_protected ) {
	pass = request_passphrase (_("Key Edit"), 1, &cancel);
	if( cancel )	
	    return FALSE;	    
    }
    id = do_find_userid( k->keyid, buf );
    if( id == -1 )
	BUG( dlg );
    ec = gpgme_new( &ctx );
    if( !ec )	
	ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( dlg );
    gpgme_enable_logging( ctx );
    gpgme_editkey_revsig_set( ek, id, k->is_protected? pass : NULL );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_REVSIG );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Revoke Signature"), MB_ERR );
    else {
	listview_add_sub_item( lv, j, 2, _("Revoked") );
	keycache_set_reload( 1 );
	status_box( dlg, _("User ID successfully revoked"), _("GnuPG Status") );	
    }
    free_if_alloc( pass );
    gpgme_editkey_release( ek );
    gpgme_release( ctx );
    return ec? FALSE : TRUE;
} /* do_editkey_revuid */


static int
do_editkey_setpref (winpt_key_t k, HWND dlg, listview_ctrl_t lv)
{
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    gpgme_error_t rc;
    char buf[256], * pass = NULL, * prefs;
    int j, id, cancel=0, flags=0;

    if ((j = listview_get_curr_pos (lv)) == -1) {
	msg_box (dlg, _("Please select a user ID."), _("Key Edit"), MB_ERR);
	return FALSE;
    }
    listview_get_item_text (lv, j, 1, buf, sizeof buf-1);
    id = do_find_userid (k->keyid, buf);
    if (id == -1)
	BUG (dlg);
    if (k->is_protected) {
	pass = request_passphrase (_("Key Edit"), 1, &cancel);
	if (cancel)
	    return FALSE;
    }
    rc = gpgme_new (&ctx);
    if (!rc)
	rc = gpgme_editkey_new (&ek);
    if (rc)
	BUG (NULL);

    get_userid_preflist (&prefs, &flags);
    gpgme_editkey_setpref_set (ek, prefs, id, pass);
    gpgme_set_edit_ctx (ctx, ek, GPGME_EDITKEY_SETPREF);
    rc = gpgme_op_editkey (ctx, k->keyid);
    free_if_alloc (pass);

    free_if_alloc (prefs);
    gpgme_release (ctx);
    gpgme_editkey_release (ek);
    return 0;
}


static int
do_editkey_primary( winpt_key_t k, HWND dlg, listview_ctrl_t lv )
{
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    gpgme_error_t ec;
    int j, id, cancel=0;
    char buf[256], * pass = NULL;

    if( (j = listview_get_curr_pos( lv )) == -1 ) {
	msg_box( dlg, _("Please select a user ID."), _("Key Edit"), MB_ERR );
	return FALSE;
    }
    listview_get_item_text( lv, j, 1, buf, sizeof buf-1 );
    id = do_find_userid (k->keyid, buf);
    if( id == -1 )
	BUG( dlg );
    if( k->is_protected ) {
	pass = request_passphrase( _("Key Edit"), 1, &cancel );
	if( cancel )
	    return FALSE;
    }

    ec = gpgme_new( &ctx );
    if( !ec )
	ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( dlg );
    gpgme_enable_logging( ctx );
    gpgme_editkey_primary_set( ek, id, k->is_protected? pass : NULL );
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_PRIMARY );
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Primary"), MB_ERR );
    else {
	keycache_set_reload( 1 );
	status_box( dlg, _("User ID successfully flagged"), _("GnuPG Status") );
    }

    free_if_alloc( pass );
    gpgme_editkey_release( ek );
    gpgme_release( ctx );
    return ec? FALSE : TRUE;
} /* do_editkey_primary */


static int
do_editkey_deluid( winpt_key_t k, HWND dlg, listview_ctrl_t lv )
{
    gpgme_ctx_t ctx;
    gpgme_editkey_t ek;
    gpgme_error_t ec;
    char buf[256], t[512];
    int j, id = 0;

    if( listview_count_items( lv, 0 ) == 1 ) {
	msg_box( dlg, _("Primary user ID can not be deleted!"), _("Key Edit"), MB_ERR );
        return FALSE;
    }
    if( (j = listview_get_curr_pos( lv )) == -1 ) {
	msg_box( dlg, _("Please select a user ID."), _("Key Edit"), MB_ERR );
        return FALSE;
    }
    
    listview_get_item_text( lv, j, 1, buf, sizeof buf -1 );
    _snprintf( t, sizeof t -1, _("user ID \"%s\".\n\n"
	"Do you really want to delete this user ID?"), buf );
    if( msg_box( dlg, t, _("Key Edit"), MB_YESNO|MB_ICONWARNING ) == IDNO )
	return FALSE;
    
    id = do_find_userid( k->keyid, buf );
    if( id == -1 )
	BUG( dlg );        
    ec = gpgme_new( &ctx );
    if( !ec )
	ec = gpgme_editkey_new( &ek );
    if( ec )
	BUG( dlg );
    gpgme_enable_logging( ctx );
    gpgme_editkey_deluid_set_id( ek,  id );    
    gpgme_set_edit_ctx( ctx, ek, GPGME_EDITKEY_DELUID );    
    ec = gpgme_op_editkey( ctx, k->keyid );
    if( ec )
	gpgme_show_error( dlg, ec, ctx, _("Delete user ID"), MB_ERR );
    else {
	listview_del_item( lv, j );
	keycache_set_reload( 1 );
	status_box( dlg, _("User ID successfully deleted"), _("GnuPG Status") );
    }
    gpgme_editkey_release( ek );    
    gpgme_release( ctx );
    return ec? FALSE : TRUE;
} /* do_editkey_deluid */



static BOOL CALLBACK
subkey_subclass_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    switch( msg ) {
    case WM_KEYUP:
	int virt_key = (int)wparam;
	switch( virt_key ) {	    
	case VK_DELETE:
	    SendDlgItemMessage( keyedit_subkey_proc.dlg, IDC_KEYEDIT_CMD, 
				CB_SETCURSEL, CMD_DELKEY, 0 );
	    send_cmd_id( keyedit_subkey_proc.dlg, IDOK );
	    break;

	case VK_INSERT:
	    SendDlgItemMessage( keyedit_subkey_proc.dlg, IDC_KEYEDIT_CMD, 
				CB_SETCURSEL, CMD_ADDKEY, 0 );
	    send_cmd_id( keyedit_subkey_proc.dlg, IDOK );
	    break;
	}
    }
    return CallWindowProc( keyedit_subkey_proc.old, dlg, msg, wparam, lparam );
} /* subkey_subclass_proc */


static BOOL CALLBACK
uid_subclass_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    switch( msg ) {
    case WM_KEYUP:
	int virt_key = (int)wparam;
	switch( virt_key ) {
	case VK_DELETE:
	    SendDlgItemMessage( keyedit_uid_proc.dlg, IDC_KEYEDIT_CMD,
				CB_SETCURSEL, CMD_DELUID, 0 );
	    send_cmd_id( keyedit_uid_proc.dlg, IDOK );
	    break;

	case VK_INSERT:
	    SendDlgItemMessage( keyedit_uid_proc.dlg, IDC_KEYEDIT_CMD,
				CB_SETCURSEL, CMD_ADDUID, 0 );
	    send_cmd_id( keyedit_uid_proc.dlg, IDOK );
	    break;
	}
    }
    return CallWindowProc( keyedit_uid_proc.old, dlg, msg, wparam, lparam );
} /* uid_subclass_proc */


BOOL CALLBACK
keyedit_main_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static winpt_key_t k;
    static listview_ctrl_t lvsub = NULL, lvuid = NULL;
    int cmd, idxsub = 0;
    HWND item;

    switch( msg ) {
    case WM_INITDIALOG:
	k = (winpt_key_t)lparam;
	if( !k )
	    BUG( NULL );
	do_init_cmdlist( dlg );
	lvsub = subkey_list_init( dlg, k );
	if( !lvsub )
	    BUG( NULL );
	lvuid = userid_list_init (dlg, k);
	if( !lvuid )
	    BUG( NULL );
	item = GetDlgItem( dlg, IDC_KEYEDIT_KEYLIST );
	keyedit_subkey_proc.dlg = dlg;
	keyedit_subkey_proc.current = (WNDPROC)subkey_subclass_proc;
	keyedit_subkey_proc.old = (WNDPROC)GetWindowLong( item, GWL_WNDPROC );
	if( keyedit_subkey_proc.old ) {
	    if( !SetWindowLong( item, GWL_WNDPROC, (LONG)keyedit_subkey_proc.current ) ) {
		msg_box( dlg, _("Could not set subkey window procedure."), _("Key Edit"), MB_ERR );
		BUG( NULL );
	    }
	}
	item = GetDlgItem( dlg, IDC_KEYEDIT_UIDLIST );
	keyedit_uid_proc.dlg = dlg;
	keyedit_uid_proc.current = (WNDPROC)uid_subclass_proc;
	keyedit_uid_proc.old = (WNDPROC)GetWindowLong( item, GWL_WNDPROC );
	if( keyedit_uid_proc.old ) {
	    if( !SetWindowLong( item, GWL_WNDPROC, (LONG)keyedit_uid_proc.current ) ) {
		msg_box( dlg, _("Could not set user ID window procedure."), _("Key Edit"), MB_ERR );
		BUG( NULL );
	    }
	}
	if (!k->key_pair) {
	    EnableWindow (GetDlgItem (dlg, IDC_KEYEDIT_CMD), FALSE);
	    EnableWindow (GetDlgItem (dlg, IDOK), FALSE);
	}
	SetForegroundWindow( dlg );
	center_window( dlg );
	return TRUE;

    case WM_DESTROY:
	if( lvsub ) {
	    listview_release( lvsub );
	    lvsub = NULL;
	}
	if( lvuid ) {
	    listview_release( lvuid );
	    lvuid = NULL;
	}
	break;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDOK:
	    cmd = SendDlgItemMessage( dlg, IDC_KEYEDIT_CMD, CB_GETCURSEL, 0, 0 );
	    if( cmd == LB_ERR ) {
		msg_box( dlg, _("Please select a command."), _("Key Edit"), MB_INFO );
		return FALSE;
	    }
	    idxsub = listview_get_curr_pos( lvsub );
	    if( km_key_is_v3( lvsub, idxsub==-1? 0 : idxsub ) && is_cmd_openpgp( cmd ) ) {
		msg_box( dlg, _("This command cannot be used with PGP 2 (v3) keys\n"
				" because it is not OpenPGP compliant."), 
				_("Key Edit"), MB_ERR );
		return FALSE;
	    }
	    switch( cmd ) {
	    case CMD_DELKEY: do_editkey_delkey( k, dlg, lvsub ); break;
	    case CMD_ADDKEY: keyedit_add_subkey( k, dlg, lvsub ); break;
	    case CMD_EXPIRE: do_editkey_expire( k, dlg, lvsub ); break;
	    case CMD_REVKEY: do_editkey_revoke( k, dlg, lvsub ); break;
	    case CMD_UPDPREF:do_editkey_setpref( k, dlg, lvuid ); break;
	    case CMD_ADDUID: keyedit_add_userid( k, dlg, lvuid ); break;
	    case CMD_ADDREVOKER: keyedit_add_revoker( k, dlg ); break;
	    case CMD_ADDPHOTO: keyedit_add_photo( k, dlg ); break;
	    case CMD_REVUID: do_editkey_revuid( k, dlg, lvuid ); break;
	    case CMD_DELUID: do_editkey_deluid( k, dlg, lvuid ); break;
	    case CMD_PASSWD: keyedit_change_passwd( k, dlg ); break;
	    case CMD_PRIMARY: do_editkey_primary( k, dlg, lvuid ); break;
	    case CMD_ENABLE: km_enable_disable_key( lvsub, dlg, idxsub, 1 ); break;
	    case CMD_DISABLE: km_enable_disable_key( lvsub, dlg, idxsub, 0 ); break;
	    }
	    break;

	case IDCANCEL:
	    EndDialog( dlg, FALSE );
	    break;

	case IDC_KEYEDIT_HELP:
	    do_show_help( dlg );
	    break;
	}
	break;
    }
    return FALSE;
} /* keyedit_main_dlg_proc */