## -*-Tcl-*-
 # ###################################################################
 #  AECoerce - Coersion functions for AEGizmo values
 # 
 #  FILE: "aecoerce.tcl"
 #                                    created: 3/3/98 {11:53:59 PM} 
 #                                last update: 22/4/1999 {6:08:56 pm} 
 #                                    version: 1.1
 #  Author: Jonathan Guyer
 #  E-mail: <jguyer@his.com>
 #     www: <http://www.his.com/~jguyer/>
 #  
 # Copyright (c) 1998  Jonathan Guyer
 # 
 # This program is free software; you can redistribute it and/or modify
 # it under the terms of the GNU General Public License as published by
 # the Free Software Foundation; either version 2 of the License, or
 # (at your option) any later version.
 # 
 # This program is distributed in the hope that it will be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 # GNU General Public License for more details.
 # 
 # You should have received a copy of the GNU General Public License
 # along with this program; if not, write to the Free Software
 # Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 # See the file "license.terms" for information on usage and 
 # redistribution of this file, and for a DISCLAIMER OF ALL 
 # WARRANTIES.
 #  
 # ###################################################################
 ##

namespace eval aecoerce {}

ensureset aecoerce::overrides {}
ensureset aecoerce::noCoerce {}
	
proc aecoerce::identity {value} {
	return $value
}

## 
 # bool ::= bool(00|01)
 ##
proc aecoerce::hexd:bool {value} {
	set value [aecoerce::hexd $value]
	set bool [expr {"0x$value"}]
	if {($bool != 0) && ($bool != 1)} {
		set msg "Can't coerce $value from 'hexd' to 'bool'"
		error $msg "" [list AECoerce -1700 $msg]
	} 
	return $bool
}

proc aecoerce::hexd:TEXT {value} {
	# make sure input is really hexd
	set value [aecoerce::hexd $value]
	
	set TEXT ""
	while {[string length $value]} {
		append TEXT [uplevel 0 "set temp \\x[string range $value 0 1]"]
		set value [string range $value 2 end]
	}
	return $TEXT
}

proc aecoerce::hexd {value} {
	regsub -all -nocase "\[ \t\r\n\]" $value "" newval
	if {[expr {[string length $newval] % 2}]} {
		# left pad with zero to make even number of digits
		set newval "0${newval}"
	} 
	if {![is::Hexadecimal $newval]} {
		set msg "Non-hex-digit in ${value}" 
		error $msg "" [list AECoerce 6 $msg]
	} else {
		return ${newval}
	}
}

proc aecoerce::null:TEXT {value} {
	return ""
}

proc aecoerce::hexd:alis {value} {
	return [aeparse::keywordValue ---- \
		[aeparse::event \
			[AEBuild -r 'MACS' core getd ---- \
				"obj {form:alis, want:file, from:'null'(), \
						seld:[aebuild::coercion "alis" [aebuild::hexd $value]] \
				}" \
				rtyp TEXT
			] \
		] \
	]
}

proc aecoerce::TEXT:alis {value} {
	return [coerce TEXT $value -x alis]
}

proc aecoerce::register {from to proc} {
	global aecoerce::coercions
	
	if {$from == $to} {
		error "Coercing '$from' to '$to' is just stupid!"
	} 
	
	set procs ""
	if {![info exists aecoerce::coercions]} {
		set aecoerce::coercions ""
	}
	set coercions ${aecoerce::coercions}
	
	set new [list $from $to *]
	while {[set i [lsearch -glob $coercions $new]] != -1} {
		lappend procs [lindex [lindex $coercions $i] 2]
		set coercions [lrange $coercions [incr i] end]
	}
	
	if {[llength $procs]} {
		
		set procs [lsort [lunique [lappend procs $proc]]]
		if {[llength $procs] > 1} {
			set proc \
			  [listpick -p \
			  	"Only one coersion from '$from' to '$to' is allowed:" \
				$procs \
			  ]
			set procs [lremove -all $procs $proc]
			
			foreach oldproc $procs {
				set aecoerce::coercions \
				  [lremove -all ${aecoerce::coercions} \
				    [list $from $to $oldproc] \
				  ]
			}
		}
	}
	lappend aecoerce::coercions [list $from $to $proc]
	set aecoerce::coercions [lunique ${aecoerce::coercions}]
}

proc aecoerce::apply {value to {typed 0}} {
	global aecoerce::coercions aecoerce::overrides aecoerce::noCoerce
	
	set from [lindex $value 0]
	set value [lindex $value 1]
	
	if {$from == "list"} {
		set msg "Cannot coerce a list"
		error $msg "" [list AECoerce 18 $msg]
	} 
	
	# no need to do anything for an identity coercion
	if {$from != $to} {		
		set coerce [list $from $to]
		
		foreach noCoerce ${aecoerce::noCoerce} {
			if {[string match $noCoerce $coerce]} {
				# return what was sent
				return [list $from $value]
			} 	
		}
		
		# coercion not blocked, so see if we know how to do it
		if {[set i [lsearch -glob ${aecoerce::overrides} [list $from $to *]]] != -1} {
			set value [[lindex [lindex ${aecoerce::overrides} $i] 2] $value]
		} elseif {[set i [lsearch -glob ${aecoerce::coercions} [list $from $to *]]] != -1} {
			set value [[lindex [lindex ${aecoerce::coercions} $i] 2] $value]
		} else {
			# -1700 is a coercion failure.
			# That's not exactly what we want; coercion didn't
			# fail, we just don't know how to do it.
			set msg "Can't coerce '$from' to '$to'"
			error $msg "" [list AECoerce 1700 $msg]
		}
	}
	if {$typed} {
		return [list $to $value]
	} else {
		return $value
	} 
}
	
# !!! NEEDS TO BE IMPLEMENTED !!!
proc aecoerce::deregister {hook {procname ""} args} {
    if {![llength $args]} {set args "*"}
    namesp hook::${hook}
    global hook::${hook}
    if {$procname == ""} { 
		# clear all hooks
		unset hook::${hook} 
	} else {		
		foreach mode $args {
		    if {[info exists hook::${hook}($mode)] \
			&& ([set i [lsearch -exact [set hook::${hook}($mode)] $procname]] != -1)} {
			set new [lreplace [set hook::${hook}($mode)] $i $i]
			if {$new != ""} {
			    set hook::${hook}($mode) $new
			} else {
			    unset hook::${hook}($mode)
			}
		    }
		}
	}
}

#  Default Coercions  #

aecoerce::register "hexd" "bool" aecoerce::hexd:bool
aecoerce::register "hexd" "TEXT" aecoerce::hexd:TEXT
aecoerce::register "null" "TEXT" aecoerce::null:TEXT
aecoerce::register "hexd" "alis" aecoerce::hexd:alis
aecoerce::register "hexd" "fss " specToPathName
aecoerce::register "TEXT" "alis" aecoerce::TEXT:alis
aecoerce::register "shor" "long" aecoerce::identity
aecoerce::register "long" "shor" aecoerce::identity

