## -*-Tcl-*-
 # ###################################################################
 #  Alpha - new Tcl folder configuration
 # 
 #  FILE: "prefsHandling.tcl"
 #                                    created: 24/2/95 {9:52:30 pm} 
 #                                last update: 05/28/1999 {10:22:55 AM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: Division of Engineering and Applied Sciences, Harvard University
 #          Oxford Street, Cambridge MA 02138, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Reorganisation carried out by Vince Darley with much help from Tom 
 # Fetherston, Johan Linde and suggestions from the Alpha-D mailing list.  
 # Alpha is shareware; please register with the author using the register 
 # button in the about box.
 #  
 #  Description: 
 # 
 # Procedures for dealing with the user's preferences
 # ###################################################################
 ##

namespace eval mode {}
namespace eval global {}

proc addArrDef {arr def val} {
    addDef [list $arr $def] $val arr
}

proc removeArrDef {arr def} {
    removeDef [list $arr $def] arr
}

proc addDef {def val {prefix {}}} {
    global ${prefix}prefDefs
    
    readDefs $prefix
    set ${prefix}prefDefs($def) $val
    writeDefs $prefix
    catch {unset ${prefix}prefDefs}
}

proc removeDef {def {prefix {}}} {
    global ${prefix}prefDefs
    
    readDefs $prefix
    catch {unset ${prefix}prefDefs($def)}
    writeDefs $prefix
    catch {unset ${prefix}prefDefs}
}

##
 # -------------------------------------------------------------------------
 #
 # "removeArr" --
 #
 #  Remove all elements of $arr from arrdefs.tcl
 # -------------------------------------------------------------------------
 ##
proc removeArr {arr} {
    global arrprefDefs $arr
    
    readDefs arr
    foreach def [array names $arr] {
	catch {unset arrprefDefs([list $arr $def])}
    }
    writeDefs arr
    catch {unset arrprefDefs}
}

proc addArr {arr} {
    global arrprefDefs $arr
    
    readDefs arr
    foreach def [array names $arr] {
	catch {set arrprefDefs([list $arr $def]) [set ${arr}($def)]}
    }
    writeDefs arr
    catch {unset arrprefDefs}
}

proc readDefs {{prefix {}}} {
    global PREFS
    if {![file exists [file join $PREFS ${prefix}defs.tcl]]} return
    uplevel \#0 [list source [file join $PREFS ${prefix}defs.tcl]]
}

proc writeDefs {{prefix {}}} {
    global HOME PREFS ${prefix}prefDefs 
    
    if {![info exists ${prefix}prefDefs]} {
	catch {file delete [file join $PREFS ${prefix}defs.tcl]}
	return
    }
    
    if {![file exists "$PREFS"]} {
	file mkdir "$PREFS"
    }
    set fd [open [file join $PREFS ${prefix}defs.tcl] "w"]
    foreach nm [array names ${prefix}prefDefs] {
	puts $fd [list set ${prefix}prefDefs($nm) [set ${prefix}prefDefs($nm)]]
    }
    close $fd
}


proc alpha::readUserDefs {} {
    namespace eval :: {
	global prefDefs arrprefDefs PREFS
	
	if {[file exists [file join $PREFS defs.tcl]]} {
	    source [file join $PREFS defs.tcl]
	    
	    foreach nm [array names prefDefs] {
		global $nm
		if {[catch {set $nm $prefDefs($nm)}]} {
		    set ns ""
		    while {[regexp "^($ns\[a-zA-Z_\]+::)" $nm ns]} {
			namespace eval $ns {}
		    }
		    set $nm $prefDefs($nm)
		}
		
	    }
	    catch {unset prefDefs}
	}
	
	if {[file exists [file join $PREFS arrdefs.tcl]]} {
	    source [file join $PREFS arrdefs.tcl]
	    
	    foreach nm [array names arrprefDefs] {
		set arr [lindex $nm 0]
		set field [lindex $nm 1]
		set val $arrprefDefs($nm)
		global $arr
		set ${arr}($field) $val
		if {[catch {set ${arr}($field) $val}]} {
		    set ns ""
		    while {[regexp "^($ns\[a-zA-Z_\]+::)" $arr ns]} {
			namespace eval $ns {}
		    }
		    set ${arr}($field) $val
		}
	    }
	    catch {unset arrprefDefs}
	}
    }
    
}

proc alpha::readUserPrefs {} {
    global PREFS
    # Use "prefs.tcl" to define or change any tcl information. 
    if {![file exists [file join $PREFS prefs.tcl]]} {
	if {![file exists "$PREFS"]} {
	    file mkdir "$PREFS"
	}
	set fd [open [file join $PREFS prefs.tcl] "w"]
	close $fd
    }
    uplevel #0 {
	if {[catch {source [file join $PREFS prefs.tcl]}]} {
	    if {[dialog::yesno "An error occurred while loading \"prefs.tcl\".  Shall I make a trace on the error?"]} {
		dumpTraces "prefs.tcl error" $errorInfo
	    }
	}
    }
}

	
proc viewSavedSetting {} {
    global prefDefs arrprefDefs
    
    saveModifiedVars
    
    if {[catch {listpick -p "The following settings have been saved:" [getSavedSettings]} res]} {
	return
    }
    
    if {[regexp {([^(]+)\(([^)]+)\)} $res dummy arr field]} {
	set arg [list $arr $field]
	set val $arrprefDefs($arg)
    } else {
	global $res
	set val $prefDefs($res)
    }	
    viewValue $res $val
    catch {unset prefDefs}
    catch {unset arrprefDefs}
}

proc viewValue {name val} {
    set header "'$name's value is:"
    set response "\r$val\r"
    if {[string length $val] > 80} {
	if {([llength $val] > 3) && ([llength $val] > 6 || [string length $val] > 160)} {
	    listpick -p "'$name's value is:" $val
	} else {
	    if {[tclLog $header$response]} {
		global tileLeft tileTop tileWidth
		new -g $tileLeft $tileTop $tileWidth 100 -n "* $name *" -m Text \
		  -info "'$name's value is:\r\r$val\r"
	    }
	}
    } else {
	global mode
	if {$mode == "Shel"} {
	    goto [maxPos]
	    tclLog $header$response
	    insertText [Alpha::Prompt]
	} else {
	    alertnote "$header\r$response"
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "removeSavedSetting" --
 # 
 #  This proc shouldn't 'unset' the variables it removes, because most
 #  such variables will be in use/have default values until restart.
 # -------------------------------------------------------------------------
 ##
proc removeSavedSetting {} {
    global prefDefs arrprefDefs
    
    saveModifiedVars
    if {[catch {listpick -p "Remove which setting?" [lsort -ignore [getSavedSettings]]} res]} {
	return
    }
    
    if {$res == ""} return
    if {[regexp {([^(]+)\(([^)]+)\)} $res "" arr field]} {
	global $arr
	removeArrDef $arr $field
    } else {
	global $res
	removeDef $res
    }
    
    catch {unset prefDefs}
    catch {unset arrprefDefs}
}


proc getSavedSettings {} {
    global prefDefs arrprefDefs
    
    readDefs
    readDefs arr
    
    set names [array names prefDefs]
    foreach pair [array names arrprefDefs] {
	lappend names "[lindex $pair 0]([lindex $pair 1])"
    }
    
    return [lsort $names]
}

#===============================================================================

proc global::editPrefsFile {} {
    global PREFS
    if {![file exists [file join $PREFS prefs.tcl]]} {
	set fd [open [file join $PREFS prefs.tcl] "w"]
	close $fd
    }
    edit [file join $PREFS prefs.tcl]
}

# Automatically add a line to the user input file
proc addUserLine {line} {
    global PREFS
    
    if {![file exists "$PREFS"]} {
	file mkdir "$PREFS"
    }
    set fid [open [file join $PREFS prefs.tcl] "a+"]
    if {![catch {seek $fid -1 end}]} {
	if {[read $fid 1] == "\r"} {
	    set line "\r$line"
	}
    }
    seek $fid 0 end
    puts $fid $line
    close $fid
}

# Automatically add a line to a mode's pref file -trf
proc mode::addUserLine {line} {
    global PREFS mode
    
    if {![file exists "$PREFS"]} {
	file mkdir "$PREFS"
    }
    set fid [open [file join $PREFS ${mode}prefs.tcl] "a+"]
    if {![catch {seek $fid -1 end}]} {
	if {[read $fid 1] != "\r"} {
	    set line "\r$line"
	}
    }
    seek $fid 0 end
    puts $fid $line
    close $fid
}



#===============================================================================

## 
 # -------------------------------------------------------------------------
 # 
 # "mode::sourcePrefs" --
 # 
 #  Fixes 'uplevel #0' problem
 # -------------------------------------------------------------------------
 ##
proc mode::sourcePrefsFile {} { 
    global mode PREFS
    if {[file exists [file join ${PREFS} ${mode}Prefs.tcl]]} {
	uplevel #0 [list source [file join ${PREFS} ${mode}Prefs.tcl]]
    } else {
	beep; message "Sorry, no preferences for '$mode' mode"
    }
}

proc mode::editPrefsFile {{m ""}} { 
    global PREFS mode
    if {$m == ""} { set m $mode }
    message $m
    # assume it is a mode, since we made the menu
    
    set f [file join $PREFS ${m}Prefs.tcl]
    if {[file exists $f]} {
	edit $f
    } else {
	if {[dialog::yesno "No '$m' prefs file exists, do you want to create one?"]} {
	    close [open $f "w"]
	    edit $f
	    insertText {
## 
 # This	file will be sourced automatically, immediately after 
 # the _first_ time the file which defines its mode is sourced.
 # Use this file to insert your own mode-specific preferences
 # and changes,	rather than altering the originals.
 # 
 # You can redefine menus, procedures, variables,...
 ##

	}}}
	
	hook::callAll mode::editPrefsFile
    }

proc saveModifiedVars {} {
    global modifiedVars modifiedModeVars modifiedArrVars \
      mode::features prefDefs modifiedArrayElements global::features \
      alpha::earlyPrefs
    
    cache::delete configuration
    cache::add configuration list global::features
    
    if {[info exists alpha::earlyPrefs]} {
	foreach f [set alpha::earlyPrefs] {
	    global $f
	    if {[info exists $f]} {
		cache::add configuration variable $f
	    }
	}
    } else {
	set alpha::earlyPrefs {}
    }
    
    foreach f [lunique $modifiedArrVars] {
	addArr $f
    }
    foreach f [lunique $modifiedVars] {
	if {[lsearch -exact [set alpha::earlyPrefs] $f] == -1} {
	    global $f
	    if {[info exists $f]} {
		addDef $f [set $f]
	    } else {
		removeDef $f
	    }
	}
    }
    # these two lists actually behave identically
    foreach f [concat [lunique $modifiedArrayElements]  [lunique $modifiedModeVars]] {
	set elt [lindex $f 0]
	set arr [lindex $f 1]
	global $arr
	if {[info exists [set arr]($elt)]} {
	    addArrDef [set arr] $elt [set [set arr]($elt)]
	} else {
	    removeArrDef [set arr] $elt
	}
    }
    
    set modifiedVars {}
    set modifiedArrVars {}
    set modifiedModeVars {}
    set modifiedArrayElements {}
}
