/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.util.DiskChecker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class LedgerDirsMonitor {
    private static final Logger LOG = LoggerFactory.getLogger(LedgerDirsMonitor.class);
    private final int interval;
    private final ServerConfiguration conf;
    private final DiskChecker diskChecker;
    private final List<LedgerDirsManager> dirsManagers;
    private long minUsableSizeForHighPriorityWrites;
    private ScheduledExecutorService executor;
    private ScheduledFuture<?> checkTask;

    public LedgerDirsMonitor(ServerConfiguration conf, DiskChecker diskChecker, List<LedgerDirsManager> dirsManagers) {
        this.interval = conf.getDiskCheckInterval();
        this.minUsableSizeForHighPriorityWrites = conf.getMinUsableSizeForHighPriorityWrites();
        this.conf = conf;
        this.diskChecker = diskChecker;
        this.dirsManagers = dirsManagers;
    }

    private void check(LedgerDirsManager ldm) {
        ConcurrentMap<File, Float> diskUsages = ldm.getDiskUsages();
        try {
            List<File> writableDirs = ldm.getWritableLedgerDirs();
            for (File dir : writableDirs) {
                try {
                    diskUsages.put(dir, Float.valueOf(this.diskChecker.checkDir(dir)));
                }
                catch (DiskChecker.DiskErrorException e) {
                    LOG.error("Ledger directory {} failed on disk checking : ", (Object)dir, (Object)e);
                    for (LedgerDirsManager.LedgerDirsListener listener : ldm.getListeners()) {
                        listener.diskFailed(dir);
                    }
                }
                catch (DiskChecker.DiskWarnThresholdException e) {
                    diskUsages.compute(dir, (d, prevUsage) -> {
                        if (null == prevUsage || e.getUsage() != prevUsage.floatValue()) {
                            LOG.warn("Ledger directory {} is almost full : usage {}", (Object)dir, (Object)Float.valueOf(e.getUsage()));
                        }
                        return Float.valueOf(e.getUsage());
                    });
                    for (LedgerDirsManager.LedgerDirsListener listener : ldm.getListeners()) {
                        listener.diskAlmostFull(dir);
                    }
                }
                catch (DiskChecker.DiskOutOfSpaceException e) {
                    diskUsages.compute(dir, (d, prevUsage) -> {
                        if (null == prevUsage || e.getUsage() != prevUsage.floatValue()) {
                            LOG.error("Ledger directory {} is out-of-space : usage {}", (Object)dir, (Object)Float.valueOf(e.getUsage()));
                        }
                        return Float.valueOf(e.getUsage());
                    });
                    ldm.addToFilledDirs(dir);
                }
            }
            ldm.getWritableLedgerDirs();
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException e) {
            LOG.warn("LedgerDirsMonitor check process: All ledger directories are non writable");
            boolean highPriorityWritesAllowed = true;
            try {
                ldm.getDirsAboveUsableThresholdSize(this.minUsableSizeForHighPriorityWrites, false);
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException e1) {
                highPriorityWritesAllowed = false;
            }
            for (LedgerDirsManager.LedgerDirsListener listener : ldm.getListeners()) {
                listener.allDisksFull(highPriorityWritesAllowed);
            }
        }
        ArrayList<File> fullfilledDirs = new ArrayList<File>(ldm.getFullFilledLedgerDirs());
        boolean makeWritable = ldm.hasWritableLedgerDirs();
        try {
            if (!makeWritable) {
                float totalDiskUsage = this.diskChecker.getTotalDiskUsage(ldm.getAllLedgerDirs());
                if (totalDiskUsage < this.conf.getDiskLowWaterMarkUsageThreshold()) {
                    makeWritable = true;
                } else {
                    LOG.debug("Current TotalDiskUsage: {} is greater than LWMThreshold: {}. So not adding any filledDir to WritableDirsList", (Object)Float.valueOf(totalDiskUsage), (Object)Float.valueOf(this.conf.getDiskLowWaterMarkUsageThreshold()));
                }
            }
            for (File dir : fullfilledDirs) {
                try {
                    diskUsages.put(dir, Float.valueOf(this.diskChecker.checkDir(dir)));
                    if (!makeWritable) continue;
                    ldm.addToWritableDirs(dir, true);
                }
                catch (DiskChecker.DiskErrorException e) {
                    for (LedgerDirsManager.LedgerDirsListener listener : ldm.getListeners()) {
                        listener.diskFailed(dir);
                    }
                }
                catch (DiskChecker.DiskWarnThresholdException e) {
                    diskUsages.put(dir, Float.valueOf(e.getUsage()));
                    if (!makeWritable) continue;
                    ldm.addToWritableDirs(dir, false);
                }
                catch (DiskChecker.DiskOutOfSpaceException e) {
                    diskUsages.put(dir, Float.valueOf(e.getUsage()));
                }
            }
        }
        catch (IOException ioe) {
            LOG.error("Got IOException while monitoring Dirs", (Throwable)ioe);
            for (LedgerDirsManager.LedgerDirsListener listener : ldm.getListeners()) {
                listener.fatalError();
            }
        }
    }

    private void check() {
        this.dirsManagers.forEach(this::check);
    }

    public void init() throws DiskChecker.DiskErrorException, LedgerDirsManager.NoWritableLedgerDirException {
        this.checkDirs();
    }

    public void start() {
        this.executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("LedgerDirsMonitorThread").setDaemon(true).build());
        this.checkTask = this.executor.scheduleAtFixedRate(this::check, this.interval, this.interval, TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        LOG.info("Shutting down LedgerDirsMonitor");
        if (null != this.checkTask && this.checkTask.cancel(true)) {
            LOG.debug("Failed to cancel check task in LedgerDirsMonitor");
        }
        if (null != this.executor) {
            this.executor.shutdown();
        }
    }

    private void checkDirs() throws LedgerDirsManager.NoWritableLedgerDirException, DiskChecker.DiskErrorException {
        for (LedgerDirsManager dirsManager : this.dirsManagers) {
            this.checkDirs(dirsManager);
        }
    }

    private void checkDirs(LedgerDirsManager ldm) throws DiskChecker.DiskErrorException, LedgerDirsManager.NoWritableLedgerDirException {
        for (File dir : ldm.getWritableLedgerDirs()) {
            try {
                this.diskChecker.checkDir(dir);
            }
            catch (DiskChecker.DiskWarnThresholdException diskWarnThresholdException) {
            }
            catch (DiskChecker.DiskOutOfSpaceException e) {
                ldm.addToFilledDirs(dir);
            }
        }
        ldm.getWritableLedgerDirs();
    }
}

