/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.frame;

import com.google.common.primitives.Ints;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.WritableByteChannel;
import javax.annotation.Nullable;
import net.jpountz.lz4.LZ4Compressor;
import net.jpountz.lz4.LZ4Factory;
import net.jpountz.lz4.LZ4SafeDecompressor;
import org.apache.datasketches.memory.Memory;
import org.apache.datasketches.memory.WritableMemory;
import org.apache.druid.frame.FrameType;
import org.apache.druid.frame.channel.ByteTracker;
import org.apache.druid.io.Channels;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.segment.data.CompressionStrategy;

public class Frame {
    public static final long HEADER_SIZE = 18L;
    public static final int COMPRESSED_FRAME_HEADER_SIZE = 17;
    public static final int COMPRESSED_FRAME_TRAILER_SIZE = 8;
    public static final int COMPRESSED_FRAME_ENVELOPE_SIZE = 25;
    private static final LZ4Compressor LZ4_COMPRESSOR = LZ4Factory.fastestInstance().fastCompressor();
    private static final LZ4SafeDecompressor LZ4_DECOMPRESSOR = LZ4Factory.fastestInstance().safeDecompressor();
    private static final int CHECKSUM_SEED = 0;
    private final Memory memory;
    private final FrameType frameType;
    private final long numBytes;
    private final int numRows;
    private final int numRegions;
    private final boolean permuted;

    private Frame(Memory memory, FrameType frameType, long numBytes, int numRows, int numRegions, boolean permuted) {
        this.memory = memory;
        this.frameType = frameType;
        this.numBytes = numBytes;
        this.numRows = numRows;
        this.numRegions = numRegions;
        this.permuted = permuted;
    }

    public static Frame wrap(Memory memory) {
        if (memory.getTypeByteOrder() != ByteOrder.LITTLE_ENDIAN) {
            throw new IAE("Memory must be little-endian", new Object[0]);
        }
        if (memory.getCapacity() < 18L) {
            throw new IAE("Memory too short for a header", new Object[0]);
        }
        byte version = memory.getByte(0L);
        FrameType frameType = FrameType.forVersion(version);
        if (frameType == null) {
            throw new IAE("Unexpected byte [%s] at start of frame", version);
        }
        long numBytes = memory.getLong(1L);
        int numRows = memory.getInt(9L);
        int numRegions = memory.getInt(13L);
        boolean permuted = memory.getByte(17L) != 0;
        Frame.validate(memory, numBytes, numRows, numRegions, permuted);
        return new Frame(memory, frameType, numBytes, numRows, numRegions, permuted);
    }

    public static Frame wrap(ByteBuffer buffer) {
        return Frame.wrap(Memory.wrap((ByteBuffer)buffer, (ByteOrder)ByteOrder.LITTLE_ENDIAN));
    }

    public static Frame wrap(byte[] bytes) {
        return Frame.wrap(ByteBuffer.wrap(bytes));
    }

    public static Frame decompress(Memory memory, long position, long length) {
        long actualChecksum;
        if (memory.getCapacity() < position + length) {
            throw new ISE("Provided position, length is out of bounds", new Object[0]);
        }
        if (length < 25L) {
            throw new ISE("Region too short", new Object[0]);
        }
        long expectedChecksum = memory.getLong(position + length - 8L);
        if (expectedChecksum != (actualChecksum = memory.xxHash64(position, length - 8L, 0L))) {
            throw new ISE("Checksum mismatch", new Object[0]);
        }
        byte compressionTypeId = memory.getByte(position);
        CompressionStrategy compressionStrategy = CompressionStrategy.forId(compressionTypeId);
        if (compressionStrategy != CompressionStrategy.LZ4) {
            throw new ISE("Unsupported compression strategy [%s]", new Object[]{compressionStrategy});
        }
        int compressedFrameLength = Ints.checkedCast((long)memory.getLong(position + 1L));
        int uncompressedFrameLength = Ints.checkedCast((long)memory.getLong(position + 1L + 8L));
        int compressedFrameLengthFromRegionLength = Ints.checkedCast((long)(length - 25L));
        long frameStart = position + 17L;
        if (compressedFrameLength != compressedFrameLengthFromRegionLength) {
            throw new ISE("Compressed sizes disagree: [%d] (embedded) vs [%d] (region length)", compressedFrameLength, compressedFrameLengthFromRegionLength);
        }
        if (memory.hasByteBuffer()) {
            ByteBuffer srcBuffer = memory.getByteBuffer();
            ByteBuffer dstBuffer = ByteBuffer.allocate(uncompressedFrameLength);
            int numBytesDecompressed = LZ4_DECOMPRESSOR.decompress(srcBuffer, Ints.checkedCast((long)(memory.getRegionOffset() + frameStart)), compressedFrameLength, dstBuffer, 0, uncompressedFrameLength);
            if (numBytesDecompressed != uncompressedFrameLength) {
                throw new ISE("Expected to decompress [%d] bytes but got [%d] bytes", uncompressedFrameLength, numBytesDecompressed);
            }
            return Frame.wrap(dstBuffer);
        }
        byte[] compressedFrame = new byte[compressedFrameLength];
        memory.getByteArray(frameStart, compressedFrame, 0, compressedFrameLength);
        return Frame.wrap(LZ4_DECOMPRESSOR.decompress(compressedFrame, uncompressedFrameLength));
    }

    public static int compressionBufferSize(long frameBytes) {
        return 25 + LZ4_COMPRESSOR.maxCompressedLength(Ints.checkedCast((long)frameBytes));
    }

    public FrameType type() {
        return this.frameType;
    }

    public long numBytes() {
        return this.numBytes;
    }

    public int numRows() {
        return this.numRows;
    }

    public int numRegions() {
        return this.numRegions;
    }

    public boolean isPermuted() {
        return this.permuted;
    }

    public int physicalRow(int logicalRow) {
        if (logicalRow < 0 || logicalRow >= this.numRows) {
            throw new IAE("Row [%,d] out of bounds", logicalRow);
        }
        if (this.permuted) {
            int rowPosition = this.memory.getInt(18L + 4L * (long)logicalRow);
            if (rowPosition < 0 || rowPosition >= this.numRows) {
                throw new ISE("Invalid physical row position for logical row [%,d]. Corrupt frame?", logicalRow);
            }
            return rowPosition;
        }
        return logicalRow;
    }

    public Memory region(int regionNumber) {
        if (regionNumber < 0 || regionNumber >= this.numRegions) {
            throw new IAE("Region [%,d] out of bounds", regionNumber);
        }
        long regionEndPositionSectionStart = 18L + (this.permuted ? (long)this.numRows * 4L : 0L);
        long regionEndPosition = this.memory.getLong(regionEndPositionSectionStart + (long)regionNumber * 8L);
        long regionStartPosition = regionNumber == 0 ? regionEndPositionSectionStart + (long)this.numRegions * 8L : this.memory.getLong(regionEndPositionSectionStart + (long)(regionNumber - 1) * 8L);
        return this.memory.region(regionStartPosition, regionEndPosition - regionStartPosition);
    }

    public WritableMemory writableMemory() {
        if (this.memory instanceof WritableMemory) {
            return (WritableMemory)this.memory;
        }
        throw new ISE("Frame memory is not writable", new Object[0]);
    }

    public long writeTo(WritableByteChannel channel, boolean compress, @Nullable ByteBuffer compressionBuffer, ByteTracker byteTracker) throws IOException {
        if (compress) {
            int frameBufferPosition;
            ByteBuffer frameBuffer;
            if (compressionBuffer == null) {
                throw new NullPointerException("compression buffer");
            }
            if (compressionBuffer.capacity() < Frame.compressionBufferSize(this.numBytes)) {
                throw new ISE("Compression buffer too small: expected [%,d] bytes but got [%,d] bytes", Frame.compressionBufferSize(this.numBytes), compressionBuffer.capacity());
            }
            if (this.memory.hasByteBuffer()) {
                frameBuffer = this.memory.getByteBuffer();
                frameBufferPosition = Ints.checkedCast((long)this.memory.getRegionOffset());
            } else {
                byte[] frameBytes = new byte[Ints.checkedCast((long)this.numBytes)];
                this.memory.getByteArray(0L, frameBytes, 0, Ints.checkedCast((long)this.numBytes));
                frameBuffer = ByteBuffer.wrap(frameBytes);
                frameBufferPosition = 0;
            }
            int compressedFrameLength = LZ4_COMPRESSOR.compress(frameBuffer, frameBufferPosition, Ints.checkedCast((long)this.numBytes), compressionBuffer, 17, compressionBuffer.capacity() - 25);
            compressionBuffer.order(ByteOrder.LITTLE_ENDIAN).limit(25 + compressedFrameLength).position(0);
            compressionBuffer.put(0, CompressionStrategy.LZ4.getId()).putLong(1, compressedFrameLength).putLong(9, this.numBytes);
            long checksum = Memory.wrap((ByteBuffer)compressionBuffer).xxHash64(0L, (long)(17 + compressedFrameLength), 0L);
            compressionBuffer.putLong(17 + compressedFrameLength, checksum);
            byteTracker.reserve(compressionBuffer.remaining());
            Channels.writeFully(channel, compressionBuffer);
            return 25 + compressedFrameLength;
        }
        byteTracker.reserve(this.numBytes);
        this.memory.writeTo(0L, this.numBytes, channel);
        return this.numBytes;
    }

    private static void validate(Memory memory, long numBytes, int numRows, int numRegions, boolean permuted) {
        long regionEndSize;
        if (numBytes != memory.getCapacity()) {
            throw new IAE("Declared size [%,d] does not match actual size [%,d]", numBytes, memory.getCapacity());
        }
        long rowOrderSize = permuted ? (long)numRows * 4L : 0L;
        long expectedSizeForPreamble = 18L + rowOrderSize + (regionEndSize = (long)numRegions * 8L);
        if (numBytes < expectedSizeForPreamble) {
            throw new IAE("Memory too short for preamble", new Object[0]);
        }
        long regionStart = expectedSizeForPreamble;
        for (int regionNumber = 0; regionNumber < numRegions; ++regionNumber) {
            long regionEnd = memory.getLong(18L + rowOrderSize + (long)regionNumber * 8L);
            if (regionEnd < regionStart || regionEnd > numBytes) {
                throw new IAE("Region [%d] invalid: end [%,d] out of range [%,d -> %,d]", regionNumber, regionEnd, expectedSizeForPreamble, numBytes);
            }
            if (regionNumber > 0) {
                regionStart = memory.getLong(18L + rowOrderSize + (long)(regionNumber - 1) * 8L);
            }
            if (regionStart >= expectedSizeForPreamble && regionStart <= numBytes) continue;
            throw new IAE("Region [%d] invalid: start [%,d] out of range [%,d -> %,d]", regionNumber, regionStart, expectedSizeForPreamble, numBytes);
        }
    }
}

