/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.catalog.http;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import org.apache.curator.shaded.com.google.common.collect.Lists;
import org.apache.druid.catalog.CatalogException;
import org.apache.druid.catalog.http.TableEditRequest;
import org.apache.druid.catalog.http.TableEditor;
import org.apache.druid.catalog.model.SchemaRegistry;
import org.apache.druid.catalog.model.TableId;
import org.apache.druid.catalog.model.TableMetadata;
import org.apache.druid.catalog.model.TableSpec;
import org.apache.druid.catalog.storage.CatalogStorage;
import org.apache.druid.common.utils.IdUtils;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.Pair;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.server.security.Access;
import org.apache.druid.server.security.Action;
import org.apache.druid.server.security.AuthorizationUtils;
import org.apache.druid.server.security.AuthorizerMapper;
import org.apache.druid.server.security.ForbiddenException;
import org.apache.druid.server.security.Resource;
import org.apache.druid.server.security.ResourceAction;

@Path(value="/druid/coordinator/v1/catalog")
public class CatalogResource {
    public static final String ROOT_PATH = "/druid/coordinator/v1/catalog";
    public static final String NAME_FORMAT = "name";
    public static final String PATH_FORMAT = "path";
    public static final String METADATA_FORMAT = "metadata";
    public static final String STATUS_FORMAT = "status";
    private final CatalogStorage catalog;
    private final AuthorizerMapper authorizerMapper;
    public static final String SCHEMA_SYNC = "/sync/schemas/{schema}";
    public static final String TABLE_SYNC = "/sync/schemas/{schema}/{name}";

    @Inject
    public CatalogResource(CatalogStorage catalog, AuthorizerMapper authorizerMapper) {
        this.catalog = catalog;
        this.authorizerMapper = authorizerMapper;
    }

    @POST
    @Path(value="/schemas/{schema}/tables/{name}")
    @Consumes(value={"application/json"})
    @Produces(value={"application/json"})
    public Response postTable(@PathParam(value="schema") String schemaName, @PathParam(value="name") String tableName, TableSpec spec, @QueryParam(value="version") long version, @QueryParam(value="overwrite") boolean overwrite, @Context HttpServletRequest req) {
        try {
            long newVersion;
            SchemaRegistry.SchemaSpec schema = this.validateSchema(schemaName, true);
            this.validateTableName(tableName);
            this.authorizeTable(schema, tableName, Action.WRITE, req);
            this.validateTableSpec(schema, spec);
            TableMetadata table = TableMetadata.newTable((TableId)TableId.of((String)schemaName, (String)tableName), (TableSpec)spec);
            try {
                this.catalog.validate(table);
            }
            catch (IAE e) {
                throw CatalogException.badRequest(e.getMessage(), new Object[0]);
            }
            if (version != 0L) {
                newVersion = this.catalog.tables().update(table, version);
            } else {
                try {
                    newVersion = this.catalog.tables().create(table);
                }
                catch (CatalogException.DuplicateKeyException e) {
                    if (overwrite) {
                        newVersion = this.catalog.tables().replace(table);
                    }
                    throw e;
                }
            }
            return CatalogResource.okWithVersion(newVersion);
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @GET
    @Path(value="/schemas/{schema}/tables/{name}")
    @Produces(value={"application/json"})
    public Response getTable(@PathParam(value="schema") String schemaName, @PathParam(value="name") String tableName, @Context HttpServletRequest req) {
        try {
            SchemaRegistry.SchemaSpec schema = this.validateSchema(schemaName, false);
            this.authorizeTable(schema, tableName, Action.READ, req);
            TableMetadata table = this.catalog.tables().read(new TableId(schemaName, tableName));
            return Response.ok().entity((Object)table).build();
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @DELETE
    @Path(value="/schemas/{schema}/tables/{name}")
    @Produces(value={"application/json"})
    public Response deleteTable(@PathParam(value="schema") String schemaName, @PathParam(value="name") String tableName, @Context HttpServletRequest req) {
        try {
            SchemaRegistry.SchemaSpec schema = this.validateSchema(schemaName, true);
            this.authorizeTable(schema, tableName, Action.WRITE, req);
            this.catalog.tables().delete(new TableId(schemaName, tableName));
            return CatalogResource.ok();
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @POST
    @Path(value="/schemas/{schema}/tables/{name}/edit")
    @Consumes(value={"application/json"})
    @Produces(value={"application/json"})
    public Response editTable(@PathParam(value="schema") String schemaName, @PathParam(value="name") String tableName, TableEditRequest editRequest, @Context HttpServletRequest req) {
        try {
            SchemaRegistry.SchemaSpec schema = this.validateSchema(schemaName, true);
            this.authorizeTable(schema, tableName, Action.WRITE, req);
            long newVersion = new TableEditor(this.catalog, TableId.of((String)schemaName, (String)tableName), editRequest).go();
            return CatalogResource.okWithVersion(newVersion);
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @GET
    @Path(value="/schemas")
    @Produces(value={"application/json"})
    public Response getSchemas(@QueryParam(value="format") String format, @Context HttpServletRequest req) {
        try {
            switch (format = Strings.isNullOrEmpty((String)format) ? NAME_FORMAT : StringUtils.toLowerCase((String)format)) {
                case "name": {
                    this.authorizeAccess("STATE", "schemas", Action.READ, req);
                    return Response.ok().entity((Object)this.catalog.schemaRegistry().names()).build();
                }
                case "path": {
                    return this.listTablePaths(req);
                }
                case "metadata": {
                    return this.listAllTableMetadata(req);
                }
            }
            throw CatalogException.badRequest("Unknown format: [%s]", format);
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @GET
    @Path(value="/schemas/{schema}/tables")
    @Produces(value={"application/json"})
    public Response getSchemaTables(@PathParam(value="schema") String schemaName, @QueryParam(value="format") String format, @Context HttpServletRequest req) {
        try {
            SchemaRegistry.SchemaSpec schema = this.validateSchema(schemaName, false);
            switch (format = Strings.isNullOrEmpty((String)format) ? NAME_FORMAT : StringUtils.toLowerCase((String)format)) {
                case "name": {
                    return this.tableNamesInSchema(schema, req);
                }
                case "metadata": {
                    return Response.ok().entity(this.getTableMetadataForSchema(schema, req)).build();
                }
                case "status": {
                    return Response.ok().entity(this.getTableStatusForSchema(schema, req)).build();
                }
            }
            throw CatalogException.badRequest("Unknown format: [%s]", format);
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @GET
    @Path(value="/sync/schemas/{schema}")
    @Produces(value={"application/json"})
    public Response syncSchema(@PathParam(value="schema") String schemaName, @Context HttpServletRequest req) {
        try {
            SchemaRegistry.SchemaSpec schema = this.validateSchema(schemaName, false);
            return Response.ok().entity(this.getTableMetadataForSchema(schema, req)).build();
        }
        catch (CatalogException e) {
            return e.toResponse();
        }
    }

    @GET
    @Path(value="/sync/schemas/{schema}/{name}")
    @Produces(value={"application/json"})
    public Response syncTable(@PathParam(value="schema") String schemaName, @PathParam(value="name") String tableName, @Context HttpServletRequest req) {
        return this.getTable(schemaName, tableName, req);
    }

    private Response listTablePaths(HttpServletRequest req) {
        List<TableId> tables = this.catalog.tables().allTablePaths();
        Iterable filtered = AuthorizationUtils.filterAuthorizedResources((HttpServletRequest)req, tables, tableId -> {
            SchemaRegistry.SchemaSpec schema = this.catalog.resolveSchema(tableId.schema());
            if (schema == null) {
                return null;
            }
            return Collections.singletonList(CatalogResource.resourceAction(schema, tableId.name(), Action.READ));
        }, (AuthorizerMapper)this.authorizerMapper);
        return Response.ok().entity((Object)Lists.newArrayList((Iterable)filtered)).build();
    }

    private List<TableMetadata> getTableMetadataForSchema(SchemaRegistry.SchemaSpec schema, HttpServletRequest req) {
        List<TableMetadata> tables = this.catalog.tables().tablesInSchema(schema.name());
        Iterable filtered = AuthorizationUtils.filterAuthorizedResources((HttpServletRequest)req, tables, table -> {
            TableId tableId = table.id();
            return Collections.singletonList(CatalogResource.resourceAction(schema, tableId.name(), Action.READ));
        }, (AuthorizerMapper)this.authorizerMapper);
        return Lists.newArrayList((Iterable)filtered);
    }

    private List<TableMetadata> getTableStatusForSchema(SchemaRegistry.SchemaSpec schema, HttpServletRequest req) {
        return this.getTableMetadataForSchema(schema, req).stream().map(table -> table.withSpec(new TableSpec(table.spec().type(), null, null))).collect(Collectors.toList());
    }

    private Response listAllTableMetadata(HttpServletRequest req) {
        ArrayList tables = new ArrayList();
        for (SchemaRegistry.SchemaSpec schema : this.catalog.schemaRegistry().schemas()) {
            tables.addAll(this.catalog.tables().tablesInSchema(schema.name()).stream().map(table -> Pair.of((Object)schema, (Object)table)).collect(Collectors.toList()));
        }
        Iterable filtered = AuthorizationUtils.filterAuthorizedResources((HttpServletRequest)req, tables, entry -> Collections.singletonList(CatalogResource.resourceAction((SchemaRegistry.SchemaSpec)entry.lhs, ((TableMetadata)entry.rhs).id().name(), Action.READ)), (AuthorizerMapper)this.authorizerMapper);
        List metadata = Lists.newArrayList((Iterable)filtered).stream().map(pair -> (TableMetadata)pair.rhs).collect(Collectors.toList());
        return Response.ok().entity(metadata).build();
    }

    private Response tableNamesInSchema(SchemaRegistry.SchemaSpec schema, HttpServletRequest req) {
        List<String> tables = this.catalog.tables().tableNamesInSchema(schema.name());
        Iterable filtered = AuthorizationUtils.filterAuthorizedResources((HttpServletRequest)req, tables, name -> Collections.singletonList(CatalogResource.resourceAction(schema, name, Action.READ)), (AuthorizerMapper)this.authorizerMapper);
        return Response.ok().entity((Object)Lists.newArrayList((Iterable)filtered)).build();
    }

    private void validateTableName(String name) throws CatalogException {
        try {
            IdUtils.validateId((String)"table", (String)name);
        }
        catch (Exception e) {
            throw CatalogException.badRequest(e.getMessage(), new Object[0]);
        }
        if (!name.equals(name.trim())) {
            throw CatalogException.badRequest("Table name cannot start or end with spaces", new Object[0]);
        }
    }

    private void validateTableSpec(SchemaRegistry.SchemaSpec schema, TableSpec spec) throws CatalogException {
        try {
            spec.validate();
        }
        catch (IAE e) {
            throw CatalogException.badRequest(e.getMessage(), new Object[0]);
        }
        if (!schema.accepts(spec.type())) {
            throw CatalogException.badRequest("Cannot create tables of type %s in schema %s", spec.type(), schema.name());
        }
    }

    private SchemaRegistry.SchemaSpec validateSchema(String schemaName, boolean forWrite) throws CatalogException {
        if (Strings.isNullOrEmpty((String)schemaName)) {
            throw CatalogException.badRequest("Schema name is required", new Object[0]);
        }
        SchemaRegistry.SchemaSpec schema = this.catalog.resolveSchema(schemaName);
        if (schema == null) {
            throw new CatalogException.NotFoundException("Unknown schema %s", schemaName);
        }
        if (forWrite && !schema.writable()) {
            throw CatalogException.badRequest("Cannot modify schema %s", schemaName);
        }
        return schema;
    }

    private static ResourceAction resourceAction(SchemaRegistry.SchemaSpec schema, String tableName, Action action) {
        return new ResourceAction(new Resource(tableName, schema.securityResource()), action);
    }

    private void authorizeTable(SchemaRegistry.SchemaSpec schema, String tableName, Action action, HttpServletRequest request) throws CatalogException {
        if (Strings.isNullOrEmpty((String)tableName)) {
            throw CatalogException.badRequest("Table name is required", new Object[0]);
        }
        if (action == Action.WRITE && !schema.writable()) {
            throw new ForbiddenException("Cannot create table definitions in schema: " + schema.name());
        }
        this.authorize(schema.securityResource(), tableName, action, request);
    }

    private void authorize(String resource, String key, Action action, HttpServletRequest request) {
        Access authResult = this.authorizeAccess(resource, key, action, request);
        if (!authResult.isAllowed()) {
            throw new ForbiddenException(authResult.toString());
        }
    }

    private Access authorizeAccess(String resource, String key, Action action, HttpServletRequest request) {
        return AuthorizationUtils.authorizeResourceAction((HttpServletRequest)request, (ResourceAction)new ResourceAction(new Resource(key, resource), action), (AuthorizerMapper)this.authorizerMapper);
    }

    private static Response okWithVersion(long version) {
        return Response.ok().entity((Object)ImmutableMap.of((Object)"version", (Object)version)).build();
    }

    private static Response ok() {
        return Response.ok().build();
    }
}

