#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.
#

##
## @fn 010-generate-guacamole-home.sh
##
## Automatically generates a temporary, skeleton GUACAMOLE_HOME to be used for
## this run of the container. GUACAMOLE_HOMEs from previous runs are
## automatically deleted prior to creating the new skeleton. A
## randomly-generated temporary directory is used instead of a standard
## directory like "/etc/guacamole" to allow users to use "/etc/guacamole" as a
## basis for their own configuration.
##

##
## The directory to copy/link over as a basis for the GUACAMOLE_HOME actually
## used by the Guacamole web application. Any configuration generated by this
## container will be overlaid on top of this configuration. To achieve the
## overlay, symbolic links will be created for all files inside and beneath
## this directory. Only the guacamole.properties file will be copied instead of
## using symbolic links (to ensure property generation performed by the
## container does not potentially modify an external file).
##
GUACAMOLE_HOME_TEMPLATE="$GUACAMOLE_HOME"

##
## Tests whether a given property is set within the guacamole.properties file
## in GUACAMOLE_HOME.
##
## @param PROPERTY_NAME
##     The name of the property to check.
##
## @returns
##     Zero if the given property is set to any value within
##     guacamole.properties, non-zero otherwise.
##
is_property_set() {
    local PROPERTY_NAME="$1"
    grep "^[[:space:]]*$PROPERTY_NAME\>" "$GUACAMOLE_HOME/guacamole.properties" &> /dev/null
}

#
# Start with a fresh GUACAMOLE_HOME
#

rm -rf /tmp/guacamole-home.*
GUACAMOLE_HOME="`mktemp -p /tmp -d guacamole-home.XXXXXXXXXX`"
mkdir -p "$GUACAMOLE_HOME/"{lib,extensions}

cat > "$GUACAMOLE_HOME/guacamole.properties" <<EOF
# guacamole.properties - generated `date`
EOF

#
# Copy contents of provided GUACAMOLE_HOME template, if any
#

if [ -e "$GUACAMOLE_HOME_TEMPLATE" ]; then

    # Create links for any libraries provided in the template GUACAMOLE_HOME
    if [ -e "$GUACAMOLE_HOME_TEMPLATE/lib" ]; then
        find "$GUACAMOLE_HOME_TEMPLATE/lib" -mindepth 1 -maxdepth 1 \
            -exec ln -sv "{}" "$GUACAMOLE_HOME/lib/" ";"
    fi

    # Create links for any extensions provided in the template GUACAMOLE_HOME
    if [ -e "$GUACAMOLE_HOME_TEMPLATE/extensions" ]; then
        find "$GUACAMOLE_HOME_TEMPLATE/extensions" -mindepth 1 -maxdepth 1 \
            -exec ln -sv "{}" "$GUACAMOLE_HOME/extensions/" ";"
    fi

    # Create links for all other files directly within the template
    # GUACAMOLE_HOME
    find "$GUACAMOLE_HOME_TEMPLATE" -mindepth 1 -maxdepth 1 \
        -name guacamole.properties -o -name lib -o -name extensions -prune \
        -o -exec ln -sv "{}" "$GUACAMOLE_HOME/" ";"

    # Add any properties provided within template GUACAMOLE_HOME
    if [ -e "$GUACAMOLE_HOME_TEMPLATE/guacamole.properties" ]; then
        cat "$GUACAMOLE_HOME_TEMPLATE/guacamole.properties" >> "$GUACAMOLE_HOME/guacamole.properties"
    fi

fi

# Enable reading of properties directly from environment variables unless
# overridden
if ! is_property_set "enable-environment-properties"; then
    cat >> "$GUACAMOLE_HOME/guacamole.properties" <<'EOF'
#
# NOTE: The following was automatically added by the container entrypoint to
# allow all Guacamole configuration properties to be automatically read from
# environment variables. If this is not desired, you can override this behavior
# by specifying the "enable-environment-properties" variable yourself in your
# own guacamole.properties file.
#
enable-environment-properties: true
EOF
fi

