/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.StripedFileTestUtil;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicy;
import org.apache.hadoop.shaded.com.google.common.base.Preconditions;
import org.apache.hadoop.util.StopWatch;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

public class ErasureCodeBenchmarkThroughput
extends Configured
implements Tool {
    private static final int BUFFER_SIZE_MB = 128;
    private static final String DFS_TMP_DIR = System.getProperty("test.benchmark.data", "/tmp/benchmark/data");
    public static final String REP_DIR = DFS_TMP_DIR + "/replica";
    public static final String EC_DIR = DFS_TMP_DIR + "/ec";
    private static final String REP_FILE_BASE = "rep-file-";
    private static final String EC_FILE_BASE = "ec-file-";
    private static final String TMP_FILE_SUFFIX = ".tmp";
    private static final ErasureCodingPolicy ecPolicy = StripedFileTestUtil.getDefaultECPolicy();
    private static final byte[] data = new byte[0x8000000];
    private final FileSystem fs;

    public static ErasureCodingPolicy getEcPolicy() {
        return ecPolicy;
    }

    public ErasureCodeBenchmarkThroughput(FileSystem fs) {
        Preconditions.checkArgument((boolean)(fs instanceof DistributedFileSystem));
        this.fs = fs;
    }

    public static String getFilePath(int dataSizeMB, boolean isEc) {
        String parent = isEc ? EC_DIR : REP_DIR;
        String file = isEc ? EC_FILE_BASE : REP_FILE_BASE;
        return parent + "/" + file + dataSizeMB + "MB";
    }

    private static void printUsage(String msg) {
        if (msg != null) {
            System.out.println(msg);
        }
        System.err.println("Usage: ErasureCodeBenchmarkThroughput <read|write|gen|clean> <size in MB> <ec|rep> [num clients] [stf|pos]\nStateful and positional option is only available for read.");
        System.exit(1);
    }

    private List<Long> doBenchmark(boolean isRead, int dataSizeMB, int numClients, boolean isEc, boolean statefulRead, boolean isGen) throws Exception {
        ExecutorCompletionService<Long> cs = new ExecutorCompletionService<Long>(Executors.newFixedThreadPool(numClients));
        for (int i = 0; i < numClients; ++i) {
            cs.submit(isRead ? new ReadCallable(dataSizeMB, isEc, i, statefulRead) : new WriteCallable(dataSizeMB, isEc, i, isGen));
        }
        ArrayList<Long> results = new ArrayList<Long>(numClients);
        for (int i = 0; i < numClients; ++i) {
            results.add((Long)cs.take().get());
        }
        return results;
    }

    private void setReadThreadPoolSize(int numClients) {
        int numThread = numClients * ecPolicy.getNumDataUnits();
        this.getConf().setInt("dfs.client.read.striped.threadpool.size", numThread);
    }

    private DecimalFormat getDecimalFormat() {
        return new DecimalFormat("#.##");
    }

    private void benchmark(OpType type, int dataSizeMB, int numClients, boolean isEc, boolean statefulRead) throws Exception {
        List<Long> sizes = null;
        StopWatch sw = new StopWatch().start();
        switch (type) {
            case READ: {
                sizes = this.doBenchmark(true, dataSizeMB, numClients, isEc, statefulRead, false);
                break;
            }
            case WRITE: {
                sizes = this.doBenchmark(false, dataSizeMB, numClients, isEc, statefulRead, false);
                break;
            }
            case GEN: {
                sizes = this.doBenchmark(false, dataSizeMB, numClients, isEc, statefulRead, true);
            }
        }
        long elapsedSec = sw.now(TimeUnit.SECONDS);
        double totalDataSizeMB = 0.0;
        for (Long size : sizes) {
            if (size < 0L) continue;
            totalDataSizeMB += size.doubleValue() / 1024.0 / 1024.0;
        }
        double throughput = totalDataSizeMB / (double)elapsedSec;
        DecimalFormat df = this.getDecimalFormat();
        System.out.println((Object)((Object)type) + " " + df.format(totalDataSizeMB) + " MB data takes: " + elapsedSec + " s.\nTotal throughput: " + df.format(throughput) + " MB/s.");
    }

    private void setUpDir() throws IOException {
        DistributedFileSystem dfs = (DistributedFileSystem)this.fs;
        dfs.mkdirs(new Path(DFS_TMP_DIR));
        Path repPath = new Path(REP_DIR);
        Path ecPath = new Path(EC_DIR);
        if (!dfs.exists(repPath)) {
            dfs.mkdirs(repPath);
        } else {
            Preconditions.checkArgument((dfs.getClient().getErasureCodingPolicy(repPath.toString()) == null ? 1 : 0) != 0);
        }
        if (!dfs.exists(ecPath)) {
            dfs.mkdirs(ecPath);
            dfs.getClient().setErasureCodingPolicy(ecPath.toString(), ecPolicy.getName());
        } else {
            Preconditions.checkArgument((boolean)dfs.getClient().getErasureCodingPolicy(ecPath.toString()).equals((Object)ecPolicy));
        }
    }

    public int run(String[] args) throws Exception {
        OpType type = null;
        int dataSizeMB = 0;
        boolean isEc = true;
        int numClients = 1;
        boolean statefulRead = true;
        if (args.length >= 3) {
            if (args[0].equals("read")) {
                type = OpType.READ;
            } else if (args[0].equals("write")) {
                type = OpType.WRITE;
            } else if (args[0].equals("gen")) {
                type = OpType.GEN;
            } else if (args[0].equals("clean")) {
                type = OpType.CLEAN;
            } else {
                ErasureCodeBenchmarkThroughput.printUsage("Unknown operation: " + args[0]);
            }
            try {
                dataSizeMB = Integer.parseInt(args[1]);
                if (dataSizeMB <= 0) {
                    ErasureCodeBenchmarkThroughput.printUsage("Invalid data size: " + dataSizeMB);
                }
            }
            catch (NumberFormatException e) {
                ErasureCodeBenchmarkThroughput.printUsage("Invalid data size: " + e.getMessage());
            }
            isEc = args[2].equals("ec");
            if (!isEc && !args[2].equals("rep")) {
                ErasureCodeBenchmarkThroughput.printUsage("Unknown storage policy: " + args[2]);
            }
        } else {
            ErasureCodeBenchmarkThroughput.printUsage(null);
        }
        if (args.length >= 4 && type != OpType.CLEAN) {
            try {
                numClients = Integer.parseInt(args[3]);
                if (numClients <= 0) {
                    ErasureCodeBenchmarkThroughput.printUsage("Invalid num of clients: " + numClients);
                }
            }
            catch (NumberFormatException e) {
                ErasureCodeBenchmarkThroughput.printUsage("Invalid num of clients: " + e.getMessage());
            }
        }
        if (args.length >= 5 && type == OpType.READ && !(statefulRead = args[4].equals("stf")) && !args[4].equals("pos")) {
            ErasureCodeBenchmarkThroughput.printUsage("Unknown read mode: " + args[4]);
        }
        this.setUpDir();
        if (type == OpType.CLEAN) {
            this.cleanUp(dataSizeMB, isEc);
        } else {
            if (type == OpType.READ && isEc) {
                this.setReadThreadPoolSize(numClients);
            }
            this.benchmark(type, dataSizeMB, numClients, isEc, statefulRead);
        }
        return 0;
    }

    private void cleanUp(int dataSizeMB, boolean isEc) throws IOException {
        FileStatus[] fileStatuses;
        final String fileName = ErasureCodeBenchmarkThroughput.getFilePath(dataSizeMB, isEc);
        Path path = isEc ? new Path(EC_DIR) : new Path(REP_DIR);
        for (FileStatus fileStatus : fileStatuses = this.fs.listStatus(path, new PathFilter(){

            public boolean accept(Path path) {
                return path.toString().contains(fileName);
            }
        })) {
            this.fs.delete(fileStatus.getPath(), false);
        }
    }

    public static void main(String[] args) throws Exception {
        HdfsConfiguration conf = new HdfsConfiguration();
        FileSystem fs = FileSystem.get((Configuration)conf);
        int res = ToolRunner.run((Configuration)conf, (Tool)new ErasureCodeBenchmarkThroughput(fs), (String[])args);
        System.exit(res);
    }

    static {
        Random random = new Random();
        random.nextBytes(data);
    }

    private class ReadCallable
    extends CallableBase {
        private final boolean statefulRead;

        public ReadCallable(int dataSizeMB, boolean isEc, int id, boolean statefulRead) throws IOException {
            super(dataSizeMB, isEc, id);
            this.statefulRead = statefulRead;
        }

        private long doStateful(FSDataInputStream inputStream) throws IOException {
            long bytesRead;
            long count = 0L;
            ByteBuffer buffer = ByteBuffer.allocate(0x8000000);
            while ((bytesRead = (long)inputStream.read(buffer)) >= 0L) {
                count += bytesRead;
                buffer.clear();
            }
            return count;
        }

        private long doPositional(FSDataInputStream inputStream) throws IOException {
            long bytesRead;
            long count = 0L;
            byte[] buf = new byte[0x8000000];
            while ((bytesRead = (long)inputStream.read(count, buf, 0, buf.length)) >= 0L) {
                count += bytesRead;
            }
            return count;
        }

        private long readFile(Path path) throws IOException {
            try (FSDataInputStream inputStream = ErasureCodeBenchmarkThroughput.this.fs.open(path);){
                StopWatch sw = new StopWatch().start();
                System.out.println((this.statefulRead ? "Stateful reading " : "Positional reading ") + path);
                long totalRead = this.statefulRead ? this.doStateful(inputStream) : this.doPositional(inputStream);
                System.out.println((this.statefulRead ? "Finished stateful read " : "Finished positional read ") + path + ". Time taken: " + sw.now(TimeUnit.SECONDS) + " s.");
                long l = totalRead;
                return l;
            }
        }

        @Override
        public Long call() throws Exception {
            long dataSize;
            Path path = new Path(this.getFilePathForThread());
            if (!ErasureCodeBenchmarkThroughput.this.fs.exists(path) || ErasureCodeBenchmarkThroughput.this.fs.isDirectory(path)) {
                System.out.println("File not found at " + path + ". Call gen first?");
                return 0L;
            }
            long bytesRead = this.readFile(path);
            Preconditions.checkArgument((bytesRead == (dataSize = (long)(this.dataSizeMB * 1024) * 1024L) ? 1 : 0) != 0, (Object)("Specified data size: " + dataSize + ", actually read " + bytesRead));
            return bytesRead;
        }
    }

    private class WriteCallable
    extends CallableBase {
        private final boolean isGen;

        public WriteCallable(int dataSizeMB, boolean isEc, int id, boolean isGen) throws IOException {
            super(dataSizeMB, isEc, id);
            this.isGen = isGen;
        }

        private long writeFile(Path path) throws IOException {
            long dataSize;
            StopWatch sw = new StopWatch().start();
            System.out.println("Writing " + path);
            long remaining = dataSize = (long)(this.dataSizeMB * 1024) * 1024L;
            try (FSDataOutputStream outputStream = ErasureCodeBenchmarkThroughput.this.fs.create(path);){
                if (!this.isGen) {
                    ErasureCodeBenchmarkThroughput.this.fs.deleteOnExit(path);
                }
                while (remaining > 0L) {
                    int toWrite = (int)Math.min(remaining, (long)data.length);
                    outputStream.write(data, 0, toWrite);
                    remaining -= (long)toWrite;
                }
                System.out.println("Finished writing " + path + ". Time taken: " + sw.now(TimeUnit.SECONDS) + " s.");
                long l = dataSize - remaining;
                return l;
            }
        }

        @Override
        public Long call() throws Exception {
            String pathStr = this.getFilePathForThread();
            if (!this.isGen) {
                pathStr = pathStr + ErasureCodeBenchmarkThroughput.TMP_FILE_SUFFIX;
            }
            Path path = new Path(pathStr);
            if (ErasureCodeBenchmarkThroughput.this.fs.exists(path)) {
                if (this.isGen) {
                    System.out.println("Data already generated at " + path);
                } else {
                    System.out.println("Previous tmp data not cleaned " + path);
                }
                return 0L;
            }
            return this.writeFile(path);
        }
    }

    private abstract class CallableBase
    implements Callable<Long> {
        protected final int dataSizeMB;
        protected final boolean isEc;
        protected final int id;

        public CallableBase(int dataSizeMB, boolean isEc, int id) throws IOException {
            this.dataSizeMB = dataSizeMB;
            this.isEc = isEc;
            this.id = id;
        }

        protected String getFilePathForThread() {
            return ErasureCodeBenchmarkThroughput.getFilePath(this.dataSizeMB, this.isEc) + "_" + this.id;
        }
    }

    static enum OpType {
        READ,
        WRITE,
        GEN,
        CLEAN;

    }
}

