/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.txn.compactor;

import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinWorkerThread;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.common.ValidReadTxnList;
import org.apache.hadoop.hive.common.ValidTxnList;
import org.apache.hadoop.hive.common.ValidWriteIdList;
import org.apache.hadoop.hive.conf.Constants;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.metastore.HMSHandler;
import org.apache.hadoop.hive.metastore.IMetaStoreClient;
import org.apache.hadoop.hive.metastore.LockComponentBuilder;
import org.apache.hadoop.hive.metastore.LockRequestBuilder;
import org.apache.hadoop.hive.metastore.api.CompactionRequest;
import org.apache.hadoop.hive.metastore.api.CompactionResponse;
import org.apache.hadoop.hive.metastore.api.CompactionType;
import org.apache.hadoop.hive.metastore.api.DataOperationType;
import org.apache.hadoop.hive.metastore.api.Database;
import org.apache.hadoop.hive.metastore.api.GetOpenTxnsResponse;
import org.apache.hadoop.hive.metastore.api.GetValidWriteIdsRequest;
import org.apache.hadoop.hive.metastore.api.LockRequest;
import org.apache.hadoop.hive.metastore.api.LockType;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.api.NoSuchObjectException;
import org.apache.hadoop.hive.metastore.api.NoSuchTxnException;
import org.apache.hadoop.hive.metastore.api.Partition;
import org.apache.hadoop.hive.metastore.api.StorageDescriptor;
import org.apache.hadoop.hive.metastore.api.Table;
import org.apache.hadoop.hive.metastore.api.TableValidWriteIds;
import org.apache.hadoop.hive.metastore.metrics.AcidMetricService;
import org.apache.hadoop.hive.metastore.metrics.Metrics;
import org.apache.hadoop.hive.metastore.txn.TxnCommonUtils;
import org.apache.hadoop.hive.metastore.txn.TxnStore;
import org.apache.hadoop.hive.metastore.txn.TxnUtils;
import org.apache.hadoop.hive.metastore.txn.entities.CompactionInfo;
import org.apache.hadoop.hive.metastore.utils.MetaStoreUtils;
import org.apache.hadoop.hive.metastore.utils.StringableMap;
import org.apache.hadoop.hive.ql.io.AcidDirectory;
import org.apache.hadoop.hive.ql.io.AcidUtils;
import org.apache.hadoop.hive.ql.txn.compactor.Initiator;
import org.apache.hadoop.hive.ql.txn.compactor.MetadataCache;
import org.apache.hadoop.hive.ql.txn.compactor.RemoteCompactorUtil;
import org.apache.hadoop.hive.shims.HadoopShims;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hive.common.util.Ref;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactorUtil {
    private static final Logger LOG = LoggerFactory.getLogger(CompactorUtil.class);
    public static final String COMPACTOR = "compactor";
    public static final String COMPACTOR_PREFIX = "compactor.";
    private static final String COMPACTOR_THRESHOLD_PREFIX = "compactorthreshold.";
    private static final List<String> QUEUE_PROPERTIES = Arrays.asList("compactor." + HiveConf.ConfVars.COMPACTOR_JOB_QUEUE.varname, "compactor.mapreduce.job.queuename", "compactor.mapred.job.queue.name");

    public static ExecutorService createExecutorWithThreadFactory(int parallelism, String threadNameFormat) {
        return new ForkJoinPool(parallelism, pool -> {
            ForkJoinWorkerThread worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
            worker.setName(String.format(threadNameFormat, worker.getPoolIndex()));
            return worker;
        }, null, false);
    }

    static String getCompactorJobQueueName(HiveConf conf, CompactionInfo ci, Table table) {
        ArrayList<Function<String, String>> propertyGetters = new ArrayList<Function<String, String>>(2);
        if (ci.properties != null) {
            StringableMap ciProperties = new StringableMap(ci.properties);
            propertyGetters.add(arg_0 -> ciProperties.get(arg_0));
        }
        if (table.getParameters() != null) {
            propertyGetters.add(table.getParameters()::get);
        }
        for (Function function : propertyGetters) {
            for (String p : QUEUE_PROPERTIES) {
                String queueName = (String)function.apply(p);
                if (queueName == null || queueName.isEmpty()) continue;
                return queueName;
            }
        }
        return conf.getVar(HiveConf.ConfVars.COMPACTOR_JOB_QUEUE);
    }

    public static StorageDescriptor resolveStorageDescriptor(Table t, Partition p) {
        return p == null ? t.getSd() : p.getSd();
    }

    public static boolean isDynPartAbort(Table t, String partName) {
        return Optional.ofNullable(t).map(Table::getPartitionKeys).filter(pk -> !pk.isEmpty()).isPresent() && partName == null;
    }

    public static List<Partition> getPartitionsByNames(HiveConf conf, String dbName, String tableName, String partName) throws MetaException {
        try {
            return HMSHandler.getMSForConf((Configuration)conf).getPartitionsByNames(MetaStoreUtils.getDefaultCatalog((Configuration)conf), dbName, tableName, Collections.singletonList(partName));
        }
        catch (Exception e) {
            LOG.error("Unable to get partitions by name = {}.{}.{}", new Object[]{dbName, tableName, partName});
            throw new MetaException(e.toString());
        }
    }

    public static Database resolveDatabase(HiveConf conf, String dbName) throws MetaException, NoSuchObjectException {
        try {
            return HMSHandler.getMSForConf((Configuration)conf).getDatabase(MetaStoreUtils.getDefaultCatalog((Configuration)conf), dbName);
        }
        catch (NoSuchObjectException e) {
            LOG.error("Unable to find database {}, {}", (Object)dbName, (Object)e.getMessage());
            throw e;
        }
    }

    public static Table resolveTable(HiveConf conf, String dbName, String tableName) throws MetaException {
        try {
            return HMSHandler.getMSForConf((Configuration)conf).getTable(MetaStoreUtils.getDefaultCatalog((Configuration)conf), dbName, tableName);
        }
        catch (MetaException e) {
            LOG.error("Unable to find table {}.{}, {}", new Object[]{dbName, tableName, e.getMessage()});
            throw e;
        }
    }

    public static String getDebugInfo(List<Path> paths) {
        return "[" + paths.stream().map(Path::getName).collect(Collectors.joining(",")) + "]";
    }

    public static boolean runJobAsSelf(String owner) {
        return owner.equals(System.getProperty("user.name"));
    }

    public static List<Path> getObsoleteDirs(AcidDirectory dir, boolean isDynPartAbort) {
        List<Path> obsoleteDirs = dir.getObsolete();
        obsoleteDirs.addAll(dir.getAbortedDirectories());
        if (isDynPartAbort) {
            obsoleteDirs = dir.getAbortedDirectories();
        }
        return obsoleteDirs;
    }

    public static Partition resolvePartition(HiveConf conf, IMetaStoreClient msc, String dbName, String tableName, String partName, METADATA_FETCH_MODE fetchMode) throws MetaException {
        if (partName != null) {
            List<Partition> parts = null;
            try {
                switch (fetchMode) {
                    case LOCAL: {
                        parts = CompactorUtil.getPartitionsByNames(conf, dbName, tableName, partName);
                        break;
                    }
                    case REMOTE: {
                        parts = RemoteCompactorUtil.getPartitionsByNames(msc, dbName, tableName, partName);
                    }
                }
                if (CollectionUtils.isEmpty(parts)) {
                    return null;
                }
            }
            catch (Exception e) {
                LOG.error("Unable to find partition " + CompactorUtil.getFullPartitionName(dbName, tableName, partName), (Throwable)e);
                throw e;
            }
            if (parts.size() != 1) {
                LOG.error(CompactorUtil.getFullPartitionName(dbName, tableName, partName) + " does not refer to a single partition. " + Arrays.toString(parts.toArray()));
                throw new MetaException("Too many partitions for : " + CompactorUtil.getFullPartitionName(dbName, tableName, partName));
            }
            return parts.get(0);
        }
        return null;
    }

    public static String getFullPartitionName(String dbName, String tableName, String partName) {
        StringBuilder buf = new StringBuilder();
        buf.append(dbName);
        buf.append('.');
        buf.append(tableName);
        if (partName != null) {
            buf.append('.');
            buf.append(partName);
        }
        return buf.toString();
    }

    public static void checkInterrupt(String callerClassName) throws InterruptedException {
        if (Thread.interrupted()) {
            throw new InterruptedException(callerClassName + " execution is interrupted.");
        }
    }

    public static boolean isMinorCompactionSupported(HiveConf conf, Map<String, String> tblProperties, AcidDirectory dir) {
        return AcidUtils.isInsertOnlyTable(tblProperties) || !conf.getBoolVar(HiveConf.ConfVars.COMPACTOR_CRUD_QUERY_BASED) || dir.getOriginalFiles().isEmpty() && dir.getCurrentDirectories().stream().noneMatch(AcidUtils.ParsedDelta::isRawFormat);
    }

    public static LockRequest createLockRequest(HiveConf conf, CompactionInfo ci, long txnId, LockType lockType, DataOperationType opType) {
        String agentInfo = Thread.currentThread().getName();
        LockComponentBuilder lockCompBuilder = new LockComponentBuilder().setLock(lockType).setOperationType(opType).setDbName(ci.dbname).setTableName(ci.tableName).setIsTransactional(true);
        if (ci.partName != null) {
            lockCompBuilder.setPartitionName(ci.partName);
        }
        return new LockRequestBuilder(agentInfo).setTransactionId(txnId).setUser(ci.runAs).setZeroWaitReadEnabled(!conf.getBoolVar(HiveConf.ConfVars.TXN_OVERWRITE_X_LOCK) || !conf.getBoolVar(HiveConf.ConfVars.TXN_WRITE_X_LOCK)).addLockComponent(lockCompBuilder.build()).build();
    }

    private static CompactionResponse requestCompaction(CompactionInfo ci, String runAs, String hostname, TxnStore txnHandler) throws MetaException {
        CompactionRequest compactionRequest = new CompactionRequest(ci.dbname, ci.tableName, ci.type);
        if (ci.partName != null) {
            compactionRequest.setPartitionname(ci.partName);
        }
        compactionRequest.setRunas(runAs);
        if (StringUtils.isEmpty((CharSequence)ci.initiatorId)) {
            compactionRequest.setInitiatorId(hostname + "-" + Thread.currentThread().getId());
        } else {
            compactionRequest.setInitiatorId(ci.initiatorId);
        }
        compactionRequest.setInitiatorVersion(ci.initiatorVersion);
        compactionRequest.setPoolName(ci.poolName);
        compactionRequest.setOrderByClause(ci.orderByClause);
        LOG.info("Requesting compaction: " + compactionRequest);
        CompactionResponse resp = txnHandler.compact(compactionRequest);
        if (resp.isAccepted()) {
            ci.id = resp.getId();
        }
        return resp;
    }

    private static CompactionType determineCompactionType(CompactionInfo ci, AcidDirectory dir, Map<String, String> tblProperties, long baseSize, long deltaSize, HiveConf conf) {
        boolean enough;
        boolean noBase = false;
        List<AcidUtils.ParsedDelta> deltas = dir.getCurrentDirectories();
        if (baseSize == 0L && deltaSize > 0L) {
            noBase = true;
        } else {
            boolean initiateMajor;
            String deltaPctProp = tblProperties.get(COMPACTOR_THRESHOLD_PREFIX + HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_PCT_THRESHOLD);
            float deltaPctThreshold = deltaPctProp == null ? HiveConf.getFloatVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_PCT_THRESHOLD) : Float.parseFloat(deltaPctProp);
            boolean bigEnough = (float)deltaSize / (float)baseSize > deltaPctThreshold;
            boolean multiBase = dir.getObsolete().stream().anyMatch(path -> path.getName().startsWith("base_"));
            boolean bl = initiateMajor = bigEnough || deltaSize == 0L && multiBase;
            if (LOG.isDebugEnabled()) {
                StringBuilder msg = new StringBuilder("delta size: ");
                msg.append(deltaSize);
                msg.append(" base size: ");
                msg.append(baseSize);
                msg.append(" multiBase ");
                msg.append(multiBase);
                msg.append(" deltaSize ");
                msg.append(deltaSize);
                msg.append(" threshold: ");
                msg.append(deltaPctThreshold);
                msg.append(" delta/base ratio > ").append(HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_PCT_THRESHOLD.varname).append(": ");
                msg.append(bigEnough);
                msg.append(".");
                if (!initiateMajor) {
                    msg.append("not");
                }
                msg.append(" initiating major compaction.");
                LOG.debug(msg.toString());
            }
            if (initiateMajor) {
                return CompactionType.MAJOR;
            }
        }
        String deltaNumProp = tblProperties.get(COMPACTOR_THRESHOLD_PREFIX + HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_NUM_THRESHOLD);
        int deltaNumThreshold = deltaNumProp == null ? HiveConf.getIntVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_COMPACTOR_DELTA_NUM_THRESHOLD) : Integer.parseInt(deltaNumProp);
        boolean bl = enough = deltas.size() > deltaNumThreshold;
        if (!enough) {
            LOG.debug("Not enough deltas to initiate compaction for table=" + ci.tableName + "partition=" + ci.partName + ". Found: " + deltas.size() + " deltas, threshold is " + deltaNumThreshold);
            return null;
        }
        LOG.debug("Found " + deltas.size() + " delta files, and " + (noBase ? "no" : "has") + " base,requesting " + (noBase ? "major" : "minor") + " compaction");
        return noBase || !CompactorUtil.isMinorCompactionSupported(conf, tblProperties, dir) ? CompactionType.MAJOR : CompactionType.MINOR;
    }

    private static long getBaseSize(AcidDirectory dir) throws IOException {
        long baseSize = 0L;
        if (dir.getBase() != null) {
            baseSize = CompactorUtil.getDirSize(dir.getFs(), dir.getBase());
        } else {
            for (HadoopShims.HdfsFileStatusWithId origStat : dir.getOriginalFiles()) {
                baseSize += origStat.getFileStatus().getLen();
            }
        }
        return baseSize;
    }

    private static long getDirSize(FileSystem fs, AcidUtils.ParsedDirectory dir) throws IOException {
        return dir.getFiles(fs, (Ref<Boolean>)Ref.from((Object)false)).stream().map(HadoopShims.HdfsFileStatusWithId::getFileStatus).mapToLong(FileStatus::getLen).sum();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static CompactionType checkForCompaction(CompactionInfo ci, ValidWriteIdList writeIds, StorageDescriptor sd, Map<String, String> tblProperties, String runAs, TxnStore txnHandler, HiveConf conf) throws IOException, InterruptedException {
        CompactionType compactionType;
        if (MetaStoreUtils.isIcebergTable(tblProperties)) {
            return ci.type;
        }
        if (ci.tooManyAborts) {
            LOG.debug("Found too many aborted transactions for " + ci.getFullPartitionName() + ", initiating major compaction");
            return CompactionType.MAJOR;
        }
        if (ci.hasOldAbort) {
            HiveConf.ConfVars oldAbortedTimeoutProp = HiveConf.ConfVars.HIVE_COMPACTOR_ABORTEDTXN_TIME_THRESHOLD;
            LOG.debug("Found an aborted transaction for " + ci.getFullPartitionName() + " with age older than threshold " + oldAbortedTimeoutProp + ": " + conf.getTimeVar(oldAbortedTimeoutProp, TimeUnit.HOURS) + " hours. Initiating minor compaction.");
            return CompactionType.MINOR;
        }
        AcidDirectory acidDirectory = AcidUtils.getAcidState(sd, writeIds, conf);
        long baseSize = CompactorUtil.getBaseSize(acidDirectory);
        FileSystem fs = acidDirectory.getFs();
        HashMap<Path, Long> deltaSizes = new HashMap<Path, Long>();
        for (AcidUtils.ParsedDelta delta : acidDirectory.getCurrentDirectories()) {
            deltaSizes.put(delta.getPath(), CompactorUtil.getDirSize(fs, delta));
        }
        long deltaSize = deltaSizes.values().stream().reduce(0L, Long::sum);
        AcidMetricService.updateMetricsFromInitiator((String)ci.dbname, (String)ci.tableName, (String)ci.partName, (Configuration)conf, (TxnStore)txnHandler, (long)baseSize, deltaSizes, acidDirectory.getObsolete());
        if (CompactorUtil.runJobAsSelf(runAs)) {
            return CompactorUtil.determineCompactionType(ci, acidDirectory, tblProperties, baseSize, deltaSize, conf);
        }
        LOG.info("Going to initiate as user " + runAs + " for " + ci.getFullPartitionName());
        UserGroupInformation ugi = UserGroupInformation.createProxyUser((String)runAs, (UserGroupInformation)UserGroupInformation.getLoginUser());
        try {
            compactionType = (CompactionType)ugi.doAs(() -> CompactorUtil.determineCompactionType(ci, acidDirectory, tblProperties, baseSize, deltaSize, conf));
        }
        finally {
            try {
                FileSystem.closeAllForUGI((UserGroupInformation)ugi);
            }
            catch (IOException exception) {
                LOG.error("Could not clean up file-system handles for UGI: " + ugi + " for " + ci.getFullPartitionName(), (Throwable)exception);
            }
        }
        return compactionType;
    }

    private static ValidWriteIdList resolveValidWriteIds(Table t, TxnStore txnHandler, HiveConf conf) throws NoSuchTxnException, MetaException {
        ValidReadTxnList validTxnList = ValidReadTxnList.fromValue((String)conf.get("hive.txn.valid.txns"));
        String fullTableName = TxnUtils.getFullTableName((String)t.getDbName(), (String)t.getTableName());
        GetValidWriteIdsRequest validWriteIdsRequest = new GetValidWriteIdsRequest(Collections.singletonList(fullTableName));
        validWriteIdsRequest.setValidTxnList(validTxnList.writeToString());
        return TxnUtils.createValidCompactWriteIdList((TableValidWriteIds)((TableValidWriteIds)txnHandler.getValidWriteIds(validWriteIdsRequest).getTblValidWriteIds().get(0)));
    }

    public static CompactionResponse scheduleCompactionIfRequired(CompactionInfo ci, Table t, Partition p, String runAs, boolean metricsEnabled, String hostName, TxnStore txnHandler, HiveConf conf) throws MetaException {
        StorageDescriptor sd = CompactorUtil.resolveStorageDescriptor(t, p);
        try {
            ValidWriteIdList validWriteIds = CompactorUtil.resolveValidWriteIds(t, txnHandler, conf);
            CompactorUtil.checkInterrupt(Initiator.class.getName());
            CompactionType type = CompactorUtil.checkForCompaction(ci, validWriteIds, sd, t.getParameters(), runAs, txnHandler, conf);
            if (type != null) {
                ci.type = type;
                return CompactorUtil.requestCompaction(ci, runAs, hostName, txnHandler);
            }
        }
        catch (InterruptedException e) {
            LOG.info("Initiator pool is being shut down, task received interruption.");
        }
        catch (Throwable ex) {
            String errorMessage = "Caught exception while trying to determine if we should compact " + ci + ". Marking failed to avoid repeated failures, " + ex;
            LOG.error(errorMessage);
            ci.errorMessage = errorMessage;
            if (metricsEnabled) {
                Metrics.getOrCreateCounter((String)"compaction_initiator_failure_counter").inc();
            }
            txnHandler.markFailed(ci);
        }
        return null;
    }

    public static CompactionResponse initiateCompactionForPartition(Table table, Partition partition, CompactionRequest compactionRequest, String hostName, TxnStore txnHandler, HiveConf inputConf) throws MetaException {
        CompactionResponse compactionResponse;
        ValidTxnList validTxnList = TxnCommonUtils.createValidReadTxnList((GetOpenTxnsResponse)txnHandler.getOpenTxns(), (long)0L);
        HiveConf conf = new HiveConf(inputConf);
        conf.set("hive.txn.valid.txns", validTxnList.writeToString());
        CompactionInfo compactionInfo = new CompactionInfo(table.getDbName(), table.getTableName(), compactionRequest.getPartitionname(), compactionRequest.getType());
        compactionInfo.initiatorId = compactionRequest.getInitiatorId();
        compactionInfo.orderByClause = compactionRequest.getOrderByClause();
        compactionInfo.initiatorVersion = compactionRequest.getInitiatorVersion();
        if (compactionRequest.getNumberOfBuckets() > 0) {
            compactionInfo.numberOfBuckets = compactionRequest.getNumberOfBuckets();
        }
        compactionInfo.poolName = compactionRequest.getPoolName();
        try {
            StorageDescriptor sd = CompactorUtil.resolveStorageDescriptor(table, partition);
            String runAs = TxnUtils.findUserToRunAs((String)sd.getLocation(), (Table)table, (Configuration)conf);
            LOG.info("Checking to see if we should compact partition {} of table {}.{}", new Object[]{compactionInfo.partName, table.getDbName(), table.getTableName()});
            compactionResponse = CompactorUtil.scheduleCompactionIfRequired(compactionInfo, table, partition, runAs, false, hostName, txnHandler, conf);
        }
        catch (IOException | InterruptedException | MetaException e) {
            LOG.error("Error occurred while Checking if we should compact partition {} of table {}.{} Exception: {}", new Object[]{compactionInfo.partName, table.getDbName(), table.getTableName(), e.getMessage()});
            throw new RuntimeException(e);
        }
        return compactionResponse;
    }

    public static Map<String, Integer> getPoolConf(HiveConf hiveConf) {
        HashMap poolConf = Maps.newHashMap();
        for (Map.Entry entry : hiveConf) {
            Matcher matcher = Constants.COMPACTION_POOLS_PATTERN.matcher((CharSequence)entry.getKey());
            if (!matcher.matches()) continue;
            poolConf.put(matcher.group(1), NumberUtils.toInt((String)((String)entry.getValue()), (int)0));
        }
        return poolConf;
    }

    public static String getPoolName(HiveConf conf, Table t, MetadataCache metadataCache) throws Exception {
        Map params = (Map)ObjectUtils.defaultIfNull((Object)t.getParameters(), Collections.emptyMap());
        String poolName = (String)params.get("hive.compactor.worker.pool");
        if (StringUtils.isBlank((CharSequence)poolName)) {
            params = (Map)ObjectUtils.defaultIfNull((Object)metadataCache.computeIfAbsent(t.getDbName(), () -> CompactorUtil.resolveDatabase(conf, t.getDbName())).getParameters(), Collections.emptyMap());
            poolName = (String)params.get("hive.compactor.worker.pool");
        }
        return poolName;
    }

    public static void overrideConfProps(HiveConf conf, CompactionInfo ci, Map<String, String> properties) {
        CompactorUtil.overrideConfProps(conf, (Map<String, String>)new StringableMap(ci.properties), properties);
    }

    public static void overrideConfProps(HiveConf conf, Map<String, String> ciProperties, Map<String, String> properties) {
        Stream.of(properties, ciProperties).filter(Objects::nonNull).flatMap(map -> map.entrySet().stream()).filter(entry -> ((String)entry.getKey()).startsWith(COMPACTOR_PREFIX)).forEach(entry -> {
            String property = ((String)entry.getKey()).substring(COMPACTOR_PREFIX.length());
            conf.set(property, (String)entry.getValue());
        });
    }

    public static enum METADATA_FETCH_MODE {
        LOCAL,
        REMOTE;

    }

    public static interface ThrowingRunnable<E extends Exception> {
        public void run() throws E;

        public static Runnable unchecked(ThrowingRunnable<?> r) {
            return () -> {
                try {
                    r.run();
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            };
        }
    }
}

