#!/bin/bash
#
# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

set -euo pipefail

pushd ${IMPALA_HOME} > /dev/null 2>&1

RETCODE=0
for file in $(git ls-files '**/*.py'); do
    # Skip the shell's ext-py code
    if [[ "${file}" =~ "shell/ext-py" ]]; then
        continue
    fi
    # Skip the shell's pkg_resources.py
    if [[ "${file}" == "shell/legacy/pkg_resources.py" ]]; then
        continue
    fi

    # Python 2 checks
    # -l = no recursion
    # -q = only print errors
    # -f = force recompile
    # Ignore files that are created to run with python3.
    FIRST_LINE=$(head -n1 ${file})
    if [[ "${FIRST_LINE}" =~ "python3" ]]; then
        >&2 echo "SKIPPING: ${file} is already using python3: ${FIRST_LINE}"
    elif ! python2 -m compileall -l -q -f ${file} > /dev/null 2>&1; then
        RETCODE=1
        echo "Python 2 compilation failed for ${file}:"
        set +e
        python2 -m compileall -l -q -f ${file}
        set -e
    fi
    # Clean up the .pyc files generated by compilation
    if [[ -f "${file}c" ]]; then
        rm "${file}c"
    fi

    # Python 3 checks
    # -l = no recursion
    # -q = only print errors
    # -f = force recompile
    if ! python3 -m compileall -l -q -f ${file} > /dev/null 2>&1  ; then
        RETCODE=1
        echo "Python 3 compilation failed for ${file}:"
        set +e
        python3 -m compileall -l -q -f ${file}
        set -e
    fi
    # Clean up the __pycache__ directories generated by compilation
    py_cache_dir="$(dirname ${file})/__pycache__"
    if [[ -d "${py_cache_dir}" ]]; then
        rm -rf ${py_cache_dir}
    fi
done

popd > /dev/null 2>&1

exit ${RETCODE}
