/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils.db;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.utils.IOUtils;
import org.apache.hadoop.hdds.utils.RocksDBStoreMetrics;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.CodecRegistry;
import org.apache.hadoop.hdds.utils.db.DBCheckpoint;
import org.apache.hadoop.hdds.utils.db.DBStore;
import org.apache.hadoop.hdds.utils.db.DBUpdatesWrapper;
import org.apache.hadoop.hdds.utils.db.RDBBatchOperation;
import org.apache.hadoop.hdds.utils.db.RDBCheckpointManager;
import org.apache.hadoop.hdds.utils.db.RDBMetrics;
import org.apache.hadoop.hdds.utils.db.RDBTable;
import org.apache.hadoop.hdds.utils.db.RocksDatabase;
import org.apache.hadoop.hdds.utils.db.SequenceNumberNotFoundException;
import org.apache.hadoop.hdds.utils.db.StringCodec;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableConfig;
import org.apache.hadoop.hdds.utils.db.TypedTable;
import org.apache.hadoop.hdds.utils.db.cache.TableCache;
import org.apache.hadoop.hdds.utils.db.managed.ManagedCompactRangeOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedDBOptions;
import org.apache.hadoop.hdds.utils.db.managed.ManagedStatistics;
import org.apache.hadoop.hdds.utils.db.managed.ManagedTransactionLogIterator;
import org.apache.hadoop.hdds.utils.db.managed.ManagedWriteOptions;
import org.apache.ozone.rocksdiff.RocksDBCheckpointDiffer;
import org.rocksdb.RocksDBException;
import org.rocksdb.Statistics;
import org.rocksdb.TransactionLogIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RDBStore
implements DBStore {
    private static final Logger LOG = LoggerFactory.getLogger(RDBStore.class);
    private final RocksDatabase db;
    private final File dbLocation;
    private final CodecRegistry codecRegistry;
    private RocksDBStoreMetrics metrics;
    private final RDBCheckpointManager checkPointManager;
    private final String checkpointsParentDir;
    private final String snapshotsParentDir;
    private final RDBMetrics rdbMetrics;
    private final RocksDBCheckpointDiffer rocksDBCheckpointDiffer;
    private final long maxDbUpdatesSizeThreshold;
    private final ManagedDBOptions dbOptions;
    private final ManagedStatistics statistics;
    private final String threadNamePrefix;

    public RDBStore(File dbFile, ManagedDBOptions dbOptions, ManagedStatistics statistics, ManagedWriteOptions writeOptions, Set<TableConfig> families, CodecRegistry registry, boolean readOnly, int maxFSSnapshots, String dbJmxBeanName, boolean enableCompactionDag, long maxDbUpdatesSizeThreshold, boolean createCheckpointDirs, ConfigurationSource configuration, String threadNamePrefix) throws IOException {
        this.threadNamePrefix = threadNamePrefix;
        Preconditions.checkNotNull((Object)dbFile, (Object)"DB file location cannot be null");
        Preconditions.checkNotNull(families);
        Preconditions.checkArgument((!families.isEmpty() ? 1 : 0) != 0);
        this.maxDbUpdatesSizeThreshold = maxDbUpdatesSizeThreshold;
        this.codecRegistry = registry;
        this.dbLocation = dbFile;
        this.dbOptions = dbOptions;
        this.statistics = statistics;
        try {
            if (enableCompactionDag) {
                this.rocksDBCheckpointDiffer = RocksDBCheckpointDiffer.RocksDBCheckpointDifferHolder.getInstance((String)this.getSnapshotMetadataDir(), (String)"compaction-sst-backup", (String)"compaction-log", (String)this.dbLocation.toString(), (ConfigurationSource)configuration);
                this.rocksDBCheckpointDiffer.setRocksDBForCompactionTracking(dbOptions);
            } else {
                this.rocksDBCheckpointDiffer = null;
            }
            this.db = RocksDatabase.open(dbFile, dbOptions, writeOptions, families, readOnly);
            if (dbJmxBeanName == null) {
                dbJmxBeanName = dbFile.getName();
            }
            this.metrics = RocksDBStoreMetrics.create((Statistics)statistics, this.db, dbJmxBeanName);
            if (this.metrics == null) {
                LOG.warn("Metrics registration failed during RocksDB init, db path :{}", (Object)dbJmxBeanName);
            } else {
                LOG.debug("Metrics registration succeed during RocksDB init, db path :{}", (Object)dbJmxBeanName);
            }
            if (!createCheckpointDirs) {
                this.checkpointsParentDir = null;
                this.snapshotsParentDir = null;
            } else {
                Path checkpointsParentDirPath = Paths.get(this.dbLocation.getParent(), "db.checkpoints");
                this.checkpointsParentDir = checkpointsParentDirPath.toString();
                Files.createDirectories(checkpointsParentDirPath, new FileAttribute[0]);
                Path snapshotsParentDirPath = Paths.get(this.dbLocation.getParent(), "db.snapshots/checkpointState");
                this.snapshotsParentDir = snapshotsParentDirPath.toString();
                Files.createDirectories(snapshotsParentDirPath, new FileAttribute[0]);
            }
            if (enableCompactionDag) {
                RocksDatabase.ColumnFamily ssInfoTableCF = this.db.getColumnFamily("snapshotInfoTable");
                Preconditions.checkNotNull((Object)ssInfoTableCF, (Object)"SnapshotInfoTable column family handle should not be null");
                this.rocksDBCheckpointDiffer.setSnapshotInfoTableCFHandle(ssInfoTableCF.getHandle());
                RocksDatabase.ColumnFamily compactionLogTableCF = this.db.getColumnFamily("compactionLogTable");
                Preconditions.checkNotNull((Object)compactionLogTableCF, (Object)"CompactionLogTable column family handle should not be null.");
                this.rocksDBCheckpointDiffer.setCompactionLogTableCFHandle(compactionLogTableCF.getHandle());
                this.rocksDBCheckpointDiffer.setActiveRocksDB(this.db.getManagedRocksDb());
                this.rocksDBCheckpointDiffer.loadAllCompactionLogs();
            }
            this.checkPointManager = new RDBCheckpointManager(this.db, this.dbLocation.getName());
            this.rdbMetrics = RDBMetrics.create();
        }
        catch (Exception e) {
            this.close();
            String msg = "Failed init RocksDB, db path : " + dbFile.getAbsolutePath() + ", exception :" + (e.getCause() == null ? e.getClass().getCanonicalName() + " " + e.getMessage() : e.getCause().getClass().getCanonicalName() + " " + e.getCause().getMessage());
            throw new IOException(msg, e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("RocksDB successfully opened.");
            LOG.debug("[Option] dbLocation= {}", (Object)this.dbLocation.getAbsolutePath());
            LOG.debug("[Option] createIfMissing = {}", (Object)dbOptions.createIfMissing());
            LOG.debug("[Option] maxOpenFiles= {}", (Object)dbOptions.maxOpenFiles());
        }
    }

    public String getSnapshotMetadataDir() {
        return this.dbLocation.getParent() + "/" + "db.snapshots/diffState";
    }

    public String getSnapshotsParentDir() {
        return this.snapshotsParentDir;
    }

    @Override
    public RocksDBCheckpointDiffer getRocksDBCheckpointDiffer() {
        return this.rocksDBCheckpointDiffer;
    }

    public ManagedDBOptions getDbOptions() {
        return this.dbOptions;
    }

    @Override
    public void compactDB() throws IOException {
        try (ManagedCompactRangeOptions options = new ManagedCompactRangeOptions();){
            this.db.compactDB(options);
        }
    }

    @Override
    public void close() throws IOException {
        if (this.metrics != null) {
            this.metrics.unregister();
            this.metrics = null;
        }
        RDBMetrics.unRegister();
        IOUtils.close((Logger)LOG, (AutoCloseable[])new AutoCloseable[]{this.checkPointManager});
        if (this.rocksDBCheckpointDiffer != null) {
            RocksDBCheckpointDiffer.RocksDBCheckpointDifferHolder.invalidateCacheEntry((String)this.rocksDBCheckpointDiffer.getMetadataDir());
        }
        if (this.statistics != null) {
            IOUtils.close((Logger)LOG, (AutoCloseable[])new AutoCloseable[]{this.statistics});
        }
        IOUtils.close((Logger)LOG, (AutoCloseable[])new AutoCloseable[]{this.db});
    }

    public <K, V> void move(K key, Table<K, V> source, Table<K, V> dest) throws IOException {
        try (BatchOperation batchOperation = this.initBatchOperation();){
            V value = source.get(key);
            dest.putWithBatch(batchOperation, key, value);
            source.deleteWithBatch(batchOperation, key);
            this.commitBatchOperation(batchOperation);
        }
    }

    public <K, V> void move(K key, V value, Table<K, V> source, Table<K, V> dest) throws IOException {
        this.move(key, key, value, source, dest);
    }

    public <K, V> void move(K sourceKey, K destKey, V value, Table<K, V> source, Table<K, V> dest) throws IOException {
        try (BatchOperation batchOperation = this.initBatchOperation();){
            dest.putWithBatch(batchOperation, destKey, value);
            source.deleteWithBatch(batchOperation, sourceKey);
            this.commitBatchOperation(batchOperation);
        }
    }

    @Override
    public long getEstimatedKeyCount() throws IOException {
        return this.db.estimateNumKeys();
    }

    @Override
    public BatchOperation initBatchOperation() {
        return new RDBBatchOperation();
    }

    @Override
    public void commitBatchOperation(BatchOperation operation) throws IOException {
        ((RDBBatchOperation)operation).commit(this.db);
    }

    public RDBTable getTable(String name) throws IOException {
        RocksDatabase.ColumnFamily handle = this.db.getColumnFamily(name);
        if (handle == null) {
            throw new IOException("No such table in this DB. TableName : " + name);
        }
        return new RDBTable(this.db, handle, this.rdbMetrics);
    }

    public <K, V> TypedTable<K, V> getTable(String name, Class<K> keyType, Class<V> valueType) throws IOException {
        return new TypedTable<K, V>(this.getTable(name), this.codecRegistry, keyType, valueType);
    }

    public <K, V> Table<K, V> getTable(String name, Class<K> keyType, Class<V> valueType, TableCache.CacheType cacheType) throws IOException {
        return new TypedTable<K, V>(this.getTable(name), this.codecRegistry, keyType, valueType, cacheType, this.threadNamePrefix);
    }

    @Override
    public ArrayList<Table> listTables() {
        ArrayList<Table> returnList = new ArrayList<Table>();
        for (RocksDatabase.ColumnFamily family : this.getColumnFamilies()) {
            returnList.add(new RDBTable(this.db, family, this.rdbMetrics));
        }
        return returnList;
    }

    @Override
    public void flushDB() throws IOException {
        this.db.flush();
    }

    @Override
    public void flushLog(boolean sync) throws IOException {
        this.db.flushWal(sync);
    }

    @Override
    public DBCheckpoint getCheckpoint(boolean flush) throws IOException {
        if (flush) {
            this.flushDB();
        }
        return this.checkPointManager.createCheckpoint(this.checkpointsParentDir);
    }

    public DBCheckpoint getSnapshot(String name) throws IOException {
        this.flushLog(true);
        return this.checkPointManager.createCheckpoint(this.snapshotsParentDir, name);
    }

    @Override
    public File getDbLocation() {
        return this.dbLocation;
    }

    @Override
    public Map<Integer, String> getTableNames() {
        HashMap<Integer, String> tableNames = new HashMap<Integer, String>();
        StringCodec stringCodec = StringCodec.get();
        for (RocksDatabase.ColumnFamily columnFamily : this.getColumnFamilies()) {
            tableNames.put(columnFamily.getID(), columnFamily.getName(stringCodec));
        }
        return tableNames;
    }

    public Collection<RocksDatabase.ColumnFamily> getColumnFamilies() {
        return this.db.getExtraColumnFamilies();
    }

    @Override
    public DBUpdatesWrapper getUpdatesSince(long sequenceNumber) throws IOException {
        return this.getUpdatesSince(sequenceNumber, Long.MAX_VALUE);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public DBUpdatesWrapper getUpdatesSince(long sequenceNumber, long limitCount) throws IOException {
        if (limitCount <= 0L) {
            throw new IllegalArgumentException("Illegal count for getUpdatesSince.");
        }
        long cumulativeDBUpdateLogBatchSize = 0L;
        DBUpdatesWrapper dbUpdatesWrapper = new DBUpdatesWrapper();
        try (ManagedTransactionLogIterator logIterator = this.db.getUpdatesSince(sequenceNumber);){
            if (this.db.getLatestSequenceNumber() != sequenceNumber && !((TransactionLogIterator)logIterator.get()).isValid()) {
                throw new SequenceNumberNotFoundException("Invalid transaction log iterator when getting updates since sequence number " + sequenceNumber);
            }
            boolean checkValidStartingSeqNumber = true;
            while (((TransactionLogIterator)logIterator.get()).isValid()) {
                TransactionLogIterator.BatchResult result = ((TransactionLogIterator)logIterator.get()).getBatch();
                try {
                    long currSequenceNumber = result.sequenceNumber();
                    if (checkValidStartingSeqNumber && currSequenceNumber > 1L + sequenceNumber) {
                        throw new SequenceNumberNotFoundException("Unable to read full data from RocksDB wal to get delta updates. It may have partially been flushed to SSTs. Requested sequence number is " + sequenceNumber + " and first available sequence number is " + currSequenceNumber + " in wal.");
                    }
                    checkValidStartingSeqNumber = false;
                    if (currSequenceNumber <= sequenceNumber) {
                        ((TransactionLogIterator)logIterator.get()).next();
                        continue;
                    }
                    dbUpdatesWrapper.addWriteBatch(result.writeBatch().data(), result.sequenceNumber());
                    if (currSequenceNumber - sequenceNumber >= limitCount || (cumulativeDBUpdateLogBatchSize += result.writeBatch().getDataSize()) >= this.maxDbUpdatesSizeThreshold) {
                        break;
                    }
                }
                finally {
                    result.writeBatch().close();
                    continue;
                }
                ((TransactionLogIterator)logIterator.get()).next();
            }
            dbUpdatesWrapper.setLatestSequenceNumber(this.db.getLatestSequenceNumber());
        }
        catch (SequenceNumberNotFoundException e) {
            LOG.warn("Unable to get delta updates since sequenceNumber {}. This exception will be thrown to the client", (Object)sequenceNumber, (Object)e);
            dbUpdatesWrapper.setDBUpdateSuccess(false);
            throw e;
        }
        catch (IOException | RocksDBException e) {
            LOG.error("Unable to get delta updates since sequenceNumber {}. This exception will not be thrown to the client ", (Object)sequenceNumber, (Object)e);
            dbUpdatesWrapper.setDBUpdateSuccess(false);
        }
        finally {
            if (dbUpdatesWrapper.getData().size() > 0) {
                this.rdbMetrics.incWalUpdateDataSize(cumulativeDBUpdateLogBatchSize);
                this.rdbMetrics.incWalUpdateSequenceCount(dbUpdatesWrapper.getCurrentSequenceNumber() - sequenceNumber);
            }
        }
        if (!dbUpdatesWrapper.isDBUpdateSuccess()) {
            LOG.warn("Returned DBUpdates isDBUpdateSuccess: {}", (Object)dbUpdatesWrapper.isDBUpdateSuccess());
        }
        return dbUpdatesWrapper;
    }

    @Override
    public boolean isClosed() {
        return this.db.isClosed();
    }

    public RocksDatabase getDb() {
        return this.db;
    }

    public String getProperty(String property) throws IOException {
        return this.db.getProperty(property);
    }

    public String getProperty(RocksDatabase.ColumnFamily family, String property) throws IOException {
        return this.db.getProperty(family, property);
    }

    public RDBMetrics getMetrics() {
        return this.rdbMetrics;
    }

    public static Logger getLogger() {
        return LOG;
    }
}

