/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.iterate;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import org.apache.phoenix.compile.ExplainPlanAttributes;
import org.apache.phoenix.compile.QueryPlan;
import org.apache.phoenix.compile.StatementContext;
import org.apache.phoenix.iterate.PeekingResultIterator;
import org.apache.phoenix.iterate.ResultIterator;
import org.apache.phoenix.iterate.ResultIterators;
import org.apache.phoenix.monitoring.GlobalClientMetrics;
import org.apache.phoenix.query.ConnectionQueryServices;
import org.apache.phoenix.schema.tuple.Tuple;
import org.apache.phoenix.util.ServerUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RoundRobinResultIterator
implements ResultIterator {
    private static final Logger LOGGER = LoggerFactory.getLogger(RoundRobinResultIterator.class);
    private final int threshold;
    private int numScannersCacheExhausted = 0;
    private ResultIterators resultIterators;
    private List<RoundRobinIterator> openIterators = new ArrayList<RoundRobinIterator>();
    private int index;
    private boolean closed;
    private final QueryPlan plan;
    private int numParallelFetches;

    public RoundRobinResultIterator(ResultIterators iterators, QueryPlan plan) {
        this.resultIterators = iterators;
        this.plan = plan;
        this.threshold = this.getThreshold();
    }

    public RoundRobinResultIterator(List<PeekingResultIterator> iterators, QueryPlan plan) {
        this.resultIterators = null;
        this.plan = plan;
        this.threshold = this.getThreshold();
        this.initOpenIterators(this.wrapToRoundRobinIterators(iterators));
    }

    public static ResultIterator newIterator(List<PeekingResultIterator> iterators, QueryPlan plan) {
        if (iterators.isEmpty()) {
            return EMPTY_ITERATOR;
        }
        if (iterators.size() == 1) {
            return iterators.get(0);
        }
        return new RoundRobinResultIterator(iterators, plan);
    }

    @Override
    public Tuple next() throws SQLException {
        List<RoundRobinIterator> iterators;
        int size;
        while ((size = (iterators = this.getIterators()).size()) > 0) {
            this.index %= size;
            RoundRobinIterator itr = iterators.get(this.index);
            if (itr.getNumRecordsRead() < this.threshold) {
                Tuple tuple = itr.peek();
                if (tuple != null) {
                    tuple = itr.next();
                    if (itr.getNumRecordsRead() == this.threshold) {
                        ++this.numScannersCacheExhausted;
                    }
                    this.index = (this.index + 1) % size;
                    return tuple;
                }
                itr.close();
                iterators.remove(this.index);
                if (iterators.size() != 0) continue;
                this.close();
                continue;
            }
            this.index = (this.index + 1) % size;
        }
        this.close();
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws SQLException {
        block24: {
            if (this.closed) {
                return;
            }
            this.closed = true;
            SQLException toThrow = null;
            try {
                if (this.resultIterators != null) {
                    this.resultIterators.close();
                }
            }
            catch (Exception e) {
                toThrow = ServerUtil.parseServerException(e);
            }
            finally {
                try {
                    if (this.openIterators.size() > 0) {
                        for (RoundRobinIterator itr : this.openIterators) {
                            try {
                                itr.close();
                            }
                            catch (Exception e) {
                                if (toThrow == null) {
                                    toThrow = ServerUtil.parseServerException(e);
                                    continue;
                                }
                                toThrow.setNextException(ServerUtil.parseServerException(e));
                            }
                        }
                    }
                }
                finally {
                    if (toThrow == null) break block24;
                    throw toThrow;
                }
            }
        }
    }

    @Override
    public void explain(List<String> planSteps) {
        if (this.resultIterators != null) {
            this.resultIterators.explain(planSteps);
        }
    }

    @Override
    public void explain(List<String> planSteps, ExplainPlanAttributes.ExplainPlanAttributesBuilder explainPlanAttributesBuilder) {
        if (this.resultIterators != null) {
            this.resultIterators.explain(planSteps, explainPlanAttributesBuilder);
        }
    }

    @VisibleForTesting
    int getNumberOfParallelFetches() {
        return this.numParallelFetches;
    }

    @VisibleForTesting
    QueryPlan getQueryPlan() {
        return this.plan;
    }

    private List<RoundRobinIterator> getIterators() throws SQLException {
        if (this.closed) {
            return Collections.emptyList();
        }
        if (this.openIterators.size() > 0 && this.openIterators.size() == this.numScannersCacheExhausted) {
            this.initOpenIterators(this.fetchNextBatch());
        } else if (this.openIterators.size() == 0 && this.resultIterators != null) {
            List<PeekingResultIterator> iterators = this.resultIterators.getIterators();
            this.initOpenIterators(this.wrapToRoundRobinIterators(iterators));
        }
        return this.openIterators;
    }

    private List<RoundRobinIterator> wrapToRoundRobinIterators(List<PeekingResultIterator> iterators) {
        ArrayList<RoundRobinIterator> roundRobinItrs = new ArrayList<RoundRobinIterator>(iterators.size());
        for (PeekingResultIterator itr : iterators) {
            roundRobinItrs.add(new RoundRobinIterator(itr, null));
        }
        return roundRobinItrs;
    }

    private void initOpenIterators(List<RoundRobinIterator> iterators) {
        this.openIterators.clear();
        this.openIterators.addAll(iterators);
        this.index = 0;
        this.numScannersCacheExhausted = 0;
    }

    private int getThreshold() {
        int cacheSize = this.getScannerCacheSize();
        Preconditions.checkArgument((cacheSize > 1 ? 1 : 0) != 0, (Object)"RoundRobinResultIterator doesn't work when cache size is less than or equal to 1");
        return cacheSize - 1;
    }

    private int getScannerCacheSize() {
        try {
            return this.plan.getContext().getStatement().getFetchSize();
        }
        catch (Throwable e) {
            Throwables.propagate((Throwable)e);
            return -1;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<RoundRobinIterator> fetchNextBatch() throws SQLException {
        block31: {
            int numExpectedIterators = this.openIterators.size();
            ArrayList<Future> futures = new ArrayList<Future>(numExpectedIterators);
            ArrayList<RoundRobinIterator> results = new ArrayList<RoundRobinIterator>();
            Collections.shuffle(this.openIterators);
            boolean success = false;
            SQLException toThrow = null;
            try {
                StatementContext context = this.plan.getContext();
                ConnectionQueryServices services = context.getConnection().getQueryServices();
                ThreadPoolExecutor executor = services.getExecutor();
                ++this.numParallelFetches;
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Performing parallel fetch for " + this.openIterators.size() + " iterators. ");
                }
                for (final RoundRobinIterator itr : this.openIterators) {
                    Future future = executor.submit(new Callable<Tuple>(){

                        @Override
                        public Tuple call() throws Exception {
                            return itr.next();
                        }
                    });
                    futures.add(future);
                }
                int i = 0;
                for (Future future : futures) {
                    Tuple tuple = (Tuple)future.get();
                    if (tuple != null) {
                        results.add(new RoundRobinIterator(this.openIterators.get(i).delegate, tuple));
                    } else {
                        this.openIterators.get(i).close();
                    }
                    ++i;
                }
                success = true;
                ArrayList<RoundRobinIterator> arrayList = results;
                return arrayList;
            }
            catch (SQLException e) {
                toThrow = e;
                return toThrow;
            }
            catch (Exception e) {
                toThrow = ServerUtil.parseServerException(e);
                return toThrow;
            }
            finally {
                try {
                    if (!success) {
                        try {
                            this.close();
                        }
                        catch (Exception e) {
                            if (toThrow == null) {
                                toThrow = ServerUtil.parseServerException(e);
                            }
                            toThrow.setNextException(ServerUtil.parseServerException(e));
                        }
                    }
                }
                finally {
                    if (toThrow == null) break block31;
                    GlobalClientMetrics.GLOBAL_FAILED_QUERY_COUNTER.increment();
                    throw toThrow;
                }
            }
        }
        return null;
    }

    private static class RoundRobinIterator
    implements PeekingResultIterator {
        private PeekingResultIterator delegate;
        private Tuple tuple;
        private int numRecordsRead;

        private RoundRobinIterator(PeekingResultIterator itr, Tuple tuple) {
            this.delegate = itr;
            this.tuple = tuple;
            this.numRecordsRead = 0;
        }

        @Override
        public void close() throws SQLException {
            this.delegate.close();
        }

        @Override
        public Tuple next() throws SQLException {
            if (this.tuple != null) {
                Tuple t = this.tuple;
                this.tuple = null;
                return t;
            }
            ++this.numRecordsRead;
            return this.delegate.next();
        }

        @Override
        public void explain(List<String> planSteps) {
            this.delegate.explain(planSteps);
        }

        @Override
        public void explain(List<String> planSteps, ExplainPlanAttributes.ExplainPlanAttributesBuilder explainPlanAttributesBuilder) {
            this.delegate.explain(planSteps, explainPlanAttributesBuilder);
        }

        @Override
        public Tuple peek() throws SQLException {
            if (this.tuple != null) {
                return this.tuple;
            }
            return this.delegate.peek();
        }

        public int getNumRecordsRead() {
            return this.numRecordsRead;
        }
    }
}

