/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.index;

import com.google.common.base.Stopwatch;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.store.logfile.DefaultMappedFile;
import org.apache.rocketmq.store.logfile.MappedFile;
import org.apache.rocketmq.tieredstore.MessageStoreConfig;
import org.apache.rocketmq.tieredstore.MessageStoreExecutor;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.index.IndexFile;
import org.apache.rocketmq.tieredstore.index.IndexItem;
import org.apache.rocketmq.tieredstore.provider.FileSegment;
import org.apache.rocketmq.tieredstore.provider.PosixFileSegment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexStoreFile
implements IndexFile {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    public static final int INDEX_MAGIC_CODE = 0;
    public static final int INDEX_BEGIN_TIME_STAMP = 4;
    public static final int INDEX_END_TIME_STAMP = 12;
    public static final int INDEX_SLOT_COUNT = 20;
    public static final int INDEX_ITEM_INDEX = 24;
    public static final int INDEX_HEADER_SIZE = 28;
    public static final int BEGIN_MAGIC_CODE = -1127939447;
    public static final int END_MAGIC_CODE = -1127939451;
    private static final int INVALID_INDEX = 0;
    private static final int HASH_SLOT_SIZE = 8;
    private static final int MAX_QUERY_COUNT = 512;
    private final int hashSlotMaxCount;
    private final int indexItemMaxCount;
    private final ReadWriteLock fileReadWriteLock;
    private final AtomicReference<IndexFile.IndexStatusEnum> fileStatus;
    private final AtomicLong beginTimestamp = new AtomicLong(-1L);
    private final AtomicLong endTimestamp = new AtomicLong(-1L);
    private final AtomicInteger hashSlotCount = new AtomicInteger(0);
    private final AtomicInteger indexItemCount = new AtomicInteger(0);
    private MappedFile mappedFile;
    private ByteBuffer byteBuffer;
    private MappedFile compactMappedFile;
    private FileSegment fileSegment;

    public IndexStoreFile(MessageStoreConfig storeConfig, long timestamp) throws IOException {
        this.hashSlotMaxCount = storeConfig.getTieredStoreIndexFileMaxHashSlotNum();
        this.indexItemMaxCount = storeConfig.getTieredStoreIndexFileMaxIndexNum();
        this.fileStatus = new AtomicReference<IndexFile.IndexStatusEnum>(IndexFile.IndexStatusEnum.UNSEALED);
        this.fileReadWriteLock = new ReentrantReadWriteLock();
        this.mappedFile = new DefaultMappedFile(Paths.get(storeConfig.getStorePathRootDir(), "tiered_index_file", String.valueOf(timestamp)).toString(), this.getItemPosition(this.indexItemMaxCount));
        this.byteBuffer = this.mappedFile.getMappedByteBuffer();
        this.beginTimestamp.set(timestamp);
        this.endTimestamp.set(this.byteBuffer.getLong(4));
        this.hashSlotCount.set(this.byteBuffer.getInt(20));
        this.indexItemCount.set(this.byteBuffer.getInt(24));
        this.flushNewMetadata(this.byteBuffer, this.indexItemMaxCount == this.indexItemCount.get() + 1);
    }

    public IndexStoreFile(MessageStoreConfig storeConfig, FileSegment fileSegment) {
        this.fileSegment = fileSegment;
        this.fileStatus = new AtomicReference<IndexFile.IndexStatusEnum>(IndexFile.IndexStatusEnum.UPLOAD);
        this.fileReadWriteLock = new ReentrantReadWriteLock();
        this.beginTimestamp.set(fileSegment.getMinTimestamp());
        this.endTimestamp.set(fileSegment.getMaxTimestamp());
        this.hashSlotCount.set(storeConfig.getTieredStoreIndexFileMaxHashSlotNum());
        this.indexItemCount.set(storeConfig.getTieredStoreIndexFileMaxIndexNum());
        this.hashSlotMaxCount = this.hashSlotCount.get();
        this.indexItemMaxCount = this.indexItemCount.get();
    }

    @Override
    public long getTimestamp() {
        return this.beginTimestamp.get();
    }

    @Override
    public long getEndTimestamp() {
        return this.endTimestamp.get();
    }

    public long getHashSlotCount() {
        return this.hashSlotCount.get();
    }

    public long getIndexItemCount() {
        return this.indexItemCount.get();
    }

    @Override
    public IndexFile.IndexStatusEnum getFileStatus() {
        return this.fileStatus.get();
    }

    protected String buildKey(String topic, String key) {
        return String.format("%s#%s", topic, key);
    }

    protected int hashCode(String keyStr) {
        int keyHash = keyStr.hashCode();
        return keyHash < 0 ? -keyHash : keyHash;
    }

    protected void flushNewMetadata(ByteBuffer byteBuffer, boolean end) {
        byteBuffer.putInt(0, !end ? -1127939447 : -1127939451);
        byteBuffer.putLong(4, this.beginTimestamp.get());
        byteBuffer.putLong(12, this.endTimestamp.get());
        byteBuffer.putInt(20, this.hashSlotCount.get());
        byteBuffer.putInt(24, this.indexItemCount.get());
    }

    protected int getSlotPosition(int slotIndex) {
        return 28 + slotIndex * 8;
    }

    protected int getSlotValue(int slotPosition) {
        return Math.max(this.byteBuffer.getInt(slotPosition), 0);
    }

    protected int getItemPosition(int itemIndex) {
        return 28 + this.hashSlotMaxCount * 8 + itemIndex * 32;
    }

    @Override
    public void start() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public AppendResult putKey(String topic, int topicId, int queueId, Set<String> keySet, long offset, int size, long timestamp) {
        if (StringUtils.isBlank((CharSequence)topic)) {
            return AppendResult.UNKNOWN_ERROR;
        }
        if (keySet == null || keySet.isEmpty()) {
            return AppendResult.SUCCESS;
        }
        try {
            this.fileReadWriteLock.writeLock().lock();
            if (!IndexFile.IndexStatusEnum.UNSEALED.equals((Object)this.fileStatus.get())) {
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            if (this.indexItemCount.get() + keySet.size() >= this.indexItemMaxCount) {
                this.fileStatus.set(IndexFile.IndexStatusEnum.SEALED);
                AppendResult appendResult = AppendResult.FILE_FULL;
                return appendResult;
            }
            for (String key : keySet) {
                int hashCode = this.hashCode(this.buildKey(topic, key));
                int slotPosition = this.getSlotPosition(hashCode % this.hashSlotMaxCount);
                int slotOldValue = this.getSlotValue(slotPosition);
                int timeDiff = (int)((timestamp - this.beginTimestamp.get()) / 1000L);
                IndexItem indexItem = new IndexItem(topicId, queueId, offset, size, hashCode, timeDiff, slotOldValue);
                int itemIndex = this.indexItemCount.incrementAndGet();
                this.byteBuffer.position(this.getItemPosition(itemIndex));
                this.byteBuffer.put(indexItem.getByteBuffer());
                this.byteBuffer.putInt(slotPosition, itemIndex);
                if (slotOldValue <= 0) {
                    this.hashSlotCount.incrementAndGet();
                }
                if (this.endTimestamp.get() < timestamp) {
                    this.endTimestamp.set(timestamp);
                }
                this.flushNewMetadata(this.byteBuffer, this.indexItemMaxCount == this.indexItemCount.get() + 1);
                log.trace("IndexStoreFile put key, timestamp: {}, topic: {}, key: {}, slot: {}, item: {}, previous item: {}, content: {}", new Object[]{this.getTimestamp(), topic, key, hashCode % this.hashSlotMaxCount, itemIndex, slotOldValue, indexItem});
            }
            Object object = AppendResult.SUCCESS;
            return object;
        }
        catch (Exception e) {
            log.error("IndexStoreFile put key error, topic: {}, topicId: {}, queueId: {}, keySet: {}, offset: {}, size: {}, timestamp: {}", new Object[]{topic, topicId, queueId, keySet, offset, size, timestamp, e});
        }
        finally {
            this.fileReadWriteLock.writeLock().unlock();
        }
        return AppendResult.UNKNOWN_ERROR;
    }

    @Override
    public CompletableFuture<List<IndexItem>> queryAsync(String topic, String key, int maxCount, long beginTime, long endTime) {
        switch (this.fileStatus.get()) {
            case UNSEALED: 
            case SEALED: {
                return this.queryAsyncFromUnsealedFile(this.buildKey(topic, key), maxCount, beginTime, endTime);
            }
            case UPLOAD: {
                return this.queryAsyncFromSegmentFile(this.buildKey(topic, key), maxCount, beginTime, endTime);
            }
        }
        return CompletableFuture.completedFuture(new ArrayList());
    }

    protected CompletableFuture<List<IndexItem>> queryAsyncFromUnsealedFile(String key, int maxCount, long beginTime, long endTime) {
        return CompletableFuture.supplyAsync(() -> {
            ArrayList<IndexItem> result = new ArrayList<IndexItem>();
            try {
                this.fileReadWriteLock.readLock().lock();
                if (!IndexFile.IndexStatusEnum.UNSEALED.equals((Object)this.fileStatus.get()) && !IndexFile.IndexStatusEnum.SEALED.equals((Object)this.fileStatus.get())) {
                    ArrayList<IndexItem> arrayList = result;
                    return arrayList;
                }
                if (this.mappedFile == null || !this.mappedFile.hold()) {
                    ArrayList<IndexItem> arrayList = result;
                    return arrayList;
                }
                int hashCode = this.hashCode(key);
                int slotPosition = this.getSlotPosition(hashCode % this.hashSlotMaxCount);
                int slotValue = this.getSlotValue(slotPosition);
                for (int left = 512; left > 0 && slotValue > 0 && slotValue <= this.indexItemCount.get(); --left) {
                    byte[] bytes = new byte[32];
                    ByteBuffer buffer = this.byteBuffer.duplicate();
                    buffer.position(this.getItemPosition(slotValue));
                    buffer.get(bytes);
                    IndexItem indexItem = new IndexItem(bytes);
                    if (hashCode == indexItem.getHashCode()) {
                        result.add(indexItem);
                        if (result.size() > maxCount) break;
                    }
                    slotValue = indexItem.getItemIndex();
                }
                log.debug("IndexStoreFile query from unsealed mapped file, timestamp: {}, result size: {}, key: {}, hashCode: {}, maxCount: {}, timestamp={}-{}", new Object[]{this.getTimestamp(), result.size(), key, hashCode, maxCount, beginTime, endTime});
            }
            catch (Exception e) {
                log.error("IndexStoreFile query from unsealed mapped file error, timestamp: {}, key: {}, maxCount: {}, timestamp={}-{}", new Object[]{this.getTimestamp(), key, maxCount, beginTime, endTime, e});
            }
            finally {
                this.fileReadWriteLock.readLock().unlock();
                this.mappedFile.release();
            }
            return result;
        }, MessageStoreExecutor.getInstance().bufferFetchExecutor);
    }

    protected CompletableFuture<List<IndexItem>> queryAsyncFromSegmentFile(String key, int maxCount, long beginTime, long endTime) {
        if (this.fileSegment == null || !IndexFile.IndexStatusEnum.UPLOAD.equals((Object)this.fileStatus.get())) {
            return CompletableFuture.completedFuture(Collections.emptyList());
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        int hashCode = this.hashCode(key);
        int slotPosition = this.getSlotPosition(hashCode % this.hashSlotMaxCount);
        CompletionStage future = ((CompletableFuture)this.fileSegment.readAsync(slotPosition, 8).thenCompose(slotBuffer -> {
            if (slotBuffer.remaining() < 8) {
                log.error("IndexStoreFile query from tiered storage return error slot buffer, key: {}, maxCount: {}, timestamp={}-{}", new Object[]{key, maxCount, beginTime, endTime});
                return CompletableFuture.completedFuture(null);
            }
            int indexPosition = slotBuffer.getInt();
            int indexTotalSize = Math.min(slotBuffer.getInt(), 28672);
            if (indexPosition <= 0 || indexTotalSize <= 0) {
                return CompletableFuture.completedFuture(null);
            }
            return this.fileSegment.readAsync(indexPosition, indexTotalSize);
        })).thenApply(itemBuffer -> {
            ArrayList<IndexItem> result = new ArrayList<IndexItem>();
            if (itemBuffer == null) {
                return result;
            }
            if (itemBuffer.remaining() % 28 != 0) {
                log.error("IndexStoreFile query from tiered storage return error item buffer, key: {}, maxCount: {}, timestamp={}-{}", new Object[]{key, maxCount, beginTime, endTime});
                return result;
            }
            int size = itemBuffer.remaining() / 28;
            byte[] bytes = new byte[28];
            for (int i = 0; i < size; ++i) {
                itemBuffer.get(bytes);
                IndexItem indexItem = new IndexItem(bytes);
                long storeTimestamp = (long)indexItem.getTimeDiff() + this.beginTimestamp.get();
                if (hashCode != indexItem.getHashCode() || beginTime > storeTimestamp || storeTimestamp > endTime || result.size() >= maxCount) continue;
                result.add(indexItem);
            }
            return result;
        });
        return ((CompletableFuture)future).whenComplete((result, throwable) -> {
            long costTime = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            if (throwable != null) {
                log.error("IndexStoreFile query from segment file, cost: {}ms, timestamp: {}, key: {}, hashCode: {}, maxCount: {}, timestamp={}-{}", new Object[]{costTime, this.getTimestamp(), key, hashCode, maxCount, beginTime, endTime, throwable});
            } else {
                String details = Optional.ofNullable(result).map(r -> r.stream().map(item -> String.format("%d-%d", item.getQueueId(), item.getOffset())).collect(Collectors.joining(", "))).orElse("");
                log.debug("IndexStoreFile query from segment file, cost: {}ms, timestamp: {}, result size: {}, ({}), key: {}, hashCode: {}, maxCount: {}, timestamp={}-{}", new Object[]{costTime, this.getTimestamp(), result != null ? result.size() : 0, details, key, hashCode, maxCount, beginTime, endTime});
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ByteBuffer doCompaction() {
        ByteBuffer buffer;
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            buffer = this.compactToNewFile();
            log.debug("IndexStoreFile do compaction, timestamp: {}, file size: {}, cost: {}ms", new Object[]{this.getTimestamp(), buffer.capacity(), stopwatch.elapsed(TimeUnit.MICROSECONDS)});
        }
        catch (Exception e) {
            log.error("IndexStoreFile do compaction, timestamp: {}, cost: {}ms", new Object[]{this.getTimestamp(), stopwatch.elapsed(TimeUnit.MICROSECONDS), e});
            return null;
        }
        try {
            this.fileReadWriteLock.writeLock().lock();
            this.fileStatus.set(IndexFile.IndexStatusEnum.SEALED);
        }
        catch (Exception e) {
            log.error("IndexStoreFile change file status to sealed error, timestamp={}", (Object)this.getTimestamp());
        }
        finally {
            this.fileReadWriteLock.writeLock().unlock();
        }
        return buffer;
    }

    protected String getCompactedFilePath() {
        return Paths.get(this.mappedFile.getFileName(), new String[0]).getParent().resolve("compacting").resolve(String.valueOf(this.getTimestamp())).toString();
    }

    protected ByteBuffer compactToNewFile() throws IOException {
        byte[] payload = new byte[32];
        ByteBuffer payloadBuffer = ByteBuffer.wrap(payload);
        int writePosition = 28 + this.hashSlotMaxCount * 8;
        int fileMaxLength = writePosition + 28 * this.indexItemCount.get();
        this.compactMappedFile = new DefaultMappedFile(this.getCompactedFilePath(), fileMaxLength);
        MappedByteBuffer newBuffer = this.compactMappedFile.getMappedByteBuffer();
        for (int i = 0; i < this.hashSlotMaxCount; ++i) {
            int slotPosition = this.getSlotPosition(i);
            int slotValue = this.getSlotValue(slotPosition);
            int writeBeginPosition = writePosition;
            while (slotValue > 0 && writePosition < fileMaxLength) {
                ByteBuffer buffer = this.byteBuffer.duplicate();
                buffer.position(this.getItemPosition(slotValue));
                buffer.get(payload);
                int newSlotValue = payloadBuffer.getInt(28);
                buffer.limit(28);
                newBuffer.position(writePosition);
                newBuffer.put(payload, 0, 28);
                log.trace("IndexStoreFile do compaction, write item, slot: {}, current: {}, next: {}", new Object[]{i, slotValue, newSlotValue});
                slotValue = newSlotValue;
                writePosition += 28;
            }
            int length = writePosition - writeBeginPosition;
            newBuffer.putInt(slotPosition, writeBeginPosition);
            newBuffer.putInt(slotPosition + 4, length);
            if (length <= 0) continue;
            log.trace("IndexStoreFile do compaction, write slot, slot: {}, begin: {}, length: {}", new Object[]{i, writeBeginPosition, length});
        }
        this.flushNewMetadata(newBuffer, true);
        newBuffer.flip();
        return newBuffer;
    }

    @Override
    public void shutdown() {
        try {
            this.fileReadWriteLock.writeLock().lock();
            this.fileStatus.set(IndexFile.IndexStatusEnum.SHUTDOWN);
            if (this.fileSegment != null && this.fileSegment instanceof PosixFileSegment) {
                ((PosixFileSegment)this.fileSegment).close();
            }
            if (this.mappedFile != null) {
                this.mappedFile.shutdown(TimeUnit.SECONDS.toMillis(10L));
            }
            if (this.compactMappedFile != null) {
                this.compactMappedFile.shutdown(TimeUnit.SECONDS.toMillis(10L));
            }
        }
        catch (Exception e) {
            log.error("IndexStoreFile shutdown failed, timestamp: {}, status: {}", new Object[]{this.getTimestamp(), this.fileStatus.get(), e});
        }
        finally {
            this.fileReadWriteLock.writeLock().unlock();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void destroy() {
        try {
            this.fileReadWriteLock.writeLock().lock();
            this.shutdown();
            switch (this.fileStatus.get()) {
                case UNSEALED: 
                case SEALED: 
                case SHUTDOWN: {
                    if (this.mappedFile != null) {
                        this.mappedFile.destroy(TimeUnit.SECONDS.toMillis(10L));
                    }
                    if (this.compactMappedFile != null) {
                        this.compactMappedFile.destroy(TimeUnit.SECONDS.toMillis(10L));
                    }
                    log.debug("IndexStoreService destroy local file, timestamp: {}, status: {}", (Object)this.getTimestamp(), (Object)this.fileStatus.get());
                    return;
                }
                case UPLOAD: {
                    log.warn("[BUG] IndexStoreService destroy remote file, timestamp: {}", (Object)this.getTimestamp());
                    return;
                }
            }
            return;
        }
        catch (Exception e) {
            log.error("IndexStoreService destroy failed, timestamp: {}, status: {}", new Object[]{this.getTimestamp(), this.fileStatus.get(), e});
            return;
        }
        finally {
            this.fileReadWriteLock.writeLock().unlock();
        }
    }
}

