/*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

import { type ComponentFixture, TestBed } from "@angular/core/testing";
import { MatMenuModule } from "@angular/material/menu";
import { Params } from "@angular/router";
import { RouterTestingModule } from "@angular/router/testing";
import { AgGridModule } from "ag-grid-angular";
import type { CellContextMenuEvent, ColDef, GridApi, RowNode, ValueGetterParams } from "ag-grid-community";
import { BehaviorSubject } from "rxjs";

import { type ContextMenuAction, GenericTableComponent, getColType, ContextMenuItem } from "./generic-table.component";

/**
 * TestData is the type of a row of data for the generic table component tests.
 */
interface TestData {
	bigIntProp: bigint;
	boolProp: boolean;
	complexProp: Record<PropertyKey, unknown>;
	id: number;
	lastUpdated: Date;
	name: string;
	propWithNoCol: unknown;
	regExpProp: RegExp;
	urlProp: URL;
	weirdProp: unknown;
}

/**
 * Generates some testing data for the generic table. This avoids reference
 * re-use and pollution.
 *
 * @returns Some testing data.
 */
function testingData(): Array<TestData> {
	return [
		{
			bigIntProp: BigInt(2),
			boolProp: true,
			complexProp: {
				properties: "and values",
				some: "other"
			},
			id: 1,
			lastUpdated: new Date(),
			name: "test item",
			propWithNoCol: "this property is never rendered because there is no column definition for it",
			regExpProp: /\.{1000}/g,
			urlProp: new URL("https://localhost"),
			weirdProp: null
		},
		{
			bigIntProp: BigInt(4),
			boolProp: false,
			complexProp: {
				properties: "and values",
				some: "other"
			},
			id: 2,
			lastUpdated: new Date(),
			name: "test item 2",
			propWithNoCol: "see above",
			regExpProp: /^$/,
			urlProp: new URL("http://apache.org/path?query#frag"),
			weirdProp: Symbol()
		}
	];
}

/**
 * Generates column definitions for data generated by {@link testingData}. This
 * avoids reference re-use and pollution.
 *
 * @returns Some testing columns.
 */
function testingCols(): Array<ColDef> {
	return [
		{
			field: "boolProp",
			headerName: "Boolean Property",
			hide: false
		},
		{
			field: "complexProp",
			headerName: "Complex Property",
			valueGetter: (p: ValueGetterParams): string => JSON.stringify(p.data.complexProp)
		},
		{
			field: "id",
			filter: "agNumberColumnFilter",
			headerName: "ID",
			hide: false
		},
		{
			field: "lastUpdated",
			filter: "agDateColumnFilter",
			headerName: "Last Updated",
			hide: true
		},
		{
			field: "name",
			headerName: "Name",
			hide: false
		},
		{
			field: "regExpProp",
			headerName: "Regular Expression Property",
			hide: false
		},
		{
			field: "urlProp",
			headerName: "URL Property",
			hide: false
		},
		{
			field: "weirdProp",
			headerName: "Weird Property",
			valueGetter: "weird",
		}
	];
}

describe("GenericTableComponent", () => {
	let component: GenericTableComponent<unknown>;
	let fixture: ComponentFixture<GenericTableComponent<unknown>>;
	let fuzzySearch: BehaviorSubject<string>;

	beforeEach(async () => {
		fuzzySearch = new BehaviorSubject("");
		await TestBed.configureTestingModule({
			declarations: [
				GenericTableComponent,

			],
			imports: [
				AgGridModule,
				RouterTestingModule,
				MatMenuModule
			]
		}).compileComponents();

		fixture = TestBed.createComponent<GenericTableComponent<unknown>>(GenericTableComponent);
		component = fixture.componentInstance;
		component.fuzzySearch = fuzzySearch;
		component.contextMenuItems = [
			{
				action: "test",
				name: "Test"
			}
		];
		component.data = testingData();
		component.cols = testingCols();
		fixture.detectChanges();
	});

	it("should create", () => {
		expect(component).toBeTruthy();
	});

	it("can tell if a context menu item is an action", () => {
		expect(component.isAction({href: "/core/dashboard", name: "Dashboard"})).toBeFalse();
		expect(component.isAction({href: "/core/dashboard", name: "Dashboard", newTab: true})).toBeFalse();
		expect(component.isAction({action: "do something", name: "Something"})).toBeTrue();
		expect(component.isAction({action: "do something", multiRow: true, name: "Something"})).toBeTrue();
	});

	it("makes all data pass the filter when there is no search box", () => {
		expect(component.filter({} as RowNode)).toBeTrue();
	});

	it("always says a context menu item is disabled with no selection", () => {
		expect(component.isDisabled({action: "anything", name: "whatever"})).toBeTrue();
		expect(component.isDisabled({action: "anything", disabled: ()=>false, name: "who cares"})).toBeTrue();
	});

	it("throws an error trying to emit a context menu action with no selection", ()=>{
		expect(()=>component.emitContextMenuAction("anything", false, new MouseEvent("click"))).toThrow();
		expect(()=>component.emitContextMenuAction("anything", true, new MouseEvent("click"))).toThrow();
		expect(()=>component.emitContextMenuAction("anything", undefined, new MouseEvent("click"))).toThrow();
	});

	it("opens the context menu on contextmenu events and closes when the user clicks outside of it or on a menu item", () => {
		expect(component.data.length).toBeGreaterThan(1);
		expect(component.showContextMenu).toBeFalse();
		component.onCellContextMenu({} as CellContextMenuEvent);
		expect(component.showContextMenu).toBeFalse();

		const oldItems = [...component.contextMenuItems];
		component.contextMenuItems = [];
		fixture.detectChanges();
		component.onCellContextMenu({
			event: new MouseEvent("contextmenu"),
		} as unknown as CellContextMenuEvent);
		expect(component.showContextMenu).toBeFalse();

		component.toggleMenu(new Event("click"));
		expect(component.showContextMenu).toBeFalse();

		component.contextMenuItems = oldItems;
		fixture.detectChanges();
		component.onCellContextMenu({
			event: new MouseEvent("contextmenu")
		} as unknown as CellContextMenuEvent);
		expect(component.showContextMenu).toBeTrue();

		component.clickOutside(new MouseEvent("click"));
		expect(component.showContextMenu).toBeFalse();

		component.onCellContextMenu({
			event: new MouseEvent("contextmenu")
		} as unknown as CellContextMenuEvent);
		expect(component.showContextMenu).toBeTrue();
		component.selected = component.data[0];
		component.emitContextMenuAction("some action", false, new MouseEvent("click"));
		expect(component.showContextMenu).toBeFalse();
	});

	it("(de)selects all rows", async () => {
		await fixture.whenStable();
		component.selectAll();
		expect(component.selectionCount).toBe(component.data.length);
		for (let i = 0; i < component.selectionCount; ++i) {
			expect(component.fullSelection[i]).toEqual(component.data[i]);
		}
		component.selectAll(true);
		expect(component.selectionCount).toBe(0);
		expect(component.fullSelection.length).toBe(0);
	});

	it("exposes its columns", async () => {
		await fixture.whenStable();
		expect(
			component.columns.map(c=>c.getDefinition())
		).toEqual(
			component.cols.map(c=>({... component.gridOptions.defaultColDef, ...c})).reverse()
		);
	});

	it("knows if it should filter", () => {
		expect(component.shouldFilter()).toBeTrue();
		component.fuzzySearch = undefined;
		expect(component.shouldFilter()).toBeFalse();
	});

	it("fuzzy filters", async () => {
		await fixture.whenStable();
		fuzzySearch.next("");
		component.toggleVisibility(new Event("toggle"), "lastUpdated");
		const api = component.gridOptions.api;
		if (!api) {
			return fail("api not set after rendering is complete");
		}

		const rows = api.getRenderedNodes();
		for (const row of rows) {
			expect(component.filter(row)).toBeTrue();
		}

		fuzzySearch.next("test item 2");
		for (const row of rows) {
			expect(component.filter(row)).toBe(row.data.name === "test item 2");
		}

		const now = new Date();
		const val = `${now.getFullYear()}`;
		fuzzySearch.next(val);
		for (const row of rows) {
			expect(component.filter(row)).toBeTrue();
		}

		fuzzySearch.next("this property is never rendered because there is no column definition for it");
		for (const row of rows) {
			expect(component.filter(row)).toBeFalse();
		}

		component.fuzzySearch = undefined;
		for (const row of rows) {
			expect(component.filter(row)).toBeTrue();
		}
	});

	it("doesn't panic when columns are requested before ag-grid loads", () => {
		expect(()=>component.columns).not.toThrow();
	});

	it("toggles column visibility", async () => {
		await fixture.whenStable();
		for (const col of component.columns) {
			if (!col.isVisible()) {
				component.toggleVisibility(new Event("toggle"), col.getId());
			}
		}

		for (const col of component.columns) {
			expect(col.isVisible()).toBeTrue();
			component.toggleVisibility(new Event("toggle"), col.getId());
			expect(col.isVisible()).toBeFalse();
		}

		component.toggleVisibility(new Event("toggle"), "not a real column ID");
		for (const col of component.columns) {
			expect(col.isVisible()).toBeFalse();
			component.toggleVisibility(new Event("toggle"), col.getId());
		}
		component.toggleVisibility(new Event("toggle"), "not a real column ID");
		for (const col of component.columns) {
			expect(col.isVisible()).toBeTrue();
		}
	});

	it("prevents default on Events", () => {
		const e = new Event("doesn't matter", {cancelable: true});
		expect(e.defaultPrevented).toBeFalse();
		component.preventDefault(e);
		expect(e.defaultPrevented).toBeTrue();
	});

	it("triggers a download of CSV data properly", async () => {
		component.selected = {};
		await fixture.whenStable();
		const spy = spyOn(component.gridOptions.api as GridApi, "exportDataAsCsv");
		component.download();
		expect(spy).toHaveBeenCalledWith({onlySelected: false});
		component.context = "test-context";
		component.download();
		expect(spy).toHaveBeenCalledWith({fileName: "test-context.csv", onlySelected: false});
	});

	it("checks if a menu action is disabled", async () => {
		let action: ContextMenuAction<unknown> = {action: "do something", name: "Do Something"};
		expect(component.isDisabled(action)).toBeTrue();
		component.selected = {};
		await fixture.whenStable();
		expect(component.isDisabled(action)).toBeFalse();

		action.disabled = (): boolean => true;
		expect(component.isDisabled(action)).toBeTrue();

		action.disabled = (): boolean => false;
		expect(component.isDisabled(action)).toBeFalse();

		action = {
			...action,
			multiRow: true
		};
		expect(component.isDisabled(action)).toBeFalse();

		action.disabled = (): boolean => true;
		expect(component.isDisabled(action)).toBeTrue();
	});

	it("calculates an href for a context menu link item", () => {
		const hrefValue = "href value";
		const menuLinkItem: ContextMenuItem<unknown> = {
			href: hrefValue,
			name: "test",
		};
		component.selected = {};
		expect(component.href(menuLinkItem)).toBe(hrefValue);
		menuLinkItem.href = (): string => hrefValue;
		expect(component.href(menuLinkItem)).toBe(hrefValue);
		component.selected = null;
		expect(component.href(menuLinkItem)).toBe("");
	});

	it("calculates a fragment for a context menu link item", () => {
		const fragmentValue = "fragment value";
		const menuLinkItem: ContextMenuItem<unknown> = {
			href: "inconsequential",
			name: "test",
		};
		component.selected = {};
		expect(component.fragment(menuLinkItem)).toBeNull();
		menuLinkItem.fragment = fragmentValue;
		expect(component.fragment(menuLinkItem)).toBe(fragmentValue);
		menuLinkItem.fragment = (): string => fragmentValue;
		expect(component.fragment(menuLinkItem)).toBe(fragmentValue);
		component.selected = null;
		expect(component.fragment(menuLinkItem)).toBeNull();
	});

	it("calculates query parameters for a context menu link item", () => {
		const qParamsValue = {query: "params"};
		const menuLinkItem: ContextMenuItem<unknown> = {
			href: "inconsequential",
			name: "test",
		};
		component.selected = {};
		expect(component.queryParameters(menuLinkItem)).toBeNull();
		menuLinkItem.queryParams = qParamsValue;
		expect(component.queryParameters(menuLinkItem)).toEqual(qParamsValue);
		menuLinkItem.queryParams = (): Params => qParamsValue;
		expect(component.queryParameters(menuLinkItem)).toEqual(qParamsValue);
		component.selected = null;
		expect(component.queryParameters(menuLinkItem)).toBeNull();
	});
});

describe("generic table utility functions", () => {
	it("gets the correct column type from a definition", () => {
		expect(getColType({})).toBe("string");
		expect(getColType({filter: true})).toBe("string");
		expect(getColType({filter: "textFilter"})).toBe("string");
		expect(getColType({filter: "agTextColumnFilter"})).toBe("string");
		expect(getColType({filter: "agNumberColumnFilter"})).toBe("number");
		expect(getColType({filter: "agDateColumnFilter"})).toBe("date");
		expect(getColType({filter: "unrecognized filter name"})).toBeNull();
		expect(getColType({filter: {}})).toBeNull();
	});
});
