/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 * 
 *  Copyright (C) 2000 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors: Rebecca Schulman <rebecka@eazel.com>
 *  
 */


/* medusa-index-service-test.c -- API for externals users of the medusa
   indexing service. */


#include <medusa-index-service.h>
#include <glib.h>
#include <stdio.h>
#include <time.h>

int main ()
{
  MedusaIndexingRequestResult reindex_request_result, index_service_status_result;
  time_t last_index_unix_time;
  struct tm *last_index_time;

  g_print ("Testing whether the reindexing service is available\n");
  index_service_status_result = medusa_index_service_is_available ();
  switch (index_service_status_result) {
  case MEDUSA_INDEXER_REQUEST_OK:
          printf ("The reindexing service is available\n");
          break;
  case MEDUSA_INDEXER_ERROR_BUSY:
          printf ("The reindexing service is currently busy\n");
          break;
  case MEDUSA_INDEXER_ERROR_NO_RESPONSE:
          printf ("The reindexing service is not responding\n");
          break;
  case MEDUSA_INDEXER_ERROR_INVALID_RESPONSE:
          printf ("The reindexing service provided an invalid response\n");
          break;
  default:
          printf ("An invalid value was returned to the reindexing service availability request\n");
          g_assert_not_reached ();
  }

  reindex_request_result = medusa_index_service_request_reindex ();

  switch (reindex_request_result) {
  case MEDUSA_INDEXER_REQUEST_OK:
          printf ("A reindex request succeeded.\n");
          break;
  case MEDUSA_INDEXER_ERROR_BUSY:
          printf ("A reindex request failed because the indexer was busy\n");
          break;
  case MEDUSA_INDEXER_ERROR_NO_RESPONSE:
          printf ("The reindexing service did not response to a reindexing request\n");
          break;
  case MEDUSA_INDEXER_ERROR_INVALID_RESPONSE:
          printf ("The reindexing service provided an invalid response\n");
          break;
  default:
          printf ("An invalid value was returned to the reindexing request\n");
          g_assert_not_reached ();
  }

  if (reindex_request_result != index_service_status_result) {
          printf ("An error occurred in either the index service availability request or in the reindex request\n");
  }
  
  if (reindex_request_result == MEDUSA_INDEXER_REQUEST_OK) {
          index_service_status_result = medusa_index_service_is_available ();
          if (index_service_status_result == MEDUSA_INDEXER_ERROR_BUSY) {
                  printf ("A subsequent status check confirmed that the index is now busy\n");
          }
          else {
                  printf ("While a reindex request succeeded, the status of the reindexing service is not busy.  The availability was: ");
                  switch (index_service_status_result) {
                  case MEDUSA_INDEXER_REQUEST_OK:
                          printf ("Service available\n");
                          break;
                  case MEDUSA_INDEXER_ERROR_NO_RESPONSE:
                          printf ("No response from the indexer\n");
                          break;
                  case MEDUSA_INDEXER_ERROR_INVALID_RESPONSE:
                          printf ("Indexer returned invalid response\n");
                          break;
                  default:
                          printf ("Invalid return value was received from the function.\n");
                          g_assert_not_reached ();
                  }
          }
  }

  last_index_unix_time = medusa_index_service_get_last_index_update_time ();
  last_index_time = localtime (&last_index_unix_time);
  if (last_index_unix_time == -1) {
          printf ("Indexing error occurred\n");
  }
  printf ("index was last updated at %d/%d/%d, %d:%d:%d\n", 
	  last_index_time->tm_mon + 1, last_index_time->tm_mday,
	  last_index_time->tm_year + 1900, 
	  last_index_time->tm_hour, last_index_time->tm_min,
	  last_index_time->tm_sec);
  return 0;
}
