/*
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "htmlboxtextaccessible.h"
#include "layout/htmlboxtext.h"

static void           html_box_text_accessible_class_init          (HtmlBoxTextAccessibleClass *klass);
static void           html_box_text_accessible_text_interface_init (AtkTextIface        *iface);
static gchar*         html_box_text_accessible_get_text            (AtkText             *text,
                                                              gint                start_offset,
                                                              gint                end_offset);
static gchar*         html_box_text_accessible_get_text_after_offset 
                                                             (AtkText             *text,
                                                              gint                offset,
                                                              AtkTextBoundary     boundary_type,
                                                              gint                *start_offset,
                                                              gint                *end_offset);
static gchar*         html_box_text_accessible_get_text_at_offset  (AtkText             *text,
                                                              gint                offset,
                                                              AtkTextBoundary     boundary_type,
                                                              gint                *start_offset,
                                                              gint                *end_offset);
static gchar*         html_box_text_accessible_get_text_before_offset 
                                                             (AtkText             *text,
                                                              gint                offset,
                                                              AtkTextBoundary     boundary_type,
                                                              gint                *start_offset,
                                                              gint                *end_offset);
static gunichar       html_box_text_accessible_get_character_at_offset 
                                                              (AtkText            *text,
                                                               gint               offset);
static gint           html_box_text_accessible_get_character_count  (AtkText            *text);
static gint           html_box_text_accessible_get_caret_offset     (AtkText            *text);
static gboolean       html_box_text_accessible_set_caret_offset     (AtkText            *text,
                                                               gint               offset);
static gint           html_box_text_accessible_get_offset_at_point  (AtkText            *text,
                                                               gint               x,
                                                               gint               y,
                                                               AtkCoordType       coords);
static void           html_box_text_accessible_get_character_extents (AtkText           *text,
                                                                gint              offset,
                                                                gint              *x,
                                                                gint              *y,
                                                                gint              *width,
                                                                gint              *height,
                                                                AtkCoordType      coords);
static AtkAttributeSet* 
                      html_box_text_accessible_get_run_attributes    (AtkText           *text,
                                                                gint              offset,
                                                                gint              *start_offset,
                                                                gint              *end_offset);
static AtkAttributeSet* 
                      html_box_text_accessible_get_default_attributes (AtkText          *text);
static gint           html_box_text_accessible_get_n_selections      (AtkText           *text);
static gchar*         html_box_text_accessible_get_selection         (AtkText           *text,
                                                                gint              selection_num,
                                                                gint              *start_pos,
                                                                gint              *end_pos);
static gboolean       html_box_text_accessible_add_selection         (AtkText           *text,
                                                                gint              start_pos,
                                                                gint              end_pos);
static gboolean       html_box_text_accessible_remove_selection      (AtkText           *text,
                                                                gint              selection_num);
static gboolean       html_box_text_accessible_set_selection         (AtkText           *text,
                                                                gint              selection_num,
                                                                gint              start_pos,
                                                                gint              end_pos);
static gchar*         get_text_near_offset                     (AtkText           *text,
                                                                GailOffsetType    function,
                                                                AtkTextBoundary   boundary_type,
                                                                gint              offset,
                                                                gint              *start_offset,
                                                                gint              *end_offset);

GType
html_box_text_accessible_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo tinfo = {
			sizeof (HtmlBoxTextAccessibleClass),
			(GBaseInitFunc) NULL, /* base init */
			(GBaseFinalizeFunc) NULL, /* base finalize */
			(GClassInitFunc) html_box_text_accessible_class_init,
			(GClassFinalizeFunc) NULL, /* class finalize */
			NULL, /* class data */
			sizeof (HtmlBoxTextAccessible),
			0, /* nb preallocs */
			(GInstanceInitFunc) NULL, /* instance init */
			NULL /* value table */
		};

		static const GInterfaceInfo atk_text_info = {
			(GInterfaceInitFunc) html_box_text_accessible_text_interface_init,
			(GInterfaceFinalizeFunc) NULL,
			NULL
		};
		type = g_type_register_static (HTML_TYPE_BOX_ACCESSIBLE, "HtmlBoxTextAccessible", &tinfo, 0);
		g_type_add_interface_static (type, ATK_TYPE_TEXT, &atk_text_info);      
	}

	return type;
}

AtkObject*
html_box_text_accessible_new (GObject *obj)
{
  gpointer object;
  AtkObject *atk_object;
  HtmlBoxTextAccessible *text;

  g_return_val_if_fail (HTML_IS_BOX_TEXT (obj), NULL);
  object = g_object_new (HTML_TYPE_BOX_TEXT_ACCESSIBLE, NULL);
  atk_object = ATK_OBJECT (object);
  text = HTML_BOX_TEXT_ACCESSIBLE (object);

  atk_object_initialize (atk_object, obj);
  text->textutil = gail_text_util_new ();
#if 0
  gail_text_util_text_setup (text->textutil,
                             HTML_BOX_TEXT (obj)->master->text);
#endif

  atk_object->role =  ATK_ROLE_TEXT;
  return atk_object;
}

static void
html_box_text_accessible_class_init (HtmlBoxTextAccessibleClass *klass)
{
}

static void
html_box_text_accessible_text_interface_init (AtkTextIface *iface)
{
  g_return_if_fail (iface != NULL);

  iface->get_text = html_box_text_accessible_get_text;
  iface->get_text_after_offset = html_box_text_accessible_get_text_after_offset;
  iface->get_text_at_offset = html_box_text_accessible_get_text_at_offset;
  iface->get_text_before_offset = html_box_text_accessible_get_text_before_offset;
  iface->get_character_at_offset = html_box_text_accessible_get_character_at_offset;
  iface->get_character_count = html_box_text_accessible_get_character_count;
  iface->get_caret_offset = html_box_text_accessible_get_caret_offset;
  iface->set_caret_offset = html_box_text_accessible_set_caret_offset;
  iface->get_offset_at_point = html_box_text_accessible_get_offset_at_point;
  iface->get_character_extents = html_box_text_accessible_get_character_extents;
  iface->get_n_selections = html_box_text_accessible_get_n_selections;
  iface->get_selection = html_box_text_accessible_get_selection;
  iface->add_selection = html_box_text_accessible_add_selection;
  iface->remove_selection = html_box_text_accessible_remove_selection;
  iface->set_selection = html_box_text_accessible_set_selection;
  iface->get_run_attributes = html_box_text_accessible_get_run_attributes;
  iface->get_default_attributes = html_box_text_accessible_get_default_attributes;
}

static gchar*
html_box_text_accessible_get_text (AtkText *text,
                                   gint    start_offset,
                                   gint    end_offset)
{
  HtmlBoxTextAccessible *text_accessible;
  GtkTextBuffer *buffer;
  GtkTextIter start, end;

  g_return_val_if_fail (HTML_IS_BOX_TEXT_ACCESSIBLE (text), NULL);
  text_accessible = HTML_BOX_TEXT_ACCESSIBLE (text);
  g_return_val_if_fail (text_accessible->textutil, NULL);

  buffer = text_accessible->textutil->buffer;
  gtk_text_buffer_get_iter_at_offset (buffer, &start, start_offset);
  gtk_text_buffer_get_iter_at_offset (buffer, &end, end_offset);

  return gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
}

static gchar*
html_box_text_accessible_get_text_after_offset (AtkText         *text,
                                                gint            offset,
                                                AtkTextBoundary boundary_type,
                                                gint            *start_offset,
                                                gint            *end_offset)
{
  return get_text_near_offset (text, GAIL_AFTER_OFFSET,
                               boundary_type, offset, 
                               start_offset, end_offset);
}

static gchar*
html_box_text_accessible_get_text_at_offset (AtkText         *text,
                                             gint            offset,
                                             AtkTextBoundary boundary_type,
                                             gint            *start_offset,
                                             gint            *end_offset)
{
  return get_text_near_offset (text, GAIL_AT_OFFSET,
                               boundary_type, offset, 
                               start_offset, end_offset);
}

static gchar*
html_box_text_accessible_get_text_before_offset (AtkText         *text,
                                                 gint            offset,
                                                 AtkTextBoundary boundary_type,
                                                 gint            *start_offset,
                                                 gint            *end_offset)
{
  return get_text_near_offset (text, GAIL_BEFORE_OFFSET,
                               boundary_type, offset, 
                               start_offset, end_offset);
}

static gunichar
html_box_text_accessible_get_character_at_offset (AtkText *text,
                                                  gint    offset)
{
  HtmlBoxTextAccessible *text_accessible;
  GtkTextIter start, end;
  GtkTextBuffer *buffer;
  gchar *string;
  gchar *index;

  g_return_val_if_fail (HTML_IS_BOX_TEXT_ACCESSIBLE (text), 0);
  text_accessible = HTML_BOX_TEXT_ACCESSIBLE (text);
  buffer = text_accessible->textutil->buffer;
  gtk_text_buffer_get_start_iter (buffer, &start);
  gtk_text_buffer_get_end_iter (buffer, &end);
  string = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);
  index = g_utf8_offset_to_pointer (string, offset);
  g_free (string);

  return g_utf8_get_char (index);
}

static gint
html_box_text_accessible_get_character_count (AtkText *text)
{
  GtkTextBuffer *buffer;
  HtmlBoxTextAccessible *text_accessible;

  g_return_val_if_fail (HTML_IS_BOX_TEXT_ACCESSIBLE (text), 0);
  text_accessible = HTML_BOX_TEXT_ACCESSIBLE (text);
  g_return_val_if_fail (text_accessible->textutil, 0);
  buffer = text_accessible->textutil->buffer;
  return gtk_text_buffer_get_char_count (buffer);
}

static gint
html_box_text_accessible_get_caret_offset (AtkText *text)
{
  HtmlBoxTextAccessible *text_accessible;
  GtkTextBuffer *buffer;
  GtkTextMark *cursor_mark;
  GtkTextIter cursor_itr;

  g_return_val_if_fail (HTML_IS_BOX_TEXT_ACCESSIBLE (text), 0);
  text_accessible = HTML_BOX_TEXT_ACCESSIBLE (text);
  g_return_val_if_fail (text_accessible->textutil, 0);
  buffer = text_accessible->textutil->buffer;
  cursor_mark = gtk_text_buffer_get_insert (buffer);
  gtk_text_buffer_get_iter_at_mark (buffer, &cursor_itr, cursor_mark);
  return gtk_text_iter_get_offset (&cursor_itr);
}

static gboolean
html_box_text_accessible_set_caret_offset (AtkText *text,
                                           gint    offset)
{
  HtmlBoxTextAccessible *text_accessible;
  GtkTextBuffer *buffer;
  GtkTextIter pos_itr;

  g_return_val_if_fail (HTML_IS_BOX_TEXT_ACCESSIBLE (text), FALSE);
  text_accessible = HTML_BOX_TEXT_ACCESSIBLE (text);
  g_return_val_if_fail (text_accessible->textutil, FALSE);
  buffer = text_accessible->textutil->buffer;
  gtk_text_buffer_get_iter_at_offset (buffer,  &pos_itr, offset);
  gtk_text_buffer_move_mark_by_name (buffer, "insert", &pos_itr);
  return TRUE;
}

static gint
html_box_text_accessible_get_offset_at_point (AtkText      *text,
                                              gint         x,
                                              gint         y,
                                              AtkCoordType coords)
{
  return -1;
}

static void
html_box_text_accessible_get_character_extents (AtkText      *text,
                                                gint         offset,
                                                gint         *x,
                                                gint         *y,
                                                gint         *width,
                                                gint         *height,
                                                AtkCoordType coords)
{
  return;
}

static AtkAttributeSet*
html_box_text_accessible_get_run_attributes (AtkText *text,
                                             gint    offset,
                                             gint    *start_offset,
                                             gint    *end_offset)
{
  return NULL;
}

static AtkAttributeSet*
html_box_text_accessible_get_default_attributes (AtkText *text)
{
  return NULL;
}

static gint
html_box_text_accessible_get_n_selections (AtkText *text)
{
  return 0;
}

static gchar*
html_box_text_accessible_get_selection (AtkText *text,
                                        gint    selection_num,
                                        gint    *start_pos,
                                        gint    *end_pos)
{
  return NULL;
}

static gboolean
html_box_text_accessible_add_selection (AtkText *text,
                                        gint    start_pos,
                                        gint    end_pos)
{
  return FALSE;
}

static gboolean
html_box_text_accessible_remove_selection (AtkText *text,
                                           gint    selection_num)
{
  return FALSE;
}

static gboolean
html_box_text_accessible_set_selection (AtkText *text,
                                        gint    selection_num,
                                        gint    start_pos,
                                        gint    end_pos)
{
  return FALSE;
}

static gchar*
get_text_near_offset (AtkText          *text,
                      GailOffsetType   function,
                      AtkTextBoundary  boundary_type,
                      gint             offset,
                      gint             *start_offset,
                      gint             *end_offset)
{
  return gail_text_util_get_text (HTML_BOX_TEXT_ACCESSIBLE (text)->textutil, NULL,
                                  function, boundary_type, offset, 
                                  start_offset, end_offset);
}
