import dbus, dbus.glib
from gettext import gettext as _
import deskbar, deskbar.core.Indexer, deskbar.interfaces.Match, deskbar.interfaces.Module, deskbar.core.Utils
from deskbar.defs import VERSION
import deskbar.handlers.gdmclient
import gtk, gnome, gnome.ui

HANDLERS = ["GdmHandler"]

class GpmMatch(deskbar.interfaces.Match):
	def __init__(self, name=None, **args):
		deskbar.interfaces.Match.__init__(self, name=name, category="actions", **args)
		bus = dbus.Bus(dbus.Bus.TYPE_SESSION)
		obj = bus.get_object('org.gnome.PowerManager', '/org/gnome/PowerManager')
		self._gpm = dbus.Interface (obj, "org.gnome.PowerManager")


class SuspendMatch(GpmMatch):
	def __init__(self, name=None, **args):
		GpmMatch.__init__(self, name, category="actions")
		self.icon = "gpm-suspend-to-ram.png"

	def action(self, text=None):
		try:
			self._gpm.Suspend()
		except dbus.DBusException:
			# this will trigger a method timeout exception.
			# As a workaround we swallow it silently
			pass

	def get_verb(self):
		return _("Suspend the machine")

class HibernateMatch(GpmMatch):
	def __init__(self, name=None, **args):
		GpmMatch.__init__(self, name)
		self.icon = "gpm-suspend-to-disk.png"

	def action(self, text=None):
		try:
			self._gpm.Hibernate()
		except dbus.DBusException:
			# this will trigger a method timeout exception.
			# As a workaround we swallow it silently
			pass

	def get_verb(self):
		return _("Hibernate the machine")

class ShutdownMatch(GpmMatch):
	def __init__(self, name=None, **args):
		GpmMatch.__init__(self, name)
		self.icon = gtk.STOCK_QUIT

	def action(self, text=None):
		try:
			self._gpm.Shutdown()
		except dbus.DBusException:
			# this will trigger a method timeout exception.
			# As a workaround we swallow it silently
			pass

	def get_verb(self):
		return _("Shutdown the machine")

class LockScreenMatch(deskbar.interfaces.Match):
	def __init__(self, name=None, **args):
		deskbar.interfaces.Match.__init__(self, name=name, **args)
		self.icon = gtk.STOCK_FULLSCREEN
		
		bus = dbus.Bus(dbus.Bus.TYPE_SESSION)
		obj = bus.get_object('org.gnome.ScreenSaver', '/org/gnome/ScreenSaver')
		# FIXME : This timeouts ?
		self._scrsvr = dbus.Interface (obj, "org.gnome.ScreenSaver")

	def action(self, text=None):
		try:
			self._scrsvr.Lock()
		except dbus.DBusException:
			# this will trigger a method timeout exception.
			# As a workaround we swallow it silently
			pass

	def get_category(self):
		return "actions"

	def get_verb(self):
		return _("Lock the screen")

class GdmMatch(deskbar.interfaces.Match):
	def __init__(self, name, **args):
		deskbar.interfaces.Match.__init__(self, name=name, category="actions", **args)
		self.logout_reentrance = 0
	
	def request_logout(self):
		if self.logout_reentrance == 0:
			self.logout_reentrance += 1

			client = gnome.ui.master_client()
			if client:
				client.request_save(gnome.ui.SAVE_GLOBAL,
					True, # Shutdown?
					gnome.ui.INTERACT_ANY,
					True, # Fast?
					True) # Global?

				self.logout_reentrance -= 1
			
class GdmShutdownMatch(GdmMatch):
	def __init__(self, **args):
		GdmMatch.__init__(self, _("Shut Down"), **args)
		
	def action(self, text=None):
		deskbar.handlers.gdmclient.set_logout_action(deskbar.handlers.gdmclient.LOGOUT_ACTION_SHUTDOWN)
		self.request_logout()
		
	def get_verb(self):
		return _("Turn off the computer")

class GdmLogoutMatch(GdmMatch):
	def __init__(self, **args):
		GdmMatch.__init__(self, _("Log Out"), **args)
		
	def action(self, text=None):
		deskbar.handlers.gdmclient.set_logout_action(deskbar.handlers.gdmclient.LOGOUT_ACTION_NONE)
		self.request_logout()
		
	def get_verb(self):
		return _("Log out")
		
class GdmRebootMatch(GdmMatch):
	def __init__(self, **args):
		GdmMatch.__init__(self, _("Restart"), **args)
		
	def action(self, text=None):
		deskbar.handlers.gdmclient.set_logout_action(deskbar.handlers.gdmclient.LOGOUT_ACTION_REBOOT)
		self.request_logout()
		
	def get_verb(self):
		return _("Restart the computer")
		
class GdmSwitchUserMatch(GdmMatch):
	def __init__(self, **args):
		GdmMatch.__init__(self, _("Switch User"), **args)
		
	def action(self, text=None):
		deskbar.handlers.gdmclient.new_login()
		
	def get_verb(self):
		return _("Switch User")
				
class GdmHandler(deskbar.interfaces.Module):
	
	INFOS = {'icon':  deskbar.core.Utils.load_icon("gpm-suspend-to-ram.png"),
			 "name": _("Computer Actions"),
			 "description": _("Logoff, shutdown, restart, suspend and related actions."),
			 "version": VERSION}
	
	def __init__(self):
		deskbar.interfaces.Module.__init__(self)	
		self.indexer = deskbar.core.Indexer.Indexer()
		
	def initialize(self):
		for klass in (GdmShutdownMatch,GdmSwitchUserMatch,GdmRebootMatch,GdmLogoutMatch):
			match = klass()
			self.indexer.add(match.get_verb(), match)
		
		self.init_gpm_matches()
		self.init_screensaver_matches()

	def init_screensaver_matches(self):
		try:
			bus = dbus.Bus(dbus.Bus.TYPE_SESSION)
			obj = bus.get_object('org.gnome.ScreenSaver', '/org/gnome/ScreenSaver')
			scrsvr = dbus.Interface (obj, "org.gnome.ScreenSaver")
			self.indexer.add(_("Lock"), LockScreenMatch())
			return True
		except dbus.DBusException:
			return False

	def init_gpm_matches(self):
		try:
			bus = dbus.Bus(dbus.Bus.TYPE_SESSION)
			obj = bus.get_object('org.gnome.PowerManager', '/org/gnome/PowerManager')
			gpm = dbus.Interface (obj, "org.gnome.PowerManager")
			if gpm.canSuspend():
				self.indexer.add(_("Suspend"), SuspendMatch())
			if gpm.canHibernate():
				self.indexer.add(_("Hibernate"), HibernateMatch())
			if gpm.canShutdown():
				self.indexer.add(_("Shutdown"), ShutdownMatch())
		except dbus.DBusException:
			return False
			
	def query(self, query):
		matches = self.indexer.look_up(query)
		self.set_priority_for_matches( matches )
		self._emit_query_ready(query, matches )
