/* GStreamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* Based on xqcam.c by Paul Chinn <loomer@svpal.org>:
 *
 *  XWindow Support for QuickCam
 *  by Paul Chinn <loomer@svpal.org>
 *  Modified by Scott Laird <scott@laird.com>
 *
 * xqcam.c is under the X license.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

/* gcc -ansi -pedantic on GNU/Linux causes warnings and errors
 * unless this is defined:
 * warning: #warning "Files using this header must be compiled with _SVID_SOURCE or _XOPEN_SOURCE"
 */
#ifndef _XOPEN_SOURCE
#  define _XOPEN_SOURCE 1
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xproto.h>
#include <X11/Xmd.h>

#include "gstxwindow.h"
#include "gstximage.h"
#ifdef HAVE_XVIDEO
#include "gstxvimage.h"
#endif

GstXWindow *
_gst_xwindow_new (gint width, gint height, XID parent)
{
  XGCValues values;
  GstXWindow *new;
  int i;
  gboolean toplevel = (parent == 0);

  new = g_new0 (GstXWindow, 1);

  new->width = width;
  new->height = height;

  new->die = FALSE;
  new->imagecount = 0;

  new->disp = XOpenDisplay (NULL);
  if (!new->disp) {
    g_free (new);
    return NULL;
  }

  new->screen = DefaultScreenOfDisplay (new->disp);
  new->screen_num = DefaultScreen (new->disp);
  new->white = XWhitePixel (new->disp, new->screen_num);
  new->black = XBlackPixel (new->disp, new->screen_num);

  new->root = DefaultRootWindow (new->disp);

  if (parent == 0)
    parent = DefaultRootWindow (new->disp);
  new->win = XCreateSimpleWindow (new->disp, parent, 
		                  0, 0, new->width, new->height, 
                                  0, 0, new->black);

  if (!new->win) {
    XCloseDisplay (new->disp);
    g_free (new);
    return NULL;
  }

  XSelectInput (new->disp, new->win, ExposureMask | StructureNotifyMask);

  new->gc = XCreateGC (new->disp, new->win, 0, &values);
  new->depth = DefaultDepthOfScreen (new->screen);

  if (toplevel) {
    XMapRaised (new->disp, new->win);
  } else {
    XMapWindow (new->disp, new->win);
  }

  new->image_formats = g_ptr_array_new();

#ifdef HAVE_XVIDEO
  _gst_xvimage_add_formats(new);
#endif
  _gst_ximage_add_formats(new);

  for(i=0;i<new->image_formats->len;i++){
    GstImageFormat *image_format;

    image_format = g_ptr_array_index(new->image_formats,i);

    GST_DEBUG ("%d: %d %s",i,image_format->type,
	gst_caps_to_string(image_format->caps));
  }

  return new;
}

static inline void
gst_xwindow_real_destroy (GstXWindow * window)
{
  if (window->die && window->imagecount == 0) {
    XFreeGC (window->disp, window->gc);
    XCloseDisplay (window->disp);
    g_free (window);
  }
}

void
_gst_xwindow_destroy (GstXWindow * window)
{
  window->die = TRUE;
  XDestroyWindow (window->disp, window->win);
  gst_xwindow_real_destroy (window);
}

void
_gst_xwindow_ref (GstXWindow * window)
{
  window->imagecount++;
}

void
_gst_xwindow_unref (GstXWindow * window)
{
  if (window->imagecount > 0) {
    window->imagecount--;
  } else {
    g_warning ("XWindow ressource count leakage");
  }
  gst_xwindow_real_destroy (window);
}

void
_gst_xwindow_resize (GstXWindow * window, gint width, gint height)
{
  XResizeWindow (window->disp, window->win, width, height);
}
