/* $Id: gdict-app.c,v 1.86 2004/11/24 22:15:27 vnoel Exp $ */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict main window
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>

#include <gnome.h>

#include "dict.h"
#include "gdict-about.h"
#include "gdict-pref.h"
#include "gdict-app.h"
#include "gdict-defbox.h"
#include "gdict-speller.h"
#include "gdict-utils.h"

#ifdef HAVE_GNOME_PRINT
#  include <libgnomeprint/gnome-printer-dialog.h>
#endif /* HAVE_GNOME_PRINT */

GtkWidget *gdict_app;
GtkUIManager *ui;
GtkActionGroup *menu_actions;
GtkActionGroup *edit_actions;
GtkWidget *gdict_appbar;
GtkWidget *gnome_word_entry;
GtkWidget *word_entry;
GDictDefbox *defbox;
gboolean  gail_loaded = FALSE;

static gchar *find_text = NULL;
static gboolean search_from_beginning = TRUE;

#ifdef HAVE_GNOME_PRINT
GnomePrinter *gdict_printer = NULL;
#endif /* HAVE_GNOME_PRINT */

dict_context_t *context;


/* gdict_init_context
 *
 * Initialises the context object with information on the current server
 *
 * Retrurns 0 on success, -1 if the server could not be found
 */
gint
gdict_init_context (void) 
{
	if (context) 
		dict_context_destroy (context);

	context = dict_context_new (gdict_pref.server, gdict_pref.port);
	context->command = dict_disconnect_command_new ();

	defbox->context = context;

#ifdef ENABLE_IPV6
	if (context->host6info)
		return 0;
#endif

	if (context->hostinfo)
		return 0;
	else
		return -1;
}

void
gdict_app_clear (void) 
{
	gchar *word;

	gdict_defbox_clear (defbox);

	/* Update entry */
	if ((word = gdict_defbox_get_word (defbox))) {
		gtk_entry_set_text (GTK_ENTRY(word_entry), word);
		gtk_editable_select_region (GTK_EDITABLE(word_entry), 0, strlen(word));
	}
}

static void
spell_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
	gtk_statusbar_pop (GTK_STATUSBAR(gdict_appbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR(gdict_appbar), 0, _("Spell-checking..."));
}

static void
spell_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
	gtk_statusbar_pop (GTK_STATUSBAR (gdict_appbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR (gdict_appbar), 0, _("Spell check done"));
}

static void
spell_not_found_cb (GtkWidget *widget, gpointer data) 
{
	gtk_statusbar_pop (GTK_STATUSBAR (gdict_appbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR (gdict_appbar), 0, _("No matches found"));
}

static void
socket_error_cb (GtkWidget *widget, const gchar *message, gpointer data) 
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
						     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
						     _("Connexion error"), message); 
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

gint
gdict_spell (gchar *text, gboolean pattern) 
{
	static GDictSpeller *speller;

	g_return_val_if_fail (text != NULL, 0);

	if (!speller) {
		speller = GDICT_SPELLER (gdict_speller_new ());

		speller->context = context;
		gdict_speller_reset_strat (speller);

		g_signal_connect (speller, "word_lookup_start",
				  G_CALLBACK (spell_lookup_start_cb), NULL);
		g_signal_connect (speller, "word_lookup_done",
				  G_CALLBACK (spell_lookup_done_cb), NULL);
		g_signal_connect (speller, "word_not_found",
				  G_CALLBACK (spell_not_found_cb), NULL);
		g_signal_connect (speller, "socket_error",
				  G_CALLBACK (socket_error_cb), NULL);
	}

	if (pattern) 
		speller->strat = "re";
	else
		speller->strat = gdict_pref.dfl_strat;

	gtk_window_present (GTK_WINDOW (speller));

	if (gdict_speller_lookup (speller, text) == -1)
		return -1;

	return 0;
}

static gboolean
is_pattern (gchar *text) 
{
	if (strpbrk (text, "*|{}()[]"))
		return TRUE;
	else
		return FALSE;
}

void
gdict_app_do_lookup (gchar *text) 
{
	gint retval;
	gchar *word_entry_text;

	g_return_if_fail (text != NULL);

	/* If needed update the the word_entry */
	word_entry_text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
	if ((word_entry_text == NULL) ||
	    (*word_entry_text == 0)   ||
	    (strcmp (word_entry_text, text) != 0))
	{
		gnome_entry_prepend_history (GNOME_ENTRY (gnome_word_entry), 1, text);
		gtk_entry_set_text (GTK_ENTRY (word_entry), text);
	}

	g_free (word_entry_text);

	if (gdict_pref.smart && is_pattern (text)) {
		retval = gdict_spell (text, TRUE);
	}
	else {
		retval = gdict_defbox_lookup (defbox, text);
		if (!retval)
			gtk_widget_show (gdict_app);
	}

	if (retval) {
		gdict_not_online ();
	}
}

static void
lookup_entry (void) 
{
	gchar *text = NULL;

	text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
	gdict_app_do_lookup (text);
	g_free (text);
}

static void
lookup_defbox (void) 
{
	gchar *text = NULL;

	g_signal_emit_by_name (G_OBJECT (defbox), "copy_clipboard", 0);
	gtk_entry_set_text (GTK_ENTRY (word_entry), "");
	g_signal_emit_by_name (G_OBJECT (word_entry), "paste_clipboard");
	text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
	gnome_entry_prepend_history (GNOME_ENTRY (gnome_word_entry), 1, text);
	gdict_app_do_lookup (text);
	g_free (text);
}

static void
def_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
	gtk_window_set_title (GTK_WINDOW (gdict_app), _("Dictionary"));
	gtk_statusbar_push (GTK_STATUSBAR (gdict_appbar), 0, _("Looking up word..."));
}

static void
def_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
	gchar *word, *title;

	word = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
	title = g_strconcat (word, " - ", _("Dictionary"), NULL);
	gtk_window_set_title (GTK_WINDOW (gdict_app), title);
	g_free (word);
	g_free (title);

	gtk_statusbar_pop (GTK_STATUSBAR (gdict_appbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR (gdict_appbar), 0, _("Lookup done"));

	if (edit_actions)
		gtk_action_group_set_sensitive (edit_actions, TRUE);
}

static void
def_not_found_cb (GtkWidget *widget, gpointer data) 
{
	gtk_statusbar_pop (GTK_STATUSBAR (gdict_appbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR (gdict_appbar), 0, _("No matches found"));
 
	if (edit_actions)
		gtk_action_group_set_sensitive (edit_actions, FALSE);

	if (gdict_pref.smart)
		gdict_spell (gdict_defbox_get_word (defbox), FALSE);
}

static void
def_substr_not_found_cb (GtkWidget *widget, gpointer data) 
{
	gtk_statusbar_pop (GTK_STATUSBAR (gdict_appbar), 0);
	gtk_statusbar_push (GTK_STATUSBAR (gdict_appbar), 0, _("String not found"));
}

static void
lookup_any (void) 
{
	if (GTK_WIDGET_HAS_FOCUS (defbox))
		lookup_defbox ();
	else
		lookup_entry ();
}

static void
lookup_cb (GtkAction *action, GtkWindow *window)
{
	lookup_any();
}

static void
lookup_button_cb (GtkButton *button, gpointer data)
{
	lookup_any();
}

static void
lookup_button_drag_cb (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
		       GtkSelectionData *sd, guint info, guint t, gpointer data)
{
	gchar *text;

	text = gtk_selection_data_get_text (sd);

	if (text) {
		gtk_entry_set_text (GTK_ENTRY (word_entry), text);
		gnome_entry_prepend_history (GNOME_ENTRY (gnome_word_entry), 1, text);
		gdict_app_do_lookup (text);
		g_free (text);
	}
}

static void
spell_cb (GtkAction *action, GtkWindow *window)
{
	gchar *text;

	text = gtk_editable_get_chars (GTK_EDITABLE (word_entry), 0, -1);
	if (gdict_spell (text, FALSE) < 0)
		gdict_not_online ();

	g_free (text);
}

#ifdef HAVE_GNOME_PRINT

static void
print_cb (GtkAction *action, GtkWindow *window)
{
	gdict_defbox_print (defbox);
}

static void
print_setup_cb (GtkAction *action, GtkWindow *window)
{
	gdict_printer = gnome_printer_dialog_new_modal();
}

#endif /* HAVE_GNOME_PRINT */

static void
close_cb (GtkAction *action, GtkWindow *window)
{
	gtk_widget_hide (gdict_app);
}

static void
quit_cb (GtkAction *action, GtkWindow *window)
{
	g_free (find_text);
	gtk_main_quit();
}

static void
cut_cb (GtkAction *action, GtkWindow *window)
{
	g_signal_emit_by_name (G_OBJECT (word_entry), "cut_clipboard");
}

static void
copy_cb (GtkAction *action, GtkWindow *window)
{
	if (GTK_WIDGET_HAS_FOCUS (defbox))
		g_signal_emit_by_name (G_OBJECT (defbox), "copy_clipboard");
	else if (GTK_WIDGET_HAS_FOCUS (word_entry))
		g_signal_emit_by_name (G_OBJECT (word_entry), "copy_clipboard");
}

static void
paste_cb (GtkAction *action, GtkWindow *window) 
{
	g_signal_emit_by_name (G_OBJECT (word_entry), "paste_clipboard");
}

static void
clear_cb (GtkAction *action, GtkWindow *window)
{
	gdict_defbox_clear (defbox);

	if (edit_actions)
		gtk_action_group_set_sensitive (edit_actions, FALSE);
}

static void
select_all_cb (GtkAction *action, GtkWindow *window)
{
	GtkTextBuffer* buffer = NULL;
	GtkTextIter start_iter, end_iter;

	gtk_widget_grab_focus (GTK_WIDGET (defbox));
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
	gtk_text_buffer_get_bounds (buffer, &start_iter, &end_iter);
	gtk_text_buffer_select_range (buffer, &start_iter, &end_iter);
}

static void
find_word_dialog_cb (GtkDialog *dialog, gint id, gpointer data)
{
	GtkEntry *entry;
	const gchar *str;

	entry = GTK_ENTRY (data);

	gtk_statusbar_pop (GTK_STATUSBAR (gdict_appbar), 0);

	switch (id) {
	case GTK_RESPONSE_APPLY:
		str = gtk_entry_get_text (entry);

		if (str == NULL)
			return;
			
		if (gdict_defbox_find (defbox, str, search_from_beginning))
		{
			search_from_beginning = FALSE;
			
			if (find_text != NULL)
				g_free (find_text);
			
			find_text = g_strdup (str);
		}
		else
			search_from_beginning = TRUE;
		
		break;

	default:
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	}
}

static void
find_cb (GtkAction *action, GtkWindow *window)
{
	static GtkWidget *dialog = NULL;
	static GtkWidget *entry = NULL;
	GtkWidget *hbox;
	GtkWidget *label;

	if (dialog == NULL) {
		dialog = gtk_dialog_new_with_buttons (_("Find"), window,
				GTK_DIALOG_DESTROY_WITH_PARENT,
				GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
				GTK_STOCK_FIND, GTK_RESPONSE_APPLY,
				NULL);
		gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	    	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_APPLY);

    		hbox = gtk_hbox_new (FALSE, 8);
   		label = gtk_label_new_with_mnemonic (_("_Search for:"));
		entry = gtk_entry_new ();
		gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
		gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);

    		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    		gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);

    		gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);

    		gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox);
		    
   		g_signal_connect (G_OBJECT (dialog), "response", 
				  G_CALLBACK (find_word_dialog_cb), entry);
		g_signal_connect(G_OBJECT (dialog), "destroy",
				 G_CALLBACK (gtk_widget_destroyed), &dialog);

		gtk_widget_show_all (GTK_WIDGET (dialog));

		search_from_beginning = TRUE;
	}
	else {
		gtk_window_present (GTK_WINDOW (dialog));
	}

	if (find_text != NULL)
		gtk_entry_set_text (GTK_ENTRY (entry), find_text);

	gtk_entry_select_region (GTK_ENTRY (entry), 0, -1);
	gtk_widget_grab_focus (entry);
}

static void
find_next_cb (GtkAction *action, GtkWindow *window)
{
	if (find_text)
		search_from_beginning = !gdict_defbox_find (defbox, find_text, search_from_beginning);
	else
		find_cb (NULL, NULL);
}

void
gdict_not_online () 
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (gdict_app),
						     GTK_DIALOG_DESTROY_WITH_PARENT,
						     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
						     _("Unable to perform requested operation."),
						     _("Either the server you are using is not available \n"
						     "or you are not connected to the Internet."));

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

void
gdict_app_show_preferences (void) 
{
	static GtkWidget *pref_dialog;

	if (!pref_dialog)
		pref_dialog = gdict_pref_dialog_new (context);

	gtk_window_present (GTK_WINDOW (pref_dialog));
}

static void
preferences_cb (GtkAction *action, GtkWindow *window)
{
	gdict_app_show_preferences ();
}

static void
help_contents_cb (GtkAction *action, GtkWindow *window)
{
	GError *error = NULL;

	gnome_help_display ("gnome-dictionary", NULL, &error);
	if (error) {
		GtkWidget *dialog;

		dialog = gtk_message_dialog_new_with_markup (window,
						 GTK_DIALOG_DESTROY_WITH_PARENT,
						 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						 "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
						 _("Could not display help"), error->message); 

		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_error_free (error);
	}
}

static void
about_cb (GtkAction *action, GtkWindow *window)
{
	gdict_about (window);
}


/* Menu bar */

static GtkActionEntry menu_entries[] = {

  /* Top */
  { "DictionaryMenu", NULL, "_Dictionary" },
  { "EditMenu", NULL, "_Edit" },
  { "HelpMenu", NULL, "_Help" },

  /* Dictionary menu */
  { "LookUp", GTK_STOCK_FIND, N_("_Look Up Word"), NULL,
    N_("Lookup word in dictionary"), G_CALLBACK (lookup_cb) },
  { "CheckSpell", GTK_STOCK_SPELL_CHECK, N_("Check _Spelling"),  NULL,
    N_("Check word spelling"), G_CALLBACK (spell_cb) },
#ifdef HAVE_GNOME_PRINT
  { "Print", GTK_STOCK_PRINT, N_("_Print"), "<control>P",
    N_("Print the current definition"), G_CALLBACK (print_setup_cb) },
#endif /* HAVE_GNOME_PRINT */

  /* Edit menu, see below for cut, paste etc. */
  { "Find", GTK_STOCK_FIND, N_("_Find..."), "<control>F",
    N_("Find a word in the text"), G_CALLBACK (find_cb) },
  { "FindNext", NULL, N_("Find Ne_xt"), "<control>G",
    N_("Find next occurrence of the word"), G_CALLBACK (find_next_cb) },
  { "Preferences", GTK_STOCK_PREFERENCES, N_("P_references"), NULL,
    N_("Configure the application"), G_CALLBACK (preferences_cb) },

  /* Help menu */
  { "HelpContents", GTK_STOCK_HELP, N_("_Contents"), "F1",
     N_("View help for this application"), G_CALLBACK (help_contents_cb) },
  { "About", GTK_STOCK_ABOUT, N_("_About"), NULL,
    N_("About this application"), G_CALLBACK (about_cb) }
};

/* Put Quit and Close in separately since we use one for the app
 * and the other for the applet. (note that they have the same name) */

static GtkActionEntry quit_entries[] = {
  { "Quit", GTK_STOCK_QUIT, N_("_Quit"), "<control>Q",
    N_("Quit the application"), G_CALLBACK (quit_cb) },
};

static GtkActionEntry close_entries[] = {
  { "Quit", GTK_STOCK_CLOSE, N_("_Close"), "<control>W",
    N_("Close the application"), G_CALLBACK (close_cb) },
};

/* a separate group for items that change sensitivity */
static GtkActionEntry edit_entries[] = {
  { "Cut", GTK_STOCK_CUT, N_("Cu_t"), "<control>X",
    N_("Cut the selection"), G_CALLBACK (cut_cb) },
  { "Copy", GTK_STOCK_COPY, N_("_Copy"), "<control>C",
    N_("Copy the selection"), G_CALLBACK (copy_cb) },
  { "Paste", GTK_STOCK_PASTE, N_("_Paste"), "<control>V",
    N_("Paste clipboard"), G_CALLBACK (paste_cb) },
  { "SelectAll", NULL, N_("Select _All"), "<control>A",
    N_("Select everything"), G_CALLBACK (select_all_cb) }
};

static const gchar *ui_info = 
"<ui>"
"  <menubar name='MenuBar'>"
"    <menu action='DictionaryMenu'>"
"      <menuitem action='LookUp'/>"
"      <menuitem action='CheckSpell'/>"
#ifdef HAVE_GNOME_PRINT
"      <separator/>"
"      <menuitem action='Print'/>"
#endif /* HAVE_GNOME_PRINT */
"      <separator/>"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu action='EditMenu'>"
"      <menuitem action='Cut'/>"
"      <menuitem action='Copy'/>"
"      <menuitem action='Paste'/>"
"      <menuitem action='SelectAll'/>"
"      <separator/>"
"      <menuitem action='Find'/>"
"      <menuitem action='FindNext'/>"
"      <separator/>"
"      <menuitem action='Preferences'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='HelpContents'/>"
"      <menuitem action='About'/>"
"    </menu>"
"  </menubar>"
"</ui>";


enum
{
	TARGET_STRING,
	TARGET_TEXT,
	TARGET_COMPOUND_TEXT,
	TARGET_UTF8_STRING,
	TARGET_TEXT_BUFFER_CONTENTS
};

static void
follow_if_link (GtkWidget *defbox, GtkTextIter *iter)
{
	GSList *tags = NULL, *tagp = NULL;
	gchar *str;

	tags = gtk_text_iter_get_tags (iter);
	for (tagp = tags;  tagp != NULL;  tagp = tagp->next) {
		GtkTextTag *tag = tagp->data;
		str = g_object_get_data (G_OBJECT (tag), "page");
		gdict_defbox_clear (GDICT_DEFBOX (defbox));		
		gdict_app_do_lookup (str);
		gtk_widget_grab_focus (word_entry);
		gtk_editable_set_position (GTK_EDITABLE (word_entry), -1);
		g_free (str);
	}

	if (tags) 
		g_slist_free (tags);
}

static gboolean
event_after (GtkWidget *defbox, GdkEvent *ev)
{
	GtkTextIter iter;
	GtkTextBuffer *buffer;
	GdkEventButton *event;
	gint x, y;

	if (ev->type != GDK_BUTTON_RELEASE)
		return FALSE;

	event = (GdkEventButton *)ev;
	if (event->button != 1)
		return FALSE;

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

	if (gtk_text_buffer_get_selection_bounds (buffer, NULL, NULL))
		return FALSE;

	gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (defbox), 
					       GTK_TEXT_WINDOW_WIDGET,
					       event->x, event->y, &x, &y);

	gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (defbox), &iter, x, y);

	follow_if_link (defbox, &iter);

	return FALSE;
}

static void
set_cursor_if_appropriate (GtkWidget *defbox, gint x, gint y)
{
	GSList *tags = NULL, *tagp = NULL;
	GtkTextBuffer *buffer;
	GtkTextIter iter;
	static gboolean hovering_over_link = FALSE;
	static GdkCursor *hand_cursor = NULL;
	static GdkCursor *regular_cursor = NULL;
	gboolean hovering = FALSE;

	if (hand_cursor == NULL)
		hand_cursor = gdk_cursor_new (GDK_HAND2);
	if (regular_cursor == NULL)
		regular_cursor = gdk_cursor_new (GDK_XTERM);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

	gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (defbox), &iter, x, y);

	tags = gtk_text_iter_get_tags (&iter);
	for (tagp = tags;  tagp != NULL;  tagp = tagp->next) {
		GtkTextTag *tag = tagp->data;
		gint page = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (tag), "page"));

		if (page != 0) {
			hovering = TRUE;
			break;
		}
	}

	if (hovering != hovering_over_link) {
		hovering_over_link = hovering;

		if (hovering_over_link)
			gdk_window_set_cursor (gtk_text_view_get_window (GTK_TEXT_VIEW (defbox),
					       GTK_TEXT_WINDOW_TEXT), hand_cursor);
		else
			gdk_window_set_cursor (gtk_text_view_get_window (GTK_TEXT_VIEW (defbox),
					       GTK_TEXT_WINDOW_TEXT), regular_cursor);
	}

	if (tags)
		g_slist_free (tags);
}

static gboolean
motion_notify_event ( GtkWidget *defbox,
		     GdkEventMotion *event)
{
	gint x, y;

	gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (defbox), 
					       GTK_TEXT_WINDOW_WIDGET,
					       event->x, event->y, &x, &y);
  
	set_cursor_if_appropriate (defbox, x, y);
	gdk_window_get_pointer (GTK_WIDGET(defbox)->window, NULL, NULL, NULL);

	return FALSE;
}

/* update the cursor image if the window becomes visible */
static gboolean
visibility_notify_event (GtkWidget *defbox,
			 GdkEventVisibility *event)
{
	gint wx, wy, bx, by;

	gdk_window_get_pointer (GTK_WIDGET(defbox)->window, &wx, &wy, NULL);

	gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (defbox), 
					       GTK_TEXT_WINDOW_WIDGET,
					       wx, wy, &bx, &by);

	set_cursor_if_appropriate (defbox, bx, by);

	return FALSE;
}

GtkWidget *
gdict_app_create (gboolean applet)
{
	GtkWidget *menubar;
	GtkWidget *container;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *button;
	GtkWidget *word_label;
	GtkWidget *scrolled;
	GtkTooltips *tooltips;
	GError *error = NULL;
	static GtkTargetEntry drop_targets [] = {
		{ "UTF8_STRING", 0, 0 },
		{ "COMPOUND_TEXT", 0, 0 },
		{ "TEXT", 0, 0 },
		{ "text/plain", 0, 0 },
		{ "STRING",     0, 0 }
	};

	tooltips = gtk_tooltips_new ();

	gdict_app = gtk_window_new (GTK_WINDOW_TOPLEVEL);

	gtk_window_set_default_size (GTK_WINDOW (gdict_app), 450, 330);

	menu_actions = gtk_action_group_new ("MenuActions");
	gtk_action_group_set_translation_domain (menu_actions, NULL);
	gtk_action_group_add_actions (menu_actions, menu_entries,
				      G_N_ELEMENTS (menu_entries), NULL);
	if (!applet)
		gtk_action_group_add_actions (menu_actions, quit_entries,
					      G_N_ELEMENTS (quit_entries), NULL);
	else
		gtk_action_group_add_actions (menu_actions, close_entries,
					      G_N_ELEMENTS (close_entries), NULL);

	/* a separate group to update sensitivity */
	edit_actions = gtk_action_group_new ("EditActions");
	gtk_action_group_set_translation_domain (edit_actions, NULL);
	gtk_action_group_add_actions (edit_actions, edit_entries,
				      G_N_ELEMENTS (edit_entries), NULL);
	gtk_action_group_set_sensitive (edit_actions, FALSE);

	ui = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui, menu_actions, 0);
	gtk_ui_manager_insert_action_group (ui, edit_actions, 0);
	gtk_window_add_accel_group (GTK_WINDOW (gdict_app), 
				    gtk_ui_manager_get_accel_group (ui));

	if (!gtk_ui_manager_add_ui_from_string (ui, ui_info, -1, &error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
		return NULL;
	}

	container = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (gdict_app), container);

	menubar = gtk_ui_manager_get_widget (ui, "/MenuBar");
	gtk_box_pack_start (GTK_BOX (container), menubar, FALSE, FALSE, 0);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);
	gtk_box_set_spacing (GTK_BOX (vbox), 6);
	gtk_box_pack_start (GTK_BOX (container), vbox, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_set_spacing (GTK_BOX (hbox), 6);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	word_label = gtk_label_new_with_mnemonic (_("_Word:"));
	gtk_box_pack_start (GTK_BOX (hbox), word_label, FALSE, FALSE, 0);

	button = gdict_button_new_with_stock_image (_("_Look Up Word"), GTK_STOCK_FIND);
	gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
	gtk_drag_dest_set (button, GTK_DEST_DEFAULT_ALL, drop_targets, 
			   G_N_ELEMENTS (drop_targets), GDK_ACTION_COPY | GDK_ACTION_MOVE);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (lookup_button_cb), defbox);
	g_signal_connect (G_OBJECT (button), "drag_data_received",
			  G_CALLBACK (lookup_button_drag_cb), defbox);

	gnome_word_entry = gnome_entry_new ("Wordentry");
	gnome_entry_set_max_saved (GNOME_ENTRY (gnome_word_entry), 10);
	gtk_box_pack_start (GTK_BOX (hbox), gnome_word_entry, TRUE, TRUE, 0);
	word_entry = gnome_entry_gtk_entry(GNOME_ENTRY (gnome_word_entry));

	gtk_label_set_mnemonic_widget (GTK_LABEL (word_label), word_entry);
	GTK_WIDGET_SET_FLAGS (word_entry, GTK_CAN_FOCUS);
	gtk_widget_grab_focus (word_entry);

	if (GTK_IS_ACCESSIBLE (gtk_widget_get_accessible (gnome_word_entry))) {
		gail_loaded = TRUE;
		gdict_add_atk_namedesc (GTK_WIDGET(word_label), _("Word"), _("Word"));
		gdict_add_atk_namedesc (gnome_word_entry, _("Word Entry"),
					_("Enter a Word or select one from the list below"));
		gdict_add_atk_namedesc (word_entry, _("Word Entry"), NULL);
		gdict_add_atk_relation (gnome_word_entry, GTK_WIDGET(word_label),
					ATK_RELATION_LABELLED_BY);
		gdict_add_atk_namedesc (GTK_WIDGET(button), NULL, _("Look Up for a Word"));
	}

	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled),
					     GTK_SHADOW_NONE);
	defbox = GDICT_DEFBOX (gdict_defbox_new ());
	defbox_setup_tags (defbox);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (defbox), GNOME_PAD_SMALL);
	g_signal_connect (G_OBJECT (defbox), "word_lookup_start",
			  G_CALLBACK (def_lookup_start_cb), defbox);
	g_signal_connect (G_OBJECT (defbox), "word_lookup_done",
			  G_CALLBACK (def_lookup_done_cb), defbox);
	g_signal_connect (G_OBJECT (defbox), "word_not_found",
			  G_CALLBACK (def_not_found_cb), defbox);
	g_signal_connect (G_OBJECT (defbox), "substr_not_found",
			  G_CALLBACK (def_substr_not_found_cb), defbox);
	g_signal_connect (G_OBJECT (defbox), "socket_error",
			  G_CALLBACK (socket_error_cb), defbox);

	gtk_text_view_set_editable (GTK_TEXT_VIEW (defbox), FALSE);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled),
					     GTK_SHADOW_ETCHED_IN);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (defbox));

	gtk_box_pack_start (GTK_BOX (vbox), scrolled, TRUE, TRUE, 0);

	gdict_appbar = gtk_statusbar_new ();
	gtk_box_pack_start (GTK_BOX (container), gdict_appbar, FALSE, FALSE, 0);

	if (!applet)
		g_signal_connect (gdict_app, "delete_event",
				  G_CALLBACK (gtk_main_quit), NULL);
	else
		g_signal_connect (gdict_app, "delete_event",
				  G_CALLBACK (gtk_widget_hide), NULL);

	g_signal_connect (word_entry, "activate",
			  G_CALLBACK (lookup_cb), NULL);
	g_signal_connect (defbox, "event-after", 
			  G_CALLBACK (event_after), NULL);
	g_signal_connect (defbox, "motion-notify-event", 
			  G_CALLBACK (motion_notify_event), NULL);
	g_signal_connect (defbox, "visibility-notify-event", 
			  G_CALLBACK (visibility_notify_event), NULL);

	if (!applet)
		gtk_widget_show_all (gdict_app);
	else
		gtk_widget_show_all (container);

	return gdict_app;
}

