/*
   Copyright (C) 2004 Fernando Herrera <fherrera@onirica.com>
   Copyright (C) 2004 Mariano Suárez-Alvarez <mariano@gnome.org>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-password-dialog.h"

struct _GKMPasswordDialogPrivate
{
  GtkWidget *message_label;
  GtkWidget *password_entry;
  GtkWidget *accept_button;
};

static void gkm_password_dialog_class_init (GKMPasswordDialog *class);
static void gkm_password_dialog_init       (GKMPasswordDialog *dialog);
static void gkm_password_dialog_finalize   (GObject *object);
static void gkm_password_dialog_destroy    (GtkObject *object);

static void entry_changed_callback         (GtkWidget         *entry,
                                            GKMPasswordDialog *dialog);

static GtkDialogClass *parent_class = NULL;

GType
gkm_password_dialog_get_type (void)
{
  static GType type = 0;
  
  if (!type)
    {
      static const GTypeInfo info =
      {
	sizeof (GKMPasswordDialogClass),
	NULL,           /* base_init */
	NULL,           /* base_finalize */
	(GClassInitFunc) gkm_password_dialog_class_init,
	NULL,           /* class_finalize */
	NULL,           /* class_data */
	sizeof (GKMPasswordDialog),
	0,              /* n_preallocs */
	(GInstanceInitFunc) gkm_password_dialog_init,
        0
      };

      type = g_type_register_static (GTK_TYPE_DIALOG, "GKMPasswordDialog", &info, 0);
    }
  
  return type;
}

static void
gkm_password_dialog_class_init (GKMPasswordDialog *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class);

  parent_class = g_type_class_peek_parent (class);

  gobject_class->finalize = gkm_password_dialog_finalize;

  object_class->destroy = gkm_password_dialog_destroy;
}

static void
gkm_password_dialog_init (GKMPasswordDialog *dialog)
{
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *image;
  
  dialog->priv = g_new0 (GKMPasswordDialogPrivate, 1);
  
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 6);
  gtk_window_set_default_size (GTK_WINDOW (dialog), 300, -1);
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 12);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  
  hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, FALSE, FALSE, 0);

  image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_AUTHENTICATION, GTK_ICON_SIZE_DIALOG);
  gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0.0);
  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (vbox), FALSE, FALSE, 0);
	
  dialog->priv->message_label = gtk_label_new (NULL);
  gtk_label_set_use_markup (GTK_LABEL (dialog->priv->message_label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (dialog->priv->message_label), 0.0, 0.5);
  gtk_label_set_line_wrap (GTK_LABEL (dialog->priv->message_label), TRUE);
  gtk_label_set_justify (GTK_LABEL (dialog->priv->message_label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start (GTK_BOX (vbox), dialog->priv->message_label, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  label = gtk_label_new_with_mnemonic (_("_Password:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  dialog->priv->password_entry = gtk_entry_new ();
  gtk_entry_set_activates_default (GTK_ENTRY (dialog->priv->password_entry), TRUE);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), dialog->priv->password_entry);
  gtk_entry_set_visibility (GTK_ENTRY (dialog->priv->password_entry), FALSE);
  gtk_box_pack_start (GTK_BOX (hbox), dialog->priv->password_entry, TRUE, TRUE, 0);

  gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
  dialog->priv->accept_button = gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_OK, GTK_RESPONSE_ACCEPT);

  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, FALSE);
  g_signal_connect (G_OBJECT (dialog->priv->password_entry), "changed", G_CALLBACK (entry_changed_callback), dialog);

  gtk_widget_show_all (GTK_DIALOG (dialog)->vbox);
}

static void
gkm_password_dialog_finalize (GObject *object)
{
  GKMPasswordDialog *dialog;

  g_return_if_fail (GKM_IS_PASSWORD_DIALOG (object));

  dialog = GKM_PASSWORD_DIALOG (object);

  g_free (dialog->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gkm_password_dialog_destroy (GtkObject *object)
{
  GKMPasswordDialog *dialog;

  g_return_if_fail (GKM_IS_PASSWORD_DIALOG (object));

  dialog = GKM_PASSWORD_DIALOG (object);

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
entry_changed_callback (GtkWidget         *entry,
                        GKMPasswordDialog *dialog)
{
  const gchar *text;
  gboolean empty;

  g_return_if_fail (GTK_IS_ENTRY (entry));
  g_return_if_fail (GKM_IS_PASSWORD_DIALOG (dialog));

  text = gtk_entry_get_text (GTK_ENTRY (entry));

  empty = text == NULL || *text == 0;

  gtk_dialog_set_response_sensitive (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT, !empty);
}

GtkWidget *
gkm_password_dialog_new (const char     *title,
                         GtkWindow      *parent,
                         GtkDialogFlags  flags,
                         const char     *primary_text,
                         const char     *secondary_text)
{
  GKMPasswordDialog *dialog;
  GString *text;

  dialog = g_object_new (GKM_TYPE_PASSWORD_DIALOG, NULL);

  if (title != NULL)
    {
      gtk_window_set_title (GTK_WINDOW (dialog), title);
    }

  if (parent != NULL)
    {
      gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    }

  if (flags & GTK_DIALOG_MODAL)
    {
      gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    }
  if (flags & GTK_DIALOG_DESTROY_WITH_PARENT)
    {
      gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
    }
  /* Note we ignore GTK_DIALOG_NO_SEPARATOR, since this dialogs don't have them */

  text = g_string_new ("");
  if (primary_text != NULL)
    {
      char *escaped;

      escaped = g_markup_escape_text (primary_text, -1);
      g_string_append_printf (text, "<span weight='bold'>%s</span>", escaped);
      g_free(escaped);
    }
  if (secondary_text != NULL)
    {
      char *escaped;

      if (primary_text != NULL)
        {
          g_string_append (text, "\n\n");
        }

      escaped = g_markup_escape_text (secondary_text, -1);
      g_string_append_printf (text, "%s", escaped);
      g_free(escaped);
    }
  gtk_label_set_markup (GTK_LABEL (dialog->priv->message_label), text->str);
  g_string_free (text, TRUE);

  return GTK_WIDGET (dialog);
}

void
gkm_password_dialog_accept_button_set_label (GKMPasswordDialog *dialog,
                                             const char        *text)
{
  gtk_button_set_use_stock (GTK_BUTTON (dialog->priv->accept_button), FALSE);
  gtk_button_set_label (GTK_BUTTON (dialog->priv->accept_button), text);
}

void
gkm_password_dialog_accept_button_set_from_stock (GKMPasswordDialog *dialog,
                                                  const char        *stock_id)
{
  gtk_button_set_use_stock (GTK_BUTTON (dialog->priv->accept_button), TRUE);
  gtk_button_set_label (GTK_BUTTON (dialog->priv->accept_button), stock_id);
}

char *
gkm_password_dialog_get_password (GKMPasswordDialog *dialog)
{
  return gtk_editable_get_chars (GTK_EDITABLE (dialog->priv->password_entry), 0, -1);
}
