/* $Id: gdict-defbox.c,v 1.35 2005/01/05 14:27:58 vnoel Exp $ */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict main window
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib/gi18n.h>
#include "gdict-defbox.h"
#include "gdict-app.h"
#include <libgnomeprint/gnome-print-pango.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

enum {
    WORD_LOOKUP_START_SIGNAL,
    WORD_LOOKUP_DONE_SIGNAL,
    WORD_NOT_FOUND_SIGNAL,
    SUBSTR_NOT_FOUND_SIGNAL,
    SOCKET_ERROR_SIGNAL,
    LAST_SIGNAL
};

#define LEFT_MARGIN 72.
#define RIGHT_MARGIN 72.
#define TOP_MARGIN 72.

static gint gdict_defbox_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };

static void gdict_defbox_init (GDictDefbox *defbox);
static void gdict_defbox_class_init (GDictDefboxClass *class);

static void def_error_cb  (dict_command_t *command, DictStatusCode code, 
                           gchar *message, gpointer data);
static void def_status_cb (dict_command_t *command, DictStatusCode code, 
                           int num_found, gpointer data);
static void def_data_cb   (dict_command_t *command, dict_res_t *res,
                           gpointer data);

static gboolean is_xref   (gchar *str, int len);
static gboolean is_number (gchar *str, int len);
static gboolean is_part   (gchar *str, int len);
static void defbox_add    (GDictDefbox *defbox, gchar *def);

GType 
gdict_defbox_get_type (void)
{
    static GType gdict_defbox_type = 0;
    
    if (!gdict_defbox_type) {
        static const GTypeInfo gdict_defbox_info = {
            sizeof (GDictDefboxClass),
            NULL,
            NULL,
            (GClassInitFunc) gdict_defbox_class_init,
            NULL,
            NULL,
	    sizeof (GDictDefbox),
	    0,
            (GInstanceInitFunc) gdict_defbox_init
        };
        
        gdict_defbox_type = 
            g_type_register_static (GTK_TYPE_TEXT_VIEW, "GDictDefboxClass", &gdict_defbox_info, 0);

    }
    
    return gdict_defbox_type;
}

static void 
gdict_defbox_init (GDictDefbox *defbox)
{
    defbox->context = NULL;
    defbox->def_cmd = NULL;
}

static void 
gdict_defbox_class_init (GDictDefboxClass *class)
{
    GObjectClass *object_class;

    object_class = G_OBJECT_CLASS (class);
    
    gdict_defbox_signals[WORD_LOOKUP_START_SIGNAL] =
        g_signal_new ("word_lookup_start",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, word_lookup_start),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[WORD_LOOKUP_DONE_SIGNAL] =
        g_signal_new ("word_lookup_done",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, word_lookup_done),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[WORD_NOT_FOUND_SIGNAL] =
        g_signal_new ("word_not_found",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, word_not_found),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[SUBSTR_NOT_FOUND_SIGNAL] =
        g_signal_new ("substr_not_found",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, substr_not_found),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[SOCKET_ERROR_SIGNAL] =
        g_signal_new ("socket_error",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, socket_error),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE,
                      1,
                      G_TYPE_STRING);

    class->word_lookup_start = NULL;
    class->word_lookup_done = NULL;
    class->word_not_found = NULL;
    class->substr_not_found = NULL;
}

GtkWidget *
gdict_defbox_new (gpointer gdict)
{
    GDictDefbox *defbox;

    defbox = GDICT_DEFBOX (g_object_new (GDICT_TYPE_DEFBOX, NULL));
    defbox->gdict = gdict;

    return GTK_WIDGET (defbox);
}

/* Setup textview tags
 */
void
defbox_setup_tags (GDictDefbox *defbox)
{
    GtkTextBuffer *buffer;
	
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
	
    gtk_text_buffer_create_tag (buffer, "italic",
			        "style", PANGO_STYLE_ITALIC, NULL);
    gtk_text_buffer_create_tag (buffer, "bold",
			        "weight", PANGO_WEIGHT_BOLD, NULL);
    gtk_text_buffer_create_tag (buffer, "underline",
			        "underline", PANGO_UNDERLINE_SINGLE, NULL);
}

/* gdict_defbox_lookup
 *
 * Sends the command to the server to commence looking up the definition
 * of a word and sets the callbacks so that the definition will be displayed
 * in this defbox
 *
 * Returns 0 on success and -1 on command invocation error
 */

gint
gdict_defbox_lookup (GDictDefbox *defbox, gchar *text)
{
    GDictWindow *gdict = defbox->gdict;

    g_return_val_if_fail (defbox != NULL, -1);
    g_return_val_if_fail (GDICT_IS_DEFBOX (defbox), -1);
    g_return_val_if_fail (text != NULL, -1);
    
    while (isspace (*text)) text++;
    
    if (*text == '\0')
        return 0;

    g_signal_emit (G_OBJECT (defbox), gdict_defbox_signals[WORD_LOOKUP_START_SIGNAL], 0);

    gdict_defbox_clear (defbox);
    
    if (defbox->database)
	    g_free (defbox->database);
    
    defbox->database = g_strdup (gdict->pref->database);
    
    defbox->def_cmd = dict_define_command_new (defbox->database, text);
    defbox->def_cmd->error_notify_cb = def_error_cb;
    defbox->def_cmd->status_notify_cb = def_status_cb;
    defbox->def_cmd->data_notify_cb = def_data_cb;
    defbox->def_cmd->user_data = defbox;

    if (dict_command_invoke (defbox->def_cmd, defbox->context) < 0)
	return -1;

    return 0;
}

/* gdict_defbox_clear
 *
 * Clears the text in a defbox and eliminates the current command structure
 */

void 
gdict_defbox_clear (GDictDefbox *defbox)
{
    GtkTextBuffer *buffer;
    GtkTextIter start, end;

    g_return_if_fail (defbox != NULL);
    g_return_if_fail (GDICT_IS_DEFBOX (defbox));

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
    gtk_text_buffer_get_bounds (buffer, &start, &end);
    gtk_text_buffer_delete (buffer, &start, &end);

    if (defbox->def_cmd) {
        dict_command_destroy (defbox->def_cmd);
        defbox->def_cmd = NULL;
    }
}

/* gdict_defbox_find
 *
 * Finds a string of text in the current definition
 */

gboolean 
gdict_defbox_find (GDictDefbox *defbox, const gchar *text, gboolean start)
{
    GtkTextBuffer *buffer;
    GtkTextIter start_iter, end_iter, iter;
    GtkTextIter match_start, match_end;
    GtkTextMark *mark = NULL;

    g_return_val_if_fail (defbox != NULL, FALSE);
    g_return_val_if_fail (GDICT_IS_DEFBOX (defbox), FALSE);
    
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

    gtk_text_buffer_get_bounds (buffer, &start_iter, &end_iter);

    if (start) 
    	iter = start_iter;
    else {
        mark = gtk_text_buffer_get_mark (buffer, "last_search");
    	
    	if (mark)
    	    gtk_text_buffer_get_iter_at_mark (buffer, &iter, mark);
    	else {
		GDictWindow *gdict = defbox->gdict;
		gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
		iter = start_iter;
	}
    }
    
    if (gtk_text_iter_forward_search (&iter, text,
                                      GTK_TEXT_SEARCH_VISIBLE_ONLY |
                                      GTK_TEXT_SEARCH_TEXT_ONLY,
                                      &match_start, &match_end,
                                      NULL))
    {
        gtk_text_view_scroll_to_iter (GTK_TEXT_VIEW (defbox), &match_start,
        			      0.0, TRUE, 0.5, 0.5);
        gtk_text_buffer_place_cursor (buffer, &match_end);
        gtk_text_buffer_move_mark (buffer,
                               gtk_text_buffer_get_mark (buffer, "selection_bound"),
                               &match_start);     
        gtk_text_buffer_create_mark (buffer, "last_search", &match_end, FALSE);

	return TRUE;
    }
    else
    {
        g_signal_emit (G_OBJECT (defbox), gdict_defbox_signals[SUBSTR_NOT_FOUND_SIGNAL], 0);

	return FALSE;
    }
}

/* gdict_defbox_reset
 *
 * Resets the defbox by re-invoking the define command on a new database
 * and/or server
 */

void
gdict_defbox_reset (GDictDefbox *defbox, dict_context_t *context)
{
    gchar *word;
    GDictWindow *gdict = defbox->gdict;
    
    if (context != defbox->context || 
        defbox->database == NULL ||
        strcmp (defbox->database, gdict->pref->database))
    {
        defbox->context = context;
        
        if (defbox->def_cmd) {
            word = g_strdup (defbox->def_cmd->search_term);
            dict_command_destroy (defbox->def_cmd);
            defbox->def_cmd = NULL;
            gdict_defbox_lookup (defbox, word);
            g_free (word);
        }
    }
}

void
gdict_defbox_print_preview (GDictDefbox *defbox)
{
	GtkWidget *preview;
	GDictWindow *gdict = defbox->gdict;
	preview = gnome_print_job_preview_new (gdict->print_job, _("Definition preview"));
	gtk_widget_show (preview);
}

/* gdict_defbox_print
 * 
 * prepare the printing job for the definition printing
 */

void 
gdict_defbox_prepare_print (GDictDefbox *defbox)
{
    GDictWindow *gdict = defbox->gdict;
    GnomePrintContext *pc;
    GList *node;
    dict_res_t *res;
    int nb_node = 0, i=0, width, height;
    double page_width, page_height;
    gchar **node_definition;
    gchar *text;
    PangoLayout *layout;
    PangoFontDescription *desc;

    g_return_if_fail (defbox != NULL);
    g_return_if_fail (GDICT_IS_DEFBOX (defbox));

    if (defbox->def_cmd == NULL)
	    return;

    pc = gnome_print_job_get_context (gdict->print_job);
    g_return_if_fail(pc != NULL);
    gnome_print_beginpage (pc, "1");
    gnome_print_job_get_page_size (gdict->print_job, &page_width, &page_height);
    gnome_print_moveto (pc, 100, page_height - TOP_MARGIN);

    layout = gnome_print_pango_create_layout (pc);
    pango_layout_set_width (layout, (page_width - LEFT_MARGIN - RIGHT_MARGIN) * PANGO_SCALE);
    desc = pango_font_description_from_string ("Times 10");
    pango_layout_set_font_description (layout, desc);
    pango_font_description_free (desc);

    pango_layout_get_size (layout, &width, &height);

    nb_node = g_list_length (defbox->def_cmd->res_list);

    node_definition = (gchar **) g_new0 (gpointer, nb_node+1);
    for (node = defbox->def_cmd->res_list; node; node = g_list_next (node)) {
	    res = (dict_res_t *) node->data;
	    node_definition[i++] = res->desc;
    }
    node_definition[i] = NULL;

    text = g_strjoinv ("\n", node_definition);

    pango_layout_set_text (layout, text, -1);
    gnome_print_pango_layout (pc, layout);
	    
    gnome_print_showpage (pc);

    g_object_unref (G_OBJECT (pc));
}

/* gdict_defbox_get_word
 *
 * Returns the word defined in the defbox, if any
 */

gchar *
gdict_defbox_get_word (GDictDefbox *defbox)
{
    g_return_val_if_fail (defbox != NULL, NULL);
    g_return_val_if_fail (GDICT_IS_DEFBOX (defbox), NULL);
    
    return defbox->def_cmd ? defbox->def_cmd->search_term : NULL;
}

/* is_xref
 *
 * Returns true if the substring str[0..len-1] is a cross reference
 */

static gboolean 
is_xref (gchar *str, int len)
{
    gint i;
    for (i = 0;  i < len;  i++)
        if (!isupper(str[i]) && !ispunct(str[i]))
            return FALSE;
    return TRUE;
}

/* is_number
 *
 * Returns true if the substring given by str[0..len-1] is a heading number
 */

static gboolean 
is_number (gchar *str, int len)
{
    gint i;

    if (str[len-1] != '.')
        return FALSE;
        
    for (i = 0;  i < len - 1;  i++)
        if (!isdigit(str[i]))
            return FALSE;

    return TRUE;
}

/* is_part
 *
 * Returns true if the substring given by str[0..len-1] is a part of speech
 */

static gboolean 
is_part (gchar *str, int len)
{
    gchar buf[3];

    if ((len < 1) || (len > 2))
        return FALSE;

    strncpy(buf, str, 2);
    buf[len] = 0;

    return (strcmp(buf, "n") == 0) || \
           (strcmp(buf, "vt") == 0) || \
           (strcmp(buf, "vi") == 0) || \
           (strcmp(buf, "vb") == 0) || \
           (strcmp(buf, "aj") == 0) || \
           (strcmp(buf, "av") == 0);
}

static void
insert_text_with_tags (GtkTextBuffer *buffer, GtkTextIter *iter, gchar *p, gint len,
		       gboolean bold, gboolean italic, gboolean underline)
{
	GtkTextMark *mark;
	GtkTextIter start;
	
	mark = gtk_text_buffer_create_mark (buffer, "start_insert", iter, TRUE);
	gtk_text_buffer_insert (buffer, iter, p, len);
	gtk_text_buffer_get_iter_at_mark (buffer, &start, mark);
	gtk_text_buffer_delete_mark (buffer, mark);

	if (bold)
		gtk_text_buffer_apply_tag_by_name (buffer, "bold", &start, iter);
	if (italic)
		gtk_text_buffer_apply_tag_by_name (buffer, "italic", &start, iter);
	if (underline)
		gtk_text_buffer_apply_tag_by_name (buffer, "underline", &start, iter);
}

static void
insert_link (GtkTextBuffer *buffer, GtkTextIter *iter, gchar *text, gint len)
{
	GtkTextTag *tag;

	tag = gtk_text_buffer_create_tag (buffer, NULL,
					  "foreground", "blue",
					  "underline", PANGO_UNDERLINE_SINGLE,
					  NULL);
	g_object_set_data (G_OBJECT (tag), "page", text);

	gtk_text_buffer_insert_with_tags (buffer, iter, text, len , tag, NULL);
}


/* defbox_add
 *
 * Adds a definition to the defbox, performing all necessary formatting
 */

static void 
defbox_add (GDictDefbox *defbox, gchar *def)
{
    GtkTextBuffer *buffer;
    GtkTextIter iter;
    GDictWindow *gdict = defbox->gdict;
    gchar *p, *q, *text;
    gint len;
    GdkFont *font;
    GdkColor *color;
    gboolean italic, bold, underline;
    int length = 0;

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
    gtk_text_buffer_get_end_iter (buffer, &iter);
    
    gtk_text_buffer_insert (buffer, &iter, "\n", -1);
         
    font = gdict->pref->typefaces[TYPEFACE_HEADWORD].font;
    color = gdict->pref->typefaces[TYPEFACE_HEADWORD].color;

    p = defbox->def_cmd->db_full_name;
    len = strlen(p);

    gtk_text_buffer_insert (buffer, &iter, p, len);

    p = def;
    len = strcspn(def, " ");
    
    bold = TRUE;
    italic = FALSE;
    underline = TRUE; /* only for the header */

    while (*p) {
        /* Handle word token */
        insert_text_with_tags (buffer, &iter, p, len, bold, italic, underline);
        
        if ((font == gdict->pref->typefaces[TYPEFACE_HEADWORD].font) || \
            (font == gdict->pref->typefaces[TYPEFACE_NUMBER].font) || \
            (font == gdict->pref->typefaces[TYPEFACE_PART].font) || \
            (font == gdict->pref->typefaces[TYPEFACE_XREF].font))
        {
            font = gdict->pref->typefaces[TYPEFACE_BODY].font;
            color = gdict->pref->typefaces[TYPEFACE_BODY].color;
            bold = FALSE;
            italic = FALSE;
            underline = FALSE;
        }
        p += len;
        
        /* ... then handle spaces ... */
        len = strspn(p, " ");
        
        if (len > 0) {
            insert_text_with_tags (buffer, &iter, p, len, bold, italic, underline);
        }
#if 0
        if (len > 0)
            gtk_text_insert (GTK_TEXT (defbox),  font, color, NULL, p, len);
#endif
        p += len;

        /* ... handle special characters ... */
        if (*p == '\\') {
            /*font = (font == gdict->pref->typefaces[TYPEFACE_BODY].font) ? \
                gdict->pref->typefaces[TYPEFACE_PRONUNCIATION].font : \
                gdict->pref->typefaces[TYPEFACE_BODY].font;*/
            color = (color == gdict->pref->typefaces[TYPEFACE_BODY].color) ? \
                gdict->pref->typefaces[TYPEFACE_PRONUNCIATION].color : \
                gdict->pref->typefaces[TYPEFACE_BODY].color;
            if (font == gdict->pref->typefaces[TYPEFACE_BODY].font) {
                font = gdict->pref->typefaces[TYPEFACE_PRONUNCIATION].font;
                bold = FALSE;
                italic = TRUE;
                underline = FALSE;
            }
            else {
            	font = gdict->pref->typefaces[TYPEFACE_BODY].font;
            	bold = FALSE;
            	italic = FALSE;
            	underline = FALSE;
            }
            ++p;
        } else if (*p == '[') {
            font = gdict->pref->typefaces[TYPEFACE_ETYMOLOGY].font;
            color = gdict->pref->typefaces[TYPEFACE_ETYMOLOGY].color;
            bold = FALSE;
            italic = TRUE;
            underline = FALSE;
            ++p;
        } else if (*p == '{') {
            font = gdict->pref->typefaces[TYPEFACE_EXAMPLE].font;
            color = gdict->pref->typefaces[TYPEFACE_EXAMPLE].color;
            bold = TRUE;
            italic = TRUE;
            underline = FALSE;
			q = p;
			while (*q != '}')
			{
				 length++;
				 q++;
                        }				
			len = length;
			++p;	
			text = g_strndup (p, len - 1);
			insert_link (buffer, &iter, text, len - 1);
			p += length;
			length = 0;
        } else if (*p == ']' ) {
            font = gdict->pref->typefaces[TYPEFACE_BODY].font;
            color = gdict->pref->typefaces[TYPEFACE_BODY].color;
            bold = FALSE;
            italic = FALSE;
            underline = FALSE;
            ++p;
        }
        
        len = strcspn (p, " \\[]{}");

        if (font == gdict->pref->typefaces[TYPEFACE_BODY].font) {
            if (is_xref(p, len)) {
                font = gdict->pref->typefaces[TYPEFACE_XREF].font;
                color = gdict->pref->typefaces[TYPEFACE_XREF].color;
                bold = FALSE;
                italic = FALSE;
                underline = FALSE;
            } else if (is_number(p, len)) {
                font = gdict->pref->typefaces[TYPEFACE_NUMBER].font;
                color = gdict->pref->typefaces[TYPEFACE_NUMBER].color;
                bold = TRUE;
                italic = FALSE;
                underline = FALSE;
            } else if (is_part(p, len)) {
                font = gdict->pref->typefaces[TYPEFACE_PART].font;
                color = gdict->pref->typefaces[TYPEFACE_PART].color;
                bold = FALSE;
                italic = FALSE;
                underline = FALSE;
            }
        }
    }    
   
}

/* def_error_cb
 *
 * Callback invoked when there was an error in the last query
 */

static void
def_error_cb (dict_command_t *command, DictStatusCode code,
              gchar *message, gpointer data)
{
    GObject *defbox;
    dict_command_t *sec_cmd;

    defbox = G_OBJECT (data);

    if (code != DICT_SOCKET_ERROR) {
        GtkWidget *dialog;
        
        dialog = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
						     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
						     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
						     _("Error invoking query"), message);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	
        sec_cmd = dict_disconnect_command_new ();
        dict_command_invoke (sec_cmd, command->context);
        
        g_signal_emit (defbox, gdict_defbox_signals[WORD_LOOKUP_DONE_SIGNAL], 0);
    }
    else {
        g_signal_emit (defbox, gdict_defbox_signals[SOCKET_ERROR_SIGNAL], 0, message);
    }
}

/* def_data_cb
 *
 * Callback used when a new definition has arrived over the link
 */

static void 
def_data_cb (dict_command_t *command, dict_res_t *res, gpointer data)
{
    GDictDefbox *defbox;
    
    defbox = GDICT_DEFBOX (data);
    defbox_add (defbox, res->desc);
}

/* def_status_cb
 *
 * Callback used when a status code has arrived over the link
 */

static void 
def_status_cb (dict_command_t *command, DictStatusCode code, 
               int num_found, gpointer data)
{
    GObject *defbox;

    defbox = G_OBJECT (data);

    if (code == DICT_STATUS_OK)
        g_signal_emit (defbox, gdict_defbox_signals[WORD_LOOKUP_DONE_SIGNAL], 0);
    else if (code == DICT_STATUS_NO_MATCH)
        g_signal_emit (defbox, gdict_defbox_signals[WORD_NOT_FOUND_SIGNAL], 0);
}
