/*
 *  Authors: Rodney Dawes <dobey@ximian.com>
 *
 *  Copyright 2004-2005 Novell, Inc. (www.novell.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */

#include <config.h>

#include "evolution-webcal-main.h"

static void e_webcal_open_cal_http (const gchar * uri, const gchar * old);

static SoupSession * session;

static gboolean e_webcal_has_kind (icalcomponent * comp, icalcomponent_kind kind)
{
  g_return_val_if_fail (comp != NULL, FALSE);

  return (icalcomponent_get_first_component (comp, kind) != NULL);
}

static void e_webcal_load (const gchar * body, const gchar * uri)
{
  icalcomponent * comp;
  icalproperty * prop;
  gint i, numprops;
  gchar * tmpname;
  const gchar * name = NULL, * desc = NULL;
  gboolean has_events, has_tasks;

  comp = icalparser_parse_string (body);
  if (comp == NULL) {
    SoupUri * tmpuri;
    gchar * message, * tmpname;

    tmpuri = soup_uri_new (uri);
    tmpname = g_path_get_basename (tmpuri->path);

    message = g_strdup_printf (_("There was an error parsing the calendar, "
				 "\"%s\". Please verify that it is a valid "
				 "calendar, and try again."),
			       tmpname);

    e_webcal_display_error (_("Error Parsing Calendar"),
			       message,
			       NULL);

    g_free (tmpname);
    soup_uri_free (tmpuri);
    g_free (message);

    bonobo_main_quit ();
    return;
  }

  numprops = icalcomponent_count_properties (comp, ICAL_X_PROPERTY);
  for (i = 0; i < numprops; i++) {
    if (i == 0) {
      prop = icalcomponent_get_first_property (comp, ICAL_X_PROPERTY);
    } else {
      prop = icalcomponent_get_next_property (comp, ICAL_X_PROPERTY);
    }
    if (prop != NULL) {
      const gchar * propname = icalproperty_get_x_name (prop);
      
      if (propname != NULL) {
	if (!strcmp (propname, "X-WR-CALNAME")) {
	  name = icalproperty_get_value_as_string (prop);
	} else if (!strcmp (propname, "X-WR-CALDESC")) {
	  desc = icalproperty_get_value_as_string (prop);
	}
      }
      icalproperty_free (prop);
    }
  }

  has_tasks  = e_webcal_has_kind (comp, ICAL_VTODO_COMPONENT);
  has_events = e_webcal_has_kind (comp, ICAL_VEVENT_COMPONENT);

  icalcomponent_free (comp);

  e_webcal_query_user (name, desc, uri, has_events, has_tasks);

  bonobo_main_quit ();
}

static void e_webcal_read (SoupMessage * msg, gpointer data) {
  const SoupUri * tmpuri;
  const gchar * uri = (const gchar *) data;
  const gchar * header;

  tmpuri = soup_message_get_uri (msg);

  if (SOUP_STATUS_IS_REDIRECTION (msg->status_code)) {
    header = soup_message_get_header (msg->response_headers, "Location");
    if (header) {
      e_webcal_open_cal_http (header, uri);
    }
    return;
  }

  if (!SOUP_STATUS_IS_SUCCESSFUL (msg->status_code)) {
    gchar * errorstring;
    gchar * errorname;

    errorname = g_path_get_basename (tmpuri->path);
    switch (msg->status_code) {
      /* Handle some common error codes here */
    case SOUP_STATUS_FORBIDDEN:
      errorstring = g_strdup_printf (_("Access to the calendar, \"%s\", "
				       "is forbidden."),
				     errorname);
      break;
    case SOUP_STATUS_NOT_FOUND:
      errorstring = g_strdup_printf (_("The calendar, \"%s\", was not "
				       "found on the server."),
				     errorname);
      break;
    case SOUP_STATUS_INTERNAL_SERVER_ERROR:
      errorstring = g_strdup_printf (_("There was an internal server error "
				       "while trying to load \"%s\"."),
				     errorname);
      break;
    default:
      errorstring = g_strdup_printf (_("There was an error loading the "
				       "calendar, \"%s\"."),
				     errorname);
      break;
    }
    e_webcal_display_error (_("Error Loading Calendar"),
			    errorstring,
			    NULL);
    g_free (errorname);
    g_free (errorstring);
    g_free (data);
    bonobo_main_quit ();
    return;
  }

  e_webcal_load (msg->response.body, uri);

  g_free (data);
}

static void e_webcal_open_cal_http (const gchar * uri, const gchar * old) {
  SoupMessage * message;
  gchar * olduri;

  if (old != NULL) {
    olduri = g_strdup (old);
  } else {
    olduri = g_strdup (uri);
  }

  message = soup_message_new (SOUP_METHOD_GET, uri);
  if (!SOUP_IS_MESSAGE (message)) {
    gchar * errorstring;

    errorstring = g_strdup_printf (_("The URI \"%s\" is invalid."), olduri);
    e_webcal_display_error (_("Invalid URI Specified"),
			    errorstring,
			    NULL);
    g_free (errorstring);
    g_free (olduri);
    bonobo_main_quit ();
    return;
  }

  soup_message_set_flags (message, SOUP_MESSAGE_NO_REDIRECT);

  soup_session_queue_message (session, message,
			      (SoupMessageCallbackFn) e_webcal_read,
			      (gpointer) olduri);
}

static void e_webcal_open_cal_vfs (const gchar * uri) {
}

static void e_webcal_open_cal (const gchar * uri) {
  gchar * newuri;

  if (strstr (uri, "webcal://")) {
    newuri = g_strdup_printf ("http://%s", uri + strlen ("webcal://"));
    e_webcal_open_cal_http (newuri, uri);
    g_free (newuri);
  } else if (!strstr (uri, "://")) {
    newuri = g_strconcat ("http://", uri, NULL);
    e_webcal_open_cal_http (newuri, newuri);
    g_free (newuri);
  } else {
    e_webcal_open_cal_http (uri, NULL);
  }
}

static gboolean e_webcal_idle_callback (void * data) {
  GSList * p = (GSList *) data;

  if (p != NULL) {
    for (; p && p->data; p = p->next) {
      const gchar * uri = p->data;

      e_webcal_open_cal (uri);
    }
    g_slist_free (p);
  } else {
    e_webcal_display_error (_("No URI Specified"),
			    _("No URI to load was specified. You need to "
			      "pass the URI of the calendar to subscribe "
			      "to, as an argument on the command line."),
			    NULL);
    bonobo_main_quit ();
  }

  return FALSE;
}

gint main (gint argc, gchar ** argv) {
  CORBA_ORB orb;
  GnomeProgram * program;
  GSList * uri_list = NULL;
  poptContext ctx;
  const gchar ** args;
  GValue context = { 0 };

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  g_thread_init (NULL);
  program = gnome_program_init ("evolution-webcal", VERSION,
				LIBGNOMEUI_MODULE,
				argc, argv, GNOME_PARAM_POPT_TABLE,
				NULL, NULL);

  if (!bonobo_activation_is_initialized ()) {
    orb = bonobo_activation_init (argc, argv);
  } else {
    orb = bonobo_activation_orb_get ();
  }

  bonobo_init ();

  g_object_get_property (G_OBJECT (program), GNOME_PARAM_POPT_CONTEXT,
			 g_value_init (&context, G_TYPE_POINTER));
  ctx = g_value_get_pointer (&context);

  args = poptGetArgs (ctx);
  if (args != NULL) {
    const gchar ** p;

    for (p = args; *p != NULL; p++) {
      uri_list = g_slist_prepend (uri_list, (char *) *p);
    }
  }
  poptFreeContext (ctx);

  session = soup_session_async_new ();

  g_idle_add (e_webcal_idle_callback, uri_list);
  bonobo_main ();

  return 0;
}
