/* Atomixed -- the level-/themeeditor for atomix.
 * Copyright (C) 1999-2000 Jens Finke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "main.h"
#include "support.h"
#include "theme_img_editor.h"
#include "image_win_ctrl.h"
#include "theme_ctrl.h"
#include "util.h"
#include "interface.h"

/*=================================================================
 
  Declaration of internal structures.

  ---------------------------------------------------------------*/

typedef struct
{
	GnomeIconList *gil;
	Theme *theme;
} InsertImageHelper;


/*=================================================================
 
  Declaration of internal functions

  ---------------------------------------------------------------*/

GnomeIconList* theme_img_editor_get_gil(ThemeImgEditor *tie);

void create_sorted_img_list(gpointer key, ThemeElement *elem, GList **list);

gint compare_theme_elements(ThemeElement *elem1, ThemeElement *elem2);

void insert_single_image(ThemeElement *element, InsertImageHelper *helper);

void on_add_image(GtkWidget *menu_item, ThemeImgEditor *tie);

void on_delete_image(GtkWidget *menu_item, ThemeImgEditor *tie);

void on_change_id(GtkWidget *menu_item, ThemeImgEditor *tie);

/*=================================================================
 
  Level creation, initialisation and clean up

  ---------------------------------------------------------------*/

ThemeImgEditor* 
theme_img_editor_new(const gchar *gil_name, ThemeImageKind kind)
{
	ThemeImgEditor *tie; 
	
	tie = g_malloc(sizeof(ThemeImgEditor));
	
	tie->gil_name = g_strdup(gil_name);
	tie->gil = NULL;
	tie->kind = kind;
	tie->current_theme = NULL;
	tie->element_list = NULL;
	
	return tie;
}

void
theme_img_editor_destroy(ThemeImgEditor *tie)
{
	g_return_if_fail(tie!=NULL);

	if(tie->gil_name != NULL)
	{
		g_free(tie->gil_name);
	}
	if(tie->element_list != NULL)
	{
		g_list_free(tie->element_list);
	}

	g_free(tie);
}

void 
theme_img_editor_set_theme(ThemeImgEditor *tie, Theme *theme)
{	
	const gint row_spacing = 20;
	const gint col_spacing = 20;
	const gint text_spacing = 5;
	gint tile_width;
	gint tile_height;
	GnomeIconList *gil;
	InsertImageHelper *helper;

	g_return_if_fail(tie!=NULL);
	g_return_if_fail(theme!=NULL);

	gil = theme_img_editor_get_gil(tie);

	gnome_icon_list_freeze(gil);
	gnome_icon_list_clear(gil);

	gnome_icon_list_set_row_spacing (gil, row_spacing);
        gnome_icon_list_set_col_spacing (gil, col_spacing);
        gnome_icon_list_set_text_spacing (gil, text_spacing);
	theme_get_tile_size(theme, &tile_width, &tile_height);
	if(tile_width > 0)
	{
		gnome_icon_list_set_icon_width(gil, 2*tile_width); 
	}
	else
	{
		gnome_icon_list_set_icon_width(gil, 60); 
	}
	
	if(tie->element_list != NULL)
	{
		g_list_free(tie->element_list);
		tie->element_list = NULL;
	}
	
	g_hash_table_foreach(theme->image_list[tie->kind], 
			     (GHFunc)create_sorted_img_list,
			     &(tie->element_list));

	helper = g_malloc(sizeof(InsertImageHelper));
	helper->gil = gil;
	helper->theme = theme;
	g_list_foreach(tie->element_list, (GFunc)insert_single_image, helper);
	g_free(helper);

	tie->current_theme = theme;

	gnome_icon_list_thaw(gil);	
}

void
insert_single_image(ThemeElement *element, InsertImageHelper *helper)
{
	gint pos;
	GdkImlibImage *image = theme_get_element_image(helper->theme, element);
	if(image != NULL)
	{
		pos = gnome_icon_list_append_imlib(helper->gil, image, element->name);	
		gnome_icon_list_set_icon_data(helper->gil, pos, element);	
	}
}

void
create_sorted_img_list(gpointer key, ThemeElement *elem, GList **list)
{
	*list = g_list_insert_sorted(*list, elem, (GCompareFunc)compare_theme_elements);
}

gint
compare_theme_elements(ThemeElement *elem1, ThemeElement *elem2)
{
	return (elem1->id - elem2->id);
}

GnomeIconList* 
theme_img_editor_get_gil(ThemeImgEditor *tie)
{
	g_return_val_if_fail(tie!=NULL, NULL);
	
	if(tie->gil == NULL)
	{
		tie->gil = (GnomeIconList*)lookup_widget(get_app(), tie->gil_name);
	}
	
	return tie->gil;
}

void 
theme_img_editor_change_text(ThemeImgEditor *tie, gint num, gchar *text)
{
	ThemeElement *element;

	g_return_if_fail(tie!=NULL);
	g_return_if_fail(text!=NULL);

	element = (ThemeElement*) gnome_icon_list_get_icon_data(tie->gil, num);
	if(element->name != NULL)
	{
		g_free(element->name);
	}
	element->name = g_strdup(text);	
	image_win_ctrl_update_theme(tie->current_theme, tie->kind);
	theme_ctrl_set_actual_theme_modified();
}

void theme_img_editor_show_context_menu(ThemeImgEditor *tie)
{
	GnomeIconList *gil;
	gint length;
	GtkWidget *context_menu;

	GnomeUIInfo context_uiinfo[] =
	{
		{GNOME_APP_UI_ITEM, N_("Add Image"),
		 N_("Add an image to the list."),
		 on_add_image, tie, 
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		{GNOME_APP_UI_ITEM, N_("Delete Image"),
		 N_("Delete selected image(s)."),
		 on_delete_image, tie,
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		{GNOME_APP_UI_ITEM, N_("Change ID"),
		 N_("Change the ID of the image."),
		 on_change_id, tie,
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		GNOMEUIINFO_END
	};
	
	context_menu = gtk_menu_new ();
	gnome_app_fill_menu (GTK_MENU_SHELL (context_menu), context_uiinfo,
			     NULL, FALSE, 0);

	gil = theme_img_editor_get_gil(tie);
	length = g_list_length(gil->selection);
	if(length == 0)
	{
		GtkWidget *widget = context_uiinfo[1].widget;
		gtk_widget_set_sensitive(GTK_WIDGET(widget), FALSE);
		widget = context_uiinfo[2].widget;
		gtk_widget_set_sensitive(GTK_WIDGET(widget), FALSE);		
	}
	else if(length > 1)
	{
		GtkWidget *widget = context_uiinfo[2].widget;
		gtk_widget_set_sensitive(GTK_WIDGET(widget), FALSE);		
	}

	gtk_menu_popup(GTK_MENU(context_menu), NULL, NULL, NULL, NULL, 3,
		       GDK_CURRENT_TIME);
}

void on_change_id(GtkWidget *menu_item, ThemeImgEditor *tie)
{
	GnomeIconList *gil;
	gint pos;
	gint button;
	ThemeElement *element;
	GtkWidget *dlg;
	GtkWidget *current_id_widget;
	GtkWidget *new_id_widget;

	/* check if only one image is selected and get this one */
	gil = theme_img_editor_get_gil(tie);
	if(g_list_length(gil->selection) != 1) return;

	pos = GPOINTER_TO_INT(g_list_nth_data(gil->selection, 0));
	element = (ThemeElement*) gnome_icon_list_get_icon_data(gil, pos);

	dlg = create_change_id_win();
	
	current_id_widget = lookup_widget(dlg, "current_id");
	new_id_widget = lookup_widget(dlg, "new_id");
	
	/* initialise dialog */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(current_id_widget), element->id);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(new_id_widget), element->id);
	gtk_entry_select_region(GTK_ENTRY(new_id_widget), 0, -1);
	gnome_dialog_editable_enters(GNOME_DIALOG(dlg), GTK_EDITABLE(new_id_widget));
	gtk_widget_grab_focus(GTK_WIDGET(new_id_widget));
	gnome_dialog_set_default(GNOME_DIALOG(dlg), 0);

	button = gnome_dialog_run(GNOME_DIALOG(dlg));
	if(button == 0 /* ok */)
	{
		gint new_id; 
		
		/* check if id is the same*/
		new_id = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(new_id_widget));
		if(new_id == element->id)
		{
			gtk_widget_destroy(GTK_WIDGET(dlg));
			return;
		}

		/* check if theme is locale */
		if(!theme_ctrl_is_locale_theme(tie->current_theme))
		{
			if(!theme_ctrl_make_theme_locale(tie->current_theme))
			{
				g_print("** Warning: couldn't create locale copy of theme.\n");
				gtk_widget_destroy(GTK_WIDGET(dlg));
				return;
			}
		}
		
		/* change the id and update the view */
		if(theme_change_element_id(tie->current_theme, tie->kind, element, new_id))
		{
			theme_img_editor_set_theme(tie, tie->current_theme);
			image_win_ctrl_update_theme(tie->current_theme, tie->kind);
			theme_ctrl_set_actual_theme_modified();
		}
		else
		{
			set_appbar_temporary("Error: ID already in use.");
		}
	}
		
        gtk_widget_destroy(GTK_WIDGET(dlg));
}

void on_delete_image(GtkWidget *menu_item, ThemeImgEditor *tie)
{
	GnomeIconList *gil;
	GtkWidget *dlg;
	gint button;

	gil = theme_img_editor_get_gil(tie);
	if(g_list_length(gil->selection) == 0) return;

	
	dlg = GTK_WIDGET(gnome_message_box_new("Do you really want to delete this image(s)?",
					       GNOME_MESSAGE_BOX_QUESTION,
					       GNOME_STOCK_BUTTON_YES, 
					       GNOME_STOCK_BUTTON_NO, NULL));
	button = gnome_dialog_run(GNOME_DIALOG(dlg));
	if(button == 0 /* yes */)
	{
		gint length = g_list_length(gil->selection);
		gint i; 

		if(!theme_ctrl_is_locale_theme(tie->current_theme))
		{
			if(!theme_ctrl_make_theme_locale(tie->current_theme))
			{
				g_print("** Warning: couldn't create locale copy of theme.\n");
				return;
			}
		}
		

		for(i = 0; i < length; i++)
		{
			ThemeElement *element;
			gint pos;

			pos = GPOINTER_TO_INT(g_list_nth_data(gil->selection, i));
			element = (ThemeElement*) gnome_icon_list_get_icon_data(gil, pos);

			theme_remove_element(tie->current_theme, tie->kind, element);
			gnome_icon_list_remove(gil, pos);
			theme_destroy_element(element);

			image_win_ctrl_update_theme(tie->current_theme, tie->kind);
			theme_ctrl_set_actual_theme_modified();
		}
	}		
}

void on_add_image(GtkWidget *menu_item, ThemeImgEditor *tie)
{
	GtkWidget *add_dlg;
	GtkWidget *img_text;
	GtkWidget *img_entry;
	gint button;

	add_dlg = GTK_WIDGET(create_add_image_win());
	
	/* get widgets */
	img_text = lookup_widget(add_dlg, "img_text");
	img_entry = lookup_widget(add_dlg, "img_entry");

	/* set default text and select it */
	gtk_entry_set_text(GTK_ENTRY(img_text), "Text");
	gtk_entry_select_region(GTK_ENTRY(img_text), 0, -1);

	button = gnome_dialog_run(GNOME_DIALOG(add_dlg));

	if(button == 0 /* ok */)
	{
		gchar *name = gtk_editable_get_chars(GTK_EDITABLE(img_text), 0, -1);
		gchar *path = gnome_pixmap_entry_get_filename(GNOME_PIXMAP_ENTRY(img_entry));
		
		if((name != NULL) && (path != NULL))
		{
			ThemeElement *element;
			gchar *file = g_basename(path);

			if(!theme_ctrl_is_locale_theme(tie->current_theme))
			{
				if(!theme_ctrl_make_theme_locale(tie->current_theme))
				{
					g_print("** Warning: couldn't create locale copy of theme.\n");
					g_free(name);
					g_free(path);
					gtk_widget_destroy(add_dlg);
					return;
				}
			}

			element = theme_add_image(tie->current_theme, name, file, tie->kind);
			if(element != NULL)
			{
				InsertImageHelper *helper;
				gboolean result;
				
				result = theme_ctrl_copy_image_into_theme_dir(path,
									      tie->current_theme);
				if(result)
				{
				
					helper = g_malloc(sizeof(InsertImageHelper));
					helper->gil = theme_img_editor_get_gil(tie);
					helper->theme = tie->current_theme;
					
					insert_single_image(element, helper);
					
					image_win_ctrl_update_theme(tie->current_theme, tie->kind);
					theme_ctrl_set_actual_theme_modified();
					
					g_free(helper);				
				}
				else
				{
					g_print("** Warning: Error during image copying.\n");
				}
			}

			g_free(name);
			g_free(path);
		}
	}
	gtk_widget_destroy(add_dlg);
}

