using System;
using System.IO;
using System.Globalization;
using System.Collections;
using Lucene.Net.Index;
using Lucene.Net.Analysis;
using Lucene.Net.Documents;
using Lucene.Net.Search;

namespace Lucene.Net.QueryParsers
{
	/// <summary>
	/// This class is generated by JavaCC.  The only method that clients should need
	/// to call is <a href="#parse">parse()</a>.
	///
	/// The syntax for query strings is as follows:
	/// A Query is a series of clauses.
	/// A clause may be prefixed by:
	/// <ul>
	/// <li> a plus (<code>+</code>) or a minus (<code>-</code>) sign, indicating
	/// that the clause is required or prohibited respectively; or</li>
	/// <li> a term followed by a colon, indicating the field to be searched.
	/// This enables one to construct queries which search multiple fields.</li>
	/// </ul>
	///
	/// A clause may be either:
	/// <ul>
	/// <li> a term, indicating all the documents that contain this term; or</li>
	/// <li> a nested query, enclosed in parentheses.  Note that this may be used
	/// with a <code>+</code>/<code>-</code> prefix to require any of a set of
	/// terms.</li>
	/// </ul>
	///
	/// Thus, in BNF, the query grammar is:
	/// <pre>
	///   Query  ::= ( Clause )*
	///   Clause ::= ["+", "-"] [&lt;TERM&gt; ":"] ( &lt;TERM&gt; | "(" Query ")" )
	/// </pre>
	///
	/// <p>
	/// Examples of appropriately formatted queries can be found in the <a
	/// href="http://jakarta.apache.org/lucene/src/test/org/apache/lucene/queryParser/TestQueryParser.java">test cases</a>.
	/// </p>
	/// </summary>
	/// <author>Brian Goetz</author>
	/// <author>Peter Halacsy</author>
	/// <author>Tatu Saloranta</author>
	public class QueryParser : QueryParserConstants  
	{

		private const int CONJ_NONE   = 0;
		private const int CONJ_AND    = 1;
		private const int CONJ_OR     = 2;

		private const int MOD_NONE    = 0;
		private const int MOD_NOT     = 10;
		private const int MOD_REQ     = 11;

		public const int DEFAULT_OPERATOR_OR  = 0;
		public const int DEFAULT_OPERATOR_AND = 1;

		/// <summary>
		/// The actual _operator that parser uses to combine query terms
		/// </summary>
		private int _operator = DEFAULT_OPERATOR_OR;

		/// <summary>
		/// Whether terms of wildcard and prefix queries are to be automatically
		/// lower-cased or not.  Default is <code>true</code>.
		/// </summary>
		bool lowercaseWildcardTerms = true;

		Analyzer analyzer;
		String field;
		int phraseSlop = 0;
		IFormatProvider locale = CultureInfo.CurrentCulture;

		/// <summary>
		/// Parses a query string, returning a Lucene.Search.Query.
		/// </summary>
		/// <param name="query">the query string to be parsed.</param>
		/// <param name="field">the default field for query terms.</param>
		/// <param name="analyzer">used to find terms in the query text.</param>
		/// <returns></returns>
		/// <throws>ParseException if the parsing fails</throws>
		static public Query Parse(String query, String field, Analyzer analyzer)
		{
			try 
			{
				QueryParser parser = new QueryParser(field, analyzer);
				return parser.Parse(query);
			}
			catch (TokenMgrError tme) 
			{
				throw new ParseException(tme.Message);
			}
		}

		/// <summary>
		/// Constructs a query parser.
		/// </summary>
		/// <param name="f">the default field for query terms.</param>
		/// <param name="a">used to find terms in the query text.</param>
		public QueryParser(String f, Analyzer a) : 
			this(new FastCharStream(new StringReader("")))
		{
			analyzer = a;
			field = f;
		}

		/// <summary>
		/// Parses a query string, returning a
		/// <a href="Lucene.Net.Search.Query.Html">Query</a>.
		/// </summary>
		/// <param name="query">the query string to be parsed.</param>
		/// <returns></returns>
		/// <throws>ParseException if the parsing fails</throws>
		/// <throws>TokenMgrError if ther parsing fails</throws>
		public Query Parse(String query) 
		{
			ReInit(new FastCharStream(new StringReader(query)));
			return Query(field);
		}

		/// <summary>
		/// Sets the default slop for phrases.  If zero, then exact phrase matches
		/// are required.  Default value is zero.
		/// </summary>
		/// <param name="phraseSlop"></param>
		public void SetPhraseSlop(int phraseSlop) 
		{
			this.phraseSlop = phraseSlop;
		}

		/// <summary>
		/// Gets the default slop for phrases.
		/// </summary>
		/// <returns></returns>
		public int GetPhraseSlop() 
		{
			return phraseSlop;
		}

		/// <summary>
		/// Sets the bool operator of the QueryParser.
		/// In classic mode (<code>DEFAULT_OPERATOR_OR</code>) terms without any modifiers
		/// are considered optional: for example <code>capital of Hungary</code> is equal to
		/// <code>capital OR of OR Hungary</code>.<br/>
		/// In <code>DEFAULT_OPERATOR_AND</code> terms are considered to be in conjuction: the
		/// above mentioned query is parsed as <code>capital AND of AND Hungary</code>
		/// </summary>
		/// <param name="_operator"></param>
		public void SetOperator(int _operator) 
		{
			this._operator = _operator;
		}

		/// <summary>
		/// Gets implicit operator setting, which will be either DEFAULT_OPERATOR_AND
		/// or DEFAULT_OPERATOR_OR.
		/// </summary>
		public int GetOperator() 
		{
			return _operator;
		}

		public void SetLowercaseWildcardTerms(bool lowercaseWildcardTerms) 
		{
			this.lowercaseWildcardTerms = lowercaseWildcardTerms;
		}

		public bool GetLowercaseWildcardTerms() 
		{
			return lowercaseWildcardTerms;
		}

		/// <summary>
		/// Get and set locale used by date range parsing.
		/// </summary>
		public IFormatProvider Locale
		{
			get
			{
				return locale;
			}
			set
			{
				this.locale = value;
			}
		}

		protected void AddClause(ArrayList clauses, int conj, int mods, Query q) 
		{
			bool required, prohibited;

			// If this term is introduced by AND, make the preceding term required,
			// unless it's already prohibited
			if (conj == CONJ_AND) 
			{
				BooleanClause c = (BooleanClause) clauses[clauses.Count-1];
				if (!c.prohibited)
					c.required = true;
			}

			if (_operator == DEFAULT_OPERATOR_AND && conj == CONJ_OR) 
			{
				// If this term is introduced by OR, make the preceding term optional,
				// unless it's prohibited (that means we leave -a OR b but +a OR b-->a OR b)
				// notice if the input is a OR b, first term is parsed as required; without
				// this modification a OR b would parsed as +a OR b
				BooleanClause c = (BooleanClause) clauses[clauses.Count-1];
				if (!c.prohibited)
					c.required = false;
			}

			// We might have been passed a null query; the term might have been
			// filtered away by the analyzer.
			if (q == null)
				return;

			if (_operator == DEFAULT_OPERATOR_OR) 
			{
				// We set REQUIRED if we're introduced by AND or +; PROHIBITED if
				// introduced by NOT or -; make sure not to set both.
				prohibited = (mods == MOD_NOT);
				required = (mods == MOD_REQ);
				if (conj == CONJ_AND && !prohibited) 
				{
					required = true;
				}
			} 
			else 
			{
				// We set PROHIBITED if we're introduced by NOT or -; We set REQUIRED
				// if not PROHIBITED and not introduced by OR
				prohibited = (mods == MOD_NOT);
				required   = (!prohibited && conj != CONJ_OR);
			}
			clauses.Add(new BooleanClause(q, required, prohibited));
		}

		protected virtual Query GetFieldQuery(String field,
			Analyzer analyzer,
			String queryText) 
		{
			// Use the analyzer to get all the tokens, and then build a TermQuery,
			// PhraseQuery, or nothing based on the term count

			TokenStream source = analyzer.TokenStream(field,
				new StringReader(queryText));
			ArrayList v = new ArrayList();
			Lucene.Net.Analysis.Token t;

			while (true) 
			{
				try 
				{
					t = source.Next();
				}
				catch (IOException) 
				{
					t = null;
				}
				if (t == null)
					break;
				v.Add(t.TermText());
			}
			try 
			{
				source.Close();
			}
			catch (System.IO.IOException) 
			{
				// ignore
			}

			if (v.Count == 0)
				return null;
			else if (v.Count == 1)
				return new TermQuery(new Term(field, (String) v[0]));
			else 
			{
				PhraseQuery q = new PhraseQuery();
				q.SetSlop(phraseSlop);
				for (int i=0; i<v.Count; i++) 
				{
					q.Add(new Term(field, (String)v[i]));
				}
				return q;
			}
		}

		protected virtual Query GetRangeQuery(String field,
			Analyzer analyzer,
			String part1,
			String part2,
			bool inclusive)
		{
			try 
			{
				DateTime d1 = DateTime.Parse(part1, locale);
				DateTime d2 = DateTime.Parse(part2, locale);
				part1 = DateField.DateToString(d1);
				part2 = DateField.DateToString(d2);
			}
			catch (Exception) { }

			return new RangeQuery(new Term(field, part1),
				new Term(field, part2),
				inclusive);
		}

		/// <summary>
		/// Factory method for generating query, given a set of clauses.
		/// By default creates a bool query composed of clauses passed in.
		///
		/// Can be overridden by extending classes, to modify query being
		/// returned.
		/// </summary>
		/// <param name="clauses">
		///		ArrayList that contains BooleanClause instances to join.
		///	</param>
		/// <returns>Resulting Query object.</returns>
		protected virtual Query GetBoolQuery(ArrayList clauses)
		{
			BooleanQuery query = new BooleanQuery();
			for (int i = 0; i < clauses.Count; i++) 
			{
				query.Add((BooleanClause)clauses[i]);
			}
			return query;
		}

		/// <summary>
		/// Factory method for generating a query. Called when parser
		/// parses an input term token that contains one or more wildcard
		/// characters (? and *), but is not a prefix term token (one
		/// that has just a single * character at the end)
		/// <p>
		/// Depending on settings, prefix term may be lower-cased
		/// automatically. It will not go through the default Analyzer,
		/// however, since normal Analyzers are unlikely to work properly
		/// with wildcard templates.
		/// </p>
		/// <p>
		/// Can be overridden by extending classes, to provide custom handling for
		/// wildcard queries, which may be necessary due to missing analyzer calls.</p>
		/// </summary>
		/// <param name="field">Name of the field query will use.</param>
		/// <param name="termStr">Term token that contains one or more wild card
		/// characters (? or *), but is not simple prefix term
		/// </param>
		/// <returns>Resulting Query built for the term</returns>
		protected virtual Query GetWildcardQuery(String field, String termStr)
		{
			if (lowercaseWildcardTerms) 
			{
				termStr = termStr.ToLower();
			}
			Term t = new Term(field, termStr);
			return new WildcardQuery(t);
		}

		/// <summary>
		/// Factory method for generating a query (similar to
		/// GetWildcardQuery). Called when parser parses an input term
		/// token that uses prefix notation; that is, contains a single '*' wildcard
		/// character as its last character. Since this is a special case
		/// of generic wildcard term, and such a query can be optimized easily,
		/// this usually results in a different query object.
		/// <p>
		/// Depending on settings, a prefix term may be lower-cased
		/// automatically. It will not go through the default Analyzer,
		/// however, since normal Analyzers are unlikely to work properly
		/// with wildcard templates.
		/// </p>
		/// <p>
		/// Can be overridden by extending classes, to provide custom handling for
		/// wild card queries, which may be necessary due to missing analyzer calls.
		/// </p>
		///    (<b>without</b> trailing '*' character!)
		/// </summary>
		/// <param name="field">Name of the field query will use.</param>
		/// <param name="termStr">Term token to use for building term for the query</param>
		/// <returns>Resulting Query built for the term</returns>
		protected Query GetPrefixQuery(String field, String termStr)
		{
			if (lowercaseWildcardTerms) 
			{
				termStr = termStr.ToLower();
			}
			Term t = new Term(field, termStr);
			return new PrefixQuery(t);
		}

		/// <summary>
		/// Factory method for generating a query (similar to
		/// GetWildcardQuery). Called when parser parses
		/// an input term token that has the fuzzy suffix (~) appended.
		/// </summary>
		/// <param name="field">Name of the field query will use.</param>
		/// <param name="termStr">Term token to use for building term for the query</param>
		/// <returns>Resulting Query built for the term</returns>
		protected virtual Query GetFuzzyQuery(String field, String termStr)
		{
			Term t = new Term(field, termStr);
			return new FuzzyQuery(t);
		}

		public static void Main(String[] args) 
		{
			QueryParser qp = new QueryParser(
				"field",
				new Lucene.Net.Analysis.SimpleAnalyzer()
			);
			Query q = qp.Parse(args[0]);
			Console.WriteLine(q.ToString("field"));
		}

		/// <summary>
		/// Query  ::= ( Clause )*
		/// Clause ::= ["+", "-"] [&lt;TERM> ":"] ( &lt;TERM> | "(" Query ")" )
		/// </summary>
		/// <returns></returns>
		public int Conjunction() 
		{
			int ret = CONJ_NONE;
			switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
			{
				case AND:
					goto case OR;
				case OR:
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case AND:
						jj_consume_token(AND);
						ret = CONJ_AND;
						break;
					case OR:
						jj_consume_token(OR);
						ret = CONJ_OR;
						break;
					default:
						jj_la1[0] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
					break;
				default:
					jj_la1[1] = jj_gen;;
					break;
			}
		{
			if (true) return ret;
		}
		}

		public int Modifiers() 
		{
			int ret = MOD_NONE;
			switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
			{
				case NOT:
					goto case MINUS;
				case PLUS:
					goto case MINUS;
				case MINUS:
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case PLUS:
						jj_consume_token(PLUS);
						ret = MOD_REQ;
						break;
					case MINUS:
						jj_consume_token(MINUS);
						ret = MOD_NOT;
						break;
					case NOT:
						jj_consume_token(NOT);
						ret = MOD_NOT;
						break;
					default:
						jj_la1[2] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
					break;
				default:
					jj_la1[3] = jj_gen;
					break;
			}
		{if (true) return ret;}
		}

		public Query Query(String field) 
		{
			ArrayList clauses = new ArrayList();
			Query q, firstQuery=null;
			int conj, mods;
			mods = Modifiers();
			q = Clause(field);
			AddClause(clauses, CONJ_NONE, mods, q);
			if (mods == MOD_NONE)
				firstQuery=q;

			while (true) 
			{
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case AND:
						break;
					case OR:
						break;
					case NOT:
						break;
					case PLUS:
						break;
					case MINUS:
						break;
					case LPAREN:
						break;
					case QUOTED:
						break;
					case TERM:
						break;
					case PREFIXTERM:
						break;
					case WILDTERM:
						break;
					case RANGEIN_START:
						break;
					case RANGEEX_START:
						break;
					case NUMBER:
						break;
					default:
						jj_la1[4] = jj_gen;
						goto l1;
				}
				conj = Conjunction();
				mods = Modifiers();
				q = Clause(field);
				AddClause(clauses, conj, mods, q);
			}
			l1: 

			if (clauses.Count == 1 && firstQuery != null)
			{return firstQuery;}
			else 
			{
				return GetBoolQuery(clauses);
			}
		}

		public Query Clause(String field) 
		{
			Query q;
			Token fieldToken=null, boost=null;
			if (jj_2_1(2)) 
			{
				fieldToken = jj_consume_token(TERM);
				jj_consume_token(COLON);
				field = fieldToken.image;
			} 
			else 
			{
				;
			}
			switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
			{
				case QUOTED:
				case TERM:
				case PREFIXTERM:
				case WILDTERM:
				case RANGEIN_START:
				case RANGEEX_START:
				case NUMBER:
					q = Term(field);
					break;
				case LPAREN:
					jj_consume_token(LPAREN);
					q = Query(field);
					jj_consume_token(RPAREN);
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case CARAT:
						jj_consume_token(CARAT);
						boost = jj_consume_token(NUMBER);
						break;
					default:
						jj_la1[5] = jj_gen;
						break;
				}
					break;
				default:
					jj_la1[6] = jj_gen;
					jj_consume_token(-1);
					throw new ParseException();
			}
			if (boost != null) 
			{
				float f = (float)1.0;
				try 
				{
					f = Single.Parse(boost.image, NumberFormatInfo.InvariantInfo);
					q.SetBoost(f);
				} 
				catch (Exception) { }
			}
		{if (true) return q;}
		}

		public Query Term(String field) 
		{
			Token term, boost=null, slop=null, goop1, goop2;
			bool prefix = false;
			bool wildcard = false;
			bool fuzzy = false;
			Query q;
			switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
			{
				case TERM:
				case PREFIXTERM:
				case WILDTERM:
				case NUMBER:
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case TERM:
						term = jj_consume_token(TERM);
						break;
					case PREFIXTERM:
						term = jj_consume_token(PREFIXTERM);
						prefix=true;
						break;
					case WILDTERM:
						term = jj_consume_token(WILDTERM);
						wildcard=true;
						break;
					case NUMBER:
						term = jj_consume_token(NUMBER);
						break;
					default:
						jj_la1[7] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case FUZZY:
						jj_consume_token(FUZZY);
						fuzzy=true;
						break;
					default:
						jj_la1[8] = jj_gen;
						break;
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case CARAT:
						jj_consume_token(CARAT);
						boost = jj_consume_token(NUMBER);

					switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
					{
						case FUZZY:
							jj_consume_token(FUZZY);
							fuzzy=true;
							break;
						default:
							jj_la1[9] = jj_gen;
							break;
					}
						break;
					default:
						jj_la1[10] = jj_gen;
						break;
				}
					if (wildcard) 
					{
						q = GetWildcardQuery(field, term.image);
					} 
					else if (prefix) 
					{
						q = GetPrefixQuery(field, term.image.Substring
							(0, term.image.Length-1));
					} 
					else if (fuzzy) 
					{
						q = GetFuzzyQuery(field, term.image);
					} 
					else 
					{
						q = GetFieldQuery(field, analyzer, term.image);
					}
					break;
				case RANGEIN_START:
					jj_consume_token(RANGEIN_START);
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case RANGEIN_GOOP:
						goop1 = jj_consume_token(RANGEIN_GOOP);
						break;
					case RANGEIN_QUOTED:
						goop1 = jj_consume_token(RANGEIN_QUOTED);
						break;
					default:
						jj_la1[11] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case RANGEIN_TO:
						jj_consume_token(RANGEIN_TO);
						break;
					default:
						jj_la1[12] = jj_gen;
						break;
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case RANGEIN_GOOP:
						goop2 = jj_consume_token(RANGEIN_GOOP);
						break;
					case RANGEIN_QUOTED:
						goop2 = jj_consume_token(RANGEIN_QUOTED);
						break;
					default:
						jj_la1[13] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
					jj_consume_token(RANGEIN_END);
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case CARAT:
						jj_consume_token(CARAT);
						boost = jj_consume_token(NUMBER);
						break;
					default:
						jj_la1[14] = jj_gen;
						break;
				}
					if (goop1.kind == RANGEIN_QUOTED)
						goop1.image = goop1.image.Substring(1, goop1.image.Length-2);
					if (goop2.kind == RANGEIN_QUOTED)
						goop2.image = goop2.image.Substring(1, goop2.image.Length-2);

					q = GetRangeQuery(field, analyzer, goop1.image, goop2.image, true);
					break;
				case RANGEEX_START:
					jj_consume_token(RANGEEX_START);
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case RANGEEX_GOOP:
						goop1 = jj_consume_token(RANGEEX_GOOP);
						break;
					case RANGEEX_QUOTED:
						goop1 = jj_consume_token(RANGEEX_QUOTED);
						break;
					default:
						jj_la1[15] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case RANGEEX_TO:
						jj_consume_token(RANGEEX_TO);
						break;
					default:
						jj_la1[16] = jj_gen;
						break;
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case RANGEEX_GOOP:
						goop2 = jj_consume_token(RANGEEX_GOOP);
						break;
					case RANGEEX_QUOTED:
						goop2 = jj_consume_token(RANGEEX_QUOTED);
						break;
					default:
						jj_la1[17] = jj_gen;
						jj_consume_token(-1);
						throw new ParseException();
				}
					jj_consume_token(RANGEEX_END);
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case CARAT:
						jj_consume_token(CARAT);
						boost = jj_consume_token(NUMBER);
						break;
					default:
						jj_la1[18] = jj_gen;
						break;
				}
					if (goop1.kind == RANGEEX_QUOTED)
						goop1.image = goop1.image.Substring(1, goop1.image.Length-2);
					if (goop2.kind == RANGEEX_QUOTED)
						goop2.image = goop2.image.Substring(1, goop2.image.Length-2);

					q = GetRangeQuery(field, analyzer, goop1.image, goop2.image, false);
					break;
				case QUOTED:
					term = jj_consume_token(QUOTED);
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case SLOP:
						slop = jj_consume_token(SLOP);
						break;
					default:
						jj_la1[19] = jj_gen;
						break;
				}
				switch ((_jj_ntk==-1)?jj_ntk():_jj_ntk) 
				{
					case CARAT:
						jj_consume_token(CARAT);
						boost = jj_consume_token(NUMBER);
						break;
					default:
						jj_la1[20] = jj_gen;
						break;
				}
					q = GetFieldQuery(field, analyzer,
						term.image.Substring(1, term.image.Length-2));
					if (slop != null && q is  PhraseQuery) 
					{
						try 
						{
							int s = (int)Single.Parse(slop.image.Substring(1), NumberFormatInfo.InvariantInfo);
							((PhraseQuery) q).SetSlop(s);
						}
						catch (Exception) { }
					}
					break;
				default:
					jj_la1[21] = jj_gen;
					jj_consume_token(-1);
					throw new ParseException();
			}
			if (boost != null) 
			{
				float f = (float) 1.0;
				try 
				{
					f = Single.Parse(boost.image, NumberFormatInfo.InvariantInfo);
				}
				catch (Exception ) 
				{
					/* Should this be handled somehow? (defaults to "no boost", if
				 * boost number is invalid)
				 */
				}

				// avoid boosting null queries, such as those caused by stop words
				if (q != null) 
				{
					q.SetBoost(f);
				}
			}
		{if (true) return q;}
		}

		private bool jj_2_1(int xla) 
		{
			jj_la = xla; jj_lastpos = jj_scanpos = token;
			try { return !jj_3_1(); }
			catch(LookaheadSuccess) { return true; }
			finally { jj_save(0, xla); }
		}

		private bool jj_3_1() 
		{
			if (jj_scan_token(TERM)) return true;
			if (jj_scan_token(COLON)) return true;
			return false;
		}

		public QueryParserTokenManager token_source;
		public Token token, jj_nt;
		private int _jj_ntk;
		private Token jj_scanpos, jj_lastpos;
		private int jj_la;
		public bool lookingAhead = false;
		//private bool jj_semLA;
		private int jj_gen;
		readonly private int[] jj_la1 = new int[22];
		static private uint[] _jj_la1_0;
		static private int[] _jj_la1_1;
		static QueryParser() 
		{
			jj_la1_0();
			jj_la1_1();
		}
		private static void jj_la1_0() 
		{
			_jj_la1_0 = new uint[] {0x180,0x180,0xe00,0xe00,0x1f31f80,0x8000,0x1f31000,0x1320000,0x40000,0x40000,0x8000,0x18000000,0x2000000,0x18000000,0x8000,0x80000000,0x20000000,0x80000000,0x8000,0x80000,0x8000,0x1f30000,};
		}
		private static void jj_la1_1() 
		{
			_jj_la1_1 = new int[] {0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x1,0x0,0x1,0x0,0x0,0x0,0x0,};
		}
		readonly private JJCalls[] jj_2_rtns = new JJCalls[1];
		private bool jj_rescan = false;
		private int jj_gc = 0;

		public QueryParser(CharStream stream) 
		{
			token_source = new QueryParserTokenManager(stream);
			token = new Token();
			_jj_ntk = -1;
			jj_gen = 0;
			for (int i = 0; i < 22; i++) jj_la1[i] = -1;
			for (int i = 0; i < jj_2_rtns.Length; i++) jj_2_rtns[i] = new JJCalls();
		}

		public void ReInit(CharStream stream) 
		{
			token_source.ReInit(stream);
			token = new Token();
			_jj_ntk = -1;
			jj_gen = 0;
			for (int i = 0; i < 22; i++) jj_la1[i] = -1;
			for (int i = 0; i < jj_2_rtns.Length; i++) jj_2_rtns[i] = new JJCalls();
		}

		public QueryParser(QueryParserTokenManager tm) 
		{
			token_source = tm;
			token = new Token();
			_jj_ntk = -1;
			jj_gen = 0;
			for (int i = 0; i < 22; i++) jj_la1[i] = -1;
			for (int i = 0; i < jj_2_rtns.Length; i++) jj_2_rtns[i] = new JJCalls();
		}

		public void ReInit(QueryParserTokenManager tm) 
		{
			token_source = tm;
			token = new Token();
			_jj_ntk = -1;
			jj_gen = 0;
			for (int i = 0; i < 22; i++) jj_la1[i] = -1;
			for (int i = 0; i < jj_2_rtns.Length; i++) jj_2_rtns[i] = new JJCalls();
		}

		private Token jj_consume_token(int kind) 
		{
			Token oldToken = null;
			if ((oldToken = token).next != null) token = token.next;
			else token = token.next = token_source.getNextToken();
			_jj_ntk = -1;
			if (token.kind == kind) 
			{
				jj_gen++;
				if (++jj_gc > 100) 
				{
					jj_gc = 0;
					for (int i = 0; i < jj_2_rtns.Length; i++) 
					{
						JJCalls c = jj_2_rtns[i];
						while (c != null) 
						{
							if (c.gen < jj_gen) c.first = null;
							c = c.next;
						}
					}
				}
				return token;
			}
			token = oldToken;
			jj_kind = kind;
			throw GenerateParseException();
		}
		
		private sealed class LookaheadSuccess : Exception { }
		private LookaheadSuccess jj_ls = new LookaheadSuccess();

		private bool jj_scan_token(int kind) 
		{
			if (jj_scanpos == jj_lastpos) 
			{
				jj_la--;
				if (jj_scanpos.next == null) 
				{
					jj_lastpos = jj_scanpos = jj_scanpos.next = token_source.getNextToken();
				} 
				else 
				{
					jj_lastpos = jj_scanpos = jj_scanpos.next;
				}
			} 
			else 
			{
				jj_scanpos = jj_scanpos.next;
			}
			if (jj_rescan) 
			{
				int i = 0; Token tok = token;
				while (tok != null && tok != jj_scanpos) { i++; tok = tok.next; }
				if (tok != null) jj_add_error_token(kind, i);
			}
			return (jj_scanpos.kind != kind);
		}

		public Token GetNextToken() 
		{
			if (token.next != null) token = token.next;
			else token = token.next = token_source.getNextToken();
			_jj_ntk = -1;
			jj_gen++;
			return token;
		}

		public Token GetToken(int index) 
		{
			Token t = lookingAhead ? jj_scanpos : token;
			for (int i = 0; i < index; i++) 
			{
				if (t.next != null) t = t.next;
				else t = t.next = token_source.getNextToken();
			}
			return t;
		}

		private int jj_ntk() 
		{
			jj_nt = token.next;
			if (jj_nt == null)
			{
				token.next = token_source.getNextToken();
				return (_jj_ntk = token.next.kind);
			}
			else
				return (_jj_ntk = jj_nt.kind);
		}

		private ArrayList jj_expentries = new ArrayList();
		private int[] jj_expentry;
		private int jj_kind = -1;
		private int[] jj_lasttokens = new int[100];
		private int jj_endpos;

		private void jj_add_error_token(int kind, int pos) 
		{
			if (pos >= 100) return;
			if (pos == jj_endpos + 1) 
			{
				jj_lasttokens[jj_endpos++] = kind;
			} 
			else if (jj_endpos != 0) 
			{
				jj_expentry = new int[jj_endpos];
				for (int i = 0; i < jj_endpos; i++) 
				{
					jj_expentry[i] = jj_lasttokens[i];
				}
				bool exists = false;
				foreach (int[] oldentry in jj_expentries) 
				{
					if (oldentry.Length == jj_expentry.Length) 
					{
						exists = true;
						for (int i = 0; i < jj_expentry.Length; i++) 
						{
							if (oldentry[i] != jj_expentry[i]) 
							{
								exists = false;
								break;
							}
						}
						if (exists) break;
					}
				}
				if (!exists) jj_expentries.Add(jj_expentry);
				if (pos != 0) jj_lasttokens[(jj_endpos = pos) - 1] = kind;
			}
		}

		public ParseException GenerateParseException() 
		{
			jj_expentries.Clear();
			bool[] la1tokens = new bool[33];
			for (int i = 0; i < 33; i++) 
			{
				la1tokens[i] = false;
			}
			if (jj_kind >= 0) 
			{
				la1tokens[jj_kind] = true;
				jj_kind = -1;
			}
			for (int i = 0; i < 22; i++) 
			{
				if (jj_la1[i] == jj_gen) 
				{
					for (int j = 0; j < 32; j++) 
					{
						if ((_jj_la1_0[i] & (1<<j)) != 0) 
						{
							la1tokens[j] = true;
						}
						if ((_jj_la1_1[i] & (1<<j)) != 0) 
						{
							la1tokens[32+j] = true;
						}
					}
				}
			}
			for (int i = 0; i < 33; i++) 
			{
				if (la1tokens[i]) 
				{
					jj_expentry = new int[1];
					jj_expentry[0] = i;
					jj_expentries.Add(jj_expentry);
				}
			}
			jj_endpos = 0;
			jj_rescan_token();
			jj_add_error_token(0, 0);
			int[][] exptokseq = new int[jj_expentries.Count][];
			for (int i = 0; i < jj_expentries.Count; i++) 
			{
				exptokseq[i] = (int[])jj_expentries[i];
			}
			return new ParseException(token, exptokseq, tokenImage);
		}

		public void enable_tracing() 
		{
		}

		public void disable_tracing() 
		{
		}

		private void jj_rescan_token() 
		{
			jj_rescan = true;
			for (int i = 0; i < 1; i++) 
			{
				JJCalls p = jj_2_rtns[i];
				do 
				{
					if (p.gen > jj_gen) 
					{
						jj_la = p.arg; jj_lastpos = jj_scanpos = p.first;
						switch (i) 
						{
							case 0: jj_3_1(); break;
						}
					}
					p = p.next;
				} while (p != null);
			}
			jj_rescan = false;
		}

		private void jj_save(int index, int xla) 
		{
			JJCalls p = jj_2_rtns[index];
			while (p.gen > jj_gen) 
			{
				if (p.next == null) { p = p.next = new JJCalls(); break; }
				p = p.next;
			}
			p.gen = jj_gen + xla - jj_la; p.first = token; p.arg = xla;
		}

		class JJCalls 
		{
			internal int gen;
			internal Token first;
			internal int arg;
			internal JJCalls next;
		}
	}
}