/* sqldb.h
 *
 * Copyright (C) 1999,2000 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef __SQL_DB__
#define __SQL_DB__

#include <gtk/gtksignal.h>
#include <gtk/gtkobject.h>
#include <gtk/gtk.h>
#include "sqlqueryres.h"
#include "sqlaccess.h"
#include "gasql_xml.h"

typedef struct _gASQL_Main_Config gASQL_Main_Config;

#ifdef __cplusplus
extern "C"
{
#endif				/* __cplusplus */


/*
 *
 * Main object: SqlDb
 *
 */

#define SQL_DB(obj)          GTK_CHECK_CAST (obj, sql_db_get_type(), SqlDb)
#define SQL_DB_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_db_get_type (), SqlDbClass)
#define IS_SQL_DB(obj)       GTK_CHECK_TYPE (obj, sql_db_get_type ())

	typedef struct _SqlDb SqlDb;
	typedef struct _SqlDbClass SqlDbClass;
	typedef struct _SqlMemItem SqlMemItem;
	typedef struct _SqlMemItemClass SqlMemItemClass;
	typedef struct _SqlMemTable SqlMemTable;
	typedef struct _SqlMemTableClass SqlMemTableClass;
	typedef struct _SqlMemField SqlMemField;
	typedef struct _SqlMemFieldClass SqlMemFieldClass;
	typedef struct _SqlMemSeq SqlMemSeq;
	typedef struct _SqlMemSeqClass SqlMemSeqClass;
	typedef struct _SqlMemFlink SqlMemFlink;
	typedef struct _SqlMemFlinkClass SqlMemFlinkClass;


	/* struct for the object's data */
	struct _SqlDb
	{
		GtkObject object;

		/* Db accessibility */
		gchar *name;
		gboolean is_fault;
		GSList *tables;	/* list of data on SqlMemTable objects */
		GSList *sequences;	/* list of data on SqlMemSeq objects */
		GSList *field_links;	/* list of data on SqlMemFlink objects */
		GSList *users;	/* holds the list of users (ptrs to gchar) */

		SqlAccess *srv;
	};

	/* struct for the object's class */
	struct _SqlDbClass
	{
		GtkObjectClass parent_class;

		void (*updated) (SqlDb * db);
		void (*struct_saved) (SqlDb * db);
		void (*fault) (SqlDb * db);
		void (*table_created) (SqlDb * db, SqlMemTable * new_table);
		void (*table_created_f) (SqlDb * db, SqlMemTable * new_table);
		void (*table_dropped) (SqlDb * db, SqlMemTable * table);
		void (*seq_created) (SqlDb * db, SqlMemSeq * new_sequence);
		void (*seq_dropped) (SqlDb * db, SqlMemSeq * sequence);
		void (*field_created) (SqlDb * db, SqlMemTable * table,
				       SqlMemField * new_field);
		void (*field_dropped) (SqlDb * db, SqlMemTable * table,
				       SqlMemField * field);
		void (*ff_link_created) (SqlDb * db, SqlMemFlink * link);
		void (*ff_link_dropped) (SqlDb * db, SqlMemFlink * link);
		void (*fs_link_created) (SqlDb * db, SqlMemSeq * seq,
					 SqlMemField * f);
		void (*fs_link_dropped) (SqlDb * db, SqlMemSeq * seq,
					 SqlMemField * f);
		void (*progress) (SqlDb * db, gchar * msg, guint now,
				  guint total);
	};

	/* 
	 * generic widget's functions 
	 */
	GtkType sql_db_get_type (void);
	GtkObject *sql_db_new (SqlAccess * srv);
	void sql_db_dump_tables (SqlDb * db);	/* prints the structure */
	void sql_db_dump_links (SqlDb * db);
	void sql_db_dump_struct_as_graph (SqlDb * db, FILE * st);
	void sql_db_empty_all_cb (GtkObject * obj, SqlDb * db);
	GSList *find_user_name (SqlDb * db, gchar * uname);

	/*
	 * memory structure lookup
	 */
	SqlMemTable *sql_db_find_table_by_name (SqlDb * db, gchar * name);
	SqlMemTable *sql_db_find_table_by_xml_name (SqlDb * db,
						    gchar * xmlname);
	SqlMemTable *sql_db_find_table_from_field (SqlDb * db,
						   SqlMemField * field);

	SqlMemField *sql_db_find_field_from_names (SqlDb * db, gchar * table,
						   char *field);
	SqlMemField *sql_db_find_field_from_xml_name (SqlDb * db,
						      char *xmlname);
	/* returns the order of a field in the field list (1 for first element and
	   0 if field does not belong to any table).
	   if table = NULL, looks for the right table first. */
	guint sql_db_get_field_order (SqlDb * db, SqlMemTable * table,
				      SqlMemField * field);

	SqlMemSeq *sql_db_find_sequence_by_name (SqlDb * db, gchar * name);
	SqlMemSeq *sql_db_find_sequence_by_xml_name (SqlDb * db,
						     gchar * xmlname);
	SqlMemSeq *sql_db_find_sequence_to_field (SqlDb * db,
						  SqlMemField * field);

	/*
	 * Field links manipulation
	 */
	void sql_db_insert_seq_field_link (SqlDb * db, SqlMemSeq * seq,
					   SqlMemField * field);
	/* if seq==NULL, removes all sequence links to that field 
	   if field=NULL, removes all links from the sequence */
	void sql_db_delete_seq_field_link (SqlDb * db, SqlMemSeq * seq,
					   SqlMemField * field);
	void sql_db_insert_field_field_link (SqlDb * db, SqlMemField * from,
					     SqlMemField * to);
	/* if "from" = NULL, removes all links to the "to" field
	   if "to" = NULL, removes all links from the "from" field */
	void sql_db_delete_field_field_link (SqlDb * db, SqlMemField * from,
					     SqlMemField * to);
	/*
	 * Field links lookup
	 */
	SqlMemFlink *sql_db_find_flink_from_fields (SqlDb * db,
						    SqlMemField * fromf,
						    SqlMemField * tof);
	SqlMemFlink *sql_db_find_flink_from_xml_name (SqlDb * db,
						      gchar * xmlname);


	/*
	 * XML storage
	 */
	void sql_db_refresh (SqlDb * db, SqlAccess * srv);
	void sql_db_build_xml_tree (SqlDb * db, xmlDocPtr doc);
	gboolean sql_db_build_db_from_xml_tree (SqlDb * db, xmlNodePtr node);
	void sql_mem_link_load_from_xml_node (SqlDb * db,
					      xmlNodePtr tabletree);


/*
 *
 * Object common to memory representation of the DB: SqlMemItems
 * items such as tables, fields, sequences, links inherit this 
 * object. Not to be instanciated!
 *
 */
#define SQL_MEM_ITEM(obj)          GTK_CHECK_CAST (obj, sql_mem_item_get_type(), SqlMemItem)
#define SQL_MEM_ITEM_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_mem_item_get_type (), SqlMemItemClass)
#define IS_SQL_MEM_ITEM(obj)       GTK_CHECK_TYPE (obj, sql_mem_item_get_type ())

	/* struct for the object's data */
	struct _SqlMemItem
	{
		GtkObject object;
		gboolean updated;
	};

	/* struct for the object's class */
	struct _SqlMemItemClass
	{
		GtkObjectClass parent_class;

		void (*load_fault) (SqlMemItem * item);
	};

	/* generic widget's functions */
	GtkType sql_mem_item_get_type (void);
	GtkObject *sql_mem_item_new (void);



/*
 *
 * Object representing a table
 *
 */

#define SQL_MEM_TABLE(obj)          GTK_CHECK_CAST (obj, sql_mem_table_get_type(), SqlMemTable)
#define SQL_MEM_TABLE_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_mem_table_get_type (), SqlMemTableClass)
#define IS_SQL_MEM_TABLE(obj)       GTK_CHECK_TYPE (obj, sql_mem_table_get_type ())

	/* struct for the object's data */
	struct _SqlMemTable
	{
		SqlMemItem object;
		gchar *name;
		gchar *comments;
		gboolean is_user_comments;
		gchar *owner;	/* ptr do a gchar* which is part of a list => don't free */
		gboolean is_view;	/* view or table? */
		GSList *fields;	/* list data on SqlMemField */
		GSList *parents;	/* list data on SqlMemTable */
	};

	/* struct for the object's class */
	struct _SqlMemTableClass
	{
		SqlMemItemClass parent_class;

		void (*field_created) (SqlDb * db, SqlMemField * new_field);
		void (*field_dropped) (SqlDb * db, SqlMemField * field);
		void (*comments_changed) (SqlMemTable * table);
	};

	/* generic widget's functions */
	GtkType sql_mem_table_get_type (void);
	GtkObject *sql_mem_table_new (void);

	void sql_mem_table_dump_fields (SqlMemTable * t);
	void sql_mem_table_dump_as_graph (SqlMemTable * t, FILE * st);
	SqlMemField *sql_mem_table_find_field_by_name (SqlMemTable * t,
						       gchar * name);
	void sql_mem_table_set_comments (SqlMemTable * table,
					 gASQL_Main_Config * conf,
					 gchar * comments,
					 gboolean is_user_comments);
	void sql_mem_table_set_owner (SqlMemTable * table, gchar * owner);

/*
 *
 * Object representing a field
 *
 */

#define SQL_MEM_FIELD(obj)          GTK_CHECK_CAST (obj, sql_mem_field_get_type(), SqlMemField)
#define SQL_MEM_FIELD_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_mem_field_get_type (), SqlMemFieldClass)
#define IS_SQL_MEM_FIELD(obj)       GTK_CHECK_TYPE (obj, sql_mem_field_get_type ())

	/* struct for the object's data */
	struct _SqlMemField
	{
		SqlMemItem object;
		gchar *name;
		SqlDataType *type;
		gint length;	/* length of the field and -1 if variable (f.e. text) */
		gboolean null_allowed;	/* defaults to TRUE */
		gboolean is_key;	/* defaults to FALSE */
		/* put a pointer to a default value here */
		gchar *default_val;	/* the str is the string which needs to be included
					   into an insert string, without any modification
					   fe: 'hello' */
	};

	/* struct for the object's class */
	struct _SqlMemFieldClass
	{
		SqlMemItemClass parent_class;
	};

	/* generic widget's functions */
	GtkType sql_mem_field_get_type (void);
	GtkObject *sql_mem_field_new (void);

	void sql_mem_field_dump (SqlMemField * field);



/*
 *
 * Object representing a sequence
 *
 */

#define SQL_MEM_SEQ(obj)          GTK_CHECK_CAST (obj, sql_mem_seq_get_type(), SqlMemSeq)
#define SQL_MEM_SEQ_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_mem_seq_get_type (), SqlMemSeqClass)
#define IS_SQL_MEM_SEQ(obj)       GTK_CHECK_TYPE (obj, sql_mem_seq_get_type ())

	/* struct for the object's data */
	struct _SqlMemSeq
	{
		SqlMemItem object;
		gchar *name;
		gchar *comments;
		gchar *owner;	/* ptr do a gchar* which is part of a list => don't free */
		GSList *field_links;	/* data to SqlMemField objects on which the seq. acts */
	};

	/* struct for the object's class */
	struct _SqlMemSeqClass
	{
		SqlMemItemClass parent_class;
	};

	/* generic widget's functions */
	GtkType sql_mem_seq_get_type (void);
	GtkObject *sql_mem_seq_new (void);

	void sql_mem_seq_dump_as_graph (SqlMemSeq * seq, SqlDb * db,
					FILE * st);


/*
 *
 * Object representing a Field link
 *
 */

#define SQL_MEM_FLINK(obj)          GTK_CHECK_CAST (obj, sql_mem_flink_get_type(), SqlMemFlink)
#define SQL_MEM_FLINK_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, sql_mem_flink_get_type (), SqlMemFlinkClass)
#define IS_SQL_MEM_FLINK(obj)       GTK_CHECK_TYPE (obj, sql_mem_flink_get_type ())

	/* struct for the object's data */
	struct _SqlMemFlink
	{
		SqlMemItem object;
		gchar *name;
		SqlMemField *from;
		SqlMemField *to;
	};

	/* struct for the object's class */
	struct _SqlMemFlinkClass
	{
		SqlMemItemClass parent_class;
	};

	/* generic widget's functions */
	GtkType sql_mem_flink_get_type (void);
	GtkObject *sql_mem_flink_new (void);

	void sql_mem_flink_dump_as_graph (SqlMemFlink * flink,
					  SqlDb * db, FILE * st);

#ifdef __cplusplus
}
#endif				/* __cplusplus */

#endif
