/* sqlwiddbrel.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlwiddbrel.h"

#define DEFAULT_REL_FONT "-misc-fixed-medium-r-semicondensed-*-13-*-*-*-c-*-*"

static void sql_wid_db_rel_class_init (SqlWidDbRelClass * class);
static void sql_wid_db_rel_init (SqlWidDbRel * rel);
static void sql_wid_db_rel_destroy (GtkObject * obj);

static SqlTableItem *find_table (SqlWidDbRel * rel, SqlMemTable * table);
static SqlSeqItem *find_seq (SqlWidDbRel * rel, SqlMemSeq * seq);

/* Callbacks for signals of SqlDb Object */
static void sql_db_table_dropped_cb (SqlDb * db, SqlMemTable * table,
				     SqlWidDbRel * rel);
static void sql_db_seq_dropped_cb (SqlDb * db, SqlMemSeq * seq,
				   SqlWidDbRel * rel);
static void sql_db_updated_cb (SqlDb * db, SqlWidDbRel * rel);
static void sql_wid_db_load_shown_items (SqlWidDbRel * rel);

/* updates the SqlLinkItems on the sheet (called whenever an object is
 added or removed on the canvas or the links have been modified) */
static void update_all_links (SqlWidDbRel * rel);

/* functions to display a line with arrow to the parents and child
   tables when the mouse enters or leaves a table */
static void arrows_parents_get_coord (SqlTableItem * parentitem,
				      SqlTableItem * childitem,
				      double *px, double *py,
				      double *cx, double *cy);
static void table_parents_links_display_cb (SqlTableItem * item,
					    SqlWidDbRel * rel);
static void table_parents_links_remove_cb (SqlTableItem * item,
					   SqlWidDbRel * rel);


static void sql_link_item_update_pos (SqlLinkItem * item);

/* CB when tables or sequences are moved around */
static void titem_moved_cb (SqlTableItem * item,
			    gfloat x, gfloat y, SqlWidDbRel * rel);
static void sitem_moved_cb (SqlSeqItem * item,
			    gfloat x, gfloat y, SqlWidDbRel * rel);

/* signals */
enum
{
	TABLE_SHOWN,
	TABLE_HIDDEN,
	SEQ_SHOWN,
	SEQ_HIDDEN,
	LASTREL_SIGNAL
};

static gint sql_wid_db_rel_signals[LASTREL_SIGNAL] = { 0, 0, 0, 0 };


guint
sql_wid_db_rel_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Wid_Db_Rel",
			sizeof (SqlWidDbRel),
			sizeof (SqlWidDbRelClass),
			(GtkClassInitFunc) sql_wid_db_rel_class_init,
			(GtkObjectInitFunc) sql_wid_db_rel_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_vbox_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_wid_db_rel_class_init (SqlWidDbRelClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
	sql_wid_db_rel_signals[TABLE_SHOWN] =
		gtk_signal_new ("table_shown",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlWidDbRelClass,
						   table_shown),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_wid_db_rel_signals[TABLE_HIDDEN] =
		gtk_signal_new ("table_hidden",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlWidDbRelClass,
						   table_hidden),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_wid_db_rel_signals[SEQ_SHOWN] =
		gtk_signal_new ("seq_shown", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlWidDbRelClass,
						   seq_shown),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_wid_db_rel_signals[SEQ_HIDDEN] =
		gtk_signal_new ("seq_hidden",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlWidDbRelClass,
						   seq_hidden),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, sql_wid_db_rel_signals,
				      LASTREL_SIGNAL);
	class->table_shown = NULL;
	class->table_hidden = NULL;
	class->seq_shown = NULL;
	class->seq_hidden = NULL;
	object_class->destroy = sql_wid_db_rel_destroy;
}

static void
canvas_allocated_cb (GtkObject * canvas, GtkAllocation * allocation,
		     SqlWidDbRel * rel)
{
	gint xoff, yoff;
	GtkWidget *sw;
	
	sw =  gtk_object_get_data (GTK_OBJECT (rel), "sw");
	/* The 18 comes from the size of the scrollbars */
	xoff = GTK_SCROLLED_WINDOW (sw)->vscrollbar_visible ? 18 : 0;
	yoff = GTK_SCROLLED_WINDOW (sw)->hscrollbar_visible ? 18 : 0;
	rel->mem_link->size.width = allocation->width + xoff;
	rel->mem_link->size.height = allocation->height + yoff;

	gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 0.0, 0.0,
	 				allocation->width, allocation->height);
}

#define CANVAS_H_SIZE 300
#define CANVAS_V_SIZE 200
static void h_adj_changed_cb(GtkAdjustment *adj, SqlWidDbRel * rel);
static void v_adj_changed_cb(GtkAdjustment *adj, SqlWidDbRel * rel);

static void
sql_wid_db_rel_init (SqlWidDbRel * rel)
{
	GtkWidget *frame, *canvas, *sw;
	GtkAdjustment *adj;
	
	/* A little frame */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	gtk_box_pack_start (GTK_BOX (rel), frame, TRUE, TRUE, 0);
	gtk_widget_show (frame);

	/* scrolled window */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (frame), sw);
	gtk_object_set_data (GTK_OBJECT (rel), "sw", sw);

	/* canvas */
	canvas = gnome_canvas_new ();
	rel->canvas = GNOME_CANVAS (canvas);
	gtk_widget_set_usize (canvas, CANVAS_H_SIZE, CANVAS_V_SIZE);
	gtk_signal_connect (GTK_OBJECT (canvas), "size_allocate",
			    GTK_SIGNAL_FUNC (canvas_allocated_cb), rel);
	gtk_container_add (GTK_CONTAINER (sw), canvas);
	
	gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 0.0, 0.0,
	 				CANVAS_H_SIZE, CANVAS_V_SIZE);

	/* setting adjustments' increments */
	adj = gtk_layout_get_hadjustment (GTK_LAYOUT (canvas));
	GTK_ADJUSTMENT (adj)->step_increment = 10;
	adj = gtk_layout_get_vadjustment (GTK_LAYOUT (canvas));
	GTK_ADJUSTMENT (adj)->step_increment = 10;

	gtk_widget_show_all (sw);
}

static void 
h_adj_changed_cb(GtkAdjustment *adj, SqlWidDbRel * rel)
{
	rel->mem_link->h_adj_val = adj->value;
	rel->mem_link->h_adj_upper = adj->upper;
	rel->mem_link->h_adj_lower = adj->lower;
	if (rel->mem_link->set_canvas_scroll) {
		rel->mem_link->set_canvas_scroll = FALSE;
		gnome_canvas_set_scroll_region (GNOME_CANVAS (rel->canvas),
						rel->mem_link->h_adj_lower, rel->mem_link->v_adj_lower,
						rel->mem_link->h_adj_upper -1, rel->mem_link->v_adj_upper -1);
	}
}

static void 
v_adj_changed_cb(GtkAdjustment *adj, SqlWidDbRel * rel)
{
	rel->mem_link->v_adj_val = adj->value;
	rel->mem_link->v_adj_upper = adj->upper;
	rel->mem_link->v_adj_lower = adj->lower;
	if (rel->mem_link->set_canvas_scroll) {
		rel->mem_link->set_canvas_scroll = FALSE;
		gnome_canvas_set_scroll_region (GNOME_CANVAS (rel->canvas),
						rel->mem_link->h_adj_lower, rel->mem_link->v_adj_lower,
						rel->mem_link->h_adj_upper - 1, rel->mem_link->v_adj_upper - 1);
	}
}


/* increase or descrease the horizontal and vertical adjustments to
 * have a window matching as closely as possible the current canvas contents
 */
static void update_canvas_adjustments(SqlWidDbRel *rel, gboolean reduce);


/* BEWARE: name must not be NULL !*/
GtkWidget *
sql_wid_db_rel_new (gASQL_Main_Config * conf, gchar * name)
{
	GtkObject *obj;
	SqlWidDbRel *rel;
	GSList *list;
	WidRelData *found;
	GtkAdjustment *hadj, *vadj;
	gfloat hhold, vhold;

	obj = gtk_type_new (sql_wid_db_rel_get_type ());
	rel = SQL_WID_DB_REL (obj);
	rel->conf = conf;
	rel->dis_tables = NULL;
	rel->dis_seqs = NULL;
	rel->dis_links = NULL;
	rel->tmp_links = NULL;
	conf->save_up_to_date = FALSE;

	/* find if a WidRelData exists for that name */
	list = conf->widdbrel_list;
	found = NULL;
	while (list && !found) {
		found = (WidRelData *) (list->data);
		if (strcmp (found->name, name)) {
			found = NULL;
			list = g_slist_next (list);
		}
	}
	if (!found) {
		found = g_new0(WidRelData, 1);
		found->name = g_strdup (name);
		found->nodes = NULL;
		found->size.width = 300;	/* default width for the canvas */
		found->size.height = 200;	/* default height for the canvas */
		found->h_adj_upper = CANVAS_H_SIZE;
		found->v_adj_upper = CANVAS_V_SIZE;
		found->h_adj_lower = 0.;
		found->v_adj_lower = 0.;
		found->set_canvas_scroll = FALSE;
		conf->widdbrel_list =
			g_slist_append (conf->widdbrel_list, found);
	}
	else
		gtk_widget_set_usize (GTK_WIDGET (rel->canvas),
				      found->size.width, found->size.height);
	rel->mem_link = found;
	hhold = rel->mem_link->h_adj_val;
	vhold = rel->mem_link->v_adj_val;

	/* loading the tables that were displayed the last time */
	sql_wid_db_load_shown_items (rel);

	/* setting up the signals connexions */
	gtk_signal_connect_while_alive (GTK_OBJECT (conf->db),
					"table_dropped",
					GTK_SIGNAL_FUNC
					(sql_db_table_dropped_cb), rel,
					GTK_OBJECT (rel));
	gtk_signal_connect_while_alive (GTK_OBJECT (conf->db), "seq_dropped",
					GTK_SIGNAL_FUNC
					(sql_db_seq_dropped_cb), rel,
					GTK_OBJECT (rel));
	gtk_signal_connect_while_alive (GTK_OBJECT (conf->db), "updated",
					GTK_SIGNAL_FUNC (sql_db_updated_cb),
					rel, GTK_OBJECT (rel));



	hadj = gtk_layout_get_hadjustment (GTK_LAYOUT (rel->canvas));
	vadj = gtk_layout_get_vadjustment (GTK_LAYOUT (rel->canvas));

	/* adjustments signals */
	gtk_signal_connect (GTK_OBJECT (hadj), "changed",
			    GTK_SIGNAL_FUNC (h_adj_changed_cb), rel);
	gtk_signal_connect (GTK_OBJECT (hadj), "value_changed",
			    GTK_SIGNAL_FUNC (h_adj_changed_cb), rel);


	gtk_signal_connect (GTK_OBJECT (vadj), "changed",
			    GTK_SIGNAL_FUNC (v_adj_changed_cb), rel);
	gtk_signal_connect (GTK_OBJECT (vadj), "value_changed",
			    GTK_SIGNAL_FUNC (v_adj_changed_cb), rel);

	/* setting the right values for scrollbars */
	rel->mem_link->set_canvas_scroll = TRUE;
	hadj->upper = rel->mem_link->h_adj_upper;
	hadj->lower = rel->mem_link->h_adj_lower;

	gtk_adjustment_changed (hadj);
	gtk_adjustment_set_value (hadj, hhold);

	vadj->upper = rel->mem_link->v_adj_upper;
	vadj->lower = rel->mem_link->v_adj_lower;

	gtk_adjustment_changed (vadj);
	gtk_adjustment_set_value (vadj, vhold);
	rel->mem_link->set_canvas_scroll = FALSE;

	update_canvas_adjustments(rel, TRUE);

	return GTK_WIDGET (obj);
}

static void
sql_wid_db_load_shown_items (SqlWidDbRel * rel)
{
	GSList *list;
	WidRelNode *node;
	GtkObject *item;

	list = rel->mem_link->nodes;
	while (list) {
		node = (WidRelNode *) (list->data);
		if (node->shown) {
			if (g_slist_find
			    (rel->conf->db->tables, node->object)) {
				item = sql_table_item_new (rel, node);
				rel->dis_tables =
					g_slist_append (rel->dis_tables,
							item);
				gtk_signal_connect (GTK_OBJECT (item),
						    "moved",
						    GTK_SIGNAL_FUNC
						    (titem_moved_cb), rel);
				gtk_signal_connect (GTK_OBJECT (item),
						    "entered",
						    GTK_SIGNAL_FUNC
						    (table_parents_links_display_cb),
						    rel);
				gtk_signal_connect (GTK_OBJECT (item), "left",
						    GTK_SIGNAL_FUNC
						    (table_parents_links_remove_cb),
						    rel);
			}
			else {
				item = sql_seq_item_new (rel, node);
				rel->dis_seqs =
					g_slist_append (rel->dis_seqs, item);
				gtk_signal_connect (GTK_OBJECT (item),
						    "moved",
						    GTK_SIGNAL_FUNC
						    (sitem_moved_cb), rel);
			}
		}
		list = g_slist_next (list);
	}
	update_all_links (rel);
}

static void
sql_wid_db_rel_destroy (GtkObject * object)
{
	SqlWidDbRel *rel;
	GtkObjectClass *parent_class;
	GSList *list;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_WID_DB_REL (object));

	/* the item's data to be freeed */
	rel = SQL_WID_DB_REL (object);

	list = rel->dis_tables;
	while (list) {
		gtk_object_destroy (GTK_OBJECT (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (rel->dis_tables);
	rel->dis_tables = NULL;


	list = rel->dis_links;
	while (list) {
		gtk_object_destroy (GTK_OBJECT (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (rel->dis_links);
	rel->dis_links = NULL;


	gtk_object_destroy (GTK_OBJECT (rel->canvas));

	/* what about the parent class */
	parent_class = gtk_type_class (gtk_object_get_type ());

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void
sql_wid_db_rel_build_xml_tree (WidRelData * data, xmlNodePtr doc,
			       gASQL_Main_Config * conf)
{
	xmlNodePtr toptree, tree;
	GSList *list;
	WidRelNode *node;
	gchar *str;

	toptree = xmlNewChild (doc, NULL, "dbrel", NULL);
	str = g_strdup_printf ("DR%s", data->name);
	xmlSetProp (toptree, "name", str);
	g_free (str);
	str = g_strdup_printf ("%d", data->size.width);
	xmlSetProp (toptree, "width", str);
	g_free (str);
	str = g_strdup_printf ("%d", data->size.height);
	xmlSetProp (toptree, "height", str);
	g_free (str);
	str = g_strdup_printf ("%d", (int) data->h_adj_upper);
	xmlSetProp (toptree, "hmax", str);
	g_free (str);
	str = g_strdup_printf ("%d", (int) data->v_adj_upper);
	xmlSetProp (toptree, "vmax", str);
	g_free (str);
	str = g_strdup_printf ("%d", (int) data->h_adj_val);
	xmlSetProp (toptree, "hpos", str);
	g_free (str);
	str = g_strdup_printf ("%d", (int) data->v_adj_val);
	xmlSetProp (toptree, "vpos", str);
	g_free (str);
	list = data->nodes;
	while (list) {
		node = (WidRelNode *) (list->data);
		tree = xmlNewChild (toptree, NULL, "relnode", NULL);
		if (g_slist_find (conf->db->tables, node->object)) {	/* table ? */
			str = g_strdup_printf ("TV%s",
					       SQL_MEM_TABLE (node->object)->
					       name);
			xmlSetProp (tree, "object", str);
			g_free (str);
		}
		else {
			if (g_slist_find (conf->db->sequences, node->object)) {	/* sequence ? */
				str = g_strdup_printf ("SE%s",
						       SQL_MEM_SEQ (node->
								    object)->
						       name);
				xmlSetProp (tree, "object", str);
				g_free (str);
			}
		}

		if (node->shown)
			xmlSetProp (tree, "shown", "t");
		else
			xmlSetProp (tree, "shown", "f");
		str = g_strdup_printf ("%d", (int) (node->x * 100.));
		xmlSetProp (tree, "x", str);
		g_free (str);
		str = g_strdup_printf ("%d", (int) (node->y * 100.));
		xmlSetProp (tree, "y", str);
		g_free (str);

		list = g_slist_next (list);
	}
}

static GSList *sql_wid_db_rel_build_from_xml_tree (GSList * list,
						   SqlDb * db,
						   xmlNodePtr node);

gboolean
sql_wid_db_rels_build_from_xml_tree (gASQL_Main_Config * conf,
				     xmlNodePtr node)
{
	gchar *str;
	xmlNodePtr subtree, subsubtree;
	WidRelData *reldata;

	/* loading all the 'dbrel' tags */
	subtree = node->xmlChildrenNode;
	while (subtree) {
		reldata = g_new0 (WidRelData, 1);
		reldata->h_adj_lower = 0.;
		reldata->v_adj_lower = 0.;

		str = xmlGetProp (subtree, "name");
		reldata->name = g_strdup (str + 2);
		g_free (str);

		str = xmlGetProp (subtree, "width");
		if (str) {
			reldata->size.width = atoi (str);
			g_free (str);
		}
		else
			reldata->size.width = 300;

		str = xmlGetProp (subtree, "height");
		if (str) {
			reldata->size.height = atoi (str);
			g_free (str);
		}
		else
			reldata->size.height = 200;

		str = xmlGetProp (subtree, "hmax");
		if (str) {
			reldata->h_adj_upper = atoi (str);
			g_free (str);
		}
		else
			reldata->h_adj_upper = CANVAS_H_SIZE;

		str = xmlGetProp (subtree, "vmax");
		if (str) {
			reldata->v_adj_upper = atoi (str);
			g_free (str);
		}
		else
			reldata->v_adj_upper = CANVAS_V_SIZE;

		str = xmlGetProp (subtree, "hpos");
		if (str) {
			reldata->h_adj_val = atoi (str);
			g_free (str);
		}

		str = xmlGetProp (subtree, "vpos");
		if (str) {
			reldata->v_adj_val = atoi (str);
			g_free (str);
		}

		reldata->nodes = NULL;
		subsubtree = subtree->xmlChildrenNode;
		/* loading all the 'relnode' tags */
		while (subsubtree) {
			reldata->nodes =
				sql_wid_db_rel_build_from_xml_tree (reldata->
								    nodes,
								    conf->db,
								    subsubtree);
			subsubtree = subsubtree->next;
		}
		subtree = subtree->next;

		conf->widdbrel_list = g_slist_append (conf->widdbrel_list,
						      reldata);
	}

	return TRUE;
}

static GSList *
sql_wid_db_rel_build_from_xml_tree (GSList * list,
				    SqlDb * db, xmlNodePtr node)
{
	gchar *txt;
	double xy;
	WidRelNode *rnode;

	rnode = g_new0 (WidRelNode, 1);
	txt = xmlGetProp (node, "x");
	xy = atof (txt);
	rnode->x = xy / 100.;
	g_free (txt);

	txt = xmlGetProp (node, "y");
	xy = atof (txt);
	rnode->y = xy / 100.;
	g_free (txt);

	txt = xmlGetProp (node, "shown");
	rnode->shown = FALSE;
	if (txt) {
		if (*txt == 't')
			rnode->shown = TRUE;
		g_free (txt);
	}

	txt = xmlGetProp (node, "object");
	if ((*txt == 'T') && (*(txt + 1) == 'V')) {	/* field */
		SqlMemTable *table;
		table = sql_db_find_table_by_xml_name (db, txt);
		rnode->object = table;
	}
	else {			/* sequence */

		SqlMemSeq *seq;
		seq = sql_db_find_sequence_by_xml_name (db, txt);
		rnode->object = seq;
	}
	g_free (txt);

	list = g_slist_append (list, rnode);

	return list;
}

static void
update_all_links (SqlWidDbRel * rel)
{
	GSList *list, *list2, *hlist;
	SqlMemTable *t1, *t2;
	SqlTableItem *ti1, *ti2;
	SqlSeqItem *si;
	gboolean found;

	/* -1- going through all the known links to see if some are missing on the 
	   canvas */
	/* -1.1- field to field links */
	list = rel->conf->db->field_links;
	while (list) {
		t1 = sql_db_find_table_from_field (rel->conf->db,
						   SQL_MEM_FLINK (list->
								  data)->
						   from);
		t2 = sql_db_find_table_from_field (rel->conf->db,
						   SQL_MEM_FLINK (list->
								  data)->to);
		if ((ti1 = find_table (rel, t1))
		    && (ti2 = find_table (rel, t2))) {
			/* here a link exists between two tables displayed. let's see if there is
			   a SqlLinkItem for that link */
			GSList *l2 = rel->dis_links;
			found = FALSE;
			while (l2 && !found) {
				if (SQL_LINK_ITEM (l2->data)->link_mem ==
				    list->data)
					found = TRUE;
				l2 = g_slist_next (l2);
			}
			if (!found) {	/* create the link */
				rel->dis_links =
					g_slist_append (rel->dis_links,
							sql_link_item_new_field_field
							(rel->canvas, ti1,
							 ti2,
							 SQL_MEM_FLINK (list->
									data)));
			}
		}

		list = g_slist_next (list);
	}

	/* -1.2- links from a sequence to fields */
	list = rel->dis_seqs;
	while (list) {
		list2 = SQL_SEQ_ITEM (list->data)->seq->field_links;
		while (list2) {	/* list2->data is a SQL_MEM_FIELD */
			t1 = sql_db_find_table_from_field (rel->conf->db,
							   SQL_MEM_FIELD
							   (list2->data));
			if ((ti1 = find_table (rel, t1))) {
				GSList *l2 = rel->dis_links;
				gboolean found = FALSE;
				while (l2 && !found) {
					if (SQL_LINK_ITEM (l2->data)->
					    link_mem == list2->data)
						found = TRUE;
					l2 = g_slist_next (l2);
				}
				if (!found) {	/* create the link */
					rel->dis_links =
						g_slist_append (rel->
								dis_links,
								sql_link_item_new_seq_field
								(rel->canvas,
								 SQL_SEQ_ITEM
								 (list->data),
								 ti1,
								 SQL_MEM_FIELD
								 (list2->
								  data)));
				}
			}
			list2 = g_slist_next (list2);
		}
		list = g_slist_next (list);
	}

	/* -2- going through all the SqlLinkItem list to see if each link 
	   should still be there */
	list = rel->dis_links;
	while (list) {
		/* is the link still in memory */
		found = FALSE;
		switch (SQL_LINK_ITEM (list->data)->link_type) {
		case SQL_LINK_FIELD:
			if (g_slist_find (rel->conf->db->field_links,
					  SQL_LINK_ITEM (list->data)->
					  link_mem))
				found = TRUE;
			break;
		case SQL_LINK_SEQUENCE:
			if (g_slist_find
			    (SQL_SEQ_ITEM (SQL_LINK_ITEM (list->data)->from)->
			     seq->field_links,
			     SQL_LINK_ITEM (list->data)->link_mem))
				found = TRUE;
			break;
		}

		if (!found) {
			gtk_object_destroy (GTK_OBJECT (list->data));
			hlist = g_slist_next (list);
			rel->dis_links = g_slist_remove_link (rel->dis_links, list);
			g_slist_free_1 (list);
			list = hlist;
		}
		else {
			/* are the tables for the link represented on the canvas ? */
			switch (SQL_LINK_ITEM (list->data)->link_type) {
			case SQL_LINK_FIELD:
				t1 = sql_db_find_table_from_field (rel->conf->
								   db,
								   SQL_MEM_FLINK
								   (SQL_LINK_ITEM
								    (list->
								     data)->
								    link_mem)->
								   from);
				t2 = sql_db_find_table_from_field (rel->conf->
								   db,
								   SQL_MEM_FLINK
								   (SQL_LINK_ITEM
								    (list->
								     data)->
								    link_mem)->
								   to);
				ti1 = find_table (rel, t1);
				ti2 = find_table (rel, t2);
				if (!ti1 || !ti2) {
					gtk_object_destroy (GTK_OBJECT
							    (list->data));
					hlist = g_slist_next (list);
					rel->dis_links = g_slist_remove_link (rel->dis_links,
									      list);
					g_slist_free_1 (list);
					list = hlist;
				}
				else
					list = g_slist_next (list);
				break;
			case SQL_LINK_SEQUENCE:
				t2 = sql_db_find_table_from_field (rel->conf->
								   db,
								   SQL_MEM_FIELD
								   (SQL_LINK_ITEM
								    (list->
								     data)->
								    link_mem));
				ti2 = find_table (rel, t2);
				si = find_seq (rel,
					       SQL_SEQ_ITEM (SQL_LINK_ITEM
							     (list->data)->
							     from)->seq);
				if (!si || !ti2) {
					gtk_object_destroy (GTK_OBJECT
							    (list->data));
					hlist = g_slist_next (list);
					rel->dis_links = g_slist_remove_link (rel->dis_links,
									      list);
					g_slist_free_1 (list);
					list = hlist;
				}
				else
					list = g_slist_next (list);
				break;
			}
		}
	}
}

/***********************************/
/* Signals CALLBACKS               */
/***********************************/

static void
sql_db_table_dropped_cb (SqlDb * db, SqlMemTable * table, SqlWidDbRel * rel)
{
	GSList *list, *hold;
	gboolean found;

	sql_wid_db_rel_hide_table (rel, table);

	/* now remove the WidRelNode for that sequence */
	found = FALSE;
	list = rel->mem_link->nodes;
	while (list && !found) {
		if (((WidRelNode *) (list->data))->object == table) {
			found = TRUE;
			g_free (list->data);
			hold = list;
			rel->mem_link->nodes = g_slist_remove_link (rel->mem_link->nodes,
								    list);
			g_slist_free_1 (hold);
		}
		else
			list = g_slist_next (list);
	}
}

static void
sql_db_seq_dropped_cb (SqlDb * db, SqlMemSeq * seq, SqlWidDbRel * rel)
{
	GSList *list, *hold;
	gboolean found;

	sql_wid_db_rel_hide_seq (rel, seq);

	/* now remove the WidRelNode for that sequence */
	found = FALSE;
	list = rel->mem_link->nodes;
	while (list && !found) {
		if (((WidRelNode *) (list->data))->object == seq) {
			found = TRUE;
			g_free (list->data);
			hold = list;
			rel->mem_link->nodes = g_slist_remove_link (rel->mem_link->nodes,
								    list);
			g_slist_free_1 (hold);
		}
		else
			list = g_slist_next (list);
	}
}

static void
sql_db_updated_cb (SqlDb * db, SqlWidDbRel * rel)
{
	update_all_links (rel);
}


static void
table_parents_links_display_cb (SqlTableItem * item, SqlWidDbRel * rel)
{
	GSList *l, *node;
	GnomeCanvasItem *newitem;
	GnomeCanvasPoints *points;
	SqlTableItem *ptitem;

	points = gnome_canvas_points_new (2);
	/* find this table parents */
	l = item->table->parents;
	while (l) {
		ptitem = find_table (rel, SQL_MEM_TABLE (l->data));
		if (ptitem) {
			arrows_parents_get_coord (ptitem, item,
						  points->coords,
						  points->coords + 1,
						  points->coords + 2,
						  points->coords + 3);
			newitem =
				gnome_canvas_item_new (gnome_canvas_root
						       (rel->canvas),
						       gnome_canvas_line_get_type
						       (), "points", points,
						       "fill_color",
						       "RoyalBlue1",
						       "width_units", 5.,
						       "last_arrowhead", TRUE,
						       "arrow_shape_a", 10.,
						       "arrow_shape_b", 10.,
						       "arrow_shape_c", 10.,
						       NULL);
			gnome_canvas_item_lower_to_bottom (newitem);
			gtk_object_set_data (GTK_OBJECT (newitem), "pitem",
					     ptitem);
			gtk_object_set_data (GTK_OBJECT (newitem), "citem",
					     item);
			rel->tmp_links =
				g_slist_prepend (rel->tmp_links, newitem);
		}
		l = g_slist_next (l);
	}
	/* find this table childs */
	l = rel->conf->db->tables;
	while (l) {
		if ((node =
		     g_slist_find (SQL_MEM_TABLE (l->data)->parents,
				   item->table))) {
			ptitem = find_table (rel, SQL_MEM_TABLE (l->data));
			if (ptitem) {
				arrows_parents_get_coord (item, ptitem,
							  points->coords,
							  points->coords + 1,
							  points->coords + 2,
							  points->coords + 3);
				newitem =
					gnome_canvas_item_new
					(gnome_canvas_root (rel->canvas),
					 gnome_canvas_line_get_type (),
					 "points", points, "fill_color",
					 "DodgerBlue1", "width_units", 5.,
					 "last_arrowhead", TRUE,
					 "arrow_shape_a", 10.,
					 "arrow_shape_b", 10.,
					 "arrow_shape_c", 10., NULL);
				gnome_canvas_item_lower_to_bottom (newitem);
				gtk_object_set_data (GTK_OBJECT (newitem),
						     "pitem", item);
				gtk_object_set_data (GTK_OBJECT (newitem),
						     "citem", ptitem);
				rel->tmp_links =
					g_slist_prepend (rel->tmp_links,
							 newitem);
			}
		}
		l = g_slist_next (l);
	}

	/* free the points object */
	gnome_canvas_points_free (points);
}
static void
table_parents_links_remove_cb (SqlTableItem * item, SqlWidDbRel * rel)
{
	GSList *hold;
	while (rel->tmp_links) {
		gtk_object_destroy (GTK_OBJECT (rel->tmp_links->data));
		hold = rel->tmp_links;
		rel->tmp_links = g_slist_remove_link (rel->tmp_links, rel->tmp_links);
		g_slist_free_1 (hold);
	}
}

/***********************************/
/* ADD/Remove objects CALLBACKS/FNs*/
/***********************************/
static void
titem_moved_cb (SqlTableItem * item, gfloat x, gfloat y, SqlWidDbRel * rel)
{
	GnomeCanvasPoints *points;
	GSList *list;
	SqlTableItem *pitem, *citem;

	/* move the parents' links that have been displayed to follow the 
	   movement! */
	points = gnome_canvas_points_new (2);
	list = rel->tmp_links;
	while (list) {
		pitem = gtk_object_get_data (GTK_OBJECT (list->data),
					     "pitem");
		citem = gtk_object_get_data (GTK_OBJECT (list->data),
					     "citem");
		if (pitem && citem) {
			arrows_parents_get_coord (pitem, citem,
						  points->coords,
						  points->coords + 1,
						  points->coords + 2,
						  points->coords + 3);
			gnome_canvas_item_set (GNOME_CANVAS_ITEM (list->data),
					       "points", points, NULL);
		}
		list = g_slist_next (list);
	}
	gnome_canvas_points_free (points);
	rel->conf->save_up_to_date = FALSE;
}

static void
sitem_moved_cb (SqlSeqItem * item, gfloat x, gfloat y, SqlWidDbRel * rel)
{
	rel->conf->save_up_to_date = FALSE;
}

/********************************/
/* showing / hiding tables      */
/********************************/
void
sql_wid_db_rel_show_table (SqlWidDbRel * rel, SqlMemTable * table)
{
	gboolean send_sig = FALSE;
	GSList *list;
	WidRelNode *node;

	if (table && !find_table (rel, table)) {
		GtkObject *titem;

		/* if necessary create a new node for the table */
		list = rel->mem_link->nodes;
		node = NULL;
		while (!node && list) {
			if (((WidRelNode *) (list->data))->object == table)
				node = (WidRelNode *) (list->data);
			list = g_slist_next (list);
		}
		if (!node) {
			node = (WidRelNode *) g_malloc (sizeof (WidRelNode));
			node->x = 100.;
			node->y = 100.;
			node->object = table;
			rel->mem_link->nodes =
				g_slist_append (rel->mem_link->nodes, node);
		}
		node->shown = TRUE;

		titem = sql_table_item_new (rel, node);
		rel->dis_tables = g_slist_append (rel->dis_tables, titem);
		gtk_signal_connect (GTK_OBJECT (titem), "moved",
				    GTK_SIGNAL_FUNC (titem_moved_cb), rel);
		gtk_signal_connect (GTK_OBJECT (titem), "entered",
				    GTK_SIGNAL_FUNC
				    (table_parents_links_display_cb), rel);
		gtk_signal_connect (GTK_OBJECT (titem), "left",
				    GTK_SIGNAL_FUNC
				    (table_parents_links_remove_cb), rel);
		rel->conf->save_up_to_date = FALSE;
		send_sig = TRUE;
	}
	update_all_links (rel);
	if (send_sig) {
#ifdef debug_signal
		g_print (">> 'TABLE_SHOWN' from sql_wid_db_rel_show_table\n");
#endif
		gtk_signal_emit (GTK_OBJECT (rel),
				 sql_wid_db_rel_signals[TABLE_SHOWN], table);
#ifdef debug_signal
		g_print ("<< 'TABLE_SHOWN' from sql_wid_db_rel_show_table\n");
#endif
	}
}

void
sql_wid_db_rel_show_table_cb (GtkObject * obj, SqlMemTable * table,
			      SqlWidDbRel * rel)
{
	sql_wid_db_rel_show_table (rel, table);
}

void
sql_wid_db_rel_hide_table (SqlWidDbRel * rel, SqlMemTable * table)
{
	SqlTableItem *found;
	gboolean send_sig = FALSE;

	if (table && (found = find_table (rel, table))) {
		found->node->shown = FALSE;
		gtk_object_destroy (GTK_OBJECT (found));
		rel->dis_tables = g_slist_remove (rel->dis_tables, found);
		rel->conf->save_up_to_date = FALSE;
		send_sig = TRUE;
	}
	update_all_links (rel);
	if (send_sig) {
#ifdef debug_signal
		g_print (">> 'TABLE_HIDDEN' from sql_wid_db_rel_hide_table\n");
#endif
		gtk_signal_emit (GTK_OBJECT (rel),
				 sql_wid_db_rel_signals[TABLE_HIDDEN], table);
#ifdef debug_signal
		g_print ("<< 'TABLE_HIDDEN' from sql_wid_db_rel_hide_table\n");
#endif
	}
}

void
sql_wid_db_rel_hide_table_cb (GtkObject * obj, SqlMemTable * table,
			      SqlWidDbRel * rel)
{
	sql_wid_db_rel_hide_table (rel, table);
}

void
sql_wid_db_rel_toggle_table (SqlWidDbRel * rel, SqlMemTable * table)
{
	SqlTableItem *found;

	found = find_table (rel, table);
	if (found) {
		sql_wid_db_rel_hide_table (rel, table);
	}
	else if (table) {
		sql_wid_db_rel_show_table (rel, table);
	}
}

void
sql_wid_db_rel_toggle_table_cb (GtkObject * obj,
				SqlMemTable * table, SqlWidDbRel * rel)
{
	sql_wid_db_rel_toggle_table (rel, table);
}

/********************************/
/* showing / hiding sequences   */
/********************************/
void
sql_wid_db_rel_show_seq (SqlWidDbRel * rel, SqlMemSeq * seq)
{
	gboolean send_sig = FALSE;
	GSList *list;
	WidRelNode *node;

	if (seq && !find_seq (rel, seq)) {
		GtkObject *sitem;

		/* if necessary create a new node for the table */
		list = rel->mem_link->nodes;
		node = NULL;
		while (!node && list) {
			if (((WidRelNode *) (list->data))->object == seq)
				node = (WidRelNode *) (list->data);
			list = g_slist_next (list);
		}
		if (!node) {
			node = (WidRelNode *) g_malloc (sizeof (WidRelNode));
			node->x = 100.;
			node->y = 100.;
			node->object = seq;
			rel->mem_link->nodes =
				g_slist_append (rel->mem_link->nodes, node);
		}
		node->shown = TRUE;

		sitem = sql_seq_item_new (rel, node);
		rel->dis_seqs = g_slist_append (rel->dis_seqs, sitem);
		gtk_signal_connect (GTK_OBJECT (sitem), "moved",
				    GTK_SIGNAL_FUNC (sitem_moved_cb), rel);
		rel->conf->save_up_to_date = FALSE;
		send_sig = TRUE;
	}
	update_all_links (rel);
	if (send_sig) {
#ifdef debug_signal
		g_print (">> 'SEQ_SHOWN' from sql_wid_db_rel_show_seq\n");
#endif
		gtk_signal_emit (GTK_OBJECT (rel),
				 sql_wid_db_rel_signals[SEQ_SHOWN], seq);
#ifdef debug_signal
		g_print ("<< 'SEQ_SHOWN' from sql_wid_db_rel_show_seq\n");
#endif
	}
}
void
sql_wid_db_rel_show_seq_cb (GtkObject * obj, SqlMemSeq * seq,
			    SqlWidDbRel * rel)
{
	sql_wid_db_rel_show_seq (rel, seq);
}

void
sql_wid_db_rel_hide_seq (SqlWidDbRel * rel, SqlMemSeq * seq)
{
	SqlSeqItem *found;
	gboolean send_sig = FALSE;

	if (seq && (found = find_seq (rel, seq))) {
		found->node->shown = FALSE;
		gtk_object_destroy (GTK_OBJECT (found));
		rel->dis_seqs = g_slist_remove (rel->dis_seqs, found);
		rel->conf->save_up_to_date = FALSE;
		send_sig = TRUE;
	}
	update_all_links (rel);
	if (send_sig) {
#ifdef debug_signal
		g_print (">> 'SEQ_HIDDEN' from sql_wid_db_rel_hide_seq\n");
#endif
		gtk_signal_emit (GTK_OBJECT (rel),
				 sql_wid_db_rel_signals[SEQ_HIDDEN], seq);
#ifdef debug_signal
		g_print ("<< 'SEQ_HIDDEN' from sql_wid_db_rel_hide_seq\n");
#endif
	}
}
void
sql_wid_db_rel_hide_seq_cb (GtkObject * obj, SqlMemSeq * seq,
			    SqlWidDbRel * rel)
{
	sql_wid_db_rel_hide_seq (rel, seq);
}

void
sql_wid_db_rel_toggle_seq (SqlWidDbRel * rel, SqlMemSeq * seq)
{
	SqlSeqItem *found;

	found = find_seq (rel, seq);
	if (found) {
		sql_wid_db_rel_hide_seq (rel, seq);
	}
	else if (seq) {
		sql_wid_db_rel_show_seq (rel, seq);
	}
}
void
sql_wid_db_rel_toggle_seq_cb (GtkObject * obj, SqlMemSeq * seq,
			      SqlWidDbRel * rel)
{
	sql_wid_db_rel_toggle_seq (rel, seq);
}


/***********************************/
/* Other Functions                 */
/***********************************/

static SqlTableItem *
find_table (SqlWidDbRel * rel, SqlMemTable * table)
{
	SqlTableItem *found = NULL;
	GSList *list = rel->dis_tables;

	while (list && !found) {
		if (SQL_TABLE_ITEM (list->data)->table == table) {
			found = SQL_TABLE_ITEM (list->data);
		}
		else
			list = g_slist_next (list);
	}
	return found;
}

static SqlSeqItem *
find_seq (SqlWidDbRel * rel, SqlMemSeq * seq)
{
	SqlSeqItem *found = NULL;
	GSList *list = rel->dis_seqs;

	while (list && !found) {
		if (SQL_SEQ_ITEM (list->data)->seq == seq) {
			found = SQL_SEQ_ITEM (list->data);
		}
		else
			list = g_slist_next (list);
	}
	return found;
}



/**************************************************************************/
/*                                                                        */
/*                                                                        */
/* SqlTableItem                                                           */
/*                                                                        */
/*                                                                        */
/**************************************************************************/

static void sql_table_item_class_init (SqlTableItemClass * class);
static void sql_table_item_init (SqlTableItem * rel);
static void sql_table_item_destroy (GtkObject * object);
static void sql_table_item_initialize (GnomeCanvasGroup * group,
				       SqlTableItem * item);
static double compute_y (SqlTableItem * item, guint field_num);
static gint sql_table_item_event (GnomeCanvasItem * item, GdkEvent * event,
				  gpointer data);
static void sql_table_item_fields_cb (SqlMemTable * t, SqlMemField * field,
				      SqlTableItem * ti);
static void sql_table_item_reload_fields (SqlTableItem * item);
/* -1 if not found, starts at 0 */
static int sql_table_item_get_pos (SqlTableItem * item, SqlMemField * field);

enum
{
	MOVED,
	ENTERED,
	LEFT,
	LAST_SIGNAL
};

static gint sql_table_item_signals[LAST_SIGNAL] = { 0, 0, 0 };

guint
sql_table_item_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Table_Item",
			sizeof (SqlTableItem),
			sizeof (SqlTableItemClass),
			(GtkClassInitFunc) sql_table_item_class_init,
			(GtkObjectInitFunc) sql_table_item_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_table_item_class_init (SqlTableItemClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
	sql_table_item_signals[MOVED] =
		gtk_signal_new ("moved",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlTableItemClass, moved),
				gtk_marshal_NONE__INT_INT, GTK_TYPE_NONE, 2,
				GTK_TYPE_INT, GTK_TYPE_INT);
	sql_table_item_signals[ENTERED] =
		gtk_signal_new ("entered",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlTableItemClass,
						   entered),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	sql_table_item_signals[LEFT] =
		gtk_signal_new ("left", GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (SqlTableItemClass, left),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	gtk_object_class_add_signals (object_class, sql_table_item_signals,
				      LAST_SIGNAL);
	object_class->destroy = sql_table_item_destroy;
	class->moved = NULL;
	class->entered = NULL;
	class->left = NULL;
}

static void
sql_table_item_init (SqlTableItem * item)
{
	item->rel = NULL;
	item->table = NULL;
	item->canvas_group = NULL;
	item->x_text_space = 3;
	item->y_text_space = 3;
	item->title_text_height = 0;	/* not important */
	item->fields_text_height = 0;	/* not important */
	item->max_text_width = 0;
	item->frame = NULL;
	item->bg_frame = NULL;
	item->field_items = NULL;
}

GtkObject *
sql_table_item_new (SqlWidDbRel * rel, WidRelNode * node)
{
	GtkObject *obj;
	SqlTableItem *item;

	obj = gtk_type_new (sql_table_item_get_type ());
	item = SQL_TABLE_ITEM (obj);
	item->rel = rel;
	item->table = SQL_MEM_TABLE (node->object);
	item->node = node;

	item->canvas_group =
		gnome_canvas_item_new (gnome_canvas_root (rel->canvas),
				       gnome_canvas_group_get_type (), "x",
				       node->x, "y", node->y, NULL);
	sql_table_item_initialize (GNOME_CANVAS_GROUP (item->canvas_group),
				   item);
	gtk_signal_connect_while_alive (GTK_OBJECT (item->table),
					"field_created",
					GTK_SIGNAL_FUNC
					(sql_table_item_fields_cb), item,
					GTK_OBJECT (item));
	gtk_signal_connect_while_alive (GTK_OBJECT (item->table),
					"field_dropped",
					GTK_SIGNAL_FUNC
					(sql_table_item_fields_cb), item,
					GTK_OBJECT (item));

	/* request the update of the canvas before going on, this is because
	   the links which will be created need to have an up to date canvas display. */
	gnome_canvas_update_now (rel->canvas);
	/* signals */
	gtk_signal_connect_while_alive (GTK_OBJECT (item->canvas_group),
					"event",
					(GtkSignalFunc) sql_table_item_event,
					item, GTK_OBJECT (item));

	return GTK_OBJECT (obj);
}

gfloat
sql_table_item_get_xpos (SqlTableItem * titem)
{
	GtkArg arg;

	arg.name = "GnomeCanvasGroup::x";
	gtk_object_getv (GTK_OBJECT (titem->canvas_group), 1, &arg);

	g_assert (arg.type == GTK_TYPE_DOUBLE);
	return GTK_VALUE_DOUBLE (arg);
}

gfloat
sql_table_item_get_ypos (SqlTableItem * titem)
{
	GtkArg arg;

	arg.name = "GnomeCanvasGroup::y";
	gtk_object_getv (GTK_OBJECT (titem->canvas_group), 1, &arg);

	g_assert (arg.type == GTK_TYPE_DOUBLE);
	return GTK_VALUE_DOUBLE (arg);
}

static void
sql_table_item_fields_cb (SqlMemTable * t, SqlMemField * field,
			  SqlTableItem * ti)
{
	sql_table_item_reload_fields (ti);
}

static void
sql_table_item_destroy (GtkObject * object)
{
	SqlTableItem *item;
	GtkObjectClass *parent_class;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_TABLE_ITEM (object));
	item = SQL_TABLE_ITEM (object);

	/* the item's data to be freeed */
	gtk_object_destroy (GTK_OBJECT (item->canvas_group));
	item->canvas_group = NULL;

	/* what about the parent class */
	parent_class = gtk_type_class (gtk_object_get_type ());

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


/* first field <-> field_num = 0 */
static double
compute_y (SqlTableItem * item, guint field_num)
{
	double calc;

	calc = (3 * (item->y_text_space) + item->title_text_height) +
		field_num * (item->y_text_space + item->fields_text_height);
	return calc;
}

static void
sql_table_item_reload_fields (SqlTableItem * item)
{
	GnomeCanvasItem *newitem;
	GnomeCanvasGroup *group = GNOME_CANVAS_GROUP (item->canvas_group);
	GSList *list, *hold;
	guint field_num;

	/* first destroy existing fields */
	list = item->field_items;
	while (list) {
		gtk_object_destroy (GTK_OBJECT (list->data));
		hold = list;
		list = g_slist_remove_link (list, list);
		g_slist_free_1 (hold);
	}
	item->field_items = NULL;

	/* now putting the new fields */
	field_num = 0;
	list = item->table->fields;
	while (list) {
		newitem = gnome_canvas_item_new (group,
						 GNOME_TYPE_CANVAS_TEXT,
						 "font", DEFAULT_REL_FONT,
						 "x",
						 (double) (item->
							   x_text_space), "y",
						 compute_y (item, field_num),
						 "text",
						 SQL_MEM_FIELD (list->data)->
						 name, "fill_color", "black",
						 "justification",
						 GTK_JUSTIFY_RIGHT, "anchor",
						 GTK_ANCHOR_NORTH_WEST, NULL);
		/* height: does not change, but easier this way:) */
		item->fields_text_height = GNOME_CANVAS_ITEM (newitem)->y2 -
			GNOME_CANVAS_ITEM (newitem)->y1;

		/* max width */
		if (GNOME_CANVAS_ITEM (newitem)->x2 -
		    GNOME_CANVAS_ITEM (newitem)->x1 > item->max_text_width)
			item->max_text_width =
				GNOME_CANVAS_ITEM (newitem)->x2 -
				GNOME_CANVAS_ITEM (newitem)->x1;

		field_num++;
		/* list of the fields */
		gtk_object_set_data (GTK_OBJECT (newitem), "fieldptr",
				     list->data);
		item->field_items =
			g_slist_append (item->field_items, newitem);
		list = g_slist_next (list);
	}
	/* adding the outline big frame */
	if (item->bg_frame)
		gtk_object_destroy (GTK_OBJECT (item->bg_frame));
	if (item->table->is_view)
		newitem = gnome_canvas_item_new (group,
						 GNOME_TYPE_CANVAS_RECT,
						 "x1", (double) 0,
						 "y1", (double) 0,
						 "x2",
						 (double) (item->
							   max_text_width +
							   2 *
							   item->
							   x_text_space),
						 "y2", compute_y (item,
								  field_num),
						 "outline_color", "black",
						 "fill_color", "lightblue",
						 "width_units", 1.0, NULL);
	else
		newitem = gnome_canvas_item_new (group,
						 GNOME_TYPE_CANVAS_RECT,
						 "x1", (double) 0,
						 "y1", (double) 0,
						 "x2",
						 (double) (item->
							   max_text_width +
							   2 *
							   item->
							   x_text_space),
						 "y2", compute_y (item,
								  field_num),
						 "outline_color", "black",
						 "fill_color", "white",
						 "width_units", 1.0, NULL);

	item->bg_frame = newitem;
	gnome_canvas_item_lower_to_bottom (item->bg_frame);
}

static void
sql_table_item_initialize (GnomeCanvasGroup * group, SqlTableItem * item)
{
	GnomeCanvasItem *newitem;

	/* table's title */
	newitem = gnome_canvas_item_new (group,
					 GNOME_TYPE_CANVAS_TEXT,
					 "text", item->table->name,
					 "font", DEFAULT_REL_FONT,
					 "x", (double) (item->x_text_space),
					 "y", (double) (item->y_text_space),
					 "fill_color", "black",
					 "justification", GTK_JUSTIFY_RIGHT,
					 "anchor", GTK_ANCHOR_NORTH_WEST,
					 NULL);
	item->title_text_height = GNOME_CANVAS_ITEM (newitem)->y2 -
		GNOME_CANVAS_ITEM (newitem)->y1;
	item->max_text_width = GNOME_CANVAS_ITEM (newitem)->x2 -
		GNOME_CANVAS_ITEM (newitem)->x1;
	/* going through all the fields */
	sql_table_item_reload_fields (item);
	newitem = gnome_canvas_item_new (group,
					 GNOME_TYPE_CANVAS_RECT,
					 "x1", (double) 0,
					 "y1", (double) 0,
					 "x2",
					 (double) (item->max_text_width +
						   2 * item->x_text_space),
					 "y2",
					 (double) (item->title_text_height +
						   2 * item->y_text_space),
					 "outline_color", "black",
					 "fill_color", "white", "width_units",
					 1.0, NULL);
	item->frame = newitem;

	gnome_canvas_item_lower_to_bottom (newitem);
	gnome_canvas_item_lower_to_bottom (item->bg_frame);
}

static int
sql_table_item_get_pos (SqlTableItem * item, SqlMemField * field)
{
	gboolean found = FALSE;
	gint pos = 0;
	GSList *list;
	SqlMemField *data_field;

	list = item->field_items;
	while (list && !found) {
		data_field =
			gtk_object_get_data (GTK_OBJECT (list->data),
					     "fieldptr");
		if (data_field == field)
			found = TRUE;
		else {
			list = g_slist_next (list);
			pos++;
		}
	}

	if (found)
		return pos;
	else
		return -1;
}

/* returns +1 for right link */
static void
sql_table_item_elect_r_or_l (SqlTableItem * first, gint * firstpos,
			     SqlTableItem * sec, gint * secpos)
{
	if ((first->canvas_group->x1 + first->max_text_width / 2. +
	     first->x_text_space) >
	    (sec->canvas_group->x1 + sec->max_text_width / 2. +
	     sec->x_text_space)) {
		/* first is on the right of second */
		if ((sec->canvas_group->x1 + sec->max_text_width +
		     2 * sec->x_text_space) > first->canvas_group->x1) {
			*secpos = +1;
			*firstpos = +1;
		}
		else {
			*secpos = +1;
			*firstpos = -1;
		}
	}
	else {			/* first is on the left of second */
		if ((first->canvas_group->x1 + first->max_text_width +
		     2 * first->x_text_space) < sec->canvas_group->x1) {
			*secpos = -1;
			*firstpos = +1;
		}
		else {
			*secpos = -1;
			*firstpos = -1;
		}
	}
}

void
sql_table_item_get_left_anchor_point (SqlTableItem * ti,
				      SqlMemField * field,
				      double *x, double *y)
{
	gint pos = sql_table_item_get_pos (ti, field);
	double ypos;

	if (pos >= 0) {
		ypos = 3 * (ti->y_text_space) + ti->title_text_height +
			pos * (ti->y_text_space + ti->fields_text_height);
		*x = sql_table_item_get_xpos (ti);
		*y = sql_table_item_get_ypos (ti) + ypos +
			ti->fields_text_height / 2.;
	}
}

void
sql_table_item_get_right_anchor_point (SqlTableItem * ti,
				       SqlMemField * field,
				       double *x, double *y)
{
	gint pos = sql_table_item_get_pos (ti, field);
	double ypos, x1, x2;
	GtkArg args[2];

	args[0].name = "GnomeCanvasRE::x1";
	args[1].name = "GnomeCanvasRE::x2";
	gtk_object_getv (GTK_OBJECT (ti->frame), 2, args);

	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	g_assert (args[1].type == GTK_TYPE_DOUBLE);
	x1 = GTK_VALUE_DOUBLE (args[0]);
	x2 = GTK_VALUE_DOUBLE (args[1]);

	if (pos >= 0) {
		ypos = 3 * (ti->y_text_space) + ti->title_text_height +
			pos * (ti->y_text_space + ti->fields_text_height);

		*x = sql_table_item_get_xpos (ti) + (x2 - x1);
		*y = sql_table_item_get_ypos (ti) + ypos +
			ti->fields_text_height / 2.;
	}
}


/* data points to the SqlTableItem */
static gint
sql_table_item_event (GnomeCanvasItem * item, GdkEvent * event, gpointer data)
{
	SqlTableItem *ti = SQL_TABLE_ITEM (data);
	static double x, y;	/* used to keep track of motion coordinates */
	double new_x, new_y, dx = 0, dy = 0;
	gboolean has_moved;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (event->button.button == 1) {
			/* Remember starting position */
			gnome_canvas_item_raise_to_top (item);
			x = event->button.x;
			y = event->button.y;
			return TRUE;
		}
		break;

	case GDK_BUTTON_RELEASE:
		if (event->button.button == 1) 
			update_canvas_adjustments(ti->rel, TRUE);			
		break;

	case GDK_MOTION_NOTIFY:
		if (event->button.state & GDK_BUTTON1_MASK) {
			/* Get the new position and move by the difference */
			has_moved = FALSE;
			new_x = event->motion.x;
			new_y = event->motion.y;
			gnome_canvas_item_move (item, new_x - x, new_y - y);
			ti->node->x += new_x - x;
			ti->node->y += new_y - y;
			if ((new_x != x) || (new_y != y)) {
				has_moved = TRUE;
				dx = new_x - x;
				dy = new_y - y;
			}
			x = new_x;
			y = new_y;

			if (has_moved) {
				/* management of the adjustments */
				update_canvas_adjustments(ti->rel, FALSE);

				gtk_signal_emit (GTK_OBJECT (data),
						 sql_table_item_signals
						 [MOVED],
						 (int) (item->x1 + dx),
						 (int) (item->y1 + dy));
			}
			
			return TRUE;
		}
		break;

	case GDK_ENTER_NOTIFY:
		gnome_canvas_item_set (ti->frame,
				       "fill_color", "IndianRed1", NULL);
		gtk_signal_emit (GTK_OBJECT (ti),
				 sql_table_item_signals[ENTERED]);
		return TRUE;

	case GDK_LEAVE_NOTIFY:
		gnome_canvas_item_set (ti->frame,
				       "fill_color", "white", NULL);
		gtk_signal_emit (GTK_OBJECT (ti),
				 sql_table_item_signals[LEFT]);
		return TRUE;
		break;
	default:
		break;
	}

	return FALSE;
}


static void compute_boundaries(SqlWidDbRel *rel, gdouble *cxmin, gdouble *cymin, gdouble *cxmax, gdouble *cymax);
/* reduce tells if we also reduce the canvas as necessary
 * instead of just expanding as necessary */
static void 
update_canvas_adjustments(SqlWidDbRel *rel, gboolean reduce)
{
	gdouble cxmin=0, cymin=0, cxmax=0, cymax=0;
	
	GtkAdjustment *adj;
	gboolean changed;
	
	/* get the canvas boundaries */
	compute_boundaries(rel, &cxmin, &cymin, &cxmax, &cymax);

	/* horizontal adjustment */
#define STEPPING 2*GNOME_PAD
	adj = gtk_layout_get_hadjustment (GTK_LAYOUT (rel->canvas));
	changed = FALSE;
	if (cxmax > adj->upper) { /* maximum */
		adj->upper = cxmax + STEPPING/2.;
		changed = TRUE;
	}
	else {
		if (reduce && (cxmax < adj->upper - STEPPING)  && (cxmax + STEPPING > rel->mem_link->size.width)){
			adj->upper = cxmax + STEPPING;
			changed = TRUE;
		}
	}

	if (cxmin < adj->lower) { /* minimum */
		/* TODO */
	}

	if (changed) {
		rel->mem_link->set_canvas_scroll = TRUE;
		gtk_adjustment_changed (GTK_ADJUSTMENT(adj));
		rel->mem_link->set_canvas_scroll = FALSE;
	}
				
	/* vertical adjustment */
	adj = gtk_layout_get_vadjustment (GTK_LAYOUT (rel->canvas));
	changed = FALSE;
	if (cymax > adj->upper) { /* maximum */
		adj->upper = cymax + STEPPING/2.;
		changed = TRUE;
	}
	else {
		if (reduce && (cymax < adj->upper - STEPPING)  && (cymax+ STEPPING > rel->mem_link->size.height)){
			adj->upper = cymax + STEPPING;
			changed = TRUE;
		}
	}
	if (cymin < adj->lower) { /* minimum */
		/* TODO */
	}

	if (changed) {
		rel->mem_link->set_canvas_scroll = TRUE;
		gtk_adjustment_changed (GTK_ADJUSTMENT(adj));
		rel->mem_link->set_canvas_scroll = FALSE;
	}
}

static void 
compute_boundaries(SqlWidDbRel *rel, gdouble *cxmin, gdouble *cymin, gdouble *cxmax, gdouble *cymax)
{
	gdouble cx, cy;

	SqlTableItem *ti;
	SqlSeqItem *si;
	GSList *items;
	
	*cxmin = 0.;
	*cymin = 0.;
	*cxmax = 0.;
	*cymax = 0.;

	/* get tables boundaries */
	items = rel->dis_tables;
	while (items) {
		ti = SQL_TABLE_ITEM (items->data);
		*cxmax = (ti->canvas_group->x2 > *cxmax) ? ti->canvas_group->x2 : *cxmax;
		*cymax = (ti->canvas_group->y2 > *cymax) ? ti->canvas_group->y2 : *cymax;
		*cxmin = (ti->canvas_group->x1 < *cxmin) ? ti->canvas_group->x1 : *cxmin;
		*cymin = (ti->canvas_group->y1 < *cymin) ? ti->canvas_group->y1 : *cymin;
		
		items = g_slist_next(items);
	}

	/* get sequences boundaries */
	items = rel->dis_seqs;
	while (items) {
		si = SQL_SEQ_ITEM (items->data);
		*cxmax = (si->canvas_group->x2 > *cxmax) ? si->canvas_group->x2 : *cxmax;
		*cymax = (si->canvas_group->y2 > *cymax) ? si->canvas_group->y2 : *cymax;
		*cxmin = (si->canvas_group->x1 < *cxmin) ? si->canvas_group->x1 : *cxmin;
		*cymin = (si->canvas_group->y1 < *cymin) ? si->canvas_group->y1 : *cymin;
		
		items = g_slist_next(items);
	}

	/* NO convertion of coordinates needed because it is already in World coordinates */
}


static void
arrows_parents_get_coord (SqlTableItem * parentitem,
			  SqlTableItem * childitem,
			  double *px, double *py, double *cx, double *cy)
{
	GtkArg args[4];
	gdouble pW, pH, cW, cH;	/* half of width and height */
	gdouble pcx, pcy, ccx, ccy;	/* coord. of centers of rectangles */

	args[0].name = "GnomeCanvasRE::x1";
	args[1].name = "GnomeCanvasRE::y1";
	args[2].name = "GnomeCanvasRE::x2";
	args[3].name = "GnomeCanvasRE::y2";
	gtk_object_getv (GTK_OBJECT (parentitem->bg_frame), 4, args);
	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	g_assert (args[1].type == GTK_TYPE_DOUBLE);
	g_assert (args[2].type == GTK_TYPE_DOUBLE);
	g_assert (args[3].type == GTK_TYPE_DOUBLE);
	pW = (GTK_VALUE_DOUBLE (args[2]) - GTK_VALUE_DOUBLE (args[0])) / 2.;
	pH = (GTK_VALUE_DOUBLE (args[3]) - GTK_VALUE_DOUBLE (args[1])) / 2.;
	pcx = sql_table_item_get_xpos (parentitem) + pW;
	pcy = sql_table_item_get_ypos (parentitem) + pH;

	gtk_object_getv (GTK_OBJECT (childitem->bg_frame), 4, args);
	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	g_assert (args[1].type == GTK_TYPE_DOUBLE);
	g_assert (args[2].type == GTK_TYPE_DOUBLE);
	g_assert (args[3].type == GTK_TYPE_DOUBLE);
	cW = (GTK_VALUE_DOUBLE (args[2]) - GTK_VALUE_DOUBLE (args[0])) / 2.;
	cH = (GTK_VALUE_DOUBLE (args[3]) - GTK_VALUE_DOUBLE (args[1])) / 2.;
	ccx = sql_table_item_get_xpos (childitem) + cW;
	ccy = sql_table_item_get_ypos (childitem) + cH;

	/* only offsets from centers */
	if (pcx == ccx) {
		*px = 0.;
		*cx = 0.;
		if (pcy < ccy) {
			*py = pH;
			*cy = -cH;
		}
		else {
			*py = -pH;
			*cy = cH;
		}
	}
	else {			/* ccx != pxc */
		/* parents */
		*py = abs (pW * (ccy - pcy) / (ccx - pcx));
		if (*py < pH) {
			*px = (pcx < ccx) ? pW : -pW;
			*py = *px * (ccy - pcy) / (ccx - pcx);
		}
		else {
			*py = (pcy < ccy) ? pH : -pH;
			*px = *py * (ccx - pcx) / (ccy - pcy);
		}

		/* child */
		*cy = abs (cW * (ccy - pcy) / (ccx - pcx));
		if (*cy < cH) {
			*cx = (pcx > ccx) ? cW : -cW;
			*cy = *cx * (ccy - pcy) / (ccx - pcx);
		}
		else {
			*cy = (pcy < ccy) ? -cH : cH;
			*cx = *cy * (ccx - pcx) / (ccy - pcy);
		}
	}

	/* completion to have absolute coordinates */
	*px += pcx;
	*py += pcy;
	*cx += ccx;
	*cy += ccy;
}

/**************************************************************************/
/*                                                                        */
/*                                                                        */
/* SqlSeqItem                                                           */
/*                                                                        */
/*                                                                        */
/**************************************************************************/

static void sql_seq_item_class_init (SqlSeqItemClass * class);
static void sql_seq_item_init (SqlSeqItem * rel);
static void sql_seq_item_destroy (GtkObject * object);
static void sql_seq_item_initialize (GnomeCanvasGroup * group,
				     SqlSeqItem * item);
static gint sql_seq_item_event (GnomeCanvasItem * item, GdkEvent * event,
				gpointer data);
enum
{
	SMOVED,
	SENTERED,
	SLEFT,
	LAST_SEQ_SIGNAL
};

static gint sql_seq_item_signals[LAST_SEQ_SIGNAL] = { 0, 0, 0 };

guint
sql_seq_item_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Seq_Item",
			sizeof (SqlSeqItem),
			sizeof (SqlSeqItemClass),
			(GtkClassInitFunc) sql_seq_item_class_init,
			(GtkObjectInitFunc) sql_seq_item_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_seq_item_class_init (SqlSeqItemClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
	sql_seq_item_signals[SMOVED] =
		gtk_signal_new ("moved",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlSeqItemClass, moved),
				gtk_marshal_NONE__INT_INT, GTK_TYPE_NONE, 2,
				GTK_TYPE_INT, GTK_TYPE_INT);
	sql_seq_item_signals[SENTERED] =
		gtk_signal_new ("entered",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlSeqItemClass, entered),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	sql_seq_item_signals[SLEFT] =
		gtk_signal_new ("left", GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (SqlSeqItemClass, left),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	gtk_object_class_add_signals (object_class, sql_seq_item_signals,
				      LAST_SIGNAL);
	object_class->destroy = sql_seq_item_destroy;
	class->moved = NULL;
	class->entered = NULL;
	class->left = NULL;
}

static void
sql_seq_item_init (SqlSeqItem * item)
{
	item->rel = NULL;
	item->seq = NULL;
	item->canvas_group = NULL;
	item->text = NULL;
	item->bg_frame = NULL;
	item->x_text_space = 6;
	item->y_text_space = 6;
}

GtkObject *
sql_seq_item_new (SqlWidDbRel * rel, WidRelNode * node)
{
	GtkObject *obj;
	SqlSeqItem *item;

	obj = gtk_type_new (sql_seq_item_get_type ());
	item = SQL_SEQ_ITEM (obj);
	item->rel = rel;
	item->seq = SQL_MEM_SEQ (node->object);
	item->node = node;

	item->canvas_group =
		gnome_canvas_item_new (gnome_canvas_root (rel->canvas),
				       gnome_canvas_group_get_type (), "x",
				       node->x, "y", node->y, NULL);
	sql_seq_item_initialize (GNOME_CANVAS_GROUP (item->canvas_group),
				 item);

	/* signals */
	gtk_signal_connect_while_alive (GTK_OBJECT (item->canvas_group),
					"event",
					(GtkSignalFunc) sql_seq_item_event,
					item, GTK_OBJECT (item));

	return GTK_OBJECT (obj);
}

gfloat
sql_seq_item_get_center_xpos (SqlSeqItem * sitem)
{
	GtkArg args[2];
	double X;

	args[0].name = "GnomeCanvasGroup::x";
	gtk_object_getv (GTK_OBJECT (sitem->canvas_group), 1, args);

	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	X = GTK_VALUE_DOUBLE (args[0]);

	args[0].name = "GnomeCanvasRE::x1";
	args[1].name = "GnomeCanvasRE::x2";
	gtk_object_getv (GTK_OBJECT (sitem->bg_frame), 2, args);
	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	g_assert (args[1].type == GTK_TYPE_DOUBLE);

	return (X +
		(GTK_VALUE_DOUBLE (args[0]) +
		 GTK_VALUE_DOUBLE (args[1])) / 2.);
}

gfloat
sql_seq_item_get_center_ypos (SqlSeqItem * sitem)
{
	GtkArg args[2];
	double Y;

	args[0].name = "GnomeCanvasGroup::y";
	gtk_object_getv (GTK_OBJECT (sitem->canvas_group), 1, args);

	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	Y = GTK_VALUE_DOUBLE (args[0]);

	args[0].name = "GnomeCanvasRE::y1";
	args[1].name = "GnomeCanvasRE::y2";
	gtk_object_getv (GTK_OBJECT (sitem->bg_frame), 2, args);
	g_assert (args[0].type == GTK_TYPE_DOUBLE);
	g_assert (args[1].type == GTK_TYPE_DOUBLE);

	return (Y +
		(GTK_VALUE_DOUBLE (args[0]) +
		 GTK_VALUE_DOUBLE (args[1])) / 2.);
}

static void
sql_seq_item_destroy (GtkObject * object)
{
	SqlSeqItem *item;
	GtkObjectClass *parent_class;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_SEQ_ITEM (object));
	item = SQL_SEQ_ITEM (object);

	/* the item's data to be freeed */
	gtk_object_destroy (GTK_OBJECT (item->canvas_group));
	item->canvas_group = NULL;

	/*g_free(item->node); Not free here! */

	/* what about the parent class */
	parent_class = gtk_type_class (gtk_object_get_type ());

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
sql_seq_item_initialize (GnomeCanvasGroup * group, SqlSeqItem * item)
{
	GnomeCanvasItem *newitem;

	/* sequence's title */
	newitem = gnome_canvas_item_new (group,
					 GNOME_TYPE_CANVAS_TEXT,
					 "text", item->seq->name,
					 "font", DEFAULT_REL_FONT,
					 "anchor", GTK_ANCHOR_CENTER,
					 "x", 0.,
					 "y", 0.,
					 "fill_color", "black", NULL);
	item->text = newitem;
	/* ellipse */
	newitem = gnome_canvas_item_new (group,
					 GNOME_TYPE_CANVAS_ELLIPSE,
					 "x1", (double) 0,
					 "y1", (double) 0,
					 "x2",
					 GNOME_CANVAS_ITEM (item->text)->x2 -
					 GNOME_CANVAS_ITEM (item->text)->x1 +
					 2. * item->x_text_space, "y2",
					 GNOME_CANVAS_ITEM (item->text)->y2 -
					 GNOME_CANVAS_ITEM (item->text)->y1 +
					 2. * item->y_text_space,
					 "outline_color", "black",
					 "fill_color", "MediumTurquoise",
					 "width_units", 1.0, NULL);
	item->bg_frame = newitem;
	/* moving title to center */
	gnome_canvas_item_move (item->text,
				(GNOME_CANVAS_ITEM (item->text)->x2 -
				 GNOME_CANVAS_ITEM (item->text)->x1) / 2. +
				item->x_text_space,
				(GNOME_CANVAS_ITEM (item->text)->y2 -
				 GNOME_CANVAS_ITEM (item->text)->y1) / 2. +
				item->y_text_space);

	gnome_canvas_item_raise_to_top (item->text);
}

/* data points to the SqlSeqItem */
static gint
sql_seq_item_event (GnomeCanvasItem * item, GdkEvent * event, gpointer data)
{
	SqlSeqItem *si = SQL_SEQ_ITEM (data);
	static double x, y;	/* used to keep track of motion coordinates */
	double new_x, new_y, dx = 0, dy = 0;
	gboolean has_moved;

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		if (event->button.button == 1) {
			/* Remember starting position */
			gnome_canvas_item_raise_to_top (item);
			x = event->button.x;
			y = event->button.y;
			return TRUE;
		}
		break;

	case GDK_BUTTON_RELEASE:
		if (event->button.button == 1) 
			update_canvas_adjustments(si->rel, TRUE);			
		break;

	case GDK_MOTION_NOTIFY:
		if (event->button.state & GDK_BUTTON1_MASK) {
			/* Get the new position and move by the difference */
			has_moved = FALSE;
			new_x = event->motion.x;
			new_y = event->motion.y;
			gnome_canvas_item_move (item, new_x - x, new_y - y);
			si->node->x += new_x - x;
			si->node->y += new_y - y;
			if ((new_x != x) || (new_y != y)) {
				has_moved = TRUE;
				dx = new_x - x;
				dy = new_y - y;
			}
			x = new_x;
			y = new_y;

			if (has_moved) {
				/* management of the adjustments */
				update_canvas_adjustments(si->rel, FALSE);

				gtk_signal_emit (GTK_OBJECT (data),
						 sql_seq_item_signals[SMOVED],
						 (int) (item->x1 + dx),
						 (int) (item->y1 + dy));
			}
			return TRUE;
		}
		break;

	case GDK_ENTER_NOTIFY:
		gnome_canvas_item_set (si->bg_frame,
				       "fill_color", "IndianRed1", NULL);
		gtk_signal_emit (GTK_OBJECT (si),
				 sql_seq_item_signals[SENTERED]);
		return TRUE;

	case GDK_LEAVE_NOTIFY:
		gnome_canvas_item_set (si->bg_frame,
				       "fill_color", "MediumTurquoise", NULL);
		gtk_signal_emit (GTK_OBJECT (si),
				 sql_seq_item_signals[SLEFT]);
		return TRUE;
		break;
	default:
		break;
	}

	return FALSE;
}


/**************************************************************************/
/*                                                                        */
/*                                                                        */
/* SqlLinkItem                                                            */
/*                                                                        */
/*                                                                        */
/**************************************************************************/

static void sql_link_item_class_init (SqlLinkItemClass * class);
static void sql_link_item_init (SqlLinkItem * rel);
static void sql_link_item_destroy (GtkObject * object);

/* moving CB */
static void sql_link_item_moved (gpointer tsi, gint x, gint y,
				 SqlLinkItem * item);

guint
sql_link_item_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Link_Item",
			sizeof (SqlLinkItem),
			sizeof (SqlLinkItemClass),
			(GtkClassInitFunc) sql_link_item_class_init,
			(GtkObjectInitFunc) sql_link_item_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_link_item_class_init (SqlLinkItemClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	object_class->destroy = sql_link_item_destroy;
}

static void
sql_link_item_init (SqlLinkItem * item)
{
	item->link_type = SQL_LINK_FIELD;
	item->from = NULL;
	item->to = NULL;
	item->link_mem = NULL;
	item->line = NULL;
}

GtkObject *
sql_link_item_new_field_field (GnomeCanvas * canvas,
			       SqlTableItem * from,
			       SqlTableItem * to, SqlMemFlink * in_mem)
{
	GtkObject *obj;
	SqlLinkItem *item;
	int i;
	GnomeCanvasPoints *points;

	obj = gtk_type_new (sql_link_item_get_type ());
	item = SQL_LINK_ITEM (obj);

	/* basic setup */
	item->link_type = SQL_LINK_FIELD;
	item->link_mem = (gpointer) in_mem;
	item->from = (gpointer) from;
	item->to = to;

	points = gnome_canvas_points_new (4);
	for (i = 0; i < 8; i++)
		points->coords[i] = 0.;

	item->line = gnome_canvas_item_new (gnome_canvas_root (canvas),
					    gnome_canvas_line_get_type (),
					    "points", points,
					    "width_pixels", 1,
					    "last_arrowhead", TRUE,
					    "arrow_shape_a", 5.,
					    "arrow_shape_b", 5.,
					    "arrow_shape_c", 5., NULL);
	gnome_canvas_points_free (points);
	gnome_canvas_item_lower_to_bottom (item->line);


	sql_link_item_moved (NULL, 0, 0, item);
	gtk_signal_connect_while_alive (GTK_OBJECT (from), "moved",
					GTK_SIGNAL_FUNC (sql_link_item_moved),
					item, obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (to), "moved",
					GTK_SIGNAL_FUNC (sql_link_item_moved),
					item, obj);
	return GTK_OBJECT (obj);
}

GtkObject *
sql_link_item_new_seq_field (GnomeCanvas * canvas,
			     SqlSeqItem * from,
			     SqlTableItem * to, SqlMemField * in_mem)
{
	GtkObject *obj;
	SqlLinkItem *item;
	int i;
	GnomeCanvasPoints *points;

	obj = gtk_type_new (sql_link_item_get_type ());
	item = SQL_LINK_ITEM (obj);

	/* basic setup */
	item->link_type = SQL_LINK_SEQUENCE;
	item->link_mem = (gpointer) in_mem;
	item->from = (gpointer) from;
	item->to = to;

	points = gnome_canvas_points_new (3);
	for (i = 0; i < 6; i++)
		points->coords[i] = 0.;

	item->line = gnome_canvas_item_new (gnome_canvas_root (canvas),
					    gnome_canvas_line_get_type (),
					    "points", points,
					    "width_pixels", 1,
					    "fill_color", "red",
					    "last_arrowhead", TRUE,
					    "arrow_shape_a", 5.,
					    "arrow_shape_b", 5.,
					    "arrow_shape_c", 5., NULL);
	gnome_canvas_points_free (points);
	gnome_canvas_item_lower_to_bottom (item->line);

	sql_link_item_moved (NULL, 0, 0, item);
	gtk_signal_connect_while_alive (GTK_OBJECT (from), "moved",
					GTK_SIGNAL_FUNC (sql_link_item_moved),
					item, obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (to), "moved",
					GTK_SIGNAL_FUNC (sql_link_item_moved),
					item, obj);
	return GTK_OBJECT (obj);
}

static void
sql_link_item_destroy (GtkObject * object)
{
	SqlLinkItem *item;
	GtkObjectClass *parent_class;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_LINK_ITEM (object));

	/* the item's data to be freeed */
	item = SQL_LINK_ITEM (object);
	gtk_object_destroy (GTK_OBJECT (item->line));
	item->line = NULL;

	/* what about the parent class */
	parent_class = gtk_type_class (gtk_object_get_type ());

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
sql_link_item_update_pos (SqlLinkItem * item)
{
	if (item->link_type == SQL_LINK_FIELD)
		sql_link_item_moved (NULL, 0, 0, item);
}

/*  x and y are not used here */
/* tsi is either a SqlTableItem or SqlSeqItem depending on the nature
   written in item */
static void
sql_link_item_moved (gpointer tsi, gint x, gint y, SqlLinkItem * item)
{
	double x1, y1, xr1, xl1, yo1, delta;
	gint from, to, index = 0;
	GnomeCanvasPoints *points = NULL;

	switch (item->link_type) {
	case SQL_LINK_FIELD:
		points = gnome_canvas_points_new (4);
		sql_table_item_elect_r_or_l (SQL_TABLE_ITEM (item->from),
					     &from, item->to, &to);
		/* FROM point */
		if (from > 0)
			sql_table_item_get_right_anchor_point (SQL_TABLE_ITEM
							       (item->from),
							       SQL_MEM_FLINK
							       (item->
								link_mem)->
							       from, &x1,
							       &y1);
		else
			sql_table_item_get_left_anchor_point (SQL_TABLE_ITEM
							      (item->from),
							      SQL_MEM_FLINK
							      (item->
							       link_mem)->
							      from, &x1, &y1);
		points->coords[0] = x1;
		points->coords[1] = y1;
		points->coords[2] = x1 + from * 10.;
		points->coords[3] = y1;
		/* TO point */
		if (to > 0)
			sql_table_item_get_right_anchor_point (item->to,
							       SQL_MEM_FLINK
							       (item->
								link_mem)->to,
							       &x1, &y1);
		else
			sql_table_item_get_left_anchor_point (item->to,
							      SQL_MEM_FLINK
							      (item->
							       link_mem)->to,
							      &x1, &y1);
		points->coords[6] = x1;
		points->coords[7] = y1;
		points->coords[4] = x1 + to * 10.;
		points->coords[5] = y1;
		break;
	case SQL_LINK_SEQUENCE:

		/* FROM point */
		x1 = sql_seq_item_get_center_xpos (SQL_SEQ_ITEM (item->from));
		y1 = sql_seq_item_get_center_ypos (SQL_SEQ_ITEM (item->from));
		/* TO point */
		sql_table_item_get_right_anchor_point (item->to,
						       SQL_MEM_FIELD (item->
								      link_mem),
						       &xr1, &yo1);
		sql_table_item_get_left_anchor_point (item->to,
						      SQL_MEM_FIELD (item->
								     link_mem),
						      &xl1, &yo1);
		index = 2;
		delta = abs ((xr1 - xl1) / 2.);
		if (abs (x1 - xr1) < abs (x1 - xl1)) {
			to = 1;
		}
		else {
			to = -1;
			xr1 = xl1;
		}

		if (((to > 0) && (x1 < xr1 + 10.)) || ((to < 0) && (x1 > xr1 - 10.))) {	/* needs a 4th point */
			index = 4;
			points = gnome_canvas_points_new (4);
			points->coords[2] = xr1 + to * 10.;
			points->coords[3] = y1;
		}
		else {
			index = 2;
			points = gnome_canvas_points_new (3);
		}
		points->coords[0] = x1;
		points->coords[1] = y1;
		points->coords[index + 2] = xr1;
		points->coords[index + 3] = yo1;
		points->coords[index] = xr1 + to * 10.;
		points->coords[index + 1] = yo1;
		break;
	}

	gnome_canvas_item_set (item->line, "points", points, NULL);

	gnome_canvas_points_free (points);
}
